#!/bin/sh -efu
#
# Copyright (C) 2006 Securedog
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY AUTHOR AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL AUTHOR OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# $Id: pkg_replace.sh,v 1.15 2006/10/31 14:37:39 securedog Exp $

usage() {
	echo "usage: ${0##*/} [-habBcCiInqrRuvVwW] [-l file] [-m make_args]"
	echo "         [-x pkgname] [[pkgname[=package]] [package] ...]"
	exit ${1:-0}
}

is_yes() {
	case ${1-} in
	[Yy][Ee][Ss]|[Yy]|[Tt][Rr][Uu][Ee])
		return 0 ;;
	*)	return 1 ;;
	esac
}

warn() {
	echo "** ${@-}" >&2
}

info() {
	echo "--->  ${@-}"
}

prompt_yesno() {
	local prompt default input

	prompt=${1-"OK?"}
	default=${2-"yes"}

	echo -n "${prompt} [${default}] " >&2
	read input

	is_yes ${input:-"${default}"}
}

init_options() {
	opt_all=NO
	opt_backup=YES
	opt_keep_backup=NO
	opt_beforeclean=NO
	opt_afterclean=YES
	opt_force=NO
	opt_interactive=NO
	opt_use_index=NO
	opt_log_output=
	opt_make_args=
	opt_noexecute=NO
	opt_noconf=NO
	opt_depends=NO
	opt_required_by=NO
	opt_preserve_libs=YES
	opt_verbose=NO
	opt_version=NO
	opt_exclude=
	opt_debug=0
	opt_replace=
	log_format="+:done -:ignored *:skipped !:failed"
#ifdef WITH_CONFIG
	MAKE_ARGS=
	BEFOREBUILD=
	BEFOREDEINSTALL=
	AFTERINSTALL=
	IGNORE=
	USE_PKGS=
#endif
}

init_variables() {
#ifdef WITH_PKGSRC
	: ${PKGSRCDIR="/usr/pkgsrc"}
	: ${PKGREPOSITORY="${PKGSRCDIR}/packages/All"}
	: ${PKG_SUFX=".tgz"}
#else
	: ${PORTSDIR="/usr/ports"}
	: ${PKGREPOSITORY="${PORTSDIR}/packages/All"}
#ifdef WITH_OPENBSD
	: ${PKG_SUFX=".tgz"}
#else
	: ${PKG_SUFX=".tbz"}
#endif
#endif
	: ${PKG_BACKUP_DIR=${PKGREPOSITORY}}
	: ${PKG_DBDIR="/var/db/pkg"}
	: ${PKG_TMPDIR=${TMPDIR:-"/var/tmp"}}
	: ${PKGCOMPATDIR="/usr/local/lib/compat/pkg"}
	: ${MAKE="make"}
	cwd=${PWD}
#ifdef WITH_PKGSRC
	export PKGSRCDIR
#else
	export PORTSDIR
#endif
	export PKG_DBDIR PKG_TMPDIR PKG_SUFX PKGCOMPATDIR
}

compat_variables() {
#ifdef WITH_PKGSRC
	PORTSDIR=${PKGSRCDIR}

#endif
	if [ ${PKG_REPLACE:+1} ]; then
		parse_options ${PKG_REPLACE}
	fi
}

parse_options() {
	local OPT OPTARG OPTIND

	while getopts habBcCfiIl:m:nqrRuvVwWx:X OPT; do
		case ${OPT} in
		a)	opt_all=YES ;;
		b)	opt_keep_backup=YES ;;
		B)	opt_backup=NO ;;
		c)	opt_beforeclean=YES ;;
		C)	opt_afterclean=YES ;;
		h)	usage ;;
		f)	opt_force=YES ;;
		i)	opt_interactive=YES ;;
		I)	opt_use_index=YES ;;
		l)	opt_log_output=${OPTARG} ;;
		m)	opt_make_args="${opt_make_args} ${OPTARG}" ;;
		n)	opt_noexecute=YES ;;
		q)	opt_noconf=YES ;;
		r)	opt_required_by=YES ;;
		R)	opt_depends=YES ;;
		u)	opt_preserve_libs=NO ;;
		v)	opt_verbose=YES ;;
		V)	opt_version=YES ;;
		w)	opt_beforeclean=NO ;;
		W)	opt_afterclean=NO ;;
		x)	opt_exclude="${opt_exclude} ${OPTARG}" ;;
		X)	opt_debug=$((${opt_debug}+1)) ;;
		\?)	usage 1 ;;
		esac
	done

	OPTC=$((${OPTIND}-1))
}

parse_args() {
	local port_name file

	do_replace=
	while [ $# -gt 0 ]; do
		case $1 in
		*.t[bg]z|?*=${PORTSDIR}/?*/*[!/])
			file=${1##*=}
			expand_path '$file'
			file_exist "${file}" || return 1 ;;
		*=*)
			warn "Syntax error: $1"
			return 1 ;;
		esac

		case $1 in
		*=*)
			opt_replace="${opt_replace} ${1%=*}=${file}"
			do_replace="${do_replace} ${1%=*}" ;;
		*.t[bg]z)
			set_port_name_from_package "${file}" || return 1
			opt_replace="${opt_replace} ${port_name%-*}=${file}"
			do_replace="${do_replace} ${port_name%-*}" ;;
		*)
			do_replace="${do_replace} $1" ;;
		esac
		shift
	done
}

#ifdef WITH_CONFIG
parse_config() {
	local LINE q lineno in_array var val hash_key hash_val

	[ -r "$1" ] || return 0

	q="['\"]"
	lineno=0
	in_array=

	while read LINE; do
		lineno=$((${lineno}+1))

		case ${LINE} in
		''|\#*) continue ;;
		esac

		if [ -n "${in_array}" ]; then
			case ${LINE} in
			*[\)}]|*[\)}]\;) in_array=; continue ;;
			*$q?*$q*)
				hash_key=${LINE#*$q}
				hash_key=${hash_key%%$q*} ;;
			*)
				warn "Syntax error at line ${lineno}: ${LINE}"
				return 1 ;;
			esac

			case ${LINE} in
			*=\>*$q*$q*) # hash
				hash_val=${LINE#*=>*$q}
				hash_val=${hash_val%$q*}
				eval ${var}=\"\${${var}:+\$${var}'
'}\${hash_key}=\${hash_val}\" ;;
			*) # array
				eval ${var}=\"\${${var}-} \${hash_key}\" ;;
			esac
			continue
		fi

		case ${LINE} in
		?*=*)	var=${LINE%%=*}; val=${LINE#*=} ;;
		*)	var="syntax-error" ;;
		esac

		case ${var} in
		*[!0-9A-Za-z_]*|[0-9]*)
			warn "Syntax error at line ${lineno}: ${LINE}"
			return 1 ;;
		esac

		case ${val} in
		*[\({])	eval ${var}=; in_array=1 ;;
		*)	eval ${var}=${val} ;;
		esac
	done < "$1"
}

parse_exec_script() {
	local IFS hash key command

	IFS='
'
	eval hash=\${$1-}
	for key in ${hash}; do
		if pkg_match "${key%%=*}"; then
			command=${key#*=}
			info "Executing a $1 command: ${command}"
			( eval "${command}" ) || :
		fi
	done
}
#endif

set_port_name() {
	port_name=
	if is_yes ${opt_use_index}; then
		# XXX: requires GNU grep.
		port_name=$(grep --mmap -m 1 -o "^[^|]*|[^|]*${1#${PORTSDIR}/}|" "${PORTSDIR}/INDEX")
		port_name=${port_name%%\|*}
	elif [ -e "$1/Makefile" ] && cd "$1"; then
#ifdef WITH_PKGSRC
		port_name=$(${MAKE_CMD} -V '${PKGNAME}')
#else
#ifdef WITH_OPENBSD
		port_name=$(${MAKE_CMD} show=FULLPKGNAME\${SUBPACKAGE})
#else
		port_name=$(${MAKE_CMD} -VPKGNAME)
#endif
#endif
	fi

	if [ -z "${port_name}" ]; then
		return 1
	fi
}

set_port_name_from_package() {
	local opt

	opt=
	case $1 in
	*.gz|*.tgz)	opt=z ;;
	*.bz2|*.tbz)	opt=j ;;
	esac

#ifdef WITH_OPENBSD
	port_name=`tar x${opt}Of "$1" "+CONTENTS" |
#else
	port_name=`tar x${opt}Of "$1" --fast-read "+CONTENTS" |
#endif
	while read LINE; do
		case ${LINE} in
		@name\ *)
			echo "${LINE#@name }"
			break ;;
		[!@]*)  break ;;
		esac
	done`

	if [ -z "${port_name}" ]; then
		warn "Unable to open contents file: $1 (not a package?)"
		return 1
	fi
}

set_pkg_name() {
	pkg_name=$1
}

set_pkg_origin() {
	local LINE

	pkg_origin=
#ifdef WITH_PKGSRC
	if [ -r "${PKG_DBDIR}/$1/+BUILD_INFO" ]; then
		while read LINE; do
			case ${LINE} in
			PKGPATH=*)
				pkg_origin=${LINE#PKGPATH=}
				break ;;
			esac
		done < "${PKG_DBDIR}/$1/+BUILD_INFO"
	fi
#else
	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
		while read LINE; do
			case ${LINE} in
#ifdef WITH_OPENBSD
			@comment*\ subdir=*)
				LINE=${LINE##*subdir=}
				pkg_origin=${LINE%% *}
				break ;;
#else
			@comment\ ORIGIN:*)
				pkg_origin=${LINE#@comment ORIGIN:}
				break ;;
#endif
			[!@]*)	break ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
	fi
#endif
}

set_pkg_pkgdir() {
	pkg_pkgdir=${PKG_DBDIR}/$1
}

set_pkg_portdir() {
	pkg_portdir=${PORTSDIR}/$1
}

correct_pkg() {
	if [ -z "${pkg_origin}" ]; then
		if [ -d "${pkg_pkgdir}" ]; then
			warn "'${pkg_name}' has no origin recorded."
		else
			warn "'${pkg_name}' is not installed. (broken dependencies?)"
		fi
		return 1
	elif [ ! -d "${pkg_portdir}" ]; then
		trace_moved "${pkg_origin}" || return 1
	fi
}

pkg_glob() {
	local - pattern name contents

	set +f
	while [ $# -gt 0 ]; do
		pattern=${1#${PKG_DBDIR}/}

		case ${pattern} in
		''|*[\<\>/]*)	shift; continue ;;
		*\**|*-pl[0-9]*|*-[0-9]*[0-9.][a-z]|*-[0-9]*[0-9]) ;;
		*)		pattern="${pattern}-[0-9]*[0-9a-z]" ;;
		esac

		for contents in ${PKG_DBDIR}/${pattern}/+CONTENTS; do
			if [ -e "${contents}" ]; then
				name=${contents%/+CONTENTS}

				if is_yes ${opt_depends}; then
					pkg_depends "${name##*/}"
				fi

				echo "${name##*/}"

				if is_yes ${opt_required_by}; then
					pkg_required_by "${name##*/}"
				fi
			else
				warn "No such installed package: $1"
			fi
		done
		shift
	done
}

pkg_depends() {
	local dep xvar

	case ${2-} in
	\$*)	xvar=${2#$}; eval ${xvar}= ;;
	*)	xvar= ;;
	esac

#ifdef WITH_OPENBSD
	if [ -r "${PKG_DBDIR}/$1/+REQUIRING" ]; then
		while read dep; do
			if [ -n "${xvar}" ]; then
				eval ${xvar}=\"\$${xvar} ${dep}\"
			else
				echo "${dep}"
			fi
		done < "${PKG_DBDIR}/$1/+REQUIRING"
	fi
#else
	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
		while read dep; do
			case ${dep} in
#ifdef WITH_PKGSRC
			@pkgdep\ *)
				if [ -n "${xvar}" ]; then
					eval ${xvar}=\"\$${xvar} $(pkg_info -e "${dep#@pkgdep }")\"
				else
					pkg_info -e "${dep#@pkgdep }"
				fi ;;
#else
			@pkgdep\ *)
				if [ -n "${xvar}" ]; then
					eval ${xvar}=\"\$${xvar} ${dep#@pkgdep }\"
				else
					echo "${dep#@pkgdep }"
				fi ;;
#endif
			[!@]*)	break ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
	fi
#endif
}

pkg_required_by() {
	local req xvar
	
	case ${2-} in
	\$*)	xvar=${2#$}; eval ${xvar}= ;;
	*)	xvar= ;;
	esac

	if [ -r "${PKG_DBDIR}/$1/+REQUIRED_BY" ]; then
		while read req; do
			if [ -n "${xvar}" ]; then
				eval ${xvar}=\"\$${xvar} ${req}\"
			else
				echo "${req}"
			fi
		done < "${PKG_DBDIR}/$1/+REQUIRED_BY"
	fi
}

pkg_sort() {
	local leaves checked pkg_node pkg depends

	leaves=
	checked=
	pkg_node=" $@ "

	while [ $# -gt 0 ]; do
		case ${checked}${leaves} in
		*" $1 "*) shift; continue ;;
		esac

		if [ $# -gt 1 ] && [ -s "${PKG_DBDIR}/$1/+REQUIRED_BY" ]; then
			pkg_depends "$1" '$depends'

			for pkg in ${depends}; do
				case ${checked} in
				*" ${pkg} "*) continue ;;
				esac
				case ${pkg_node} in
				*" ${pkg} "*) checked="${checked:- }${pkg} " ;;
				esac
			done

			checked="${checked:- }$1 "
		else
			leaves="${leaves:- }$1 "
		fi
		shift
	done

	echo ${checked} ${leaves}
}

pkg_match() {
	local i

	for i in ${1+"$@"}; do
		case "|${pkg_name}|${pkg_name%-*}|${pkg_origin}|" in
		*\|$i\|*) return 0 ;;
		esac
	done

	return 1
}

#ifndef WITH_PKGSRC
#ifndef WITH_OPENBSD
guess_up_to_date() {
	local IFS all pkg

	all=" $@ "
	do_replace=
	IFS='
'
	for pkg in $(pkg_version -qvIl\< "${PORTSDIR}/INDEX"); do
		case ${all} in
		*" ${pkg%% *} "*) do_replace="${do_replace} ${pkg%% *}" ;;
		esac
	done
}
#endif
#endif

try() {
	local exit

	if [ ${opt_debug} -ge 2 ]; then
		warn "$@"
	elif "$@"; exit=$?; [ ${exit} != 0 ]; then
		warn "Command failed (exit code ${exit}): $@"
		return ${exit}
	fi
}

build_port() {
	local msg

#ifdef WITH_OPENBSD
	msg="${MAKE_CMD#env }"
	msg="${MAKE_CMD% ${MAKE}}"
#else
	msg="${MAKE_CMD#${MAKE} }"
#endif

	info "Building '$1'${msg:+ with make flags:${msg}}"
	cd "$1" || return 1

#ifdef WITH_CONFIG
	parse_exec_script BEFOREBUILD

#endif
	if is_yes ${opt_beforeclean}; then
		clean_port "$1" || return 1
	fi

	try ${MAKE_CMD} || return 1
}

install_port() {
	local install_args

	install_args=
	is_yes ${opt_force} && install_args="-DFORCE_PKG_REGISTER"

	info "Installing '$1'"
	cd "$1" || return 1

	try ${MAKE_CMD} ${install_args} install || return 1

	if is_yes ${opt_afterclean}; then
		clean_port "$1" || return 1
	fi
#ifdef WITH_CONFIG

	parse_exec_script AFTERINSTALL
#endif
}

install_package() {
	local install_args

	install_args=
	is_yes ${opt_force} && install_args="-f"
	is_yes ${opt_verbose} && install_args="${install_args} -v"

	info "Installing '$1'"

	try pkg_add ${install_args} "$1" || return 1
#ifdef WITH_CONFIG

	parse_exec_script AFTERINSTALL
#endif
}

deinstall_package() {
	local deinstall_args

	deinstall_args=
	is_yes ${opt_force} && deinstall_args="-f"
	is_yes ${opt_verbose} && deinstall_args="${deinstall_args} -v"

	info "Deinstalling '$1'"
#ifdef WITH_CONFIG
	parse_exec_script BEFOREDEINSTALL
#endif

	if [ ${opt_debug} -lt 2 ] && [ ! -w "${PKG_DBDIR}" ]; then
		warn "You do not own ${PKG_DBDIR}."
		return 1
	fi

	try pkg_delete ${deinstall_args} "$1" || return 1
}

clean_port() {
	local clean_args

#ifdef WITH_PKGSRC
	clean_args="CLEANDEPENDS=yes"
#else
#ifdef WITH_OPENBSD
	clean_args="CLEANDEPENDS=yes"
#else
	clean_args=
#endif
#endif

	info "Cleaning '$1'"
	cd "$1" || return 1

	try ${MAKE_CMD} ${clean_args} clean || return 1
}

#ifdef WITH_CONFIG
fetch_package() {
	local pkg_uri prog subdir OPSYS OS_VERSION OS_REVISION OS_MAJOR ARCH

	info "Fetching the binary package for '$1'"

	set -- "$1" "${2:-${cwd}}" ${UNAME=$(uname -smr)}

	OPSYS=$3
	OS_VERSION=$4
	OS_REVISION=${OS_VERSION%%-*}
	OS_MAJOR=${OS_REVISION%%.*}
	ARCH=$5

	if [ ! -w "$2" ]; then
		warn "You do not own $2."
		return 1
	fi

	if [ ${PACKAGESITE:+1} ]; then
		pkg_uri=${PACKAGESITE}
	else
#ifdef WITH_PKGSRC
		pkg_uri="${PACKAGEROOT:-ftp://ftp.NetBSD.org}/pub/pkgsrc/packages/${OPSYS}-${OS_REVISION}/${ARCH}/All/"
#else
#ifdef WITH_OPENBSD
		pkg_uri="${PACKAGEROOT:-ftp://ftp.OpenBSD.org}/pub/OpenBSD/${OS_REVISION}/packages/${ARCH}/"
#else
		case ${OS_VERSION} in
		*-CURRENT)	subdir="${OS_MAJOR}-current" ;;
		*-RELEASE*)	subdir="${OS_REVISION}-release" ;;
		*)		subdir="${OS_MAJOR}-stable" ;;
		esac

		pkg_uri="${PACKAGEROOT:-ftp://ftp.FreeBSD.org}/pub/FreeBSD/ports/${ARCH}/packages-${subdir}/All/"
#endif
#endif
	fi
	pkg_uri="${pkg_uri}$1${PKG_SUFX}"

	cd "$2" || return 1

	for prog in "/usr/bin/fetch -Ap" /usr/bin/ftp /usr/local/bin/ftp; do
		[ -x "${prog%% *}" ] || continue

		if try ${prog} "${pkg_uri}" && [ -e "$1${PKG_SUFX}" ]; then
			return 0
		else
			return 1
		fi
	done
}

find_package() {
	use_package="${PKGREPOSITORY}/${1}${PKG_SUFX}"

	if ! {
		[ -e "${use_package}" ] || \
		fetch_package "$1" "${PKGREPOSITORY}"
	}; then
		warn "Couldn't find the latest binary package."

		if is_yes ${opt_force}; then
			use_package=
			warn "Using the source files instead of a binary package."
		else
			warn "Please put a binary package (${use_package##*/})" \
			"in '${PKGREPOSITORY}'. (or specify -f to force)"
			return 1
		fi
	fi
}
#endif

#ifdef WITH_PKGSRC
pkg_backup() {
	local pkg_pkgdir package plist create_args arg

	set_pkg_pkgdir "$1"
	package=$2
	create_args=

	if ! { cd "${pkg_pkgdir}" && file_exist "+CONTENTS" && \
		plist=$(mktemp -t "$1"); }; then
		return 1
	fi

	sed  -e '/^@blddep /d' -e '/^@comment [^[:space:]]*:/d' \
	     -e '/^@name /d' -e '/^@mtree /d' \
	     -e '/^@pkgcfl /d' -e '/^@pkgdep /d' \
		"+CONTENTS" > "${plist}" || return 1

	while read LINE; do
		case ${LINE} in
		@pkgcfl\ *)	create_args="${create_args} -C ${LINE#@pkgcfl }" ;;
		@pkgdep\ *)	create_args="${create_args} -P ${LINE#@pkgdep }" ;;
		[!@]*)	break ;;
		esac
	done < "+CONTENTS"

	for arg in \
		b:BUILD_VERSION B:BUILD_INFO c:COMMENT d:DESC \
		D:DISPLAY i:INSTALL k:DEINSTALL m:MTREE n:PRESERVE \
		r:REQUIRE s:SIZE_PKG S:SIZE_ALL; do

		if [ -e "+${arg#*:}" ]; then
			create_args="${create_args} -${arg%%:*} +${arg#*:}"
		fi
	done

	try pkg_create ${create_args} -lf "${plist}" "${package}" || {
		try rm "${plist}"; return 1; }

	try rm "${plist}" && cd "${cwd}"
}
#endif

backup_package() {
	if is_yes ${opt_backup} && [ ! -e "$2" ]; then
		info "Backing up the old version"
#ifdef WITH_PKGSRC
		try pkg_backup "$1" "$2" || return 1
#else
#ifdef WITH_OPENBSD
		if ! {
			cd "${2%/*}" && \
			try pkg_create -f "${PKG_DBDIR}/$1/+CONTENTS" && \
			cd "${cwd}"
		}; then
			return 1
		fi
#else
		try pkg_create -b "$1" "$2" || return 1
#endif
#endif
	fi
}

backup_file() {
	if [ -e "$1" ]; then
		info "Backing up the ${1##*/} file"
		try cp -f "$1" "$2" || return 1
	fi
}

restore_package() {
	if [ -e "$1" ]; then
		info "Restoring the old version"
		opt_force=YES install_package "$1" || return 1
	fi
}

restore_file() {
	if [ -e "$1" ] && [ ! -e "$2" ]; then
		info "Restoring the ${1##*/} file"
		try mv -f "$1" "$2" || return 1
	fi
}

process_package() {
	if is_yes ${opt_keep_backup} && [ -e "$1" ] && [ ! -e "${PKG_BACKUP_DIR}/${1##*/}" ]; then
		info "Keeping old package in '${PKG_BACKUP_DIR}'"
		make_dirs "${PKG_BACKUP_DIR}" || return 1
		try mv -f "$1" "${PKG_BACKUP_DIR}" || return 1
	fi
}

cleanup() {
	try rm -rf "$@" || warn "Couldn't remove '$@'."
}

trace_moved() {
#ifdef WITH_PKGSRC
	warn "'$1' has been moved or obsoleted."
	warn "See ${PORTSDIR}/doc/CHANGES-*."
	return 1
#else
#ifdef WITH_OPENBSD
	warn "'$1' has been moved or obsoleted."
	return 1
#else
	local IFS msg checked moved origin

	file_exist "${PORTSDIR}/MOVED" || return 1

	msg=
	origin=$1
	checked=
	IFS='|'

	while moved=$(grep "^$1|" "${PORTSDIR}/MOVED"); do
		set -- ${moved}

		if [ $# -ne 4 ]; then
			warn "'${PORTSDIR}/MOVED' has broken!"
			break
		elif [ -n "$2" ]; then
			case ${checked} in
			*"|$2|"*)
				warn "'${PORTSDIR}/MOVED' has broken!"
				break ;;
			esac

			checked="${checked:-|}$1|"
			msg="${msg:+${msg} }-> $2"
			set_pkg_portdir "$2"

			if [ -d "${pkg_portdir}" ]; then
				warn "'${origin}' has been moved. (${msg})"
				pkg_origin=$2
				return 0
			fi

			set -- "$2"
		else
			warn "'$1' has been removed from ports tree:"
			warn "\"$4\" ($3)"
			return 1
		fi
	done

	warn "'$1' has been moved or obsoleted."
	return 1
#endif
#endif
}

perform() {
	local MAKE_CMD pkg_name pkg_origin pkg_pkgdir pkg_portdir port_name use_package status log

	while [ $# -gt 0 ]; do
		log=
		status=

		if perform_preprocess "$1"; then
			if is_yes ${opt_version}; then
				perform_version
			elif ! perform_replace; then
				status=failed
				warn "Fix the problem and try again."
				echo
			fi
		else
			status=skipped
			warn "Skipping '$1'."
		fi

		add_log "${status:-ignored}" "$1" ${log:+" (${log})"}
		shift
	done
}

perform_preprocess() {
	local IFS i replace_with

	set_pkg_name "$1"
	set_pkg_pkgdir "$1"
	use_package=
	replace_with=

	for i in ${opt_replace}; do
		case "|${pkg_name}|${pkg_name%-*}|" in
		*\|${i%=*}\|*)
			replace_with=${i##*=}
			break ;;
		esac
	done

	case ${replace_with} in
	*.t[bg]z|'')
		set_pkg_origin "${pkg_name}" 
		[ -z "${replace_with}" ] || use_package=${replace_with} ;;
	*)	pkg_origin=${replace_with#${PORTSDIR}/} ;;
	esac

#ifdef WITH_CONFIG
	if ! is_yes ${opt_force} && pkg_match ${IGNORE}; then
		warn "'${pkg_name}' is ignored. (specify -f to force)"
		return 1
	fi

#endif
	if pkg_match ${opt_exclude}; then
		return 1
	fi

	if [ -n "${use_package}" ]; then
		set_port_name_from_package "${use_package}" || return 1
	else
		set_pkg_portdir "${pkg_origin}"

#ifdef WITH_OPENBSD
		MAKE_CMD="env ${opt_make_args}"
#else
		MAKE_CMD="${MAKE} ${opt_make_args}"
#endif
#ifdef WITH_CONFIG
		IFS='
'
		for i in ${MAKE_ARGS}; do
			if pkg_match "${i%%=*}"; then
				MAKE_CMD="${MAKE_CMD} ${i#*=}"
			fi
		done
		IFS=' '
#endif
#ifdef WITH_OPENBSD
		MAKE_CMD="${MAKE_CMD} ${MAKE}"
#endif

		correct_pkg || return 1
		set_port_name "${pkg_portdir}" || return 1
	fi
}

perform_replace() {
	local tmpdir old_package old_required_by preserved_files

	if [ "${pkg_name}" != "${port_name}" ]; then
		info "Replacing '${pkg_name}' with '${port_name}'"
	elif is_yes ${opt_force}; then
		info "Reinstalling '${pkg_name}'"
	else
		is_yes ${opt_verbose} && \
		warn "No need to replace '${pkg_name}'. (specify -f to force)"
		return 0
	fi

	if is_yes ${opt_noexecute}; then
		status=done
		return 0
	elif is_yes ${opt_interactive}; then
		prompt_yesno || return 0
	fi

#ifdef WITH_CONFIG
	if [ -z "${use_package}" ] && pkg_match ${USE_PKGS}; then
		find_package "${port_name}" || { log="package not found"; return 1; }
	fi

#endif
	if [ -z "${use_package}" ]; then
		build_port "${pkg_portdir}" || { log="build error"; return 1; }
	fi

	if [ ${opt_debug} -ge 2 ]; then
		tmpdir="${PKG_TMPDIR}/${pkg_name}.XXXXXX"
	else
		tmpdir=$(mktemp -d "${PKG_TMPDIR}/${pkg_name}.XXXXXX") || {
			log="backup error"; return 1; }
	fi

	old_package="${pkg_name}${PKG_SUFX}"

	if [ -e "${PKG_BACKUP_DIR}/${old_package}" ]; then
		info "Found a package of the old version"
		old_package="${PKG_BACKUP_DIR}/${old_package}"
	else
#ifdef WITH_OPENBSD
		# PKG_SUFX doesn't work, see src/usr.sbin/pkg_add/pkg_create.
		old_package="${tmpdir}/${pkg_name}.tgz"
#else
		old_package="${tmpdir}/${old_package}"
#endif
	fi

	old_required_by="${tmpdir}/+REQUIRED_BY"

	if ! {
		backup_package "${pkg_name}" "${old_package}" && \
		backup_file "${pkg_pkgdir}/+REQUIRED_BY" "${old_required_by}" && \
		preserve_shlibs "${pkg_name}"
	}; then
		log="backup error"
		cleanup "${tmpdir}"
		return 1
	fi

	if opt_force=YES deinstall_package "${pkg_name}"; then
		if {
			if [ -n "${use_package}" ]; then
				install_package "${use_package}"
			else
				install_port "${pkg_portdir}" && \
				opt_use_index=NO set_port_name "${pkg_portdir}"
			fi
		}; then
			status=done
			set_pkg_pkgdir "${port_name}"
		else
			log="install error"
			restore_package "${old_package}" || {
				warn "Failed to restore the old version," \
				"please reinstall '${old_package}' manually."
				return 1
			}
		fi
	else
		log="deinstall error"
	fi

	restore_file "${old_required_by}" "${pkg_pkgdir}/+REQUIRED_BY" || \
		warn "Failed to restore the +REQUIRED_BY file."
	process_package "${old_package}" || \
		warn "Failed to save the old package."
	clean_shlibs || \
		warn "Failed to clean the old shared libraries."
	cleanup "${tmpdir}"

	[ "${status}" = done ] || return 1

#ifndef WITH_OPENBSD
	if [ "${pkg_name}" != "${port_name}" ]; then
		info "Updating the dependencies"
		try update_dependencies "${pkg_name}" "${port_name}" || return 1
	fi
#endif
}

perform_version() {
	[ "${pkg_name}" != "${port_name}" ] || return 0

	status=done
	echo -n "${pkg_name} -> "

	if [ "${pkg_name%-*}" = "${port_name%-*}" ]; then
		echo "${port_name##*-}"
	else
		echo "${port_name}"
	fi
}

#ifndef WITH_OPENBSD
update_dependencies() {
	local oldname newname required_by dep

	oldname=$1
	newname=$2

	pkg_required_by "${newname}" '$required_by'
	for dep in ${required_by}; do
		if have_depend "${dep}" "${oldname%-*}"; then
			update_contents "${dep}" "${oldname%-*}-[^-]*" "${newname}" || return 1
		fi
	done
}

update_contents() {
	local contents obj

	contents=${PKG_DBDIR}/$1/+CONTENTS
	obj=${contents}.$$

	if is_yes ${opt_verbose}; then
		info "Updating $1: $2 -> $3"
	fi

	if sed "s/^@pkgdep $2\$/@pkgdep $3/" "${contents}" > "${obj}"; then
		try mv -f "${obj}" "${contents}"
	else
		return 1
	fi
}

have_depend() {
	local pkg dep

	if [ -r "${PKG_DBDIR}/$1/+CONTENTS" ]; then
		while read dep; do
			case ${dep} in
			@pkgdep\ $2[\<\>*]*|@pkgdep\ $2*\**|[!@]*)
				break ;;
			@pkgdep\ $2-*)
				dep=${dep#@pkgdep }
				[ "$2" = "${dep%-*}" ] && return 0 ;;
			esac
		done < "${PKG_DBDIR}/$1/+CONTENTS"
	fi

	return 1
}
#endif

preserve_shlibs() {
	local file

	is_yes ${opt_preserve_libs} || return 0

	preserved_files=
	for file in $(pkg_info -qL "$1"); do
		case ${file} in
		*.so.[0-9]*)
			case $(file -b "${file}") in
			*ELF*shared\ object*)
				preserved_files="${preserved_files} ${file}"
			esac ;;
		esac
	done

	if [ -n "${preserved_files}" ]; then
		info "Preserving the shared libraries"
		make_dirs "${PKGCOMPATDIR}" || return 1
		try cp -f ${preserved_files} "${PKGCOMPATDIR}" || return 1
	fi
}

clean_shlibs() {
	local delete_files file dest

	if ! is_yes ${opt_preserve_libs} || [ -z "${preserved_files}" ]; then
		return 0
	fi

	info "Cleaning the preserved shared libraries"

	delete_files=
	for file in ${preserved_files}; do
		dest=${PKGCOMPATDIR}/${file##*/}
		if [ -e "${file}" ]; then
			delete_files="${delete_files} ${dest}"
		else
			info "Keeping ${file} as ${dest}"
		fi
	done

	if [ -n "${delete_files}" ]; then
		try rm -f ${delete_files} || return 1
	fi
}

file_exist() {
	if [ ! -e "$1" ]; then
		warn "No such file or directory: $1"
		return 1
	fi
}

make_dirs() {
	while [ $# -gt 0 ]; do
		[ -d "$1" ] || try mkdir -p "$1" || return 1
		shift
	done
}

create_log() {
	[ -n "$1" ] || return 0

	opt_log_output=$1
	expand_path '$opt_log_output'

	true > "${opt_log_output}" || {
		warn "Failed to create the log."
		return 1; }
}

add_log() {
	local sign valid

	[ -n "${opt_log_output}" ] || return 0

	valid=
	for sign in ${log_format}; do
		case $1 in
		${sign#*:})
			valid=${sign%%:*}
			break ;;
		esac
	done; shift

	echo "${valid:-?} $@" >> "${opt_log_output}"
}

show_log() {
	local LINE i descr msg

	[ -n "${opt_log_output}" ] || return 0

	all_c=0
	descr=
	msg=

	for i in ${log_format}; do
		eval ${i#*:}_c=0
		descr="${descr:+${descr} / }$i"
	done

	info "Listing the results (${descr})"

	while read LINE; do
		all_c=$((${all_c}+1))
		for i in ${log_format}; do
			case ${LINE} in
			"${i%%:*} "*)
				eval ${i#*:}_c=\$\(\(\${${i#*:}_c}+1\)\)
				break ;;
			esac
		done
		echo "        ${LINE}"
	done < "${opt_log_output}"

	for i in ${log_format}; do
		eval msg=\"${msg:+${msg}, }\${${i#*:}_c} ${i#*:}\"
	done

	info "All tasks ${all_c}: ${msg%,*} and ${msg##*, }"
	info "Saved the results to '${opt_log_output}'"
}

expand_path() {
	local val xvar

	case $1 in
	\$*)	xvar=${1#$}; eval val=$1 ;;
	*)	xvar=; val=$1 ;;
	esac

	case ${val} in
	[!/]*)	val=${cwd%/}/${val}
	esac
	
	if [ -n "${xvar}" ]; then
		eval ${xvar}=\${val}
	else
		echo "${val}"
	fi 
}

main() {
	init_variables
	init_options
	parse_options ${1+"$@"}
	shift ${OPTC}

	if is_yes ${opt_all} || { is_yes ${opt_version} && [ $# -eq 0 ]; }; then
		opt_depends=NO
		opt_required_by=NO
		set -- '*'
	elif [ $# -eq 0 ]; then
		usage
	fi

#ifdef WITH_CONFIG
	if ! is_yes ${opt_noconf}; then
		if ! parse_config "/usr/local/etc/pkg_replace.conf"; then
			warn "Fatal error in /usr/local/etc/pkg_replace.conf."
			exit 1
		fi
	fi
#endif
	compat_variables

	if is_yes ${opt_use_index} && ! file_exist "${PORTSDIR}/INDEX" ; then
		warn "Couldn't use -I option."
		usage 1
	fi

	parse_args ${1+"$@"} || usage 1
	set -- $(pkg_glob ${do_replace})

	if [ $# -gt 1 ]; then
#ifndef WITH_PKGSRC
#ifndef WITH_OPENBSD
		if is_yes ${opt_use_index} && ! is_yes ${opt_force}; then
			guess_up_to_date "$@"
			set -- ${do_replace}

			if [ $# -eq 0 ]; then
				warn "No need to replace packages. (specify -f to force)"
				exit 0
			fi
		fi
#endif
#endif
		set -- $(pkg_sort "$@")
	fi

	unset do_replace
	if [ ${opt_debug} -eq 1 ]; then
		while [ $# -gt 0 ]; do
			echo "${0##*/} $1"
			shift
		done
	elif [ $# -gt 0 ]; then
		create_log "${opt_log_output}" || usage 1
		perform ${1+"$@"}
		show_log
	fi

	exit 0
}

IFS=' 
'
main ${1+"$@"}
