/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: mime.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	"strtool.h"
#include	"pairlist.h"
#include	"dlist.h"
#include	"heady.h"
#include	"mime.h"

struct mime
{
	dlist_t        *dlist;
};

void            mime_free(void *data);

mime_t         *
mime_create(void)
{
	mime_t         *obj = NULL;

	obj = (mime_t *) calloc(1, sizeof(mime_t));
	if (!obj)
	{
		goto ERROR;
	}
	obj->dlist = dlist_create();
	if (!obj->dlist)
	{
		goto ERROR;
	}
	return (obj);
ERROR:
	mime_destroy(obj);
	return (NULL);
}

void 
mime_destroy(mime_t * obj)
{
	if (obj)
	{
		dlist_destroy(obj->dlist, mime_free);

		free(obj);
	}
}

void 
mime_free(void *data)
{
	heady_destroy((heady_t *) data);
}

int 
mime_new_data(mime_t * obj)
{
	heady_t        *data = NULL;
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	data = heady_create();
	if (!data)
	{
		goto ERROR;
	}
	err = dlist_push(obj->dlist, (void *) data);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_del_data(mime_t * obj, const int offset)
{
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	err = dlist_del_data_by_offset(obj->dlist, offset, mime_free);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_pop_header(
		mime_t * obj,
		const int offset,
		char **name,
		char **value)
{
	char           *tmp_name = NULL;
	char           *tmp_value = NULL;
	int             err;

	heady_t        *data = NULL;

	if (!obj || offset < 0 || !name || !value)
	{
		goto ERROR;
	}
	data = (heady_t *) dlist_get_data_by_offset(obj->dlist, offset);
	if (!data)
	{
		goto ERROR;
	}
	err = heady_pop_header(data, &tmp_name, &tmp_value);
	if (err)
	{
		goto ERROR;
	}
	*name = tmp_name;
	*value = tmp_value;

	return (0);
ERROR:
	free(tmp_name);
	free(tmp_value);
	*name = NULL;
	*value = NULL;
	return (1);
}

int 
mime_set_header(
		mime_t * obj,
		const int offset,
		const char *name,
		const char *value)
{
	heady_t        *data = NULL;
	int             err;

	if (!obj || strtool_isempty(name) || strtool_isempty(value))
	{
		goto ERROR;
	}
	data = (heady_t *) dlist_get_data_by_offset(obj->dlist, offset);
	if (!data || heady_refer_header_by_name(data, name))
	{
		goto ERROR;
	}
	err = heady_add_header(data, name, value);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_set_type(mime_t * obj, const int offset, const char *type)
{
	int             err;

	if (!obj || strtool_isempty(type))
	{
		goto ERROR;
	}
	err = mime_set_header(obj, offset, "Content-Type", type);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_set_encoding(mime_t * obj, const int offset, const int encoding)
{
	char           *tmp = NULL;
	int             err;

	if (!obj)
	{
		goto ERROR;
	}
	switch (encoding)
	{
	case MIME_ENCODING_7BIT:
		tmp = "7bit";
		break;
	case MIME_ENCODING_8BIT:
		tmp = "8bit";
		break;
	case MIME_ENCODING_BINARY:
		tmp = "binary";
		break;
	case MIME_ENCODING_QUOTED:
		tmp = "quoted-printable";
		break;
	case MIME_ENCODING_BASE64:
		tmp = "base64";
		break;
	default:
		goto ERROR;
	}

	err = mime_set_header(obj, offset, "Content-Transfer-Encoding", tmp);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_disposition(mime_t * obj, const int offset, const char *disposition)
{
	int             err;

	if (!obj || strtool_isempty(disposition))
	{
		goto ERROR;
	}
	err = mime_set_header(obj, offset, "Content-Disposition", disposition);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

int 
mime_set_body(mime_t * obj, const int offset, const char *body)
{
	heady_t        *data = NULL;
	int             err;

	if (!obj || strtool_isempty(body))
	{
		goto ERROR;
	}
	data = (heady_t *) dlist_get_data_by_offset(obj->dlist, offset);
	if (!data)
	{
		goto ERROR;
	}
	err = heady_set_body(data, body);
	if (err)
	{
		goto ERROR;
	}
	return (0);
ERROR:
	return (1);
}

char           *
mime_to_str(const mime_t * obj, const char *boundary, const char *eol)
{
	heady_t        *data = NULL;
	char           *dst = NULL;
	char           *tmp = NULL;
	int             n, len;

	if (!obj)
	{
		goto ERROR;
	}
	len = mime_len(obj);

	for (n = 0; n < len; n++)
	{
		data = (heady_t *) dlist_get_data_by_offset(obj->dlist, n);

		tmp = heady_to_str(data, eol);

		strtool_cat(&dst, tmp);

		if (boundary)
		{
			strtool_cat(&dst, boundary);
			if (n + 1 < len)
			{
				strtool_cat(&dst, eol);
			}
		}
		free(tmp);
		tmp = NULL;
	}

	if (dst && boundary)
	{
		strtool_cat(&dst, "--");
		strtool_cat(&dst, eol);
	}
	return (dst);
ERROR:
	return (NULL);
}

int 
mime_len(const mime_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	return (dlist_len(obj->dlist));
ERROR:
	return (0);
}
