/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: ini.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	"dlist.h"
#include	"filetool.h"
#include	"strtool.h"
#include	"strlist.h"
#include	"pairlist.h"
#include	"ini.h"

struct section
{
	char           *name;
	pairlist_t     *data;
};

struct ini
{
	dlist_t        *dlist;
};

struct section *ini_section_calloc(void);
void            ini_section_free(void *data);
void            ini_parse_remove_comment(char **line, char comment);
int             ini_parse(ini_t * ini, const char *data, const char *eol, const char *delim, const char comment);
int             ini_parse_core(pairlist_t * data, const char *delim, const char *line);
int             ini_parse_is_section(char *line);
int             ini_parse_is_new_section(const ini_t * ini, const char *name);

int 
ini_unload(ini_t * obj)
{
	if (!obj)
	{
		goto ERROR;
	}
	dlist_destroy(obj->dlist, ini_section_free);

	free(obj);

	return (0);
ERROR:
	return (1);
}

ini_t          *
ini_load(
	 const char *file,
	 const char *eol,
	 const char *delim,
	 const char comment)
{
	ini_t          *ini = NULL;
	char           *data = NULL;
	int             err;

	if (strtool_isempty(file) || strtool_isempty(eol))
	{
		goto ERROR;
	}
	filetool_read_all(file, &data);
	if (!data)
	{
		goto ERROR;
	}
	ini = (ini_t *) calloc(1, sizeof(ini_t));
	if (!ini)
	{
		goto ERROR;
	}
	ini->dlist = dlist_create();
	if (!ini->dlist)
	{
		goto ERROR;
	}
	err = ini_parse(ini, data, eol, delim, comment);
	if (err)
	{
		goto ERROR;
	}
	free(data);

	return (ini);
ERROR:
	ini_unload(ini);
	free(data);
	return (NULL);
}

struct section *
ini_section_calloc(void)
{
	struct section *sec = NULL;

	sec = (struct section *) calloc(1, sizeof(struct section));
	if (!sec)
	{
		goto ERROR;
	}
	sec->name = NULL;
	sec->data = NULL;

	return (sec);
ERROR:
	return (NULL);
}

void 
ini_section_free(void *data)
{
	struct section *sec = (struct section *) data;

	if (sec)
	{
		free(sec->name);
		pairlist_destroy(sec->data);
		free(sec);
	}
}

int 
ini_add_section(ini_t * ini, const char *name)
{
	struct section *sec = NULL;

	if (!ini || strtool_isempty(name))
	{
		goto ERROR;
	}
	if (ini_is_exist_section(ini, name))
	{
		goto ERROR;
	}
	sec = ini_section_calloc();
	if (!sec)
	{
		goto ERROR;
	}
	sec->name = strdup(name);
	sec->data = pairlist_create();

	dlist_push(ini->dlist, sec);

	return (0);
ERROR:
	return (1);
}

int 
ini_is_exist_section(const ini_t * ini, const char *name)
{
	int             n, len;
	struct section *sec;

	if (ini)
	{
		len = dlist_len(ini->dlist);

		for (n = 0; n < len; n++)
		{
			sec = (struct section *) dlist_get_data_by_offset(ini->dlist, n);

			if (sec && sec->name && strcmp(sec->name, name) == 0)
			{
				return (1);
			}
		}
	}
	return (0);
}

int 
ini_add_data(
	     ini_t * ini,
	     const char *section,
	     const char *name,
	     const char *value)
{
	pairlist_t     *data;
	int             err;

	if (!ini)
	{
		goto ERROR;
	}
	data = ini_refer_data_by_section(ini, section);
	if (!data)
	{
		err = ini_add_section(ini, section);
		if (err)
		{
			goto ERROR;
		}
		data = ini_refer_data_by_section(ini, section);
		if (!data)
		{
			goto ERROR;
		}
	}
	if (pairlist_is_exist(data, name))
	{
		err = pairlist_replace_value_by_name(data, name, value);
	} else
	{
		err = pairlist_add(data, name, value);
	}

	return (err);
ERROR:
	return (1);
}

int 
ini_parse(
	  ini_t * ini,
	  const char *data,
	  const char *eol,
	  const char *delim,
	  const char comment)
{
	strlist_t      *lines = NULL;
	pairlist_t     *current = NULL;
	struct section *section = NULL;
	char           *line;
	int             n, len;

	lines = strtool_split(data, eol, 0);

	len = strlist_len(lines);

	if (!lines || len == 0)
	{
		goto ERROR;
	}
	section = ini_section_calloc();
	section->name = NULL;
	section->data = pairlist_create();
	dlist_push(ini->dlist, section);
	current = section->data;

	for (n = 0; n < len; n++)
	{
		line = strlist_get(lines, n);

		if (comment)
		{
			ini_parse_remove_comment(&line, comment);
		}
		strtool_strip(&line);

		if (strtool_isempty(line))
		{
			free(line);
			continue;
		}
		if (ini_parse_is_section(line))
		{
			if (ini_parse_is_new_section(ini, line))
			{
				section = ini_section_calloc();
				section->name = strdup(line);
				section->data = pairlist_create();
				dlist_push(ini->dlist, section);
				current = section->data;
			} else
			{
				current = ini_refer_data_by_section(ini, line);
			}
		} else
		{
			ini_parse_core(current, delim, line);
		}

		free(line);
	}

	strlist_destroy(lines);

	return (0);
ERROR:
	strlist_destroy(lines);
	return (1);
}

int 
ini_parse_core(pairlist_t * data, const char *delim, const char *line)
{
	strlist_t      *nv = NULL;
	char           *name = NULL;
	char           *value = NULL;

	nv = strtool_split(line, delim, 1);
	if (!nv || strlist_len(nv) != 2)
	{
		goto ERROR;
	}
	name = strlist_get(nv, 0);
	value = strlist_get(nv, 1);

	if (!name || !value)
	{
		goto ERROR;
	}
	strtool_strip(&name);
	strtool_strip(&value);

	pairlist_push(data, name, value);

	free(name);
	free(value);
	strlist_destroy(nv);

	return (0);
ERROR:
	free(name);
	free(value);
	return (1);
}

int 
ini_parse_is_section(char *line)
{
	if (line[0] == '[' && line[strlen(line) - 1] == ']')
	{
		while (*(line + 1) != ']')
		{
			*line = *(line + 1);
			line++;
		}

		*line = '\0';

		return (1);
	}
	return (0);
}

int 
ini_parse_is_new_section(const ini_t * ini, const char *name)
{
	int             n, len;
	struct section *tmp;

	if (ini)
	{
		len = dlist_len(ini->dlist);

		for (n = 0; n < len; n++)
		{
			tmp = dlist_get_data_by_offset(ini->dlist, n);
			if (tmp && tmp->name && strcmp(tmp->name, name) == 0)
			{
				return (0);
			}
		}
	}
	return (1);
}

void 
ini_parse_remove_comment(char **line, char comment)
{
	char           *tmp;

	if (!line || strtool_isempty(*line) || !comment)
	{
		goto ERROR;
	}
	tmp = *line;

	while (*tmp)
	{
		if (*tmp == comment)
		{
			*tmp = '\0';
		}
		tmp++;
	}

ERROR:
	return;
}

pairlist_t     *
ini_refer_data_by_section(const ini_t * ini, const char *section)
{
	int             n, len;
	struct section *tmp;

	if (!ini)
	{
		goto ERROR;
	}
	len = dlist_len(ini->dlist);

	for (n = 0; n < len; n++)
	{
		tmp = dlist_get_data_by_offset(ini->dlist, n);

		if (tmp)
		{
			if (section && tmp->name)
			{
				if (strcmp(section, tmp->name) == 0)
				{
					return (tmp->data);
				}
			} else if ((!section || strtool_isempty(section)) && !tmp->name)
			{
				return (tmp->data);
			}
		}
	}

ERROR:
	return (NULL);
}

pairlist_t     *
ini_get_data_by_section(const ini_t * ini, const char *section)
{
	pairlist_t     *tmp = NULL;

	tmp = ini_refer_data_by_section(ini, section);
	if (!tmp)
	{
		goto ERROR;
	}
	return (pairlist_dup(tmp));
ERROR:
	return (NULL);
}

char           *
ini_refer_value_by_name(
			const ini_t * ini,
			const char *section,
			const char *name)
{
	pairlist_t     *tmp;

	if (!ini || strtool_isempty(name))
	{
		goto ERROR;
	}
	tmp = ini_refer_data_by_section(ini, section);

	if (!tmp)
	{
		goto ERROR;
	}
	return (pairlist_refer_value_by_name(tmp, name));
ERROR:
	return (NULL);
}

char           *
ini_get_value_by_name(
		      const ini_t * ini,
		      const char *section,
		      const char *name)
{
	char           *tmp;

	tmp = ini_refer_value_by_name(ini, section, name);

	if (strtool_isempty(tmp))
	{
		goto ERROR;
	}
	return (strdup(tmp));
ERROR:
	return (NULL);
}

int 
ini_save(const ini_t * ini, const char *file, const char *eol, const char *delim)
{
	FILE           *fp = NULL;
	struct section *sect = NULL;
	char           *tmp = NULL;
	char           *t_name, *t_value;
	int             n, len1;
	int             m, len2;

	if (!ini || strtool_isempty(file)
	    || strtool_isempty(eol) || strtool_isempty(delim))
	{
		goto ERROR;
	}
	fp = fopen(file, "w");
	if (!fp)
	{
		goto ERROR;
	}
	len1 = dlist_len(ini->dlist);

	for (n = 0; n < len1; n++)
	{
		sect = dlist_get_data_by_offset(ini->dlist, n);
		if (!sect)
		{
			goto ERROR;
		}
		if (sect->name)
		{
			tmp = strtool_sprintf("[%s]%s", sect->name, eol);

			fwrite(tmp, strlen(tmp), sizeof(char), fp);

			free(tmp);
		}
		len2 = pairlist_len(sect->data);
		for (m = 0; m < len2; m++)
		{
			pairlist_refer(sect->data, m, &t_name, &t_value);

			tmp = strtool_sprintf("%s%s%s%s",
					      t_name, delim, t_value, eol);

			fwrite(tmp, strlen(tmp), sizeof(char), fp);

			free(tmp);
		}
	}

	fclose(fp);

	return (0);
ERROR:
	if (fp)
	{
		fclose(fp);
	}
	return (1);
}
