/* -*- Mode: c; c-basic-offset: 8; Coding: utf-8-unix -*- ;; */
/* $Id: filetool.c 2 2008-05-27 07:44:27Z mtaneda $	 */

/*
 * Copyright 2008 The piranha Project. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE PIRANHA PROJECT ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE PIRANHA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the piranha Project.
 */

#include	<stdio.h>
#include	<string.h>
#include	<stdlib.h>
#include	<sys/types.h>
#include	<sys/stat.h>
#include	<sys/wait.h>
#include	<unistd.h>
#include	<fcntl.h>
#include	<errno.h>
#include	"strtool.h"
#include	"strlist.h"
#include	"pairlist.h"
#include	"filetool.h"

int             filetool_is(const char *file, const int mode);

int 
filetool_read_all(const char *file, char **dst)
{
	struct stat     status;
	char           *tmp = NULL;
	char           *ptr;
	int             err, fd, ret;

	if (strtool_isempty(file) || !dst)
	{
		goto ERROR;
	}
	err = stat(file, &status);

	if (err)
	{
		goto ERROR;
	}
	tmp = (char *) calloc(status.st_size + 1, sizeof(char));

	if (!tmp)
	{
		goto ERROR;
	}
	fd = open(file, O_RDONLY);

	if (fd == -1)
	{
		goto ERROR;
	}
	for (ptr = tmp; (ptr - tmp) < status.st_size;)
	{
		ret = read(fd, ptr, status.st_size - (ptr - tmp));

		if (ret == -1)
		{
			if (errno == EINTR)
			{
				continue;
			}
			break;
		}
		ptr += ret;
	}

	close(fd);

	if ((ptr - tmp) < status.st_size)
	{
		goto ERROR;
	}
	*dst = tmp;

	return (status.st_size);
ERROR:
	free(tmp);
	*dst = NULL;
	return (0);
}

int 
filetool_is(const char *file, const int mode)
{
	int             ret;

	if (strtool_isempty(file))
	{
		goto ERROR;
	}
	ret = access(file, mode);

	if (ret == -1)
	{
		goto ERROR;
	}
	return (1);
ERROR:
	return (0);
}

int 
filetool_is_exist(const char *file)
{
	return (filetool_is(file, F_OK));
}

int 
filetool_is_readable(const char *file)
{
	return (filetool_is(file, R_OK));
}

int 
filetool_is_writeable(const char *file)
{
	return (filetool_is(file, W_OK));
}

int 
filetool_is_executable(const char *file)
{
	return (filetool_is(file, X_OK));
}

char           *
filetool_get_fname_from_path(const char *path)
{
	strlist_t      *tmp = NULL;
	char           *dst = NULL;

	tmp = strtool_split(path, "/", 0);
	if (!tmp)
	{
		goto ERROR;
	}
	dst = strlist_pop(tmp);

	strlist_destroy(tmp);

	return (dst);
ERROR:
	strlist_destroy(tmp);
	return (NULL);
}

int 
filetool_execute(const char *path, const strlist_t * argv, const pairlist_t * env)
{
	char          **args = NULL;
	char          **envs = NULL;
	char           *name, *value;
	int             n, len;
	int             ret, err = 1;

	if (!filetool_is_executable(path))
	{
		goto ERROR;
	}
	len = strlist_len(argv);

	args = (char **) calloc(len + 2, sizeof(char *));

	args[0] = strdup(path);

	if (len)
	{
		for (n = 0; n < len; n++)
		{
			args[n + 1] = strlist_get(argv, n);
		}
		args[n + 1] = NULL;
	} else
	{
		args[1] = NULL;
	}

	len = pairlist_len(env);

	if (len)
	{
		envs = (char **) calloc(len + 1, sizeof(char *));

		for (n = 0; n < len; n++)
		{
			pairlist_refer(env, n, &name, &value);

			envs[n] = strtool_sprintf("%s=%s", name, value);
		}
	}
	if (fork() == 0)
	{
		if (execve(path, args, envs) == -1)
		{
			exit(1);
		}
		exit(0);
	} else
	{
		wait(&ret);
	}

	fprintf(stderr, "%d\n", WEXITSTATUS(ret));

	if (WEXITSTATUS(ret) == 1)
	{
		goto ERROR;
	}
	err = 0;

ERROR:
	if (args)
	{
		for (n = 0; args[n]; n++)
		{
			free(args[n]);
		}
		free(args);
	}
	if (envs)
	{
		for (n = 0; envs[n]; n++)
		{
			free(envs[n]);
		}
		free(envs);
	}
	return (err);
}

int 
filetool_execute_pipe(
		      const char *path,
		      const strlist_t * argv,
		      const pairlist_t * env,
		      char **data)
{
	char          **args = NULL;
	char          **envs = NULL;
	char           *name, *value;
	char            buf[1024];
	int             pfd[2];
	int             ret, n, len, err = 1;

	if (!filetool_is_executable(path))
	{
		goto ERROR;
	}
	if (pipe(pfd) == -1)
	{
		goto ERROR;
	}
	if (fork() == 0)
	{
		close(pfd[0]);
		close(1);
		dup(pfd[1]);
		close(pfd[1]);
		len = strlist_len(argv);

		args = (char **) calloc(len + 1, sizeof(char *));
		args[0] = strdup(path);

		if (len)
		{
			for (n = 0; n < len; n++)
			{
				args[n + 1] = strlist_get(argv, n);
			}
			args[n + 1] = NULL;
		} else
		{
			args[1] = NULL;
		}

		len = pairlist_len(env);
		if (len)
		{
			envs = (char **) calloc(len + 1, sizeof(char *));
			for (n = 0; n < len; n++)
			{
				pairlist_refer(env, n, &name, &value);

				envs[n] = strtool_sprintf("%s=%s", name, value);
			}
		}
		if (execve(path, args, envs) == -1)
		{
			if (args)
			{
				for (n = 0; args[n]; n++)
				{
					free(args[n]);
				}
				free(args);
			}
			if (envs)
			{
				for (n = 0; envs[n]; n++)
				{
					free(envs[n]);
				}
				free(envs);
			}
			exit(1);
		}
		if (args)
		{
			for (n = 0; args[n]; n++)
			{
				free(args[n]);
			}
			free(args);
		}
		if (envs)
		{
			for (n = 0; envs[n]; n++)
			{
				free(envs[n]);
			}
			free(envs);
		}
		exit(0);
	}
	close(pfd[1]);

	memset(buf, 0, sizeof(buf));

	read(pfd[0], buf, sizeof(buf));

	close(pfd[0]);

	wait(&ret);

	fprintf(stderr, "%d\n", WEXITSTATUS(ret));

	if (WEXITSTATUS(ret) == 1)
	{
		goto ERROR;
	}
	err = 0;
ERROR:
	*data = strtool_dup(buf);
	return (err);
}
