#-> ViewContext
require 'uri'
require 'pathname'
require 'htmlsplit'
require 'rubyplus/boolattr'

require 'pinkyblog/config'
require 'pinkyblog/module-handler'
require 'pinkyblog/request'
require 'pinkyblog/menu'

module PinkyBlog
	# ビューコンテキスト（ビューに必要な情報のセット）
	class ViewContext
	
		attr_reader :config, :request
		bool_attr_reader :master_mode
		attr_accessor :snapshot_path, :path_refered_by_menu
		bool_attr_accessor :exporting
		attr_accessor :warnings

		def initialize(config, request, master_mode = false, snapshot_path = nil)
			@config = config
			@request = request
			@master_mode = master_mode
			@snapshot_path = Pathname.new(snapshot_path) if snapshot_path
			@path_refered_by_menu = nil
			@exporting = false
			
			@warnings = []
			
		end
		
		def snapshot_mode?
			(@snapshot_path ? true : false)
		end
		
		
		WDAY_TABLE = %w(日 月 火 水 木 金 土)

		def time2str(time)
			return sprintf("%d年%d月%d日（%s） %d:%02d",
			               time.year, time.month, time.day, WDAY_TABLE[time.wday], time.hour, time.min)
		end

		def date2str(time)
			return sprintf("%d年%d月%d日（%s）",
			               time.year, time.month, time.day, WDAY_TABLE[time.wday])
		end
		
		def time2str_short(time)
			return sprintf("%02d-%02d-%02d %02d:%02d",
			               time.year, time.month, time.day, time.hour, time.min)
		end
		
		def date2str_short(time)
			return sprintf("%02d-%02d-%02d", time.year, time.month, time.day)
		end



		def route_to(path, query = nil, fragment = nil)
			
			Util.normalize_path!(path)
			if exporting? then
				absolute_uri_to(path, query, fragment)
			elsif snapshot_mode? then
				case path
				when '/'
					to = Pathname.new("index.html")
				when '/entries'
					to = Pathname.new('./files/') + Util.page_number_to_file_name('entries', 1)
				else
					to = Pathname.new("files/#{path}.html")
				end
				
				return to.relative_path_from(@snapshot_path.dirname)
			else
				if path == '/' then
					dest = URI.parse(@request.script_name)
					if dest.path.empty? then
						dest.path = '/'
					end
				else
					dest_path = (@request.script_name + path)
					dest_path.squeeze!('/')
					dest = URI.parse(dest_path)
				end
					
				dest.query = query
				dest.fragment = fragment
				
				return dest
			end
		end
		alias uri_to route_to
		
		
		def absolute_uri_to(path, query = nil, fragment = nil)
			Util.normalize_path!(path)
			if @config.use_path_info then
				dest = URI.parse(@request.url)
				
				if path == '/' then
					destp = @request.script_name
				else
					destp = (@request.script_name + path)
				end
				destp.squeeze!('/')
				dest.path = destp
				
				dest.query = query
				dest.fragment = fragment
				return dest
			else
				re = script_uri.dup
				re.path = path
				re.query = (query ? "#{query}&path_info=#{path}" : "path_info=#{path}")
				re.fragment = fragment
				
				re
			end
		end

		def file_route_to(dest_path)
			if exporting? then
				return(script_uri + dest_path.to_s)
			elsif snapshot_mode? then
				return Pathname.new(dest_path.to_s).relative_path_from(@snapshot_path)
			else
				dest = script_uri + dest_path.to_s
				return URI(@request.url).route_to(dest)
			end
			
			
		end
		alias file_uri_to file_route_to
		
		def script_uri
			@script_uri ||= URI::HTTP.build({:host => @request.host, :port => @request.port, :path => @request.script_name})
		end

		
		
		
		def get_feed_url(file_name)
			url = script_uri
			url += (config.feed_dir_path + file_name).to_s
			return url
		end
		
		
		
		def self_url
			url = @request.script_url.dup
			if @config.use_path_info? then
				url.path += @request.path_info
			end
			url.query = @request.query
			return url

		end
		
		def etag_base
			"#{CORE_VERSION} #{@request.script_name} #{@config.etag_base} #{@master_mode ? '1' : '0'}"
		end
		
		def current_menu_item
			@config.menu.get_current_item(self)
		end
		
		
		def current_caption_on_menu(default = 'no title')
			if (item = current_menu_item) then
				item.caption
			else
				default
			end
		end
		
		def current_href_on_menu(default = '')
			if (item = current_menu_item) then
				item.build_href(self)
			else
				default
			end
		end
		
		def on_top_page?
			@request.normalized_path == '/'
		end
		
		def cookie_path
			if @request.script_name.empty? then
				'/'
			else
				path = @request.script_name
				if path =~ /\.cgi$/ then
					return File.dirname(path)
				else
					return path
				end
			end
		end

		def parse_ex_link(href)
			scheme_pattern = /\Aex\:(.+?)\:(.+)\Z/
			
			if href =~ scheme_pattern then
				scheme = $1
				us_target = $2
				target = Util.encode_url($2)
				
				case scheme
				when 'google'
					return "http://www.google.com/search?ie=utf-8&q=#{target}"
				when 'wikipedia'
					return "http://ja.wikipedia.org/wiki/#{target}"
				when 'tag'
					tags = us_target.split(/[ ]|　/)
					q = tags.map{|x| "tags=#{Util.encode_base64url(x)}"}.join("&")
					return route_to("entries", q).to_s
				when 'res'
					if snapshot_mode? then
						to = Pathname.new('files/res') + us_target
						return to.relative_path_from(@snapshot_path.dirname).to_s
					else
						return file_route_to(@config.res_dir_path + us_target).to_s
					end
				when 'entry'
					return route_to("entries/#{target}").to_s
				when 'menu'
					# メニュー項目扱い
					cmd_args = Shellwords.shellwords(us_target)
					cmd_name = cmd_args.shift
					
					begin
						item = MenuItem.create('dummy caption', cmd_name, cmd_args, 1)
					
						if item then
							return item.build_href(self)
						end
					rescue MenuItem::ArgumentParseError
					end
				end
			end
			
			# 特殊リンクでなかった場合
			return href
		end

		# HTMLをパースして、属性の置換などを行う
		def parse_html(html, briefing = false, entry_id = nil)
			elements = HTMLSplit.new(html).document


			url_keys = %W(href src action cite data archive codebase background classid longdesk usemap)

			briefing_mark_index = nil
			elements.each_with_index do |elem, index|
				if elem.is_a?(StartTag) || elem.is_a?(EmptyElementTag) then
					url_keys.each do |key|
						if elem[key] then
							elem[key] = parse_ex_link(elem[key])
						end
					end
					
					if elem.name == 'hr' then
						briefing_mark_index ||= index
					end
				end
				
				
			end # elements.each
			
			if briefing and briefing_mark_index then
				buf = elements.slice(0...(briefing_mark_index - 1))
				if entry_id then
					#buf << %Q|<div class="pinky-body-briefing"><p><a href="#{absolute_uri_to('/entries/' + entry_id)}" onclick="return load_entry(this, '#{route_to('/entries/' + entry_id)}');">（続きを読む）</a></p></div>|
					buf << %Q|\n\n<div class="pinky-body-briefing">\n  <p><a href="#{absolute_uri_to('/entries/' + entry_id)}">（続きを読む）</a></p>\n</div>\n\n|
				else
					buf << %Q|<div><p>（続きは省略されています）</p></div>|
				end
				
				
				return buf.to_s
			else
				return elements.to_s
			end
		end

		
	end
end
