#-> ViewContext
require 'uri'
require 'pathname'
require 'htmlsplit'

require 'pinkyblog/config'
require 'pinkyblog/module-handler'
require 'pinkyblog/request'

module PinkyBlog
	# ビューコンテキスト（ビューに必要な情報のセット）
	class ViewContext
	
		attr_reader :config, :module_handler, :request, :master_mode
		attr_accessor :snapshot_path
		attr_reader :warnings
		
		alias master_mode? master_mode

		def initialize(config, module_handler, request, master_mode = false, snapshot_path = nil)
			@config = config
			@module_handler = module_handler
			@request = request
			@master_mode = master_mode
			@snapshot_path = Pathname.new(snapshot_path) if snapshot_path
			
			@warnings = []
		end
		
		def snapshot_mode?
			(@snapshot_path ? true : false)
		end
		
		def message
			@request.get_cookie('message')
		end
		
		
		WDAY_TABLE = %w(日 月 火 水 木 金 土)

		def time2str(time)
			return sprintf("%d年%d月%d日（%s） %d:%02d",
			               time.year, time.month, time.day, WDAY_TABLE[time.wday], time.hour, time.min)
		end

		def date2str(time)
			return sprintf("%d年%d月%d日（%s）",
			               time.year, time.month, time.day, WDAY_TABLE[time.wday])
		end
		
		def time2str_short(time)
			return sprintf("%02d-%02d-%02d %02d:%02d",
			               time.year, time.month, time.day, time.hour, time.min)
		end


		def route_to(path, query = nil, fragment = nil)
			path = path.to_s
			if snapshot_mode? then
				to = (path == '/' ? Pathname.new("index.html") : Pathname.new("files/#{path}.html"))
				return to.relative_path_from(@snapshot_path.dirname)
			else
				@request.route_to(path, query, fragment)
				
			end
		end

		alias uri_to route_to
		alias url_to route_to
		alias get_cgi_path route_to # obsolete
		
		
		def absolute_uri_to(path, query = nil, fragment = nil)
			if @config.use_path_info then
				return @request.absolute_uri_to(path, query, fragment)
			else
				re = @script_uri.dup
				
				re.query = (query ? "#{query}&path_info=#{path}" : "path_info=#{path}")
				re.fragment = fragment if fragment
				
				re
			end
		end
		
		alias absolute_url_to absolute_uri_to
		alias get_cgi_url absolute_uri_to # obsolete

		
		
		
		def get_feed_url(file_name)
			url = @request.script_uri.dup
			url += (config.feed_dir_path + file_name).to_s
			return url
		end
		
		
		
		def self_url
			url = @request.script_url.dup
			if @config.use_path_info? then
				url.path += @request.path_info
			end
			url.query = @request.query
			return url

		end


		# HTMLをパースして、属性の置換などを行う
		def parse_html(html)
			doc = HTMLSplit.new(html).document

			scheme_pattern = /\Aex\:(.+?)\:(.+)\Z/

			url_keys = %W(href src action cite data archive codebase background classid longdesk usemap)

			doc.each do |elem|
				if elem.is_a?(StartTag) || elem.is_a?(EmptyElementTag) then
					url_keys.each do |key|
						case elem[key]
						when scheme_pattern
							scheme = $1
							us_target = $2
							target = PinkyBlog.encode_url($2)
							
							case scheme
							when 'google'
								elem[key] = "http://www.google.com/search?ie=utf-8&q=#{target}"
							when 'wikipedia'
								elem[key] = "http://ja.wikipedia.org/wiki/#{target}"
							when 'tag'
								tags = us_target.split(/[ ]|　/)
								q = tags.map{|x| "tags=#{PinkyBlog.encode_tag(x)}"}.join("&")
								elem[key] = get_cgi_path("entries", q).to_s
							when 'res'
								if snapshot_mode? then
									to = Pathname.new('files/res') + us_target
									elem[key] = to.relative_path_from(@snapshot_path.dirname).to_s
								else
									elem[key] = @request.file_route_to(@config.res_dir_path + us_target).to_s
								end
							when 'entry'
								elem[key] = route_to("entries/#{target}").to_s
							end
						end
					end
					
						
				end
				
				
			end
			
			return doc.to_s
		end

		
	end
end
