#-> Section
require 'erb'
require 'amrita/template'
require 'html-gen'

require 'pinkyblog/const'
require 'pinkyblog/function'
require 'pinkyblog/config'
require 'pinkyblog/module-handler'
require 'pinkyblog/request'
require 'pinkyblog/option-check'


module PinkyBlog
	# HTML上での一つ一つのセクションを表すクラス（通常、Screenから使う）
	class Section
		include Amrita::ExpandByMember
		self.extend OptionCheck

		def initialize(context, option_args = {})
			@context = context
			
			self.class.assert_fulfill_requirement(option_args)
			self.class.set_defaults(option_args)
			
			option_args.each_pair do |key, value|
				instance_variable_set("@#{key}", value)
			end
			
		end
		
		def config
			@context.config
		end
		
		def module_handler
			@context.module_handler
		end
		
		def request
			@context.request
		end
		
		def master_mode?
			@context.master_mode?
		end
		
		def snapshot_mode?
			@context.snapshot_mode?
		end

		
		
		def body
			"content."
		end
		
		def header
			header_text
		end
		
		def header_text
			"section title"
		end
		
		
		
		def footer
			nil
		end
		
		def header_text_id
			nil
		end
		
		def modori
			{:href => '#PAGETOP'}
		end
		
		
		def action_path
			request.script_url.path
		end
		
		def post_path
			PinkyBlog.get_html_entity(@context.get_cgi_path('post').to_s)
		end
		
		
		
		def expand_amrita_template(tmpl_path, model = {})
			path = config.lib_dir_path + 'pinkyblog/template/' + tmpl_path
			tmpl = Amrita::TemplateText.new(path.read.untaint)
			tmpl.amrita_id = 'amrita_src'
			buf = Amrita::SanitizedString.new
			tmpl.expand(buf, model)
			return buf
		end
		
		def expand_eruby_template(tmpl_path, binding)
			path = config.lib_dir_path + 'pinkyblog/template/' + tmpl_path
			tmpl = ERB.new(path.read.gsub(/\t/, '  ').untaint)
			return Amrita::SanitizedString.new(eval(tmpl.src, binding, path.basename.to_s))
			
		end

		def translate(name, content)
			Amrita::SanitizedString.new(module_handler.translate(@context, name, content))
		end
		
	
	end
	
	class DebugInformationSection < Section
		def header_text
			'デバッグ情報'
		end
		
		def body
			re = ''
			re << "[CGI環境変数]\n"
			
			list = %w(REQUEST_METHOD PATH_INFO SCRIPT_NAME SERVER_SOFTWARE QUERY_STRING).sort
			list.each do |key|
				re << sprintf("  %-16s : %s\n", key, ENV[key])
			end
			
			re << "[script URI]\n" << "  #{request.script_uri}\n"
			
			
			return Amrita::SanitizedString.new("<pre>#{PinkyBlog.escape_html(re)}</pre>")
		end
	end
	
	class HeadlineSection < Section
		require_opt :entries


		def header_text
			config.headline_title
		end
		
		
		def body
			model = {}
			model[:items] = []
			
			counts = nil
			@entries.sort!{|a,b| b.last_modified <=> a.last_modified}.each do |entry|
				model[:items] << {
					:a => {
						:name => entry.title_caption,
						:href => @context.get_cgi_path("entries/#{entry.id}")
					}
				}
				
				if entry.edited_number == 0 then
					model[:items].last[:info] = (" - " + @context.date2str(entry.created) + "作成")
				else
					model[:items].last[:info] = (" - " + @context.date2str(entry.last_modified) + "更新")
				end
				
			end

			
			
			return expand_amrita_template('headline.html', model)
			
			
		end
		
	end
	
	class MessageFormSection < Section
		default_opt :default_name, ""
		default_opt :default_address, ""


		def header_text
			config.message_form_title
		end
		
		def body
			model = {}
			model[:post_path] = post_path
			model[:guide] = config.message_form_guide
			
			return expand_amrita_template("message_form.html", model)
			
		end
	end


	class MessageSection < Section
		
		def header_text
			nil
		end
		
		def body
			return Amrita::SanitizedString.new("<p><em>#{@context.message}</em></p>")
		end
		
		def modori
			nil
		end
	end

	

	
	class WarningSection < Section
		require_opt :warnings
		
		def header_text
			nil
		end
		
		def modori
			nil
		end
		
		def body
			model = {}
			model[:items] = []
			@warnings.each do |warning|
				model[:items] = {:message => warning.to_s}
			end
			
			return expand_amrita_template("warning.html", model)

		end
	end
	
	class ErrorSection < Section
		require_opt :body, :http_status
		
		def header_text
			"エラー : #{@http_status}"
		end
		
		def body
			return Amrita::SanitizedString.new(HTML::Tag.new('p', @body).to_s)
		end
	end
	
	class PostBlockSection < Section
		def header_text
			"エラー : #{HTTP_BAD_REQUEST}"
		end
		
		def body
			return expand_amrita_template("post_block.html", {})
		end
	end
	
	class EntrySection < Section
		require_opt :entry
		default_opt :ex_footer_visible, true
		
		def header_text
			return @entry.title_caption
		end
		

		def body
			translate(@entry.format, @entry.content || '')
		end
		
		def footer
			model = {}
			model[:items] = []
			if @ex_footer_visible then
				if @entry.is_a?(StaticEntry) then
					model[:items] << {:href => @context.get_cgi_path("entries/#{@entry.id}/edit_form"),
					                  :caption => "このエントリを編集"} if master_mode?
				else
					
					model[:items] << {:href => @context.get_cgi_path("entries/#{@entry.id}/edit_form"),
					                  :caption => "この記事を編集"} if master_mode?
					model[:items] << {:href => @context.get_cgi_path("entries/#{@entry.id}"), :caption => "この記事のURL"}
					model[:items] << {:href => @context.get_cgi_path("entries/#{@entry.id}", nil, 'COMMENT'),
					                  :caption => "コメント:#{@entry.comments.size}"} if @entry.commentable?
				end
			elsif master_mode? then
				model[:items] << {:href => @context.get_cgi_path("entries/#{@entry.id}/edit_form"), :caption => "この記事を編集"}
			end
				
			return model
		end
	end
	
	class EntryListSection < Section
		require_opt :entries, :access_counts


		def header_text
			'記事の一覧'
		end
		
		def body
			items = []

			@entries.each do |entry|
				items << {
					:caption => entry.title_caption,
					:href => @context.get_cgi_path("entries/#{entry.id}")
				}
				case request.sort
				when BY_CREATED
					items.last[:info] = @context.date2str(entry.created) + "作成"
				when BY_ACCESS
					items.last[:info] = (@access_counts[entry.id.to_s] || 0).to_s + " アクセス"
				when BY_MODIFIED
					items.last[:info] = @context.date2str(entry.last_modified) + "更新"
				end
			end
			
				
			action_path = @context.get_cgi_path("entries")
			if request.tags.empty? then
				filter_info = ''
			else
				tag_info = request.tags.map{|x| "[#{x}]"}
				filter_info = "（タグ#{tag_info.join}がついた記事のみ表示）"
			end
			
			return expand_eruby_template('entry_list.ehtml', binding)
		
			
			
		end
		
	end


	
	class PreviewSection < Section
		require_opt :content, :format
		
		def header_text
			'プレビュー'
		end
		
		def body
			translate(@format, @content)

		end

	end
	
	class EntryInformationSection < Section
		require_opt :entry, :referer_list
		
		def header_text
			return Amrita.a({:id => 'ENTRYINFO'}){"この記事の情報"}
		end

		def body
			model = {}
			
			model[:url] = @context.absolute_uri_to("entries/#{@entry.id}")
			model[:last_modified] = @context.time2str(@entry.last_modified)

			if master_mode? || config.referer_visible? then
				model[:referer_dt] = {}
				model[:referer_dd] = {:items => []}
				
				@referer_list.each do |url, site_name, count|
					unless site_name == '-'  then
						model[:referer_dd][:items] << {:href => url, :count => count.to_s, :caption => (site_name || url)}
					end
				end
			end

			
			if not @entry.is_a?(StaticEntry) and not snapshot_mode? then
				model[:tags_dt] = {}
				model[:tags_dd] = {}
				model[:tags_dd][:tags] = @entry.normal_tags.map do |tag|
					query = PinkyBlog.tags_to_query([tag])
				{:href => @context.absolute_uri_to('/entries', query), :name => tag, :title => "タグ「#{tag}」を含む記事を一覧表示"}
				end
				
			end
			return expand_amrita_template("entry_information.html", model)
		end
		
	end
	
	
	class TagListSection < Section
		require_opt :tag_list
		
		def header_text
			return (request.tags.empty? ? "タグで絞り込む" : "タグでさらに絞り込む")
		end

		def body
			model = {}
			
			model[:tags] = []
			@tag_list.each do |tag, count|
				query = PinkyBlog.tags_to_query(request.tags + [tag])
				model[:tags] << {:href => @context.route_to('entries', query), :name => "#{tag}(#{count})"}
			end

			model[:tags] << {:name => "タグ絞り込みを解除" , :href => @context.route_to('./entries', '')} unless request.tags.empty?
			
			
			return expand_amrita_template("tag_list.html", model)
		end
		
		def modori
			nil
		end
		
	end

	
	
	class EntryEditFormSection < Section
		require_opt :entry, :tag_list, :parameters

		def header_text
			if @entry then
				"#{@entry.title.empty? ? '記事' : @entry.title}の編集"
			else
				"新しい記事を書く"
			end
		end
		
		def body
			model = {}
			if @parameters then
				model[:title] = {:value => @parameters[:title]}
			elsif @entry then
				model[:title] = {:value => @entry.title}

			else
				model[:title] = {:value => ''}
			end
			
			if @parameters then
				model[:content] = Amrita::SanitizedString.new(PinkyBlog.escape_html(@parameters[:content]))
			elsif @entry then
				model[:content] = Amrita::SanitizedString.new(PinkyBlog.escape_html(@entry.content || ''))
			else
				model[:content] = ""
			end
			
			model[:format] = {}
			model[:format][:items] = []
			mod_list = module_handler.translator_modules.to_a
			mod_list.sort!{|x, y| x[0] <=> y[0]} # 名前順
			selected = (@parameters && @parameters[:format]) || (@entry && @entry.format) || config.default_translator
			mod_list.each do |name, mod|
				model[:format][:items] << {:value => name, :caption => mod::CAPTION}
				model[:format][:items].last[:selected] = 'true' if name == selected
			end
			
			model[:format_guide] = @context.get_cgi_path('format_guide')
			
			unless @entry && @entry.is_a?(StaticEntry) then
				model[:tag] = {}
				model[:tag][:list] = []
				i = 0
				@tag_list.each do |name, count|
					model[:tag][:list] << {:name => "tags_#{i}", :value => PinkyBlog.encode_tag(name), :caption => name + "(#{count})"}
					if @parameters then
						model[:tag][:list].last[:checked] = @parameters[:tags].include?(name) && 'true'
					elsif @entry then
						model[:tag][:list].last[:checked] = @entry.tags.include?(name) && 'true'
					end
					i += 1
				end
				
				model[:tag][:add_tag] = @parameters && @parameters[:add_tag]

			end
			
			if @parameters then
				# プレビュー
				model[:invisible_checked] = 'checked' if @parameters[:invisible]
			elsif @entry then
				# 既存エントリの編集画面（開いた直後）
				model[:invisible_checked] = 'checked' if @entry.invisible?
			end

			model[:id] = {:value => @entry.id} if @entry
			model[:post_path] = post_path
			
			return expand_amrita_template("entry_edit_form.html", model)
			
			
		end
	end
	
	class FormatGuideSection < Section

		def header_text
			"記事の書き方について"
		end
		
		def body
			model = {}
			model[:format_list] = {}
			model[:format_list][:items] = []
			mod_list = module_handler.translator_modules.to_a
			mod_list.sort!{|x, y| x[0] <=> y[0]} # 名前順
			mod_list.each do |name, mod|
				model[:format_list][:items] << {:name => name, :href => @context.get_cgi_path("format_guide/#{name}"),
				                                :caption => mod::CAPTION}
			end
			
			
			return expand_amrita_template("format_guide.html", model)
			
			
		end
	end


	class FormatDetailSection < Section

		def header_text
			"#{module_handler.translator_modules[format_name]::CAPTION}"
		end
		def format_name
			request.path_items[1]
		end
		def body
			translator = module_handler.get_translator(@context, format_name)
			return Amrita::SanitizedString.new(translate(format_name, translator.format_guide.gsub(/\r\n/, "\n")))
		end
	end
	

	class FormatDetailSourceSection < Section

		def header_text
			"元になったテキスト"
		end
		def format_name
			request.path_items[1]
		end
		def body
			
			translator = get_translator(format_name)
			return Amrita::SanitizedString.new("<pre>" + CGI.escapeHTML(translator.format_guide) + "</pre>")
		end
	end


	class CommentSection < Section
		require_opt :comments

		def header_text
			Amrita.a({:id => 'COMMENT'}){"コメント"}
		end
		
		def body
			model = {}
			model[:comments] = []
			@comments.each_index do |i|
				comment = @comments[i]
				if master_mode? && comment.mail_address then
					info = Amrita::SanitizedString.new(%Q|#{@context.time2str(comment.time)}<br><a href="mailto:#{comment.mail_address}">#{comment.mail_address}</a>|)
				else
					info = Amrita::SanitizedString.new("#{@context.time2str(comment.time)}")
				end
				model[:comments] << {
					:header => (@comments[i].writer ? "#{i+1} : #{@comments[i].writer}" : "#{i+1} :"),
					:content => Amrita::SanitizedString.new(comment.content_html),
					:info => info
				}
			end
			model[:action] = action_path

			
			return expand_amrita_template("comment.html", model)
			
			
		end
	end

	
	class CommentFormSection < Section
		require_opt :entry_id, :default_name, :default_address


		def header_text
			Amrita.a({:id => 'COMMENT-FORM'}){"この記事にコメントする"}
		end
		
		def body
			model = {}
			model[:entry_id] = @entry_id
			model[:name] = @default_name
			model[:address] = @default_address
			model[:post_path] = post_path
			
			model[:name_info] = (config.commentator_name_required? ? Amrita::SanitizedString.new('<em>必須</em>') : '空欄可')
			model[:address_info] = (config.commentator_address_required? ? Amrita::SanitizedString.new('<em>必須</em>') : '空欄可')
			
			return expand_amrita_template("comment_form.html", model)
			
		end
	end
	
	
	class MasterMenuSection < Section
		def header_text
			"管理者メニュー"
		end
		
		def body
			model = {}
			model[:entry_add_form] = @context.get_cgi_path('master_menu/entry_add_form')
			model[:message_list] = @context.get_cgi_path('master_menu/message_list')
			model[:entry_manager] = @context.get_cgi_path('master_menu/entry_manager')
			model[:blog_config] = @context.get_cgi_path('master_menu/blog_config')
			model[:post_limit] = @context.get_cgi_path('master_menu/post_limit')
			model[:referer_config] = @context.get_cgi_path('master_menu/referer_config')
			model[:snapshot] = @context.get_cgi_path('master_menu/snapshot')
			model[:logout] = @context.get_cgi_path('/').to_s + '?logout=1'
			return expand_amrita_template("master_menu.html", model)
			
			
		end
	end
	
	class SystemInformationSection < Section
	
		def header_text
			"システム情報"
		end
		
		def body
			model = {}
			model[:core_version] = (config.demo? ? "#{CORE_VERSION}（サンプル動作中）" : CORE_VERSION)
			model[:ruby_version] = "#{RUBY_VERSION} (#{RUBY_RELEASE_DATE}) [#{RUBY_PLATFORM}]"
			model[:disp_env] = {:href => @context.route_to('/master_menu/env')} if @context.master_mode?
			return expand_amrita_template("system_information.html", model)
			
			
		end
	end
	
	class ENVSection < Section
	
		def header_text
			"環境変数"
		end
		
		def body
			list = %w(GATEWAY_INTERFACE SERVER_SOFTWARE SCRIPT_NAME PATH_INFO PATH_TRANSLATED).sort
			rows = list.map{|k| {:key => k, :value => ENV[k]}}
			return expand_amrita_template("env.html", {:rows => rows})
			
			
		end
	end

	

	class NotifyingSection < Section
		require_opt :notifications

		def header_text
			nil
		end
		
		def body
			model = {}
			model[:items] = []
			model[:items] = @notifications.map{|x| {:caption => x} }
			return expand_amrita_template("notifying.html", model)
			
			
		end
		
		def modori
			nil
		end
	end
	
	class MessageListSection < Section
		require_opt :messages
		
		def header_text
			"ひとことメッセージ一覧"
		end
		
		def body
			model = {}
			model[:items] = []
			@messages.sort!{|a, b| b.time <=> a.time}
			@messages.each do |msg|
				model[:items] << {:id => msg.uuid, :content => msg.content, :time => @context.time2str(msg.time)}
			end
			model[:post_path] = post_path
			return expand_amrita_template("message_list.html", model)
		end
	end
	


	
	class BlogConfigFormSection < Section

		def header_text
			"blog設定"
		end
		
		def body
			model = {}
			model[:site_title] = config.site_title
			model[:writer_name] = config.writer_name
			model[:writer_address] = config.writer_address || ''
			model[:home_url] = config.home_url || ''
			model[:about_visible] = config.about_visible
			model[:recent_entries_visible] = config.recent_entries_visible
			model[:news_feed_visible] = config.news_feed_visible
			model[:headline_title] = config.headline_title
			model[:message_form_visible] = config.message_form_visible
			model[:message_form_title] = config.message_form_title
			model[:message_form_guide] = config.message_form_guide
			
			model[:commentator_name_required] = (config.commentator_name_required? && 'checked')
			model[:commentator_address_required] = (config.commentator_address_required? && 'checked')
			
			model[:menu_form_rows] = []
			MENU_KEYS.each do |key|
				model[:menu_form_rows] << {:caption_name => "menu_caption_of_#{key}", 
				                           :caption_value => config.menu_captions[key], :default => key.tr('_', ' ')}

				case key
				when 'about', 'recent_entries', 'news_feed'
					model[:menu_form_rows].last[:checkbox] = {:name => "#{key}_visible"}
					if config.send("#{key}_visible") then
						model[:menu_form_rows].last[:checkbox][:checked] = 'checked'
					end
				end
			end
			
			model[:default_translator] = []
			module_handler.translator_modules.to_a.sort.each do |name, mod|
				m = {}
				m[:caption] = mod::CAPTION
				m[:value] = name
				m[:selected] = 'true' if name == config.default_translator
				model[:default_translator] << m
			end
			
			model[:post_path] = post_path
			
			return expand_amrita_template("blog_config_form.html", model)
			
		end
	end
	
	class PostLimitFormSection < Section

		def header_text
			"投稿制限"
		end
		
		def body
			model = {}
			

			[:message_length, :comment_length].each do |sym|
				model[sym] = {}
				model[sym][:tight_length] = LIMIT_TABLE[sym]['tight']
				model[sym][:loose_length] = LIMIT_TABLE[sym]['loose']
				model[sym][:very_loose_length] = LIMIT_TABLE[sym]['very-loose']
				
				case config.send("#{sym}_limit")
				when 'very-loose'
					model[sym][:very_loose_checked] = 'checked'
				when 'tight'
					model[sym][:tight_checked] = 'checked'
				else
					model[sym][:loose_checked] = 'checked'
				end
			end
			
			model[:block_http] = 'checked' if config.block_http
			model[:block_ascii] = 'checked' if config.block_ascii
			
			model[:ng_word] = config.ng_words.join("\r\n")


		
			model[:post_path] = post_path
			
			return expand_amrita_template("post_limit_form.html", model)
			
		end
	end

	
	
	class RefererTableSection < Section
		require_opt :table

		def header_text
			"現在のURL置換リスト"
		end
		
		def body
			model = {}
			model[:list] = []
			@table.each do |url, name|
				model[:list] << {:url => url, :name => (name == '-' ? '(非表示)' : name)}
			end
			
			return expand_amrita_template("referer_table.html", model)
			
		end
	end
	
	class RefererTableEditFormSection < Section
	
		require_opt :table_text
		
		def header_text
			"URL置換リストの編集"
		end
		
		def body
			model = {}
			model[:post_path] = post_path
			model[:table] = @table_text
			
			return expand_amrita_template("referer_table_edit_form.html", model)
			
		end
	end
	
	class EntryDataListSection < Section
		require_opt :entries, :access_record, :file_data

		def initialize(context, option_args = {})
			super(context, option_args)
			@file_data[:size] ||= {}
		end
		
		def entry_rows
			model = []
			@entries.each do |entry|
				checked = !((entry.tags & request.tags).empty?)
				model << {
					:id => entry.id,
					:last_modified => @context.time2str_short(entry.last_modified),
					:href => @context.get_cgi_path("entries/#{entry.id}"),
					:caption => entry.title_caption,
					:size => sprintf("%.2f KB", @file_data[:size][entry.id] / 1024.0),
					:access => (@access_record['counts'][entry.id] || 0).to_i,
					:comment => (entry.commentable? ? entry.comments.size : '-'),
					:checked => (checked ? 'checked' : '')
				}
				model.last[:tags] = []
				entry.tags.each do |tag|
					query = PinkyBlog.tags_to_query([tag])
					model.last[:tags] << {:href => @context.get_cgi_path("master_menu/entry_manager", query), :name=> tag}
				end
			end
			return model
		end
	end
	
	class EntryManagerSection < EntryDataListSection
	

		def header_text
			"記事の管理・一括操作"
		end
		
		def body
			
			model = {}
			model[:post_path] = post_path
			@entries.sort!{|a,b| b.last_modified <=> a.last_modified}
			rows = entry_rows
			return expand_eruby_template("entry_manager.ehtml", binding)
		end
	end
	


	class EntryDeleteConfirmationSection < EntryDataListSection
	
		def header_text
			"記事の削除確認"
		end
		
		def body
			
			model = {}
			model[:post_path] = post_path
			@entries.sort!{|a,b| b.last_modified <=> a.last_modified}
			rows = entry_rows
			return expand_eruby_template("entry_delete_confirmation.ehtml", binding)
		end
	end



	
	
	
	
	class NavigationSection < Section
	
		require_opt :prev_href, :prev_caption, :next_href, :next_caption

	
		def header_text
			nil
		end
		
		def body
			model = {}
			if @prev_href && @prev_caption then
				model[:prev_a] = {}
				model[:prev_a][:href] = @prev_href
				model[:prev_a][:caption] = @prev_caption
			end
			if @next_href && @next_caption then
				model[:next_a] = {}
				model[:next_a][:href] = @next_href
				model[:next_a][:caption] = @next_caption
			end

			return expand_amrita_template("navigation.html", model)
			
			
		end
		
		def footer
			return nil
		end
		
		def modori
			return nil
		end
	end
	
	class SearchFormSection < Section
	
		require_opt :keywords

		def header_text
			"blog内の記事を検索"
		end
		
		def body
			expand_amrita_template("search_form.html", {:action => @context.get_cgi_path('search'), :keyword => @keywords.join(' ')})
		end
	end
	
	class SearchResultSection < Section
		require_opt :keywords, :hit_list
		
	
		def header_text
			"検索結果"
		end
		
		def body
			model = {}
			model[:info] = @keywords.map{|x| "「#{x}」"}.join + "で検索し、#{@hit_list.size}記事が見つかりました"
			model[:items] = []
			@hit_list.each do |entry, hits, score|
				model[:items] << {:caption => entry.title_caption, :href => @context.get_cgi_path("entries/#{entry.id}")}
				hit_list = []
				hit_list << "スコア:#{score}"
				hit_list += hits.map{|x| x.to_s}

				
				model[:items] << {:hit_list => hit_list}
				
			end
			expand_amrita_template("search_result.html", model)
		end
	end

	
	class LoginFormSection < Section
		def header_text
			"管理者パスワード入力"
		end
		
		def body
			expand_amrita_template("login_form.html", {:post_path => post_path, :auto_login => (config.auto_login? && 'checked')})
		end
	end
	
	class SnapshotSection < Section
		def header_text
			"スナップショット（実行確認）"
		end
		
		def body
			expand_amrita_template("snapshot.html", {:post_path => post_path})
		end
	end

	
	class NewsFeedSection < Section
		def header_text
			"ニュースフィード配信"
		end
		
		
		def body
			
			model = {}
			
			%w(created modified comment).each do |name|
				if (config.feed_dir_path + "#{name}.xml").exist? then
					model[name.to_sym] = {:url => @context.get_feed_url("#{name}.xml")}
				else
					model[name.to_sym] = "（まだフィードが作成されていません）"
				end
			end

			uri = request.file_route_to(config.res_dir_path + 'pinkyblog/atom10.png')
			model[:atom10_img] = Amrita.a({:src => uri.to_s})
			expand_amrita_template("news_feed.html", model)
		end
	end

	
end
