/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util.draw;

import java.awt.AlphaComposite;
import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Composite;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.image.BufferedImage;

import com.otk.application.TheConstants;
import com.otk.application.image.filter.AbstractFilter;
import com.otk.application.image.filter.BoundingBox;
import com.otk.application.image.filter.FilteringContext;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.MathUtils;

public class FaceFrame {

	public final AbstractFilter filter = new AbstractFilter() {

		@Override
		public FilteringContext doExecute(FilteringContext context) {
			BufferedImage newImage = ImageUtils.copy(context.getImage());
			Graphics2D g2d = newImage.createGraphics();
			paint(g2d, newImage.getWidth(), newImage.getHeight());
			g2d.dispose();
			return context.withImage(newImage);
		}

	};
	private double widthOverHeightRatio = 35.0 / 45.0;
	private double heightRatio = 0.6;
	private double horizontalOffsetRatio;
	private double verticalOffsetRatio = 0.0;
	private double borderWidthRatio = 0.01;
	private boolean positioningOverlayHeadShapeVisible = true;
	private boolean positioningOverlayRulersVisible = true;
	private boolean positioningOverlayYellowSquareVisible;
	private double positioningOverlayVerticalOffsetRatio = 0.0;
	private double positioningOverlayHeightRatio = 0.7;
	private double positioningOverlayWidthRatio = 0.9;

	public double getPositioningOverlayVerticalOffsetRatio() {
		return positioningOverlayVerticalOffsetRatio;
	}

	public void setPositioningOverlayVerticalOffsetRatio(double positioningOverlayVerticalOffsetRatio) {
		this.positioningOverlayVerticalOffsetRatio = positioningOverlayVerticalOffsetRatio;
	}

	public double getPositioningOverlayHeightRatio() {
		return positioningOverlayHeightRatio;
	}

	public void setPositioningOverlayHeightRatio(double positioningOverlayHeightRatio) {
		this.positioningOverlayHeightRatio = positioningOverlayHeightRatio;
	}

	public double getPositioningOverlayWidthRatio() {
		return positioningOverlayWidthRatio;
	}

	public void setPositioningOverlayWidthRatio(double positioningOverlayWidthRatio) {
		this.positioningOverlayWidthRatio = positioningOverlayWidthRatio;
	}

	public double getHeightRatio() {
		return heightRatio;
	}

	public void setHeightRatio(double heightRatio) {
		this.heightRatio = heightRatio;
	}

	public double getHorizontalOffsetRatio() {
		return horizontalOffsetRatio;
	}

	public void setHorizontalOffsetRatio(double horizontalOffsetRatio) {
		this.horizontalOffsetRatio = horizontalOffsetRatio;
	}

	public double getVerticalOffsetRatio() {
		return verticalOffsetRatio;
	}

	public void setVerticalOffsetRatio(double verticalOffsetRatio) {
		this.verticalOffsetRatio = verticalOffsetRatio;
	}

	public boolean isPositioningOverlayYellowSquareVisible() {
		return positioningOverlayYellowSquareVisible;
	}

	public void setPositioningOverlayYellowSquareVisible(boolean positioningOverlayYellowSquareVisible) {
		this.positioningOverlayYellowSquareVisible = positioningOverlayYellowSquareVisible;
	}

	public boolean isPositioningOverlayHeadShapeVisible() {
		return positioningOverlayHeadShapeVisible;
	}

	public void setPositioningOverlayHeadShapeVisible(boolean positioningOverlayHeadShapeVisible) {
		this.positioningOverlayHeadShapeVisible = positioningOverlayHeadShapeVisible;
	}

	public boolean isPositioningOverlayRulersVisible() {
		return positioningOverlayRulersVisible;
	}

	public void setPositioningOverlayRulersVisible(boolean positioningOverlayRulersVisible) {
		this.positioningOverlayRulersVisible = positioningOverlayRulersVisible;
	}

	public double getWidthOverHeightRatio() {
		return widthOverHeightRatio;
	}

	public void setWidthOverHeightRatio(double widthOverHeightRatio) {
		this.widthOverHeightRatio = widthOverHeightRatio;
	}

	public AbstractFilter getFilter() {
		return filter;
	}

	private void paintHeadShape(Graphics g, int canvasWidth, int canvasHeight) {
		Rectangle bounds = getPositioningOverlayBounds(canvasWidth, canvasHeight);
		paintHeadShape(g, bounds);
	}

	private void paintHeadShape(Graphics g, Rectangle bounds) {
		paintPositioningOverlay(TheConstants.positioningRectangleHeadShapeImage, g, bounds);
	}

	private void paintRulers(Graphics g, int canvasWidth, int canvasHeight) {
		Rectangle bounds = getPositioningOverlayBounds(canvasWidth, canvasHeight);
		paintRulers(g, bounds);
	}

	private void paintBorder(Graphics2D g, int canvasWidth, int canvasHeight) {
		Rectangle bounds = getBounds(canvasWidth, canvasHeight);
		paintBorder(g, bounds);
	}

	private void paintRulers(Graphics g, Rectangle bounds) {
		paintPositioningOverlay(TheConstants.positioningRectangleRulersImage, g, bounds);
	}

	private void paintBorder(Graphics2D g, Rectangle bounds) {
		int borderLineWidth = MathUtils.round(borderWidthRatio * (bounds.width + bounds.height) / 2.0);
		int x = bounds.x - borderLineWidth;
		int y = bounds.y - borderLineWidth;
		int width = bounds.width + 2 * borderLineWidth;
		int height = bounds.height + 2 * borderLineWidth;
		g.setStroke(new BasicStroke(borderLineWidth));
		g.setColor(Color.RED);
		g.drawRect(x, y, width, height);
	}

	private void paintPositioningOverlay(Image overlayImage, Graphics g, Rectangle bounds) {
		Graphics2D g2d = (Graphics2D) g;
		AlphaComposite newComposite = AlphaComposite.getInstance(AlphaComposite.SRC_OVER, 1.0f);
		Composite compositeTorestore = g2d.getComposite();
		g2d.setComposite(newComposite);
		ImageUtils.drawSmoothlyScaled(g2d, overlayImage, bounds.x, bounds.y, bounds.width, bounds.height);
		g2d.setComposite(compositeTorestore);
	}

	public BufferedImage crop(Image picture) {
		BufferedImage bufferedPicture = ImageUtils.getBufferedImage(picture);
		Rectangle croppingRect = getBounds(bufferedPicture.getWidth(), bufferedPicture.getHeight());
		BufferedImage result = new BufferedImage(croppingRect.width, croppingRect.height,
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		int dx1 = 0;
		int dx2 = croppingRect.width;
		int dy1 = 0;
		int dy2 = croppingRect.height;
		int sx1 = croppingRect.x;
		int sx2 = croppingRect.x + croppingRect.width;
		int sy1 = croppingRect.y;
		int sy2 = croppingRect.y + croppingRect.height;
		g2d.drawImage(bufferedPicture, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);
		g2d.dispose();
		return result;
	}

	public Rectangle getBounds(int canvasWidth, int canvasHeight) {
		return getBounds(canvasWidth, canvasHeight, getStandardizingWidthRatio(canvasWidth, canvasHeight), heightRatio,
				horizontalOffsetRatio, verticalOffsetRatio);
	}

	public boolean isOutOfbounds(int canvasWidth, int canvasHeight) {
		Rectangle bounds = getBounds(canvasWidth, canvasHeight);
		return !new Rectangle(canvasWidth, canvasHeight).contains(bounds);
	}

	public double getStandardizingWidthRatio(double canvasWidth, double canvasHeight) {
		double height = canvasHeight * heightRatio;
		double width = height * widthOverHeightRatio;
		return width / canvasWidth;
	}

	public double getHeightRatio(int canvasWidth, int canvasHeight, double widthRatio) {
		double width = canvasWidth * widthRatio;
		double height = width / widthOverHeightRatio;
		return height / canvasHeight;
	}

	private Rectangle getBounds(int canvasWidth, int canvasHeight, double widthRatio, double heightRatio,
			double horizontalOffsetRatio, double verticalOffsetRatio) {
		double centerX = (0.5 + horizontalOffsetRatio) * canvasWidth;
		double centerY = (0.5 + verticalOffsetRatio) * canvasHeight;
		double width = canvasWidth * widthRatio;
		double height = canvasHeight * heightRatio;
		Rectangle result = MathUtils.getBoundsAroundCenter(centerX, centerY, width, height).getBounds();
		return result;
	}

	private Rectangle getFrameRelativePositioningOverlayBounds(int frameWidth, int frameHeight) {
		return getBounds(frameWidth, frameHeight, positioningOverlayWidthRatio, positioningOverlayHeightRatio, 0.0,
				positioningOverlayVerticalOffsetRatio);
	}

	public Rectangle getPositioningOverlayBounds(int canvasWidth, int canvasHeight) {
		Rectangle frameBounds = getBounds(canvasWidth, canvasHeight);
		Rectangle result = getFrameRelativePositioningOverlayBounds(frameBounds.width, frameBounds.height);
		result.x += frameBounds.x;
		result.y += frameBounds.y;
		return result;
	}

	private Dimension getCroppingSize(int canvasWidth, int canvasHeight) {
		return getBounds(canvasWidth, canvasHeight).getSize();
	}

	public void paint(Graphics2D g2d, int canvasWidth, int canvasHeight) {
		if (positioningOverlayHeadShapeVisible) {
			paintHeadShape(g2d, canvasWidth, canvasHeight);
		}
		if (positioningOverlayRulersVisible) {
			paintRulers(g2d, canvasWidth, canvasHeight);
		}
		if (positioningOverlayYellowSquareVisible) {
			paintYellowSquare(g2d, canvasWidth, canvasHeight);
		}
		paintBorder(g2d, canvasWidth, canvasHeight);
	}

	public void paintPositioningOverlayOnCroppedImage(Graphics2D g2d, Dimension croppingSize) {
		Rectangle overlayBounds = getFrameRelativePositioningOverlayBounds(croppingSize.width, croppingSize.height);
		if (positioningOverlayHeadShapeVisible) {
			paintHeadShape(g2d, overlayBounds);
		}
		if (positioningOverlayRulersVisible) {
			paintRulers(g2d, overlayBounds);
		}
		if (positioningOverlayYellowSquareVisible) {
			paintYellowSquare(g2d, overlayBounds);
		}
	}

	private void paintYellowSquare(Graphics2D g2d, int canvasWidth, int canvasHeight) {
		Rectangle bounds = getPositioningOverlayBounds(canvasWidth, canvasHeight);
		paintYellowSquare(g2d, bounds);
	}

	private void paintYellowSquare(Graphics2D g2d, Rectangle bounds) {
		paintPositioningOverlay(TheConstants.positioningRectangleYellowSquareImage, g2d, bounds);

	}

	public BufferedImage correctSize(Image image, Dimension destSize, Color backgroundColor) {
		BoundingBox filter = new BoundingBox();
		filter.width = destSize.width;
		filter.height = destSize.height;
		filter.cutEmptySides = false;
		filter.scaleToHeight = false;
		filter.scaleToWidth = true;
		BufferedImage result = ImageUtils.filter(image, filter);
		if (backgroundColor != null) {
			result = ImageUtils.changeColor(result, new Color(0, 0, 0, 0), backgroundColor);
		}
		return result;
	}

	public BufferedImage correctCroppedImageSize(Image croppedImage, int canvasWidth, int canvasHeight,
			Color backgroundColor) {
		Dimension faceFrameSize = getCroppingSize(canvasWidth, canvasHeight);
		return correctSize(croppedImage, faceFrameSize, backgroundColor);
	}

}
