/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.RenderingHints;
import java.awt.image.BufferedImage;
import java.awt.image.RescaleOp;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.imageio.ImageIO;

import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.image.camera.Camera;
import com.otk.application.image.camera.FrameFormat;
import com.otk.application.image.filter.AbstractFilter;
import com.otk.application.image.filter.ContrastAndBrightness;
import com.otk.application.image.filter.FilteringContext;
import com.otk.application.image.filter.Rotation;
import com.otk.application.image.filter.Saturation;
import com.otk.application.util.draw.DropShadow;

public class ImageUtils {

	public static final BufferedImage NULL_IMAGE = new BufferedImage(1, 1, ImageUtils.getAdaptedBufferedImageType());
	public static final String[] IMAGE_EXTENSIONS;
	static {
		List<String> result = Arrays.asList(ImageIO.getReaderFileSuffixes());
		result = new ArrayList<String>(result);
		result.retainAll(Arrays.asList(ImageIO.getWriterFileSuffixes()));
		IMAGE_EXTENSIONS = result.toArray(new String[result.size()]);
	}

	public static boolean isNullImage(BufferedImage image) {
		return image.getWidth() * image.getHeight() == 1;
	}

	public static BufferedImage getBufferedImage(Image image) {
		if (image instanceof BufferedImage) {
			return (BufferedImage) image;
		}
		return ImageUtils.copy(image);
	}

	public static BufferedImage changeColor(BufferedImage image, Color mask, Color replacement) {
		BufferedImage destImage = new BufferedImage(image.getWidth(), image.getHeight(),
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g = destImage.createGraphics();
		g.drawImage(image, null, 0, 0);
		g.dispose();
		for (int i = 0; i < destImage.getWidth(); i++) {
			for (int j = 0; j < destImage.getHeight(); j++) {
				int destRGB = destImage.getRGB(i, j);
				if (mask.getRGB() == destRGB) {
					destImage.setRGB(i, j, replacement.getRGB());
				}
			}
		}
		return destImage;
	}

	public static void historicizeImage(Image image, String baseFileName, int historySize) {
		File file = new File(baseFileName);
		FileUtils.historicizeFile(file.getPath(), historySize);
		String fileExtension = FileUtils.getFileNameExtension(file.getName());
		try {
			if (!ImageIO.write(getBufferedImage(image), fileExtension, file)) {
				throw new UnexpectedError("Cannot write the image file: '" + file + "'");
			}
		} catch (IOException e) {
			throw new UnexpectedError(e);
		}
	}

	public static BufferedImage filter(Image im, List<AbstractFilter> filters) {
		FilteringContext filteringContext = new FilteringContext().withImage(getBufferedImage(im));
		for (AbstractFilter f : filters) {
			filteringContext = f.execute(filteringContext);
		}
		return filteringContext.getImage();
	}

	public static BufferedImage blendImages(BufferedImage im1, double im1Weight, BufferedImage im2, double im2Weight) {
		BufferedImage result = new BufferedImage(im1.getWidth(), im1.getHeight(),
				ImageUtils.getAdaptedBufferedImageType());
		for (int i = 0; i < im1.getWidth(); i++) {
			for (int j = 0; j < im1.getHeight(); j++) {
				Color c1 = new Color(im1.getRGB(i, j), true);
				Color c2 = new Color(im2.getRGB(i, j), true);
				int r = Math.max(0, Math.min(255, MathUtils
						.round((c1.getRed() * im1Weight + c2.getRed() * im2Weight) / (im1Weight + im2Weight))));
				int g = Math.max(0, Math.min(255, MathUtils
						.round((c1.getGreen() * im1Weight + c2.getGreen() * im2Weight) / (im1Weight + im2Weight))));
				int b = Math.max(0, Math.min(255, MathUtils
						.round((c1.getBlue() * im1Weight + c2.getBlue() * im2Weight) / (im1Weight + im2Weight))));
				int a = c1.getAlpha();
				result.setRGB(i, j, new Color(r, g, b, a).getRGB());
			}
		}
		return result;
	}

	public static BufferedImage unblendImages(BufferedImage im1, BufferedImage im2) {
		BufferedImage result = new BufferedImage(im1.getWidth(), im1.getHeight(),
				ImageUtils.getAdaptedBufferedImageType());
		for (int i = 0; i < im1.getWidth(); i++) {
			for (int j = 0; j < im1.getHeight(); j++) {
				Color c1 = new Color(im1.getRGB(i, j), true);
				Color c2 = new Color(im2.getRGB(i, j), true);
				int r = Math.max(0, Math.min(255, 2 * c1.getRed() - c2.getRed()));
				int g = Math.max(0, Math.min(255, 2 * c1.getGreen() - c2.getGreen()));
				int b = Math.max(0, Math.min(255, 2 * c1.getBlue() - c2.getBlue()));
				int a = c1.getAlpha();
				result.setRGB(i, j, new Color(r, g, b, a).getRGB());
			}
		}
		return result;
	}

	public static int getAdaptedBufferedImageType() {
		return BufferedImage.TYPE_INT_ARGB;
	}

	public static Color getDistantColor(Color c) {
		int brightness = MathUtils.round((c.getRed() + c.getGreen() + c.getBlue()) / 3.0);
		if (brightness >= 128) {
			return Color.BLACK;
		} else {
			return Color.WHITE;
		}
	}

	public static Dimension scale(Dimension d, double wScale, double hScale) {
		int w = MathUtils.round(d.getWidth() * wScale);
		int h = MathUtils.round(d.getHeight() * hScale);
		return new Dimension(w, h);
	}

	public static void drawLineBorder(Graphics2D g2d, Color color, int thickness, int width, int height) {
		g2d.setColor(color);
		g2d.setStroke(new BasicStroke(thickness));
		g2d.drawRect(thickness / 2, thickness / 2, width - thickness, height - thickness);
	}

	public static BufferedImage requireAdaptedImage(String filePath) {
		try {
			return ImageUtils.loadAdaptedImage(filePath);
		} catch (Exception e) {
			throw new UnexpectedError("Could not load the image file: '" + filePath + "': " + e.toString(), e);
		}
	}

	public static BufferedImage requireAdaptedImage(InputStream in) {
		try {
			return ImageUtils.loadAdaptedImage(in);
		} catch (Exception e) {
			throw new UnexpectedError("Could not load the image from stream: " + e.toString(), e);
		}
	}

	public static BufferedImage requireAdaptedImage(File file) {
		try {
			return ImageUtils.loadAdaptedImage(file);
		} catch (Exception e) {
			throw new UnexpectedError("Could not load the image file: '" + file + "': " + e.toString(), e);
		}
	}

	public static BufferedImage rotate(BufferedImage image, int angleDegrees) {
		Rotation filter = new Rotation();
		filter.angleDegrees = angleDegrees;
		FilteringContext context = new FilteringContext().withImage(getBufferedImage(image));
		context = filter.execute(context);
		return context.getImage();
	}

	public static BufferedImage scalePreservingRatio(Image image, int newWidth, int newHeight, boolean cutEmptySides,
			boolean boundArea, boolean cutBoundingOverflow, boolean smoothScale) {
		Dimension imageSize = new Dimension(image.getWidth(null), image.getHeight(null));
		Dimension boxSize = new Dimension(newWidth, newHeight);

		Dimension resultSize;
		Rectangle srcBounds;
		Rectangle dstBounds;

		if (boundArea) {
			if (cutBoundingOverflow) {
				dstBounds = new Rectangle(boxSize);
				resultSize = boxSize;
				srcBounds = MathUtils.scaletoFitInside(boxSize, imageSize);
			} else {
				dstBounds = MathUtils.scaleToBound(imageSize, boxSize);
				resultSize = dstBounds.getSize();
				srcBounds = new Rectangle(imageSize);
			}
		} else {
			if (cutEmptySides) {
				dstBounds = new Rectangle(MathUtils.scaletoFitInside(imageSize, boxSize).getSize());
				resultSize = dstBounds.getSize();
				srcBounds = new Rectangle(imageSize);
			} else {
				dstBounds = MathUtils.scaletoFitInside(imageSize, boxSize);
				resultSize = boxSize;
				srcBounds = new Rectangle(imageSize);
			}
		}

		BufferedImage result = new BufferedImage(resultSize.width, resultSize.height,
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g = result.createGraphics();
		int dx1 = dstBounds.x;
		int dy1 = dstBounds.y;
		int dx2 = dstBounds.x + dstBounds.width;
		int dy2 = dstBounds.y + dstBounds.height;
		int sx1 = srcBounds.x;
		int sy1 = srcBounds.y;
		int sx2 = srcBounds.x + srcBounds.width;
		int sy2 = srcBounds.y + srcBounds.height;
		if (smoothScale) {
			setSmoothScalingRenderingHints(g);
		}
		g.drawImage(image, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);
		g.dispose();

		return result;
	}

	public static BufferedImage scalePreservingRatio(Image image, int newWidth, int newHeight, boolean cutEmptySides,
			boolean smoothScale) {
		return scalePreservingRatio(image, newWidth, newHeight, cutEmptySides, false, false, smoothScale);
	}

	public static Dimension getSize(Image image) {
		return new Dimension(image.getWidth(null), image.getHeight(null));
	}

	public static BufferedImage getScreenShot(Component c) {
		if (c.getParent() == null) {
			c.addNotify();
			c.setSize(c.getPreferredSize());
			c.validate();
		}
		BufferedImage image = new BufferedImage(c.getWidth(), c.getHeight(), getAdaptedBufferedImageType());
		Graphics2D g = image.createGraphics();
		c.paint(g);
		g.dispose();
		return image;
	}

	public static BufferedImage getTextImage(String text, int fontSize, Color color) {
		Font font = new Font("Arial", Font.PLAIN, fontSize);
		return getTextImage(text, font, color);
	}

	public static BufferedImage getTextImage(String text, Font font, Color color) {
		if (text.length() == 0) {
			text = " ";
		}
		List<BufferedImage> lineImages = new ArrayList<BufferedImage>();
		for (String line : TextUtils.splitLines(text)) {
			Graphics2D g2d = new BufferedImage(1, 1, BufferedImage.TYPE_INT_ARGB).createGraphics();
			g2d.setFont(font);
			FontMetrics fontMetrics = g2d.getFontMetrics();
			int width = fontMetrics.stringWidth(line);
			if (width == 0) {
				width = 1;
			}
			int height = fontMetrics.getHeight();
			g2d.dispose();
			BufferedImage singleLineImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
			g2d = singleLineImage.createGraphics();
			g2d.setFont(font);
			fontMetrics = g2d.getFontMetrics();
			g2d.setColor(color);
			g2d.drawString(line, 0, fontMetrics.getAscent());
			g2d.dispose();
			lineImages.add(singleLineImage);
		}
		if (lineImages.size() == 1) {
			return lineImages.get(0);
		} else {
			int spacing = 0;
			return joinImages(lineImages, false, spacing, false);
		}
	}

	public static BufferedImage scaleToWidth(BufferedImage image, int width) {
		double imageWidthOverHeightRatio = image.getWidth() / (double) image.getHeight();
		image = ImageUtils.getBufferedImage(
				image.getScaledInstance(width, MathUtils.round(width / imageWidthOverHeightRatio), Image.SCALE_SMOOTH));
		return image;
	}

	public static BufferedImage scaleToHeight(BufferedImage image, int height) {
		double imageWidthOverHeightRatio = image.getWidth() / (double) image.getHeight();
		image = ImageUtils.getBufferedImage(image.getScaledInstance(MathUtils.round(height * imageWidthOverHeightRatio),
				height, Image.SCALE_SMOOTH));
		return image;
	}

	public static void drawSmoothlyScaled(Graphics2D g, Image image, int x, int y, int w, int h) {
		RenderingHints renderingHintsBackup = g.getRenderingHints();
		setSmoothScalingRenderingHints(g);
		g.drawImage(image, x, y, w, h, null);
		g.setRenderingHints(renderingHintsBackup);
	}

	public static void setSmoothScalingRenderingHints(Graphics2D g) {
		g.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
		g.setRenderingHint(RenderingHints.KEY_INTERPOLATION, RenderingHints.VALUE_INTERPOLATION_BICUBIC);
	}

	public static int getBrightness(Color color) {
		return MathUtils.round((color.getRed() + color.getGreen() + color.getBlue()) / 3.0);
	}

	public static BufferedImage adapt(Image image) {
		return ImageUtils.copy(image);
	}

	public static BufferedImage copy(Image image) {
		BufferedImage result = new BufferedImage(image.getWidth(null), image.getHeight(null),
				getAdaptedBufferedImageType());
		Graphics2D g = result.createGraphics();
		g.drawImage(image, 0, 0, null);
		return result;
	}

	public static BufferedImage filter(Image im, AbstractFilter... filters) {
		return filter(im, Arrays.asList(filters));
	}

	public static BufferedImage loadAdaptedImage(InputStream in) throws IOException {
		try {
			BufferedImage result = ImageIO.read(in);
			if (result == null) {
				throw new IOException("Failed to read the image format");
			}
			return adapt(result);
		} catch (IOException e) {
			throw new IOException("Error while loading image: " + e.getMessage(), e);
		}
	}

	public static BufferedImage loadAdaptedImage(File file) throws IOException {
		FileInputStream in = new FileInputStream(file);
		try {
			return loadAdaptedImage(in);
		} catch (IOException e) {
			throw new IOException("File '" + file + "': " + e.getMessage(), e);
		} finally {
			try {
				in.close();
			} catch (Exception ignore) {
			}
		}
	}

	public static BufferedImage loadAdaptedImage(String filePath) throws IOException {
		return loadAdaptedImage(new File(filePath));
	}

	public static Object stringToColor(String s) {
		try {
			String[] rgb = s.split(",");
			return new Color(Integer.valueOf(rgb[0]), Integer.valueOf(rgb[1]), Integer.valueOf(rgb[2]));
		} catch (Exception e) {
			return Color.decode("0x" + s);
		}
	}

	public static String toString(Color c) {
		return c.getRed() + "," + c.getGreen() + "," + c.getBlue();
	}

	public static BufferedImage addLineBorder(Image image, Color color, int thickness) {
		BufferedImage result = new BufferedImage(image.getWidth(null) + 2 * thickness,
				image.getHeight(null) + 2 * thickness, ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		g2d.drawImage(image, thickness, thickness, null);
		drawLineBorder(g2d, color, thickness, result.getWidth(), result.getHeight());
		g2d.dispose();
		return result;
	}

	public static BufferedImage removeTransparency(Image image, Color replacingColor) {
		BufferedImage result = new BufferedImage(image.getWidth(null), image.getHeight(null),
				getAdaptedBufferedImageType());
		Graphics2D g = result.createGraphics();
		g.setColor(replacingColor);
		g.drawImage(image, 0, 0, replacingColor, null);
		return result;
	}

	public static void annotateImage(BufferedImage image, String text, int fontSize, Color fontColor) {
		Graphics2D g2d = image.createGraphics();
		BufferedImage annotationImage = ImageUtils.getTextImage(text, fontSize, fontColor);
		annotationImage = ImageUtils.getBufferedImage(
				ImageUtils.scalePreservingRatio(annotationImage, image.getWidth(), image.getHeight(), false, true));
		g2d.drawImage(annotationImage, 0, 0, null);
		g2d.dispose();
	}

	public static Image addDisabledEffect(Image image) {
		image = getBufferedImage(image);
		Saturation filter = new Saturation();
		filter.amount = 0;
		return filter(image, filter);
	}

	public static Image addActivatedEffect(Image image) {
		image = getBufferedImage(image);
		ContrastAndBrightness filter = new ContrastAndBrightness();
		filter.brightnessCompensation = 20;
		filter.contrastCompensation = 0;
		return filter(image, filter);
	}

	public static Color getGrayScale(Color color) {
		int grayScaleComponent = (int) Math.round((color.getRed() + color.getGreen() + color.getBlue()) / 3);
		return new Color(grayScaleComponent, grayScaleComponent, grayScaleComponent);
	}

	public static boolean equals(BufferedImage img1, BufferedImage img2) {
		if (img1.getWidth(null) == img2.getWidth(null) && img1.getHeight(null) == img2.getHeight(null)) {
			for (int x = 0; x < img1.getWidth(null); x++) {
				for (int y = 0; y < img1.getHeight(null); y++) {
					if (img1.getRGB(x, y) != img2.getRGB(x, y))
						return false;
				}
			}
		} else {
			return false;
		}
		return true;
	}

	public static void assertImagesEquals(BufferedImage image1, BufferedImage image2) {
		if (!equals(image1, image2)) {
			throw new RuntimeException("Images differ:" + ":\nimage1(" + image1.getWidth() + "x" + image1.getHeight()
					+ "):\n" + imageToText(image1) + "\nimage2(" + image2.getWidth() + "x" + image2.getHeight() + "):\n"
					+ imageToText(image2));
		}
	}

	public static String imageToText(BufferedImage image) {
		StringBuilder result = new StringBuilder();
		for (int y = 0; y < image.getHeight(null); y++) {
			for (int x = 0; x < image.getWidth(null); x++) {
				result.append("\t" + image.getRGB(x, y));
			}
			result.append("\n");
		}
		return result.toString();
	}

	public static List<FrameFormat> getSortedVideoFormats(Camera camera) {
		List<FrameFormat> result;
		try {
			result = new ArrayList<FrameFormat>(camera.getVideoFormats());
			Collections.sort(result);
		} catch (Exception e) {
			result = Collections.emptyList();
		}
		return result;
	}

	public static FrameFormat getSmallestVideoFormat(Camera camera) {
		List<FrameFormat> formats = getSortedVideoFormats(camera);
		if (formats.size() > 0) {
			return formats.get(0);
		} else {
			return new FrameFormat(160, 120);
		}
	}

	public static BufferedImage joinImages(List<? extends Image> images, boolean horizontallyElseVertically) {
		return joinImages(images, horizontallyElseVertically, 0, false);
	}

	public static BufferedImage joinImages(List<? extends Image> images, boolean horizontallyElseVertically,
			int spacing, boolean centerAlignment) {
		int resultWidth = 0;
		int resultHeight = 0;
		for (int i = 0; i < images.size(); i++) {
			Image image = images.get(i);
			if (horizontallyElseVertically) {
				if (i > 0) {
					resultWidth += spacing;
				}
				resultWidth += image.getWidth(null);
				resultHeight = Math.max(resultHeight, image.getHeight(null));
			} else {
				if (i > 0) {
					resultHeight += spacing;
				}
				resultHeight += image.getHeight(null);
				resultWidth = Math.max(resultWidth, image.getWidth(null));
			}
		}
		BufferedImage result = new BufferedImage(resultWidth, resultHeight, BufferedImage.TYPE_INT_ARGB);
		Graphics2D g = result.createGraphics();
		int x = 0;
		int y = 0;
		for (Image image : images) {
			if (horizontallyElseVertically) {
				if (centerAlignment) {
					y = Math.round((resultHeight - image.getHeight(null)) / 2f);
				}
				g.drawImage(image, x, y, null);
				x += image.getWidth(null) + spacing;
			} else {
				if (centerAlignment) {
					x = Math.round((resultWidth - image.getWidth(null)) / 2f);
				}
				g.drawImage(image, x, y, null);
				y += image.getHeight(null) + spacing;
			}
		}
		g.dispose();
		return result;
	}

	public static List<BufferedImage> splitImage(BufferedImage image, int slices, boolean horizontallyElseVertically,
			boolean sharedMemory) {
		List<BufferedImage> result = new ArrayList<BufferedImage>();
		for (int i = 0; i < slices; i++) {
			int x, y, w, h;
			if (horizontallyElseVertically) {
				w = image.getWidth() / slices;
				h = image.getHeight();
				x = i * w;
				y = 0;
			} else {
				w = image.getWidth();
				h = image.getHeight() / slices;
				x = 0;
				y = i * h;
			}
			BufferedImage subImage = image.getSubimage(x, y, w, h);
			if (!sharedMemory) {
				subImage = copy(subImage);
			}
			result.add(subImage);
		}
		return result;
	}

	public static BufferedImage getOverflowingSubImage(BufferedImage inputImage, int x, int y, int width, int height) {
		BufferedImage result = new BufferedImage(width, height, getAdaptedBufferedImageType());
		Graphics2D g = result.createGraphics();
		g.drawImage(inputImage, -x, -y, null);
		g.dispose();
		return result;
	}

	public static BufferedImage changeBrightnessAndContrast(BufferedImage bufferedImage, int brightnessCompensation,
			int contrastCompensation, Color contrastPivot) {
		if ((brightnessCompensation != 0) || (contrastCompensation != 0)) {
			int MAX_SCALE_FACTOR = 20;
			int MAX_CONTRAST_COMPENSATION = 127;
			float scaleFactor = (float) Math.pow(2.0,
					contrastCompensation * (Math.exp(Math.log(MAX_SCALE_FACTOR) / 2.0) / MAX_CONTRAST_COMPENSATION));
			float rOffset = ((+brightnessCompensation - contrastPivot.getRed()) * scaleFactor) + contrastPivot.getRed();
			float gOffset = ((+brightnessCompensation - contrastPivot.getGreen()) * scaleFactor)
					+ contrastPivot.getGreen();
			float bOffset = ((+brightnessCompensation - contrastPivot.getBlue()) * scaleFactor)
					+ contrastPivot.getBlue();
			RescaleOp rescale = new RescaleOp(new float[] { scaleFactor, scaleFactor, scaleFactor, 1f },
					new float[] { rOffset, gOffset, bOffset, 0f }, null);
			return rescale.filter(bufferedImage, null);
		}
		return bufferedImage;
	}

	public static BufferedImage addRealisticPaperEffect(BufferedImage image) {
		return DropShadow.add(image);
	}

	public static void saveInExtensionFormat(BufferedImage image, File file) {
		String extension = FileUtils.getFileNameExtension(file.getName());
		if ((extension == null) || (extension.trim().length() == 0)) {
			throw new StandardError("The image file extension was not specified. Expected: "
					+ Arrays.toString(ImageUtils.IMAGE_EXTENSIONS));
		}
		try {
			if (!ImageIO.write(image, extension, file)) {
				throw new StandardError("Failed to write the image in the specified format: '" + extension + "'");
			}
		} catch (IOException e) {
			throw new StandardError(e.toString(), e);
		}
	}

}
