/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.io.IOException;
import java.io.Serializable;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.otk.application.BuildProperties;
import com.otk.application.IdPhotoProject;
import com.otk.application.Main;
import com.otk.application.error.UnexpectedError;

/**
 * This class allows to optionally gather the application usage statistics.
 * 
 * @author olitank
 *
 */
public class Analytics {

	public static final String TRACKINGS_DELIVERY_URL = Main.WEBSITE_URL
			+ "/custom/forward-http-to-google-analytics.php";
	public static final String TRACKING_ID = "UA-71918184-1";

	private static final int TRACKINGS_TRANSMISSION_PACKET_SIZE = 1000;

	private static Thread regularSender;
	private static boolean initialized;
	private static List<Tracking> trackings = Collections.synchronizedList(new ArrayList<Tracking>());

	public static void initialize() {
		if (initialized) {
			return;
		}
		startRegularSender();
		initialized = true;
	}

	public static void shutdown() {
		if (!initialized) {
			return;
		}
		regularSender.interrupt();
		try {
			regularSender.join();
		} catch (InterruptedException e) {
			throw new RuntimeException(e);
		}
		try {
			sendAllTrackings();
		} catch (Throwable t) {
			Log.error(t);
		}
		initialized = false;
	}

	private static void startRegularSender() {
		regularSender = new Thread(MiscUtils.formatThreadName(Analytics.class, "RegularSender")) {
			@Override
			public synchronized void run() {
				while (true) {
					MiscUtils.sleepSafely(1000);
					if (isInterrupted()) {
						break;
					}
					if (trackings.size() >= TRACKINGS_TRANSMISSION_PACKET_SIZE) {
						sendAllTrackings();
					}
				}
			}
		};
		regularSender.setDaemon(true);
		regularSender.setPriority(Thread.MIN_PRIORITY);
		regularSender.start();
	}

	public static void track(String event, String... details) {
		for (int i = 0; i < details.length; i++) {
			details[i] = TextUtils.escapeNewLines(details[i]);
		}
		Log.info("Tracking: " + event);
		trackings.add(new Tracking(new Date(), event, details));
	}

	public synchronized static void sendAllTrackings() {
		if (!IdPhotoProject.isUsageStatisticsShared()) {
			Log.info("Skipping trackings delivery");
			trackings.clear();
			return;
		}
		Log.info("Delivering trackings");
		try {
			for (Tracking tracking : trackings) {
				sendTracking(tracking.getDateTime(), tracking.getUsed(), tracking.getDetails());
			}
		} catch (Exception e) {
			Log.error(e);
		} finally {
			trackings.clear();
		}
	}

	public static void sendTracking(Date when, String used, String... details) {
		used = anonymize(used);
		for (int i = 0; i < details.length; i++) {
			details[i] = anonymize(details[i]);
		}
		Map<String, String> arguments = new HashMap<String, String>();
		StringBuilder categoryId = new StringBuilder();
		{
			categoryId.append("PhoyoId");
			categoryId.append("-");
			categoryId.append("b" + BuildProperties.get().getId());
			arguments.put("ec", categoryId.toString());
		}
		arguments.put("ea", used);
		arguments.put("qt", Long.toString((new Date().getTime() - when.getTime())));
		if (details.length > 0) {
			arguments.put("el", TextUtils.stringJoin(Arrays.asList(details), " - "));
		}
		try {
			arguments.put("cid", hexEncode(System.getProperty("user.name")));
		} catch (Exception e) {
			arguments.put("cid", "UnknownHost");
		}
		try {
			MiscUtils.sendHttpPost(new URL(TRACKINGS_DELIVERY_URL), arguments);
		} catch (IOException e) {
			throw new UnexpectedError(e);
		}
	}

	private static String hexEncode(String s) {
		StringBuilder result = new StringBuilder();
		for (byte b : s.getBytes()) {
			result.append(Integer.toHexString(MathUtils.unsignedByte(b)));
		}
		return result.toString();
	}

	public static String anonymize(String message) {
		message = message.replaceAll("([Pp]assword)=.*", "$1=**********");
		message = message.replaceAll("([Ll]ogin)=.*", "$1=**********");
		message = message.replaceAll("(user\\..*)=.*", "$1=**********");
		return message;
	}

	public static class Tracking implements Serializable {

		private static final long serialVersionUID = 1L;

		private long id;

		private Date dateTime;

		private String used;

		private String[] details;

		public Tracking() {
		}

		public Tracking(Date dateTime, String used, String[] details) {
			super();
			this.dateTime = dateTime;
			this.used = used;
			this.details = details;
		}

		public long getId() {
			return id;
		}

		public void setId(long id) {
			this.id = id;
		}

		public Date getDateTime() {
			return dateTime;
		}

		public void setDateTime(Date dateTime) {
			this.dateTime = dateTime;
		}

		public String getUsed() {
			return used;
		}

		public void setUsed(String used) {
			this.used = used;
		}

		public String[] getDetails() {
			return details;
		}

		public void setDetails(String[] details) {
			this.details = details;
		}

		@Override
		public String toString() {
			return "Tracking [dateTime=" + dateTime + ", used=" + used + ", details=" + Arrays.toString(details) + "]";
		}

	}

	public static void main(String[] args) {
		initialize();
		track("used", "details");
		track("used");
		shutdown();
	}

}
