/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.filter;

import java.awt.Point;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.util.ArrayList;
import java.util.List;

public class FastestRGBAccess {

	public static boolean isRGBConversionRequired(BufferedImage image) {
		return image.getType() != BufferedImage.TYPE_INT_ARGB;
	}

	public static int get(BufferedImage image, int x, int y) {
		if (isRGBConversionRequired(image)) {
			return image.getRGB(x, y);
		} else {
			int[] rgbArray = ((DataBufferInt) image.getRaster().getDataBuffer()).getData();
			int pixelIndex = x + (image.getWidth() * y);
			return rgbArray[pixelIndex];
		}
	}

	public static void set(BufferedImage image, int x, int y, int rgb) {
		if (isRGBConversionRequired(image)) {
			image.setRGB(x, y, rgb);
		} else {
			int[] rgbArray = ((DataBufferInt) image.getRaster().getDataBuffer()).getData();
			int pixelIndex = x + (image.getWidth() * y);
			rgbArray[pixelIndex] = rgb;
		}
	}

	public static int[] get(BufferedImage image) {
		int width = image.getWidth();
		int height = image.getHeight();
		boolean rgbConversionRequired = isRGBConversionRequired(image);
		if (rgbConversionRequired) {
			return image.getRGB(0, 0, width, height, null, 0, width);
		} else {
			WritableRaster raster = image.getRaster();
			return (int[]) raster.getDataElements(0, 0, width, height, null);
		}

	}

	public static void set(int[] pixels, BufferedImage image) {
		int width = image.getWidth();
		int height = image.getHeight();
		boolean dstRGBConversionRequired = isRGBConversionRequired(image);
		if (dstRGBConversionRequired) {
			image.setRGB(0, 0, width, height, pixels, 0, width);
		} else {
			WritableRaster raster = image.getRaster();
			raster.setDataElements(0, 0, width, height, pixels);
		}
	}

	public static void scan(BufferedImage srcImage, ScanFilter filter) {
		int width = srcImage.getWidth();
		int height = srcImage.getHeight();
		List<Point> scan = new ArrayList<Point>();
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				scan.add(new Point(x, y));
			}
		}
		scan(srcImage, filter, scan);
	}

	public static void scan(BufferedImage srcImage, ScanFilter filter, Iterable<Point> scan) {
		int width = srcImage.getWidth();
		int[] srcPixels = get(srcImage);
		for (Point p : scan) {
			int pixelIndex = p.y * width + p.x;
			filter.handle(p.x, p.y, srcPixels[pixelIndex], srcPixels);
		}
	}

	public static void transfer(BufferedImage srcImage, BufferedImage dstImage, TransferFilter filter) {
		int width = dstImage.getWidth();
		int height = dstImage.getHeight();
		List<Point> scan = new ArrayList<Point>();
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				scan.add(new Point(x, y));
			}
		}
		transfer(srcImage, dstImage, filter, scan);
	}

	public static void transfer(BufferedImage srcImage, BufferedImage dstImage, TransferFilter filter,
			Iterable<Point> scan) {
		int width = dstImage.getWidth();
		int height = dstImage.getHeight();
		int[] dstPixels = new int[width * height];
		int[] srcPixels = get(srcImage);
		for (Point p : scan) {
			int pixelIndex = p.y * width + p.x;
			dstPixels[pixelIndex] = filter.get(p.x, p.y, srcPixels[pixelIndex], srcPixels);
		}
		set(dstPixels, dstImage);
	}

	public static void blend(BufferedImage srcImage1, BufferedImage srcImage2, BufferedImage dstImage,
			BlendFilter filter) {
		int width = dstImage.getWidth();
		int height = dstImage.getHeight();
		List<Point> scan = new ArrayList<Point>();
		for (int x = 0; x < width; x++) {
			for (int y = 0; y < height; y++) {
				scan.add(new Point(x, y));
			}
		}
		blend(srcImage1, srcImage2, dstImage, filter, scan);
	}

	public static void blend(BufferedImage srcImage1, BufferedImage srcImage2, BufferedImage dstImage,
			BlendFilter filter, Iterable<Point> scan) {
		int width = dstImage.getWidth();
		int height = dstImage.getHeight();
		WritableRaster srcRaster1 = srcImage1.getRaster();
		WritableRaster srcRaster2 = srcImage2.getRaster();
		WritableRaster dstRaster = dstImage.getRaster();
		int[] srcPixels1 = new int[width * height];
		int[] srcPixels2 = new int[width * height];
		int[] dstPixels = new int[width * height];
		boolean src1RGBConversionRequired = isRGBConversionRequired(srcImage1);
		boolean src2RGBConversionRequired = isRGBConversionRequired(srcImage2);
		boolean dstRGBConversionRequired = isRGBConversionRequired(dstImage);
		if (src1RGBConversionRequired) {
			srcPixels1 = srcImage1.getRGB(0, 0, width, height, null, 0, width);
		} else {
			srcPixels1 = (int[]) srcRaster1.getDataElements(0, 0, width, height, null);
		}
		if (src2RGBConversionRequired) {
			srcPixels2 = srcImage2.getRGB(0, 0, width, height, null, 0, width);
		} else {
			srcPixels2 = (int[]) srcRaster2.getDataElements(0, 0, width, height, null);
		}
		for (Point p : scan) {
			int pixelIndex = p.y * width + p.x;
			dstPixels[pixelIndex] = filter.get(p.x, p.y, srcPixels1[pixelIndex], srcPixels2[pixelIndex], srcPixels1,
					srcPixels2);
		}
		if (dstRGBConversionRequired) {
			dstImage.setRGB(0, 0, width, height, dstPixels, 0, width);
		} else {
			dstRaster.setDataElements(0, 0, width, height, dstPixels);
		}
	}

	public interface ScanFilter {
		void handle(int x, int y, int rgb, int[] srcPixels);
	}

	public interface TransferFilter {
		int get(int x, int y, int rgb, int[] srcPixels);
	}

	public interface BlendFilter {
		int get(int x, int y, int rgb1, int rgb2, int[] srcPixels1, int[] srcPixels2);
	}

}
