/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.Color;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.util.Collections;
import java.util.List;

import com.otk.application.resource.image.ClassInImagesPackage;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.MiscUtils;

/**
 * Virtual {@link Camera} device driver. Continuously outputs an image
 * indicating that no camera device was selected.
 * 
 * @author olitank
 *
 */
public class NOCamDriver extends AbstractCameraDriver {

	public static final String LOCAL_DEVICE_NAME = "No Device";
	public static final List<FrameFormat> FORMATS = FrameFormat.COMMON;
	public static final BufferedImage NO_CAMERA_IMAGE = ImageUtils
			.requireAdaptedImage(ClassInImagesPackage.class.getResourceAsStream("noCamera.png"));

	private Thread captureThread;
	private boolean mustStop;

	public NOCamDriver(Camera camera) {
		super(camera);
	}

	@Override
	public String getName() {
		return "Nocam";
	}

	@Override
	public void doSetUp() throws Exception {
	}

	@Override
	public void doCleanUp() throws Exception {
	}

	@Override
	public void startCapture() {
		mustStop = false;
		captureThread = new Thread(MiscUtils.formatThreadName(NOCamDriver.class, "CaptureSimulator")) {
			int brightnessCompensation = 0;
			int brightnessCompensationIncrement = 3;
			int brightnessCompensationMaxAmount = 30;
			long brightnessCompensationIncrementPeriodMilliseconds = 20;
			FrameFormat format = camera.getVideoFormat();
			final BufferedImage baseImage = ImageUtils.getBufferedImage(
					NO_CAMERA_IMAGE.getScaledInstance(format.width, format.height, Image.SCALE_SMOOTH));

			@Override
			public void run() {
				while (!mustStop) {
					BufferedImage image = baseImage;
					image = addAnimationEffect(image);
					camera.handleNewImageFromDevice(image);
					MiscUtils.sleepSafely(brightnessCompensationIncrementPeriodMilliseconds);
				}
				super.run();
			}

			BufferedImage addAnimationEffect(BufferedImage image) {
				image = ImageUtils.changeBrightnessAndContrast(image, brightnessCompensation, 0, Color.GRAY);
				brightnessCompensation += brightnessCompensationIncrement;
				if (Math.abs(brightnessCompensation) == brightnessCompensationMaxAmount) {
					brightnessCompensationIncrement = -brightnessCompensationIncrement;
				}
				return image;
			}

		};
		captureThread.setDaemon(true);
		captureThread.setPriority(Thread.MIN_PRIORITY);
		captureThread.start();
	}

	@Override
	public void stopCapture() {
		mustStop = true;
	}

	@Override
	public void doWaitForCompleteInterruption() {
		while (captureThread.isAlive()) {
			MiscUtils.relieveCPU();
		}
		captureThread = null;
	}

	@Override
	public List<FrameFormat> getVideoFormatsWhileCameraInitializedAndNotActive() {
		return FORMATS;
	}

	@Override
	public List<String> listDeviceLocalNamesWhileCameraNotInitialized() {
		return Collections.singletonList(LOCAL_DEVICE_NAME);
	}

}
