/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.otk.application.error.AbstractApplicationError;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.util.Accessor;
import com.otk.application.util.CommandLineInterface;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.Listener;
import com.otk.application.util.MiscUtils;

/**
 * Base class allowing to implement {@link Camera} drivers using command line
 * tools.
 * 
 * @author olitank
 *
 */
public abstract class AbstractCommandLineBasedCameraDriver extends AbstractCameraDriver {

	protected CameraControlInterface<Void> liveViewProcess;
	protected boolean liveViewInterruptionRequested;
	protected CameraControlInterface<?> synchronizedProcess;
	protected List<CameraControlInterface<?>> nonSynchronizedProcesses = Collections
			.synchronizedList(new ArrayList<AbstractCommandLineBasedCameraDriver.CameraControlInterface<?>>());

	protected String lastSetUpDeviceLocalName;

	protected abstract String cleanLogs(String outputLogs);

	protected abstract CameraControlInterface<Void> getCompatibilityCheckingCommandInterface();

	protected abstract CameraControlInterface<List<String>> getDeviceListingCommandInterface();

	protected abstract CommandLineInterface<List<FrameFormat>> getVideoFormatListingCommandInterface(
			String localDeviceName);

	protected abstract CameraControlInterface<Void> getLiveViewCommandInterface(String localDeviceName,
			FrameFormat format, Listener<BufferedImage> deviceImageListener, Accessor<Boolean> interruptionRequested);

	protected abstract boolean isErrorDiagnosticEnabled();

	public abstract boolean isActive();

	public abstract void setActive(boolean b);

	public AbstractCommandLineBasedCameraDriver(Camera camera) {
		super(camera);
		ensureRunningProcessDestroyedOnExit();
	}

	protected String getLastSetUpDeviceLocalName() {
		return lastSetUpDeviceLocalName;
	}

	protected void ensureRunningProcessDestroyedOnExit() {
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				try {
					if (synchronizedProcess != null) {
						synchronizedProcess.getCommandExecutor().killProcess();
					}
				} catch (Throwable ignore) {
				}
				for (CameraControlInterface<?> process : nonSynchronizedProcesses) {
					try {
						process.getCommandExecutor().killProcess();
					} catch (Throwable ignore) {
					}
				}
			}

		});
	}

	public void checkCompatibility() {
		if (!isActive()) {
			return;
		}
		if (camera.isVerbose()) {
			camera.logInfo("Checking digital camera support");
		}
		getCompatibilityCheckingCommandInterface().execute();
	}

	@Override
	public void doSetUp() throws Exception {
		List<String> connectedDevices = listDeviceLocalNamesWhileCameraNotInitialized();
		String configuredDeviceLocalName = camera.getLocalDeviceName();
		if (!connectedDevices.contains(configuredDeviceLocalName)) {
			throw new StandardError("Camera device not found: '" + configuredDeviceLocalName + "'");
		}
		lastSetUpDeviceLocalName = configuredDeviceLocalName;
	}

	@Override
	public void doCleanUp() throws Exception {
	}

	@Override
	public List<String> listDeviceLocalNamesWhileCameraNotInitialized() {
		if (!isActive()) {
			return Collections.emptyList();
		}
		checkCompatibility();
		return getDeviceListingCommandInterface().execute();
	}

	@Override
	public List<FrameFormat> getVideoFormatsWhileCameraInitializedAndNotActive() {
		return getVideoFormatListingCommandInterface(camera.getLocalDeviceName()).execute();
	}

	protected BufferedImage adaptDeviceImage(BufferedImage inputImage, FrameFormat targetFormat) {
		BufferedImage result = new BufferedImage(targetFormat.getWidth(), targetFormat.getHeight(),
				ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		g2d.translate(result.getWidth(), 0);
		double xScale = -((double) result.getWidth()) / ((double) inputImage.getWidth());
		double yScale = ((double) result.getHeight()) / ((double) inputImage.getHeight());
		g2d.scale(xScale, yScale);
		g2d.drawImage(inputImage, getDeviceImagePrecorrectionOp(inputImage), 0, 0);
		g2d.dispose();
		return result;
	}

	@Override
	public void startCapture() {
		liveViewInterruptionRequested = false;
		Listener<BufferedImage> deviceImageListener = new Listener<BufferedImage>() {
			@Override
			public void handle(BufferedImage newImage) {
				BufferedImage adaptedImage = adaptDeviceImage(newImage, camera.getVideoFormat());
				camera.handleNewImageFromDevice(adaptedImage);
			}
		};
		Accessor<Boolean> interruptionRequested = new Accessor<Boolean>() {

			@Override
			public Boolean get() {
				return liveViewInterruptionRequested;
			}
		};
		liveViewProcess = getLiveViewCommandInterface(camera.getLocalDeviceName(), camera.getVideoFormat(),
				deviceImageListener, interruptionRequested);
		liveViewProcess.executeAsynchronously();
		while (true) {
			if (liveViewProcess.getCommandExecutor() != null) {
				if (liveViewProcess.getCommandExecutor().getLaunchedProcess() != null) {
					break;
				}
			}
			MiscUtils.relieveCPU();
		}
	}

	@Override
	public void stopCapture() {
		if (!liveViewProcess.isExecuting()) {
			throw new StandardError("The live view was abnormally interrupted");
		}
		liveViewInterruptionRequested = true;
	}

	@Override
	public void doWaitForCompleteInterruption() {
		try {
			liveViewProcess.waitForExecutionEnd();
		} catch (Throwable t) {
			liveViewProcess.requestExecutionInterruption();
			throw new StandardError("Failed to trigger the live view interruption normally: " + t.toString(), t);
		} finally {
			liveViewProcess = null;
		}
	}

	public abstract class CameraControlInterface<TT> extends CommandLineInterface<TT> {

		private boolean processKilledAfterCommunication;
		private boolean processSynchronized;

		protected abstract TT doCommunicate(Process process);

		public CameraControlInterface(File executableFile) {
			super(new File("."), executableFile, "");
			setSafeReadingTimeoutMilliseconds(camera.getFreezeTimeoutMilliSeconds());
		}

		@Override
		final protected TT communicate(Process process) {
			try {
				TT result = doCommunicate(process);
				if (!isProcessKilledAfterCommunication()) {
					waitSafelyFor(process, camera.getFreezeTimeoutMilliSeconds());
				}
				return result;
			} catch (Throwable t) {
				if (isErrorDiagnosticEnabled()) {
					logErrorDiagnosticData(t);
				}
				throw AbstractApplicationError.getToRethrow(t);
			}
		}

		protected boolean isProcessKilledAfterCommunication() {
			return processKilledAfterCommunication;
		}

		public void setProcessKilledAfterCommunication(boolean processKilledAfterCommunication) {
			this.processKilledAfterCommunication = processKilledAfterCommunication;
		}

		protected boolean isProcessSynchronized() {
			return processSynchronized;
		}

		public void setProcessSynchronized(boolean processSynchronized) {
			this.processSynchronized = processSynchronized;
		}

		@Override
		protected Thread createOrchestrationThread() {
			Thread result = super.createOrchestrationThread();
			result.setDaemon(true);
			result.setPriority((Integer) MiscUtils.getFieldValue(Thread.class, camera.getThreadPriority(), null));
			return result;
		}

		@Override
		protected Thread createSafeReadingThread(Runnable runnable) {
			Thread result = super.createSafeReadingThread(runnable);
			result.setDaemon(true);
			result.setPriority((Integer) MiscUtils.getFieldValue(Thread.class, camera.getThreadPriority(), null));
			return result;
		}

		protected void logErrorDiagnosticData(Throwable t) {
			try {
				camera.logError(t.toString() + "\nContext:\n" + collectAvailableDiagnosticData());
			} catch (Throwable t2) {
				camera.logError(MiscUtils.getPrintedStackTrace(t));
				camera.logError("Failed to log diagnostic data: " + t2);
			}
		}

		@Override
		protected void orchestrate() {
			if (!isProcessSynchronized()) {
				AbstractCommandLineBasedCameraDriver.this.nonSynchronizedProcesses.add(CameraControlInterface.this);
				try {
					super.orchestrate();
				} finally {
					AbstractCommandLineBasedCameraDriver.this.nonSynchronizedProcesses
							.remove(CameraControlInterface.this);
				}
			} else {
				if (AbstractCommandLineBasedCameraDriver.this.synchronizedProcess != null) {
					throw new UnexpectedError("Cannot run camera control process <" + getCommandOptions()
							+ ">. A camera control process was found already running: <"
							+ AbstractCommandLineBasedCameraDriver.this.synchronizedProcess.getCommandOptions() + ">");
				}
				AbstractCommandLineBasedCameraDriver.this.synchronizedProcess = CameraControlInterface.this;
				try {
					super.orchestrate();
				} finally {
					AbstractCommandLineBasedCameraDriver.this.synchronizedProcess = null;
				}
			}
		}

		@Override
		protected String cleanLogs(String outputLogs) {
			return AbstractCommandLineBasedCameraDriver.this.cleanLogs(outputLogs);
		}

		@Override
		public String toString() {
			return "CameraControlProcess [" + getCommandOptions() + "]";
		}

	}

}
