/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application;

import java.awt.Frame;

import javax.swing.JDialog;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;

import com.otk.application.error.ErrorManager;
import com.otk.application.error.UnexpectedError;
import com.otk.application.util.Analytics;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.Log;

/**
 * This class serves as the main entry point of the application. It handles all
 * initialization and cleanup tasks.
 * 
 * @author olitank
 *
 */
public class Main {

	public static void main(String[] args) {
		initializeApplication();
		Runtime.getRuntime().addShutdownHook(new Thread() {
			@Override
			public void run() {
				finalizeApplication();
			}

		});
		Log.info("Build Id: " + BuildProperties.get().getId());
		SwingUtilities.invokeLater(new Runnable() {
			@Override
			public void run() {
				IdPhotoUI.RENDERER.openObjectFrame(new IdPhotoProject());
			}
		});
	}

	/**
	 * System property name allowing to identify at runtime the type of packaging
	 * used for the application.
	 */
	public static final String LAUNCHER_PROPERTY_KEY = Main.class.getPackage().getName() + ".launcher";

	/**
	 * The support website.
	 */
	public static final String WEBSITE_URL = "http://www.phoyosystem.com";

	/**
	 * Whether the application is initialized or not.
	 */
	private static boolean initialized = false;

	/**
	 * Sets up the application.
	 */
	private static void initializeApplication() {
		if (initialized) {
			return;
		}
		checkCompatibility();
		ErrorManager.initialize();
		Analytics.initialize();
		initialized = true;
		Log.info("Starting application");
		Analytics.track("Start",
				"os.name=" + System.getProperty("os.name") + ", os.arch=" + System.getProperty("os.arch") + ", "
						+ LAUNCHER_PROPERTY_KEY + "=" + System.getProperty(LAUNCHER_PROPERTY_KEY) + ", java.home="
						+ System.getProperty("java.home"));
	}

	/**
	 * Checks the compatibility of the application with the current environment. An
	 * exception is thrown if an issue is detected.
	 */
	private static void checkCompatibility() {
		String javaVersion = System.getProperty("java.version");
		if (javaVersion == null) {
			return;
		}
		if (!javaVersion.startsWith("1.6") && !javaVersion.startsWith("1.7") && !javaVersion.startsWith("1.8")) {
			handleCompatibilityIssue("Java 1.6 to 1.8 is required: Current java version: " + javaVersion);
		}
	}

	/**
	 * Handle compatibility issues (display, tracking, ...).
	 * 
	 * @param errorMessage
	 *            The compatibility error message.
	 */
	private static void handleCompatibilityIssue(String errorMessage) {
		try {
			Analytics.initialize();
			Analytics.track("CompatibilityIssue", errorMessage);
		} catch (Throwable ignore) {
		}
		try {
			JOptionPane pane = new JOptionPane(errorMessage, JOptionPane.ERROR_MESSAGE);
			JDialog dialog = pane.createDialog(null, "Compatibility Issue");
			((Frame) dialog.getParent()).setIconImage(ImageUtils.NULL_IMAGE);
			dialog.setResizable(true);
			dialog.setVisible(true);
		} catch (Throwable ignore) {
		}
		throw new UnexpectedError(errorMessage);
	}

	/**
	 * Cleans up the application.
	 */
	private static void finalizeApplication() {
		Log.info("Exiting application");
		Analytics.track("Exit");
		if (!initialized) {
			return;
		}
		try {
			Analytics.shutdown();
		} catch (Throwable ignore) {
		}
		initialized = false;
	}

}
