﻿// 
// Hiero
// Copyright (c) 2015  Barry Block 
// 
// This program is free software: you can redistribute it and/or modify it under
// the terms of the GNU General Public License as published by the Free Software
// Foundation, either version 3 of the License, or (at your option) any later
// version. 
// 
// This program is distributed in the hope that it will be useful, but WITHOUT ANY
// WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
// PARTICULAR PURPOSE.  See the GNU General Public License for more details. 
// 
// You should have received a copy of the GNU General Public License along with
// this program.  If not, see <http://www.gnu.org/licenses/>. 
//

using System;
using System.IO;
using Gtk;

namespace Hiero
{
   class MainClass
   {
      // The static fields declared here are effectively application globals
      // as each is referenced by multiple classes (and this ought to be better
      // managed).
       
      internal static string appName = "Hiero";
         // The applications name.
      internal static bool firstRun;
         // Stores whether this is the first run of the app. or not.
      internal static string homeFolder = "/home/" + System.Environment.UserName;
         // The user's home folder.
      internal static string appFolder = homeFolder + "/." + appName.ToLower();
         // The app's main folder.
      internal static string logFolder = appFolder + "/" + "Log";
         // The app's logging folder;
      internal static string crashFolder = appFolder + "/Crashes";
         // The app's crash folder.
      internal static string lockFile = homeFolder + "/." + appName.ToLower() + "/instlock";
         // File used to determine if Hiero instance is already running.
      internal static string newOutlinesFolder = appFolder + "/New";
         // Folder where all "new" outlines reside (as log files).
      internal static string newNodePlaceholder = "[new node]";
         // The node's initial text.
      internal static string initRootText = "This is your outline's root node. To edit it, either double-click it or press SPACE.";
         // The initial text used for the root of a new outline.
      internal static string escapedNewline = "&#xA;";  
         // Acts as a stand in for real newlines when an outline is stored to disk.

      public static void Main(string[] args)
      {
         Application.Init();
         ResponseType response = ResponseType.No;

         // Create the "warning" dialog:

         MessageDialog warningMesgDlg = new MessageDialog(null, DialogFlags.DestroyWithParent, 
         MessageType.Warning, ButtonsType.YesNo, "The previous " + appName + 
         " session either terminated unexpectedly or another instance of " + appName + 
         " is currently running (and only one is allowed). If " + appName + " terminated, you can " +
         "choose \"Yes\" to continue. Any outline document that you had open during the previous session " + 
         "will be re-opened and those documents that had unsaved changes will be recovered to the point at " +
         "which termination occurred and you can then continue editing them (see the topic, \"Crash Recovery\" in " + 
         appName + "'s user guide for more information).\n\nNote: For each outline recovered, the original " +
         "outline and its command log will be copied to the folder, \"" + crashFolder + "\" in case " +
         "you'd like to contact the software's author to report a bug.\n\nIf previously, " + appName + 
         " didn't terminate and instead, another instance of it is currently running, you can either choose " +
         "\"No\" to prevent this one from loading or you can close the other instance and then press " +
         "\"Yes\".\n\nDo you want to continue?");

         warningMesgDlg.WindowPosition = WindowPosition.Center;
         
         // Determine if Hiero has been run before:
         
         firstRun = !Directory.Exists(appFolder);
         if (firstRun)
         {
            // Hiero hasn't been run before. Create Hiero's folder and subfolders
            // (Note: Hiero's top-level folder, "appFolder" is automatically created 
            // whenever the first of its subfolders is created):
   
            Directory.CreateDirectory(crashFolder); 
            Directory.CreateDirectory(logFolder); 
            Directory.CreateDirectory(newOutlinesFolder); 
         }

         // Display the crash dialog if, during the prior session, Hiero 
         // crashed or another instance of it is currently running:
         
         try 
         {
            if (File.Exists(lockFile)) 
            {
               // Display dialog:

               response = (ResponseType) warningMesgDlg.Run();
            }
         }
         finally 
         {
            warningMesgDlg.Destroy();
         }

         if (!File.Exists(lockFile) || (response == ResponseType.Yes))
         {
            // Hiero isn't already running (but it may 
            // have previously crashed). Start it up:

            if (!File.Exists(lockFile)) File.Create(lockFile);  // Note that I didn't use "appIsLocked" for 
            MainWindow win = new MainWindow(args);              // this "if" statement. That's because the 
            win.Show();                                         // user could close a potential other 
            Application.Run();                                  // instance of Hiero while dialog is displayed
            if (File.Exists(lockFile)) File.Delete(lockFile);   // causing instlock file to be deleted!
         }
      }
   }
}
