/*  csc_iso639.c - convert table of ISO639

    Copyright (C) 2013  "Andy Xuming" <xuming@users.sourceforge.net>

    This file is part of CSOUP, Chicken Soup library

    CSOUP is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    CSOUP is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* see http://www.loc.gov/standards/iso639-2/php/code_list.php */
struct	_ISO639LUT	{
	char		*iso639_2;
	char		*iso639_1;
	char		*name;
} ISO639LUT[] = {
	{ "aar",	"aa",	"Afar" },
	{ "abk",	"ab",	"Abkhazian" },
	{ "ace",	"",	"Achinese" },
	{ "ach",	"",	"Acoli" },
	{ "ada",	"",	"Adangme" },
	{ "ady",	"",	"Adyghe;" },
	{ "afa",	"",	"Afro-Asiatic" },
	{ "afh",	"",	"Afrihili" },
	{ "afr",	"af",	"Afrikaans" },
	{ "ain",	"",	"Ainu" },
	{ "aka",	"ak",	"Akan" },
	{ "akk",	"",	"Akkadian" },
	{ "alb",	"sq",	"Albanian" },
	{ "sqi",	"sq",	"Albanian" },
	{ "ale",	"",	"Aleut" },
	{ "alg",	"",	"Algonquian" },
	{ "alt",	"",	"Southern" },
	{ "amh",	"am",	"Amharic" },
	{ "ang",	"",	"English," },
	{ "anp",	"",	"Angika" },
	{ "apa",	"",	"Apache" },
	{ "ara",	"ar",	"Arabic" },
	{ "arc",	"",	"Official" },
	{ "arg",	"an",	"Aragonese" },
	{ "arm",	"hy",	"Armenian" },
	{ "hye",	"hy",	"Armenian" },
	{ "arn",	"",	"Mapudungun;" },
	{ "arp",	"",	"Arapaho" },
	{ "art",	"",	"Artificial" },
	{ "arw",	"",	"Arawak" },
	{ "asm",	"as",	"Assamese" },
	{ "ast",	"",	"Asturian;" },
	{ "ath",	"",	"Athapascan" },
	{ "aus",	"",	"Australian" },
	{ "ava",	"av",	"Avaric" },
	{ "ave",	"ae",	"Avestan" },
	{ "awa",	"",	"Awadhi" },
	{ "aym",	"ay",	"Aymara" },
	{ "aze",	"az",	"Azerbaijani" },
	{ "bad",	"",	"Banda" },
	{ "bai",	"",	"Bamileke" },
	{ "bak",	"ba",	"Bashkir" },
	{ "bal",	"",	"Baluchi" },
	{ "bam",	"bm",	"Bambara" },
	{ "ban",	"",	"Balinese" },
	{ "baq",	"eu",	"Basque" },
	{ "eus",	"eu",	"Basque" },
	{ "bas",	"",	"Basa" },
	{ "bat",	"",	"Baltic" },
	{ "bej",	"",	"Beja;" },
	{ "bel",	"be",	"Belarusian" },
	{ "bem",	"",	"Bemba" },
	{ "ben",	"bn",	"Bengali" },
	{ "ber",	"",	"Berber" },
	{ "bho",	"",	"Bhojpuri" },
	{ "bih",	"bh",	"Bihari" },
	{ "bik",	"",	"Bikol" },
	{ "bin",	"",	"Bini;" },
	{ "bis",	"bi",	"Bislama" },
	{ "bla",	"",	"Siksika" },
	{ "bnt",	"",	"Bantu" },
	{ "tib",	"bo",	"Tibetan" },
	{ "bod",	"bo",	"Tibetan" },
	{ "bos",	"bs",	"Bosnian" },
	{ "bra",	"",	"Braj" },
	{ "bre",	"br",	"Breton" },
	{ "btk",	"",	"Batak" },
	{ "bua",	"",	"Buriat" },
	{ "bug",	"",	"Buginese" },
	{ "bul",	"bg",	"Bulgarian" },
	{ "bur",	"my",	"Burmese" },
	{ "mya",	"my",	"Burmese" },
	{ "byn",	"",	"Blin;" },
	{ "cad",	"",	"Caddo" },
	{ "cai",	"",	"Central" },
	{ "car",	"",	"Galibi" },
	{ "cat",	"ca",	"Catalan;" },
	{ "cau",	"",	"Caucasian" },
	{ "ceb",	"",	"Cebuano" },
	{ "cel",	"",	"Celtic" },
	{ "cze",	"cs",	"Czech" },
	{ "ces",	"cs",	"Czech" },
	{ "cha",	"ch",	"Chamorro" },
	{ "chb",	"",	"Chibcha" },
	{ "che",	"ce",	"Chechen" },
	{ "chg",	"",	"Chagatai" },
	{ "chi",	"zh",	"Chinese" },
	{ "zho",	"zh",	"Chinese" },
	{ "chk",	"",	"Chuukese" },
	{ "chm",	"",	"Mari" },
	{ "chn",	"",	"Chinook" },
	{ "cho",	"",	"Choctaw" },
	{ "chp",	"",	"Chipewyan;" },
	{ "chr",	"",	"Cherokee" },
	{ "chu",	"cu",	"Church" },
	{ "chv",	"cv",	"Chuvash" },
	{ "chy",	"",	"Cheyenne" },
	{ "cmc",	"",	"Chamic" },
	{ "cop",	"",	"Coptic" },
	{ "cor",	"kw",	"Cornish" },
	{ "cos",	"co",	"Corsican" },
	{ "cpe",	"",	"Creoles" },
	{ "cpf",	"",	"Creoles" },
	{ "cpp",	"",	"Creoles" },
	{ "cre",	"cr",	"Cree" },
	{ "crh",	"",	"Crimean" },
	{ "crp",	"",	"Creoles" },
	{ "csb",	"",	"Kashubian" },
	{ "cus",	"",	"Cushitic" },
	{ "wel",	"cy",	"Welsh" },
	{ "cym",	"cy",	"Welsh" },
	{ "cze",	"cs",	"Czech" },
	{ "ces",	"cs",	"Czech" },
	{ "dak",	"",	"Dakota" },
	{ "dan",	"da",	"Danish" },
	{ "dar",	"",	"Dargwa" },
	{ "day",	"",	"Land" },
	{ "del",	"",	"Delaware" },
	{ "den",	"",	"Slave" },
	{ "ger",	"de",	"German" },
	{ "deu",	"de",	"German" },
	{ "dgr",	"",	"Dogrib" },
	{ "din",	"",	"Dinka" },
	{ "div",	"dv",	"Divehi;" },
	{ "doi",	"",	"Dogri" },
	{ "dra",	"",	"Dravidian" },
	{ "dsb",	"",	"Lower" },
	{ "dua",	"",	"Duala" },
	{ "dum",	"",	"Dutch," },
	{ "dut",	"nl",	"Dutch;" },
	{ "nld",	"nl",	"Dutch;" },
	{ "dyu",	"",	"Dyula" },
	{ "dzo",	"dz",	"Dzongkha" },
	{ "efi",	"",	"Efik" },
	{ "egy",	"",	"Egyptian" },
	{ "eka",	"",	"Ekajuk" },
	{ "gre",	"el",	"Greek," },
	{ "ell",	"el",	"Greek," },
	{ "elx",	"",	"Elamite" },
	{ "eng",	"en",	"English" },
	{ "enm",	"",	"English," },
	{ "epo",	"eo",	"Esperanto" },
	{ "est",	"et",	"Estonian" },
	{ "baq",	"eu",	"Basque" },
	{ "eus",	"eu",	"Basque" },
	{ "ewe",	"ee",	"Ewe" },
	{ "ewo",	"",	"Ewondo" },
	{ "fan",	"",	"Fang" },
	{ "fao",	"fo",	"Faroese" },
	{ "per",	"fa",	"Persian" },
	{ "fas",	"fa",	"Persian" },
	{ "fat",	"",	"Fanti" },
	{ "fij",	"fj",	"Fijian" },
	{ "fil",	"",	"Filipino;" },
	{ "fin",	"fi",	"Finnish" },
	{ "fiu",	"",	"Finno-Ugrian" },
	{ "fon",	"",	"Fon" },
	{ "fre",	"fr",	"French" },
	{ "fra",	"fr",	"French" },
	{ "fre",	"fr",	"French" },
	{ "fra",	"fr",	"French" },
	{ "frm",	"",	"French," },
	{ "fro",	"",	"French," },
	{ "frr",	"",	"Northern" },
	{ "frs",	"",	"Eastern" },
	{ "fry",	"fy",	"Western" },
	{ "ful",	"ff",	"Fulah" },
	{ "fur",	"",	"Friulian" },
	{ "gaa",	"",	"Ga" },
	{ "gay",	"",	"Gayo" },
	{ "gba",	"",	"Gbaya" },
	{ "gem",	"",	"Germanic" },
	{ "geo",	"ka",	"Georgian" },
	{ "kat",	"ka",	"Georgian" },
	{ "ger",	"de",	"German" },
	{ "deu",	"de",	"German" },
	{ "gez",	"",	"Geez" },
	{ "gil",	"",	"Gilbertese" },
	{ "gla",	"gd",	"Gaelic;" },
	{ "gle",	"ga",	"Irish" },
	{ "glg",	"gl",	"Galician" },
	{ "glv",	"gv",	"Manx" },
	{ "gmh",	"",	"German," },
	{ "goh",	"",	"German," },
	{ "gon",	"",	"Gondi" },
	{ "gor",	"",	"Gorontalo" },
	{ "got",	"",	"Gothic" },
	{ "grb",	"",	"Grebo" },
	{ "grc",	"",	"Greek," },
	{ "gre",	"el",	"Greek," },
	{ "ell",	"el",	"Greek," },
	{ "grn",	"gn",	"Guarani" },
	{ "gsw",	"",	"Swiss" },
	{ "guj",	"gu",	"Gujarati" },
	{ "gwi",	"",	"Gwich'in" },
	{ "hai",	"",	"Haida" },
	{ "hat",	"ht",	"Haitian;" },
	{ "hau",	"ha",	"Hausa" },
	{ "haw",	"",	"Hawaiian" },
	{ "heb",	"he",	"Hebrew" },
	{ "her",	"hz",	"Herero" },
	{ "hil",	"",	"Hiligaynon" },
	{ "him",	"",	"Himachali" },
	{ "hin",	"hi",	"Hindi" },
	{ "hit",	"",	"Hittite" },
	{ "hmn",	"",	"Hmong;" },
	{ "hmo",	"ho",	"Hiri" },
	{ "hrv",	"hr",	"Croatian" },
	{ "hsb",	"",	"Upper" },
	{ "hun",	"hu",	"Hungarian" },
	{ "hup",	"",	"Hupa" },
	{ "arm",	"hy",	"Armenian" },
	{ "hye",	"hy",	"Armenian" },
	{ "iba",	"",	"Iban" },
	{ "ibo",	"ig",	"Igbo" },
	{ "ice",	"is",	"Icelandic" },
	{ "isl",	"is",	"Icelandic" },
	{ "ido",	"io",	"Ido" },
	{ "iii",	"ii",	"Sichuan" },
	{ "ijo",	"",	"Ijo" },
	{ "iku",	"iu",	"Inuktitut" },
	{ "ile",	"ie",	"Interlingue;" },
	{ "ilo",	"",	"Iloko" },
	{ "ina",	"ia",	"Interlingua" },
	{ "inc",	"",	"Indic" },
	{ "ind",	"id",	"Indonesian" },
	{ "ine",	"",	"Indo-European" },
	{ "inh",	"",	"Ingush" },
	{ "ipk",	"ik",	"Inupiaq" },
	{ "ira",	"",	"Iranian" },
	{ "iro",	"",	"Iroquoian" },
	{ "ice",	"is",	"Icelandic" },
	{ "isl",	"is",	"Icelandic" },
	{ "ita",	"it",	"Italian" },
	{ "jav",	"jv",	"Javanese" },
	{ "jbo",	"",	"Lojban" },
	{ "jpn",	"ja",	"Japanese" },
	{ "jpr",	"",	"Judeo-Persian" },
	{ "jrb",	"",	"Judeo-Arabic" },
	{ "kaa",	"",	"Kara-Kalpak" },
	{ "kab",	"",	"Kabyle" },
	{ "kac",	"",	"Kachin;" },
	{ "kal",	"kl",	"Kalaallisut;" },
	{ "kam",	"",	"Kamba" },
	{ "kan",	"kn",	"Kannada" },
	{ "kar",	"",	"Karen" },
	{ "kas",	"ks",	"Kashmiri" },
	{ "geo",	"ka",	"Georgian" },
	{ "kat",	"ka",	"Georgian" },
	{ "kau",	"kr",	"Kanuri" },
	{ "kaw",	"",	"Kawi" },
	{ "kaz",	"kk",	"Kazakh" },
	{ "kbd",	"",	"Kabardian" },
	{ "kha",	"",	"Khasi" },
	{ "khi",	"",	"Khoisan" },
	{ "khm",	"km",	"Central" },
	{ "kho",	"",	"Khotanese;" },
	{ "kik",	"ki",	"Kikuyu;" },
	{ "kin",	"rw",	"Kinyarwanda" },
	{ "kir",	"ky",	"Kirghiz;" },
	{ "kmb",	"",	"Kimbundu" },
	{ "kok",	"",	"Konkani" },
	{ "kom",	"kv",	"Komi" },
	{ "kon",	"kg",	"Kongo" },
	{ "kor",	"ko",	"Korean" },
	{ "kos",	"",	"Kosraean" },
	{ "kpe",	"",	"Kpelle" },
	{ "krc",	"",	"Karachay-Balkar" },
	{ "krl",	"",	"Karelian" },
	{ "kro",	"",	"Kru" },
	{ "kru",	"",	"Kurukh" },
	{ "kua",	"kj",	"Kuanyama;" },
	{ "kum",	"",	"Kumyk" },
	{ "kur",	"ku",	"Kurdish" },
	{ "kut",	"",	"Kutenai" },
	{ "lad",	"",	"Ladino" },
	{ "lah",	"",	"Lahnda" },
	{ "lam",	"",	"Lamba" },
	{ "lao",	"lo",	"Lao" },
	{ "lat",	"la",	"Latin" },
	{ "lav",	"lv",	"Latvian" },
	{ "lez",	"",	"Lezghian" },
	{ "lim",	"li",	"Limburgan;" },
	{ "lin",	"ln",	"Lingala" },
	{ "lit",	"lt",	"Lithuanian" },
	{ "lol",	"",	"Mongo" },
	{ "loz",	"",	"Lozi" },
	{ "ltz",	"lb",	"Luxembourgish;" },
	{ "lua",	"",	"Luba-Lulua" },
	{ "lub",	"lu",	"Luba-Katanga" },
	{ "lug",	"lg",	"Ganda" },
	{ "lui",	"",	"Luiseno" },
	{ "lun",	"",	"Lunda" },
	{ "luo",	"",	"Luo" },
	{ "lus",	"",	"Lushai" },
	{ "mac",	"mk",	"Macedonian" },
	{ "mkd",	"mk",	"Macedonian" },
	{ "mad",	"",	"Madurese" },
	{ "mag",	"",	"Magahi" },
	{ "mah",	"mh",	"Marshallese" },
	{ "mai",	"",	"Maithili" },
	{ "mak",	"",	"Makasar" },
	{ "mal",	"ml",	"Malayalam" },
	{ "man",	"",	"Mandingo" },
	{ "mao",	"mi",	"Maori" },
	{ "mri",	"mi",	"Maori" },
	{ "map",	"",	"Austronesian" },
	{ "mar",	"mr",	"Marathi" },
	{ "mas",	"",	"Masai" },
	{ "may",	"ms",	"Malay" },
	{ "msa",	"ms",	"Malay" },
	{ "mdf",	"",	"Moksha" },
	{ "mdr",	"",	"Mandar" },
	{ "men",	"",	"Mende" },
	{ "mga",	"",	"Irish," },
	{ "mic",	"",	"Mi'kmaq;" },
	{ "min",	"",	"Minangkabau" },
	{ "mis",	"",	"Uncoded" },
	{ "mac",	"mk",	"Macedonian" },
	{ "mkd",	"mk",	"Macedonian" },
	{ "mkh",	"",	"Mon-Khmer" },
	{ "mlg",	"mg",	"Malagasy" },
	{ "mlt",	"mt",	"Maltese" },
	{ "mnc",	"",	"Manchu" },
	{ "mni",	"",	"Manipuri" },
	{ "mno",	"",	"Manobo" },
	{ "moh",	"",	"Mohawk" },
	{ "mon",	"mn",	"Mongolian" },
	{ "mos",	"",	"Mossi" },
	{ "mao",	"mi",	"Maori" },
	{ "mri",	"mi",	"Maori" },
	{ "may",	"ms",	"Malay" },
	{ "msa",	"ms",	"Malay" },
	{ "mul",	"",	"Multiple" },
	{ "mun",	"",	"Munda" },
	{ "mus",	"",	"Creek" },
	{ "mwl",	"",	"Mirandese" },
	{ "mwr",	"",	"Marwari" },
	{ "bur",	"my",	"Burmese" },
	{ "mya",	"my",	"Burmese" },
	{ "myn",	"",	"Mayan" },
	{ "myv",	"",	"Erzya" },
	{ "nah",	"",	"Nahuatl" },
	{ "nai",	"",	"North" },
	{ "nap",	"",	"Neapolitan" },
	{ "nau",	"na",	"Nauru" },
	{ "nav",	"nv",	"Navajo;" },
	{ "nbl",	"nr",	"Ndebele," },
	{ "nde",	"nd",	"Ndebele," },
	{ "ndo",	"ng",	"Ndonga" },
	{ "nds",	"",	"Low" },
	{ "nep",	"ne",	"Nepali" },
	{ "new",	"",	"Nepal" },
	{ "nia",	"",	"Nias" },
	{ "nic",	"",	"Niger-Kordofanian" },
	{ "niu",	"",	"Niuean" },
	{ "dut",	"nl",	"Dutch;" },
	{ "nld",	"nl",	"Dutch;" },
	{ "nno",	"nn",	"Norwegian" },
	{ "nob",	"nb",	"Bokml," },
	{ "nog",	"",	"Nogai" },
	{ "non",	"",	"Norse," },
	{ "nor",	"no",	"Norwegian" },
	{ "nqo",	"",	"N'Ko" },
	{ "nso",	"",	"Pedi;" },
	{ "nub",	"",	"Nubian" },
	{ "nwc",	"",	"Classical" },
	{ "nya",	"ny",	"Chichewa;" },
	{ "nym",	"",	"Nyamwezi" },
	{ "nyn",	"",	"Nyankole" },
	{ "nyo",	"",	"Nyoro" },
	{ "nzi",	"",	"Nzima" },
	{ "oci",	"oc",	"Occitan" },
	{ "oji",	"oj",	"Ojibwa" },
	{ "ori",	"or",	"Oriya" },
	{ "orm",	"om",	"Oromo" },
	{ "osa",	"",	"Osage" },
	{ "oss",	"os",	"Ossetian;" },
	{ "ota",	"",	"Turkish," },
	{ "oto",	"",	"Otomian" },
	{ "paa",	"",	"Papuan" },
	{ "pag",	"",	"Pangasinan" },
	{ "pal",	"",	"Pahlavi" },
	{ "pam",	"",	"Pampanga;" },
	{ "pan",	"pa",	"Panjabi;" },
	{ "pap",	"",	"Papiamento" },
	{ "pau",	"",	"Palauan" },
	{ "peo",	"",	"Persian," },
	{ "per",	"fa",	"Persian" },
	{ "fas",	"fa",	"Persian" },
	{ "phi",	"",	"Philippine" },
	{ "phn",	"",	"Phoenician" },
	{ "pli",	"pi",	"Pali" },
	{ "pol",	"pl",	"Polish" },
	{ "pon",	"",	"Pohnpeian" },
	{ "por",	"pt",	"Portuguese" },
	{ "pra",	"",	"Prakrit" },
	{ "pro",	"",	"Provenal," },
	{ "pus",	"ps",	"Pushto;" },
	{ "que",	"qu",	"Quechua" },
	{ "raj",	"",	"Rajasthani" },
	{ "rap",	"",	"Rapanui" },
	{ "rar",	"",	"Rarotongan;" },
	{ "roa",	"",	"Romance" },
	{ "roh",	"rm",	"Romansh" },
	{ "rom",	"",	"Romany" },
	{ "rum",	"ro",	"Romanian;" },
	{ "ron",	"ro",	"Romanian;" },
	{ "rum",	"ro",	"Romanian;" },
	{ "ron",	"ro",	"Romanian;" },
	{ "run",	"rn",	"Rundi" },
	{ "rup",	"",	"Aromanian;" },
	{ "rus",	"ru",	"Russian" },
	{ "sad",	"",	"Sandawe" },
	{ "sag",	"sg",	"Sango" },
	{ "sah",	"",	"Yakut" },
	{ "sai",	"",	"South" },
	{ "sal",	"",	"Salishan" },
	{ "sam",	"",	"Samaritan" },
	{ "san",	"sa",	"Sanskrit" },
	{ "sas",	"",	"Sasak" },
	{ "sat",	"",	"Santali" },
	{ "scn",	"",	"Sicilian" },
	{ "sco",	"",	"Scots" },
	{ "sel",	"",	"Selkup" },
	{ "sem",	"",	"Semitic" },
	{ "sga",	"",	"Irish," },
	{ "sgn",	"",	"Sign" },
	{ "shn",	"",	"Shan" },
	{ "sid",	"",	"Sidamo" },
	{ "sin",	"si",	"Sinhala;" },
	{ "sio",	"",	"Siouan" },
	{ "sit",	"",	"Sino-Tibetan" },
	{ "sla",	"",	"Slavic" },
	{ "slo",	"sk",	"Slovak" },
	{ "slk",	"sk",	"Slovak" },
	{ "slo",	"sk",	"Slovak" },
	{ "slk",	"sk",	"Slovak" },
	{ "slv",	"sl",	"Slovenian" },
	{ "sma",	"",	"Southern" },
	{ "sme",	"se",	"Northern" },
	{ "smi",	"",	"Sami" },
	{ "smj",	"",	"Lule" },
	{ "smn",	"",	"Inari" },
	{ "smo",	"sm",	"Samoan" },
	{ "sms",	"",	"Skolt" },
	{ "sna",	"sn",	"Shona" },
	{ "snd",	"sd",	"Sindhi" },
	{ "snk",	"",	"Soninke" },
	{ "sog",	"",	"Sogdian" },
	{ "som",	"so",	"Somali" },
	{ "son",	"",	"Songhai" },
	{ "sot",	"st",	"Sotho," },
	{ "spa",	"es",	"Spanish;" },
	{ "alb",	"sq",	"Albanian" },
	{ "sqi",	"sq",	"Albanian" },
	{ "srd",	"sc",	"Sardinian" },
	{ "srn",	"",	"Sranan" },
	{ "srp",	"sr",	"Serbian" },
	{ "srr",	"",	"Serer" },
	{ "ssa",	"",	"Nilo-Saharan" },
	{ "ssw",	"ss",	"Swati" },
	{ "suk",	"",	"Sukuma" },
	{ "sun",	"su",	"Sundanese" },
	{ "sus",	"",	"Susu" },
	{ "sux",	"",	"Sumerian" },
	{ "swa",	"sw",	"Swahili" },
	{ "swe",	"sv",	"Swedish" },
	{ "syc",	"",	"Classical" },
	{ "syr",	"",	"Syriac" },
	{ "tah",	"ty",	"Tahitian" },
	{ "tai",	"",	"Tai" },
	{ "tam",	"ta",	"Tamil" },
	{ "tat",	"tt",	"Tatar" },
	{ "tel",	"te",	"Telugu" },
	{ "tem",	"",	"Timne" },
	{ "ter",	"",	"Tereno" },
	{ "tet",	"",	"Tetum" },
	{ "tgk",	"tg",	"Tajik" },
	{ "tgl",	"tl",	"Tagalog" },
	{ "tha",	"th",	"Thai" },
	{ "tib",	"bo",	"Tibetan" },
	{ "bod",	"bo",	"Tibetan" },
	{ "tig",	"",	"Tigre" },
	{ "tir",	"ti",	"Tigrinya" },
	{ "tiv",	"",	"Tiv" },
	{ "tkl",	"",	"Tokelau" },
	{ "tlh",	"",	"Klingon;" },
	{ "tli",	"",	"Tlingit" },
	{ "tmh",	"",	"Tamashek" },
	{ "tog",	"",	"Tonga" },
	{ "ton",	"to",	"Tonga" },
	{ "tpi",	"",	"Tok" },
	{ "tsi",	"",	"Tsimshian" },
	{ "tsn",	"tn",	"Tswana" },
	{ "tso",	"ts",	"Tsonga" },
	{ "tuk",	"tk",	"Turkmen" },
	{ "tum",	"",	"Tumbuka" },
	{ "tup",	"",	"Tupi" },
	{ "tur",	"tr",	"Turkish" },
	{ "tut",	"",	"Altaic" },
	{ "tvl",	"",	"Tuvalu" },
	{ "twi",	"tw",	"Twi" },
	{ "tyv",	"",	"Tuvinian" },
	{ "udm",	"",	"Udmurt" },
	{ "uga",	"",	"Ugaritic" },
	{ "uig",	"ug",	"Uighur;" },
	{ "ukr",	"uk",	"Ukrainian" },
	{ "umb",	"",	"Umbundu" },
	{ "und",	"",	"Undetermined" },
	{ "urd",	"ur",	"Urdu" },
	{ "uzb",	"uz",	"Uzbek" },
	{ "vai",	"",	"Vai" },
	{ "ven",	"ve",	"Venda" },
	{ "vie",	"vi",	"Vietnamese" },
	{ "vol",	"vo",	"Volapk" },
	{ "vot",	"",	"Votic" },
	{ "wak",	"",	"Wakashan" },
	{ "wal",	"",	"Wolaitta;" },
	{ "war",	"",	"Waray" },
	{ "was",	"",	"Washo" },
	{ "wel",	"cy",	"Welsh" },
	{ "cym",	"cy",	"Welsh" },
	{ "wen",	"",	"Sorbian" },
	{ "wln",	"wa",	"Walloon" },
	{ "wol",	"wo",	"Wolof" },
	{ "xal",	"",	"Kalmyk;" },
	{ "xho",	"xh",	"Xhosa" },
	{ "yao",	"",	"Yao" },
	{ "yap",	"",	"Yapese" },
	{ "yid",	"yi",	"Yiddish" },
	{ "yor",	"yo",	"Yoruba" },
	{ "ypk",	"",	"Yupik" },
	{ "zap",	"",	"Zapotec" },
	{ "zbl",	"",	"Blissymbols;" },
	{ "zen",	"",	"Zenaga" },
	{ "zgh",	"",	"Standard" },
	{ "zha",	"za",	"Zhuang;" },
	{ "chi",	"zh",	"Chinese" },
	{ "zho",	"zh",	"Chinese" },
	{ "znd",	"",	"Zande" },
	{ "zul",	"zu",	"Zulu" },
	{ "zun",	"",	"Zuni" },
	{ "zxx",	"",	"No" },
	{ "zza",	"",	"Zaza;" },
	{ NULL, NULL, NULL }
};


char *csc_iso639_lang_to_iso(char *lang)
{
	int	i;

	for (i = 0; ISO639LUT[i].name; i++) {
		if (!strcmp(ISO639LUT[i].name, lang)) {
			return ISO639LUT[i].iso639_2;
		}
	}
	return 0;
}

char *csc_iso639_lang_to_short(char *lang)
{
	int	i;

	for (i = 0; ISO639LUT[i].name; i++) {
		if (!strcmp(ISO639LUT[i].name, lang)) {
			return ISO639LUT[i].iso639_1;
		}
	}
	return 0;
}

char *csc_iso639_iso_to_lang(char *iso)
{
	int	i;

	switch (strlen(iso)) {
	case 3:
		for (i = 0; ISO639LUT[i].name; i++) {
			if (!strcmp(ISO639LUT[i].iso639_2, iso)) {
				return ISO639LUT[i].name;
			}
		}
		break;
	case 2:
		for (i = 0; ISO639LUT[i].name; i++) {
			if (!strcmp(ISO639LUT[i].iso639_1, iso)) {
				return ISO639LUT[i].name;
			}
		}
		break;
	}
	return NULL;
}

