/* Mapbox GL JS is Copyright © 2020 Mapbox and subject to the Mapbox Terms of Service ((https://www.mapbox.com/legal/tos/). */
(function (global, factory) {
typeof exports === 'object' && typeof module !== 'undefined' ? module.exports = factory() :
typeof define === 'function' && define.amd ? define(factory) :
(global = typeof globalThis !== 'undefined' ? globalThis : global || self, global.mapboxgl = factory());
})(this, (function () { 'use strict';

/* eslint-disable */

var shared, worker, mapboxgl;
// define gets called three times: one for each chunk. we rely on the order
// they're imported to know which is which
function define(_, chunk) {
if (!shared) {
    shared = chunk;
} else if (!worker) {
    worker = chunk;
} else {
    var workerBundleString = "self.onerror = function() { console.error('An error occurred while parsing the WebWorker bundle. This is most likely due to improper transpilation by Babel; please see https://docs.mapbox.com/mapbox-gl-js/guides/install/#transpiling'); }; var sharedChunk = {}; (" + shared + ")(sharedChunk); (" + worker + ")(sharedChunk); self.onerror = null;"

    var sharedChunk = {};
    shared(sharedChunk);
    mapboxgl = chunk(sharedChunk);
    if (typeof window !== 'undefined' && window && window.URL && window.URL.createObjectURL) {
        mapboxgl.workerUrl = window.URL.createObjectURL(new Blob([workerBundleString], { type: 'text/javascript' }));
    }
}
}


define(['exports'], (function (exports) { 'use strict';

function getDefaultExportFromCjs (x) {
	return x && x.__esModule && Object.prototype.hasOwnProperty.call(x, 'default') ? x['default'] : x;
}

var cjs = {};

var common = {};

var hasRequiredCommon;

function requireCommon () {
	if (hasRequiredCommon) return common;
	hasRequiredCommon = 1;
	Object.defineProperty(common, '__esModule', { value: true });
	common.setMatrixArrayType = setMatrixArrayType;
	common.toRadian = toRadian;
	common.equals = equals;
	common.RANDOM = common.ARRAY_TYPE = common.EPSILON = void 0;
	/**
	 * Common utilities
	 * @module glMatrix
	 */
	// Configuration Constants
	var EPSILON = 0.000001;
	common.EPSILON = EPSILON;
	var ARRAY_TYPE = typeof Float32Array !== 'undefined' ? Float32Array : Array;
	common.ARRAY_TYPE = ARRAY_TYPE;
	var RANDOM = Math.random;
	/**
	 * Sets the type of array used when creating new vectors and matrices
	 *
	 * @param {Float32ArrayConstructor | ArrayConstructor} type Array type, such as Float32Array or Array
	 */
	common.RANDOM = RANDOM;
	function setMatrixArrayType(type) {
	    common.ARRAY_TYPE = ARRAY_TYPE = type;
	}
	var degree = Math.PI / 180;
	/**
	 * Convert Degree To Radian
	 *
	 * @param {Number} a Angle in Degrees
	 */
	function toRadian(a) {
	    return a * degree;
	}
	/**
	 * Tests whether or not the arguments have approximately the same value, within an absolute
	 * or relative tolerance of glMatrix.EPSILON (an absolute tolerance is used for values less
	 * than or equal to 1.0, and a relative tolerance is used for larger values)
	 *
	 * @param {Number} a The first number to test.
	 * @param {Number} b The second number to test.
	 * @returns {Boolean} True if the numbers are approximately equal, false otherwise.
	 */
	function equals(a, b) {
	    return Math.abs(a - b) <= EPSILON * Math.max(1, Math.abs(a), Math.abs(b));
	}
	if (!Math.hypot)
	    Math.hypot = function () {
	        var y = 0, i = arguments.length;
	        while (i--) {
	            y += arguments[i] * arguments[i];
	        }
	        return Math.sqrt(y);
	    };
	return common;
}

var mat2 = {};

var hasRequiredMat2;

function requireMat2 () {
	if (hasRequiredMat2) return mat2;
	hasRequiredMat2 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(mat2, '__esModule', { value: true });
	mat2.create = create;
	mat2.clone = clone;
	mat2.copy = copy;
	mat2.identity = identity;
	mat2.fromValues = fromValues;
	mat2.set = set;
	mat2.transpose = transpose;
	mat2.invert = invert;
	mat2.adjoint = adjoint;
	mat2.determinant = determinant;
	mat2.multiply = multiply;
	mat2.rotate = rotate;
	mat2.scale = scale;
	mat2.fromRotation = fromRotation;
	mat2.fromScaling = fromScaling;
	mat2.str = str;
	mat2.frob = frob;
	mat2.LDU = LDU;
	mat2.add = add;
	mat2.subtract = subtract;
	mat2.exactEquals = exactEquals;
	mat2.equals = equals;
	mat2.multiplyScalar = multiplyScalar;
	mat2.multiplyScalarAndAdd = multiplyScalarAndAdd;
	mat2.sub = mat2.mul = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 2x2 Matrix
	 * @module mat2
	 */
	/**
	 * Creates a new identity mat2
	 *
	 * @returns {mat2} a new 2x2 matrix
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[1] = 0;
	        out[2] = 0;
	    }
	    out[0] = 1;
	    out[3] = 1;
	    return out;
	}
	/**
	 * Creates a new mat2 initialized with values from an existing matrix
	 *
	 * @param {ReadonlyMat2} a matrix to clone
	 * @returns {mat2} a new 2x2 matrix
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Copy the values from one mat2 to another
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the source matrix
	 * @returns {mat2} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Set a mat2 to the identity matrix
	 *
	 * @param {mat2} out the receiving matrix
	 * @returns {mat2} out
	 */
	function identity(out) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    return out;
	}
	/**
	 * Create a new mat2 with the given values
	 *
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m10 Component in column 1, row 0 position (index 2)
	 * @param {Number} m11 Component in column 1, row 1 position (index 3)
	 * @returns {mat2} out A new 2x2 matrix
	 */
	function fromValues(m00, m01, m10, m11) {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m10;
	    out[3] = m11;
	    return out;
	}
	/**
	 * Set the components of a mat2 to the given values
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m10 Component in column 1, row 0 position (index 2)
	 * @param {Number} m11 Component in column 1, row 1 position (index 3)
	 * @returns {mat2} out
	 */
	function set(out, m00, m01, m10, m11) {
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m10;
	    out[3] = m11;
	    return out;
	}
	/**
	 * Transpose the values of a mat2
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the source matrix
	 * @returns {mat2} out
	 */
	function transpose(out, a) {
	    // If we are transposing ourselves we can skip a few steps but have to cache
	    // some values
	    if (out === a) {
	        var a1 = a[1];
	        out[1] = a[2];
	        out[2] = a1;
	    } else {
	        out[0] = a[0];
	        out[1] = a[2];
	        out[2] = a[1];
	        out[3] = a[3];
	    }
	    return out;
	}
	/**
	 * Inverts a mat2
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the source matrix
	 * @returns {mat2} out
	 */
	function invert(out, a) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    // Calculate the determinant
	    var det = a0 * a3 - a2 * a1;
	    if (!det) {
	        return null;
	    }
	    det = 1 / det;
	    out[0] = a3 * det;
	    out[1] = -a1 * det;
	    out[2] = -a2 * det;
	    out[3] = a0 * det;
	    return out;
	}
	/**
	 * Calculates the adjugate of a mat2
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the source matrix
	 * @returns {mat2} out
	 */
	function adjoint(out, a) {
	    // Caching this value is nessecary if out == a
	    var a0 = a[0];
	    out[0] = a[3];
	    out[1] = -a[1];
	    out[2] = -a[2];
	    out[3] = a0;
	    return out;
	}
	/**
	 * Calculates the determinant of a mat2
	 *
	 * @param {ReadonlyMat2} a the source matrix
	 * @returns {Number} determinant of a
	 */
	function determinant(a) {
	    return a[0] * a[3] - a[2] * a[1];
	}
	/**
	 * Multiplies two mat2's
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the first operand
	 * @param {ReadonlyMat2} b the second operand
	 * @returns {mat2} out
	 */
	function multiply(out, a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
	    out[0] = a0 * b0 + a2 * b1;
	    out[1] = a1 * b0 + a3 * b1;
	    out[2] = a0 * b2 + a2 * b3;
	    out[3] = a1 * b2 + a3 * b3;
	    return out;
	}
	/**
	 * Rotates a mat2 by the given angle
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat2} out
	 */
	function rotate(out, a, rad) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    out[0] = a0 * c + a2 * s;
	    out[1] = a1 * c + a3 * s;
	    out[2] = a0 * -s + a2 * c;
	    out[3] = a1 * -s + a3 * c;
	    return out;
	}
	/**
	 * Scales the mat2 by the dimensions in the given vec2
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the matrix to rotate
	 * @param {ReadonlyVec2} v the vec2 to scale the matrix by
	 * @returns {mat2} out
	 **/
	function scale(out, a, v) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var v0 = v[0], v1 = v[1];
	    out[0] = a0 * v0;
	    out[1] = a1 * v0;
	    out[2] = a2 * v1;
	    out[3] = a3 * v1;
	    return out;
	}
	/**
	 * Creates a matrix from a given angle
	 * This is equivalent to (but much faster than):
	 *
	 *     mat2.identity(dest);
	 *     mat2.rotate(dest, dest, rad);
	 *
	 * @param {mat2} out mat2 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat2} out
	 */
	function fromRotation(out, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    out[0] = c;
	    out[1] = s;
	    out[2] = -s;
	    out[3] = c;
	    return out;
	}
	/**
	 * Creates a matrix from a vector scaling
	 * This is equivalent to (but much faster than):
	 *
	 *     mat2.identity(dest);
	 *     mat2.scale(dest, dest, vec);
	 *
	 * @param {mat2} out mat2 receiving operation result
	 * @param {ReadonlyVec2} v Scaling vector
	 * @returns {mat2} out
	 */
	function fromScaling(out, v) {
	    out[0] = v[0];
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = v[1];
	    return out;
	}
	/**
	 * Returns a string representation of a mat2
	 *
	 * @param {ReadonlyMat2} a matrix to represent as a string
	 * @returns {String} string representation of the matrix
	 */
	function str(a) {
	    return 'mat2(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
	}
	/**
	 * Returns Frobenius norm of a mat2
	 *
	 * @param {ReadonlyMat2} a the matrix to calculate Frobenius norm of
	 * @returns {Number} Frobenius norm
	 */
	function frob(a) {
	    return Math.hypot(a[0], a[1], a[2], a[3]);
	}
	/**
	 * Returns L, D and U matrices (Lower triangular, Diagonal and Upper triangular) by factorizing the input matrix
	 * @param {ReadonlyMat2} L the lower triangular matrix
	 * @param {ReadonlyMat2} D the diagonal matrix
	 * @param {ReadonlyMat2} U the upper triangular matrix
	 * @param {ReadonlyMat2} a the input matrix to factorize
	 */
	function LDU(L, D, U, a) {
	    L[2] = a[2] / a[0];
	    U[0] = a[0];
	    U[1] = a[1];
	    U[3] = a[3] - L[2] * U[1];
	    return [
	        L,
	        D,
	        U
	    ];
	}
	/**
	 * Adds two mat2's
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the first operand
	 * @param {ReadonlyMat2} b the second operand
	 * @returns {mat2} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    return out;
	}
	/**
	 * Subtracts matrix b from matrix a
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the first operand
	 * @param {ReadonlyMat2} b the second operand
	 * @returns {mat2} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    out[3] = a[3] - b[3];
	    return out;
	}
	/**
	 * Returns whether or not the matrices have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyMat2} a The first matrix.
	 * @param {ReadonlyMat2} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3];
	}
	/**
	 * Returns whether or not the matrices have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyMat2} a The first matrix.
	 * @param {ReadonlyMat2} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3));
	}
	/**
	 * Multiply each element of the matrix by a scalar.
	 *
	 * @param {mat2} out the receiving matrix
	 * @param {ReadonlyMat2} a the matrix to scale
	 * @param {Number} b amount to scale the matrix's elements by
	 * @returns {mat2} out
	 */
	function multiplyScalar(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    return out;
	}
	/**
	 * Adds two mat2's after multiplying each element of the second operand by a scalar value.
	 *
	 * @param {mat2} out the receiving vector
	 * @param {ReadonlyMat2} a the first operand
	 * @param {ReadonlyMat2} b the second operand
	 * @param {Number} scale the amount to scale b's elements by before adding
	 * @returns {mat2} out
	 */
	function multiplyScalarAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    out[3] = a[3] + b[3] * scale;
	    return out;
	}
	/**
	 * Alias for {@link mat2.multiply}
	 * @function
	 */
	var mul = multiply;
	/**
	 * Alias for {@link mat2.subtract}
	 * @function
	 */
	mat2.mul = mul;
	var sub = subtract;
	mat2.sub = sub;
	return mat2;
}

var mat2d = {};

var hasRequiredMat2d;

function requireMat2d () {
	if (hasRequiredMat2d) return mat2d;
	hasRequiredMat2d = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(mat2d, '__esModule', { value: true });
	mat2d.create = create;
	mat2d.clone = clone;
	mat2d.copy = copy;
	mat2d.identity = identity;
	mat2d.fromValues = fromValues;
	mat2d.set = set;
	mat2d.invert = invert;
	mat2d.determinant = determinant;
	mat2d.multiply = multiply;
	mat2d.rotate = rotate;
	mat2d.scale = scale;
	mat2d.translate = translate;
	mat2d.fromRotation = fromRotation;
	mat2d.fromScaling = fromScaling;
	mat2d.fromTranslation = fromTranslation;
	mat2d.str = str;
	mat2d.frob = frob;
	mat2d.add = add;
	mat2d.subtract = subtract;
	mat2d.multiplyScalar = multiplyScalar;
	mat2d.multiplyScalarAndAdd = multiplyScalarAndAdd;
	mat2d.exactEquals = exactEquals;
	mat2d.equals = equals;
	mat2d.sub = mat2d.mul = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 2x3 Matrix
	 * @module mat2d
	 * @description
	 * A mat2d contains six elements defined as:
	 * <pre>
	 * [a, b,
	 *  c, d,
	 *  tx, ty]
	 * </pre>
	 * This is a short form for the 3x3 matrix:
	 * <pre>
	 * [a, b, 0,
	 *  c, d, 0,
	 *  tx, ty, 1]
	 * </pre>
	 * The last column is ignored so the array is shorter and operations are faster.
	 */
	/**
	 * Creates a new identity mat2d
	 *
	 * @returns {mat2d} a new 2x3 matrix
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(6);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[1] = 0;
	        out[2] = 0;
	        out[4] = 0;
	        out[5] = 0;
	    }
	    out[0] = 1;
	    out[3] = 1;
	    return out;
	}
	/**
	 * Creates a new mat2d initialized with values from an existing matrix
	 *
	 * @param {ReadonlyMat2d} a matrix to clone
	 * @returns {mat2d} a new 2x3 matrix
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(6);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    return out;
	}
	/**
	 * Copy the values from one mat2d to another
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the source matrix
	 * @returns {mat2d} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    return out;
	}
	/**
	 * Set a mat2d to the identity matrix
	 *
	 * @param {mat2d} out the receiving matrix
	 * @returns {mat2d} out
	 */
	function identity(out) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    out[4] = 0;
	    out[5] = 0;
	    return out;
	}
	/**
	 * Create a new mat2d with the given values
	 *
	 * @param {Number} a Component A (index 0)
	 * @param {Number} b Component B (index 1)
	 * @param {Number} c Component C (index 2)
	 * @param {Number} d Component D (index 3)
	 * @param {Number} tx Component TX (index 4)
	 * @param {Number} ty Component TY (index 5)
	 * @returns {mat2d} A new mat2d
	 */
	function fromValues(a, b, c, d, tx, ty) {
	    var out = new glMatrix.ARRAY_TYPE(6);
	    out[0] = a;
	    out[1] = b;
	    out[2] = c;
	    out[3] = d;
	    out[4] = tx;
	    out[5] = ty;
	    return out;
	}
	/**
	 * Set the components of a mat2d to the given values
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {Number} a Component A (index 0)
	 * @param {Number} b Component B (index 1)
	 * @param {Number} c Component C (index 2)
	 * @param {Number} d Component D (index 3)
	 * @param {Number} tx Component TX (index 4)
	 * @param {Number} ty Component TY (index 5)
	 * @returns {mat2d} out
	 */
	function set(out, a, b, c, d, tx, ty) {
	    out[0] = a;
	    out[1] = b;
	    out[2] = c;
	    out[3] = d;
	    out[4] = tx;
	    out[5] = ty;
	    return out;
	}
	/**
	 * Inverts a mat2d
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the source matrix
	 * @returns {mat2d} out
	 */
	function invert(out, a) {
	    var aa = a[0], ab = a[1], ac = a[2], ad = a[3];
	    var atx = a[4], aty = a[5];
	    var det = aa * ad - ab * ac;
	    if (!det) {
	        return null;
	    }
	    det = 1 / det;
	    out[0] = ad * det;
	    out[1] = -ab * det;
	    out[2] = -ac * det;
	    out[3] = aa * det;
	    out[4] = (ac * aty - ad * atx) * det;
	    out[5] = (ab * atx - aa * aty) * det;
	    return out;
	}
	/**
	 * Calculates the determinant of a mat2d
	 *
	 * @param {ReadonlyMat2d} a the source matrix
	 * @returns {Number} determinant of a
	 */
	function determinant(a) {
	    return a[0] * a[3] - a[1] * a[2];
	}
	/**
	 * Multiplies two mat2d's
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the first operand
	 * @param {ReadonlyMat2d} b the second operand
	 * @returns {mat2d} out
	 */
	function multiply(out, a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3], b4 = b[4], b5 = b[5];
	    out[0] = a0 * b0 + a2 * b1;
	    out[1] = a1 * b0 + a3 * b1;
	    out[2] = a0 * b2 + a2 * b3;
	    out[3] = a1 * b2 + a3 * b3;
	    out[4] = a0 * b4 + a2 * b5 + a4;
	    out[5] = a1 * b4 + a3 * b5 + a5;
	    return out;
	}
	/**
	 * Rotates a mat2d by the given angle
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat2d} out
	 */
	function rotate(out, a, rad) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5];
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    out[0] = a0 * c + a2 * s;
	    out[1] = a1 * c + a3 * s;
	    out[2] = a0 * -s + a2 * c;
	    out[3] = a1 * -s + a3 * c;
	    out[4] = a4;
	    out[5] = a5;
	    return out;
	}
	/**
	 * Scales the mat2d by the dimensions in the given vec2
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the matrix to translate
	 * @param {ReadonlyVec2} v the vec2 to scale the matrix by
	 * @returns {mat2d} out
	 **/
	function scale(out, a, v) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5];
	    var v0 = v[0], v1 = v[1];
	    out[0] = a0 * v0;
	    out[1] = a1 * v0;
	    out[2] = a2 * v1;
	    out[3] = a3 * v1;
	    out[4] = a4;
	    out[5] = a5;
	    return out;
	}
	/**
	 * Translates the mat2d by the dimensions in the given vec2
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the matrix to translate
	 * @param {ReadonlyVec2} v the vec2 to translate the matrix by
	 * @returns {mat2d} out
	 **/
	function translate(out, a, v) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5];
	    var v0 = v[0], v1 = v[1];
	    out[0] = a0;
	    out[1] = a1;
	    out[2] = a2;
	    out[3] = a3;
	    out[4] = a0 * v0 + a2 * v1 + a4;
	    out[5] = a1 * v0 + a3 * v1 + a5;
	    return out;
	}
	/**
	 * Creates a matrix from a given angle
	 * This is equivalent to (but much faster than):
	 *
	 *     mat2d.identity(dest);
	 *     mat2d.rotate(dest, dest, rad);
	 *
	 * @param {mat2d} out mat2d receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat2d} out
	 */
	function fromRotation(out, rad) {
	    var s = Math.sin(rad), c = Math.cos(rad);
	    out[0] = c;
	    out[1] = s;
	    out[2] = -s;
	    out[3] = c;
	    out[4] = 0;
	    out[5] = 0;
	    return out;
	}
	/**
	 * Creates a matrix from a vector scaling
	 * This is equivalent to (but much faster than):
	 *
	 *     mat2d.identity(dest);
	 *     mat2d.scale(dest, dest, vec);
	 *
	 * @param {mat2d} out mat2d receiving operation result
	 * @param {ReadonlyVec2} v Scaling vector
	 * @returns {mat2d} out
	 */
	function fromScaling(out, v) {
	    out[0] = v[0];
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = v[1];
	    out[4] = 0;
	    out[5] = 0;
	    return out;
	}
	/**
	 * Creates a matrix from a vector translation
	 * This is equivalent to (but much faster than):
	 *
	 *     mat2d.identity(dest);
	 *     mat2d.translate(dest, dest, vec);
	 *
	 * @param {mat2d} out mat2d receiving operation result
	 * @param {ReadonlyVec2} v Translation vector
	 * @returns {mat2d} out
	 */
	function fromTranslation(out, v) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    out[4] = v[0];
	    out[5] = v[1];
	    return out;
	}
	/**
	 * Returns a string representation of a mat2d
	 *
	 * @param {ReadonlyMat2d} a matrix to represent as a string
	 * @returns {String} string representation of the matrix
	 */
	function str(a) {
	    return 'mat2d(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ', ' + a[4] + ', ' + a[5] + ')';
	}
	/**
	 * Returns Frobenius norm of a mat2d
	 *
	 * @param {ReadonlyMat2d} a the matrix to calculate Frobenius norm of
	 * @returns {Number} Frobenius norm
	 */
	function frob(a) {
	    return Math.hypot(a[0], a[1], a[2], a[3], a[4], a[5], 1);
	}
	/**
	 * Adds two mat2d's
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the first operand
	 * @param {ReadonlyMat2d} b the second operand
	 * @returns {mat2d} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    out[4] = a[4] + b[4];
	    out[5] = a[5] + b[5];
	    return out;
	}
	/**
	 * Subtracts matrix b from matrix a
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the first operand
	 * @param {ReadonlyMat2d} b the second operand
	 * @returns {mat2d} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    out[3] = a[3] - b[3];
	    out[4] = a[4] - b[4];
	    out[5] = a[5] - b[5];
	    return out;
	}
	/**
	 * Multiply each element of the matrix by a scalar.
	 *
	 * @param {mat2d} out the receiving matrix
	 * @param {ReadonlyMat2d} a the matrix to scale
	 * @param {Number} b amount to scale the matrix's elements by
	 * @returns {mat2d} out
	 */
	function multiplyScalar(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    out[4] = a[4] * b;
	    out[5] = a[5] * b;
	    return out;
	}
	/**
	 * Adds two mat2d's after multiplying each element of the second operand by a scalar value.
	 *
	 * @param {mat2d} out the receiving vector
	 * @param {ReadonlyMat2d} a the first operand
	 * @param {ReadonlyMat2d} b the second operand
	 * @param {Number} scale the amount to scale b's elements by before adding
	 * @returns {mat2d} out
	 */
	function multiplyScalarAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    out[3] = a[3] + b[3] * scale;
	    out[4] = a[4] + b[4] * scale;
	    out[5] = a[5] + b[5] * scale;
	    return out;
	}
	/**
	 * Returns whether or not the matrices have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyMat2d} a The first matrix.
	 * @param {ReadonlyMat2d} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3] && a[4] === b[4] && a[5] === b[5];
	}
	/**
	 * Returns whether or not the matrices have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyMat2d} a The first matrix.
	 * @param {ReadonlyMat2d} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3], b4 = b[4], b5 = b[5];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3)) && Math.abs(a4 - b4) <= glMatrix.EPSILON * Math.max(1, Math.abs(a4), Math.abs(b4)) && Math.abs(a5 - b5) <= glMatrix.EPSILON * Math.max(1, Math.abs(a5), Math.abs(b5));
	}
	/**
	 * Alias for {@link mat2d.multiply}
	 * @function
	 */
	var mul = multiply;
	/**
	 * Alias for {@link mat2d.subtract}
	 * @function
	 */
	mat2d.mul = mul;
	var sub = subtract;
	mat2d.sub = sub;
	return mat2d;
}

var mat3 = {};

var hasRequiredMat3;

function requireMat3 () {
	if (hasRequiredMat3) return mat3;
	hasRequiredMat3 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(mat3, '__esModule', { value: true });
	mat3.create = create;
	mat3.fromMat4 = fromMat4;
	mat3.clone = clone;
	mat3.copy = copy;
	mat3.fromValues = fromValues;
	mat3.set = set;
	mat3.identity = identity;
	mat3.transpose = transpose;
	mat3.invert = invert;
	mat3.adjoint = adjoint;
	mat3.determinant = determinant;
	mat3.multiply = multiply;
	mat3.translate = translate;
	mat3.rotate = rotate;
	mat3.scale = scale;
	mat3.fromTranslation = fromTranslation;
	mat3.fromRotation = fromRotation;
	mat3.fromScaling = fromScaling;
	mat3.fromMat2d = fromMat2d;
	mat3.fromQuat = fromQuat;
	mat3.normalFromMat4 = normalFromMat4;
	mat3.projection = projection;
	mat3.str = str;
	mat3.frob = frob;
	mat3.add = add;
	mat3.subtract = subtract;
	mat3.multiplyScalar = multiplyScalar;
	mat3.multiplyScalarAndAdd = multiplyScalarAndAdd;
	mat3.exactEquals = exactEquals;
	mat3.equals = equals;
	mat3.sub = mat3.mul = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 3x3 Matrix
	 * @module mat3
	 */
	/**
	 * Creates a new identity mat3
	 *
	 * @returns {mat3} a new 3x3 matrix
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(9);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[1] = 0;
	        out[2] = 0;
	        out[3] = 0;
	        out[5] = 0;
	        out[6] = 0;
	        out[7] = 0;
	    }
	    out[0] = 1;
	    out[4] = 1;
	    out[8] = 1;
	    return out;
	}
	/**
	 * Copies the upper-left 3x3 values into the given mat3.
	 *
	 * @param {mat3} out the receiving 3x3 matrix
	 * @param {ReadonlyMat4} a   the source 4x4 matrix
	 * @returns {mat3} out
	 */
	function fromMat4(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[4];
	    out[4] = a[5];
	    out[5] = a[6];
	    out[6] = a[8];
	    out[7] = a[9];
	    out[8] = a[10];
	    return out;
	}
	/**
	 * Creates a new mat3 initialized with values from an existing matrix
	 *
	 * @param {ReadonlyMat3} a matrix to clone
	 * @returns {mat3} a new 3x3 matrix
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(9);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    out[8] = a[8];
	    return out;
	}
	/**
	 * Copy the values from one mat3 to another
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the source matrix
	 * @returns {mat3} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    out[8] = a[8];
	    return out;
	}
	/**
	 * Create a new mat3 with the given values
	 *
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m02 Component in column 0, row 2 position (index 2)
	 * @param {Number} m10 Component in column 1, row 0 position (index 3)
	 * @param {Number} m11 Component in column 1, row 1 position (index 4)
	 * @param {Number} m12 Component in column 1, row 2 position (index 5)
	 * @param {Number} m20 Component in column 2, row 0 position (index 6)
	 * @param {Number} m21 Component in column 2, row 1 position (index 7)
	 * @param {Number} m22 Component in column 2, row 2 position (index 8)
	 * @returns {mat3} A new mat3
	 */
	function fromValues(m00, m01, m02, m10, m11, m12, m20, m21, m22) {
	    var out = new glMatrix.ARRAY_TYPE(9);
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m02;
	    out[3] = m10;
	    out[4] = m11;
	    out[5] = m12;
	    out[6] = m20;
	    out[7] = m21;
	    out[8] = m22;
	    return out;
	}
	/**
	 * Set the components of a mat3 to the given values
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m02 Component in column 0, row 2 position (index 2)
	 * @param {Number} m10 Component in column 1, row 0 position (index 3)
	 * @param {Number} m11 Component in column 1, row 1 position (index 4)
	 * @param {Number} m12 Component in column 1, row 2 position (index 5)
	 * @param {Number} m20 Component in column 2, row 0 position (index 6)
	 * @param {Number} m21 Component in column 2, row 1 position (index 7)
	 * @param {Number} m22 Component in column 2, row 2 position (index 8)
	 * @returns {mat3} out
	 */
	function set(out, m00, m01, m02, m10, m11, m12, m20, m21, m22) {
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m02;
	    out[3] = m10;
	    out[4] = m11;
	    out[5] = m12;
	    out[6] = m20;
	    out[7] = m21;
	    out[8] = m22;
	    return out;
	}
	/**
	 * Set a mat3 to the identity matrix
	 *
	 * @param {mat3} out the receiving matrix
	 * @returns {mat3} out
	 */
	function identity(out) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 1;
	    out[5] = 0;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 1;
	    return out;
	}
	/**
	 * Transpose the values of a mat3
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the source matrix
	 * @returns {mat3} out
	 */
	function transpose(out, a) {
	    // If we are transposing ourselves we can skip a few steps but have to cache some values
	    if (out === a) {
	        var a01 = a[1], a02 = a[2], a12 = a[5];
	        out[1] = a[3];
	        out[2] = a[6];
	        out[3] = a01;
	        out[5] = a[7];
	        out[6] = a02;
	        out[7] = a12;
	    } else {
	        out[0] = a[0];
	        out[1] = a[3];
	        out[2] = a[6];
	        out[3] = a[1];
	        out[4] = a[4];
	        out[5] = a[7];
	        out[6] = a[2];
	        out[7] = a[5];
	        out[8] = a[8];
	    }
	    return out;
	}
	/**
	 * Inverts a mat3
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the source matrix
	 * @returns {mat3} out
	 */
	function invert(out, a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2];
	    var a10 = a[3], a11 = a[4], a12 = a[5];
	    var a20 = a[6], a21 = a[7], a22 = a[8];
	    var b01 = a22 * a11 - a12 * a21;
	    var b11 = -a22 * a10 + a12 * a20;
	    var b21 = a21 * a10 - a11 * a20;
	    // Calculate the determinant
	    var det = a00 * b01 + a01 * b11 + a02 * b21;
	    if (!det) {
	        return null;
	    }
	    det = 1 / det;
	    out[0] = b01 * det;
	    out[1] = (-a22 * a01 + a02 * a21) * det;
	    out[2] = (a12 * a01 - a02 * a11) * det;
	    out[3] = b11 * det;
	    out[4] = (a22 * a00 - a02 * a20) * det;
	    out[5] = (-a12 * a00 + a02 * a10) * det;
	    out[6] = b21 * det;
	    out[7] = (-a21 * a00 + a01 * a20) * det;
	    out[8] = (a11 * a00 - a01 * a10) * det;
	    return out;
	}
	/**
	 * Calculates the adjugate of a mat3
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the source matrix
	 * @returns {mat3} out
	 */
	function adjoint(out, a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2];
	    var a10 = a[3], a11 = a[4], a12 = a[5];
	    var a20 = a[6], a21 = a[7], a22 = a[8];
	    out[0] = a11 * a22 - a12 * a21;
	    out[1] = a02 * a21 - a01 * a22;
	    out[2] = a01 * a12 - a02 * a11;
	    out[3] = a12 * a20 - a10 * a22;
	    out[4] = a00 * a22 - a02 * a20;
	    out[5] = a02 * a10 - a00 * a12;
	    out[6] = a10 * a21 - a11 * a20;
	    out[7] = a01 * a20 - a00 * a21;
	    out[8] = a00 * a11 - a01 * a10;
	    return out;
	}
	/**
	 * Calculates the determinant of a mat3
	 *
	 * @param {ReadonlyMat3} a the source matrix
	 * @returns {Number} determinant of a
	 */
	function determinant(a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2];
	    var a10 = a[3], a11 = a[4], a12 = a[5];
	    var a20 = a[6], a21 = a[7], a22 = a[8];
	    return a00 * (a22 * a11 - a12 * a21) + a01 * (-a22 * a10 + a12 * a20) + a02 * (a21 * a10 - a11 * a20);
	}
	/**
	 * Multiplies two mat3's
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the first operand
	 * @param {ReadonlyMat3} b the second operand
	 * @returns {mat3} out
	 */
	function multiply(out, a, b) {
	    var a00 = a[0], a01 = a[1], a02 = a[2];
	    var a10 = a[3], a11 = a[4], a12 = a[5];
	    var a20 = a[6], a21 = a[7], a22 = a[8];
	    var b00 = b[0], b01 = b[1], b02 = b[2];
	    var b10 = b[3], b11 = b[4], b12 = b[5];
	    var b20 = b[6], b21 = b[7], b22 = b[8];
	    out[0] = b00 * a00 + b01 * a10 + b02 * a20;
	    out[1] = b00 * a01 + b01 * a11 + b02 * a21;
	    out[2] = b00 * a02 + b01 * a12 + b02 * a22;
	    out[3] = b10 * a00 + b11 * a10 + b12 * a20;
	    out[4] = b10 * a01 + b11 * a11 + b12 * a21;
	    out[5] = b10 * a02 + b11 * a12 + b12 * a22;
	    out[6] = b20 * a00 + b21 * a10 + b22 * a20;
	    out[7] = b20 * a01 + b21 * a11 + b22 * a21;
	    out[8] = b20 * a02 + b21 * a12 + b22 * a22;
	    return out;
	}
	/**
	 * Translate a mat3 by the given vector
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the matrix to translate
	 * @param {ReadonlyVec2} v vector to translate by
	 * @returns {mat3} out
	 */
	function translate(out, a, v) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a10 = a[3], a11 = a[4], a12 = a[5], a20 = a[6], a21 = a[7], a22 = a[8], x = v[0], y = v[1];
	    out[0] = a00;
	    out[1] = a01;
	    out[2] = a02;
	    out[3] = a10;
	    out[4] = a11;
	    out[5] = a12;
	    out[6] = x * a00 + y * a10 + a20;
	    out[7] = x * a01 + y * a11 + a21;
	    out[8] = x * a02 + y * a12 + a22;
	    return out;
	}
	/**
	 * Rotates a mat3 by the given angle
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat3} out
	 */
	function rotate(out, a, rad) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a10 = a[3], a11 = a[4], a12 = a[5], a20 = a[6], a21 = a[7], a22 = a[8], s = Math.sin(rad), c = Math.cos(rad);
	    out[0] = c * a00 + s * a10;
	    out[1] = c * a01 + s * a11;
	    out[2] = c * a02 + s * a12;
	    out[3] = c * a10 - s * a00;
	    out[4] = c * a11 - s * a01;
	    out[5] = c * a12 - s * a02;
	    out[6] = a20;
	    out[7] = a21;
	    out[8] = a22;
	    return out;
	}
	/**
	 * Scales the mat3 by the dimensions in the given vec2
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the matrix to rotate
	 * @param {ReadonlyVec2} v the vec2 to scale the matrix by
	 * @returns {mat3} out
	 **/
	function scale(out, a, v) {
	    var x = v[0], y = v[1];
	    out[0] = x * a[0];
	    out[1] = x * a[1];
	    out[2] = x * a[2];
	    out[3] = y * a[3];
	    out[4] = y * a[4];
	    out[5] = y * a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    out[8] = a[8];
	    return out;
	}
	/**
	 * Creates a matrix from a vector translation
	 * This is equivalent to (but much faster than):
	 *
	 *     mat3.identity(dest);
	 *     mat3.translate(dest, dest, vec);
	 *
	 * @param {mat3} out mat3 receiving operation result
	 * @param {ReadonlyVec2} v Translation vector
	 * @returns {mat3} out
	 */
	function fromTranslation(out, v) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 1;
	    out[5] = 0;
	    out[6] = v[0];
	    out[7] = v[1];
	    out[8] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a given angle
	 * This is equivalent to (but much faster than):
	 *
	 *     mat3.identity(dest);
	 *     mat3.rotate(dest, dest, rad);
	 *
	 * @param {mat3} out mat3 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat3} out
	 */
	function fromRotation(out, rad) {
	    var s = Math.sin(rad), c = Math.cos(rad);
	    out[0] = c;
	    out[1] = s;
	    out[2] = 0;
	    out[3] = -s;
	    out[4] = c;
	    out[5] = 0;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a vector scaling
	 * This is equivalent to (but much faster than):
	 *
	 *     mat3.identity(dest);
	 *     mat3.scale(dest, dest, vec);
	 *
	 * @param {mat3} out mat3 receiving operation result
	 * @param {ReadonlyVec2} v Scaling vector
	 * @returns {mat3} out
	 */
	function fromScaling(out, v) {
	    out[0] = v[0];
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = v[1];
	    out[5] = 0;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 1;
	    return out;
	}
	/**
	 * Copies the values from a mat2d into a mat3
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat2d} a the matrix to copy
	 * @returns {mat3} out
	 **/
	function fromMat2d(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = 0;
	    out[3] = a[2];
	    out[4] = a[3];
	    out[5] = 0;
	    out[6] = a[4];
	    out[7] = a[5];
	    out[8] = 1;
	    return out;
	}
	/**
	 * Calculates a 3x3 matrix from the given quaternion
	 *
	 * @param {mat3} out mat3 receiving operation result
	 * @param {ReadonlyQuat} q Quaternion to create matrix from
	 *
	 * @returns {mat3} out
	 */
	function fromQuat(out, q) {
	    var x = q[0], y = q[1], z = q[2], w = q[3];
	    var x2 = x + x;
	    var y2 = y + y;
	    var z2 = z + z;
	    var xx = x * x2;
	    var yx = y * x2;
	    var yy = y * y2;
	    var zx = z * x2;
	    var zy = z * y2;
	    var zz = z * z2;
	    var wx = w * x2;
	    var wy = w * y2;
	    var wz = w * z2;
	    out[0] = 1 - yy - zz;
	    out[3] = yx - wz;
	    out[6] = zx + wy;
	    out[1] = yx + wz;
	    out[4] = 1 - xx - zz;
	    out[7] = zy - wx;
	    out[2] = zx - wy;
	    out[5] = zy + wx;
	    out[8] = 1 - xx - yy;
	    return out;
	}
	/**
	 * Calculates a 3x3 normal matrix (transpose inverse) from the 4x4 matrix
	 *
	 * @param {mat3} out mat3 receiving operation result
	 * @param {ReadonlyMat4} a Mat4 to derive the normal matrix from
	 *
	 * @returns {mat3} out
	 */
	function normalFromMat4(out, a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3];
	    var a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7];
	    var a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11];
	    var a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];
	    var b00 = a00 * a11 - a01 * a10;
	    var b01 = a00 * a12 - a02 * a10;
	    var b02 = a00 * a13 - a03 * a10;
	    var b03 = a01 * a12 - a02 * a11;
	    var b04 = a01 * a13 - a03 * a11;
	    var b05 = a02 * a13 - a03 * a12;
	    var b06 = a20 * a31 - a21 * a30;
	    var b07 = a20 * a32 - a22 * a30;
	    var b08 = a20 * a33 - a23 * a30;
	    var b09 = a21 * a32 - a22 * a31;
	    var b10 = a21 * a33 - a23 * a31;
	    var b11 = a22 * a33 - a23 * a32;
	    // Calculate the determinant
	    var det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
	    if (!det) {
	        return null;
	    }
	    det = 1 / det;
	    out[0] = (a11 * b11 - a12 * b10 + a13 * b09) * det;
	    out[1] = (a12 * b08 - a10 * b11 - a13 * b07) * det;
	    out[2] = (a10 * b10 - a11 * b08 + a13 * b06) * det;
	    out[3] = (a02 * b10 - a01 * b11 - a03 * b09) * det;
	    out[4] = (a00 * b11 - a02 * b08 + a03 * b07) * det;
	    out[5] = (a01 * b08 - a00 * b10 - a03 * b06) * det;
	    out[6] = (a31 * b05 - a32 * b04 + a33 * b03) * det;
	    out[7] = (a32 * b02 - a30 * b05 - a33 * b01) * det;
	    out[8] = (a30 * b04 - a31 * b02 + a33 * b00) * det;
	    return out;
	}
	/**
	 * Generates a 2D projection matrix with the given bounds
	 *
	 * @param {mat3} out mat3 frustum matrix will be written into
	 * @param {number} width Width of your gl context
	 * @param {number} height Height of gl context
	 * @returns {mat3} out
	 */
	function projection(out, width, height) {
	    out[0] = 2 / width;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = -2 / height;
	    out[5] = 0;
	    out[6] = -1;
	    out[7] = 1;
	    out[8] = 1;
	    return out;
	}
	/**
	 * Returns a string representation of a mat3
	 *
	 * @param {ReadonlyMat3} a matrix to represent as a string
	 * @returns {String} string representation of the matrix
	 */
	function str(a) {
	    return 'mat3(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ', ' + a[4] + ', ' + a[5] + ', ' + a[6] + ', ' + a[7] + ', ' + a[8] + ')';
	}
	/**
	 * Returns Frobenius norm of a mat3
	 *
	 * @param {ReadonlyMat3} a the matrix to calculate Frobenius norm of
	 * @returns {Number} Frobenius norm
	 */
	function frob(a) {
	    return Math.hypot(a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8]);
	}
	/**
	 * Adds two mat3's
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the first operand
	 * @param {ReadonlyMat3} b the second operand
	 * @returns {mat3} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    out[4] = a[4] + b[4];
	    out[5] = a[5] + b[5];
	    out[6] = a[6] + b[6];
	    out[7] = a[7] + b[7];
	    out[8] = a[8] + b[8];
	    return out;
	}
	/**
	 * Subtracts matrix b from matrix a
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the first operand
	 * @param {ReadonlyMat3} b the second operand
	 * @returns {mat3} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    out[3] = a[3] - b[3];
	    out[4] = a[4] - b[4];
	    out[5] = a[5] - b[5];
	    out[6] = a[6] - b[6];
	    out[7] = a[7] - b[7];
	    out[8] = a[8] - b[8];
	    return out;
	}
	/**
	 * Multiply each element of the matrix by a scalar.
	 *
	 * @param {mat3} out the receiving matrix
	 * @param {ReadonlyMat3} a the matrix to scale
	 * @param {Number} b amount to scale the matrix's elements by
	 * @returns {mat3} out
	 */
	function multiplyScalar(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    out[4] = a[4] * b;
	    out[5] = a[5] * b;
	    out[6] = a[6] * b;
	    out[7] = a[7] * b;
	    out[8] = a[8] * b;
	    return out;
	}
	/**
	 * Adds two mat3's after multiplying each element of the second operand by a scalar value.
	 *
	 * @param {mat3} out the receiving vector
	 * @param {ReadonlyMat3} a the first operand
	 * @param {ReadonlyMat3} b the second operand
	 * @param {Number} scale the amount to scale b's elements by before adding
	 * @returns {mat3} out
	 */
	function multiplyScalarAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    out[3] = a[3] + b[3] * scale;
	    out[4] = a[4] + b[4] * scale;
	    out[5] = a[5] + b[5] * scale;
	    out[6] = a[6] + b[6] * scale;
	    out[7] = a[7] + b[7] * scale;
	    out[8] = a[8] + b[8] * scale;
	    return out;
	}
	/**
	 * Returns whether or not the matrices have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyMat3} a The first matrix.
	 * @param {ReadonlyMat3} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3] && a[4] === b[4] && a[5] === b[5] && a[6] === b[6] && a[7] === b[7] && a[8] === b[8];
	}
	/**
	 * Returns whether or not the matrices have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyMat3} a The first matrix.
	 * @param {ReadonlyMat3} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5], a6 = a[6], a7 = a[7], a8 = a[8];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3], b4 = b[4], b5 = b[5], b6 = b[6], b7 = b[7], b8 = b[8];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3)) && Math.abs(a4 - b4) <= glMatrix.EPSILON * Math.max(1, Math.abs(a4), Math.abs(b4)) && Math.abs(a5 - b5) <= glMatrix.EPSILON * Math.max(1, Math.abs(a5), Math.abs(b5)) && Math.abs(a6 - b6) <= glMatrix.EPSILON * Math.max(1, Math.abs(a6), Math.abs(b6)) && Math.abs(a7 - b7) <= glMatrix.EPSILON * Math.max(1, Math.abs(a7), Math.abs(b7)) && Math.abs(a8 - b8) <= glMatrix.EPSILON * Math.max(1, Math.abs(a8), Math.abs(b8));
	}
	/**
	 * Alias for {@link mat3.multiply}
	 * @function
	 */
	var mul = multiply;
	/**
	 * Alias for {@link mat3.subtract}
	 * @function
	 */
	mat3.mul = mul;
	var sub = subtract;
	mat3.sub = sub;
	return mat3;
}

var mat4 = {};

var hasRequiredMat4;

function requireMat4 () {
	if (hasRequiredMat4) return mat4;
	hasRequiredMat4 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(mat4, '__esModule', { value: true });
	mat4.create = create;
	mat4.clone = clone;
	mat4.copy = copy;
	mat4.fromValues = fromValues;
	mat4.set = set;
	mat4.identity = identity;
	mat4.transpose = transpose;
	mat4.invert = invert;
	mat4.adjoint = adjoint;
	mat4.determinant = determinant;
	mat4.multiply = multiply;
	mat4.translate = translate;
	mat4.scale = scale;
	mat4.rotate = rotate;
	mat4.rotateX = rotateX;
	mat4.rotateY = rotateY;
	mat4.rotateZ = rotateZ;
	mat4.fromTranslation = fromTranslation;
	mat4.fromScaling = fromScaling;
	mat4.fromRotation = fromRotation;
	mat4.fromXRotation = fromXRotation;
	mat4.fromYRotation = fromYRotation;
	mat4.fromZRotation = fromZRotation;
	mat4.fromRotationTranslation = fromRotationTranslation;
	mat4.fromQuat2 = fromQuat2;
	mat4.getTranslation = getTranslation;
	mat4.getScaling = getScaling;
	mat4.getRotation = getRotation;
	mat4.fromRotationTranslationScale = fromRotationTranslationScale;
	mat4.fromRotationTranslationScaleOrigin = fromRotationTranslationScaleOrigin;
	mat4.fromQuat = fromQuat;
	mat4.frustum = frustum;
	mat4.perspectiveNO = perspectiveNO;
	mat4.perspectiveZO = perspectiveZO;
	mat4.perspectiveFromFieldOfView = perspectiveFromFieldOfView;
	mat4.orthoNO = orthoNO;
	mat4.orthoZO = orthoZO;
	mat4.lookAt = lookAt;
	mat4.targetTo = targetTo;
	mat4.str = str;
	mat4.frob = frob;
	mat4.add = add;
	mat4.subtract = subtract;
	mat4.multiplyScalar = multiplyScalar;
	mat4.multiplyScalarAndAdd = multiplyScalarAndAdd;
	mat4.exactEquals = exactEquals;
	mat4.equals = equals;
	mat4.sub = mat4.mul = mat4.ortho = mat4.perspective = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 4x4 Matrix<br>Format: column-major, when typed out it looks like row-major<br>The matrices are being post multiplied.
	 * @module mat4
	 */
	/**
	 * Creates a new identity mat4
	 *
	 * @returns {mat4} a new 4x4 matrix
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(16);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[1] = 0;
	        out[2] = 0;
	        out[3] = 0;
	        out[4] = 0;
	        out[6] = 0;
	        out[7] = 0;
	        out[8] = 0;
	        out[9] = 0;
	        out[11] = 0;
	        out[12] = 0;
	        out[13] = 0;
	        out[14] = 0;
	    }
	    out[0] = 1;
	    out[5] = 1;
	    out[10] = 1;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a new mat4 initialized with values from an existing matrix
	 *
	 * @param {ReadonlyMat4} a matrix to clone
	 * @returns {mat4} a new 4x4 matrix
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(16);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    out[8] = a[8];
	    out[9] = a[9];
	    out[10] = a[10];
	    out[11] = a[11];
	    out[12] = a[12];
	    out[13] = a[13];
	    out[14] = a[14];
	    out[15] = a[15];
	    return out;
	}
	/**
	 * Copy the values from one mat4 to another
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the source matrix
	 * @returns {mat4} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    out[8] = a[8];
	    out[9] = a[9];
	    out[10] = a[10];
	    out[11] = a[11];
	    out[12] = a[12];
	    out[13] = a[13];
	    out[14] = a[14];
	    out[15] = a[15];
	    return out;
	}
	/**
	 * Create a new mat4 with the given values
	 *
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m02 Component in column 0, row 2 position (index 2)
	 * @param {Number} m03 Component in column 0, row 3 position (index 3)
	 * @param {Number} m10 Component in column 1, row 0 position (index 4)
	 * @param {Number} m11 Component in column 1, row 1 position (index 5)
	 * @param {Number} m12 Component in column 1, row 2 position (index 6)
	 * @param {Number} m13 Component in column 1, row 3 position (index 7)
	 * @param {Number} m20 Component in column 2, row 0 position (index 8)
	 * @param {Number} m21 Component in column 2, row 1 position (index 9)
	 * @param {Number} m22 Component in column 2, row 2 position (index 10)
	 * @param {Number} m23 Component in column 2, row 3 position (index 11)
	 * @param {Number} m30 Component in column 3, row 0 position (index 12)
	 * @param {Number} m31 Component in column 3, row 1 position (index 13)
	 * @param {Number} m32 Component in column 3, row 2 position (index 14)
	 * @param {Number} m33 Component in column 3, row 3 position (index 15)
	 * @returns {mat4} A new mat4
	 */
	function fromValues(m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33) {
	    var out = new glMatrix.ARRAY_TYPE(16);
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m02;
	    out[3] = m03;
	    out[4] = m10;
	    out[5] = m11;
	    out[6] = m12;
	    out[7] = m13;
	    out[8] = m20;
	    out[9] = m21;
	    out[10] = m22;
	    out[11] = m23;
	    out[12] = m30;
	    out[13] = m31;
	    out[14] = m32;
	    out[15] = m33;
	    return out;
	}
	/**
	 * Set the components of a mat4 to the given values
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {Number} m00 Component in column 0, row 0 position (index 0)
	 * @param {Number} m01 Component in column 0, row 1 position (index 1)
	 * @param {Number} m02 Component in column 0, row 2 position (index 2)
	 * @param {Number} m03 Component in column 0, row 3 position (index 3)
	 * @param {Number} m10 Component in column 1, row 0 position (index 4)
	 * @param {Number} m11 Component in column 1, row 1 position (index 5)
	 * @param {Number} m12 Component in column 1, row 2 position (index 6)
	 * @param {Number} m13 Component in column 1, row 3 position (index 7)
	 * @param {Number} m20 Component in column 2, row 0 position (index 8)
	 * @param {Number} m21 Component in column 2, row 1 position (index 9)
	 * @param {Number} m22 Component in column 2, row 2 position (index 10)
	 * @param {Number} m23 Component in column 2, row 3 position (index 11)
	 * @param {Number} m30 Component in column 3, row 0 position (index 12)
	 * @param {Number} m31 Component in column 3, row 1 position (index 13)
	 * @param {Number} m32 Component in column 3, row 2 position (index 14)
	 * @param {Number} m33 Component in column 3, row 3 position (index 15)
	 * @returns {mat4} out
	 */
	function set(out, m00, m01, m02, m03, m10, m11, m12, m13, m20, m21, m22, m23, m30, m31, m32, m33) {
	    out[0] = m00;
	    out[1] = m01;
	    out[2] = m02;
	    out[3] = m03;
	    out[4] = m10;
	    out[5] = m11;
	    out[6] = m12;
	    out[7] = m13;
	    out[8] = m20;
	    out[9] = m21;
	    out[10] = m22;
	    out[11] = m23;
	    out[12] = m30;
	    out[13] = m31;
	    out[14] = m32;
	    out[15] = m33;
	    return out;
	}
	/**
	 * Set a mat4 to the identity matrix
	 *
	 * @param {mat4} out the receiving matrix
	 * @returns {mat4} out
	 */
	function identity(out) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = 1;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = 1;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Transpose the values of a mat4
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the source matrix
	 * @returns {mat4} out
	 */
	function transpose(out, a) {
	    // If we are transposing ourselves we can skip a few steps but have to cache some values
	    if (out === a) {
	        var a01 = a[1], a02 = a[2], a03 = a[3];
	        var a12 = a[6], a13 = a[7];
	        var a23 = a[11];
	        out[1] = a[4];
	        out[2] = a[8];
	        out[3] = a[12];
	        out[4] = a01;
	        out[6] = a[9];
	        out[7] = a[13];
	        out[8] = a02;
	        out[9] = a12;
	        out[11] = a[14];
	        out[12] = a03;
	        out[13] = a13;
	        out[14] = a23;
	    } else {
	        out[0] = a[0];
	        out[1] = a[4];
	        out[2] = a[8];
	        out[3] = a[12];
	        out[4] = a[1];
	        out[5] = a[5];
	        out[6] = a[9];
	        out[7] = a[13];
	        out[8] = a[2];
	        out[9] = a[6];
	        out[10] = a[10];
	        out[11] = a[14];
	        out[12] = a[3];
	        out[13] = a[7];
	        out[14] = a[11];
	        out[15] = a[15];
	    }
	    return out;
	}
	/**
	 * Inverts a mat4
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the source matrix
	 * @returns {mat4} out
	 */
	function invert(out, a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3];
	    var a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7];
	    var a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11];
	    var a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];
	    var b00 = a00 * a11 - a01 * a10;
	    var b01 = a00 * a12 - a02 * a10;
	    var b02 = a00 * a13 - a03 * a10;
	    var b03 = a01 * a12 - a02 * a11;
	    var b04 = a01 * a13 - a03 * a11;
	    var b05 = a02 * a13 - a03 * a12;
	    var b06 = a20 * a31 - a21 * a30;
	    var b07 = a20 * a32 - a22 * a30;
	    var b08 = a20 * a33 - a23 * a30;
	    var b09 = a21 * a32 - a22 * a31;
	    var b10 = a21 * a33 - a23 * a31;
	    var b11 = a22 * a33 - a23 * a32;
	    // Calculate the determinant
	    var det = b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
	    if (!det) {
	        return null;
	    }
	    det = 1 / det;
	    out[0] = (a11 * b11 - a12 * b10 + a13 * b09) * det;
	    out[1] = (a02 * b10 - a01 * b11 - a03 * b09) * det;
	    out[2] = (a31 * b05 - a32 * b04 + a33 * b03) * det;
	    out[3] = (a22 * b04 - a21 * b05 - a23 * b03) * det;
	    out[4] = (a12 * b08 - a10 * b11 - a13 * b07) * det;
	    out[5] = (a00 * b11 - a02 * b08 + a03 * b07) * det;
	    out[6] = (a32 * b02 - a30 * b05 - a33 * b01) * det;
	    out[7] = (a20 * b05 - a22 * b02 + a23 * b01) * det;
	    out[8] = (a10 * b10 - a11 * b08 + a13 * b06) * det;
	    out[9] = (a01 * b08 - a00 * b10 - a03 * b06) * det;
	    out[10] = (a30 * b04 - a31 * b02 + a33 * b00) * det;
	    out[11] = (a21 * b02 - a20 * b04 - a23 * b00) * det;
	    out[12] = (a11 * b07 - a10 * b09 - a12 * b06) * det;
	    out[13] = (a00 * b09 - a01 * b07 + a02 * b06) * det;
	    out[14] = (a31 * b01 - a30 * b03 - a32 * b00) * det;
	    out[15] = (a20 * b03 - a21 * b01 + a22 * b00) * det;
	    return out;
	}
	/**
	 * Calculates the adjugate of a mat4
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the source matrix
	 * @returns {mat4} out
	 */
	function adjoint(out, a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3];
	    var a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7];
	    var a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11];
	    var a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];
	    out[0] = a11 * (a22 * a33 - a23 * a32) - a21 * (a12 * a33 - a13 * a32) + a31 * (a12 * a23 - a13 * a22);
	    out[1] = -(a01 * (a22 * a33 - a23 * a32) - a21 * (a02 * a33 - a03 * a32) + a31 * (a02 * a23 - a03 * a22));
	    out[2] = a01 * (a12 * a33 - a13 * a32) - a11 * (a02 * a33 - a03 * a32) + a31 * (a02 * a13 - a03 * a12);
	    out[3] = -(a01 * (a12 * a23 - a13 * a22) - a11 * (a02 * a23 - a03 * a22) + a21 * (a02 * a13 - a03 * a12));
	    out[4] = -(a10 * (a22 * a33 - a23 * a32) - a20 * (a12 * a33 - a13 * a32) + a30 * (a12 * a23 - a13 * a22));
	    out[5] = a00 * (a22 * a33 - a23 * a32) - a20 * (a02 * a33 - a03 * a32) + a30 * (a02 * a23 - a03 * a22);
	    out[6] = -(a00 * (a12 * a33 - a13 * a32) - a10 * (a02 * a33 - a03 * a32) + a30 * (a02 * a13 - a03 * a12));
	    out[7] = a00 * (a12 * a23 - a13 * a22) - a10 * (a02 * a23 - a03 * a22) + a20 * (a02 * a13 - a03 * a12);
	    out[8] = a10 * (a21 * a33 - a23 * a31) - a20 * (a11 * a33 - a13 * a31) + a30 * (a11 * a23 - a13 * a21);
	    out[9] = -(a00 * (a21 * a33 - a23 * a31) - a20 * (a01 * a33 - a03 * a31) + a30 * (a01 * a23 - a03 * a21));
	    out[10] = a00 * (a11 * a33 - a13 * a31) - a10 * (a01 * a33 - a03 * a31) + a30 * (a01 * a13 - a03 * a11);
	    out[11] = -(a00 * (a11 * a23 - a13 * a21) - a10 * (a01 * a23 - a03 * a21) + a20 * (a01 * a13 - a03 * a11));
	    out[12] = -(a10 * (a21 * a32 - a22 * a31) - a20 * (a11 * a32 - a12 * a31) + a30 * (a11 * a22 - a12 * a21));
	    out[13] = a00 * (a21 * a32 - a22 * a31) - a20 * (a01 * a32 - a02 * a31) + a30 * (a01 * a22 - a02 * a21);
	    out[14] = -(a00 * (a11 * a32 - a12 * a31) - a10 * (a01 * a32 - a02 * a31) + a30 * (a01 * a12 - a02 * a11));
	    out[15] = a00 * (a11 * a22 - a12 * a21) - a10 * (a01 * a22 - a02 * a21) + a20 * (a01 * a12 - a02 * a11);
	    return out;
	}
	/**
	 * Calculates the determinant of a mat4
	 *
	 * @param {ReadonlyMat4} a the source matrix
	 * @returns {Number} determinant of a
	 */
	function determinant(a) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3];
	    var a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7];
	    var a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11];
	    var a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];
	    var b00 = a00 * a11 - a01 * a10;
	    var b01 = a00 * a12 - a02 * a10;
	    var b02 = a00 * a13 - a03 * a10;
	    var b03 = a01 * a12 - a02 * a11;
	    var b04 = a01 * a13 - a03 * a11;
	    var b05 = a02 * a13 - a03 * a12;
	    var b06 = a20 * a31 - a21 * a30;
	    var b07 = a20 * a32 - a22 * a30;
	    var b08 = a20 * a33 - a23 * a30;
	    var b09 = a21 * a32 - a22 * a31;
	    var b10 = a21 * a33 - a23 * a31;
	    var b11 = a22 * a33 - a23 * a32;
	    // Calculate the determinant
	    return b00 * b11 - b01 * b10 + b02 * b09 + b03 * b08 - b04 * b07 + b05 * b06;
	}
	/**
	 * Multiplies two mat4s
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the first operand
	 * @param {ReadonlyMat4} b the second operand
	 * @returns {mat4} out
	 */
	function multiply(out, a, b) {
	    var a00 = a[0], a01 = a[1], a02 = a[2], a03 = a[3];
	    var a10 = a[4], a11 = a[5], a12 = a[6], a13 = a[7];
	    var a20 = a[8], a21 = a[9], a22 = a[10], a23 = a[11];
	    var a30 = a[12], a31 = a[13], a32 = a[14], a33 = a[15];
	    // Cache only the current line of the second matrix
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
	    out[0] = b0 * a00 + b1 * a10 + b2 * a20 + b3 * a30;
	    out[1] = b0 * a01 + b1 * a11 + b2 * a21 + b3 * a31;
	    out[2] = b0 * a02 + b1 * a12 + b2 * a22 + b3 * a32;
	    out[3] = b0 * a03 + b1 * a13 + b2 * a23 + b3 * a33;
	    b0 = b[4];
	    b1 = b[5];
	    b2 = b[6];
	    b3 = b[7];
	    out[4] = b0 * a00 + b1 * a10 + b2 * a20 + b3 * a30;
	    out[5] = b0 * a01 + b1 * a11 + b2 * a21 + b3 * a31;
	    out[6] = b0 * a02 + b1 * a12 + b2 * a22 + b3 * a32;
	    out[7] = b0 * a03 + b1 * a13 + b2 * a23 + b3 * a33;
	    b0 = b[8];
	    b1 = b[9];
	    b2 = b[10];
	    b3 = b[11];
	    out[8] = b0 * a00 + b1 * a10 + b2 * a20 + b3 * a30;
	    out[9] = b0 * a01 + b1 * a11 + b2 * a21 + b3 * a31;
	    out[10] = b0 * a02 + b1 * a12 + b2 * a22 + b3 * a32;
	    out[11] = b0 * a03 + b1 * a13 + b2 * a23 + b3 * a33;
	    b0 = b[12];
	    b1 = b[13];
	    b2 = b[14];
	    b3 = b[15];
	    out[12] = b0 * a00 + b1 * a10 + b2 * a20 + b3 * a30;
	    out[13] = b0 * a01 + b1 * a11 + b2 * a21 + b3 * a31;
	    out[14] = b0 * a02 + b1 * a12 + b2 * a22 + b3 * a32;
	    out[15] = b0 * a03 + b1 * a13 + b2 * a23 + b3 * a33;
	    return out;
	}
	/**
	 * Translate a mat4 by the given vector
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to translate
	 * @param {ReadonlyVec3} v vector to translate by
	 * @returns {mat4} out
	 */
	function translate(out, a, v) {
	    var x = v[0], y = v[1], z = v[2];
	    var a00, a01, a02, a03;
	    var a10, a11, a12, a13;
	    var a20, a21, a22, a23;
	    if (a === out) {
	        out[12] = a[0] * x + a[4] * y + a[8] * z + a[12];
	        out[13] = a[1] * x + a[5] * y + a[9] * z + a[13];
	        out[14] = a[2] * x + a[6] * y + a[10] * z + a[14];
	        out[15] = a[3] * x + a[7] * y + a[11] * z + a[15];
	    } else {
	        a00 = a[0];
	        a01 = a[1];
	        a02 = a[2];
	        a03 = a[3];
	        a10 = a[4];
	        a11 = a[5];
	        a12 = a[6];
	        a13 = a[7];
	        a20 = a[8];
	        a21 = a[9];
	        a22 = a[10];
	        a23 = a[11];
	        out[0] = a00;
	        out[1] = a01;
	        out[2] = a02;
	        out[3] = a03;
	        out[4] = a10;
	        out[5] = a11;
	        out[6] = a12;
	        out[7] = a13;
	        out[8] = a20;
	        out[9] = a21;
	        out[10] = a22;
	        out[11] = a23;
	        out[12] = a00 * x + a10 * y + a20 * z + a[12];
	        out[13] = a01 * x + a11 * y + a21 * z + a[13];
	        out[14] = a02 * x + a12 * y + a22 * z + a[14];
	        out[15] = a03 * x + a13 * y + a23 * z + a[15];
	    }
	    return out;
	}
	/**
	 * Scales the mat4 by the dimensions in the given vec3 not using vectorization
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to scale
	 * @param {ReadonlyVec3} v the vec3 to scale the matrix by
	 * @returns {mat4} out
	 **/
	function scale(out, a, v) {
	    var x = v[0], y = v[1], z = v[2];
	    out[0] = a[0] * x;
	    out[1] = a[1] * x;
	    out[2] = a[2] * x;
	    out[3] = a[3] * x;
	    out[4] = a[4] * y;
	    out[5] = a[5] * y;
	    out[6] = a[6] * y;
	    out[7] = a[7] * y;
	    out[8] = a[8] * z;
	    out[9] = a[9] * z;
	    out[10] = a[10] * z;
	    out[11] = a[11] * z;
	    out[12] = a[12];
	    out[13] = a[13];
	    out[14] = a[14];
	    out[15] = a[15];
	    return out;
	}
	/**
	 * Rotates a mat4 by the given angle around the given axis
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @param {ReadonlyVec3} axis the axis to rotate around
	 * @returns {mat4} out
	 */
	function rotate(out, a, rad, axis) {
	    var x = axis[0], y = axis[1], z = axis[2];
	    var len = Math.hypot(x, y, z);
	    var s, c, t;
	    var a00, a01, a02, a03;
	    var a10, a11, a12, a13;
	    var a20, a21, a22, a23;
	    var b00, b01, b02;
	    var b10, b11, b12;
	    var b20, b21, b22;
	    if (len < glMatrix.EPSILON) {
	        return null;
	    }
	    len = 1 / len;
	    x *= len;
	    y *= len;
	    z *= len;
	    s = Math.sin(rad);
	    c = Math.cos(rad);
	    t = 1 - c;
	    a00 = a[0];
	    a01 = a[1];
	    a02 = a[2];
	    a03 = a[3];
	    a10 = a[4];
	    a11 = a[5];
	    a12 = a[6];
	    a13 = a[7];
	    a20 = a[8];
	    a21 = a[9];
	    a22 = a[10];
	    a23 = a[11];
	    // Construct the elements of the rotation matrix
	    b00 = x * x * t + c;
	    b01 = y * x * t + z * s;
	    b02 = z * x * t - y * s;
	    b10 = x * y * t - z * s;
	    b11 = y * y * t + c;
	    b12 = z * y * t + x * s;
	    b20 = x * z * t + y * s;
	    b21 = y * z * t - x * s;
	    b22 = z * z * t + c;
	    // Perform rotation-specific matrix multiplication
	    out[0] = a00 * b00 + a10 * b01 + a20 * b02;
	    out[1] = a01 * b00 + a11 * b01 + a21 * b02;
	    out[2] = a02 * b00 + a12 * b01 + a22 * b02;
	    out[3] = a03 * b00 + a13 * b01 + a23 * b02;
	    out[4] = a00 * b10 + a10 * b11 + a20 * b12;
	    out[5] = a01 * b10 + a11 * b11 + a21 * b12;
	    out[6] = a02 * b10 + a12 * b11 + a22 * b12;
	    out[7] = a03 * b10 + a13 * b11 + a23 * b12;
	    out[8] = a00 * b20 + a10 * b21 + a20 * b22;
	    out[9] = a01 * b20 + a11 * b21 + a21 * b22;
	    out[10] = a02 * b20 + a12 * b21 + a22 * b22;
	    out[11] = a03 * b20 + a13 * b21 + a23 * b22;
	    if (a !== out) {
	        // If the source and destination differ, copy the unchanged last row
	        out[12] = a[12];
	        out[13] = a[13];
	        out[14] = a[14];
	        out[15] = a[15];
	    }
	    return out;
	}
	/**
	 * Rotates a matrix by the given angle around the X axis
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function rotateX(out, a, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    var a10 = a[4];
	    var a11 = a[5];
	    var a12 = a[6];
	    var a13 = a[7];
	    var a20 = a[8];
	    var a21 = a[9];
	    var a22 = a[10];
	    var a23 = a[11];
	    if (a !== out) {
	        // If the source and destination differ, copy the unchanged rows
	        out[0] = a[0];
	        out[1] = a[1];
	        out[2] = a[2];
	        out[3] = a[3];
	        out[12] = a[12];
	        out[13] = a[13];
	        out[14] = a[14];
	        out[15] = a[15];
	    }
	    // Perform axis-specific matrix multiplication
	    out[4] = a10 * c + a20 * s;
	    out[5] = a11 * c + a21 * s;
	    out[6] = a12 * c + a22 * s;
	    out[7] = a13 * c + a23 * s;
	    out[8] = a20 * c - a10 * s;
	    out[9] = a21 * c - a11 * s;
	    out[10] = a22 * c - a12 * s;
	    out[11] = a23 * c - a13 * s;
	    return out;
	}
	/**
	 * Rotates a matrix by the given angle around the Y axis
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function rotateY(out, a, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    var a00 = a[0];
	    var a01 = a[1];
	    var a02 = a[2];
	    var a03 = a[3];
	    var a20 = a[8];
	    var a21 = a[9];
	    var a22 = a[10];
	    var a23 = a[11];
	    if (a !== out) {
	        // If the source and destination differ, copy the unchanged rows
	        out[4] = a[4];
	        out[5] = a[5];
	        out[6] = a[6];
	        out[7] = a[7];
	        out[12] = a[12];
	        out[13] = a[13];
	        out[14] = a[14];
	        out[15] = a[15];
	    }
	    // Perform axis-specific matrix multiplication
	    out[0] = a00 * c - a20 * s;
	    out[1] = a01 * c - a21 * s;
	    out[2] = a02 * c - a22 * s;
	    out[3] = a03 * c - a23 * s;
	    out[8] = a00 * s + a20 * c;
	    out[9] = a01 * s + a21 * c;
	    out[10] = a02 * s + a22 * c;
	    out[11] = a03 * s + a23 * c;
	    return out;
	}
	/**
	 * Rotates a matrix by the given angle around the Z axis
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to rotate
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function rotateZ(out, a, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    var a00 = a[0];
	    var a01 = a[1];
	    var a02 = a[2];
	    var a03 = a[3];
	    var a10 = a[4];
	    var a11 = a[5];
	    var a12 = a[6];
	    var a13 = a[7];
	    if (a !== out) {
	        // If the source and destination differ, copy the unchanged last row
	        out[8] = a[8];
	        out[9] = a[9];
	        out[10] = a[10];
	        out[11] = a[11];
	        out[12] = a[12];
	        out[13] = a[13];
	        out[14] = a[14];
	        out[15] = a[15];
	    }
	    // Perform axis-specific matrix multiplication
	    out[0] = a00 * c + a10 * s;
	    out[1] = a01 * c + a11 * s;
	    out[2] = a02 * c + a12 * s;
	    out[3] = a03 * c + a13 * s;
	    out[4] = a10 * c - a00 * s;
	    out[5] = a11 * c - a01 * s;
	    out[6] = a12 * c - a02 * s;
	    out[7] = a13 * c - a03 * s;
	    return out;
	}
	/**
	 * Creates a matrix from a vector translation
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.translate(dest, dest, vec);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {ReadonlyVec3} v Translation vector
	 * @returns {mat4} out
	 */
	function fromTranslation(out, v) {
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = 1;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = 1;
	    out[11] = 0;
	    out[12] = v[0];
	    out[13] = v[1];
	    out[14] = v[2];
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a vector scaling
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.scale(dest, dest, vec);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {ReadonlyVec3} v Scaling vector
	 * @returns {mat4} out
	 */
	function fromScaling(out, v) {
	    out[0] = v[0];
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = v[1];
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = v[2];
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a given angle around a given axis
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.rotate(dest, dest, rad, axis);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @param {ReadonlyVec3} axis the axis to rotate around
	 * @returns {mat4} out
	 */
	function fromRotation(out, rad, axis) {
	    var x = axis[0], y = axis[1], z = axis[2];
	    var len = Math.hypot(x, y, z);
	    var s, c, t;
	    if (len < glMatrix.EPSILON) {
	        return null;
	    }
	    len = 1 / len;
	    x *= len;
	    y *= len;
	    z *= len;
	    s = Math.sin(rad);
	    c = Math.cos(rad);
	    t = 1 - c;
	    // Perform rotation-specific matrix multiplication
	    out[0] = x * x * t + c;
	    out[1] = y * x * t + z * s;
	    out[2] = z * x * t - y * s;
	    out[3] = 0;
	    out[4] = x * y * t - z * s;
	    out[5] = y * y * t + c;
	    out[6] = z * y * t + x * s;
	    out[7] = 0;
	    out[8] = x * z * t + y * s;
	    out[9] = y * z * t - x * s;
	    out[10] = z * z * t + c;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from the given angle around the X axis
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.rotateX(dest, dest, rad);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function fromXRotation(out, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    // Perform axis-specific matrix multiplication
	    out[0] = 1;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = c;
	    out[6] = s;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = -s;
	    out[10] = c;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from the given angle around the Y axis
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.rotateY(dest, dest, rad);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function fromYRotation(out, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    // Perform axis-specific matrix multiplication
	    out[0] = c;
	    out[1] = 0;
	    out[2] = -s;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = 1;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = s;
	    out[9] = 0;
	    out[10] = c;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from the given angle around the Z axis
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.rotateZ(dest, dest, rad);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {Number} rad the angle to rotate the matrix by
	 * @returns {mat4} out
	 */
	function fromZRotation(out, rad) {
	    var s = Math.sin(rad);
	    var c = Math.cos(rad);
	    // Perform axis-specific matrix multiplication
	    out[0] = c;
	    out[1] = s;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = -s;
	    out[5] = c;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = 1;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a quaternion rotation and vector translation
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.translate(dest, vec);
	 *     let quatMat = mat4.create();
	 *     quat4.toMat4(quat, quatMat);
	 *     mat4.multiply(dest, quatMat);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {quat4} q Rotation quaternion
	 * @param {ReadonlyVec3} v Translation vector
	 * @returns {mat4} out
	 */
	function fromRotationTranslation(out, q, v) {
	    // Quaternion math
	    var x = q[0], y = q[1], z = q[2], w = q[3];
	    var x2 = x + x;
	    var y2 = y + y;
	    var z2 = z + z;
	    var xx = x * x2;
	    var xy = x * y2;
	    var xz = x * z2;
	    var yy = y * y2;
	    var yz = y * z2;
	    var zz = z * z2;
	    var wx = w * x2;
	    var wy = w * y2;
	    var wz = w * z2;
	    out[0] = 1 - (yy + zz);
	    out[1] = xy + wz;
	    out[2] = xz - wy;
	    out[3] = 0;
	    out[4] = xy - wz;
	    out[5] = 1 - (xx + zz);
	    out[6] = yz + wx;
	    out[7] = 0;
	    out[8] = xz + wy;
	    out[9] = yz - wx;
	    out[10] = 1 - (xx + yy);
	    out[11] = 0;
	    out[12] = v[0];
	    out[13] = v[1];
	    out[14] = v[2];
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a new mat4 from a dual quat.
	 *
	 * @param {mat4} out Matrix
	 * @param {ReadonlyQuat2} a Dual Quaternion
	 * @returns {mat4} mat4 receiving operation result
	 */
	function fromQuat2(out, a) {
	    var translation = new glMatrix.ARRAY_TYPE(3);
	    var bx = -a[0], by = -a[1], bz = -a[2], bw = a[3], ax = a[4], ay = a[5], az = a[6], aw = a[7];
	    var magnitude = bx * bx + by * by + bz * bz + bw * bw;
	    //Only scale if it makes sense
	    if (magnitude > 0) {
	        translation[0] = (ax * bw + aw * bx + ay * bz - az * by) * 2 / magnitude;
	        translation[1] = (ay * bw + aw * by + az * bx - ax * bz) * 2 / magnitude;
	        translation[2] = (az * bw + aw * bz + ax * by - ay * bx) * 2 / magnitude;
	    } else {
	        translation[0] = (ax * bw + aw * bx + ay * bz - az * by) * 2;
	        translation[1] = (ay * bw + aw * by + az * bx - ax * bz) * 2;
	        translation[2] = (az * bw + aw * bz + ax * by - ay * bx) * 2;
	    }
	    fromRotationTranslation(out, a, translation);
	    return out;
	}
	/**
	 * Returns the translation vector component of a transformation
	 *  matrix. If a matrix is built with fromRotationTranslation,
	 *  the returned vector will be the same as the translation vector
	 *  originally supplied.
	 * @param  {vec3} out Vector to receive translation component
	 * @param  {ReadonlyMat4} mat Matrix to be decomposed (input)
	 * @return {vec3} out
	 */
	function getTranslation(out, mat) {
	    out[0] = mat[12];
	    out[1] = mat[13];
	    out[2] = mat[14];
	    return out;
	}
	/**
	 * Returns the scaling factor component of a transformation
	 *  matrix. If a matrix is built with fromRotationTranslationScale
	 *  with a normalized Quaternion paramter, the returned vector will be
	 *  the same as the scaling vector
	 *  originally supplied.
	 * @param  {vec3} out Vector to receive scaling factor component
	 * @param  {ReadonlyMat4} mat Matrix to be decomposed (input)
	 * @return {vec3} out
	 */
	function getScaling(out, mat) {
	    var m11 = mat[0];
	    var m12 = mat[1];
	    var m13 = mat[2];
	    var m21 = mat[4];
	    var m22 = mat[5];
	    var m23 = mat[6];
	    var m31 = mat[8];
	    var m32 = mat[9];
	    var m33 = mat[10];
	    out[0] = Math.hypot(m11, m12, m13);
	    out[1] = Math.hypot(m21, m22, m23);
	    out[2] = Math.hypot(m31, m32, m33);
	    return out;
	}
	/**
	 * Returns a quaternion representing the rotational component
	 *  of a transformation matrix. If a matrix is built with
	 *  fromRotationTranslation, the returned quaternion will be the
	 *  same as the quaternion originally supplied.
	 * @param {quat} out Quaternion to receive the rotation component
	 * @param {ReadonlyMat4} mat Matrix to be decomposed (input)
	 * @return {quat} out
	 */
	function getRotation(out, mat) {
	    var scaling = new glMatrix.ARRAY_TYPE(3);
	    getScaling(scaling, mat);
	    var is1 = 1 / scaling[0];
	    var is2 = 1 / scaling[1];
	    var is3 = 1 / scaling[2];
	    var sm11 = mat[0] * is1;
	    var sm12 = mat[1] * is2;
	    var sm13 = mat[2] * is3;
	    var sm21 = mat[4] * is1;
	    var sm22 = mat[5] * is2;
	    var sm23 = mat[6] * is3;
	    var sm31 = mat[8] * is1;
	    var sm32 = mat[9] * is2;
	    var sm33 = mat[10] * is3;
	    var trace = sm11 + sm22 + sm33;
	    var S = 0;
	    if (trace > 0) {
	        S = Math.sqrt(trace + 1) * 2;
	        out[3] = 0.25 * S;
	        out[0] = (sm23 - sm32) / S;
	        out[1] = (sm31 - sm13) / S;
	        out[2] = (sm12 - sm21) / S;
	    } else if (sm11 > sm22 && sm11 > sm33) {
	        S = Math.sqrt(1 + sm11 - sm22 - sm33) * 2;
	        out[3] = (sm23 - sm32) / S;
	        out[0] = 0.25 * S;
	        out[1] = (sm12 + sm21) / S;
	        out[2] = (sm31 + sm13) / S;
	    } else if (sm22 > sm33) {
	        S = Math.sqrt(1 + sm22 - sm11 - sm33) * 2;
	        out[3] = (sm31 - sm13) / S;
	        out[0] = (sm12 + sm21) / S;
	        out[1] = 0.25 * S;
	        out[2] = (sm23 + sm32) / S;
	    } else {
	        S = Math.sqrt(1 + sm33 - sm11 - sm22) * 2;
	        out[3] = (sm12 - sm21) / S;
	        out[0] = (sm31 + sm13) / S;
	        out[1] = (sm23 + sm32) / S;
	        out[2] = 0.25 * S;
	    }
	    return out;
	}
	/**
	 * Creates a matrix from a quaternion rotation, vector translation and vector scale
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.translate(dest, vec);
	 *     let quatMat = mat4.create();
	 *     quat4.toMat4(quat, quatMat);
	 *     mat4.multiply(dest, quatMat);
	 *     mat4.scale(dest, scale)
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {quat4} q Rotation quaternion
	 * @param {ReadonlyVec3} v Translation vector
	 * @param {ReadonlyVec3} s Scaling vector
	 * @returns {mat4} out
	 */
	function fromRotationTranslationScale(out, q, v, s) {
	    // Quaternion math
	    var x = q[0], y = q[1], z = q[2], w = q[3];
	    var x2 = x + x;
	    var y2 = y + y;
	    var z2 = z + z;
	    var xx = x * x2;
	    var xy = x * y2;
	    var xz = x * z2;
	    var yy = y * y2;
	    var yz = y * z2;
	    var zz = z * z2;
	    var wx = w * x2;
	    var wy = w * y2;
	    var wz = w * z2;
	    var sx = s[0];
	    var sy = s[1];
	    var sz = s[2];
	    out[0] = (1 - (yy + zz)) * sx;
	    out[1] = (xy + wz) * sx;
	    out[2] = (xz - wy) * sx;
	    out[3] = 0;
	    out[4] = (xy - wz) * sy;
	    out[5] = (1 - (xx + zz)) * sy;
	    out[6] = (yz + wx) * sy;
	    out[7] = 0;
	    out[8] = (xz + wy) * sz;
	    out[9] = (yz - wx) * sz;
	    out[10] = (1 - (xx + yy)) * sz;
	    out[11] = 0;
	    out[12] = v[0];
	    out[13] = v[1];
	    out[14] = v[2];
	    out[15] = 1;
	    return out;
	}
	/**
	 * Creates a matrix from a quaternion rotation, vector translation and vector scale, rotating and scaling around the given origin
	 * This is equivalent to (but much faster than):
	 *
	 *     mat4.identity(dest);
	 *     mat4.translate(dest, vec);
	 *     mat4.translate(dest, origin);
	 *     let quatMat = mat4.create();
	 *     quat4.toMat4(quat, quatMat);
	 *     mat4.multiply(dest, quatMat);
	 *     mat4.scale(dest, scale)
	 *     mat4.translate(dest, negativeOrigin);
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {quat4} q Rotation quaternion
	 * @param {ReadonlyVec3} v Translation vector
	 * @param {ReadonlyVec3} s Scaling vector
	 * @param {ReadonlyVec3} o The origin vector around which to scale and rotate
	 * @returns {mat4} out
	 */
	function fromRotationTranslationScaleOrigin(out, q, v, s, o) {
	    // Quaternion math
	    var x = q[0], y = q[1], z = q[2], w = q[3];
	    var x2 = x + x;
	    var y2 = y + y;
	    var z2 = z + z;
	    var xx = x * x2;
	    var xy = x * y2;
	    var xz = x * z2;
	    var yy = y * y2;
	    var yz = y * z2;
	    var zz = z * z2;
	    var wx = w * x2;
	    var wy = w * y2;
	    var wz = w * z2;
	    var sx = s[0];
	    var sy = s[1];
	    var sz = s[2];
	    var ox = o[0];
	    var oy = o[1];
	    var oz = o[2];
	    var out0 = (1 - (yy + zz)) * sx;
	    var out1 = (xy + wz) * sx;
	    var out2 = (xz - wy) * sx;
	    var out4 = (xy - wz) * sy;
	    var out5 = (1 - (xx + zz)) * sy;
	    var out6 = (yz + wx) * sy;
	    var out8 = (xz + wy) * sz;
	    var out9 = (yz - wx) * sz;
	    var out10 = (1 - (xx + yy)) * sz;
	    out[0] = out0;
	    out[1] = out1;
	    out[2] = out2;
	    out[3] = 0;
	    out[4] = out4;
	    out[5] = out5;
	    out[6] = out6;
	    out[7] = 0;
	    out[8] = out8;
	    out[9] = out9;
	    out[10] = out10;
	    out[11] = 0;
	    out[12] = v[0] + ox - (out0 * ox + out4 * oy + out8 * oz);
	    out[13] = v[1] + oy - (out1 * ox + out5 * oy + out9 * oz);
	    out[14] = v[2] + oz - (out2 * ox + out6 * oy + out10 * oz);
	    out[15] = 1;
	    return out;
	}
	/**
	 * Calculates a 4x4 matrix from the given quaternion
	 *
	 * @param {mat4} out mat4 receiving operation result
	 * @param {ReadonlyQuat} q Quaternion to create matrix from
	 *
	 * @returns {mat4} out
	 */
	function fromQuat(out, q) {
	    var x = q[0], y = q[1], z = q[2], w = q[3];
	    var x2 = x + x;
	    var y2 = y + y;
	    var z2 = z + z;
	    var xx = x * x2;
	    var yx = y * x2;
	    var yy = y * y2;
	    var zx = z * x2;
	    var zy = z * y2;
	    var zz = z * z2;
	    var wx = w * x2;
	    var wy = w * y2;
	    var wz = w * z2;
	    out[0] = 1 - yy - zz;
	    out[1] = yx + wz;
	    out[2] = zx - wy;
	    out[3] = 0;
	    out[4] = yx - wz;
	    out[5] = 1 - xx - zz;
	    out[6] = zy + wx;
	    out[7] = 0;
	    out[8] = zx + wy;
	    out[9] = zy - wx;
	    out[10] = 1 - xx - yy;
	    out[11] = 0;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = 0;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Generates a frustum matrix with the given bounds
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {Number} left Left bound of the frustum
	 * @param {Number} right Right bound of the frustum
	 * @param {Number} bottom Bottom bound of the frustum
	 * @param {Number} top Top bound of the frustum
	 * @param {Number} near Near bound of the frustum
	 * @param {Number} far Far bound of the frustum
	 * @returns {mat4} out
	 */
	function frustum(out, left, right, bottom, top, near, far) {
	    var rl = 1 / (right - left);
	    var tb = 1 / (top - bottom);
	    var nf = 1 / (near - far);
	    out[0] = near * 2 * rl;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = near * 2 * tb;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = (right + left) * rl;
	    out[9] = (top + bottom) * tb;
	    out[10] = (far + near) * nf;
	    out[11] = -1;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = far * near * 2 * nf;
	    out[15] = 0;
	    return out;
	}
	/**
	 * Generates a perspective projection matrix with the given bounds.
	 * The near/far clip planes correspond to a normalized device coordinate Z range of [-1, 1],
	 * which matches WebGL/OpenGL's clip volume.
	 * Passing null/undefined/no value for far will generate infinite projection matrix.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {number} fovy Vertical field of view in radians
	 * @param {number} aspect Aspect ratio. typically viewport width/height
	 * @param {number} near Near bound of the frustum
	 * @param {number} far Far bound of the frustum, can be null or Infinity
	 * @returns {mat4} out
	 */
	function perspectiveNO(out, fovy, aspect, near, far) {
	    var f = 1 / Math.tan(fovy / 2), nf;
	    out[0] = f / aspect;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = f;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[11] = -1;
	    out[12] = 0;
	    out[13] = 0;
	    out[15] = 0;
	    if (far != null && far !== Infinity) {
	        nf = 1 / (near - far);
	        out[10] = (far + near) * nf;
	        out[14] = 2 * far * near * nf;
	    } else {
	        out[10] = -1;
	        out[14] = -2 * near;
	    }
	    return out;
	}
	/**
	 * Alias for {@link mat4.perspectiveNO}
	 * @function
	 */
	var perspective = perspectiveNO;
	/**
	 * Generates a perspective projection matrix suitable for WebGPU with the given bounds.
	 * The near/far clip planes correspond to a normalized device coordinate Z range of [0, 1],
	 * which matches WebGPU/Vulkan/DirectX/Metal's clip volume.
	 * Passing null/undefined/no value for far will generate infinite projection matrix.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {number} fovy Vertical field of view in radians
	 * @param {number} aspect Aspect ratio. typically viewport width/height
	 * @param {number} near Near bound of the frustum
	 * @param {number} far Far bound of the frustum, can be null or Infinity
	 * @returns {mat4} out
	 */
	mat4.perspective = perspective;
	function perspectiveZO(out, fovy, aspect, near, far) {
	    var f = 1 / Math.tan(fovy / 2), nf;
	    out[0] = f / aspect;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = f;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[11] = -1;
	    out[12] = 0;
	    out[13] = 0;
	    out[15] = 0;
	    if (far != null && far !== Infinity) {
	        nf = 1 / (near - far);
	        out[10] = far * nf;
	        out[14] = far * near * nf;
	    } else {
	        out[10] = -1;
	        out[14] = -near;
	    }
	    return out;
	}
	/**
	 * Generates a perspective projection matrix with the given field of view.
	 * This is primarily useful for generating projection matrices to be used
	 * with the still experiemental WebVR API.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {Object} fov Object containing the following values: upDegrees, downDegrees, leftDegrees, rightDegrees
	 * @param {number} near Near bound of the frustum
	 * @param {number} far Far bound of the frustum
	 * @returns {mat4} out
	 */
	function perspectiveFromFieldOfView(out, fov, near, far) {
	    var upTan = Math.tan(fov.upDegrees * Math.PI / 180);
	    var downTan = Math.tan(fov.downDegrees * Math.PI / 180);
	    var leftTan = Math.tan(fov.leftDegrees * Math.PI / 180);
	    var rightTan = Math.tan(fov.rightDegrees * Math.PI / 180);
	    var xScale = 2 / (leftTan + rightTan);
	    var yScale = 2 / (upTan + downTan);
	    out[0] = xScale;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = yScale;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = -((leftTan - rightTan) * xScale * 0.5);
	    out[9] = (upTan - downTan) * yScale * 0.5;
	    out[10] = far / (near - far);
	    out[11] = -1;
	    out[12] = 0;
	    out[13] = 0;
	    out[14] = far * near / (near - far);
	    out[15] = 0;
	    return out;
	}
	/**
	 * Generates a orthogonal projection matrix with the given bounds.
	 * The near/far clip planes correspond to a normalized device coordinate Z range of [-1, 1],
	 * which matches WebGL/OpenGL's clip volume.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {number} left Left bound of the frustum
	 * @param {number} right Right bound of the frustum
	 * @param {number} bottom Bottom bound of the frustum
	 * @param {number} top Top bound of the frustum
	 * @param {number} near Near bound of the frustum
	 * @param {number} far Far bound of the frustum
	 * @returns {mat4} out
	 */
	function orthoNO(out, left, right, bottom, top, near, far) {
	    var lr = 1 / (left - right);
	    var bt = 1 / (bottom - top);
	    var nf = 1 / (near - far);
	    out[0] = -2 * lr;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = -2 * bt;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = 2 * nf;
	    out[11] = 0;
	    out[12] = (left + right) * lr;
	    out[13] = (top + bottom) * bt;
	    out[14] = (far + near) * nf;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Alias for {@link mat4.orthoNO}
	 * @function
	 */
	var ortho = orthoNO;
	/**
	 * Generates a orthogonal projection matrix with the given bounds.
	 * The near/far clip planes correspond to a normalized device coordinate Z range of [0, 1],
	 * which matches WebGPU/Vulkan/DirectX/Metal's clip volume.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {number} left Left bound of the frustum
	 * @param {number} right Right bound of the frustum
	 * @param {number} bottom Bottom bound of the frustum
	 * @param {number} top Top bound of the frustum
	 * @param {number} near Near bound of the frustum
	 * @param {number} far Far bound of the frustum
	 * @returns {mat4} out
	 */
	mat4.ortho = ortho;
	function orthoZO(out, left, right, bottom, top, near, far) {
	    var lr = 1 / (left - right);
	    var bt = 1 / (bottom - top);
	    var nf = 1 / (near - far);
	    out[0] = -2 * lr;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    out[4] = 0;
	    out[5] = -2 * bt;
	    out[6] = 0;
	    out[7] = 0;
	    out[8] = 0;
	    out[9] = 0;
	    out[10] = nf;
	    out[11] = 0;
	    out[12] = (left + right) * lr;
	    out[13] = (top + bottom) * bt;
	    out[14] = near * nf;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Generates a look-at matrix with the given eye position, focal point, and up axis.
	 * If you want a matrix that actually makes an object look at another object, you should use targetTo instead.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {ReadonlyVec3} eye Position of the viewer
	 * @param {ReadonlyVec3} center Point the viewer is looking at
	 * @param {ReadonlyVec3} up vec3 pointing up
	 * @returns {mat4} out
	 */
	function lookAt(out, eye, center, up) {
	    var x0, x1, x2, y0, y1, y2, z0, z1, z2, len;
	    var eyex = eye[0];
	    var eyey = eye[1];
	    var eyez = eye[2];
	    var upx = up[0];
	    var upy = up[1];
	    var upz = up[2];
	    var centerx = center[0];
	    var centery = center[1];
	    var centerz = center[2];
	    if (Math.abs(eyex - centerx) < glMatrix.EPSILON && Math.abs(eyey - centery) < glMatrix.EPSILON && Math.abs(eyez - centerz) < glMatrix.EPSILON) {
	        return identity(out);
	    }
	    z0 = eyex - centerx;
	    z1 = eyey - centery;
	    z2 = eyez - centerz;
	    len = 1 / Math.hypot(z0, z1, z2);
	    z0 *= len;
	    z1 *= len;
	    z2 *= len;
	    x0 = upy * z2 - upz * z1;
	    x1 = upz * z0 - upx * z2;
	    x2 = upx * z1 - upy * z0;
	    len = Math.hypot(x0, x1, x2);
	    if (!len) {
	        x0 = 0;
	        x1 = 0;
	        x2 = 0;
	    } else {
	        len = 1 / len;
	        x0 *= len;
	        x1 *= len;
	        x2 *= len;
	    }
	    y0 = z1 * x2 - z2 * x1;
	    y1 = z2 * x0 - z0 * x2;
	    y2 = z0 * x1 - z1 * x0;
	    len = Math.hypot(y0, y1, y2);
	    if (!len) {
	        y0 = 0;
	        y1 = 0;
	        y2 = 0;
	    } else {
	        len = 1 / len;
	        y0 *= len;
	        y1 *= len;
	        y2 *= len;
	    }
	    out[0] = x0;
	    out[1] = y0;
	    out[2] = z0;
	    out[3] = 0;
	    out[4] = x1;
	    out[5] = y1;
	    out[6] = z1;
	    out[7] = 0;
	    out[8] = x2;
	    out[9] = y2;
	    out[10] = z2;
	    out[11] = 0;
	    out[12] = -(x0 * eyex + x1 * eyey + x2 * eyez);
	    out[13] = -(y0 * eyex + y1 * eyey + y2 * eyez);
	    out[14] = -(z0 * eyex + z1 * eyey + z2 * eyez);
	    out[15] = 1;
	    return out;
	}
	/**
	 * Generates a matrix that makes something look at something else.
	 *
	 * @param {mat4} out mat4 frustum matrix will be written into
	 * @param {ReadonlyVec3} eye Position of the viewer
	 * @param {ReadonlyVec3} center Point the viewer is looking at
	 * @param {ReadonlyVec3} up vec3 pointing up
	 * @returns {mat4} out
	 */
	function targetTo(out, eye, target, up) {
	    var eyex = eye[0], eyey = eye[1], eyez = eye[2], upx = up[0], upy = up[1], upz = up[2];
	    var z0 = eyex - target[0], z1 = eyey - target[1], z2 = eyez - target[2];
	    var len = z0 * z0 + z1 * z1 + z2 * z2;
	    if (len > 0) {
	        len = 1 / Math.sqrt(len);
	        z0 *= len;
	        z1 *= len;
	        z2 *= len;
	    }
	    var x0 = upy * z2 - upz * z1, x1 = upz * z0 - upx * z2, x2 = upx * z1 - upy * z0;
	    len = x0 * x0 + x1 * x1 + x2 * x2;
	    if (len > 0) {
	        len = 1 / Math.sqrt(len);
	        x0 *= len;
	        x1 *= len;
	        x2 *= len;
	    }
	    out[0] = x0;
	    out[1] = x1;
	    out[2] = x2;
	    out[3] = 0;
	    out[4] = z1 * x2 - z2 * x1;
	    out[5] = z2 * x0 - z0 * x2;
	    out[6] = z0 * x1 - z1 * x0;
	    out[7] = 0;
	    out[8] = z0;
	    out[9] = z1;
	    out[10] = z2;
	    out[11] = 0;
	    out[12] = eyex;
	    out[13] = eyey;
	    out[14] = eyez;
	    out[15] = 1;
	    return out;
	}
	/**
	 * Returns a string representation of a mat4
	 *
	 * @param {ReadonlyMat4} a matrix to represent as a string
	 * @returns {String} string representation of the matrix
	 */
	function str(a) {
	    return 'mat4(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ', ' + a[4] + ', ' + a[5] + ', ' + a[6] + ', ' + a[7] + ', ' + a[8] + ', ' + a[9] + ', ' + a[10] + ', ' + a[11] + ', ' + a[12] + ', ' + a[13] + ', ' + a[14] + ', ' + a[15] + ')';
	}
	/**
	 * Returns Frobenius norm of a mat4
	 *
	 * @param {ReadonlyMat4} a the matrix to calculate Frobenius norm of
	 * @returns {Number} Frobenius norm
	 */
	function frob(a) {
	    return Math.hypot(a[0], a[1], a[2], a[3], a[4], a[5], a[6], a[7], a[8], a[9], a[10], a[11], a[12], a[13], a[14], a[15]);
	}
	/**
	 * Adds two mat4's
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the first operand
	 * @param {ReadonlyMat4} b the second operand
	 * @returns {mat4} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    out[4] = a[4] + b[4];
	    out[5] = a[5] + b[5];
	    out[6] = a[6] + b[6];
	    out[7] = a[7] + b[7];
	    out[8] = a[8] + b[8];
	    out[9] = a[9] + b[9];
	    out[10] = a[10] + b[10];
	    out[11] = a[11] + b[11];
	    out[12] = a[12] + b[12];
	    out[13] = a[13] + b[13];
	    out[14] = a[14] + b[14];
	    out[15] = a[15] + b[15];
	    return out;
	}
	/**
	 * Subtracts matrix b from matrix a
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the first operand
	 * @param {ReadonlyMat4} b the second operand
	 * @returns {mat4} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    out[3] = a[3] - b[3];
	    out[4] = a[4] - b[4];
	    out[5] = a[5] - b[5];
	    out[6] = a[6] - b[6];
	    out[7] = a[7] - b[7];
	    out[8] = a[8] - b[8];
	    out[9] = a[9] - b[9];
	    out[10] = a[10] - b[10];
	    out[11] = a[11] - b[11];
	    out[12] = a[12] - b[12];
	    out[13] = a[13] - b[13];
	    out[14] = a[14] - b[14];
	    out[15] = a[15] - b[15];
	    return out;
	}
	/**
	 * Multiply each element of the matrix by a scalar.
	 *
	 * @param {mat4} out the receiving matrix
	 * @param {ReadonlyMat4} a the matrix to scale
	 * @param {Number} b amount to scale the matrix's elements by
	 * @returns {mat4} out
	 */
	function multiplyScalar(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    out[4] = a[4] * b;
	    out[5] = a[5] * b;
	    out[6] = a[6] * b;
	    out[7] = a[7] * b;
	    out[8] = a[8] * b;
	    out[9] = a[9] * b;
	    out[10] = a[10] * b;
	    out[11] = a[11] * b;
	    out[12] = a[12] * b;
	    out[13] = a[13] * b;
	    out[14] = a[14] * b;
	    out[15] = a[15] * b;
	    return out;
	}
	/**
	 * Adds two mat4's after multiplying each element of the second operand by a scalar value.
	 *
	 * @param {mat4} out the receiving vector
	 * @param {ReadonlyMat4} a the first operand
	 * @param {ReadonlyMat4} b the second operand
	 * @param {Number} scale the amount to scale b's elements by before adding
	 * @returns {mat4} out
	 */
	function multiplyScalarAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    out[3] = a[3] + b[3] * scale;
	    out[4] = a[4] + b[4] * scale;
	    out[5] = a[5] + b[5] * scale;
	    out[6] = a[6] + b[6] * scale;
	    out[7] = a[7] + b[7] * scale;
	    out[8] = a[8] + b[8] * scale;
	    out[9] = a[9] + b[9] * scale;
	    out[10] = a[10] + b[10] * scale;
	    out[11] = a[11] + b[11] * scale;
	    out[12] = a[12] + b[12] * scale;
	    out[13] = a[13] + b[13] * scale;
	    out[14] = a[14] + b[14] * scale;
	    out[15] = a[15] + b[15] * scale;
	    return out;
	}
	/**
	 * Returns whether or not the matrices have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyMat4} a The first matrix.
	 * @param {ReadonlyMat4} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3] && a[4] === b[4] && a[5] === b[5] && a[6] === b[6] && a[7] === b[7] && a[8] === b[8] && a[9] === b[9] && a[10] === b[10] && a[11] === b[11] && a[12] === b[12] && a[13] === b[13] && a[14] === b[14] && a[15] === b[15];
	}
	/**
	 * Returns whether or not the matrices have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyMat4} a The first matrix.
	 * @param {ReadonlyMat4} b The second matrix.
	 * @returns {Boolean} True if the matrices are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var a4 = a[4], a5 = a[5], a6 = a[6], a7 = a[7];
	    var a8 = a[8], a9 = a[9], a10 = a[10], a11 = a[11];
	    var a12 = a[12], a13 = a[13], a14 = a[14], a15 = a[15];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
	    var b4 = b[4], b5 = b[5], b6 = b[6], b7 = b[7];
	    var b8 = b[8], b9 = b[9], b10 = b[10], b11 = b[11];
	    var b12 = b[12], b13 = b[13], b14 = b[14], b15 = b[15];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3)) && Math.abs(a4 - b4) <= glMatrix.EPSILON * Math.max(1, Math.abs(a4), Math.abs(b4)) && Math.abs(a5 - b5) <= glMatrix.EPSILON * Math.max(1, Math.abs(a5), Math.abs(b5)) && Math.abs(a6 - b6) <= glMatrix.EPSILON * Math.max(1, Math.abs(a6), Math.abs(b6)) && Math.abs(a7 - b7) <= glMatrix.EPSILON * Math.max(1, Math.abs(a7), Math.abs(b7)) && Math.abs(a8 - b8) <= glMatrix.EPSILON * Math.max(1, Math.abs(a8), Math.abs(b8)) && Math.abs(a9 - b9) <= glMatrix.EPSILON * Math.max(1, Math.abs(a9), Math.abs(b9)) && Math.abs(a10 - b10) <= glMatrix.EPSILON * Math.max(1, Math.abs(a10), Math.abs(b10)) && Math.abs(a11 - b11) <= glMatrix.EPSILON * Math.max(1, Math.abs(a11), Math.abs(b11)) && Math.abs(a12 - b12) <= glMatrix.EPSILON * Math.max(1, Math.abs(a12), Math.abs(b12)) && Math.abs(a13 - b13) <= glMatrix.EPSILON * Math.max(1, Math.abs(a13), Math.abs(b13)) && Math.abs(a14 - b14) <= glMatrix.EPSILON * Math.max(1, Math.abs(a14), Math.abs(b14)) && Math.abs(a15 - b15) <= glMatrix.EPSILON * Math.max(1, Math.abs(a15), Math.abs(b15));
	}
	/**
	 * Alias for {@link mat4.multiply}
	 * @function
	 */
	var mul = multiply;
	/**
	 * Alias for {@link mat4.subtract}
	 * @function
	 */
	mat4.mul = mul;
	var sub = subtract;
	mat4.sub = sub;
	return mat4;
}

var quat = {};

var vec3 = {};

var hasRequiredVec3;

function requireVec3 () {
	if (hasRequiredVec3) return vec3;
	hasRequiredVec3 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(vec3, '__esModule', { value: true });
	vec3.create = create;
	vec3.clone = clone;
	vec3.length = length;
	vec3.fromValues = fromValues;
	vec3.copy = copy;
	vec3.set = set;
	vec3.add = add;
	vec3.subtract = subtract;
	vec3.multiply = multiply;
	vec3.divide = divide;
	vec3.ceil = ceil;
	vec3.floor = floor;
	vec3.min = min;
	vec3.max = max;
	vec3.round = round;
	vec3.scale = scale;
	vec3.scaleAndAdd = scaleAndAdd;
	vec3.distance = distance;
	vec3.squaredDistance = squaredDistance;
	vec3.squaredLength = squaredLength;
	vec3.negate = negate;
	vec3.inverse = inverse;
	vec3.normalize = normalize;
	vec3.dot = dot;
	vec3.cross = cross;
	vec3.lerp = lerp;
	vec3.hermite = hermite;
	vec3.bezier = bezier;
	vec3.random = random;
	vec3.transformMat4 = transformMat4;
	vec3.transformMat3 = transformMat3;
	vec3.transformQuat = transformQuat;
	vec3.rotateX = rotateX;
	vec3.rotateY = rotateY;
	vec3.rotateZ = rotateZ;
	vec3.angle = angle;
	vec3.zero = zero;
	vec3.str = str;
	vec3.exactEquals = exactEquals;
	vec3.equals = equals;
	vec3.forEach = vec3.sqrLen = vec3.len = vec3.sqrDist = vec3.dist = vec3.div = vec3.mul = vec3.sub = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 3 Dimensional Vector
	 * @module vec3
	 */
	/**
	 * Creates a new, empty vec3
	 *
	 * @returns {vec3} a new 3D vector
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(3);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[0] = 0;
	        out[1] = 0;
	        out[2] = 0;
	    }
	    return out;
	}
	/**
	 * Creates a new vec3 initialized with values from an existing vector
	 *
	 * @param {ReadonlyVec3} a vector to clone
	 * @returns {vec3} a new 3D vector
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(3);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    return out;
	}
	/**
	 * Calculates the length of a vec3
	 *
	 * @param {ReadonlyVec3} a vector to calculate length of
	 * @returns {Number} length of a
	 */
	function length(a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    return Math.hypot(x, y, z);
	}
	/**
	 * Creates a new vec3 initialized with the given values
	 *
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @returns {vec3} a new 3D vector
	 */
	function fromValues(x, y, z) {
	    var out = new glMatrix.ARRAY_TYPE(3);
	    out[0] = x;
	    out[1] = y;
	    out[2] = z;
	    return out;
	}
	/**
	 * Copy the values from one vec3 to another
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the source vector
	 * @returns {vec3} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    return out;
	}
	/**
	 * Set the components of a vec3 to the given values
	 *
	 * @param {vec3} out the receiving vector
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @returns {vec3} out
	 */
	function set(out, x, y, z) {
	    out[0] = x;
	    out[1] = y;
	    out[2] = z;
	    return out;
	}
	/**
	 * Adds two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    return out;
	}
	/**
	 * Subtracts vector b from vector a
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    return out;
	}
	/**
	 * Multiplies two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function multiply(out, a, b) {
	    out[0] = a[0] * b[0];
	    out[1] = a[1] * b[1];
	    out[2] = a[2] * b[2];
	    return out;
	}
	/**
	 * Divides two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function divide(out, a, b) {
	    out[0] = a[0] / b[0];
	    out[1] = a[1] / b[1];
	    out[2] = a[2] / b[2];
	    return out;
	}
	/**
	 * Math.ceil the components of a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to ceil
	 * @returns {vec3} out
	 */
	function ceil(out, a) {
	    out[0] = Math.ceil(a[0]);
	    out[1] = Math.ceil(a[1]);
	    out[2] = Math.ceil(a[2]);
	    return out;
	}
	/**
	 * Math.floor the components of a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to floor
	 * @returns {vec3} out
	 */
	function floor(out, a) {
	    out[0] = Math.floor(a[0]);
	    out[1] = Math.floor(a[1]);
	    out[2] = Math.floor(a[2]);
	    return out;
	}
	/**
	 * Returns the minimum of two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function min(out, a, b) {
	    out[0] = Math.min(a[0], b[0]);
	    out[1] = Math.min(a[1], b[1]);
	    out[2] = Math.min(a[2], b[2]);
	    return out;
	}
	/**
	 * Returns the maximum of two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function max(out, a, b) {
	    out[0] = Math.max(a[0], b[0]);
	    out[1] = Math.max(a[1], b[1]);
	    out[2] = Math.max(a[2], b[2]);
	    return out;
	}
	/**
	 * Math.round the components of a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to round
	 * @returns {vec3} out
	 */
	function round(out, a) {
	    out[0] = Math.round(a[0]);
	    out[1] = Math.round(a[1]);
	    out[2] = Math.round(a[2]);
	    return out;
	}
	/**
	 * Scales a vec3 by a scalar number
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the vector to scale
	 * @param {Number} b amount to scale the vector by
	 * @returns {vec3} out
	 */
	function scale(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    return out;
	}
	/**
	 * Adds two vec3's after scaling the second operand by a scalar value
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @param {Number} scale the amount to scale b by before adding
	 * @returns {vec3} out
	 */
	function scaleAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    return out;
	}
	/**
	 * Calculates the euclidian distance between two vec3's
	 *
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {Number} distance between a and b
	 */
	function distance(a, b) {
	    var x = b[0] - a[0];
	    var y = b[1] - a[1];
	    var z = b[2] - a[2];
	    return Math.hypot(x, y, z);
	}
	/**
	 * Calculates the squared euclidian distance between two vec3's
	 *
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {Number} squared distance between a and b
	 */
	function squaredDistance(a, b) {
	    var x = b[0] - a[0];
	    var y = b[1] - a[1];
	    var z = b[2] - a[2];
	    return x * x + y * y + z * z;
	}
	/**
	 * Calculates the squared length of a vec3
	 *
	 * @param {ReadonlyVec3} a vector to calculate squared length of
	 * @returns {Number} squared length of a
	 */
	function squaredLength(a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    return x * x + y * y + z * z;
	}
	/**
	 * Negates the components of a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to negate
	 * @returns {vec3} out
	 */
	function negate(out, a) {
	    out[0] = -a[0];
	    out[1] = -a[1];
	    out[2] = -a[2];
	    return out;
	}
	/**
	 * Returns the inverse of the components of a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to invert
	 * @returns {vec3} out
	 */
	function inverse(out, a) {
	    out[0] = 1 / a[0];
	    out[1] = 1 / a[1];
	    out[2] = 1 / a[2];
	    return out;
	}
	/**
	 * Normalize a vec3
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a vector to normalize
	 * @returns {vec3} out
	 */
	function normalize(out, a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    var len = x * x + y * y + z * z;
	    if (len > 0) {
	        //TODO: evaluate use of glm_invsqrt here?
	        len = 1 / Math.sqrt(len);
	    }
	    out[0] = a[0] * len;
	    out[1] = a[1] * len;
	    out[2] = a[2] * len;
	    return out;
	}
	/**
	 * Calculates the dot product of two vec3's
	 *
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {Number} dot product of a and b
	 */
	function dot(a, b) {
	    return a[0] * b[0] + a[1] * b[1] + a[2] * b[2];
	}
	/**
	 * Computes the cross product of two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @returns {vec3} out
	 */
	function cross(out, a, b) {
	    var ax = a[0], ay = a[1], az = a[2];
	    var bx = b[0], by = b[1], bz = b[2];
	    out[0] = ay * bz - az * by;
	    out[1] = az * bx - ax * bz;
	    out[2] = ax * by - ay * bx;
	    return out;
	}
	/**
	 * Performs a linear interpolation between two vec3's
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {vec3} out
	 */
	function lerp(out, a, b, t) {
	    var ax = a[0];
	    var ay = a[1];
	    var az = a[2];
	    out[0] = ax + t * (b[0] - ax);
	    out[1] = ay + t * (b[1] - ay);
	    out[2] = az + t * (b[2] - az);
	    return out;
	}
	/**
	 * Performs a hermite interpolation with two control points
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @param {ReadonlyVec3} c the third operand
	 * @param {ReadonlyVec3} d the fourth operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {vec3} out
	 */
	function hermite(out, a, b, c, d, t) {
	    var factorTimes2 = t * t;
	    var factor1 = factorTimes2 * (2 * t - 3) + 1;
	    var factor2 = factorTimes2 * (t - 2) + t;
	    var factor3 = factorTimes2 * (t - 1);
	    var factor4 = factorTimes2 * (3 - 2 * t);
	    out[0] = a[0] * factor1 + b[0] * factor2 + c[0] * factor3 + d[0] * factor4;
	    out[1] = a[1] * factor1 + b[1] * factor2 + c[1] * factor3 + d[1] * factor4;
	    out[2] = a[2] * factor1 + b[2] * factor2 + c[2] * factor3 + d[2] * factor4;
	    return out;
	}
	/**
	 * Performs a bezier interpolation with two control points
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the first operand
	 * @param {ReadonlyVec3} b the second operand
	 * @param {ReadonlyVec3} c the third operand
	 * @param {ReadonlyVec3} d the fourth operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {vec3} out
	 */
	function bezier(out, a, b, c, d, t) {
	    var inverseFactor = 1 - t;
	    var inverseFactorTimesTwo = inverseFactor * inverseFactor;
	    var factorTimes2 = t * t;
	    var factor1 = inverseFactorTimesTwo * inverseFactor;
	    var factor2 = 3 * t * inverseFactorTimesTwo;
	    var factor3 = 3 * factorTimes2 * inverseFactor;
	    var factor4 = factorTimes2 * t;
	    out[0] = a[0] * factor1 + b[0] * factor2 + c[0] * factor3 + d[0] * factor4;
	    out[1] = a[1] * factor1 + b[1] * factor2 + c[1] * factor3 + d[1] * factor4;
	    out[2] = a[2] * factor1 + b[2] * factor2 + c[2] * factor3 + d[2] * factor4;
	    return out;
	}
	/**
	 * Generates a random vector with the given scale
	 *
	 * @param {vec3} out the receiving vector
	 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
	 * @returns {vec3} out
	 */
	function random(out, scale) {
	    scale = scale || 1;
	    var r = glMatrix.RANDOM() * 2 * Math.PI;
	    var z = glMatrix.RANDOM() * 2 - 1;
	    var zScale = Math.sqrt(1 - z * z) * scale;
	    out[0] = Math.cos(r) * zScale;
	    out[1] = Math.sin(r) * zScale;
	    out[2] = z * scale;
	    return out;
	}
	/**
	 * Transforms the vec3 with a mat4.
	 * 4th vector component is implicitly '1'
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the vector to transform
	 * @param {ReadonlyMat4} m matrix to transform with
	 * @returns {vec3} out
	 */
	function transformMat4(out, a, m) {
	    var x = a[0], y = a[1], z = a[2];
	    var w = m[3] * x + m[7] * y + m[11] * z + m[15];
	    w = w || 1;
	    out[0] = (m[0] * x + m[4] * y + m[8] * z + m[12]) / w;
	    out[1] = (m[1] * x + m[5] * y + m[9] * z + m[13]) / w;
	    out[2] = (m[2] * x + m[6] * y + m[10] * z + m[14]) / w;
	    return out;
	}
	/**
	 * Transforms the vec3 with a mat3.
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the vector to transform
	 * @param {ReadonlyMat3} m the 3x3 matrix to transform with
	 * @returns {vec3} out
	 */
	function transformMat3(out, a, m) {
	    var x = a[0], y = a[1], z = a[2];
	    out[0] = x * m[0] + y * m[3] + z * m[6];
	    out[1] = x * m[1] + y * m[4] + z * m[7];
	    out[2] = x * m[2] + y * m[5] + z * m[8];
	    return out;
	}
	/**
	 * Transforms the vec3 with a quat
	 * Can also be used for dual quaternions. (Multiply it with the real part)
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec3} a the vector to transform
	 * @param {ReadonlyQuat} q quaternion to transform with
	 * @returns {vec3} out
	 */
	function transformQuat(out, a, q) {
	    // benchmarks: https://jsperf.com/quaternion-transform-vec3-implementations-fixed
	    var qx = q[0], qy = q[1], qz = q[2], qw = q[3];
	    var x = a[0], y = a[1], z = a[2];
	    // var qvec = [qx, qy, qz];
	    // var uv = vec3.cross([], qvec, a);
	    var uvx = qy * z - qz * y, uvy = qz * x - qx * z, uvz = qx * y - qy * x;
	    // var uuv = vec3.cross([], qvec, uv);
	    var uuvx = qy * uvz - qz * uvy, uuvy = qz * uvx - qx * uvz, uuvz = qx * uvy - qy * uvx;
	    // vec3.scale(uv, uv, 2 * w);
	    var w2 = qw * 2;
	    uvx *= w2;
	    uvy *= w2;
	    uvz *= w2;
	    // vec3.scale(uuv, uuv, 2);
	    uuvx *= 2;
	    uuvy *= 2;
	    uuvz *= 2;
	    // return vec3.add(out, a, vec3.add(out, uv, uuv));
	    out[0] = x + uvx + uuvx;
	    out[1] = y + uvy + uuvy;
	    out[2] = z + uvz + uuvz;
	    return out;
	}
	/**
	 * Rotate a 3D vector around the x-axis
	 * @param {vec3} out The receiving vec3
	 * @param {ReadonlyVec3} a The vec3 point to rotate
	 * @param {ReadonlyVec3} b The origin of the rotation
	 * @param {Number} rad The angle of rotation in radians
	 * @returns {vec3} out
	 */
	function rotateX(out, a, b, rad) {
	    var p = [], r = [];
	    //Translate point to the origin
	    p[0] = a[0] - b[0];
	    p[1] = a[1] - b[1];
	    p[2] = a[2] - b[2];
	    //perform rotation
	    r[0] = p[0];
	    r[1] = p[1] * Math.cos(rad) - p[2] * Math.sin(rad);
	    r[2] = p[1] * Math.sin(rad) + p[2] * Math.cos(rad);
	    //translate to correct position
	    out[0] = r[0] + b[0];
	    out[1] = r[1] + b[1];
	    out[2] = r[2] + b[2];
	    return out;
	}
	/**
	 * Rotate a 3D vector around the y-axis
	 * @param {vec3} out The receiving vec3
	 * @param {ReadonlyVec3} a The vec3 point to rotate
	 * @param {ReadonlyVec3} b The origin of the rotation
	 * @param {Number} rad The angle of rotation in radians
	 * @returns {vec3} out
	 */
	function rotateY(out, a, b, rad) {
	    var p = [], r = [];
	    //Translate point to the origin
	    p[0] = a[0] - b[0];
	    p[1] = a[1] - b[1];
	    p[2] = a[2] - b[2];
	    //perform rotation
	    r[0] = p[2] * Math.sin(rad) + p[0] * Math.cos(rad);
	    r[1] = p[1];
	    r[2] = p[2] * Math.cos(rad) - p[0] * Math.sin(rad);
	    //translate to correct position
	    out[0] = r[0] + b[0];
	    out[1] = r[1] + b[1];
	    out[2] = r[2] + b[2];
	    return out;
	}
	/**
	 * Rotate a 3D vector around the z-axis
	 * @param {vec3} out The receiving vec3
	 * @param {ReadonlyVec3} a The vec3 point to rotate
	 * @param {ReadonlyVec3} b The origin of the rotation
	 * @param {Number} rad The angle of rotation in radians
	 * @returns {vec3} out
	 */
	function rotateZ(out, a, b, rad) {
	    var p = [], r = [];
	    //Translate point to the origin
	    p[0] = a[0] - b[0];
	    p[1] = a[1] - b[1];
	    p[2] = a[2] - b[2];
	    //perform rotation
	    r[0] = p[0] * Math.cos(rad) - p[1] * Math.sin(rad);
	    r[1] = p[0] * Math.sin(rad) + p[1] * Math.cos(rad);
	    r[2] = p[2];
	    //translate to correct position
	    out[0] = r[0] + b[0];
	    out[1] = r[1] + b[1];
	    out[2] = r[2] + b[2];
	    return out;
	}
	/**
	 * Get the angle between two 3D vectors
	 * @param {ReadonlyVec3} a The first operand
	 * @param {ReadonlyVec3} b The second operand
	 * @returns {Number} The angle in radians
	 */
	function angle(a, b) {
	    var ax = a[0], ay = a[1], az = a[2], bx = b[0], by = b[1], bz = b[2], mag1 = Math.sqrt(ax * ax + ay * ay + az * az), mag2 = Math.sqrt(bx * bx + by * by + bz * bz), mag = mag1 * mag2, cosine = mag && dot(a, b) / mag;
	    return Math.acos(Math.min(Math.max(cosine, -1), 1));
	}
	/**
	 * Set the components of a vec3 to zero
	 *
	 * @param {vec3} out the receiving vector
	 * @returns {vec3} out
	 */
	function zero(out) {
	    out[0] = 0;
	    out[1] = 0;
	    out[2] = 0;
	    return out;
	}
	/**
	 * Returns a string representation of a vector
	 *
	 * @param {ReadonlyVec3} a vector to represent as a string
	 * @returns {String} string representation of the vector
	 */
	function str(a) {
	    return 'vec3(' + a[0] + ', ' + a[1] + ', ' + a[2] + ')';
	}
	/**
	 * Returns whether or not the vectors have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyVec3} a The first vector.
	 * @param {ReadonlyVec3} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2];
	}
	/**
	 * Returns whether or not the vectors have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyVec3} a The first vector.
	 * @param {ReadonlyVec3} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2];
	    var b0 = b[0], b1 = b[1], b2 = b[2];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2));
	}
	/**
	 * Alias for {@link vec3.subtract}
	 * @function
	 */
	var sub = subtract;
	/**
	 * Alias for {@link vec3.multiply}
	 * @function
	 */
	vec3.sub = sub;
	var mul = multiply;
	/**
	 * Alias for {@link vec3.divide}
	 * @function
	 */
	vec3.mul = mul;
	var div = divide;
	/**
	 * Alias for {@link vec3.distance}
	 * @function
	 */
	vec3.div = div;
	var dist = distance;
	/**
	 * Alias for {@link vec3.squaredDistance}
	 * @function
	 */
	vec3.dist = dist;
	var sqrDist = squaredDistance;
	/**
	 * Alias for {@link vec3.length}
	 * @function
	 */
	vec3.sqrDist = sqrDist;
	var len = length;
	/**
	 * Alias for {@link vec3.squaredLength}
	 * @function
	 */
	vec3.len = len;
	var sqrLen = squaredLength;
	/**
	 * Perform some operation over an array of vec3s.
	 *
	 * @param {Array} a the array of vectors to iterate over
	 * @param {Number} stride Number of elements between the start of each vec3. If 0 assumes tightly packed
	 * @param {Number} offset Number of elements to skip at the beginning of the array
	 * @param {Number} count Number of vec3s to iterate over. If 0 iterates over entire array
	 * @param {Function} fn Function to call for each vector in the array
	 * @param {Object} [arg] additional argument to pass to fn
	 * @returns {Array} a
	 * @function
	 */
	vec3.sqrLen = sqrLen;
	var forEach = (function () {
	    var vec = create();
	    return function (a, stride, offset, count, fn, arg) {
	        var i, l;
	        if (!stride) {
	            stride = 3;
	        }
	        if (!offset) {
	            offset = 0;
	        }
	        if (count) {
	            l = Math.min(count * stride + offset, a.length);
	        } else {
	            l = a.length;
	        }
	        for (i = offset; i < l; i += stride) {
	            vec[0] = a[i];
	            vec[1] = a[i + 1];
	            vec[2] = a[i + 2];
	            fn(vec, vec, arg);
	            a[i] = vec[0];
	            a[i + 1] = vec[1];
	            a[i + 2] = vec[2];
	        }
	        return a;
	    };
	}());
	vec3.forEach = forEach;
	return vec3;
}

var vec4 = {};

var hasRequiredVec4;

function requireVec4 () {
	if (hasRequiredVec4) return vec4;
	hasRequiredVec4 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(vec4, '__esModule', { value: true });
	vec4.create = create;
	vec4.clone = clone;
	vec4.fromValues = fromValues;
	vec4.copy = copy;
	vec4.set = set;
	vec4.add = add;
	vec4.subtract = subtract;
	vec4.multiply = multiply;
	vec4.divide = divide;
	vec4.ceil = ceil;
	vec4.floor = floor;
	vec4.min = min;
	vec4.max = max;
	vec4.round = round;
	vec4.scale = scale;
	vec4.scaleAndAdd = scaleAndAdd;
	vec4.distance = distance;
	vec4.squaredDistance = squaredDistance;
	vec4.length = length;
	vec4.squaredLength = squaredLength;
	vec4.negate = negate;
	vec4.inverse = inverse;
	vec4.normalize = normalize;
	vec4.dot = dot;
	vec4.cross = cross;
	vec4.lerp = lerp;
	vec4.random = random;
	vec4.transformMat4 = transformMat4;
	vec4.transformQuat = transformQuat;
	vec4.zero = zero;
	vec4.str = str;
	vec4.exactEquals = exactEquals;
	vec4.equals = equals;
	vec4.forEach = vec4.sqrLen = vec4.len = vec4.sqrDist = vec4.dist = vec4.div = vec4.mul = vec4.sub = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 4 Dimensional Vector
	 * @module vec4
	 */
	/**
	 * Creates a new, empty vec4
	 *
	 * @returns {vec4} a new 4D vector
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[0] = 0;
	        out[1] = 0;
	        out[2] = 0;
	        out[3] = 0;
	    }
	    return out;
	}
	/**
	 * Creates a new vec4 initialized with values from an existing vector
	 *
	 * @param {ReadonlyVec4} a vector to clone
	 * @returns {vec4} a new 4D vector
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Creates a new vec4 initialized with the given values
	 *
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @param {Number} w W component
	 * @returns {vec4} a new 4D vector
	 */
	function fromValues(x, y, z, w) {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    out[0] = x;
	    out[1] = y;
	    out[2] = z;
	    out[3] = w;
	    return out;
	}
	/**
	 * Copy the values from one vec4 to another
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the source vector
	 * @returns {vec4} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Set the components of a vec4 to the given values
	 *
	 * @param {vec4} out the receiving vector
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @param {Number} w W component
	 * @returns {vec4} out
	 */
	function set(out, x, y, z, w) {
	    out[0] = x;
	    out[1] = y;
	    out[2] = z;
	    out[3] = w;
	    return out;
	}
	/**
	 * Adds two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    return out;
	}
	/**
	 * Subtracts vector b from vector a
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    out[2] = a[2] - b[2];
	    out[3] = a[3] - b[3];
	    return out;
	}
	/**
	 * Multiplies two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function multiply(out, a, b) {
	    out[0] = a[0] * b[0];
	    out[1] = a[1] * b[1];
	    out[2] = a[2] * b[2];
	    out[3] = a[3] * b[3];
	    return out;
	}
	/**
	 * Divides two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function divide(out, a, b) {
	    out[0] = a[0] / b[0];
	    out[1] = a[1] / b[1];
	    out[2] = a[2] / b[2];
	    out[3] = a[3] / b[3];
	    return out;
	}
	/**
	 * Math.ceil the components of a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to ceil
	 * @returns {vec4} out
	 */
	function ceil(out, a) {
	    out[0] = Math.ceil(a[0]);
	    out[1] = Math.ceil(a[1]);
	    out[2] = Math.ceil(a[2]);
	    out[3] = Math.ceil(a[3]);
	    return out;
	}
	/**
	 * Math.floor the components of a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to floor
	 * @returns {vec4} out
	 */
	function floor(out, a) {
	    out[0] = Math.floor(a[0]);
	    out[1] = Math.floor(a[1]);
	    out[2] = Math.floor(a[2]);
	    out[3] = Math.floor(a[3]);
	    return out;
	}
	/**
	 * Returns the minimum of two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function min(out, a, b) {
	    out[0] = Math.min(a[0], b[0]);
	    out[1] = Math.min(a[1], b[1]);
	    out[2] = Math.min(a[2], b[2]);
	    out[3] = Math.min(a[3], b[3]);
	    return out;
	}
	/**
	 * Returns the maximum of two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {vec4} out
	 */
	function max(out, a, b) {
	    out[0] = Math.max(a[0], b[0]);
	    out[1] = Math.max(a[1], b[1]);
	    out[2] = Math.max(a[2], b[2]);
	    out[3] = Math.max(a[3], b[3]);
	    return out;
	}
	/**
	 * Math.round the components of a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to round
	 * @returns {vec4} out
	 */
	function round(out, a) {
	    out[0] = Math.round(a[0]);
	    out[1] = Math.round(a[1]);
	    out[2] = Math.round(a[2]);
	    out[3] = Math.round(a[3]);
	    return out;
	}
	/**
	 * Scales a vec4 by a scalar number
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the vector to scale
	 * @param {Number} b amount to scale the vector by
	 * @returns {vec4} out
	 */
	function scale(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    return out;
	}
	/**
	 * Adds two vec4's after scaling the second operand by a scalar value
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @param {Number} scale the amount to scale b by before adding
	 * @returns {vec4} out
	 */
	function scaleAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    out[2] = a[2] + b[2] * scale;
	    out[3] = a[3] + b[3] * scale;
	    return out;
	}
	/**
	 * Calculates the euclidian distance between two vec4's
	 *
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {Number} distance between a and b
	 */
	function distance(a, b) {
	    var x = b[0] - a[0];
	    var y = b[1] - a[1];
	    var z = b[2] - a[2];
	    var w = b[3] - a[3];
	    return Math.hypot(x, y, z, w);
	}
	/**
	 * Calculates the squared euclidian distance between two vec4's
	 *
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {Number} squared distance between a and b
	 */
	function squaredDistance(a, b) {
	    var x = b[0] - a[0];
	    var y = b[1] - a[1];
	    var z = b[2] - a[2];
	    var w = b[3] - a[3];
	    return x * x + y * y + z * z + w * w;
	}
	/**
	 * Calculates the length of a vec4
	 *
	 * @param {ReadonlyVec4} a vector to calculate length of
	 * @returns {Number} length of a
	 */
	function length(a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    var w = a[3];
	    return Math.hypot(x, y, z, w);
	}
	/**
	 * Calculates the squared length of a vec4
	 *
	 * @param {ReadonlyVec4} a vector to calculate squared length of
	 * @returns {Number} squared length of a
	 */
	function squaredLength(a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    var w = a[3];
	    return x * x + y * y + z * z + w * w;
	}
	/**
	 * Negates the components of a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to negate
	 * @returns {vec4} out
	 */
	function negate(out, a) {
	    out[0] = -a[0];
	    out[1] = -a[1];
	    out[2] = -a[2];
	    out[3] = -a[3];
	    return out;
	}
	/**
	 * Returns the inverse of the components of a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to invert
	 * @returns {vec4} out
	 */
	function inverse(out, a) {
	    out[0] = 1 / a[0];
	    out[1] = 1 / a[1];
	    out[2] = 1 / a[2];
	    out[3] = 1 / a[3];
	    return out;
	}
	/**
	 * Normalize a vec4
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a vector to normalize
	 * @returns {vec4} out
	 */
	function normalize(out, a) {
	    var x = a[0];
	    var y = a[1];
	    var z = a[2];
	    var w = a[3];
	    var len = x * x + y * y + z * z + w * w;
	    if (len > 0) {
	        len = 1 / Math.sqrt(len);
	    }
	    out[0] = x * len;
	    out[1] = y * len;
	    out[2] = z * len;
	    out[3] = w * len;
	    return out;
	}
	/**
	 * Calculates the dot product of two vec4's
	 *
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @returns {Number} dot product of a and b
	 */
	function dot(a, b) {
	    return a[0] * b[0] + a[1] * b[1] + a[2] * b[2] + a[3] * b[3];
	}
	/**
	 * Returns the cross-product of three vectors in a 4-dimensional space
	 *
	 * @param {ReadonlyVec4} result the receiving vector
	 * @param {ReadonlyVec4} U the first vector
	 * @param {ReadonlyVec4} V the second vector
	 * @param {ReadonlyVec4} W the third vector
	 * @returns {vec4} result
	 */
	function cross(out, u, v, w) {
	    var A = v[0] * w[1] - v[1] * w[0], B = v[0] * w[2] - v[2] * w[0], C = v[0] * w[3] - v[3] * w[0], D = v[1] * w[2] - v[2] * w[1], E = v[1] * w[3] - v[3] * w[1], F = v[2] * w[3] - v[3] * w[2];
	    var G = u[0];
	    var H = u[1];
	    var I = u[2];
	    var J = u[3];
	    out[0] = H * F - I * E + J * D;
	    out[1] = -(G * F) + I * C - J * B;
	    out[2] = G * E - H * C + J * A;
	    out[3] = -(G * D) + H * B - I * A;
	    return out;
	}
	/**
	 * Performs a linear interpolation between two vec4's
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the first operand
	 * @param {ReadonlyVec4} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {vec4} out
	 */
	function lerp(out, a, b, t) {
	    var ax = a[0];
	    var ay = a[1];
	    var az = a[2];
	    var aw = a[3];
	    out[0] = ax + t * (b[0] - ax);
	    out[1] = ay + t * (b[1] - ay);
	    out[2] = az + t * (b[2] - az);
	    out[3] = aw + t * (b[3] - aw);
	    return out;
	}
	/**
	 * Generates a random vector with the given scale
	 *
	 * @param {vec4} out the receiving vector
	 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
	 * @returns {vec4} out
	 */
	function random(out, scale) {
	    scale = scale || 1;
	    // Marsaglia, George. Choosing a Point from the Surface of a
	    // Sphere. Ann. Math. Statist. 43 (1972), no. 2, 645--646.
	    // http://projecteuclid.org/euclid.aoms/1177692644;
	    var v1, v2, v3, v4;
	    var s1, s2;
	    do {
	        v1 = glMatrix.RANDOM() * 2 - 1;
	        v2 = glMatrix.RANDOM() * 2 - 1;
	        s1 = v1 * v1 + v2 * v2;
	    } while (s1 >= 1);
	    do {
	        v3 = glMatrix.RANDOM() * 2 - 1;
	        v4 = glMatrix.RANDOM() * 2 - 1;
	        s2 = v3 * v3 + v4 * v4;
	    } while (s2 >= 1);
	    var d = Math.sqrt((1 - s1) / s2);
	    out[0] = scale * v1;
	    out[1] = scale * v2;
	    out[2] = scale * v3 * d;
	    out[3] = scale * v4 * d;
	    return out;
	}
	/**
	 * Transforms the vec4 with a mat4.
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the vector to transform
	 * @param {ReadonlyMat4} m matrix to transform with
	 * @returns {vec4} out
	 */
	function transformMat4(out, a, m) {
	    var x = a[0], y = a[1], z = a[2], w = a[3];
	    out[0] = m[0] * x + m[4] * y + m[8] * z + m[12] * w;
	    out[1] = m[1] * x + m[5] * y + m[9] * z + m[13] * w;
	    out[2] = m[2] * x + m[6] * y + m[10] * z + m[14] * w;
	    out[3] = m[3] * x + m[7] * y + m[11] * z + m[15] * w;
	    return out;
	}
	/**
	 * Transforms the vec4 with a quat
	 *
	 * @param {vec4} out the receiving vector
	 * @param {ReadonlyVec4} a the vector to transform
	 * @param {ReadonlyQuat} q quaternion to transform with
	 * @returns {vec4} out
	 */
	function transformQuat(out, a, q) {
	    var x = a[0], y = a[1], z = a[2];
	    var qx = q[0], qy = q[1], qz = q[2], qw = q[3];
	    // calculate quat * vec
	    var ix = qw * x + qy * z - qz * y;
	    var iy = qw * y + qz * x - qx * z;
	    var iz = qw * z + qx * y - qy * x;
	    var iw = -qx * x - qy * y - qz * z;
	    // calculate result * inverse quat
	    out[0] = ix * qw + iw * -qx + iy * -qz - iz * -qy;
	    out[1] = iy * qw + iw * -qy + iz * -qx - ix * -qz;
	    out[2] = iz * qw + iw * -qz + ix * -qy - iy * -qx;
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Set the components of a vec4 to zero
	 *
	 * @param {vec4} out the receiving vector
	 * @returns {vec4} out
	 */
	function zero(out) {
	    out[0] = 0;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 0;
	    return out;
	}
	/**
	 * Returns a string representation of a vector
	 *
	 * @param {ReadonlyVec4} a vector to represent as a string
	 * @returns {String} string representation of the vector
	 */
	function str(a) {
	    return 'vec4(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
	}
	/**
	 * Returns whether or not the vectors have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyVec4} a The first vector.
	 * @param {ReadonlyVec4} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3];
	}
	/**
	 * Returns whether or not the vectors have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyVec4} a The first vector.
	 * @param {ReadonlyVec4} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3));
	}
	/**
	 * Alias for {@link vec4.subtract}
	 * @function
	 */
	var sub = subtract;
	/**
	 * Alias for {@link vec4.multiply}
	 * @function
	 */
	vec4.sub = sub;
	var mul = multiply;
	/**
	 * Alias for {@link vec4.divide}
	 * @function
	 */
	vec4.mul = mul;
	var div = divide;
	/**
	 * Alias for {@link vec4.distance}
	 * @function
	 */
	vec4.div = div;
	var dist = distance;
	/**
	 * Alias for {@link vec4.squaredDistance}
	 * @function
	 */
	vec4.dist = dist;
	var sqrDist = squaredDistance;
	/**
	 * Alias for {@link vec4.length}
	 * @function
	 */
	vec4.sqrDist = sqrDist;
	var len = length;
	/**
	 * Alias for {@link vec4.squaredLength}
	 * @function
	 */
	vec4.len = len;
	var sqrLen = squaredLength;
	/**
	 * Perform some operation over an array of vec4s.
	 *
	 * @param {Array} a the array of vectors to iterate over
	 * @param {Number} stride Number of elements between the start of each vec4. If 0 assumes tightly packed
	 * @param {Number} offset Number of elements to skip at the beginning of the array
	 * @param {Number} count Number of vec4s to iterate over. If 0 iterates over entire array
	 * @param {Function} fn Function to call for each vector in the array
	 * @param {Object} [arg] additional argument to pass to fn
	 * @returns {Array} a
	 * @function
	 */
	vec4.sqrLen = sqrLen;
	var forEach = (function () {
	    var vec = create();
	    return function (a, stride, offset, count, fn, arg) {
	        var i, l;
	        if (!stride) {
	            stride = 4;
	        }
	        if (!offset) {
	            offset = 0;
	        }
	        if (count) {
	            l = Math.min(count * stride + offset, a.length);
	        } else {
	            l = a.length;
	        }
	        for (i = offset; i < l; i += stride) {
	            vec[0] = a[i];
	            vec[1] = a[i + 1];
	            vec[2] = a[i + 2];
	            vec[3] = a[i + 3];
	            fn(vec, vec, arg);
	            a[i] = vec[0];
	            a[i + 1] = vec[1];
	            a[i + 2] = vec[2];
	            a[i + 3] = vec[3];
	        }
	        return a;
	    };
	}());
	vec4.forEach = forEach;
	return vec4;
}

var hasRequiredQuat;

function requireQuat () {
	if (hasRequiredQuat) return quat;
	hasRequiredQuat = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(quat, '__esModule', { value: true });
	quat.create = create;
	quat.identity = identity;
	quat.setAxisAngle = setAxisAngle;
	quat.getAxisAngle = getAxisAngle;
	quat.getAngle = getAngle;
	quat.multiply = multiply;
	quat.rotateX = rotateX;
	quat.rotateY = rotateY;
	quat.rotateZ = rotateZ;
	quat.calculateW = calculateW;
	quat.exp = exp;
	quat.ln = ln;
	quat.pow = pow;
	quat.slerp = slerp;
	quat.random = random;
	quat.invert = invert;
	quat.conjugate = conjugate;
	quat.fromMat3 = fromMat3;
	quat.fromEuler = fromEuler;
	quat.str = str;
	quat.setAxes = quat.sqlerp = quat.rotationTo = quat.equals = quat.exactEquals = quat.normalize = quat.sqrLen = quat.squaredLength = quat.len = quat.length = quat.lerp = quat.dot = quat.scale = quat.mul = quat.add = quat.set = quat.copy = quat.fromValues = quat.clone = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	var mat3 = _interopRequireWildcard(requireMat3());
	var vec3 = _interopRequireWildcard(requireVec3());
	var vec4 = _interopRequireWildcard(requireVec4());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * Quaternion
	 * @module quat
	 */
	/**
	 * Creates a new identity quat
	 *
	 * @returns {quat} a new quaternion
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(4);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[0] = 0;
	        out[1] = 0;
	        out[2] = 0;
	    }
	    out[3] = 1;
	    return out;
	}
	/**
	 * Set a quat to the identity quaternion
	 *
	 * @param {quat} out the receiving quaternion
	 * @returns {quat} out
	 */
	function identity(out) {
	    out[0] = 0;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    return out;
	}
	/**
	 * Sets a quat from the given angle and rotation axis,
	 * then returns it.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyVec3} axis the axis around which to rotate
	 * @param {Number} rad the angle in radians
	 * @returns {quat} out
	 **/
	function setAxisAngle(out, axis, rad) {
	    rad = rad * 0.5;
	    var s = Math.sin(rad);
	    out[0] = s * axis[0];
	    out[1] = s * axis[1];
	    out[2] = s * axis[2];
	    out[3] = Math.cos(rad);
	    return out;
	}
	/**
	 * Gets the rotation axis and angle for a given
	 *  quaternion. If a quaternion is created with
	 *  setAxisAngle, this method will return the same
	 *  values as providied in the original parameter list
	 *  OR functionally equivalent values.
	 * Example: The quaternion formed by axis [0, 0, 1] and
	 *  angle -90 is the same as the quaternion formed by
	 *  [0, 0, 1] and 270. This method favors the latter.
	 * @param  {vec3} out_axis  Vector receiving the axis of rotation
	 * @param  {ReadonlyQuat} q     Quaternion to be decomposed
	 * @return {Number}     Angle, in radians, of the rotation
	 */
	function getAxisAngle(out_axis, q) {
	    var rad = Math.acos(q[3]) * 2;
	    var s = Math.sin(rad / 2);
	    if (s > glMatrix.EPSILON) {
	        out_axis[0] = q[0] / s;
	        out_axis[1] = q[1] / s;
	        out_axis[2] = q[2] / s;
	    } else {
	        // If s is zero, return any axis (no rotation - axis does not matter)
	        out_axis[0] = 1;
	        out_axis[1] = 0;
	        out_axis[2] = 0;
	    }
	    return rad;
	}
	/**
	 * Gets the angular distance between two unit quaternions
	 *
	 * @param  {ReadonlyQuat} a     Origin unit quaternion
	 * @param  {ReadonlyQuat} b     Destination unit quaternion
	 * @return {Number}     Angle, in radians, between the two quaternions
	 */
	function getAngle(a, b) {
	    var dotproduct = dot(a, b);
	    return Math.acos(2 * dotproduct * dotproduct - 1);
	}
	/**
	 * Multiplies two quat's
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @returns {quat} out
	 */
	function multiply(out, a, b) {
	    var ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    var bx = b[0], by = b[1], bz = b[2], bw = b[3];
	    out[0] = ax * bw + aw * bx + ay * bz - az * by;
	    out[1] = ay * bw + aw * by + az * bx - ax * bz;
	    out[2] = az * bw + aw * bz + ax * by - ay * bx;
	    out[3] = aw * bw - ax * bx - ay * by - az * bz;
	    return out;
	}
	/**
	 * Rotates a quaternion by the given angle about the X axis
	 *
	 * @param {quat} out quat receiving operation result
	 * @param {ReadonlyQuat} a quat to rotate
	 * @param {number} rad angle (in radians) to rotate
	 * @returns {quat} out
	 */
	function rotateX(out, a, rad) {
	    rad *= 0.5;
	    var ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    var bx = Math.sin(rad), bw = Math.cos(rad);
	    out[0] = ax * bw + aw * bx;
	    out[1] = ay * bw + az * bx;
	    out[2] = az * bw - ay * bx;
	    out[3] = aw * bw - ax * bx;
	    return out;
	}
	/**
	 * Rotates a quaternion by the given angle about the Y axis
	 *
	 * @param {quat} out quat receiving operation result
	 * @param {ReadonlyQuat} a quat to rotate
	 * @param {number} rad angle (in radians) to rotate
	 * @returns {quat} out
	 */
	function rotateY(out, a, rad) {
	    rad *= 0.5;
	    var ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    var by = Math.sin(rad), bw = Math.cos(rad);
	    out[0] = ax * bw - az * by;
	    out[1] = ay * bw + aw * by;
	    out[2] = az * bw + ax * by;
	    out[3] = aw * bw - ay * by;
	    return out;
	}
	/**
	 * Rotates a quaternion by the given angle about the Z axis
	 *
	 * @param {quat} out quat receiving operation result
	 * @param {ReadonlyQuat} a quat to rotate
	 * @param {number} rad angle (in radians) to rotate
	 * @returns {quat} out
	 */
	function rotateZ(out, a, rad) {
	    rad *= 0.5;
	    var ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    var bz = Math.sin(rad), bw = Math.cos(rad);
	    out[0] = ax * bw + ay * bz;
	    out[1] = ay * bw - ax * bz;
	    out[2] = az * bw + aw * bz;
	    out[3] = aw * bw - az * bz;
	    return out;
	}
	/**
	 * Calculates the W component of a quat from the X, Y, and Z components.
	 * Assumes that quaternion is 1 unit in length.
	 * Any existing W component will be ignored.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate W component of
	 * @returns {quat} out
	 */
	function calculateW(out, a) {
	    var x = a[0], y = a[1], z = a[2];
	    out[0] = x;
	    out[1] = y;
	    out[2] = z;
	    out[3] = Math.sqrt(Math.abs(1 - x * x - y * y - z * z));
	    return out;
	}
	/**
	 * Calculate the exponential of a unit quaternion.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate the exponential of
	 * @returns {quat} out
	 */
	function exp(out, a) {
	    var x = a[0], y = a[1], z = a[2], w = a[3];
	    var r = Math.sqrt(x * x + y * y + z * z);
	    var et = Math.exp(w);
	    var s = r > 0 ? et * Math.sin(r) / r : 0;
	    out[0] = x * s;
	    out[1] = y * s;
	    out[2] = z * s;
	    out[3] = et * Math.cos(r);
	    return out;
	}
	/**
	 * Calculate the natural logarithm of a unit quaternion.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate the exponential of
	 * @returns {quat} out
	 */
	function ln(out, a) {
	    var x = a[0], y = a[1], z = a[2], w = a[3];
	    var r = Math.sqrt(x * x + y * y + z * z);
	    var t = r > 0 ? Math.atan2(r, w) / r : 0;
	    out[0] = x * t;
	    out[1] = y * t;
	    out[2] = z * t;
	    out[3] = 0.5 * Math.log(x * x + y * y + z * z + w * w);
	    return out;
	}
	/**
	 * Calculate the scalar power of a unit quaternion.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate the exponential of
	 * @param {Number} b amount to scale the quaternion by
	 * @returns {quat} out
	 */
	function pow(out, a, b) {
	    ln(out, a);
	    scale(out, out, b);
	    exp(out, out);
	    return out;
	}
	/**
	 * Performs a spherical linear interpolation between two quat
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {quat} out
	 */
	function slerp(out, a, b, t) {
	    // benchmarks:
	    //    http://jsperf.com/quaternion-slerp-implementations
	    var ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    var bx = b[0], by = b[1], bz = b[2], bw = b[3];
	    var omega, cosom, sinom, scale0, scale1;
	    // calc cosine
	    cosom = ax * bx + ay * by + az * bz + aw * bw;
	    // adjust signs (if necessary)
	    if (cosom < 0) {
	        cosom = -cosom;
	        bx = -bx;
	        by = -by;
	        bz = -bz;
	        bw = -bw;
	    }
	    // calculate coefficients
	    if (1 - cosom > glMatrix.EPSILON) {
	        // standard case (slerp)
	        omega = Math.acos(cosom);
	        sinom = Math.sin(omega);
	        scale0 = Math.sin((1 - t) * omega) / sinom;
	        scale1 = Math.sin(t * omega) / sinom;
	    } else {
	        // "from" and "to" quaternions are very close
	        //  ... so we can do a linear interpolation
	        scale0 = 1 - t;
	        scale1 = t;
	    }
	    // calculate final values
	    out[0] = scale0 * ax + scale1 * bx;
	    out[1] = scale0 * ay + scale1 * by;
	    out[2] = scale0 * az + scale1 * bz;
	    out[3] = scale0 * aw + scale1 * bw;
	    return out;
	}
	/**
	 * Generates a random unit quaternion
	 *
	 * @param {quat} out the receiving quaternion
	 * @returns {quat} out
	 */
	function random(out) {
	    // Implementation of http://planning.cs.uiuc.edu/node198.html
	    // TODO: Calling random 3 times is probably not the fastest solution
	    var u1 = glMatrix.RANDOM();
	    var u2 = glMatrix.RANDOM();
	    var u3 = glMatrix.RANDOM();
	    var sqrt1MinusU1 = Math.sqrt(1 - u1);
	    var sqrtU1 = Math.sqrt(u1);
	    out[0] = sqrt1MinusU1 * Math.sin(2 * Math.PI * u2);
	    out[1] = sqrt1MinusU1 * Math.cos(2 * Math.PI * u2);
	    out[2] = sqrtU1 * Math.sin(2 * Math.PI * u3);
	    out[3] = sqrtU1 * Math.cos(2 * Math.PI * u3);
	    return out;
	}
	/**
	 * Calculates the inverse of a quat
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate inverse of
	 * @returns {quat} out
	 */
	function invert(out, a) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3];
	    var dot = a0 * a0 + a1 * a1 + a2 * a2 + a3 * a3;
	    var invDot = dot ? 1 / dot : 0;
	    // TODO: Would be faster to return [0,0,0,0] immediately if dot == 0
	    out[0] = -a0 * invDot;
	    out[1] = -a1 * invDot;
	    out[2] = -a2 * invDot;
	    out[3] = a3 * invDot;
	    return out;
	}
	/**
	 * Calculates the conjugate of a quat
	 * If the quaternion is normalized, this function is faster than quat.inverse and produces the same result.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quat to calculate conjugate of
	 * @returns {quat} out
	 */
	function conjugate(out, a) {
	    out[0] = -a[0];
	    out[1] = -a[1];
	    out[2] = -a[2];
	    out[3] = a[3];
	    return out;
	}
	/**
	 * Creates a quaternion from the given 3x3 rotation matrix.
	 *
	 * NOTE: The resultant quaternion is not normalized, so you should be sure
	 * to renormalize the quaternion yourself where necessary.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyMat3} m rotation matrix
	 * @returns {quat} out
	 * @function
	 */
	function fromMat3(out, m) {
	    // Algorithm in Ken Shoemake's article in 1987 SIGGRAPH course notes
	    // article "Quaternion Calculus and Fast Animation".
	    var fTrace = m[0] + m[4] + m[8];
	    var fRoot;
	    if (fTrace > 0) {
	        // |w| > 1/2, may as well choose w > 1/2
	        fRoot = Math.sqrt(fTrace + 1);
	        // 2w
	        out[3] = 0.5 * fRoot;
	        fRoot = 0.5 / fRoot;
	        // 1/(4w)
	        out[0] = (m[5] - m[7]) * fRoot;
	        out[1] = (m[6] - m[2]) * fRoot;
	        out[2] = (m[1] - m[3]) * fRoot;
	    } else {
	        // |w| <= 1/2
	        var i = 0;
	        if (m[4] > m[0])
	            i = 1;
	        if (m[8] > m[i * 3 + i])
	            i = 2;
	        var j = (i + 1) % 3;
	        var k = (i + 2) % 3;
	        fRoot = Math.sqrt(m[i * 3 + i] - m[j * 3 + j] - m[k * 3 + k] + 1);
	        out[i] = 0.5 * fRoot;
	        fRoot = 0.5 / fRoot;
	        out[3] = (m[j * 3 + k] - m[k * 3 + j]) * fRoot;
	        out[j] = (m[j * 3 + i] + m[i * 3 + j]) * fRoot;
	        out[k] = (m[k * 3 + i] + m[i * 3 + k]) * fRoot;
	    }
	    return out;
	}
	/**
	 * Creates a quaternion from the given euler angle x, y, z.
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {x} Angle to rotate around X axis in degrees.
	 * @param {y} Angle to rotate around Y axis in degrees.
	 * @param {z} Angle to rotate around Z axis in degrees.
	 * @returns {quat} out
	 * @function
	 */
	function fromEuler(out, x, y, z) {
	    var halfToRad = 0.5 * Math.PI / 180;
	    x *= halfToRad;
	    y *= halfToRad;
	    z *= halfToRad;
	    var sx = Math.sin(x);
	    var cx = Math.cos(x);
	    var sy = Math.sin(y);
	    var cy = Math.cos(y);
	    var sz = Math.sin(z);
	    var cz = Math.cos(z);
	    out[0] = sx * cy * cz - cx * sy * sz;
	    out[1] = cx * sy * cz + sx * cy * sz;
	    out[2] = cx * cy * sz - sx * sy * cz;
	    out[3] = cx * cy * cz + sx * sy * sz;
	    return out;
	}
	/**
	 * Returns a string representation of a quatenion
	 *
	 * @param {ReadonlyQuat} a vector to represent as a string
	 * @returns {String} string representation of the vector
	 */
	function str(a) {
	    return 'quat(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ')';
	}
	/**
	 * Creates a new quat initialized with values from an existing quaternion
	 *
	 * @param {ReadonlyQuat} a quaternion to clone
	 * @returns {quat} a new quaternion
	 * @function
	 */
	var clone = vec4.clone;
	/**
	 * Creates a new quat initialized with the given values
	 *
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @param {Number} w W component
	 * @returns {quat} a new quaternion
	 * @function
	 */
	quat.clone = clone;
	var fromValues = vec4.fromValues;
	/**
	 * Copy the values from one quat to another
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the source quaternion
	 * @returns {quat} out
	 * @function
	 */
	quat.fromValues = fromValues;
	var copy = vec4.copy;
	/**
	 * Set the components of a quat to the given values
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @param {Number} z Z component
	 * @param {Number} w W component
	 * @returns {quat} out
	 * @function
	 */
	quat.copy = copy;
	var set = vec4.set;
	/**
	 * Adds two quat's
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @returns {quat} out
	 * @function
	 */
	quat.set = set;
	var add = vec4.add;
	/**
	 * Alias for {@link quat.multiply}
	 * @function
	 */
	quat.add = add;
	var mul = multiply;
	/**
	 * Scales a quat by a scalar number
	 *
	 * @param {quat} out the receiving vector
	 * @param {ReadonlyQuat} a the vector to scale
	 * @param {Number} b amount to scale the vector by
	 * @returns {quat} out
	 * @function
	 */
	quat.mul = mul;
	var scale = vec4.scale;
	/**
	 * Calculates the dot product of two quat's
	 *
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @returns {Number} dot product of a and b
	 * @function
	 */
	quat.scale = scale;
	var dot = vec4.dot;
	/**
	 * Performs a linear interpolation between two quat's
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {quat} out
	 * @function
	 */
	quat.dot = dot;
	var lerp = vec4.lerp;
	/**
	 * Calculates the length of a quat
	 *
	 * @param {ReadonlyQuat} a vector to calculate length of
	 * @returns {Number} length of a
	 */
	quat.lerp = lerp;
	var length = vec4.length;
	/**
	 * Alias for {@link quat.length}
	 * @function
	 */
	quat.length = length;
	var len = length;
	/**
	 * Calculates the squared length of a quat
	 *
	 * @param {ReadonlyQuat} a vector to calculate squared length of
	 * @returns {Number} squared length of a
	 * @function
	 */
	quat.len = len;
	var squaredLength = vec4.squaredLength;
	/**
	 * Alias for {@link quat.squaredLength}
	 * @function
	 */
	quat.squaredLength = squaredLength;
	var sqrLen = squaredLength;
	/**
	 * Normalize a quat
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a quaternion to normalize
	 * @returns {quat} out
	 * @function
	 */
	quat.sqrLen = sqrLen;
	var normalize = vec4.normalize;
	/**
	 * Returns whether or not the quaternions have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyQuat} a The first quaternion.
	 * @param {ReadonlyQuat} b The second quaternion.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	quat.normalize = normalize;
	var exactEquals = vec4.exactEquals;
	/**
	 * Returns whether or not the quaternions have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyQuat} a The first vector.
	 * @param {ReadonlyQuat} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	quat.exactEquals = exactEquals;
	var equals = vec4.equals;
	/**
	 * Sets a quaternion to represent the shortest rotation from one
	 * vector to another.
	 *
	 * Both vectors are assumed to be unit length.
	 *
	 * @param {quat} out the receiving quaternion.
	 * @param {ReadonlyVec3} a the initial vector
	 * @param {ReadonlyVec3} b the destination vector
	 * @returns {quat} out
	 */
	quat.equals = equals;
	var rotationTo = (function () {
	    var tmpvec3 = vec3.create();
	    var xUnitVec3 = vec3.fromValues(1, 0, 0);
	    var yUnitVec3 = vec3.fromValues(0, 1, 0);
	    return function (out, a, b) {
	        var dot = vec3.dot(a, b);
	        if (dot < -0.999999) {
	            vec3.cross(tmpvec3, xUnitVec3, a);
	            if (vec3.len(tmpvec3) < 0.000001)
	                vec3.cross(tmpvec3, yUnitVec3, a);
	            vec3.normalize(tmpvec3, tmpvec3);
	            setAxisAngle(out, tmpvec3, Math.PI);
	            return out;
	        } else if (dot > 0.999999) {
	            out[0] = 0;
	            out[1] = 0;
	            out[2] = 0;
	            out[3] = 1;
	            return out;
	        } else {
	            vec3.cross(tmpvec3, a, b);
	            out[0] = tmpvec3[0];
	            out[1] = tmpvec3[1];
	            out[2] = tmpvec3[2];
	            out[3] = 1 + dot;
	            return normalize(out, out);
	        }
	    };
	}());
	/**
	 * Performs a spherical linear interpolation with two control points
	 *
	 * @param {quat} out the receiving quaternion
	 * @param {ReadonlyQuat} a the first operand
	 * @param {ReadonlyQuat} b the second operand
	 * @param {ReadonlyQuat} c the third operand
	 * @param {ReadonlyQuat} d the fourth operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {quat} out
	 */
	quat.rotationTo = rotationTo;
	var sqlerp = (function () {
	    var temp1 = create();
	    var temp2 = create();
	    return function (out, a, b, c, d, t) {
	        slerp(temp1, a, d, t);
	        slerp(temp2, b, c, t);
	        slerp(out, temp1, temp2, 2 * t * (1 - t));
	        return out;
	    };
	}());
	/**
	 * Sets the specified quaternion with values corresponding to the given
	 * axes. Each axis is a vec3 and is expected to be unit length and
	 * perpendicular to all other specified axes.
	 *
	 * @param {ReadonlyVec3} view  the vector representing the viewing direction
	 * @param {ReadonlyVec3} right the vector representing the local "right" direction
	 * @param {ReadonlyVec3} up    the vector representing the local "up" direction
	 * @returns {quat} out
	 */
	quat.sqlerp = sqlerp;
	var setAxes = (function () {
	    var matr = mat3.create();
	    return function (out, view, right, up) {
	        matr[0] = right[0];
	        matr[3] = right[1];
	        matr[6] = right[2];
	        matr[1] = up[0];
	        matr[4] = up[1];
	        matr[7] = up[2];
	        matr[2] = -view[0];
	        matr[5] = -view[1];
	        matr[8] = -view[2];
	        return normalize(out, fromMat3(out, matr));
	    };
	}());
	quat.setAxes = setAxes;
	return quat;
}

var quat2 = {};

var hasRequiredQuat2;

function requireQuat2 () {
	if (hasRequiredQuat2) return quat2;
	hasRequiredQuat2 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(quat2, '__esModule', { value: true });
	quat2.create = create;
	quat2.clone = clone;
	quat2.fromValues = fromValues;
	quat2.fromRotationTranslationValues = fromRotationTranslationValues;
	quat2.fromRotationTranslation = fromRotationTranslation;
	quat2.fromTranslation = fromTranslation;
	quat2.fromRotation = fromRotation;
	quat2.fromMat4 = fromMat4;
	quat2.copy = copy;
	quat2.identity = identity;
	quat2.set = set;
	quat2.getDual = getDual;
	quat2.setDual = setDual;
	quat2.getTranslation = getTranslation;
	quat2.translate = translate;
	quat2.rotateX = rotateX;
	quat2.rotateY = rotateY;
	quat2.rotateZ = rotateZ;
	quat2.rotateByQuatAppend = rotateByQuatAppend;
	quat2.rotateByQuatPrepend = rotateByQuatPrepend;
	quat2.rotateAroundAxis = rotateAroundAxis;
	quat2.add = add;
	quat2.multiply = multiply;
	quat2.scale = scale;
	quat2.lerp = lerp;
	quat2.invert = invert;
	quat2.conjugate = conjugate;
	quat2.normalize = normalize;
	quat2.str = str;
	quat2.exactEquals = exactEquals;
	quat2.equals = equals;
	quat2.sqrLen = quat2.squaredLength = quat2.len = quat2.length = quat2.dot = quat2.mul = quat2.setReal = quat2.getReal = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	var quat = _interopRequireWildcard(requireQuat());
	var mat4 = _interopRequireWildcard(requireMat4());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * Dual Quaternion<br>
	 * Format: [real, dual]<br>
	 * Quaternion format: XYZW<br>
	 * Make sure to have normalized dual quaternions, otherwise the functions may not work as intended.<br>
	 * @module quat2
	 */
	/**
	 * Creates a new identity dual quat
	 *
	 * @returns {quat2} a new dual quaternion [real -> rotation, dual -> translation]
	 */
	function create() {
	    var dq = new glMatrix.ARRAY_TYPE(8);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        dq[0] = 0;
	        dq[1] = 0;
	        dq[2] = 0;
	        dq[4] = 0;
	        dq[5] = 0;
	        dq[6] = 0;
	        dq[7] = 0;
	    }
	    dq[3] = 1;
	    return dq;
	}
	/**
	 * Creates a new quat initialized with values from an existing quaternion
	 *
	 * @param {ReadonlyQuat2} a dual quaternion to clone
	 * @returns {quat2} new dual quaternion
	 * @function
	 */
	function clone(a) {
	    var dq = new glMatrix.ARRAY_TYPE(8);
	    dq[0] = a[0];
	    dq[1] = a[1];
	    dq[2] = a[2];
	    dq[3] = a[3];
	    dq[4] = a[4];
	    dq[5] = a[5];
	    dq[6] = a[6];
	    dq[7] = a[7];
	    return dq;
	}
	/**
	 * Creates a new dual quat initialized with the given values
	 *
	 * @param {Number} x1 X component
	 * @param {Number} y1 Y component
	 * @param {Number} z1 Z component
	 * @param {Number} w1 W component
	 * @param {Number} x2 X component
	 * @param {Number} y2 Y component
	 * @param {Number} z2 Z component
	 * @param {Number} w2 W component
	 * @returns {quat2} new dual quaternion
	 * @function
	 */
	function fromValues(x1, y1, z1, w1, x2, y2, z2, w2) {
	    var dq = new glMatrix.ARRAY_TYPE(8);
	    dq[0] = x1;
	    dq[1] = y1;
	    dq[2] = z1;
	    dq[3] = w1;
	    dq[4] = x2;
	    dq[5] = y2;
	    dq[6] = z2;
	    dq[7] = w2;
	    return dq;
	}
	/**
	 * Creates a new dual quat from the given values (quat and translation)
	 *
	 * @param {Number} x1 X component
	 * @param {Number} y1 Y component
	 * @param {Number} z1 Z component
	 * @param {Number} w1 W component
	 * @param {Number} x2 X component (translation)
	 * @param {Number} y2 Y component (translation)
	 * @param {Number} z2 Z component (translation)
	 * @returns {quat2} new dual quaternion
	 * @function
	 */
	function fromRotationTranslationValues(x1, y1, z1, w1, x2, y2, z2) {
	    var dq = new glMatrix.ARRAY_TYPE(8);
	    dq[0] = x1;
	    dq[1] = y1;
	    dq[2] = z1;
	    dq[3] = w1;
	    var ax = x2 * 0.5, ay = y2 * 0.5, az = z2 * 0.5;
	    dq[4] = ax * w1 + ay * z1 - az * y1;
	    dq[5] = ay * w1 + az * x1 - ax * z1;
	    dq[6] = az * w1 + ax * y1 - ay * x1;
	    dq[7] = -ax * x1 - ay * y1 - az * z1;
	    return dq;
	}
	/**
	 * Creates a dual quat from a quaternion and a translation
	 *
	 * @param {ReadonlyQuat2} dual quaternion receiving operation result
	 * @param {ReadonlyQuat} q a normalized quaternion
	 * @param {ReadonlyVec3} t tranlation vector
	 * @returns {quat2} dual quaternion receiving operation result
	 * @function
	 */
	function fromRotationTranslation(out, q, t) {
	    var ax = t[0] * 0.5, ay = t[1] * 0.5, az = t[2] * 0.5, bx = q[0], by = q[1], bz = q[2], bw = q[3];
	    out[0] = bx;
	    out[1] = by;
	    out[2] = bz;
	    out[3] = bw;
	    out[4] = ax * bw + ay * bz - az * by;
	    out[5] = ay * bw + az * bx - ax * bz;
	    out[6] = az * bw + ax * by - ay * bx;
	    out[7] = -ax * bx - ay * by - az * bz;
	    return out;
	}
	/**
	 * Creates a dual quat from a translation
	 *
	 * @param {ReadonlyQuat2} dual quaternion receiving operation result
	 * @param {ReadonlyVec3} t translation vector
	 * @returns {quat2} dual quaternion receiving operation result
	 * @function
	 */
	function fromTranslation(out, t) {
	    out[0] = 0;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    out[4] = t[0] * 0.5;
	    out[5] = t[1] * 0.5;
	    out[6] = t[2] * 0.5;
	    out[7] = 0;
	    return out;
	}
	/**
	 * Creates a dual quat from a quaternion
	 *
	 * @param {ReadonlyQuat2} dual quaternion receiving operation result
	 * @param {ReadonlyQuat} q the quaternion
	 * @returns {quat2} dual quaternion receiving operation result
	 * @function
	 */
	function fromRotation(out, q) {
	    out[0] = q[0];
	    out[1] = q[1];
	    out[2] = q[2];
	    out[3] = q[3];
	    out[4] = 0;
	    out[5] = 0;
	    out[6] = 0;
	    out[7] = 0;
	    return out;
	}
	/**
	 * Creates a new dual quat from a matrix (4x4)
	 *
	 * @param {quat2} out the dual quaternion
	 * @param {ReadonlyMat4} a the matrix
	 * @returns {quat2} dual quat receiving operation result
	 * @function
	 */
	function fromMat4(out, a) {
	    //TODO Optimize this
	    var outer = quat.create();
	    mat4.getRotation(outer, a);
	    var t = new glMatrix.ARRAY_TYPE(3);
	    mat4.getTranslation(t, a);
	    fromRotationTranslation(out, outer, t);
	    return out;
	}
	/**
	 * Copy the values from one dual quat to another
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the source dual quaternion
	 * @returns {quat2} out
	 * @function
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    out[2] = a[2];
	    out[3] = a[3];
	    out[4] = a[4];
	    out[5] = a[5];
	    out[6] = a[6];
	    out[7] = a[7];
	    return out;
	}
	/**
	 * Set a dual quat to the identity dual quaternion
	 *
	 * @param {quat2} out the receiving quaternion
	 * @returns {quat2} out
	 */
	function identity(out) {
	    out[0] = 0;
	    out[1] = 0;
	    out[2] = 0;
	    out[3] = 1;
	    out[4] = 0;
	    out[5] = 0;
	    out[6] = 0;
	    out[7] = 0;
	    return out;
	}
	/**
	 * Set the components of a dual quat to the given values
	 *
	 * @param {quat2} out the receiving quaternion
	 * @param {Number} x1 X component
	 * @param {Number} y1 Y component
	 * @param {Number} z1 Z component
	 * @param {Number} w1 W component
	 * @param {Number} x2 X component
	 * @param {Number} y2 Y component
	 * @param {Number} z2 Z component
	 * @param {Number} w2 W component
	 * @returns {quat2} out
	 * @function
	 */
	function set(out, x1, y1, z1, w1, x2, y2, z2, w2) {
	    out[0] = x1;
	    out[1] = y1;
	    out[2] = z1;
	    out[3] = w1;
	    out[4] = x2;
	    out[5] = y2;
	    out[6] = z2;
	    out[7] = w2;
	    return out;
	}
	/**
	 * Gets the real part of a dual quat
	 * @param  {quat} out real part
	 * @param  {ReadonlyQuat2} a Dual Quaternion
	 * @return {quat} real part
	 */
	var getReal = quat.copy;
	/**
	 * Gets the dual part of a dual quat
	 * @param  {quat} out dual part
	 * @param  {ReadonlyQuat2} a Dual Quaternion
	 * @return {quat} dual part
	 */
	quat2.getReal = getReal;
	function getDual(out, a) {
	    out[0] = a[4];
	    out[1] = a[5];
	    out[2] = a[6];
	    out[3] = a[7];
	    return out;
	}
	/**
	 * Set the real component of a dual quat to the given quaternion
	 *
	 * @param {quat2} out the receiving quaternion
	 * @param {ReadonlyQuat} q a quaternion representing the real part
	 * @returns {quat2} out
	 * @function
	 */
	var setReal = quat.copy;
	/**
	 * Set the dual component of a dual quat to the given quaternion
	 *
	 * @param {quat2} out the receiving quaternion
	 * @param {ReadonlyQuat} q a quaternion representing the dual part
	 * @returns {quat2} out
	 * @function
	 */
	quat2.setReal = setReal;
	function setDual(out, q) {
	    out[4] = q[0];
	    out[5] = q[1];
	    out[6] = q[2];
	    out[7] = q[3];
	    return out;
	}
	/**
	 * Gets the translation of a normalized dual quat
	 * @param  {vec3} out translation
	 * @param  {ReadonlyQuat2} a Dual Quaternion to be decomposed
	 * @return {vec3} translation
	 */
	function getTranslation(out, a) {
	    var ax = a[4], ay = a[5], az = a[6], aw = a[7], bx = -a[0], by = -a[1], bz = -a[2], bw = a[3];
	    out[0] = (ax * bw + aw * bx + ay * bz - az * by) * 2;
	    out[1] = (ay * bw + aw * by + az * bx - ax * bz) * 2;
	    out[2] = (az * bw + aw * bz + ax * by - ay * bx) * 2;
	    return out;
	}
	/**
	 * Translates a dual quat by the given vector
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to translate
	 * @param {ReadonlyVec3} v vector to translate by
	 * @returns {quat2} out
	 */
	function translate(out, a, v) {
	    var ax1 = a[0], ay1 = a[1], az1 = a[2], aw1 = a[3], bx1 = v[0] * 0.5, by1 = v[1] * 0.5, bz1 = v[2] * 0.5, ax2 = a[4], ay2 = a[5], az2 = a[6], aw2 = a[7];
	    out[0] = ax1;
	    out[1] = ay1;
	    out[2] = az1;
	    out[3] = aw1;
	    out[4] = aw1 * bx1 + ay1 * bz1 - az1 * by1 + ax2;
	    out[5] = aw1 * by1 + az1 * bx1 - ax1 * bz1 + ay2;
	    out[6] = aw1 * bz1 + ax1 * by1 - ay1 * bx1 + az2;
	    out[7] = -ax1 * bx1 - ay1 * by1 - az1 * bz1 + aw2;
	    return out;
	}
	/**
	 * Rotates a dual quat around the X axis
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @param {number} rad how far should the rotation be
	 * @returns {quat2} out
	 */
	function rotateX(out, a, rad) {
	    var bx = -a[0], by = -a[1], bz = -a[2], bw = a[3], ax = a[4], ay = a[5], az = a[6], aw = a[7], ax1 = ax * bw + aw * bx + ay * bz - az * by, ay1 = ay * bw + aw * by + az * bx - ax * bz, az1 = az * bw + aw * bz + ax * by - ay * bx, aw1 = aw * bw - ax * bx - ay * by - az * bz;
	    quat.rotateX(out, a, rad);
	    bx = out[0];
	    by = out[1];
	    bz = out[2];
	    bw = out[3];
	    out[4] = ax1 * bw + aw1 * bx + ay1 * bz - az1 * by;
	    out[5] = ay1 * bw + aw1 * by + az1 * bx - ax1 * bz;
	    out[6] = az1 * bw + aw1 * bz + ax1 * by - ay1 * bx;
	    out[7] = aw1 * bw - ax1 * bx - ay1 * by - az1 * bz;
	    return out;
	}
	/**
	 * Rotates a dual quat around the Y axis
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @param {number} rad how far should the rotation be
	 * @returns {quat2} out
	 */
	function rotateY(out, a, rad) {
	    var bx = -a[0], by = -a[1], bz = -a[2], bw = a[3], ax = a[4], ay = a[5], az = a[6], aw = a[7], ax1 = ax * bw + aw * bx + ay * bz - az * by, ay1 = ay * bw + aw * by + az * bx - ax * bz, az1 = az * bw + aw * bz + ax * by - ay * bx, aw1 = aw * bw - ax * bx - ay * by - az * bz;
	    quat.rotateY(out, a, rad);
	    bx = out[0];
	    by = out[1];
	    bz = out[2];
	    bw = out[3];
	    out[4] = ax1 * bw + aw1 * bx + ay1 * bz - az1 * by;
	    out[5] = ay1 * bw + aw1 * by + az1 * bx - ax1 * bz;
	    out[6] = az1 * bw + aw1 * bz + ax1 * by - ay1 * bx;
	    out[7] = aw1 * bw - ax1 * bx - ay1 * by - az1 * bz;
	    return out;
	}
	/**
	 * Rotates a dual quat around the Z axis
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @param {number} rad how far should the rotation be
	 * @returns {quat2} out
	 */
	function rotateZ(out, a, rad) {
	    var bx = -a[0], by = -a[1], bz = -a[2], bw = a[3], ax = a[4], ay = a[5], az = a[6], aw = a[7], ax1 = ax * bw + aw * bx + ay * bz - az * by, ay1 = ay * bw + aw * by + az * bx - ax * bz, az1 = az * bw + aw * bz + ax * by - ay * bx, aw1 = aw * bw - ax * bx - ay * by - az * bz;
	    quat.rotateZ(out, a, rad);
	    bx = out[0];
	    by = out[1];
	    bz = out[2];
	    bw = out[3];
	    out[4] = ax1 * bw + aw1 * bx + ay1 * bz - az1 * by;
	    out[5] = ay1 * bw + aw1 * by + az1 * bx - ax1 * bz;
	    out[6] = az1 * bw + aw1 * bz + ax1 * by - ay1 * bx;
	    out[7] = aw1 * bw - ax1 * bx - ay1 * by - az1 * bz;
	    return out;
	}
	/**
	 * Rotates a dual quat by a given quaternion (a * q)
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @param {ReadonlyQuat} q quaternion to rotate by
	 * @returns {quat2} out
	 */
	function rotateByQuatAppend(out, a, q) {
	    var qx = q[0], qy = q[1], qz = q[2], qw = q[3], ax = a[0], ay = a[1], az = a[2], aw = a[3];
	    out[0] = ax * qw + aw * qx + ay * qz - az * qy;
	    out[1] = ay * qw + aw * qy + az * qx - ax * qz;
	    out[2] = az * qw + aw * qz + ax * qy - ay * qx;
	    out[3] = aw * qw - ax * qx - ay * qy - az * qz;
	    ax = a[4];
	    ay = a[5];
	    az = a[6];
	    aw = a[7];
	    out[4] = ax * qw + aw * qx + ay * qz - az * qy;
	    out[5] = ay * qw + aw * qy + az * qx - ax * qz;
	    out[6] = az * qw + aw * qz + ax * qy - ay * qx;
	    out[7] = aw * qw - ax * qx - ay * qy - az * qz;
	    return out;
	}
	/**
	 * Rotates a dual quat by a given quaternion (q * a)
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat} q quaternion to rotate by
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @returns {quat2} out
	 */
	function rotateByQuatPrepend(out, q, a) {
	    var qx = q[0], qy = q[1], qz = q[2], qw = q[3], bx = a[0], by = a[1], bz = a[2], bw = a[3];
	    out[0] = qx * bw + qw * bx + qy * bz - qz * by;
	    out[1] = qy * bw + qw * by + qz * bx - qx * bz;
	    out[2] = qz * bw + qw * bz + qx * by - qy * bx;
	    out[3] = qw * bw - qx * bx - qy * by - qz * bz;
	    bx = a[4];
	    by = a[5];
	    bz = a[6];
	    bw = a[7];
	    out[4] = qx * bw + qw * bx + qy * bz - qz * by;
	    out[5] = qy * bw + qw * by + qz * bx - qx * bz;
	    out[6] = qz * bw + qw * bz + qx * by - qy * bx;
	    out[7] = qw * bw - qx * bx - qy * by - qz * bz;
	    return out;
	}
	/**
	 * Rotates a dual quat around a given axis. Does the normalisation automatically
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the dual quaternion to rotate
	 * @param {ReadonlyVec3} axis the axis to rotate around
	 * @param {Number} rad how far the rotation should be
	 * @returns {quat2} out
	 */
	function rotateAroundAxis(out, a, axis, rad) {
	    //Special case for rad = 0
	    if (Math.abs(rad) < glMatrix.EPSILON) {
	        return copy(out, a);
	    }
	    var axisLength = Math.hypot(axis[0], axis[1], axis[2]);
	    rad = rad * 0.5;
	    var s = Math.sin(rad);
	    var bx = s * axis[0] / axisLength;
	    var by = s * axis[1] / axisLength;
	    var bz = s * axis[2] / axisLength;
	    var bw = Math.cos(rad);
	    var ax1 = a[0], ay1 = a[1], az1 = a[2], aw1 = a[3];
	    out[0] = ax1 * bw + aw1 * bx + ay1 * bz - az1 * by;
	    out[1] = ay1 * bw + aw1 * by + az1 * bx - ax1 * bz;
	    out[2] = az1 * bw + aw1 * bz + ax1 * by - ay1 * bx;
	    out[3] = aw1 * bw - ax1 * bx - ay1 * by - az1 * bz;
	    var ax = a[4], ay = a[5], az = a[6], aw = a[7];
	    out[4] = ax * bw + aw * bx + ay * bz - az * by;
	    out[5] = ay * bw + aw * by + az * bx - ax * bz;
	    out[6] = az * bw + aw * bz + ax * by - ay * bx;
	    out[7] = aw * bw - ax * bx - ay * by - az * bz;
	    return out;
	}
	/**
	 * Adds two dual quat's
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the first operand
	 * @param {ReadonlyQuat2} b the second operand
	 * @returns {quat2} out
	 * @function
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    out[2] = a[2] + b[2];
	    out[3] = a[3] + b[3];
	    out[4] = a[4] + b[4];
	    out[5] = a[5] + b[5];
	    out[6] = a[6] + b[6];
	    out[7] = a[7] + b[7];
	    return out;
	}
	/**
	 * Multiplies two dual quat's
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a the first operand
	 * @param {ReadonlyQuat2} b the second operand
	 * @returns {quat2} out
	 */
	function multiply(out, a, b) {
	    var ax0 = a[0], ay0 = a[1], az0 = a[2], aw0 = a[3], bx1 = b[4], by1 = b[5], bz1 = b[6], bw1 = b[7], ax1 = a[4], ay1 = a[5], az1 = a[6], aw1 = a[7], bx0 = b[0], by0 = b[1], bz0 = b[2], bw0 = b[3];
	    out[0] = ax0 * bw0 + aw0 * bx0 + ay0 * bz0 - az0 * by0;
	    out[1] = ay0 * bw0 + aw0 * by0 + az0 * bx0 - ax0 * bz0;
	    out[2] = az0 * bw0 + aw0 * bz0 + ax0 * by0 - ay0 * bx0;
	    out[3] = aw0 * bw0 - ax0 * bx0 - ay0 * by0 - az0 * bz0;
	    out[4] = ax0 * bw1 + aw0 * bx1 + ay0 * bz1 - az0 * by1 + ax1 * bw0 + aw1 * bx0 + ay1 * bz0 - az1 * by0;
	    out[5] = ay0 * bw1 + aw0 * by1 + az0 * bx1 - ax0 * bz1 + ay1 * bw0 + aw1 * by0 + az1 * bx0 - ax1 * bz0;
	    out[6] = az0 * bw1 + aw0 * bz1 + ax0 * by1 - ay0 * bx1 + az1 * bw0 + aw1 * bz0 + ax1 * by0 - ay1 * bx0;
	    out[7] = aw0 * bw1 - ax0 * bx1 - ay0 * by1 - az0 * bz1 + aw1 * bw0 - ax1 * bx0 - ay1 * by0 - az1 * bz0;
	    return out;
	}
	/**
	 * Alias for {@link quat2.multiply}
	 * @function
	 */
	var mul = multiply;
	/**
	 * Scales a dual quat by a scalar number
	 *
	 * @param {quat2} out the receiving dual quat
	 * @param {ReadonlyQuat2} a the dual quat to scale
	 * @param {Number} b amount to scale the dual quat by
	 * @returns {quat2} out
	 * @function
	 */
	quat2.mul = mul;
	function scale(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    out[2] = a[2] * b;
	    out[3] = a[3] * b;
	    out[4] = a[4] * b;
	    out[5] = a[5] * b;
	    out[6] = a[6] * b;
	    out[7] = a[7] * b;
	    return out;
	}
	/**
	 * Calculates the dot product of two dual quat's (The dot product of the real parts)
	 *
	 * @param {ReadonlyQuat2} a the first operand
	 * @param {ReadonlyQuat2} b the second operand
	 * @returns {Number} dot product of a and b
	 * @function
	 */
	var dot = quat.dot;
	/**
	 * Performs a linear interpolation between two dual quats's
	 * NOTE: The resulting dual quaternions won't always be normalized (The error is most noticeable when t = 0.5)
	 *
	 * @param {quat2} out the receiving dual quat
	 * @param {ReadonlyQuat2} a the first operand
	 * @param {ReadonlyQuat2} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {quat2} out
	 */
	quat2.dot = dot;
	function lerp(out, a, b, t) {
	    var mt = 1 - t;
	    if (dot(a, b) < 0)
	        t = -t;
	    out[0] = a[0] * mt + b[0] * t;
	    out[1] = a[1] * mt + b[1] * t;
	    out[2] = a[2] * mt + b[2] * t;
	    out[3] = a[3] * mt + b[3] * t;
	    out[4] = a[4] * mt + b[4] * t;
	    out[5] = a[5] * mt + b[5] * t;
	    out[6] = a[6] * mt + b[6] * t;
	    out[7] = a[7] * mt + b[7] * t;
	    return out;
	}
	/**
	 * Calculates the inverse of a dual quat. If they are normalized, conjugate is cheaper
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a dual quat to calculate inverse of
	 * @returns {quat2} out
	 */
	function invert(out, a) {
	    var sqlen = squaredLength(a);
	    out[0] = -a[0] / sqlen;
	    out[1] = -a[1] / sqlen;
	    out[2] = -a[2] / sqlen;
	    out[3] = a[3] / sqlen;
	    out[4] = -a[4] / sqlen;
	    out[5] = -a[5] / sqlen;
	    out[6] = -a[6] / sqlen;
	    out[7] = a[7] / sqlen;
	    return out;
	}
	/**
	 * Calculates the conjugate of a dual quat
	 * If the dual quaternion is normalized, this function is faster than quat2.inverse and produces the same result.
	 *
	 * @param {quat2} out the receiving quaternion
	 * @param {ReadonlyQuat2} a quat to calculate conjugate of
	 * @returns {quat2} out
	 */
	function conjugate(out, a) {
	    out[0] = -a[0];
	    out[1] = -a[1];
	    out[2] = -a[2];
	    out[3] = a[3];
	    out[4] = -a[4];
	    out[5] = -a[5];
	    out[6] = -a[6];
	    out[7] = a[7];
	    return out;
	}
	/**
	 * Calculates the length of a dual quat
	 *
	 * @param {ReadonlyQuat2} a dual quat to calculate length of
	 * @returns {Number} length of a
	 * @function
	 */
	var length = quat.length;
	/**
	 * Alias for {@link quat2.length}
	 * @function
	 */
	quat2.length = length;
	var len = length;
	/**
	 * Calculates the squared length of a dual quat
	 *
	 * @param {ReadonlyQuat2} a dual quat to calculate squared length of
	 * @returns {Number} squared length of a
	 * @function
	 */
	quat2.len = len;
	var squaredLength = quat.squaredLength;
	/**
	 * Alias for {@link quat2.squaredLength}
	 * @function
	 */
	quat2.squaredLength = squaredLength;
	var sqrLen = squaredLength;
	/**
	 * Normalize a dual quat
	 *
	 * @param {quat2} out the receiving dual quaternion
	 * @param {ReadonlyQuat2} a dual quaternion to normalize
	 * @returns {quat2} out
	 * @function
	 */
	quat2.sqrLen = sqrLen;
	function normalize(out, a) {
	    var magnitude = squaredLength(a);
	    if (magnitude > 0) {
	        magnitude = Math.sqrt(magnitude);
	        var a0 = a[0] / magnitude;
	        var a1 = a[1] / magnitude;
	        var a2 = a[2] / magnitude;
	        var a3 = a[3] / magnitude;
	        var b0 = a[4];
	        var b1 = a[5];
	        var b2 = a[6];
	        var b3 = a[7];
	        var a_dot_b = a0 * b0 + a1 * b1 + a2 * b2 + a3 * b3;
	        out[0] = a0;
	        out[1] = a1;
	        out[2] = a2;
	        out[3] = a3;
	        out[4] = (b0 - a0 * a_dot_b) / magnitude;
	        out[5] = (b1 - a1 * a_dot_b) / magnitude;
	        out[6] = (b2 - a2 * a_dot_b) / magnitude;
	        out[7] = (b3 - a3 * a_dot_b) / magnitude;
	    }
	    return out;
	}
	/**
	 * Returns a string representation of a dual quatenion
	 *
	 * @param {ReadonlyQuat2} a dual quaternion to represent as a string
	 * @returns {String} string representation of the dual quat
	 */
	function str(a) {
	    return 'quat2(' + a[0] + ', ' + a[1] + ', ' + a[2] + ', ' + a[3] + ', ' + a[4] + ', ' + a[5] + ', ' + a[6] + ', ' + a[7] + ')';
	}
	/**
	 * Returns whether or not the dual quaternions have exactly the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyQuat2} a the first dual quaternion.
	 * @param {ReadonlyQuat2} b the second dual quaternion.
	 * @returns {Boolean} true if the dual quaternions are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1] && a[2] === b[2] && a[3] === b[3] && a[4] === b[4] && a[5] === b[5] && a[6] === b[6] && a[7] === b[7];
	}
	/**
	 * Returns whether or not the dual quaternions have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyQuat2} a the first dual quat.
	 * @param {ReadonlyQuat2} b the second dual quat.
	 * @returns {Boolean} true if the dual quats are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1], a2 = a[2], a3 = a[3], a4 = a[4], a5 = a[5], a6 = a[6], a7 = a[7];
	    var b0 = b[0], b1 = b[1], b2 = b[2], b3 = b[3], b4 = b[4], b5 = b[5], b6 = b[6], b7 = b[7];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1)) && Math.abs(a2 - b2) <= glMatrix.EPSILON * Math.max(1, Math.abs(a2), Math.abs(b2)) && Math.abs(a3 - b3) <= glMatrix.EPSILON * Math.max(1, Math.abs(a3), Math.abs(b3)) && Math.abs(a4 - b4) <= glMatrix.EPSILON * Math.max(1, Math.abs(a4), Math.abs(b4)) && Math.abs(a5 - b5) <= glMatrix.EPSILON * Math.max(1, Math.abs(a5), Math.abs(b5)) && Math.abs(a6 - b6) <= glMatrix.EPSILON * Math.max(1, Math.abs(a6), Math.abs(b6)) && Math.abs(a7 - b7) <= glMatrix.EPSILON * Math.max(1, Math.abs(a7), Math.abs(b7));
	}
	return quat2;
}

var vec2 = {};

var hasRequiredVec2;

function requireVec2 () {
	if (hasRequiredVec2) return vec2;
	hasRequiredVec2 = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(vec2, '__esModule', { value: true });
	vec2.create = create;
	vec2.clone = clone;
	vec2.fromValues = fromValues;
	vec2.copy = copy;
	vec2.set = set;
	vec2.add = add;
	vec2.subtract = subtract;
	vec2.multiply = multiply;
	vec2.divide = divide;
	vec2.ceil = ceil;
	vec2.floor = floor;
	vec2.min = min;
	vec2.max = max;
	vec2.round = round;
	vec2.scale = scale;
	vec2.scaleAndAdd = scaleAndAdd;
	vec2.distance = distance;
	vec2.squaredDistance = squaredDistance;
	vec2.length = length;
	vec2.squaredLength = squaredLength;
	vec2.negate = negate;
	vec2.inverse = inverse;
	vec2.normalize = normalize;
	vec2.dot = dot;
	vec2.cross = cross;
	vec2.lerp = lerp;
	vec2.random = random;
	vec2.transformMat2 = transformMat2;
	vec2.transformMat2d = transformMat2d;
	vec2.transformMat3 = transformMat3;
	vec2.transformMat4 = transformMat4;
	vec2.rotate = rotate;
	vec2.angle = angle;
	vec2.zero = zero;
	vec2.str = str;
	vec2.exactEquals = exactEquals;
	vec2.equals = equals;
	vec2.forEach = vec2.sqrLen = vec2.sqrDist = vec2.dist = vec2.div = vec2.mul = vec2.sub = vec2.len = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	/**
	 * 2 Dimensional Vector
	 * @module vec2
	 */
	/**
	 * Creates a new, empty vec2
	 *
	 * @returns {vec2} a new 2D vector
	 */
	function create() {
	    var out = new glMatrix.ARRAY_TYPE(2);
	    if (glMatrix.ARRAY_TYPE != Float32Array) {
	        out[0] = 0;
	        out[1] = 0;
	    }
	    return out;
	}
	/**
	 * Creates a new vec2 initialized with values from an existing vector
	 *
	 * @param {ReadonlyVec2} a vector to clone
	 * @returns {vec2} a new 2D vector
	 */
	function clone(a) {
	    var out = new glMatrix.ARRAY_TYPE(2);
	    out[0] = a[0];
	    out[1] = a[1];
	    return out;
	}
	/**
	 * Creates a new vec2 initialized with the given values
	 *
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @returns {vec2} a new 2D vector
	 */
	function fromValues(x, y) {
	    var out = new glMatrix.ARRAY_TYPE(2);
	    out[0] = x;
	    out[1] = y;
	    return out;
	}
	/**
	 * Copy the values from one vec2 to another
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the source vector
	 * @returns {vec2} out
	 */
	function copy(out, a) {
	    out[0] = a[0];
	    out[1] = a[1];
	    return out;
	}
	/**
	 * Set the components of a vec2 to the given values
	 *
	 * @param {vec2} out the receiving vector
	 * @param {Number} x X component
	 * @param {Number} y Y component
	 * @returns {vec2} out
	 */
	function set(out, x, y) {
	    out[0] = x;
	    out[1] = y;
	    return out;
	}
	/**
	 * Adds two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function add(out, a, b) {
	    out[0] = a[0] + b[0];
	    out[1] = a[1] + b[1];
	    return out;
	}
	/**
	 * Subtracts vector b from vector a
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function subtract(out, a, b) {
	    out[0] = a[0] - b[0];
	    out[1] = a[1] - b[1];
	    return out;
	}
	/**
	 * Multiplies two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function multiply(out, a, b) {
	    out[0] = a[0] * b[0];
	    out[1] = a[1] * b[1];
	    return out;
	}
	/**
	 * Divides two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function divide(out, a, b) {
	    out[0] = a[0] / b[0];
	    out[1] = a[1] / b[1];
	    return out;
	}
	/**
	 * Math.ceil the components of a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to ceil
	 * @returns {vec2} out
	 */
	function ceil(out, a) {
	    out[0] = Math.ceil(a[0]);
	    out[1] = Math.ceil(a[1]);
	    return out;
	}
	/**
	 * Math.floor the components of a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to floor
	 * @returns {vec2} out
	 */
	function floor(out, a) {
	    out[0] = Math.floor(a[0]);
	    out[1] = Math.floor(a[1]);
	    return out;
	}
	/**
	 * Returns the minimum of two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function min(out, a, b) {
	    out[0] = Math.min(a[0], b[0]);
	    out[1] = Math.min(a[1], b[1]);
	    return out;
	}
	/**
	 * Returns the maximum of two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec2} out
	 */
	function max(out, a, b) {
	    out[0] = Math.max(a[0], b[0]);
	    out[1] = Math.max(a[1], b[1]);
	    return out;
	}
	/**
	 * Math.round the components of a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to round
	 * @returns {vec2} out
	 */
	function round(out, a) {
	    out[0] = Math.round(a[0]);
	    out[1] = Math.round(a[1]);
	    return out;
	}
	/**
	 * Scales a vec2 by a scalar number
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the vector to scale
	 * @param {Number} b amount to scale the vector by
	 * @returns {vec2} out
	 */
	function scale(out, a, b) {
	    out[0] = a[0] * b;
	    out[1] = a[1] * b;
	    return out;
	}
	/**
	 * Adds two vec2's after scaling the second operand by a scalar value
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @param {Number} scale the amount to scale b by before adding
	 * @returns {vec2} out
	 */
	function scaleAndAdd(out, a, b, scale) {
	    out[0] = a[0] + b[0] * scale;
	    out[1] = a[1] + b[1] * scale;
	    return out;
	}
	/**
	 * Calculates the euclidian distance between two vec2's
	 *
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {Number} distance between a and b
	 */
	function distance(a, b) {
	    var x = b[0] - a[0], y = b[1] - a[1];
	    return Math.hypot(x, y);
	}
	/**
	 * Calculates the squared euclidian distance between two vec2's
	 *
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {Number} squared distance between a and b
	 */
	function squaredDistance(a, b) {
	    var x = b[0] - a[0], y = b[1] - a[1];
	    return x * x + y * y;
	}
	/**
	 * Calculates the length of a vec2
	 *
	 * @param {ReadonlyVec2} a vector to calculate length of
	 * @returns {Number} length of a
	 */
	function length(a) {
	    var x = a[0], y = a[1];
	    return Math.hypot(x, y);
	}
	/**
	 * Calculates the squared length of a vec2
	 *
	 * @param {ReadonlyVec2} a vector to calculate squared length of
	 * @returns {Number} squared length of a
	 */
	function squaredLength(a) {
	    var x = a[0], y = a[1];
	    return x * x + y * y;
	}
	/**
	 * Negates the components of a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to negate
	 * @returns {vec2} out
	 */
	function negate(out, a) {
	    out[0] = -a[0];
	    out[1] = -a[1];
	    return out;
	}
	/**
	 * Returns the inverse of the components of a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to invert
	 * @returns {vec2} out
	 */
	function inverse(out, a) {
	    out[0] = 1 / a[0];
	    out[1] = 1 / a[1];
	    return out;
	}
	/**
	 * Normalize a vec2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a vector to normalize
	 * @returns {vec2} out
	 */
	function normalize(out, a) {
	    var x = a[0], y = a[1];
	    var len = x * x + y * y;
	    if (len > 0) {
	        //TODO: evaluate use of glm_invsqrt here?
	        len = 1 / Math.sqrt(len);
	    }
	    out[0] = a[0] * len;
	    out[1] = a[1] * len;
	    return out;
	}
	/**
	 * Calculates the dot product of two vec2's
	 *
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {Number} dot product of a and b
	 */
	function dot(a, b) {
	    return a[0] * b[0] + a[1] * b[1];
	}
	/**
	 * Computes the cross product of two vec2's
	 * Note that the cross product must by definition produce a 3D vector
	 *
	 * @param {vec3} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @returns {vec3} out
	 */
	function cross(out, a, b) {
	    var z = a[0] * b[1] - a[1] * b[0];
	    out[0] = out[1] = 0;
	    out[2] = z;
	    return out;
	}
	/**
	 * Performs a linear interpolation between two vec2's
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the first operand
	 * @param {ReadonlyVec2} b the second operand
	 * @param {Number} t interpolation amount, in the range [0-1], between the two inputs
	 * @returns {vec2} out
	 */
	function lerp(out, a, b, t) {
	    var ax = a[0], ay = a[1];
	    out[0] = ax + t * (b[0] - ax);
	    out[1] = ay + t * (b[1] - ay);
	    return out;
	}
	/**
	 * Generates a random vector with the given scale
	 *
	 * @param {vec2} out the receiving vector
	 * @param {Number} [scale] Length of the resulting vector. If ommitted, a unit vector will be returned
	 * @returns {vec2} out
	 */
	function random(out, scale) {
	    scale = scale || 1;
	    var r = glMatrix.RANDOM() * 2 * Math.PI;
	    out[0] = Math.cos(r) * scale;
	    out[1] = Math.sin(r) * scale;
	    return out;
	}
	/**
	 * Transforms the vec2 with a mat2
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the vector to transform
	 * @param {ReadonlyMat2} m matrix to transform with
	 * @returns {vec2} out
	 */
	function transformMat2(out, a, m) {
	    var x = a[0], y = a[1];
	    out[0] = m[0] * x + m[2] * y;
	    out[1] = m[1] * x + m[3] * y;
	    return out;
	}
	/**
	 * Transforms the vec2 with a mat2d
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the vector to transform
	 * @param {ReadonlyMat2d} m matrix to transform with
	 * @returns {vec2} out
	 */
	function transformMat2d(out, a, m) {
	    var x = a[0], y = a[1];
	    out[0] = m[0] * x + m[2] * y + m[4];
	    out[1] = m[1] * x + m[3] * y + m[5];
	    return out;
	}
	/**
	 * Transforms the vec2 with a mat3
	 * 3rd vector component is implicitly '1'
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the vector to transform
	 * @param {ReadonlyMat3} m matrix to transform with
	 * @returns {vec2} out
	 */
	function transformMat3(out, a, m) {
	    var x = a[0], y = a[1];
	    out[0] = m[0] * x + m[3] * y + m[6];
	    out[1] = m[1] * x + m[4] * y + m[7];
	    return out;
	}
	/**
	 * Transforms the vec2 with a mat4
	 * 3rd vector component is implicitly '0'
	 * 4th vector component is implicitly '1'
	 *
	 * @param {vec2} out the receiving vector
	 * @param {ReadonlyVec2} a the vector to transform
	 * @param {ReadonlyMat4} m matrix to transform with
	 * @returns {vec2} out
	 */
	function transformMat4(out, a, m) {
	    var x = a[0];
	    var y = a[1];
	    out[0] = m[0] * x + m[4] * y + m[12];
	    out[1] = m[1] * x + m[5] * y + m[13];
	    return out;
	}
	/**
	 * Rotate a 2D vector
	 * @param {vec2} out The receiving vec2
	 * @param {ReadonlyVec2} a The vec2 point to rotate
	 * @param {ReadonlyVec2} b The origin of the rotation
	 * @param {Number} rad The angle of rotation in radians
	 * @returns {vec2} out
	 */
	function rotate(out, a, b, rad) {
	    //Translate point to the origin
	    var p0 = a[0] - b[0], p1 = a[1] - b[1], sinC = Math.sin(rad), cosC = Math.cos(rad);
	    //perform rotation and translate to correct position
	    out[0] = p0 * cosC - p1 * sinC + b[0];
	    out[1] = p0 * sinC + p1 * cosC + b[1];
	    return out;
	}
	/**
	 * Get the angle between two 2D vectors
	 * @param {ReadonlyVec2} a The first operand
	 * @param {ReadonlyVec2} b The second operand
	 * @returns {Number} The angle in radians
	 */
	function angle(a, b) {
	    var x1 = a[0], y1 = a[1], x2 = b[0], y2 = b[1],
	        // mag is the product of the magnitudes of a and b
	        mag = Math.sqrt(x1 * x1 + y1 * y1) * Math.sqrt(x2 * x2 + y2 * y2),
	        // mag &&.. short circuits if mag == 0
	        cosine = mag && (x1 * x2 + y1 * y2) / mag;
	    // Math.min(Math.max(cosine, -1), 1) clamps the cosine between -1 and 1
	    return Math.acos(Math.min(Math.max(cosine, -1), 1));
	}
	/**
	 * Set the components of a vec2 to zero
	 *
	 * @param {vec2} out the receiving vector
	 * @returns {vec2} out
	 */
	function zero(out) {
	    out[0] = 0;
	    out[1] = 0;
	    return out;
	}
	/**
	 * Returns a string representation of a vector
	 *
	 * @param {ReadonlyVec2} a vector to represent as a string
	 * @returns {String} string representation of the vector
	 */
	function str(a) {
	    return 'vec2(' + a[0] + ', ' + a[1] + ')';
	}
	/**
	 * Returns whether or not the vectors exactly have the same elements in the same position (when compared with ===)
	 *
	 * @param {ReadonlyVec2} a The first vector.
	 * @param {ReadonlyVec2} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function exactEquals(a, b) {
	    return a[0] === b[0] && a[1] === b[1];
	}
	/**
	 * Returns whether or not the vectors have approximately the same elements in the same position.
	 *
	 * @param {ReadonlyVec2} a The first vector.
	 * @param {ReadonlyVec2} b The second vector.
	 * @returns {Boolean} True if the vectors are equal, false otherwise.
	 */
	function equals(a, b) {
	    var a0 = a[0], a1 = a[1];
	    var b0 = b[0], b1 = b[1];
	    return Math.abs(a0 - b0) <= glMatrix.EPSILON * Math.max(1, Math.abs(a0), Math.abs(b0)) && Math.abs(a1 - b1) <= glMatrix.EPSILON * Math.max(1, Math.abs(a1), Math.abs(b1));
	}
	/**
	 * Alias for {@link vec2.length}
	 * @function
	 */
	var len = length;
	/**
	 * Alias for {@link vec2.subtract}
	 * @function
	 */
	vec2.len = len;
	var sub = subtract;
	/**
	 * Alias for {@link vec2.multiply}
	 * @function
	 */
	vec2.sub = sub;
	var mul = multiply;
	/**
	 * Alias for {@link vec2.divide}
	 * @function
	 */
	vec2.mul = mul;
	var div = divide;
	/**
	 * Alias for {@link vec2.distance}
	 * @function
	 */
	vec2.div = div;
	var dist = distance;
	/**
	 * Alias for {@link vec2.squaredDistance}
	 * @function
	 */
	vec2.dist = dist;
	var sqrDist = squaredDistance;
	/**
	 * Alias for {@link vec2.squaredLength}
	 * @function
	 */
	vec2.sqrDist = sqrDist;
	var sqrLen = squaredLength;
	/**
	 * Perform some operation over an array of vec2s.
	 *
	 * @param {Array} a the array of vectors to iterate over
	 * @param {Number} stride Number of elements between the start of each vec2. If 0 assumes tightly packed
	 * @param {Number} offset Number of elements to skip at the beginning of the array
	 * @param {Number} count Number of vec2s to iterate over. If 0 iterates over entire array
	 * @param {Function} fn Function to call for each vector in the array
	 * @param {Object} [arg] additional argument to pass to fn
	 * @returns {Array} a
	 * @function
	 */
	vec2.sqrLen = sqrLen;
	var forEach = (function () {
	    var vec = create();
	    return function (a, stride, offset, count, fn, arg) {
	        var i, l;
	        if (!stride) {
	            stride = 2;
	        }
	        if (!offset) {
	            offset = 0;
	        }
	        if (count) {
	            l = Math.min(count * stride + offset, a.length);
	        } else {
	            l = a.length;
	        }
	        for (i = offset; i < l; i += stride) {
	            vec[0] = a[i];
	            vec[1] = a[i + 1];
	            fn(vec, vec, arg);
	            a[i] = vec[0];
	            a[i + 1] = vec[1];
	        }
	        return a;
	    };
	}());
	vec2.forEach = forEach;
	return vec2;
}

var hasRequiredCjs;

function requireCjs () {
	if (hasRequiredCjs) return cjs;
	hasRequiredCjs = 1;
	function _typeof(obj) {
	    '@babel/helpers - typeof';
	    if (typeof Symbol === 'function' && typeof Symbol.iterator === 'symbol') {
	        _typeof = function _typeof(obj) {
	            return typeof obj;
	        };
	    } else {
	        _typeof = function _typeof(obj) {
	            return obj && typeof Symbol === 'function' && obj.constructor === Symbol && obj !== Symbol.prototype ? 'symbol' : typeof obj;
	        };
	    }
	    return _typeof(obj);
	}
	Object.defineProperty(cjs, '__esModule', { value: true });
	cjs.vec4 = cjs.vec3 = cjs.vec2 = cjs.quat2 = cjs.quat = cjs.mat4 = cjs.mat3 = cjs.mat2d = cjs.mat2 = cjs.glMatrix = void 0;
	var glMatrix = _interopRequireWildcard(requireCommon());
	cjs.glMatrix = glMatrix;
	var mat2 = _interopRequireWildcard(requireMat2());
	cjs.mat2 = mat2;
	var mat2d = _interopRequireWildcard(requireMat2d());
	cjs.mat2d = mat2d;
	var mat3 = _interopRequireWildcard(requireMat3());
	cjs.mat3 = mat3;
	var mat4 = _interopRequireWildcard(requireMat4());
	cjs.mat4 = mat4;
	var quat = _interopRequireWildcard(requireQuat());
	cjs.quat = quat;
	var quat2 = _interopRequireWildcard(requireQuat2());
	cjs.quat2 = quat2;
	var vec2 = _interopRequireWildcard(requireVec2());
	cjs.vec2 = vec2;
	var vec3 = _interopRequireWildcard(requireVec3());
	cjs.vec3 = vec3;
	var vec4 = _interopRequireWildcard(requireVec4());
	cjs.vec4 = vec4;
	function _getRequireWildcardCache(nodeInterop) {
	    if (typeof WeakMap !== 'function')
	        return null;
	    var cacheBabelInterop = new WeakMap();
	    var cacheNodeInterop = new WeakMap();
	    return (_getRequireWildcardCache = function _getRequireWildcardCache(nodeInterop) {
	        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
	    })(nodeInterop);
	}
	function _interopRequireWildcard(obj, nodeInterop) {
	    if (obj && obj.__esModule) {
	        return obj;
	    }
	    if (obj === null || _typeof(obj) !== 'object' && typeof obj !== 'function') {
	        return { 'default': obj };
	    }
	    var cache = _getRequireWildcardCache(nodeInterop);
	    if (cache && cache.has(obj)) {
	        return cache.get(obj);
	    }
	    var newObj = {};
	    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
	    for (var key in obj) {
	        if (key !== 'default' && Object.prototype.hasOwnProperty.call(obj, key)) {
	            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
	            if (desc && (desc.get || desc.set)) {
	                Object.defineProperty(newObj, key, desc);
	            } else {
	                newObj[key] = obj[key];
	            }
	        }
	    }
	    newObj['default'] = obj;
	    if (cache) {
	        cache.set(obj, newObj);
	    }
	    return newObj;
	}
	return cjs;
}

var cjsExports = /*@__PURE__*/ requireCjs();

var unitbezier;
var hasRequiredUnitbezier;

function requireUnitbezier () {
	if (hasRequiredUnitbezier) return unitbezier;
	hasRequiredUnitbezier = 1;
	unitbezier = UnitBezier;
	function UnitBezier(p1x, p1y, p2x, p2y) {
	    // Calculate the polynomial coefficients, implicit first and last control points are (0,0) and (1,1).
	    this.cx = 3 * p1x;
	    this.bx = 3 * (p2x - p1x) - this.cx;
	    this.ax = 1 - this.cx - this.bx;
	    this.cy = 3 * p1y;
	    this.by = 3 * (p2y - p1y) - this.cy;
	    this.ay = 1 - this.cy - this.by;
	    this.p1x = p1x;
	    this.p1y = p1y;
	    this.p2x = p2x;
	    this.p2y = p2y;
	}
	UnitBezier.prototype = {
	    sampleCurveX: function (t) {
	        // `ax t^3 + bx t^2 + cx t' expanded using Horner's rule.
	        return ((this.ax * t + this.bx) * t + this.cx) * t;
	    },
	    sampleCurveY: function (t) {
	        return ((this.ay * t + this.by) * t + this.cy) * t;
	    },
	    sampleCurveDerivativeX: function (t) {
	        return (3 * this.ax * t + 2 * this.bx) * t + this.cx;
	    },
	    solveCurveX: function (x, epsilon) {
	        if (epsilon === undefined)
	            epsilon = 0.000001;
	        if (x < 0)
	            return 0;
	        if (x > 1)
	            return 1;
	        var t = x;
	        // First try a few iterations of Newton's method - normally very fast.
	        for (var i = 0; i < 8; i++) {
	            var x2 = this.sampleCurveX(t) - x;
	            if (Math.abs(x2) < epsilon)
	                return t;
	            var d2 = this.sampleCurveDerivativeX(t);
	            if (Math.abs(d2) < 0.000001)
	                break;
	            t = t - x2 / d2;
	        }
	        // Fall back to the bisection method for reliability.
	        var t0 = 0;
	        var t1 = 1;
	        t = x;
	        for (i = 0; i < 20; i++) {
	            x2 = this.sampleCurveX(t);
	            if (Math.abs(x2 - x) < epsilon)
	                break;
	            if (x > x2) {
	                t0 = t;
	            } else {
	                t1 = t;
	            }
	            t = (t1 - t0) * 0.5 + t0;
	        }
	        return t;
	    },
	    solve: function (x, epsilon) {
	        return this.sampleCurveY(this.solveCurveX(x, epsilon));
	    }
	};
	return unitbezier;
}

var unitbezierExports = requireUnitbezier();
var UnitBezier = /*@__PURE__*/getDefaultExportFromCjs(unitbezierExports);

var pointGeometry;
var hasRequiredPointGeometry;

function requirePointGeometry () {
	if (hasRequiredPointGeometry) return pointGeometry;
	hasRequiredPointGeometry = 1;
	pointGeometry = Point;
	/**
	 * A standalone point geometry with useful accessor, comparison, and
	 * modification methods.
	 *
	 * @class Point
	 * @param {Number} x the x-coordinate. this could be longitude or screen
	 * pixels, or any other sort of unit.
	 * @param {Number} y the y-coordinate. this could be latitude or screen
	 * pixels, or any other sort of unit.
	 * @example
	 * var point = new Point(-77, 38);
	 */
	function Point(x, y) {
	    this.x = x;
	    this.y = y;
	}
	Point.prototype = {
	    /**
	     * Clone this point, returning a new point that can be modified
	     * without affecting the old one.
	     * @return {Point} the clone
	     */
	    clone: function () {
	        return new Point(this.x, this.y);
	    },
	    /**
	     * Add this point's x & y coordinates to another point,
	     * yielding a new point.
	     * @param {Point} p the other point
	     * @return {Point} output point
	     */
	    add: function (p) {
	        return this.clone()._add(p);
	    },
	    /**
	     * Subtract this point's x & y coordinates to from point,
	     * yielding a new point.
	     * @param {Point} p the other point
	     * @return {Point} output point
	     */
	    sub: function (p) {
	        return this.clone()._sub(p);
	    },
	    /**
	     * Multiply this point's x & y coordinates by point,
	     * yielding a new point.
	     * @param {Point} p the other point
	     * @return {Point} output point
	     */
	    multByPoint: function (p) {
	        return this.clone()._multByPoint(p);
	    },
	    /**
	     * Divide this point's x & y coordinates by point,
	     * yielding a new point.
	     * @param {Point} p the other point
	     * @return {Point} output point
	     */
	    divByPoint: function (p) {
	        return this.clone()._divByPoint(p);
	    },
	    /**
	     * Multiply this point's x & y coordinates by a factor,
	     * yielding a new point.
	     * @param {Point} k factor
	     * @return {Point} output point
	     */
	    mult: function (k) {
	        return this.clone()._mult(k);
	    },
	    /**
	     * Divide this point's x & y coordinates by a factor,
	     * yielding a new point.
	     * @param {Point} k factor
	     * @return {Point} output point
	     */
	    div: function (k) {
	        return this.clone()._div(k);
	    },
	    /**
	     * Rotate this point around the 0, 0 origin by an angle a,
	     * given in radians
	     * @param {Number} a angle to rotate around, in radians
	     * @return {Point} output point
	     */
	    rotate: function (a) {
	        return this.clone()._rotate(a);
	    },
	    /**
	     * Rotate this point around p point by an angle a,
	     * given in radians
	     * @param {Number} a angle to rotate around, in radians
	     * @param {Point} p Point to rotate around
	     * @return {Point} output point
	     */
	    rotateAround: function (a, p) {
	        return this.clone()._rotateAround(a, p);
	    },
	    /**
	     * Multiply this point by a 4x1 transformation matrix
	     * @param {Array<Number>} m transformation matrix
	     * @return {Point} output point
	     */
	    matMult: function (m) {
	        return this.clone()._matMult(m);
	    },
	    /**
	     * Calculate this point but as a unit vector from 0, 0, meaning
	     * that the distance from the resulting point to the 0, 0
	     * coordinate will be equal to 1 and the angle from the resulting
	     * point to the 0, 0 coordinate will be the same as before.
	     * @return {Point} unit vector point
	     */
	    unit: function () {
	        return this.clone()._unit();
	    },
	    /**
	     * Compute a perpendicular point, where the new y coordinate
	     * is the old x coordinate and the new x coordinate is the old y
	     * coordinate multiplied by -1
	     * @return {Point} perpendicular point
	     */
	    perp: function () {
	        return this.clone()._perp();
	    },
	    /**
	     * Return a version of this point with the x & y coordinates
	     * rounded to integers.
	     * @return {Point} rounded point
	     */
	    round: function () {
	        return this.clone()._round();
	    },
	    /**
	     * Return the magitude of this point: this is the Euclidean
	     * distance from the 0, 0 coordinate to this point's x and y
	     * coordinates.
	     * @return {Number} magnitude
	     */
	    mag: function () {
	        return Math.sqrt(this.x * this.x + this.y * this.y);
	    },
	    /**
	     * Judge whether this point is equal to another point, returning
	     * true or false.
	     * @param {Point} other the other point
	     * @return {boolean} whether the points are equal
	     */
	    equals: function (other) {
	        return this.x === other.x && this.y === other.y;
	    },
	    /**
	     * Calculate the distance from this point to another point
	     * @param {Point} p the other point
	     * @return {Number} distance
	     */
	    dist: function (p) {
	        return Math.sqrt(this.distSqr(p));
	    },
	    /**
	     * Calculate the distance from this point to another point,
	     * without the square root step. Useful if you're comparing
	     * relative distances.
	     * @param {Point} p the other point
	     * @return {Number} distance
	     */
	    distSqr: function (p) {
	        var dx = p.x - this.x, dy = p.y - this.y;
	        return dx * dx + dy * dy;
	    },
	    /**
	     * Get the angle from the 0, 0 coordinate to this point, in radians
	     * coordinates.
	     * @return {Number} angle
	     */
	    angle: function () {
	        return Math.atan2(this.y, this.x);
	    },
	    /**
	     * Get the angle from this point to another point, in radians
	     * @param {Point} b the other point
	     * @return {Number} angle
	     */
	    angleTo: function (b) {
	        return Math.atan2(this.y - b.y, this.x - b.x);
	    },
	    /**
	     * Get the angle between this point and another point, in radians
	     * @param {Point} b the other point
	     * @return {Number} angle
	     */
	    angleWith: function (b) {
	        return this.angleWithSep(b.x, b.y);
	    },
	    /*
	     * Find the angle of the two vectors, solving the formula for
	     * the cross product a x b = |a||b|sin(θ) for θ.
	     * @param {Number} x the x-coordinate
	     * @param {Number} y the y-coordinate
	     * @return {Number} the angle in radians
	     */
	    angleWithSep: function (x, y) {
	        return Math.atan2(this.x * y - this.y * x, this.x * x + this.y * y);
	    },
	    _matMult: function (m) {
	        var x = m[0] * this.x + m[1] * this.y, y = m[2] * this.x + m[3] * this.y;
	        this.x = x;
	        this.y = y;
	        return this;
	    },
	    _add: function (p) {
	        this.x += p.x;
	        this.y += p.y;
	        return this;
	    },
	    _sub: function (p) {
	        this.x -= p.x;
	        this.y -= p.y;
	        return this;
	    },
	    _mult: function (k) {
	        this.x *= k;
	        this.y *= k;
	        return this;
	    },
	    _div: function (k) {
	        this.x /= k;
	        this.y /= k;
	        return this;
	    },
	    _multByPoint: function (p) {
	        this.x *= p.x;
	        this.y *= p.y;
	        return this;
	    },
	    _divByPoint: function (p) {
	        this.x /= p.x;
	        this.y /= p.y;
	        return this;
	    },
	    _unit: function () {
	        this._div(this.mag());
	        return this;
	    },
	    _perp: function () {
	        var y = this.y;
	        this.y = this.x;
	        this.x = -y;
	        return this;
	    },
	    _rotate: function (angle) {
	        var cos = Math.cos(angle), sin = Math.sin(angle), x = cos * this.x - sin * this.y, y = sin * this.x + cos * this.y;
	        this.x = x;
	        this.y = y;
	        return this;
	    },
	    _rotateAround: function (angle, p) {
	        var cos = Math.cos(angle), sin = Math.sin(angle), x = p.x + cos * (this.x - p.x) - sin * (this.y - p.y), y = p.y + sin * (this.x - p.x) + cos * (this.y - p.y);
	        this.x = x;
	        this.y = y;
	        return this;
	    },
	    _round: function () {
	        this.x = Math.round(this.x);
	        this.y = Math.round(this.y);
	        return this;
	    }
	};
	/**
	 * Construct a point from an array if necessary, otherwise if the input
	 * is already a Point, or an unknown type, return it unchanged
	 * @param {Array<Number>|Point|*} a any kind of input value
	 * @return {Point} constructed point, or passed-through value.
	 * @example
	 * // this
	 * var point = Point.convert([0, 1]);
	 * // is equivalent to
	 * var point = new Point(0, 1);
	 */
	Point.convert = function (a) {
	    if (a instanceof Point) {
	        return a;
	    }
	    if (Array.isArray(a)) {
	        return new Point(a[0], a[1]);
	    }
	    return a;
	};
	return pointGeometry;
}

var pointGeometryExports = requirePointGeometry();
var Point = /*@__PURE__*/getDefaultExportFromCjs(pointGeometryExports);

function deepEqual(a, b) {
    if (Array.isArray(a)) {
        if (!Array.isArray(b) || a.length !== b.length)
            return false;
        for (let i = 0; i < a.length; i++) {
            if (!deepEqual(a[i], b[i]))
                return false;
        }
        return true;
    }
    if (typeof a === 'object' && a !== null && b !== null) {
        if (!(typeof b === 'object'))
            return false;
        const keys = Object.keys(a);
        if (keys.length !== Object.keys(b).length)
            return false;
        for (const key in a) {
            if (!deepEqual(a[key], b[key]))
                return false;
        }
        return true;
    }
    return a === b;
}

const DEG_TO_RAD = Math.PI / 180;
const RAD_TO_DEG = 180 / Math.PI;
function degToRad(a) {
    return a * DEG_TO_RAD;
}
function radToDeg(a) {
    return a * RAD_TO_DEG;
}
const TILE_CORNERS = [
    [
        0,
        0
    ],
    [
        1,
        0
    ],
    [
        1,
        1
    ],
    [
        0,
        1
    ]
];
function furthestTileCorner(bearing) {
    const alignedBearing = (bearing + 45 + 360) % 360;
    const cornerIdx = Math.round(alignedBearing / 90) % 4;
    return TILE_CORNERS[cornerIdx];
}
function easeCubicInOut(t) {
    if (t <= 0)
        return 0;
    if (t >= 1)
        return 1;
    const t2 = t * t, t3 = t2 * t;
    return 4 * (t < 0.5 ? t3 : 3 * (t - t2) + t3 - 0.75);
}
function getBounds(points) {
    let minX = Infinity;
    let minY = Infinity;
    let maxX = -Infinity;
    let maxY = -Infinity;
    for (const p of points) {
        minX = Math.min(minX, p.x);
        minY = Math.min(minY, p.y);
        maxX = Math.max(maxX, p.x);
        maxY = Math.max(maxY, p.y);
    }
    return {
        min: new Point(minX, minY),
        max: new Point(maxX, maxY)
    };
}
function getAABBPointSquareDist(min, max, point) {
    let sqDist = 0;
    for (let i = 0; i < 2; ++i) {
        const v = 0;
        if (min[i] > v)
            sqDist += (min[i] - v) * (min[i] - v);
        if (max[i] < v)
            sqDist += (v - max[i]) * (v - max[i]);
    }
    return sqDist;
}
function polygonizeBounds(min, max, buffer = 0, close = true) {
    const offset = new Point(buffer, buffer);
    const minBuf = min.sub(offset);
    const maxBuf = max.add(offset);
    const polygon = [
        minBuf,
        new Point(maxBuf.x, minBuf.y),
        maxBuf,
        new Point(minBuf.x, maxBuf.y)
    ];
    if (close) {
        polygon.push(minBuf.clone());
    }
    return polygon;
}
function bufferConvexPolygon(ring, buffer) {
    const output = [];
    for (let currIdx = 0; currIdx < ring.length; currIdx++) {
        const prevIdx = wrap$1(currIdx - 1, -1, ring.length - 1);
        const nextIdx = wrap$1(currIdx + 1, -1, ring.length - 1);
        const prev = ring[prevIdx];
        const curr = ring[currIdx];
        const next = ring[nextIdx];
        const p1 = prev.sub(curr).unit();
        const p2 = next.sub(curr).unit();
        const interiorAngle = p2.angleWithSep(p1.x, p1.y);
        const offset = p1.add(p2).unit().mult(-1 * buffer / Math.sin(interiorAngle / 2));
        output.push(curr.add(offset));
    }
    return output;
}
function bezier(p1x, p1y, p2x, p2y) {
    const bezier2 = new UnitBezier(p1x, p1y, p2x, p2y);
    return function (t) {
        return bezier2.solve(t);
    };
}
const ease = bezier(0.25, 0.1, 0.25, 1);
function clamp(n, min, max) {
    return Math.min(max, Math.max(min, n));
}
function smoothstep(e0, e1, x) {
    x = clamp((x - e0) / (e1 - e0), 0, 1);
    return x * x * (3 - 2 * x);
}
function wrap$1(n, min, max) {
    const d = max - min;
    const w = ((n - min) % d + d) % d + min;
    return w === min ? max : w;
}
function shortestAngle(a, b) {
    const diff = (b - a + 180) % 360 - 180;
    return diff < -180 ? diff + 360 : diff;
}
function asyncAll(array, fn, callback) {
    if (!array.length) {
        return callback(null, []);
    }
    let remaining = array.length;
    const results = new Array(array.length);
    let error = null;
    array.forEach((item, i) => {
        fn(item, (err, result) => {
            if (err)
                error = err;
            results[i] = result;
            if (--remaining === 0)
                callback(error, results);
        });
    });
}
function values(obj) {
    const result = [];
    for (const k in obj) {
        result.push(obj[k]);
    }
    return result;
}
function keysDifference(obj, other) {
    const difference = [];
    for (const i in obj) {
        if (!(i in other)) {
            difference.push(i);
        }
    }
    return difference;
}
function extend$1(dest, ...sources) {
    for (const src of sources) {
        for (const k in src) {
            dest[k] = src[k];
        }
    }
    return dest;
}
function pick(src, properties) {
    const result = {};
    for (let i = 0; i < properties.length; i++) {
        const k = properties[i];
        if (k in src) {
            result[k] = src[k];
        }
    }
    return result;
}
let id = 1;
function uniqueId() {
    return id++;
}
function uuid() {
    function b(a) {
        return a ? (a ^ Math.random() * (16 >> a / 4)).toString(16) : // @ts-expect-error - TS2365 - Operator '+' cannot be applied to types 'number[]' and 'number'.
        // eslint-disable-next-line
        ([10000000] + -[1000] + -4000 + -8000 + -100000000000).replace(/[018]/g, b);
    }
    return b();
}
function nextPowerOfTwo(value) {
    if (value <= 1)
        return 1;
    return Math.pow(2, Math.ceil(Math.log(value) / Math.LN2));
}
function prevPowerOfTwo(value) {
    if (value <= 1)
        return 1;
    return Math.pow(2, Math.floor(Math.log(value) / Math.LN2));
}
function validateUuid(str) {
    return str ? /^[0-9a-f]{8}-[0-9a-f]{4}-[4][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i.test(str) : false;
}
function bindAll(fns, context) {
    fns.forEach(fn => {
        if (!context[fn]) {
            return;
        }
        context[fn] = context[fn].bind(context);
    });
}
function endsWith(string, suffix) {
    return string.indexOf(suffix, string.length - suffix.length) !== -1;
}
function mapObject(input, iterator, context) {
    const output = {};
    for (const key in input) {
        output[key] = iterator.call(this, input[key], key, input);
    }
    return output;
}
function filterObject(input, iterator, context) {
    const output = {};
    for (const key in input) {
        if (iterator.call(this, input[key], key, input)) {
            output[key] = input[key];
        }
    }
    return output;
}
function clone(input) {
    if (Array.isArray(input)) {
        return input.map(clone);
    } else if (typeof input === 'object' && input) {
        return mapObject(input, clone);
    } else {
        return input;
    }
}
function mapValue(value, min, max, outMin, outMax) {
    return clamp((value - min) / (max - min) * (outMax - outMin) + outMin, outMin, outMax);
}
function arraysIntersect(a, b) {
    for (let l = 0; l < a.length; l++) {
        if (b.indexOf(a[l]) >= 0)
            return true;
    }
    return false;
}
const warnOnceHistory = {};
function warnOnce(message) {
    if (!warnOnceHistory[message]) {
        if (typeof console !== 'undefined')
            console.warn(message);
        warnOnceHistory[message] = true;
    }
}
function isCounterClockwise(a, b, c) {
    return (c.y - a.y) * (b.x - a.x) > (b.y - a.y) * (c.x - a.x);
}
function calculateSignedArea$1(ring) {
    let sum = 0;
    for (let i = 0, len = ring.length, j = len - 1, p1, p2; i < len; j = i++) {
        p1 = ring[i];
        p2 = ring[j];
        sum += (p2.x - p1.x) * (p1.y + p2.y);
    }
    return sum;
}
function sphericalPositionToCartesian([r, azimuthal, polar]) {
    const a = degToRad(azimuthal + 90), p = degToRad(polar);
    return {
        x: r * Math.cos(a) * Math.sin(p),
        y: r * Math.sin(a) * Math.sin(p),
        z: r * Math.cos(p),
        azimuthal,
        polar
    };
}
function sphericalDirectionToCartesian([azimuthal, polar]) {
    const position = sphericalPositionToCartesian([
        1,
        azimuthal,
        polar
    ]);
    return {
        x: position.x,
        y: position.y,
        z: position.z
    };
}
function cartesianPositionToSpherical(x, y, z) {
    const radial = Math.sqrt(x * x + y * y + z * z);
    const polar = radial > 0 ? Math.acos(z / radial) * RAD_TO_DEG : 0;
    let azimuthal = x !== 0 || y !== 0 ? Math.atan2(-y, -x) * RAD_TO_DEG + 90 : 0;
    if (azimuthal < 0) {
        azimuthal += 360;
    }
    return [
        radial,
        azimuthal,
        polar
    ];
}
function isWorker() {
    return typeof WorkerGlobalScope !== 'undefined' && typeof self !== 'undefined' && self instanceof WorkerGlobalScope;
}
function parseCacheControl(cacheControl) {
    const re = /(?:^|(?:\s*\,\s*))([^\x00-\x20\(\)<>@\,;\:\\"\/\[\]\?\=\{\}\x7F]+)(?:\=(?:([^\x00-\x20\(\)<>@\,;\:\\"\/\[\]\?\=\{\}\x7F]+)|(?:\"((?:[^"\\]|\\.)*)\")))?/g;
    const header = {};
    cacheControl.replace(re, ($0, $1, $2, $3) => {
        const value = $2 || $3;
        header[$1] = value ? value.toLowerCase() : true;
        return '';
    });
    if (header['max-age']) {
        const maxAge = parseInt(header['max-age'], 10);
        if (isNaN(maxAge))
            delete header['max-age'];
        else
            header['max-age'] = maxAge;
    }
    return header;
}
let _isSafari = null;
function isSafari(scope) {
    if (_isSafari == null) {
        const userAgent = scope.navigator ? scope.navigator.userAgent : null;
        _isSafari = !!scope.safari || !!(userAgent && (/\b(iPad|iPhone|iPod)\b/.test(userAgent) || !!userAgent.match('Safari') && !userAgent.match('Chrome')));
    }
    return _isSafari;
}
function isSafariWithAntialiasingBug(scope) {
    const userAgent = scope.navigator ? scope.navigator.userAgent : null;
    if (!isSafari(scope))
        return false;
    return userAgent && (userAgent.match('Version/15.4') || userAgent.match('Version/15.5') || userAgent.match(/CPU (OS|iPhone OS) (15_4|15_5) like Mac OS X/));
}
function isFullscreen() {
    return !!document.fullscreenElement || !!document.webkitFullscreenElement;
}
function storageAvailable(type) {
    try {
        const storage = self[type];
        storage.setItem('_mapbox_test_', 1);
        storage.removeItem('_mapbox_test_');
        return true;
    } catch (e) {
        return false;
    }
}
function b64EncodeUnicode(str) {
    return btoa(encodeURIComponent(str).replace(/%([0-9A-F]{2})/g, (match, p1) => {
        return String.fromCharCode(Number('0x' + p1));
    }));
}
function b64DecodeUnicode(str) {
    return decodeURIComponent(atob(str).split('').map(c => {
        return '%' + ('00' + c.charCodeAt(0).toString(16)).slice(-2);
    }).join(''));
}
function base64DecToArr(sBase64) {
    const str = atob(sBase64);
    const arr = new Uint8Array(str.length);
    for (let i = 0; i < str.length; i++)
        arr[i] = str.codePointAt(i);
    return arr;
}
function getColumn(matrix, col) {
    return [
        matrix[col * 4],
        matrix[col * 4 + 1],
        matrix[col * 4 + 2],
        matrix[col * 4 + 3]
    ];
}
function setColumn(matrix, col, values2) {
    matrix[col * 4 + 0] = values2[0];
    matrix[col * 4 + 1] = values2[1];
    matrix[col * 4 + 2] = values2[2];
    matrix[col * 4 + 3] = values2[3];
}
function sRGBToLinearAndScale(v, s) {
    return [
        Math.pow(v[0], 2.2) * s,
        Math.pow(v[1], 2.2) * s,
        Math.pow(v[2], 2.2) * s
    ];
}
function linearVec3TosRGB(v) {
    return [
        Math.pow(v[0], 1 / 2.2),
        Math.pow(v[1], 1 / 2.2),
        Math.pow(v[2], 1 / 2.2)
    ];
}
function lowerBound(array, startIndex, finishIndex, target) {
    while (startIndex < finishIndex) {
        const middleIndex = startIndex + finishIndex >> 1;
        if (array[middleIndex] < target) {
            startIndex = middleIndex + 1;
        } else {
            finishIndex = middleIndex;
        }
    }
    return startIndex;
}
function upperBound(array, startIndex, finishIndex, target) {
    while (startIndex < finishIndex) {
        const middleIndex = startIndex + finishIndex >> 1;
        if (array[middleIndex] <= target) {
            startIndex = middleIndex + 1;
        } else {
            finishIndex = middleIndex;
        }
    }
    return startIndex;
}
function contrastFactor(contrast) {
    return contrast > 0 ? 1 / (1.001 - contrast) : 1 + contrast;
}
function saturationFactor(saturation) {
    return saturation > 0 ? 1 - 1 / (1.001 - saturation) : -saturation;
}
function computeColorAdjustmentMatrix(saturation, contrast, brightnessMin, brightnessMax) {
    saturation = saturationFactor(saturation);
    contrast = contrastFactor(contrast);
    const m = cjsExports.mat4.create();
    const sa = saturation / 3;
    const sb = 1 - 2 * sa;
    const saturationMatrix = [
        sb,
        sa,
        sa,
        0,
        sa,
        sb,
        sa,
        0,
        sa,
        sa,
        sb,
        0,
        0,
        0,
        0,
        1
    ];
    const cs = 0.5 - 0.5 * contrast;
    const contrastMatrix = [
        contrast,
        0,
        0,
        0,
        0,
        contrast,
        0,
        0,
        0,
        0,
        contrast,
        0,
        cs,
        cs,
        cs,
        1
    ];
    const hl = brightnessMax - brightnessMin;
    const brightnessMatrix = [
        hl,
        0,
        0,
        0,
        0,
        hl,
        0,
        0,
        0,
        0,
        hl,
        0,
        brightnessMin,
        brightnessMin,
        brightnessMin,
        1
    ];
    cjsExports.mat4.multiply(m, brightnessMatrix, contrastMatrix);
    cjsExports.mat4.multiply(m, m, saturationMatrix);
    return m;
}

const config = {
    API_URL: 'https://api.mapbox.com',
    get API_URL_REGEX() {
        return /^((https?:)?\/\/)?([^\/]+\.)?mapbox\.c(n|om)(\/|\?|$)/i;
    },
    get API_TILEJSON_REGEX() {
        return /^((https?:)?\/\/)?([^\/]+\.)?mapbox\.c(n|om)(\/v[0-9]*\/.*\.json.*$)/i;
    },
    get API_SPRITE_REGEX() {
        return /^((https?:)?\/\/)?([^\/]+\.)?mapbox\.c(n|om)(\/styles\/v[0-9]*\/)(.*\/sprite.*\..*$)/i;
    },
    get API_FONTS_REGEX() {
        return /^((https?:)?\/\/)?([^\/]+\.)?mapbox\.c(n|om)(\/fonts\/v[0-9]*\/)(.*\.pbf.*$)/i;
    },
    get API_STYLE_REGEX() {
        return /^((https?:)?\/\/)?([^\/]+\.)?mapbox\.c(n|om)(\/styles\/v[0-9]*\/)(.*$)/i;
    },
    get API_CDN_URL_REGEX() {
        return /^((https?:)?\/\/)?api\.mapbox\.c(n|om)(\/mapbox-gl-js\/)(.*$)/i;
    },
    get EVENTS_URL() {
        if (!config.API_URL) {
            return null;
        }
        try {
            const url = new URL(config.API_URL);
            if (url.hostname === 'api.mapbox.cn') {
                return 'https://events.mapbox.cn/events/v2';
            } else if (url.hostname === 'api.mapbox.com') {
                return 'https://events.mapbox.com/events/v2';
            } else {
                return null;
            }
        } catch (e) {
            return null;
        }
    },
    SESSION_PATH: '/map-sessions/v1',
    FEEDBACK_URL: 'https://apps.mapbox.com/feedback',
    TILE_URL_VERSION: 'v4',
    RASTER_URL_PREFIX: 'raster/v1',
    RASTERARRAYS_URL_PREFIX: 'rasterarrays/v1',
    REQUIRE_ACCESS_TOKEN: true,
    ACCESS_TOKEN: null,
    DEFAULT_STYLE: 'mapbox://styles/mapbox/standard',
    MAX_PARALLEL_IMAGE_REQUESTS: 16,
    DRACO_URL: 'https://api.mapbox.com/mapbox-gl-js/draco_decoder_gltf_v1.5.6.wasm',
    MESHOPT_URL: 'https://api.mapbox.com/mapbox-gl-js/meshopt_base_v0.20.wasm',
    MESHOPT_SIMD_URL: 'https://api.mapbox.com/mapbox-gl-js/meshopt_simd_v0.20.wasm',
    GLYPHS_URL: 'mapbox://fonts/mapbox/{fontstack}/{range}.pbf',
    TILES3D_URL_PREFIX: '3dtiles/v1'
};

function isMapboxHTTPURL(url) {
    return config.API_URL_REGEX.test(url);
}
function isMapboxURL(url) {
    return url.indexOf('mapbox:') === 0;
}
function isMapboxHTTPCDNURL(url) {
    return config.API_CDN_URL_REGEX.test(url);
}
function isMapboxHTTPSpriteURL(url) {
    return config.API_SPRITE_REGEX.test(url);
}
function isMapboxHTTPStyleURL(url) {
    return config.API_STYLE_REGEX.test(url) && !isMapboxHTTPSpriteURL(url);
}
function isMapboxHTTPTileJSONURL(url) {
    return config.API_TILEJSON_REGEX.test(url);
}
function isMapboxHTTPFontsURL(url) {
    return config.API_FONTS_REGEX.test(url);
}
function hasCacheDefeatingSku(url) {
    return url.indexOf('sku=') > 0 && isMapboxHTTPURL(url);
}

let supportsOffscreenCanvas;
function offscreenCanvasSupported() {
    if (supportsOffscreenCanvas == null) {
        supportsOffscreenCanvas = self.OffscreenCanvas && new OffscreenCanvas(1, 1).getContext('2d') && typeof self.createImageBitmap === 'function';
    }
    return supportsOffscreenCanvas;
}

let linkEl;
let reducedMotionQuery;
let stubTime;
let canvas;
let hasCanvasFingerprintNoise;
const exported$1 = {
    /**
   * Returns either performance.now() or a value set by setNow.
   * @returns {number} Time value in milliseconds.
   */
    now() {
        if (stubTime !== void 0) {
            return stubTime;
        }
        return performance.now();
    },
    setNow(time) {
        stubTime = time;
    },
    restoreNow() {
        stubTime = void 0;
    },
    frame(fn) {
        const frame = requestAnimationFrame(fn);
        return { cancel: () => cancelAnimationFrame(frame) };
    },
    getImageData(img, padding = 0) {
        const {width, height} = img;
        if (!canvas) {
            canvas = document.createElement('canvas');
        }
        const context = canvas.getContext('2d', { willReadFrequently: true });
        if (!context) {
            throw new Error('failed to create canvas 2d context');
        }
        if (width > canvas.width || height > canvas.height) {
            canvas.width = width;
            canvas.height = height;
        }
        context.clearRect(-padding, -padding, width + 2 * padding, height + 2 * padding);
        context.drawImage(img, 0, 0, width, height);
        return context.getImageData(-padding, -padding, width + 2 * padding, height + 2 * padding);
    },
    resolveURL(path) {
        if (!linkEl)
            linkEl = document.createElement('a');
        linkEl.href = path;
        return linkEl.href;
    },
    get devicePixelRatio() {
        return window.devicePixelRatio;
    },
    get prefersReducedMotion() {
        if (!window.matchMedia)
            return false;
        if (reducedMotionQuery == null) {
            reducedMotionQuery = window.matchMedia('(prefers-reduced-motion: reduce)');
        }
        return reducedMotionQuery.matches;
    },
    /**
   * Returns true if the browser has OffscreenCanvas support and
   * adds noise to Canvas2D operations used for image decoding to prevent fingerprinting.
   */
    hasCanvasFingerprintNoise() {
        if (hasCanvasFingerprintNoise !== void 0) {
            return hasCanvasFingerprintNoise;
        }
        if (!offscreenCanvasSupported()) {
            hasCanvasFingerprintNoise = false;
            return false;
        }
        const offscreenCanvas = new OffscreenCanvas(255 / 3, 1);
        const offscreenCanvasContext = offscreenCanvas.getContext('2d', { willReadFrequently: true });
        let inc = 0;
        for (let i = 0; i < offscreenCanvas.width; ++i) {
            offscreenCanvasContext.fillStyle = `rgba(${ inc++ },${ inc++ },${ inc++ }, 255)`;
            offscreenCanvasContext.fillRect(i, 0, 1, 1);
        }
        const readData = offscreenCanvasContext.getImageData(0, 0, offscreenCanvas.width, offscreenCanvas.height);
        inc = 0;
        for (let i = 0; i < readData.data.length; ++i) {
            if (i % 4 !== 3 && inc++ !== readData.data[i]) {
                hasCanvasFingerprintNoise = true;
                return true;
            }
        }
        hasCanvasFingerprintNoise = false;
        return false;
    }
};

function setQueryParameters(url, params) {
    const paramStart = url.indexOf('?');
    if (paramStart < 0)
        return `${ url }?${ new URLSearchParams(params).toString() }`;
    const searchParams = new URLSearchParams(url.slice(paramStart));
    for (const key in params) {
        searchParams.set(key, params[key]);
    }
    return `${ url.slice(0, paramStart) }?${ searchParams.toString() }`;
}
function stripQueryParameters(url, params = { persistentParams: [] }) {
    const paramStart = url.indexOf('?');
    if (paramStart < 0)
        return url;
    const nextParams = new URLSearchParams();
    const searchParams = new URLSearchParams(url.slice(paramStart));
    for (const param of params.persistentParams) {
        const value = searchParams.get(param);
        if (value)
            nextParams.set(param, value);
    }
    const nextParamsString = nextParams.toString();
    return `${ url.slice(0, paramStart) }${ nextParamsString.length > 0 ? `?${ nextParamsString }` : '' }`;
}

const CACHE_NAME = 'mapbox-tiles';
let cacheLimit = 500;
let cacheCheckThreshold = 50;
const MIN_TIME_UNTIL_EXPIRY = 1000 * 60 * 7;
const PERSISTENT_PARAMS = [
    'language',
    'worldview',
    'jobid'
];
let sharedCache;
function getCaches() {
    try {
        return caches;
    } catch (e) {
    }
}
function cacheOpen() {
    const caches2 = getCaches();
    if (caches2 && sharedCache == null) {
        sharedCache = caches2.open(CACHE_NAME);
    }
}
let responseConstructorSupportsReadableStream;
function prepareBody(response, callback) {
    if (responseConstructorSupportsReadableStream === void 0) {
        try {
            new Response(new ReadableStream());
            responseConstructorSupportsReadableStream = true;
        } catch (e) {
            responseConstructorSupportsReadableStream = false;
        }
    }
    if (responseConstructorSupportsReadableStream) {
        callback(response.body);
    } else {
        response.blob().then(callback);
    }
}
function isNullBodyStatus(status) {
    if (status === 200 || status === 404) {
        return false;
    }
    return [
        101,
        103,
        204,
        205,
        304
    ].includes(status);
}
function cachePut(request, response, requestTime) {
    cacheOpen();
    if (sharedCache == null)
        return;
    const cacheControl = parseCacheControl(response.headers.get('Cache-Control') || '');
    if (cacheControl['no-store'])
        return;
    const options = {
        status: response.status,
        statusText: response.statusText,
        headers: new Headers()
    };
    response.headers.forEach((v, k) => options.headers.set(k, v));
    if (cacheControl['max-age']) {
        options.headers.set('Expires', new Date(requestTime + cacheControl['max-age'] * 1000).toUTCString());
    }
    const expires = options.headers.get('Expires');
    if (!expires)
        return;
    const timeUntilExpiry = new Date(expires).getTime() - requestTime;
    if (timeUntilExpiry < MIN_TIME_UNTIL_EXPIRY)
        return;
    let strippedURL = stripQueryParameters(request.url, { persistentParams: PERSISTENT_PARAMS });
    if (response.status === 206) {
        const range = request.headers.get('Range');
        if (!range)
            return;
        options.status = 200;
        strippedURL = setQueryParameters(strippedURL, { range });
    }
    prepareBody(response, body => {
        const clonedResponse = new Response(isNullBodyStatus(response.status) ? null : body, options);
        cacheOpen();
        if (sharedCache == null)
            return;
        sharedCache.then(cache => cache.put(strippedURL, clonedResponse)).catch(e => warnOnce(e.message));
    });
}
function cacheGet(request, callback) {
    cacheOpen();
    if (sharedCache == null)
        return callback(null);
    sharedCache.then(cache => {
        let strippedURL = stripQueryParameters(request.url, { persistentParams: PERSISTENT_PARAMS });
        const range = request.headers.get('Range');
        if (range)
            strippedURL = setQueryParameters(strippedURL, { range });
        cache.match(strippedURL).then(response => {
            const fresh = isFresh(response);
            cache.delete(strippedURL);
            if (fresh) {
                cache.put(strippedURL, response.clone());
            }
            callback(null, response, fresh);
        }).catch(callback);
    }).catch(callback);
}
function isFresh(response) {
    if (!response)
        return false;
    const expires = new Date(response.headers.get('Expires') || 0);
    const cacheControl = parseCacheControl(response.headers.get('Cache-Control') || '');
    return expires > Date.now() && !cacheControl['no-cache'];
}
let globalEntryCounter = Infinity;
function cacheEntryPossiblyAdded(dispatcher) {
    globalEntryCounter++;
    if (globalEntryCounter > cacheCheckThreshold) {
        dispatcher.getActor().send('enforceCacheSizeLimit', cacheLimit);
        globalEntryCounter = 0;
    }
}
function enforceCacheSizeLimit(limit) {
    cacheOpen();
    if (sharedCache == null)
        return;
    sharedCache.then(cache => {
        cache.keys().then(keys => {
            for (let i = 0; i < keys.length - limit; i++) {
                cache.delete(keys[i]);
            }
        });
    });
}
function clearTileCache(callback) {
    const caches2 = getCaches();
    if (!caches2)
        return;
    const promise = caches2.delete(CACHE_NAME);
    if (callback) {
        promise.catch(callback).then(() => callback());
    }
}
function setCacheLimits(limit, checkThreshold) {
    cacheLimit = limit;
    cacheCheckThreshold = checkThreshold;
}

const exported = {
    supported: false,
    testSupport
};
let glForTesting;
let webpCheckComplete = false;
let webpImgTest;
let webpImgTestOnloadComplete = false;
const window$1 = typeof self !== 'undefined' ? self : {};
if (window$1.document) {
    webpImgTest = window$1.document.createElement('img');
    webpImgTest.onload = function () {
        if (glForTesting)
            testWebpTextureUpload(glForTesting);
        glForTesting = null;
        webpImgTestOnloadComplete = true;
    };
    webpImgTest.onerror = function () {
        webpCheckComplete = true;
        glForTesting = null;
    };
    webpImgTest.src = 'data:image/webp;base64,UklGRh4AAABXRUJQVlA4TBEAAAAvAQAAAAfQ//73v/+BiOh/AAA=';
}
function testSupport(gl) {
    if (webpCheckComplete || !webpImgTest)
        return;
    if (webpImgTestOnloadComplete) {
        testWebpTextureUpload(gl);
    } else {
        glForTesting = gl;
    }
}
function testWebpTextureUpload(gl) {
    const texture = gl.createTexture();
    gl.bindTexture(gl.TEXTURE_2D, texture);
    try {
        gl.texImage2D(gl.TEXTURE_2D, 0, gl.RGBA, gl.RGBA, gl.UNSIGNED_BYTE, webpImgTest);
        if (gl.isContextLost())
            return;
        exported.supported = true;
    } catch (e) {
    }
    gl.deleteTexture(texture);
    webpCheckComplete = true;
}

const ResourceType = {
    Unknown: 'Unknown',
    Style: 'Style',
    Source: 'Source',
    Tile: 'Tile',
    Glyphs: 'Glyphs',
    SpriteImage: 'SpriteImage',
    SpriteJSON: 'SpriteJSON',
    Iconset: 'Iconset',
    Image: 'Image',
    Model: 'Model'
};
if (typeof Object.freeze == 'function') {
    Object.freeze(ResourceType);
}
class AJAXError extends Error {
    constructor(message, status, url) {
        if (status === 401 && isMapboxHTTPURL(url)) {
            message += ': you may have provided an invalid Mapbox access token. See https://docs.mapbox.com/api/overview/#access-tokens-and-token-scopes';
        }
        super(message);
        this.status = status;
        this.url = url;
    }
    toString() {
        return `${ this.name }: ${ this.message } (${ this.status }): ${ this.url }`;
    }
}
const getReferrer = isWorker() ? // @ts-expect-error - TS2551 - Property 'worker' does not exist on type 'Window & typeof globalThis'. Did you mean 'Worker'? | TS2551 - Property 'worker' does not exist on type 'Window & typeof globalThis'. Did you mean 'Worker'?
() => self.worker && self.worker.referrer : () => (location.protocol === 'blob:' ? parent : self).location.href;
const isFileURL = url => /^file:/.test(url) || /^file:/.test(getReferrer()) && !/^\w+:/.test(url);
function makeFetchRequest(requestParameters, callback) {
    const controller = new AbortController();
    const request = new Request(requestParameters.url, {
        method: requestParameters.method || 'GET',
        body: requestParameters.body,
        credentials: requestParameters.credentials,
        headers: requestParameters.headers,
        referrer: getReferrer(),
        referrerPolicy: requestParameters.referrerPolicy,
        signal: controller.signal
    });
    let complete = false;
    let aborted = false;
    const cacheIgnoringSearch = hasCacheDefeatingSku(request.url);
    if (requestParameters.type === 'json') {
        request.headers.set('Accept', 'application/json');
    }
    const validateOrFetch = (err, cachedResponse, responseIsFresh) => {
        if (aborted)
            return;
        if (err) {
            if (err.message !== 'SecurityError') {
                warnOnce(err.toString());
            }
        }
        if (cachedResponse && responseIsFresh) {
            return finishRequest(cachedResponse);
        }
        const requestTime = Date.now();
        fetch(request).then(response => {
            if (response.ok) {
                const cacheableResponse = cacheIgnoringSearch ? response.clone() : null;
                return finishRequest(response, cacheableResponse, requestTime);
            } else {
                return callback(new AJAXError(response.statusText, response.status, requestParameters.url));
            }
        }).catch(error => {
            if (error.name === 'AbortError') {
                return;
            }
            callback(new Error(`${ error.message } ${ requestParameters.url }`));
        });
    };
    const finishRequest = (response, cacheableResponse, requestTime) => {
        (requestParameters.type === 'arrayBuffer' ? response.arrayBuffer() : requestParameters.type === 'json' ? response.json() : response.text()).then(result => {
            if (aborted)
                return;
            if (cacheableResponse && requestTime) {
                cachePut(request, cacheableResponse, requestTime);
            }
            complete = true;
            callback(null, result, response.headers.get('Cache-Control'), response.headers.get('Expires'));
        }).catch(err => {
            if (!aborted)
                callback(new Error(err.message));
        });
    };
    if (cacheIgnoringSearch) {
        cacheGet(request, validateOrFetch);
    } else {
        validateOrFetch(null, null);
    }
    return {
        cancel: () => {
            aborted = true;
            if (!complete)
                controller.abort();
        }
    };
}
function makeXMLHttpRequest(requestParameters, callback) {
    const xhr = new XMLHttpRequest();
    xhr.open(requestParameters.method || 'GET', requestParameters.url, true);
    if (requestParameters.type === 'arrayBuffer') {
        xhr.responseType = 'arraybuffer';
    }
    for (const k in requestParameters.headers) {
        xhr.setRequestHeader(k, requestParameters.headers[k]);
    }
    if (requestParameters.type === 'json') {
        xhr.responseType = 'text';
        xhr.setRequestHeader('Accept', 'application/json');
    }
    xhr.withCredentials = requestParameters.credentials === 'include';
    xhr.onerror = () => {
        callback(new Error(xhr.statusText));
    };
    xhr.onload = () => {
        if ((xhr.status >= 200 && xhr.status < 300 || xhr.status === 0) && xhr.response !== null) {
            let data = xhr.response;
            if (requestParameters.type === 'json') {
                try {
                    data = JSON.parse(xhr.response);
                } catch (err) {
                    return callback(err);
                }
            }
            callback(null, data, xhr.getResponseHeader('Cache-Control'), xhr.getResponseHeader('Expires'));
        } else {
            callback(new AJAXError(xhr.statusText, xhr.status, requestParameters.url));
        }
    };
    xhr.send(requestParameters.body);
    return { cancel: () => xhr.abort() };
}
const makeRequest = function (requestParameters, callback) {
    if (!isFileURL(requestParameters.url)) {
        if (self.fetch && self.Request && self.AbortController && Request.prototype.hasOwnProperty('signal')) {
            return makeFetchRequest(requestParameters, callback);
        }
        if (isWorker() && self.worker && self.worker.actor) {
            const queueOnMainThread = true;
            return self.worker.actor.send('getResource', requestParameters, callback, void 0, queueOnMainThread);
        }
    }
    return makeXMLHttpRequest(requestParameters, callback);
};
const getJSON = function (requestParameters, callback) {
    return makeRequest(extend$1(requestParameters, { type: 'json' }), callback);
};
const getArrayBuffer = function (requestParameters, callback) {
    return makeRequest(extend$1(requestParameters, { type: 'arrayBuffer' }), callback);
};
const postData = function (requestParameters, callback) {
    return makeRequest(extend$1(requestParameters, { method: 'POST' }), callback);
};
const getData = function (requestParameters, callback) {
    return makeRequest(extend$1(requestParameters, { method: 'GET' }), callback);
};
function sameOrigin(url) {
    const a = document.createElement('a');
    a.href = url;
    return a.protocol === location.protocol && a.host === location.host;
}
const transparentPngUrl = 'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAAAEAAAABCAYAAAAfFcSJAAAAC0lEQVQYV2NgAAIAAAUAAarVyFEAAAAASUVORK5CYII=';
function arrayBufferToImage(data, callback) {
    const img = new Image();
    img.onload = () => {
        callback(null, img);
        URL.revokeObjectURL(img.src);
        img.onload = null;
        requestAnimationFrame(() => {
            img.src = transparentPngUrl;
        });
    };
    img.onerror = () => callback(new Error('Could not load image. Please make sure to use a supported image type such as PNG or JPEG. Note that SVGs are not supported.'));
    const blob = new Blob([new Uint8Array(data)], { type: 'image/png' });
    img.src = data.byteLength ? URL.createObjectURL(blob) : transparentPngUrl;
}
function arrayBufferToImageBitmap(data, callback) {
    const blob = new Blob([new Uint8Array(data)], { type: 'image/png' });
    createImageBitmap(blob).then(imgBitmap => {
        callback(null, imgBitmap);
    }).catch(e => {
        callback(new Error(`Could not load image because of ${ e.message }. Please make sure to use a supported image type such as PNG or JPEG. Note that SVGs are not supported.`));
    });
}
let imageQueue, numImageRequests;
const resetImageRequestQueue = () => {
    imageQueue = [];
    numImageRequests = 0;
};
resetImageRequestQueue();
const getImage = function (requestParameters, callback) {
    if (exported.supported) {
        if (!requestParameters.headers) {
            requestParameters.headers = {};
        }
        requestParameters.headers.accept = 'image/webp,*/*';
    }
    if (numImageRequests >= config.MAX_PARALLEL_IMAGE_REQUESTS) {
        const queued = {
            requestParameters,
            callback,
            cancelled: false,
            cancel() {
                this.cancelled = true;
            }
        };
        imageQueue.push(queued);
        return queued;
    }
    numImageRequests++;
    let advanced = false;
    const advanceImageRequestQueue = () => {
        if (advanced)
            return;
        advanced = true;
        numImageRequests--;
        while (imageQueue.length && numImageRequests < config.MAX_PARALLEL_IMAGE_REQUESTS) {
            const request2 = imageQueue.shift();
            const {
                requestParameters: requestParameters2,
                callback: callback2,
                cancelled
            } = request2;
            if (!cancelled) {
                request2.cancel = getImage(requestParameters2, callback2).cancel;
            }
        }
    };
    const request = getArrayBuffer(requestParameters, (err, data, cacheControl, expires) => {
        advanceImageRequestQueue();
        if (err) {
            callback(err);
        } else if (data) {
            if (self.createImageBitmap) {
                arrayBufferToImageBitmap(data, (err2, imgBitmap) => callback(err2, imgBitmap, cacheControl, expires));
            } else {
                arrayBufferToImage(data, (err2, img) => callback(err2, img, cacheControl, expires));
            }
        }
    });
    return {
        cancel: () => {
            request.cancel();
            advanceImageRequestQueue();
        }
    };
};
const getVideo = function (urls, callback) {
    const video = document.createElement('video');
    video.muted = true;
    video.onloadstart = function () {
        callback(null, video);
    };
    for (let i = 0; i < urls.length; i++) {
        const s = document.createElement('source');
        if (!sameOrigin(urls[i])) {
            video.crossOrigin = 'Anonymous';
        }
        s.src = urls[i];
        video.appendChild(s);
    }
    return {
        cancel: () => {
        }
    };
};

var murmurhashJs = {exports: {}};

var murmurhash3_gc = {exports: {}};

/**
 * JS Implementation of MurmurHash3 (r136) (as of May 20, 2011)
 * 
 * @author <a href="mailto:gary.court@gmail.com">Gary Court</a>
 * @see http://github.com/garycourt/murmurhash-js
 * @author <a href="mailto:aappleby@gmail.com">Austin Appleby</a>
 * @see http://sites.google.com/site/murmurhash/
 * 
 * @param {string} key ASCII only
 * @param {number} seed Positive integer only
 * @return {number} 32-bit positive integer hash 
 */

var hasRequiredMurmurhash3_gc;

function requireMurmurhash3_gc () {
	if (hasRequiredMurmurhash3_gc) return murmurhash3_gc.exports;
	hasRequiredMurmurhash3_gc = 1;
	(function (module) {
		function murmurhash3_32_gc(key, seed) {
		    var remainder, bytes, h1, h1b, c1, c2, k1, i;
		    remainder = key.length & 3;
		    // key.length % 4
		    bytes = key.length - remainder;
		    h1 = seed;
		    c1 = 3432918353;
		    c2 = 461845907;
		    i = 0;
		    while (i < bytes) {
		        k1 = key.charCodeAt(i) & 255 | (key.charCodeAt(++i) & 255) << 8 | (key.charCodeAt(++i) & 255) << 16 | (key.charCodeAt(++i) & 255) << 24;
		        ++i;
		        k1 = (k1 & 65535) * c1 + (((k1 >>> 16) * c1 & 65535) << 16) & 4294967295;
		        k1 = k1 << 15 | k1 >>> 17;
		        k1 = (k1 & 65535) * c2 + (((k1 >>> 16) * c2 & 65535) << 16) & 4294967295;
		        h1 ^= k1;
		        h1 = h1 << 13 | h1 >>> 19;
		        h1b = (h1 & 65535) * 5 + (((h1 >>> 16) * 5 & 65535) << 16) & 4294967295;
		        h1 = (h1b & 65535) + 27492 + (((h1b >>> 16) + 58964 & 65535) << 16);
		    }
		    k1 = 0;
		    switch (remainder) {
		    case 3:
		        k1 ^= (key.charCodeAt(i + 2) & 255) << 16;
		    case 2:
		        k1 ^= (key.charCodeAt(i + 1) & 255) << 8;
		    case 1:
		        k1 ^= key.charCodeAt(i) & 255;
		        k1 = (k1 & 65535) * c1 + (((k1 >>> 16) * c1 & 65535) << 16) & 4294967295;
		        k1 = k1 << 15 | k1 >>> 17;
		        k1 = (k1 & 65535) * c2 + (((k1 >>> 16) * c2 & 65535) << 16) & 4294967295;
		        h1 ^= k1;
		    }
		    h1 ^= key.length;
		    h1 ^= h1 >>> 16;
		    h1 = (h1 & 65535) * 2246822507 + (((h1 >>> 16) * 2246822507 & 65535) << 16) & 4294967295;
		    h1 ^= h1 >>> 13;
		    h1 = (h1 & 65535) * 3266489909 + (((h1 >>> 16) * 3266489909 & 65535) << 16) & 4294967295;
		    h1 ^= h1 >>> 16;
		    return h1 >>> 0;
		}
		{
		    module.exports = murmurhash3_32_gc;
		} 
	} (murmurhash3_gc));
	return murmurhash3_gc.exports;
}

var murmurhash2_gc = {exports: {}};

/**
 * JS Implementation of MurmurHash2
 * 
 * @author <a href="mailto:gary.court@gmail.com">Gary Court</a>
 * @see http://github.com/garycourt/murmurhash-js
 * @author <a href="mailto:aappleby@gmail.com">Austin Appleby</a>
 * @see http://sites.google.com/site/murmurhash/
 * 
 * @param {string} str ASCII only
 * @param {number} seed Positive integer only
 * @return {number} 32-bit positive integer hash
 */

var hasRequiredMurmurhash2_gc;

function requireMurmurhash2_gc () {
	if (hasRequiredMurmurhash2_gc) return murmurhash2_gc.exports;
	hasRequiredMurmurhash2_gc = 1;
	(function (module) {
		function murmurhash2_32_gc(str, seed) {
		    var l = str.length, h = seed ^ l, i = 0, k;
		    while (l >= 4) {
		        k = str.charCodeAt(i) & 255 | (str.charCodeAt(++i) & 255) << 8 | (str.charCodeAt(++i) & 255) << 16 | (str.charCodeAt(++i) & 255) << 24;
		        k = (k & 65535) * 1540483477 + (((k >>> 16) * 1540483477 & 65535) << 16);
		        k ^= k >>> 24;
		        k = (k & 65535) * 1540483477 + (((k >>> 16) * 1540483477 & 65535) << 16);
		        h = (h & 65535) * 1540483477 + (((h >>> 16) * 1540483477 & 65535) << 16) ^ k;
		        l -= 4;
		        ++i;
		    }
		    switch (l) {
		    case 3:
		        h ^= (str.charCodeAt(i + 2) & 255) << 16;
		    case 2:
		        h ^= (str.charCodeAt(i + 1) & 255) << 8;
		    case 1:
		        h ^= str.charCodeAt(i) & 255;
		        h = (h & 65535) * 1540483477 + (((h >>> 16) * 1540483477 & 65535) << 16);
		    }
		    h ^= h >>> 13;
		    h = (h & 65535) * 1540483477 + (((h >>> 16) * 1540483477 & 65535) << 16);
		    h ^= h >>> 15;
		    return h >>> 0;
		}
		{
		    module.exports = murmurhash2_32_gc;
		} 
	} (murmurhash2_gc));
	return murmurhash2_gc.exports;
}

var hasRequiredMurmurhashJs;

function requireMurmurhashJs () {
	if (hasRequiredMurmurhashJs) return murmurhashJs.exports;
	hasRequiredMurmurhashJs = 1;
	var murmur3 = requireMurmurhash3_gc();
	var murmur2 = requireMurmurhash2_gc();
	murmurhashJs.exports = murmur3;
	murmurhashJs.exports.murmur3 = murmur3;
	murmurhashJs.exports.murmur2 = murmur2;
	return murmurhashJs.exports;
}

var murmurhashJsExports = requireMurmurhashJs();
var murmur3 = /*@__PURE__*/getDefaultExportFromCjs(murmurhashJsExports);

class Event {
    constructor(type, ...eventData) {
        extend$1(this, eventData[0] || {});
        this.type = type;
    }
}
class ErrorEvent extends Event {
    constructor(error, data = {}) {
        super('error', extend$1({ error }, data));
    }
}
function _addEventListener(type, listener, listenerList) {
    const listenerExists = listenerList[type] && listenerList[type].indexOf(listener) !== -1;
    if (!listenerExists) {
        listenerList[type] = listenerList[type] || [];
        listenerList[type].push(listener);
    }
}
function _removeEventListener(type, listener, listenerList) {
    if (listenerList && listenerList[type]) {
        const index = listenerList[type].indexOf(listener);
        if (index !== -1) {
            listenerList[type].splice(index, 1);
        }
    }
}
class Evented {
    /**
   * Adds a listener to a specified event type.
   *
   * @param {string} type The event type to add a listen for.
   * @param {Function} listener The function to be called when the event is fired.
   *   The listener function is called with the data object passed to `fire`,
   *   extended with `target` and `type` properties.
   * @returns {Object} Returns itself to allow for method chaining.
   */
    on(type, listener) {
        this._listeners = this._listeners || {};
        _addEventListener(type, listener, this._listeners);
        return this;
    }
    /**
   * Removes a previously registered event listener.
   *
   * @param {string} type The event type to remove listeners for.
   * @param {Function} listener The listener function to remove.
   * @returns {Object} Returns itself to allow for method chaining.
   */
    off(type, listener) {
        _removeEventListener(type, listener, this._listeners);
        _removeEventListener(type, listener, this._oneTimeListeners);
        return this;
    }
    once(type, listener) {
        if (!listener) {
            return new Promise(resolve => this.once(type, resolve));
        }
        this._oneTimeListeners = this._oneTimeListeners || {};
        _addEventListener(type, listener, this._oneTimeListeners);
        return this;
    }
    fire(e, eventData) {
        const event = typeof e === 'string' ? new Event(e, eventData) : e;
        const type = event.type;
        if (this.listens(type)) {
            event.target = this;
            const listeners = this._listeners && this._listeners[type] ? this._listeners[type].slice() : [];
            for (const listener of listeners) {
                listener.call(this, event);
            }
            const oneTimeListeners = this._oneTimeListeners && this._oneTimeListeners[type] ? this._oneTimeListeners[type].slice() : [];
            for (const listener of oneTimeListeners) {
                _removeEventListener(type, listener, this._oneTimeListeners);
                listener.call(this, event);
            }
            const parent = this._eventedParent;
            if (parent) {
                const eventedParentData = typeof this._eventedParentData === 'function' ? this._eventedParentData() : this._eventedParentData;
                extend$1(event, eventedParentData);
                parent.fire(event);
            }
        } else if (event instanceof ErrorEvent) {
            console.error(event.error);
        }
        return this;
    }
    /**
   * Returns true if this instance of Evented or any forwarded instances of Evented have a listener for the specified type.
   *
   * @param {string} type The event type.
   * @returns {boolean} Returns `true` if there is at least one registered listener for specified event type, `false` otherwise.
   * @private
   */
    listens(type) {
        return !!(this._listeners && this._listeners[type] && this._listeners[type].length > 0 || this._oneTimeListeners && this._oneTimeListeners[type] && this._oneTimeListeners[type].length > 0 || this._eventedParent && this._eventedParent.listens(type));
    }
    /**
   * Bubble all events fired by this instance of Evented to this parent instance of Evented.
   *
   * @returns {Object} `this`
   * @private
   */
    setEventedParent(parent, data) {
        this._eventedParent = parent;
        this._eventedParentData = data;
        return this;
    }
}

var csscolorparser = {};

var hasRequiredCsscolorparser;

function requireCsscolorparser () {
	if (hasRequiredCsscolorparser) return csscolorparser;
	hasRequiredCsscolorparser = 1;
	// (c) Dean McNamee <dean@gmail.com>, 2012.
	//
	// https://github.com/deanm/css-color-parser-js
	//
	// Permission is hereby granted, free of charge, to any person obtaining a copy
	// of this software and associated documentation files (the "Software"), to
	// deal in the Software without restriction, including without limitation the
	// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
	// sell copies of the Software, and to permit persons to whom the Software is
	// furnished to do so, subject to the following conditions:
	//
	// The above copyright notice and this permission notice shall be included in
	// all copies or substantial portions of the Software.
	//
	// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
	// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
	// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
	// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
	// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
	// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
	// IN THE SOFTWARE.
	// http://www.w3.org/TR/css3-color/
	var kCSSColorTable = {
	    'transparent': [
	        0,
	        0,
	        0,
	        0
	    ],
	    'aliceblue': [
	        240,
	        248,
	        255,
	        1
	    ],
	    'antiquewhite': [
	        250,
	        235,
	        215,
	        1
	    ],
	    'aqua': [
	        0,
	        255,
	        255,
	        1
	    ],
	    'aquamarine': [
	        127,
	        255,
	        212,
	        1
	    ],
	    'azure': [
	        240,
	        255,
	        255,
	        1
	    ],
	    'beige': [
	        245,
	        245,
	        220,
	        1
	    ],
	    'bisque': [
	        255,
	        228,
	        196,
	        1
	    ],
	    'black': [
	        0,
	        0,
	        0,
	        1
	    ],
	    'blanchedalmond': [
	        255,
	        235,
	        205,
	        1
	    ],
	    'blue': [
	        0,
	        0,
	        255,
	        1
	    ],
	    'blueviolet': [
	        138,
	        43,
	        226,
	        1
	    ],
	    'brown': [
	        165,
	        42,
	        42,
	        1
	    ],
	    'burlywood': [
	        222,
	        184,
	        135,
	        1
	    ],
	    'cadetblue': [
	        95,
	        158,
	        160,
	        1
	    ],
	    'chartreuse': [
	        127,
	        255,
	        0,
	        1
	    ],
	    'chocolate': [
	        210,
	        105,
	        30,
	        1
	    ],
	    'coral': [
	        255,
	        127,
	        80,
	        1
	    ],
	    'cornflowerblue': [
	        100,
	        149,
	        237,
	        1
	    ],
	    'cornsilk': [
	        255,
	        248,
	        220,
	        1
	    ],
	    'crimson': [
	        220,
	        20,
	        60,
	        1
	    ],
	    'cyan': [
	        0,
	        255,
	        255,
	        1
	    ],
	    'darkblue': [
	        0,
	        0,
	        139,
	        1
	    ],
	    'darkcyan': [
	        0,
	        139,
	        139,
	        1
	    ],
	    'darkgoldenrod': [
	        184,
	        134,
	        11,
	        1
	    ],
	    'darkgray': [
	        169,
	        169,
	        169,
	        1
	    ],
	    'darkgreen': [
	        0,
	        100,
	        0,
	        1
	    ],
	    'darkgrey': [
	        169,
	        169,
	        169,
	        1
	    ],
	    'darkkhaki': [
	        189,
	        183,
	        107,
	        1
	    ],
	    'darkmagenta': [
	        139,
	        0,
	        139,
	        1
	    ],
	    'darkolivegreen': [
	        85,
	        107,
	        47,
	        1
	    ],
	    'darkorange': [
	        255,
	        140,
	        0,
	        1
	    ],
	    'darkorchid': [
	        153,
	        50,
	        204,
	        1
	    ],
	    'darkred': [
	        139,
	        0,
	        0,
	        1
	    ],
	    'darksalmon': [
	        233,
	        150,
	        122,
	        1
	    ],
	    'darkseagreen': [
	        143,
	        188,
	        143,
	        1
	    ],
	    'darkslateblue': [
	        72,
	        61,
	        139,
	        1
	    ],
	    'darkslategray': [
	        47,
	        79,
	        79,
	        1
	    ],
	    'darkslategrey': [
	        47,
	        79,
	        79,
	        1
	    ],
	    'darkturquoise': [
	        0,
	        206,
	        209,
	        1
	    ],
	    'darkviolet': [
	        148,
	        0,
	        211,
	        1
	    ],
	    'deeppink': [
	        255,
	        20,
	        147,
	        1
	    ],
	    'deepskyblue': [
	        0,
	        191,
	        255,
	        1
	    ],
	    'dimgray': [
	        105,
	        105,
	        105,
	        1
	    ],
	    'dimgrey': [
	        105,
	        105,
	        105,
	        1
	    ],
	    'dodgerblue': [
	        30,
	        144,
	        255,
	        1
	    ],
	    'firebrick': [
	        178,
	        34,
	        34,
	        1
	    ],
	    'floralwhite': [
	        255,
	        250,
	        240,
	        1
	    ],
	    'forestgreen': [
	        34,
	        139,
	        34,
	        1
	    ],
	    'fuchsia': [
	        255,
	        0,
	        255,
	        1
	    ],
	    'gainsboro': [
	        220,
	        220,
	        220,
	        1
	    ],
	    'ghostwhite': [
	        248,
	        248,
	        255,
	        1
	    ],
	    'gold': [
	        255,
	        215,
	        0,
	        1
	    ],
	    'goldenrod': [
	        218,
	        165,
	        32,
	        1
	    ],
	    'gray': [
	        128,
	        128,
	        128,
	        1
	    ],
	    'green': [
	        0,
	        128,
	        0,
	        1
	    ],
	    'greenyellow': [
	        173,
	        255,
	        47,
	        1
	    ],
	    'grey': [
	        128,
	        128,
	        128,
	        1
	    ],
	    'honeydew': [
	        240,
	        255,
	        240,
	        1
	    ],
	    'hotpink': [
	        255,
	        105,
	        180,
	        1
	    ],
	    'indianred': [
	        205,
	        92,
	        92,
	        1
	    ],
	    'indigo': [
	        75,
	        0,
	        130,
	        1
	    ],
	    'ivory': [
	        255,
	        255,
	        240,
	        1
	    ],
	    'khaki': [
	        240,
	        230,
	        140,
	        1
	    ],
	    'lavender': [
	        230,
	        230,
	        250,
	        1
	    ],
	    'lavenderblush': [
	        255,
	        240,
	        245,
	        1
	    ],
	    'lawngreen': [
	        124,
	        252,
	        0,
	        1
	    ],
	    'lemonchiffon': [
	        255,
	        250,
	        205,
	        1
	    ],
	    'lightblue': [
	        173,
	        216,
	        230,
	        1
	    ],
	    'lightcoral': [
	        240,
	        128,
	        128,
	        1
	    ],
	    'lightcyan': [
	        224,
	        255,
	        255,
	        1
	    ],
	    'lightgoldenrodyellow': [
	        250,
	        250,
	        210,
	        1
	    ],
	    'lightgray': [
	        211,
	        211,
	        211,
	        1
	    ],
	    'lightgreen': [
	        144,
	        238,
	        144,
	        1
	    ],
	    'lightgrey': [
	        211,
	        211,
	        211,
	        1
	    ],
	    'lightpink': [
	        255,
	        182,
	        193,
	        1
	    ],
	    'lightsalmon': [
	        255,
	        160,
	        122,
	        1
	    ],
	    'lightseagreen': [
	        32,
	        178,
	        170,
	        1
	    ],
	    'lightskyblue': [
	        135,
	        206,
	        250,
	        1
	    ],
	    'lightslategray': [
	        119,
	        136,
	        153,
	        1
	    ],
	    'lightslategrey': [
	        119,
	        136,
	        153,
	        1
	    ],
	    'lightsteelblue': [
	        176,
	        196,
	        222,
	        1
	    ],
	    'lightyellow': [
	        255,
	        255,
	        224,
	        1
	    ],
	    'lime': [
	        0,
	        255,
	        0,
	        1
	    ],
	    'limegreen': [
	        50,
	        205,
	        50,
	        1
	    ],
	    'linen': [
	        250,
	        240,
	        230,
	        1
	    ],
	    'magenta': [
	        255,
	        0,
	        255,
	        1
	    ],
	    'maroon': [
	        128,
	        0,
	        0,
	        1
	    ],
	    'mediumaquamarine': [
	        102,
	        205,
	        170,
	        1
	    ],
	    'mediumblue': [
	        0,
	        0,
	        205,
	        1
	    ],
	    'mediumorchid': [
	        186,
	        85,
	        211,
	        1
	    ],
	    'mediumpurple': [
	        147,
	        112,
	        219,
	        1
	    ],
	    'mediumseagreen': [
	        60,
	        179,
	        113,
	        1
	    ],
	    'mediumslateblue': [
	        123,
	        104,
	        238,
	        1
	    ],
	    'mediumspringgreen': [
	        0,
	        250,
	        154,
	        1
	    ],
	    'mediumturquoise': [
	        72,
	        209,
	        204,
	        1
	    ],
	    'mediumvioletred': [
	        199,
	        21,
	        133,
	        1
	    ],
	    'midnightblue': [
	        25,
	        25,
	        112,
	        1
	    ],
	    'mintcream': [
	        245,
	        255,
	        250,
	        1
	    ],
	    'mistyrose': [
	        255,
	        228,
	        225,
	        1
	    ],
	    'moccasin': [
	        255,
	        228,
	        181,
	        1
	    ],
	    'navajowhite': [
	        255,
	        222,
	        173,
	        1
	    ],
	    'navy': [
	        0,
	        0,
	        128,
	        1
	    ],
	    'oldlace': [
	        253,
	        245,
	        230,
	        1
	    ],
	    'olive': [
	        128,
	        128,
	        0,
	        1
	    ],
	    'olivedrab': [
	        107,
	        142,
	        35,
	        1
	    ],
	    'orange': [
	        255,
	        165,
	        0,
	        1
	    ],
	    'orangered': [
	        255,
	        69,
	        0,
	        1
	    ],
	    'orchid': [
	        218,
	        112,
	        214,
	        1
	    ],
	    'palegoldenrod': [
	        238,
	        232,
	        170,
	        1
	    ],
	    'palegreen': [
	        152,
	        251,
	        152,
	        1
	    ],
	    'paleturquoise': [
	        175,
	        238,
	        238,
	        1
	    ],
	    'palevioletred': [
	        219,
	        112,
	        147,
	        1
	    ],
	    'papayawhip': [
	        255,
	        239,
	        213,
	        1
	    ],
	    'peachpuff': [
	        255,
	        218,
	        185,
	        1
	    ],
	    'peru': [
	        205,
	        133,
	        63,
	        1
	    ],
	    'pink': [
	        255,
	        192,
	        203,
	        1
	    ],
	    'plum': [
	        221,
	        160,
	        221,
	        1
	    ],
	    'powderblue': [
	        176,
	        224,
	        230,
	        1
	    ],
	    'purple': [
	        128,
	        0,
	        128,
	        1
	    ],
	    'rebeccapurple': [
	        102,
	        51,
	        153,
	        1
	    ],
	    'red': [
	        255,
	        0,
	        0,
	        1
	    ],
	    'rosybrown': [
	        188,
	        143,
	        143,
	        1
	    ],
	    'royalblue': [
	        65,
	        105,
	        225,
	        1
	    ],
	    'saddlebrown': [
	        139,
	        69,
	        19,
	        1
	    ],
	    'salmon': [
	        250,
	        128,
	        114,
	        1
	    ],
	    'sandybrown': [
	        244,
	        164,
	        96,
	        1
	    ],
	    'seagreen': [
	        46,
	        139,
	        87,
	        1
	    ],
	    'seashell': [
	        255,
	        245,
	        238,
	        1
	    ],
	    'sienna': [
	        160,
	        82,
	        45,
	        1
	    ],
	    'silver': [
	        192,
	        192,
	        192,
	        1
	    ],
	    'skyblue': [
	        135,
	        206,
	        235,
	        1
	    ],
	    'slateblue': [
	        106,
	        90,
	        205,
	        1
	    ],
	    'slategray': [
	        112,
	        128,
	        144,
	        1
	    ],
	    'slategrey': [
	        112,
	        128,
	        144,
	        1
	    ],
	    'snow': [
	        255,
	        250,
	        250,
	        1
	    ],
	    'springgreen': [
	        0,
	        255,
	        127,
	        1
	    ],
	    'steelblue': [
	        70,
	        130,
	        180,
	        1
	    ],
	    'tan': [
	        210,
	        180,
	        140,
	        1
	    ],
	    'teal': [
	        0,
	        128,
	        128,
	        1
	    ],
	    'thistle': [
	        216,
	        191,
	        216,
	        1
	    ],
	    'tomato': [
	        255,
	        99,
	        71,
	        1
	    ],
	    'turquoise': [
	        64,
	        224,
	        208,
	        1
	    ],
	    'violet': [
	        238,
	        130,
	        238,
	        1
	    ],
	    'wheat': [
	        245,
	        222,
	        179,
	        1
	    ],
	    'white': [
	        255,
	        255,
	        255,
	        1
	    ],
	    'whitesmoke': [
	        245,
	        245,
	        245,
	        1
	    ],
	    'yellow': [
	        255,
	        255,
	        0,
	        1
	    ],
	    'yellowgreen': [
	        154,
	        205,
	        50,
	        1
	    ]
	};
	function clamp_css_byte(i) {
	    // Clamp to integer 0 .. 255.
	    i = Math.round(i);
	    // Seems to be what Chrome does (vs truncation).
	    return i < 0 ? 0 : i > 255 ? 255 : i;
	}
	function clamp_css_float(f) {
	    // Clamp to float 0.0 .. 1.0.
	    return f < 0 ? 0 : f > 1 ? 1 : f;
	}
	function parse_css_int(str) {
	    // int or percentage.
	    if (str[str.length - 1] === '%')
	        return clamp_css_byte(parseFloat(str) / 100 * 255);
	    return clamp_css_byte(parseInt(str));
	}
	function parse_css_float(str) {
	    // float or percentage.
	    if (str[str.length - 1] === '%')
	        return clamp_css_float(parseFloat(str) / 100);
	    return clamp_css_float(parseFloat(str));
	}
	function css_hue_to_rgb(m1, m2, h) {
	    if (h < 0)
	        h += 1;
	    else if (h > 1)
	        h -= 1;
	    if (h * 6 < 1)
	        return m1 + (m2 - m1) * h * 6;
	    if (h * 2 < 1)
	        return m2;
	    if (h * 3 < 2)
	        return m1 + (m2 - m1) * (2 / 3 - h) * 6;
	    return m1;
	}
	function parseCSSColor(css_str) {
	    // Remove all whitespace, not compliant, but should just be more accepting.
	    var str = css_str.replace(/ /g, '').toLowerCase();
	    // Color keywords (and transparent) lookup.
	    if (str in kCSSColorTable)
	        return kCSSColorTable[str].slice();
	    // dup.
	    // #abc and #abc123 syntax.
	    if (str[0] === '#') {
	        if (str.length === 4) {
	            var iv = parseInt(str.substr(1), 16);
	            // TODO(deanm): Stricter parsing.
	            if (!(iv >= 0 && iv <= 4095))
	                return null;
	            // Covers NaN.
	            return [
	                (iv & 3840) >> 4 | (iv & 3840) >> 8,
	                iv & 240 | (iv & 240) >> 4,
	                iv & 15 | (iv & 15) << 4,
	                1
	            ];
	        } else if (str.length === 7) {
	            var iv = parseInt(str.substr(1), 16);
	            // TODO(deanm): Stricter parsing.
	            if (!(iv >= 0 && iv <= 16777215))
	                return null;
	            // Covers NaN.
	            return [
	                (iv & 16711680) >> 16,
	                (iv & 65280) >> 8,
	                iv & 255,
	                1
	            ];
	        }
	        return null;
	    }
	    var op = str.indexOf('('), ep = str.indexOf(')');
	    if (op !== -1 && ep + 1 === str.length) {
	        var fname = str.substr(0, op);
	        var params = str.substr(op + 1, ep - (op + 1)).split(',');
	        var alpha = 1;
	        // To allow case fallthrough.
	        switch (fname) {
	        case 'rgba':
	            if (params.length !== 4)
	                return null;
	            alpha = parse_css_float(params.pop());
	        // Fall through.
	        case 'rgb':
	            if (params.length !== 3)
	                return null;
	            return [
	                parse_css_int(params[0]),
	                parse_css_int(params[1]),
	                parse_css_int(params[2]),
	                alpha
	            ];
	        case 'hsla':
	            if (params.length !== 4)
	                return null;
	            alpha = parse_css_float(params.pop());
	        // Fall through.
	        case 'hsl':
	            if (params.length !== 3)
	                return null;
	            var h = (parseFloat(params[0]) % 360 + 360) % 360 / 360;
	            // 0 .. 1
	            // NOTE(deanm): According to the CSS spec s/l should only be
	            // percentages, but we don't bother and let float or percentage.
	            var s = parse_css_float(params[1]);
	            var l = parse_css_float(params[2]);
	            var m2 = l <= 0.5 ? l * (s + 1) : l + s - l * s;
	            var m1 = l * 2 - m2;
	            return [
	                clamp_css_byte(css_hue_to_rgb(m1, m2, h + 1 / 3) * 255),
	                clamp_css_byte(css_hue_to_rgb(m1, m2, h) * 255),
	                clamp_css_byte(css_hue_to_rgb(m1, m2, h - 1 / 3) * 255),
	                alpha
	            ];
	        default:
	            return null;
	        }
	    }
	    return null;
	}
	try {
	    csscolorparser.parseCSSColor = parseCSSColor;
	} catch (e) {
	}
	return csscolorparser;
}

var csscolorparserExports = requireCsscolorparser();

class Color {
    constructor(r, g, b, a = 1) {
        this.r = r;
        this.g = g;
        this.b = b;
        this.a = a;
    }
    /**
   * Parses valid CSS color strings and returns a `Color` instance.
   * @returns A `Color` instance, or `undefined` if the input is not a valid color string.
   */
    static parse(input) {
        if (!input) {
            return void 0;
        }
        if (input instanceof Color) {
            return input;
        }
        if (typeof input !== 'string') {
            return void 0;
        }
        const rgba = csscolorparserExports.parseCSSColor(input);
        if (!rgba) {
            return void 0;
        }
        return new Color(rgba[0] / 255 * rgba[3], rgba[1] / 255 * rgba[3], rgba[2] / 255 * rgba[3], rgba[3]);
    }
    /**
   * Returns an RGBA string representing the color value.
   *
   * @returns An RGBA string.
   * @example
   * var purple = new Color.parse('purple');
   * purple.toString; // = "rgba(128,0,128,1)"
   * var translucentGreen = new Color.parse('rgba(26, 207, 26, .73)');
   * translucentGreen.toString(); // = "rgba(26,207,26,0.73)"
   */
    toStringPremultipliedAlpha() {
        const [r, g, b, a] = this.a === 0 ? [
            0,
            0,
            0,
            0
        ] : [
            this.r * 255 / this.a,
            this.g * 255 / this.a,
            this.b * 255 / this.a,
            this.a
        ];
        return `rgba(${ Math.round(r) },${ Math.round(g) },${ Math.round(b) },${ a })`;
    }
    toString() {
        const [r, g, b, a] = [
            this.r,
            this.g,
            this.b,
            this.a
        ];
        return `rgba(${ Math.round(r * 255) },${ Math.round(g * 255) },${ Math.round(b * 255) },${ a })`;
    }
    toRenderColor(lut) {
        const {r, g, b, a} = this;
        return new RenderColor(lut, r, g, b, a);
    }
    clone() {
        return new Color(this.r, this.g, this.b, this.a);
    }
}
class RenderColor {
    constructor(lut, r, g, b, a) {
        if (!lut) {
            this.r = r;
            this.g = g;
            this.b = b;
            this.a = a;
        } else {
            const N = lut.image.height;
            const N2 = N * N;
            r = a === 0 ? 0 : r / a * (N - 1);
            g = a === 0 ? 0 : g / a * (N - 1);
            b = a === 0 ? 0 : b / a * (N - 1);
            const r0 = Math.floor(r);
            const g0 = Math.floor(g);
            const b0 = Math.floor(b);
            const r1 = Math.ceil(r);
            const g1 = Math.ceil(g);
            const b1 = Math.ceil(b);
            const rw = r - r0;
            const gw = g - g0;
            const bw = b - b0;
            const data = lut.image.data;
            const i0 = (r0 + g0 * N2 + b0 * N) * 4;
            const i1 = (r0 + g0 * N2 + b1 * N) * 4;
            const i2 = (r0 + g1 * N2 + b0 * N) * 4;
            const i3 = (r0 + g1 * N2 + b1 * N) * 4;
            const i4 = (r1 + g0 * N2 + b0 * N) * 4;
            const i5 = (r1 + g0 * N2 + b1 * N) * 4;
            const i6 = (r1 + g1 * N2 + b0 * N) * 4;
            const i7 = (r1 + g1 * N2 + b1 * N) * 4;
            if (i0 < 0 || i7 >= data.length) {
                throw new Error('out of range');
            }
            this.r = number(number(number(data[i0], data[i1], bw), number(data[i2], data[i3], bw), gw), number(number(data[i4], data[i5], bw), number(data[i6], data[i7], bw), gw), rw) / 255 * a;
            this.g = number(number(number(data[i0 + 1], data[i1 + 1], bw), number(data[i2 + 1], data[i3 + 1], bw), gw), number(number(data[i4 + 1], data[i5 + 1], bw), number(data[i6 + 1], data[i7 + 1], bw), gw), rw) / 255 * a;
            this.b = number(number(number(data[i0 + 2], data[i1 + 2], bw), number(data[i2 + 2], data[i3 + 2], bw), gw), number(number(data[i4 + 2], data[i5 + 2], bw), number(data[i6 + 2], data[i7 + 2], bw), gw), rw) / 255 * a;
            this.a = a;
        }
    }
    /**
   * Returns an RGBA array of values representing the color, unpremultiplied by A.
   *
   * @returns An array of RGBA color values in the range [0, 255].
   */
    toArray() {
        const {r, g, b, a} = this;
        return a === 0 ? [
            0,
            0,
            0,
            0
        ] : [
            r * 255 / a,
            g * 255 / a,
            b * 255 / a,
            a
        ];
    }
    /**
   * Returns an HSLA array of values representing the color, unpremultiplied by A.
   *
   * @returns An array of HSLA color values.
   */
    toHslaArray() {
        if (this.a === 0) {
            return [
                0,
                0,
                0,
                0
            ];
        }
        const {r, g, b, a} = this;
        const red = Math.min(Math.max(r / a, 0), 1);
        const green = Math.min(Math.max(g / a, 0), 1);
        const blue = Math.min(Math.max(b / a, 0), 1);
        const min = Math.min(red, green, blue);
        const max = Math.max(red, green, blue);
        const l = (min + max) / 2;
        if (min === max) {
            return [
                0,
                0,
                l * 100,
                a
            ];
        }
        const delta = max - min;
        const s = l > 0.5 ? delta / (2 - max - min) : delta / (max + min);
        let h = 0;
        if (max === red) {
            h = (green - blue) / delta + (green < blue ? 6 : 0);
        } else if (max === green) {
            h = (blue - red) / delta + 2;
        } else if (max === blue) {
            h = (red - green) / delta + 4;
        }
        h *= 60;
        return [
            Math.min(Math.max(h, 0), 360),
            Math.min(Math.max(s * 100, 0), 100),
            Math.min(Math.max(l * 100, 0), 100),
            a
        ];
    }
    /**
   * Returns a RGBA array of float values representing the color, unpremultiplied by A.
   *
   * @returns An array of RGBA color values in the range [0, 1].
   */
    toArray01() {
        const {r, g, b, a} = this;
        return a === 0 ? [
            0,
            0,
            0,
            0
        ] : [
            r / a,
            g / a,
            b / a,
            a
        ];
    }
    /**
   * Returns an RGB array of values representing the color, unpremultiplied by A and multiplied by a scalar.
   *
   * @param {number} scale A scale to apply to the unpremultiplied-alpha values.
   * @returns An array of RGB color values in the range [0, 1].
   */
    toArray01Scaled(scale) {
        const {r, g, b, a} = this;
        return a === 0 ? [
            0,
            0,
            0
        ] : [
            r / a * scale,
            g / a * scale,
            b / a * scale
        ];
    }
    /**
   * Returns an RGBA array of values representing the color, premultiplied by A.
   *
   * @returns An array of RGBA color values in the range [0, 1].
   */
    toArray01PremultipliedAlpha() {
        const {r, g, b, a} = this;
        return [
            r,
            g,
            b,
            a
        ];
    }
    /**
   * Returns an RGBA array of values representing the color, unpremultiplied by A, and converted to linear color space.
   * The color is defined by sRGB primaries, but the sRGB transfer function is reversed to obtain linear energy.
   *
   * @returns An array of RGBA color values in the range [0, 1].
   */
    toArray01Linear() {
        const {r, g, b, a} = this;
        return a === 0 ? [
            0,
            0,
            0,
            0
        ] : [
            Math.pow(r / a, 2.2),
            Math.pow(g / a, 2.2),
            Math.pow(b / a, 2.2),
            a
        ];
    }
}
Color.black = new Color(0, 0, 0, 1);
Color.white = new Color(1, 1, 1, 1);
Color.transparent = new Color(0, 0, 0, 0);
Color.red = new Color(1, 0, 0, 1);
Color.blue = new Color(0, 0, 1, 1);

function number(a, b, t) {
    return a * (1 - t) + b * t;
}
function color(from, to, t) {
    return new Color(number(from.r, to.r, t), number(from.g, to.g, t), number(from.b, to.b, t), number(from.a, to.a, t));
}
function array$1(from, to, t) {
    return from.map((d, i) => {
        return number(d, to[i], t);
    });
}

var interpolate$1 = /*#__PURE__*/Object.freeze({
__proto__: null,
array: array$1,
color: color,
number: number
});

function extend (output, ...inputs) {
    for (const input of inputs) {
        for (const k in input) {
            output[k] = input[k];
        }
    }
    return output;
}

class ParsingError extends Error {
    constructor(key, message) {
        super(message);
        this.message = message;
        this.key = key;
    }
}

class Scope {
    constructor(parent, bindings = []) {
        this.parent = parent;
        this.bindings = {};
        for (const [name, expression] of bindings) {
            this.bindings[name] = expression;
        }
    }
    concat(bindings) {
        return new Scope(this, bindings);
    }
    get(name) {
        if (this.bindings[name]) {
            return this.bindings[name];
        }
        if (this.parent) {
            return this.parent.get(name);
        }
        throw new Error(`${ name } not found in scope.`);
    }
    has(name) {
        if (this.bindings[name])
            return true;
        return this.parent ? this.parent.has(name) : false;
    }
}

const NullType = { kind: 'null' };
const NumberType = { kind: 'number' };
const StringType = { kind: 'string' };
const BooleanType = { kind: 'boolean' };
const ColorType = { kind: 'color' };
const ObjectType = { kind: 'object' };
const ValueType = { kind: 'value' };
const ErrorType = { kind: 'error' };
const CollatorType = { kind: 'collator' };
const FormattedType = { kind: 'formatted' };
const ResolvedImageType = { kind: 'resolvedImage' };
function array(itemType, N) {
    return {
        kind: 'array',
        itemType,
        N
    };
}
function toString$1(type) {
    if (type.kind === 'array') {
        const itemType = toString$1(type.itemType);
        return typeof type.N === 'number' ? `array<${ itemType }, ${ type.N }>` : type.itemType.kind === 'value' ? 'array' : `array<${ itemType }>`;
    } else {
        return type.kind;
    }
}
const valueMemberTypes = [
    NullType,
    NumberType,
    StringType,
    BooleanType,
    ColorType,
    FormattedType,
    ObjectType,
    array(ValueType),
    ResolvedImageType
];
function checkSubtype(expected, t) {
    if (t.kind === 'error') {
        return null;
    } else if (expected.kind === 'array') {
        if (t.kind === 'array' && (t.N === 0 && t.itemType.kind === 'value' || !checkSubtype(expected.itemType, t.itemType)) && (typeof expected.N !== 'number' || expected.N === t.N)) {
            return null;
        }
    } else if (expected.kind === t.kind) {
        return null;
    } else if (expected.kind === 'value') {
        for (const memberType of valueMemberTypes) {
            if (!checkSubtype(memberType, t)) {
                return null;
            }
        }
    }
    return `Expected ${ toString$1(expected) } but found ${ toString$1(t) } instead.`;
}
function isValidType(provided, allowedTypes) {
    return allowedTypes.some(t => t.kind === provided.kind);
}
function isValidNativeType(provided, allowedTypes) {
    return allowedTypes.some(t => {
        if (t === 'null') {
            return provided === null;
        } else if (t === 'array') {
            return Array.isArray(provided);
        } else if (t === 'object') {
            return provided && !Array.isArray(provided) && typeof provided === 'object';
        } else {
            return t === typeof provided;
        }
    });
}

class Collator {
    constructor(caseSensitive, diacriticSensitive, locale) {
        if (caseSensitive)
            this.sensitivity = diacriticSensitive ? 'variant' : 'case';
        else
            this.sensitivity = diacriticSensitive ? 'accent' : 'base';
        this.locale = locale;
        this.collator = new Intl.Collator(this.locale ? this.locale : [], {
            sensitivity: this.sensitivity,
            usage: 'search'
        });
    }
    compare(lhs, rhs) {
        return this.collator.compare(lhs, rhs);
    }
    resolvedLocale() {
        return new Intl.Collator(this.locale ? this.locale : []).resolvedOptions().locale;
    }
}

class FormattedSection {
    constructor(text, image, scale, fontStack, textColor) {
        this.text = text.normalize ? text.normalize() : text;
        this.image = image;
        this.scale = scale;
        this.fontStack = fontStack;
        this.textColor = textColor;
    }
}
class Formatted {
    constructor(sections) {
        this.sections = sections;
    }
    static fromString(unformatted) {
        return new Formatted([new FormattedSection(unformatted, null, null, null, null)]);
    }
    isEmpty() {
        if (this.sections.length === 0)
            return true;
        return !this.sections.some(section => section.text.length !== 0 || section.image && section.image.namePrimary);
    }
    static factory(text) {
        if (text instanceof Formatted) {
            return text;
        } else {
            return Formatted.fromString(text);
        }
    }
    toString() {
        if (this.sections.length === 0)
            return '';
        return this.sections.map(section => section.text).join('');
    }
    serialize() {
        const serialized = ['format'];
        for (const section of this.sections) {
            if (section.image) {
                serialized.push([
                    'image',
                    section.image.namePrimary
                ]);
                continue;
            }
            serialized.push(section.text);
            const options = {};
            if (section.fontStack) {
                options['text-font'] = [
                    'literal',
                    section.fontStack.split(',')
                ];
            }
            if (section.scale) {
                options['font-scale'] = section.scale;
            }
            if (section.textColor) {
                options['text-color'] = ['rgba'].concat(section.textColor.toRenderColor(null).toArray());
            }
            serialized.push(options);
        }
        return serialized;
    }
}

class ImageIdWithOptions {
    constructor(id, options) {
        this.id = id;
        this.options = options || { params: {} };
        if (!this.options.transform) {
            this.options.transform = new DOMMatrix([
                1,
                0,
                0,
                1,
                0,
                0
            ]);
        } else {
            const {a, b, c, d, e, f} = this.options.transform;
            this.options.transform = new DOMMatrix([
                a,
                b,
                c,
                d,
                e,
                f
            ]);
        }
    }
    static deserializeId(serialized) {
        return JSON.parse(serialized).id;
    }
    static deserializeFromString(serialized) {
        const deserializedObject = JSON.parse(serialized);
        ({ params: deserializedObject.options.params });
        const {a, b, c, d, e, f} = deserializedObject.options.transform;
        new DOMMatrix([
            a,
            b,
            c,
            d,
            e,
            f
        ]);
        return new ImageIdWithOptions(deserializedObject.id, deserializedObject.options);
    }
    scaleSelf(factor) {
        this.options.transform = this.options.transform.scale(factor);
        return this;
    }
    serialize() {
        const serialisedObject = { id: this.id };
        if (this.options) {
            serialisedObject.options = this.options;
        }
        const {a, b, c, d, e, f} = this.options.transform;
        serialisedObject.options.transform = {
            a,
            b,
            c,
            d,
            e,
            f
        };
        return JSON.stringify(serialisedObject);
    }
}

class ResolvedImage {
    constructor(options) {
        this.namePrimary = options.namePrimary;
        if (options.nameSecondary) {
            this.nameSecondary = options.nameSecondary;
        }
        if (options.optionsPrimary) {
            this.optionsPrimary = options.optionsPrimary;
        }
        if (options.optionsSecondary) {
            this.optionsSecondary = options.optionsSecondary;
        }
        this.available = options.available;
    }
    toString() {
        if (this.namePrimary && this.nameSecondary) {
            return `[${ this.namePrimary },${ this.nameSecondary }]`;
        }
        return this.namePrimary;
    }
    getPrimary() {
        return new ImageIdWithOptions(this.namePrimary, { params: this.optionsPrimary ? this.optionsPrimary.params || {} : {} });
    }
    getSerializedPrimary() {
        return this.getPrimary().serialize();
    }
    getSecondary() {
        if (this.nameSecondary) {
            return new ImageIdWithOptions(this.nameSecondary, { params: this.optionsSecondary ? this.optionsSecondary.params || {} : {} });
        }
        return null;
    }
    static from(image) {
        return typeof image === 'string' ? ResolvedImage.build(image) : image;
    }
    static build(namePrimary, nameSecondary, optionsPrimary, optionsSecondary) {
        if (!namePrimary)
            return null;
        return new ResolvedImage({
            namePrimary,
            nameSecondary,
            optionsPrimary,
            optionsSecondary,
            available: false
        });
    }
}

function validateRGBA(r, g, b, a) {
    if (!(typeof r === 'number' && r >= 0 && r <= 255 && typeof g === 'number' && g >= 0 && g <= 255 && typeof b === 'number' && b >= 0 && b <= 255)) {
        const value = typeof a === 'number' ? [
            r,
            g,
            b,
            a
        ] : [
            r,
            g,
            b
        ];
        return `Invalid rgba value [${ value.join(', ') }]: 'r', 'g', and 'b' must be between 0 and 255.`;
    }
    if (!(typeof a === 'undefined' || typeof a === 'number' && a >= 0 && a <= 1)) {
        return `Invalid rgba value [${ [
            r,
            g,
            b,
            a
        ].join(', ') }]: 'a' must be between 0 and 1.`;
    }
    return null;
}
function validateHSLA(h, s, l, a) {
    if (!(typeof h === 'number' && h >= 0 && h <= 360)) {
        const value = typeof a === 'number' ? [
            h,
            s,
            l,
            a
        ] : [
            h,
            s,
            l
        ];
        return `Invalid hsla value [${ value.join(', ') }]: 'h' must be between 0 and 360.`;
    }
    if (!(typeof s === 'number' && s >= 0 && s <= 100 && typeof l === 'number' && l >= 0 && l <= 100)) {
        const value = typeof a === 'number' ? [
            h,
            s,
            l,
            a
        ] : [
            h,
            s,
            l
        ];
        return `Invalid hsla value [${ value.join(', ') }]: 's', and 'l' must be between 0 and 100.`;
    }
    if (!(typeof a === 'undefined' || typeof a === 'number' && a >= 0 && a <= 1)) {
        return `Invalid hsla value [${ [
            h,
            s,
            l,
            a
        ].join(', ') }]: 'a' must be between 0 and 1.`;
    }
    return null;
}
function isValue(mixed) {
    if (mixed === null) {
        return true;
    } else if (typeof mixed === 'string') {
        return true;
    } else if (typeof mixed === 'boolean') {
        return true;
    } else if (typeof mixed === 'number') {
        return true;
    } else if (mixed instanceof Color) {
        return true;
    } else if (mixed instanceof Collator) {
        return true;
    } else if (mixed instanceof Formatted) {
        return true;
    } else if (mixed instanceof ResolvedImage) {
        return true;
    } else if (Array.isArray(mixed)) {
        for (const item of mixed) {
            if (!isValue(item)) {
                return false;
            }
        }
        return true;
    } else if (typeof mixed === 'object') {
        for (const key in mixed) {
            if (!isValue(mixed[key])) {
                return false;
            }
        }
        return true;
    } else {
        return false;
    }
}
function typeOf(value) {
    if (value === null) {
        return NullType;
    } else if (typeof value === 'string') {
        return StringType;
    } else if (typeof value === 'boolean') {
        return BooleanType;
    } else if (typeof value === 'number') {
        return NumberType;
    } else if (value instanceof Color) {
        return ColorType;
    } else if (value instanceof Collator) {
        return CollatorType;
    } else if (value instanceof Formatted) {
        return FormattedType;
    } else if (value instanceof ResolvedImage) {
        return ResolvedImageType;
    } else if (Array.isArray(value)) {
        const length = value.length;
        let itemType;
        for (const item of value) {
            const t = typeOf(item);
            if (!itemType) {
                itemType = t;
            } else if (itemType === t) {
                continue;
            } else {
                itemType = ValueType;
                break;
            }
        }
        return array(itemType || ValueType, length);
    } else {
        return ObjectType;
    }
}
function toString(value) {
    const type = typeof value;
    if (value === null) {
        return '';
    } else if (type === 'string' || type === 'number' || type === 'boolean') {
        return String(value);
    } else if (value instanceof Color) {
        return value.toStringPremultipliedAlpha();
    } else if (value instanceof Formatted || value instanceof ResolvedImage) {
        return value.toString();
    } else {
        return JSON.stringify(value);
    }
}

class Literal {
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    static parse(args, context) {
        if (args.length !== 2)
            return context.error(`'literal' expression requires exactly one argument, but found ${ args.length - 1 } instead.`);
        if (!isValue(args[1]))
            return context.error(`invalid value`);
        const value = args[1];
        let type = typeOf(value);
        const expected = context.expectedType;
        if (type.kind === 'array' && type.N === 0 && expected && expected.kind === 'array' && (typeof expected.N !== 'number' || expected.N === 0)) {
            type = expected;
        }
        return new Literal(type, value);
    }
    evaluate() {
        return this.value;
    }
    eachChild() {
    }
    outputDefined() {
        return true;
    }
    serialize() {
        if (this.type.kind === 'array' || this.type.kind === 'object') {
            return [
                'literal',
                this.value
            ];
        } else if (this.value instanceof Color) {
            return ['rgba'].concat(this.value.toRenderColor(null).toArray());
        } else if (this.value instanceof Formatted) {
            return this.value.serialize();
        } else {
            return this.value;
        }
    }
}

class RuntimeError {
    constructor(message) {
        this.name = 'ExpressionEvaluationError';
        this.message = message;
    }
    toJSON() {
        return this.message;
    }
}

const types$1 = {
    string: StringType,
    number: NumberType,
    boolean: BooleanType,
    object: ObjectType
};
class Assertion {
    constructor(type, args) {
        this.type = type;
        this.args = args;
    }
    static parse(args, context) {
        if (args.length < 2)
            return context.error(`Expected at least one argument.`);
        let i = 1;
        let type;
        const name = args[0];
        if (name === 'array') {
            let itemType;
            if (args.length > 2) {
                const type2 = args[1];
                if (typeof type2 !== 'string' || !(type2 in types$1) || type2 === 'object')
                    return context.error('The item type argument of "array" must be one of string, number, boolean', 1);
                itemType = types$1[type2];
                i++;
            } else {
                itemType = ValueType;
            }
            let N;
            if (args.length > 3) {
                if (args[2] !== null && (typeof args[2] !== 'number' || args[2] < 0 || args[2] !== Math.floor(args[2]))) {
                    return context.error('The length argument to "array" must be a positive integer literal', 2);
                }
                N = args[2];
                i++;
            }
            type = array(itemType, N);
        } else {
            type = types$1[name];
        }
        const parsed = [];
        for (; i < args.length; i++) {
            const input = context.parse(args[i], i, ValueType);
            if (!input)
                return null;
            parsed.push(input);
        }
        return new Assertion(type, parsed);
    }
    evaluate(ctx) {
        for (let i = 0; i < this.args.length; i++) {
            const value = this.args[i].evaluate(ctx);
            const error = checkSubtype(this.type, typeOf(value));
            if (!error) {
                return value;
            } else if (i === this.args.length - 1) {
                throw new RuntimeError(`The expression ${ JSON.stringify(this.args[i].serialize()) } evaluated to ${ toString$1(typeOf(value)) } but was expected to be of type ${ toString$1(this.type) }.`);
            }
        }
        return null;
    }
    eachChild(fn) {
        this.args.forEach(fn);
    }
    outputDefined() {
        return this.args.every(arg => arg.outputDefined());
    }
    serialize() {
        const type = this.type;
        const serialized = [type.kind];
        if (type.kind === 'array') {
            const itemType = type.itemType;
            if (itemType.kind === 'string' || itemType.kind === 'number' || itemType.kind === 'boolean') {
                serialized.push(itemType.kind);
                const N = type.N;
                if (typeof N === 'number' || this.args.length > 1) {
                    serialized.push(N);
                }
            }
        }
        return serialized.concat(this.args.map(arg => arg.serialize()));
    }
}

class FormatExpression {
    constructor(sections) {
        this.type = FormattedType;
        this.sections = sections;
    }
    static parse(args, context) {
        if (args.length < 2) {
            return context.error(`Expected at least one argument.`);
        }
        const firstArg = args[1];
        if (!Array.isArray(firstArg) && typeof firstArg === 'object') {
            return context.error(`First argument must be an image or text section.`);
        }
        const sections = [];
        let nextTokenMayBeObject = false;
        for (let i = 1; i <= args.length - 1; ++i) {
            const arg = args[i];
            if (nextTokenMayBeObject && typeof arg === 'object' && !Array.isArray(arg)) {
                nextTokenMayBeObject = false;
                let scale = null;
                if (arg['font-scale']) {
                    scale = context.parseObjectValue(arg['font-scale'], i, 'font-scale', NumberType);
                    if (!scale)
                        return null;
                }
                let font = null;
                if (arg['text-font']) {
                    font = context.parseObjectValue(arg['text-font'], i, 'text-font', array(StringType));
                    if (!font)
                        return null;
                }
                let textColor = null;
                if (arg['text-color']) {
                    textColor = context.parseObjectValue(arg['text-color'], i, 'text-color', ColorType);
                    if (!textColor)
                        return null;
                }
                const lastExpression = sections[sections.length - 1];
                lastExpression.scale = scale;
                lastExpression.font = font;
                lastExpression.textColor = textColor;
            } else {
                const content = context.parse(args[i], i, ValueType);
                if (!content)
                    return null;
                const kind = content.type.kind;
                if (kind !== 'string' && kind !== 'value' && kind !== 'null' && kind !== 'resolvedImage')
                    return context.error(`Formatted text type must be 'string', 'value', 'image' or 'null'.`);
                nextTokenMayBeObject = true;
                sections.push({
                    content,
                    scale: null,
                    font: null,
                    textColor: null
                });
            }
        }
        return new FormatExpression(sections);
    }
    evaluate(ctx) {
        const evaluateSection = section => {
            const evaluatedContent = section.content.evaluate(ctx);
            if (typeOf(evaluatedContent) === ResolvedImageType) {
                return new FormattedSection('', evaluatedContent, null, null, null);
            }
            return new FormattedSection(toString(evaluatedContent), null, section.scale ? section.scale.evaluate(ctx) : null, section.font ? section.font.evaluate(ctx).join(',') : null, section.textColor ? section.textColor.evaluate(ctx) : null);
        };
        return new Formatted(this.sections.map(evaluateSection));
    }
    eachChild(fn) {
        for (const section of this.sections) {
            fn(section.content);
            if (section.scale) {
                fn(section.scale);
            }
            if (section.font) {
                fn(section.font);
            }
            if (section.textColor) {
                fn(section.textColor);
            }
        }
    }
    outputDefined() {
        return false;
    }
    serialize() {
        const serialized = ['format'];
        for (const section of this.sections) {
            serialized.push(section.content.serialize());
            const options = {};
            if (section.scale) {
                options['font-scale'] = section.scale.serialize();
            }
            if (section.font) {
                options['text-font'] = section.font.serialize();
            }
            if (section.textColor) {
                options['text-color'] = section.textColor.serialize();
            }
            serialized.push(options);
        }
        return serialized;
    }
}

function isImageOptions(value) {
    if (value !== null && typeof value === 'object' && !Array.isArray(value)) {
        return true;
    }
    return false;
}
class ImageExpression {
    constructor(inputPrimary, inputSecondary, inputPrimaryParams, inputSecondaryParams) {
        this._imageWarnHistory = {};
        this.type = ResolvedImageType;
        this.inputPrimary = inputPrimary;
        this.inputSecondary = inputSecondary;
        this.inputPrimaryParams = inputPrimaryParams;
        this.inputSecondaryParams = inputSecondaryParams;
    }
    static parse(args, context) {
        if (args.length < 2) {
            return context.error(`Expected two or more arguments.`);
        }
        let nextArgId = 1;
        const imageExpression = [];
        function tryParseImage() {
            if (nextArgId < args.length) {
                const imageName = context.parse(args[nextArgId], nextArgId++, StringType);
                if (!imageName) {
                    context.error(imageExpression.length ? `Secondary image variant is not a string.` : `No image name provided.`);
                    return false;
                }
                imageExpression.push({
                    image: imageName,
                    options: void 0
                });
                return true;
            }
            return true;
        }
        function tryParseOptions() {
            if (nextArgId < args.length) {
                if (!isImageOptions(args[nextArgId])) {
                    return true;
                }
                const params = args[nextArgId].params;
                const optionsContext = context.concat(nextArgId);
                if (!params) {
                    nextArgId++;
                    return true;
                }
                if (typeof params !== 'object' || params.constructor !== Object) {
                    optionsContext.error(`Image options "params" should be an object`);
                    return false;
                }
                const parsed = {};
                const childContext = optionsContext.concat(void 0, 'params');
                for (const key in params) {
                    if (!key) {
                        childContext.error(`Image parameter name should be non-empty`);
                        return false;
                    }
                    const value = childContext.concat(void 0, key).parse(params[key], void 0, ColorType, void 0, { typeAnnotation: 'coerce' });
                    if (!value) {
                        return false;
                    }
                    parsed[key] = value;
                }
                imageExpression[imageExpression.length - 1].options = parsed;
                nextArgId++;
                return true;
            }
            return true;
        }
        for (let i = 0; i < 2; i++) {
            if (!tryParseImage() || !tryParseOptions()) {
                return;
            }
        }
        return new ImageExpression(imageExpression[0].image, imageExpression[1] ? imageExpression[1].image : void 0, imageExpression[0].options, imageExpression[1] ? imageExpression[1].options : void 0);
    }
    evaluateParams(ctx, params) {
        const result = {};
        if (params) {
            for (const key in params) {
                if (params[key]) {
                    try {
                        const color = params[key].evaluate(ctx);
                        const msg = `Ignoring image parameter "${ key }" with semi-transparent color ${ color.toString() }`;
                        if (color.a !== 1) {
                            if (!this._imageWarnHistory[msg]) {
                                console.warn(msg);
                                this._imageWarnHistory[msg] = true;
                            }
                            continue;
                        }
                        result[key] = color;
                    } catch (err) {
                        continue;
                    }
                }
            }
        } else {
            return void 0;
        }
        if (Object.keys(result).length === 0) {
            return void 0;
        }
        return { params: result };
    }
    evaluate(ctx) {
        const value = ResolvedImage.build(this.inputPrimary.evaluate(ctx), this.inputSecondary ? this.inputSecondary.evaluate(ctx) : void 0, this.inputPrimaryParams ? this.evaluateParams(ctx, this.inputPrimaryParams) : void 0, this.inputSecondaryParams ? this.evaluateParams(ctx, this.inputSecondaryParams) : void 0);
        if (value && ctx.availableImages) {
            value.available = ctx.availableImages.indexOf(value.namePrimary) > -1;
            if (value.nameSecondary && value.available && ctx.availableImages) {
                value.available = ctx.availableImages.indexOf(value.nameSecondary) > -1;
            }
        }
        return value;
    }
    eachChild(fn) {
        fn(this.inputPrimary);
        if (this.inputPrimaryParams) {
            for (const key in this.inputPrimaryParams) {
                if (this.inputPrimaryParams[key]) {
                    fn(this.inputPrimaryParams[key]);
                }
            }
        }
        if (this.inputSecondary) {
            fn(this.inputSecondary);
            if (this.inputSecondaryParams) {
                for (const key in this.inputSecondaryParams) {
                    if (this.inputSecondaryParams[key]) {
                        fn(this.inputSecondaryParams[key]);
                    }
                }
            }
        }
    }
    outputDefined() {
        return false;
    }
    serializeParams(params) {
        const result = {};
        if (params) {
            for (const key in params) {
                if (params[key]) {
                    result[key] = params[key].serialize();
                }
            }
        } else {
            return void 0;
        }
        return { params: result };
    }
    serialize() {
        const serialized = [
            'image',
            this.inputPrimary.serialize()
        ];
        if (this.inputPrimaryParams) {
            serialized.push(this.serializeParams(this.inputPrimaryParams));
        }
        if (this.inputSecondary) {
            serialized.push(this.inputSecondary.serialize());
            if (this.inputSecondaryParams) {
                serialized.push(this.serializeParams(this.inputSecondaryParams));
            }
        }
        return serialized;
    }
}

function getType(val) {
    if (val instanceof Number) {
        return 'number';
    } else if (val instanceof String) {
        return 'string';
    } else if (val instanceof Boolean) {
        return 'boolean';
    } else if (Array.isArray(val)) {
        return 'array';
    } else if (val === null) {
        return 'null';
    } else {
        return typeof val;
    }
}

const types = {
    'to-boolean': BooleanType,
    'to-color': ColorType,
    'to-number': NumberType,
    'to-string': StringType
};
class Coercion {
    constructor(type, args) {
        this.type = type;
        this.args = args;
    }
    static parse(args, context) {
        if (args.length < 2)
            return context.error(`Expected at least one argument.`);
        const name = args[0];
        const parsed = [];
        let type = NullType;
        if (name === 'to-array') {
            if (!Array.isArray(args[1])) {
                return null;
            }
            const arrayLength = args[1].length;
            if (context.expectedType) {
                if (context.expectedType.kind === 'array') {
                    type = array(context.expectedType.itemType, arrayLength);
                } else {
                    return context.error(`Expected ${ context.expectedType.kind } but found array.`);
                }
            } else if (arrayLength > 0 && isValue(args[1][0])) {
                const value = args[1][0];
                type = array(typeOf(value), arrayLength);
            } else {
                return null;
            }
            for (let i = 0; i < arrayLength; i++) {
                const member = args[1][i];
                let parsedMember;
                if (getType(member) === 'array') {
                    parsedMember = context.parse(member, void 0, type.itemType);
                } else {
                    const memberType = getType(member);
                    if (memberType !== type.itemType.kind) {
                        return context.error(`Expected ${ type.itemType.kind } but found ${ memberType }.`);
                    }
                    parsedMember = context.registry['literal'].parse([
                        'literal',
                        member === void 0 ? null : member
                    ], context);
                }
                if (!parsedMember)
                    return null;
                parsed.push(parsedMember);
            }
        } else {
            if ((name === 'to-boolean' || name === 'to-string') && args.length !== 2)
                return context.error(`Expected one argument.`);
            type = types[name];
            for (let i = 1; i < args.length; i++) {
                const input = context.parse(args[i], i, ValueType);
                if (!input)
                    return null;
                parsed.push(input);
            }
        }
        return new Coercion(type, parsed);
    }
    evaluate(ctx) {
        if (this.type.kind === 'boolean') {
            return Boolean(this.args[0].evaluate(ctx));
        } else if (this.type.kind === 'color') {
            let input;
            let error;
            for (const arg of this.args) {
                input = arg.evaluate(ctx);
                error = null;
                if (input instanceof Color) {
                    return input;
                } else if (typeof input === 'string') {
                    const c = ctx.parseColor(input);
                    if (c)
                        return c;
                } else if (Array.isArray(input)) {
                    if (input.length < 3 || input.length > 4) {
                        error = `Invalid rbga value ${ JSON.stringify(input) }: expected an array containing either three or four numeric values.`;
                    } else {
                        error = validateRGBA(input[0], input[1], input[2], input[3]);
                    }
                    if (!error) {
                        return new Color(input[0] / 255, input[1] / 255, input[2] / 255, input[3]);
                    }
                }
            }
            throw new RuntimeError(error || `Could not parse color from value '${ typeof input === 'string' ? input : String(JSON.stringify(input)) }'`);
        } else if (this.type.kind === 'number') {
            let value = null;
            for (const arg of this.args) {
                value = arg.evaluate(ctx);
                if (value === null)
                    return 0;
                const num = Number(value);
                if (isNaN(num))
                    continue;
                return num;
            }
            throw new RuntimeError(`Could not convert ${ JSON.stringify(value) } to number.`);
        } else if (this.type.kind === 'formatted') {
            return Formatted.fromString(toString(this.args[0].evaluate(ctx)));
        } else if (this.type.kind === 'resolvedImage') {
            return ResolvedImage.build(toString(this.args[0].evaluate(ctx)));
        } else if (this.type.kind === 'array') {
            return this.args.map(arg => {
                return arg.evaluate(ctx);
            });
        } else {
            return toString(this.args[0].evaluate(ctx));
        }
    }
    eachChild(fn) {
        this.args.forEach(fn);
    }
    outputDefined() {
        return this.args.every(arg => arg.outputDefined());
    }
    serialize() {
        if (this.type.kind === 'formatted') {
            return new FormatExpression([{
                    content: this.args[0],
                    scale: null,
                    font: null,
                    textColor: null
                }]).serialize();
        }
        if (this.type.kind === 'resolvedImage') {
            return new ImageExpression(this.args[0]).serialize();
        }
        const serialized = this.type.kind === 'array' ? [] : [`to-${ this.type.kind }`];
        this.eachChild(child => {
            serialized.push(child.serialize());
        });
        return serialized;
    }
}

const geometryTypes = [
    'Unknown',
    'Point',
    'LineString',
    'Polygon'
];
class EvaluationContext {
    constructor(scope, options) {
        this.globals = null;
        this.feature = null;
        this.featureState = null;
        this.formattedSection = null;
        this._parseColorCache = {};
        this.availableImages = null;
        this.canonical = null;
        this.featureTileCoord = null;
        this.featureDistanceData = null;
        this.scope = scope;
        this.options = options;
    }
    id() {
        return this.feature && this.feature.id !== void 0 ? this.feature.id : null;
    }
    geometryType() {
        return this.feature ? typeof this.feature.type === 'number' ? geometryTypes[this.feature.type] : this.feature.type : null;
    }
    geometry() {
        return this.feature && 'geometry' in this.feature ? this.feature.geometry : null;
    }
    canonicalID() {
        return this.canonical;
    }
    properties() {
        return this.feature && this.feature.properties || {};
    }
    measureLight(_) {
        return this.globals.brightness || 0;
    }
    distanceFromCenter() {
        if (this.featureTileCoord && this.featureDistanceData) {
            const c = this.featureDistanceData.center;
            const scale = this.featureDistanceData.scale;
            const {x, y} = this.featureTileCoord;
            const dX = x * scale - c[0];
            const dY = y * scale - c[1];
            const bX = this.featureDistanceData.bearing[0];
            const bY = this.featureDistanceData.bearing[1];
            const dist = bX * dX + bY * dY;
            return dist;
        }
        return 0;
    }
    parseColor(input) {
        let cached = this._parseColorCache[input];
        if (!cached) {
            cached = this._parseColorCache[input] = Color.parse(input);
        }
        return cached;
    }
    getConfig(id) {
        return this.options ? this.options.get(id) : null;
    }
}

class CompoundExpression {
    constructor(name, type, evaluate, args, overloadIndex) {
        this.name = name;
        this.type = type;
        this._evaluate = evaluate;
        this.args = args;
        this._overloadIndex = overloadIndex;
    }
    evaluate(ctx) {
        if (!this._evaluate) {
            const definition = CompoundExpression.definitions[this.name];
            this._evaluate = Array.isArray(definition) ? definition[2] : definition.overloads[this._overloadIndex][1];
        }
        return this._evaluate(ctx, this.args);
    }
    eachChild(fn) {
        this.args.forEach(fn);
    }
    outputDefined() {
        return false;
    }
    serialize() {
        return [this.name].concat(this.args.map(arg => arg.serialize()));
    }
    static parse(args, context) {
        const op = args[0];
        const definition = CompoundExpression.definitions[op];
        if (!definition) {
            return context.error(`Unknown expression "${ op }". If you wanted a literal array, use ["literal", [...]].`, 0);
        }
        const type = Array.isArray(definition) ? definition[0] : definition.type;
        const availableOverloads = Array.isArray(definition) ? [[
                definition[1],
                definition[2]
            ]] : definition.overloads;
        const overloadParams = [];
        let signatureContext = null;
        let overloadIndex = -1;
        for (const [params, evaluate] of availableOverloads) {
            if (Array.isArray(params) && params.length !== args.length - 1)
                continue;
            overloadParams.push(params);
            overloadIndex++;
            signatureContext = new ParsingContext$1(context.registry, context.path, null, context.scope, void 0, context._scope, context.options);
            const parsedArgs = [];
            let argParseFailed = false;
            for (let i = 1; i < args.length; i++) {
                const arg = args[i];
                const expectedType = Array.isArray(params) ? params[i - 1] : // @ts-expect-error - TS2339 - Property 'type' does not exist on type 'Varargs | Evaluate'.
                params.type;
                const parsed = signatureContext.parse(arg, 1 + parsedArgs.length, expectedType);
                if (!parsed) {
                    argParseFailed = true;
                    break;
                }
                parsedArgs.push(parsed);
            }
            if (argParseFailed) {
                continue;
            }
            if (Array.isArray(params)) {
                if (params.length !== parsedArgs.length) {
                    signatureContext.error(`Expected ${ params.length } arguments, but found ${ parsedArgs.length } instead.`);
                    continue;
                }
            }
            for (let i = 0; i < parsedArgs.length; i++) {
                const expected = Array.isArray(params) ? params[i] : params.type;
                const arg = parsedArgs[i];
                signatureContext.concat(i + 1).checkSubtype(expected, arg.type);
            }
            if (signatureContext.errors.length === 0) {
                return new CompoundExpression(op, type, evaluate, parsedArgs, overloadIndex);
            }
        }
        if (overloadParams.length === 1) {
            context.errors.push(...signatureContext.errors);
        } else {
            const expected = overloadParams.length ? overloadParams : availableOverloads.map(([params]) => params);
            const signatures = expected.map(stringifySignature).join(' | ');
            const actualTypes = [];
            for (let i = 1; i < args.length; i++) {
                const parsed = context.parse(args[i], 1 + actualTypes.length);
                if (!parsed)
                    return null;
                actualTypes.push(toString$1(parsed.type));
            }
            context.error(`Expected arguments of type ${ signatures }, but found (${ actualTypes.join(', ') }) instead.`);
        }
        return null;
    }
    static register(registry, definitions) {
        CompoundExpression.definitions = definitions;
        for (const name in definitions) {
            registry[name] = CompoundExpression;
        }
    }
}
function stringifySignature(signature) {
    if (Array.isArray(signature)) {
        return `(${ signature.map(toString$1).join(', ') })`;
    } else {
        return `(${ toString$1(signature.type) }...)`;
    }
}

class CollatorExpression {
    constructor(caseSensitive, diacriticSensitive, locale) {
        this.type = CollatorType;
        this.locale = locale;
        this.caseSensitive = caseSensitive;
        this.diacriticSensitive = diacriticSensitive;
    }
    static parse(args, context) {
        if (args.length !== 2)
            return context.error(`Expected one argument.`);
        const options = args[1];
        if (typeof options !== 'object' || Array.isArray(options))
            return context.error(`Collator options argument must be an object.`);
        const caseSensitive = options['case-sensitive'] === void 0 ? context.parse(false, 1, BooleanType) : context.parseObjectValue(options['case-sensitive'], 1, 'case-sensitive', BooleanType);
        if (!caseSensitive)
            return null;
        const diacriticSensitive = options['diacritic-sensitive'] === void 0 ? context.parse(false, 1, BooleanType) : context.parseObjectValue(options['diacritic-sensitive'], 1, 'diacritic-sensitive', BooleanType);
        if (!diacriticSensitive)
            return null;
        let locale = null;
        if (options['locale']) {
            locale = context.parseObjectValue(options['locale'], 1, 'locale', StringType);
            if (!locale)
                return null;
        }
        return new CollatorExpression(caseSensitive, diacriticSensitive, locale);
    }
    evaluate(ctx) {
        return new Collator(this.caseSensitive.evaluate(ctx), this.diacriticSensitive.evaluate(ctx), this.locale ? this.locale.evaluate(ctx) : null);
    }
    eachChild(fn) {
        fn(this.caseSensitive);
        fn(this.diacriticSensitive);
        if (this.locale) {
            fn(this.locale);
        }
    }
    outputDefined() {
        return false;
    }
    serialize() {
        const options = {};
        options['case-sensitive'] = this.caseSensitive.serialize();
        options['diacritic-sensitive'] = this.diacriticSensitive.serialize();
        if (this.locale) {
            options['locale'] = this.locale.serialize();
        }
        return [
            'collator',
            options
        ];
    }
}

/**
 * Rearranges items so that all items in the [left, k] are the smallest.
 * The k-th element will have the (k - left + 1)-th smallest value in [left, right].
 *
 * @template T
 * @param {T[]} arr the array to partially sort (in place)
 * @param {number} k middle index for partial sorting (as defined above)
 * @param {number} [left=0] left index of the range to sort
 * @param {number} [right=arr.length-1] right index
 * @param {(a: T, b: T) => number} [compare = (a, b) => a - b] compare function
 */
function quickselect(arr, k, left = 0, right = arr.length - 1, compare = defaultCompare) {
    while (right > left) {
        if (right - left > 600) {
            const n = right - left + 1;
            const m = k - left + 1;
            const z = Math.log(n);
            const s = 0.5 * Math.exp(2 * z / 3);
            const sd = 0.5 * Math.sqrt(z * s * (n - s) / n) * (m - n / 2 < 0 ? -1 : 1);
            const newLeft = Math.max(left, Math.floor(k - m * s / n + sd));
            const newRight = Math.min(right, Math.floor(k + (n - m) * s / n + sd));
            quickselect(arr, k, newLeft, newRight, compare);
        }
        const t = arr[k];
        let i = left;
        /** @type {number} */
        let j = right;
        swap$2(arr, left, k);
        if (compare(arr[right], t) > 0)
            swap$2(arr, left, right);
        while (i < j) {
            swap$2(arr, i, j);
            i++;
            j--;
            while (compare(arr[i], t) < 0)
                i++;
            while (compare(arr[j], t) > 0)
                j--;
        }
        if (compare(arr[left], t) === 0)
            swap$2(arr, left, j);
        else {
            j++;
            swap$2(arr, j, right);
        }
        if (j <= k)
            left = j + 1;
        if (k <= j)
            right = j - 1;
    }
}
/**
 * @template T
 * @param {T[]} arr
 * @param {number} i
 * @param {number} j
 */
function swap$2(arr, i, j) {
    const tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
}
/**
 * @template T
 * @param {T} a
 * @param {T} b
 * @returns {number}
 */
function defaultCompare(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}

function calculateSignedArea(ring) {
    let sum = 0;
    for (let i = 0, len = ring.length, j = len - 1, p1, p2; i < len; j = i++) {
        p1 = ring[i];
        p2 = ring[j];
        sum += (p2.x - p1.x) * (p1.y + p2.y);
    }
    return sum;
}
function classifyRings$1(rings, maxRings) {
    const len = rings.length;
    if (len <= 1)
        return [rings];
    const polygons = [];
    let polygon, ccw;
    for (let i = 0; i < len; i++) {
        const area = calculateSignedArea(rings[i]);
        if (area === 0)
            continue;
        rings[i].area = Math.abs(area);
        if (ccw === void 0)
            ccw = area < 0;
        if (ccw === area < 0) {
            if (polygon)
                polygons.push(polygon);
            polygon = [rings[i]];
        } else {
            polygon.push(rings[i]);
        }
    }
    if (polygon)
        polygons.push(polygon);
    return polygons;
}
function updateBBox(bbox, coord) {
    bbox[0] = Math.min(bbox[0], coord[0]);
    bbox[1] = Math.min(bbox[1], coord[1]);
    bbox[2] = Math.max(bbox[2], coord[0]);
    bbox[3] = Math.max(bbox[3], coord[1]);
}
function boxWithinBox(bbox1, bbox2) {
    if (bbox1[0] <= bbox2[0])
        return false;
    if (bbox1[2] >= bbox2[2])
        return false;
    if (bbox1[1] <= bbox2[1])
        return false;
    if (bbox1[3] >= bbox2[3])
        return false;
    return true;
}
function onBoundary(p, p1, p2) {
    const x1 = p[0] - p1[0];
    const y1 = p[1] - p1[1];
    const x2 = p[0] - p2[0];
    const y2 = p[1] - p2[1];
    return x1 * y2 - x2 * y1 === 0 && x1 * x2 <= 0 && y1 * y2 <= 0;
}
function rayIntersect(p, p1, p2) {
    return p1[1] > p[1] !== p2[1] > p[1] && p[0] < (p2[0] - p1[0]) * (p[1] - p1[1]) / (p2[1] - p1[1]) + p1[0];
}
function pointWithinPolygon(point, rings, trueOnBoundary = false) {
    let inside = false;
    for (let i = 0, len = rings.length; i < len; i++) {
        const ring = rings[i];
        for (let j = 0, len2 = ring.length, k = len2 - 1; j < len2; k = j++) {
            const q1 = ring[k];
            const q2 = ring[j];
            if (onBoundary(point, q1, q2))
                return trueOnBoundary;
            if (rayIntersect(point, q1, q2))
                inside = !inside;
        }
    }
    return inside;
}
function perp(v1, v2) {
    return v1[0] * v2[1] - v1[1] * v2[0];
}
function twoSided(p1, p2, q1, q2) {
    const x1 = p1[0] - q1[0];
    const y1 = p1[1] - q1[1];
    const x2 = p2[0] - q1[0];
    const y2 = p2[1] - q1[1];
    const x3 = q2[0] - q1[0];
    const y3 = q2[1] - q1[1];
    const det1 = x1 * y3 - x3 * y1;
    const det2 = x2 * y3 - x3 * y2;
    if (det1 > 0 && det2 < 0 || det1 < 0 && det2 > 0)
        return true;
    return false;
}
function segmentIntersectSegment(a, b, c, d) {
    const vectorP = [
        b[0] - a[0],
        b[1] - a[1]
    ];
    const vectorQ = [
        d[0] - c[0],
        d[1] - c[1]
    ];
    if (perp(vectorQ, vectorP) === 0)
        return false;
    if (twoSided(a, b, c, d) && twoSided(c, d, a, b))
        return true;
    return false;
}

const EXTENT$1 = 8192;
function mercatorXfromLng$1(lng) {
    return (180 + lng) / 360;
}
function mercatorYfromLat$1(lat) {
    return (180 - 180 / Math.PI * Math.log(Math.tan(Math.PI / 4 + lat * Math.PI / 360))) / 360;
}
function getTileCoordinates(p, canonical) {
    const x = mercatorXfromLng$1(p[0]);
    const y = mercatorYfromLat$1(p[1]);
    const tilesAtZoom = Math.pow(2, canonical.z);
    return [
        Math.round(x * tilesAtZoom * EXTENT$1),
        Math.round(y * tilesAtZoom * EXTENT$1)
    ];
}
function pointWithinPolygons(point, polygons) {
    for (let i = 0; i < polygons.length; i++) {
        if (pointWithinPolygon(point, polygons[i]))
            return true;
    }
    return false;
}
function lineIntersectPolygon(p1, p2, polygon) {
    for (const ring of polygon) {
        for (let j = 0, len = ring.length, k = len - 1; j < len; k = j++) {
            const q1 = ring[k];
            const q2 = ring[j];
            if (segmentIntersectSegment(p1, p2, q1, q2)) {
                return true;
            }
        }
    }
    return false;
}
function lineStringWithinPolygon(line, polygon) {
    for (let i = 0; i < line.length; ++i) {
        if (!pointWithinPolygon(line[i], polygon)) {
            return false;
        }
    }
    for (let i = 0; i < line.length - 1; ++i) {
        if (lineIntersectPolygon(line[i], line[i + 1], polygon)) {
            return false;
        }
    }
    return true;
}
function lineStringWithinPolygons(line, polygons) {
    for (let i = 0; i < polygons.length; i++) {
        if (lineStringWithinPolygon(line, polygons[i]))
            return true;
    }
    return false;
}
function getTilePolygon(coordinates, bbox, canonical) {
    const polygon = [];
    for (let i = 0; i < coordinates.length; i++) {
        const ring = [];
        for (let j = 0; j < coordinates[i].length; j++) {
            const coord = getTileCoordinates(coordinates[i][j], canonical);
            updateBBox(bbox, coord);
            ring.push(coord);
        }
        polygon.push(ring);
    }
    return polygon;
}
function getTilePolygons(coordinates, bbox, canonical) {
    const polygons = [];
    for (let i = 0; i < coordinates.length; i++) {
        const polygon = getTilePolygon(coordinates[i], bbox, canonical);
        polygons.push(polygon);
    }
    return polygons;
}
function updatePoint(p, bbox, polyBBox, worldSize) {
    if (p[0] < polyBBox[0] || p[0] > polyBBox[2]) {
        const halfWorldSize = worldSize * 0.5;
        let shift = p[0] - polyBBox[0] > halfWorldSize ? -worldSize : polyBBox[0] - p[0] > halfWorldSize ? worldSize : 0;
        if (shift === 0) {
            shift = p[0] - polyBBox[2] > halfWorldSize ? -worldSize : polyBBox[2] - p[0] > halfWorldSize ? worldSize : 0;
        }
        p[0] += shift;
    }
    updateBBox(bbox, p);
}
function resetBBox(bbox) {
    bbox[0] = bbox[1] = Infinity;
    bbox[2] = bbox[3] = -Infinity;
}
function getTilePoints(geometry, pointBBox, polyBBox, canonical) {
    const worldSize = Math.pow(2, canonical.z) * EXTENT$1;
    const shifts = [
        canonical.x * EXTENT$1,
        canonical.y * EXTENT$1
    ];
    const tilePoints = [];
    if (!geometry)
        return tilePoints;
    for (const points of geometry) {
        for (const point of points) {
            const p = [
                point.x + shifts[0],
                point.y + shifts[1]
            ];
            updatePoint(p, pointBBox, polyBBox, worldSize);
            tilePoints.push(p);
        }
    }
    return tilePoints;
}
function getTileLines(geometry, lineBBox, polyBBox, canonical) {
    const worldSize = Math.pow(2, canonical.z) * EXTENT$1;
    const shifts = [
        canonical.x * EXTENT$1,
        canonical.y * EXTENT$1
    ];
    const tileLines = [];
    if (!geometry)
        return tileLines;
    for (const line of geometry) {
        const tileLine = [];
        for (const point of line) {
            const p = [
                point.x + shifts[0],
                point.y + shifts[1]
            ];
            updateBBox(lineBBox, p);
            tileLine.push(p);
        }
        tileLines.push(tileLine);
    }
    if (lineBBox[2] - lineBBox[0] <= worldSize / 2) {
        resetBBox(lineBBox);
        for (const line of tileLines) {
            for (const p of line) {
                updatePoint(p, lineBBox, polyBBox, worldSize);
            }
        }
    }
    return tileLines;
}
function pointsWithinPolygons(ctx, polygonGeometry) {
    const pointBBox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    const polyBBox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    const canonical = ctx.canonicalID();
    if (!canonical) {
        return false;
    }
    if (polygonGeometry.type === 'Polygon') {
        const tilePolygon = getTilePolygon(polygonGeometry.coordinates, polyBBox, canonical);
        const tilePoints = getTilePoints(ctx.geometry(), pointBBox, polyBBox, canonical);
        if (!boxWithinBox(pointBBox, polyBBox))
            return false;
        for (const point of tilePoints) {
            if (!pointWithinPolygon(point, tilePolygon))
                return false;
        }
    }
    if (polygonGeometry.type === 'MultiPolygon') {
        const tilePolygons = getTilePolygons(polygonGeometry.coordinates, polyBBox, canonical);
        const tilePoints = getTilePoints(ctx.geometry(), pointBBox, polyBBox, canonical);
        if (!boxWithinBox(pointBBox, polyBBox))
            return false;
        for (const point of tilePoints) {
            if (!pointWithinPolygons(point, tilePolygons))
                return false;
        }
    }
    return true;
}
function linesWithinPolygons(ctx, polygonGeometry) {
    const lineBBox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    const polyBBox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    const canonical = ctx.canonicalID();
    if (!canonical) {
        return false;
    }
    if (polygonGeometry.type === 'Polygon') {
        const tilePolygon = getTilePolygon(polygonGeometry.coordinates, polyBBox, canonical);
        const tileLines = getTileLines(ctx.geometry(), lineBBox, polyBBox, canonical);
        if (!boxWithinBox(lineBBox, polyBBox))
            return false;
        for (const line of tileLines) {
            if (!lineStringWithinPolygon(line, tilePolygon))
                return false;
        }
    }
    if (polygonGeometry.type === 'MultiPolygon') {
        const tilePolygons = getTilePolygons(polygonGeometry.coordinates, polyBBox, canonical);
        const tileLines = getTileLines(ctx.geometry(), lineBBox, polyBBox, canonical);
        if (!boxWithinBox(lineBBox, polyBBox))
            return false;
        for (const line of tileLines) {
            if (!lineStringWithinPolygons(line, tilePolygons))
                return false;
        }
    }
    return true;
}
class Within {
    constructor(geojson, geometries) {
        this.type = BooleanType;
        this.geojson = geojson;
        this.geometries = geometries;
    }
    static parse(args, context) {
        if (args.length !== 2)
            return context.error(`'within' expression requires exactly one argument, but found ${ args.length - 1 } instead.`);
        if (isValue(args[1])) {
            const geojson = args[1];
            if (geojson.type === 'FeatureCollection') {
                for (let i = 0; i < geojson.features.length; ++i) {
                    const type = geojson.features[i].geometry.type;
                    if (type === 'Polygon' || type === 'MultiPolygon') {
                        return new Within(geojson, geojson.features[i].geometry);
                    }
                }
            } else if (geojson.type === 'Feature') {
                const type = geojson.geometry.type;
                if (type === 'Polygon' || type === 'MultiPolygon') {
                    return new Within(geojson, geojson.geometry);
                }
            } else if (geojson.type === 'Polygon' || geojson.type === 'MultiPolygon') {
                return new Within(geojson, geojson);
            }
        }
        return context.error(`'within' expression requires valid geojson object that contains polygon geometry type.`);
    }
    evaluate(ctx) {
        if (ctx.geometry() != null && ctx.canonicalID() != null) {
            if (ctx.geometryType() === 'Point') {
                return pointsWithinPolygons(ctx, this.geometries);
            } else if (ctx.geometryType() === 'LineString') {
                return linesWithinPolygons(ctx, this.geometries);
            }
        }
        return false;
    }
    eachChild() {
    }
    outputDefined() {
        return true;
    }
    serialize() {
        return [
            'within',
            this.geojson
        ];
    }
}

const factors = {
    kilometers: 1,
    miles: 1000 / 1609.344,
    nauticalmiles: 1000 / 1852,
    meters: 1000,
    metres: 1000,
    yards: 1000 / 0.9144,
    feet: 1000 / 0.3048,
    inches: 1000 / 0.0254
};
// Values that define WGS84 ellipsoid model of the Earth
const RE = 6378.137;
// equatorial radius
const FE = 1 / 298.257223563;
// flattening
const E2 = FE * (2 - FE);
const RAD = Math.PI / 180;
/**
 * A collection of very fast approximations to common geodesic measurements. Useful for performance-sensitive code that measures things on a city scale.
 */
class CheapRuler {
    /**
     * Creates a ruler object from tile coordinates (y and z).
     *
     * @param {number} y
     * @param {number} z
     * @param {keyof typeof factors} [units='kilometers']
     * @returns {CheapRuler}
     * @example
     * const ruler = cheapRuler.fromTile(1567, 12);
     * //=ruler
     */
    static fromTile(y, z, units) {
        const n = Math.PI * (1 - 2 * (y + 0.5) / Math.pow(2, z));
        const lat = Math.atan(0.5 * (Math.exp(n) - Math.exp(-n))) / RAD;
        return new CheapRuler(lat, units);
    }
    /**
     * Multipliers for converting between units.
     *
     * @example
     * // convert 50 meters to yards
     * 50 * CheapRuler.units.yards / CheapRuler.units.meters;
     */
    static get units() {
        return factors;
    }
    /**
     * Creates a ruler instance for very fast approximations to common geodesic measurements around a certain latitude.
     *
     * @param {number} lat latitude
     * @param {keyof typeof factors} [units='kilometers']
     * @example
     * const ruler = cheapRuler(35.05, 'miles');
     * //=ruler
     */
    constructor(lat, units) {
        if (lat === undefined)
            throw new Error('No latitude given.');
        if (units && !factors[units])
            throw new Error(`Unknown unit ${ units }. Use one of: ${ Object.keys(factors).join(', ') }`);
        // Curvature formulas from https://en.wikipedia.org/wiki/Earth_radius#Meridional
        const m = RAD * RE * (units ? factors[units] : 1);
        const coslat = Math.cos(lat * RAD);
        const w2 = 1 / (1 - E2 * (1 - coslat * coslat));
        const w = Math.sqrt(w2);
        // multipliers for converting longitude and latitude degrees into distance
        this.kx = m * w * coslat;
        // based on normal radius of curvature
        this.ky = m * w * w2 * (1 - E2);    // based on meridonal radius of curvature
    }
    /**
     * Given two points of the form [longitude, latitude], returns the distance.
     *
     * @param {[number, number]} a point [longitude, latitude]
     * @param {[number, number]} b point [longitude, latitude]
     * @returns {number} distance
     * @example
     * const distance = ruler.distance([30.5, 50.5], [30.51, 50.49]);
     * //=distance
     */
    distance(a, b) {
        const dx = wrap(a[0] - b[0]) * this.kx;
        const dy = (a[1] - b[1]) * this.ky;
        return Math.sqrt(dx * dx + dy * dy);
    }
    /**
     * Returns the bearing between two points in angles.
     *
     * @param {[number, number]} a point [longitude, latitude]
     * @param {[number, number]} b point [longitude, latitude]
     * @returns {number} bearing
     * @example
     * const bearing = ruler.bearing([30.5, 50.5], [30.51, 50.49]);
     * //=bearing
     */
    bearing(a, b) {
        const dx = wrap(b[0] - a[0]) * this.kx;
        const dy = (b[1] - a[1]) * this.ky;
        return Math.atan2(dx, dy) / RAD;
    }
    /**
     * Returns a new point given distance and bearing from the starting point.
     *
     * @param {[number, number]} p point [longitude, latitude]
     * @param {number} dist distance
     * @param {number} bearing
     * @returns {[number, number]} point [longitude, latitude]
     * @example
     * const point = ruler.destination([30.5, 50.5], 0.1, 90);
     * //=point
     */
    destination(p, dist, bearing) {
        const a = bearing * RAD;
        return this.offset(p, Math.sin(a) * dist, Math.cos(a) * dist);
    }
    /**
     * Returns a new point given easting and northing offsets (in ruler units) from the starting point.
     *
     * @param {[number, number]} p point [longitude, latitude]
     * @param {number} dx easting
     * @param {number} dy northing
     * @returns {[number, number]} point [longitude, latitude]
     * @example
     * const point = ruler.offset([30.5, 50.5], 10, 10);
     * //=point
     */
    offset(p, dx, dy) {
        return [
            p[0] + dx / this.kx,
            p[1] + dy / this.ky
        ];
    }
    /**
     * Given a line (an array of points), returns the total line distance.
     *
     * @param {[number, number][]} points [longitude, latitude]
     * @returns {number} total line distance
     * @example
     * const length = ruler.lineDistance([
     *     [-67.031, 50.458], [-67.031, 50.534],
     *     [-66.929, 50.534], [-66.929, 50.458]
     * ]);
     * //=length
     */
    lineDistance(points) {
        let total = 0;
        for (let i = 0; i < points.length - 1; i++) {
            total += this.distance(points[i], points[i + 1]);
        }
        return total;
    }
    /**
     * Given a polygon (an array of rings, where each ring is an array of points), returns the area.
     *
     * @param {[number, number][][]} polygon
     * @returns {number} area value in the specified units (square kilometers by default)
     * @example
     * const area = ruler.area([[
     *     [-67.031, 50.458], [-67.031, 50.534], [-66.929, 50.534],
     *     [-66.929, 50.458], [-67.031, 50.458]
     * ]]);
     * //=area
     */
    area(polygon) {
        let sum = 0;
        for (let i = 0; i < polygon.length; i++) {
            const ring = polygon[i];
            for (let j = 0, len = ring.length, k = len - 1; j < len; k = j++) {
                sum += wrap(ring[j][0] - ring[k][0]) * (ring[j][1] + ring[k][1]) * (i ? -1 : 1);
            }
        }
        return Math.abs(sum) / 2 * this.kx * this.ky;
    }
    /**
     * Returns the point at a specified distance along the line.
     *
     * @param {[number, number][]} line
     * @param {number} dist distance
     * @returns {[number, number]} point [longitude, latitude]
     * @example
     * const point = ruler.along(line, 2.5);
     * //=point
     */
    along(line, dist) {
        let sum = 0;
        if (dist <= 0)
            return line[0];
        for (let i = 0; i < line.length - 1; i++) {
            const p0 = line[i];
            const p1 = line[i + 1];
            const d = this.distance(p0, p1);
            sum += d;
            if (sum > dist)
                return interpolate(p0, p1, (dist - (sum - d)) / d);
        }
        return line[line.length - 1];
    }
    /**
     * Returns the distance from a point `p` to a line segment `a` to `b`.
     *
     * @pointToSegmentDistance
     * @param {[number, number]} p point [longitude, latitude]
     * @param {[number, number]} a segment point 1 [longitude, latitude]
     * @param {[number, number]} b segment point 2 [longitude, latitude]
     * @returns {number} distance
     * @example
     * const distance = ruler.pointToSegmentDistance([-67.04, 50.5], [-67.05, 50.57], [-67.03, 50.54]);
     * //=distance
     */
    pointToSegmentDistance(p, a, b) {
        let [x, y] = a;
        let dx = wrap(b[0] - x) * this.kx;
        let dy = (b[1] - y) * this.ky;
        if (dx !== 0 || dy !== 0) {
            const t = (wrap(p[0] - x) * this.kx * dx + (p[1] - y) * this.ky * dy) / (dx * dx + dy * dy);
            if (t > 1) {
                x = b[0];
                y = b[1];
            } else if (t > 0) {
                x += dx / this.kx * t;
                y += dy / this.ky * t;
            }
        }
        dx = wrap(p[0] - x) * this.kx;
        dy = (p[1] - y) * this.ky;
        return Math.sqrt(dx * dx + dy * dy);
    }
    /**
     * Returns an object of the form {point, index, t}, where point is closest point on the line
     * from the given point, index is the start index of the segment with the closest point,
     * and t is a parameter from 0 to 1 that indicates where the closest point is on that segment.
     *
     * @param {[number, number][]} line
     * @param {[number, number]} p point [longitude, latitude]
     * @returns {{point: [number, number], index: number, t: number}} {point, index, t}
     * @example
     * const point = ruler.pointOnLine(line, [-67.04, 50.5]).point;
     * //=point
     */
    pointOnLine(line, p) {
        let minDist = Infinity;
        let minX = line[0][0];
        let minY = line[0][1];
        let minI = 0;
        let minT = 0;
        for (let i = 0; i < line.length - 1; i++) {
            let x = line[i][0];
            let y = line[i][1];
            let dx = wrap(line[i + 1][0] - x) * this.kx;
            let dy = (line[i + 1][1] - y) * this.ky;
            let t = 0;
            if (dx !== 0 || dy !== 0) {
                t = (wrap(p[0] - x) * this.kx * dx + (p[1] - y) * this.ky * dy) / (dx * dx + dy * dy);
                if (t > 1) {
                    x = line[i + 1][0];
                    y = line[i + 1][1];
                } else if (t > 0) {
                    x += dx / this.kx * t;
                    y += dy / this.ky * t;
                }
            }
            dx = wrap(p[0] - x) * this.kx;
            dy = (p[1] - y) * this.ky;
            const sqDist = dx * dx + dy * dy;
            if (sqDist < minDist) {
                minDist = sqDist;
                minX = x;
                minY = y;
                minI = i;
                minT = t;
            }
        }
        return {
            point: [
                minX,
                minY
            ],
            index: minI,
            t: Math.max(0, Math.min(1, minT))
        };
    }
    /**
     * Returns a part of the given line between the start and the stop points (or their closest points on the line).
     *
     * @param {[number, number]} start point [longitude, latitude]
     * @param {[number, number]} stop point [longitude, latitude]
     * @param {[number, number][]} line
     * @returns {[number, number][]} line part of a line
     * @example
     * const line2 = ruler.lineSlice([-67.04, 50.5], [-67.05, 50.56], line1);
     * //=line2
     */
    lineSlice(start, stop, line) {
        let p1 = this.pointOnLine(line, start);
        let p2 = this.pointOnLine(line, stop);
        if (p1.index > p2.index || p1.index === p2.index && p1.t > p2.t) {
            const tmp = p1;
            p1 = p2;
            p2 = tmp;
        }
        const slice = [p1.point];
        const l = p1.index + 1;
        const r = p2.index;
        if (!equals$1(line[l], slice[0]) && l <= r)
            slice.push(line[l]);
        for (let i = l + 1; i <= r; i++) {
            slice.push(line[i]);
        }
        if (!equals$1(line[r], p2.point))
            slice.push(p2.point);
        return slice;
    }
    /**
     * Returns a part of the given line between the start and the stop points indicated by distance along the line.
     *
     * @param {number} start start distance
     * @param {number} stop stop distance
     * @param {[number, number][]} line
     * @returns {[number, number][]} part of a line
     * @example
     * const line2 = ruler.lineSliceAlong(10, 20, line1);
     * //=line2
     */
    lineSliceAlong(start, stop, line) {
        let sum = 0;
        const slice = [];
        for (let i = 0; i < line.length - 1; i++) {
            const p0 = line[i];
            const p1 = line[i + 1];
            const d = this.distance(p0, p1);
            sum += d;
            if (sum > start && slice.length === 0) {
                slice.push(interpolate(p0, p1, (start - (sum - d)) / d));
            }
            if (sum >= stop) {
                slice.push(interpolate(p0, p1, (stop - (sum - d)) / d));
                return slice;
            }
            if (sum > start)
                slice.push(p1);
        }
        return slice;
    }
    /**
     * Given a point, returns a bounding box object ([w, s, e, n]) created from the given point buffered by a given distance.
     *
     * @param {[number, number]} p point [longitude, latitude]
     * @param {number} buffer
     * @returns {[number, number, number, number]} bbox ([w, s, e, n])
     * @example
     * const bbox = ruler.bufferPoint([30.5, 50.5], 0.01);
     * //=bbox
     */
    bufferPoint(p, buffer) {
        const v = buffer / this.ky;
        const h = buffer / this.kx;
        return [
            p[0] - h,
            p[1] - v,
            p[0] + h,
            p[1] + v
        ];
    }
    /**
     * Given a bounding box, returns the box buffered by a given distance.
     *
     * @param {[number, number, number, number]} bbox ([w, s, e, n])
     * @param {number} buffer
     * @returns {[number, number, number, number]} bbox ([w, s, e, n])
     * @example
     * const bbox = ruler.bufferBBox([30.5, 50.5, 31, 51], 0.2);
     * //=bbox
     */
    bufferBBox(bbox, buffer) {
        const v = buffer / this.ky;
        const h = buffer / this.kx;
        return [
            bbox[0] - h,
            bbox[1] - v,
            bbox[2] + h,
            bbox[3] + v
        ];
    }
    /**
     * Returns true if the given point is inside in the given bounding box, otherwise false.
     *
     * @param {[number, number]} p point [longitude, latitude]
     * @param {[number, number, number, number]} bbox ([w, s, e, n])
     * @returns {boolean}
     * @example
     * const inside = ruler.insideBBox([30.5, 50.5], [30, 50, 31, 51]);
     * //=inside
     */
    insideBBox(p, bbox) {
        // eslint-disable-line
        return wrap(p[0] - bbox[0]) >= 0 && wrap(p[0] - bbox[2]) <= 0 && p[1] >= bbox[1] && p[1] <= bbox[3];
    }
}
/**
 * @param {[number, number]} a
 * @param {[number, number]} b
 */
function equals$1(a, b) {
    return a[0] === b[0] && a[1] === b[1];
}
/**
 * @param {[number, number]} a
 * @param {[number, number]} b
 * @param {number} t
 * @returns {[number, number]}
 */
function interpolate(a, b, t) {
    const dx = wrap(b[0] - a[0]);
    const dy = b[1] - a[1];
    return [
        a[0] + dx * t,
        a[1] + dy * t
    ];
}
/**
 * normalize a degree value into [-180..180] range
 * @param {number} deg
 */
function wrap(deg) {
    while (deg < -180)
        deg += 360;
    while (deg > 180)
        deg -= 360;
    return deg;
}

class TinyQueue {
    constructor(data = [], compare = (a, b) => a < b ? -1 : a > b ? 1 : 0) {
        this.data = data;
        this.length = this.data.length;
        this.compare = compare;
        if (this.length > 0) {
            for (let i = (this.length >> 1) - 1; i >= 0; i--)
                this._down(i);
        }
    }
    push(item) {
        this.data.push(item);
        this._up(this.length++);
    }
    pop() {
        if (this.length === 0)
            return undefined;
        const top = this.data[0];
        const bottom = this.data.pop();
        if (--this.length > 0) {
            this.data[0] = bottom;
            this._down(0);
        }
        return top;
    }
    peek() {
        return this.data[0];
    }
    _up(pos) {
        const {data, compare} = this;
        const item = data[pos];
        while (pos > 0) {
            const parent = pos - 1 >> 1;
            const current = data[parent];
            if (compare(item, current) >= 0)
                break;
            data[pos] = current;
            pos = parent;
        }
        data[pos] = item;
    }
    _down(pos) {
        const {data, compare} = this;
        const halfLength = this.length >> 1;
        const item = data[pos];
        while (pos < halfLength) {
            let bestChild = (pos << 1) + 1;
            // initially it is the left child
            const right = bestChild + 1;
            if (right < this.length && compare(data[right], data[bestChild]) < 0) {
                bestChild = right;
            }
            if (compare(data[bestChild], item) >= 0)
                break;
            data[pos] = data[bestChild];
            pos = bestChild;
        }
        data[pos] = item;
    }
}

var EXTENT = 8192;

function compareMax$1(a, b) {
    return b.dist - a.dist;
}
const MIN_POINT_SIZE = 100;
const MIN_LINE_POINT_SIZE = 50;
function isDefaultBBOX(bbox) {
    const defualtBBox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    if (defualtBBox.length !== bbox.length) {
        return false;
    }
    for (let i = 0; i < defualtBBox.length; i++) {
        if (defualtBBox[i] !== bbox[i]) {
            return false;
        }
    }
    return true;
}
function getRangeSize(range) {
    return range[1] - range[0] + 1;
}
function isRangeSafe(range, threshold) {
    const ret = range[1] >= range[0] && range[1] < threshold;
    if (!ret) {
        console.warn('Distance Expression: Index is out of range');
    }
    return ret;
}
function splitRange(range, isLine) {
    if (range[0] > range[1])
        return [
            null,
            null
        ];
    const size = getRangeSize(range);
    if (isLine) {
        if (size === 2) {
            return [
                range,
                null
            ];
        }
        const size1 = Math.floor(size / 2);
        const range1 = [
            range[0],
            range[0] + size1
        ];
        const range2 = [
            range[0] + size1,
            range[1]
        ];
        return [
            range1,
            range2
        ];
    } else {
        if (size === 1) {
            return [
                range,
                null
            ];
        }
        const size1 = Math.floor(size / 2) - 1;
        const range1 = [
            range[0],
            range[0] + size1
        ];
        const range2 = [
            range[0] + size1 + 1,
            range[1]
        ];
        return [
            range1,
            range2
        ];
    }
}
function getBBox(pointSets, range) {
    const bbox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    if (!isRangeSafe(range, pointSets.length))
        return bbox;
    for (let i = range[0]; i <= range[1]; ++i) {
        updateBBox(bbox, pointSets[i]);
    }
    return bbox;
}
function getPolygonBBox(polygon) {
    const bbox = [
        Infinity,
        Infinity,
        -Infinity,
        -Infinity
    ];
    for (let i = 0; i < polygon.length; ++i) {
        for (let j = 0; j < polygon[i].length; ++j) {
            updateBBox(bbox, polygon[i][j]);
        }
    }
    return bbox;
}
function bboxToBBoxDistance(bbox1, bbox2, ruler) {
    if (isDefaultBBOX(bbox1) || isDefaultBBOX(bbox2)) {
        return NaN;
    }
    let dx = 0;
    let dy = 0;
    if (bbox1[2] < bbox2[0]) {
        dx = bbox2[0] - bbox1[2];
    }
    if (bbox1[0] > bbox2[2]) {
        dx = bbox1[0] - bbox2[2];
    }
    if (bbox1[1] > bbox2[3]) {
        dy = bbox1[1] - bbox2[3];
    }
    if (bbox1[3] < bbox2[1]) {
        dy = bbox2[1] - bbox1[3];
    }
    return ruler.distance([
        0,
        0
    ], [
        dx,
        dy
    ]);
}
function lngFromMercatorX$1(x) {
    return x * 360 - 180;
}
function latFromMercatorY$1(y) {
    const y2 = 180 - y * 360;
    return 360 / Math.PI * Math.atan(Math.exp(y2 * Math.PI / 180)) - 90;
}
function getLngLatPoint(coord, canonical) {
    const tilesAtZoom = Math.pow(2, canonical.z);
    const x = (coord.x / EXTENT + canonical.x) / tilesAtZoom;
    const y = (coord.y / EXTENT + canonical.y) / tilesAtZoom;
    return [
        lngFromMercatorX$1(x),
        latFromMercatorY$1(y)
    ];
}
function getLngLatPoints(coordinates, canonical) {
    const coords = [];
    for (let i = 0; i < coordinates.length; ++i) {
        coords.push(getLngLatPoint(coordinates[i], canonical));
    }
    return coords;
}
function pointToLineDistance(point, line, ruler) {
    const nearestPoint = ruler.pointOnLine(line, point).point;
    return ruler.distance(point, nearestPoint);
}
function pointsToLineDistance(points, rangeA, line, rangeB, ruler) {
    const subLine = line.slice(rangeB[0], rangeB[1] + 1);
    let dist = Infinity;
    for (let i = rangeA[0]; i <= rangeA[1]; ++i) {
        if ((dist = Math.min(dist, pointToLineDistance(points[i], subLine, ruler))) === 0)
            return 0;
    }
    return dist;
}
function segmentToSegmentDistance(p1, p2, q1, q2, ruler) {
    const dist1 = Math.min(// @ts-expect-error - TS2345 - Argument of type 'Position' is not assignable to parameter of type 'Point'.
    ruler.pointToSegmentDistance(p1, q1, q2), // @ts-expect-error - TS2345 - Argument of type 'Position' is not assignable to parameter of type 'Point'.
    ruler.pointToSegmentDistance(p2, q1, q2));
    const dist2 = Math.min(// @ts-expect-error - TS2345 - Argument of type 'Position' is not assignable to parameter of type 'Point'.
    ruler.pointToSegmentDistance(q1, p1, p2), // @ts-expect-error - TS2345 - Argument of type 'Position' is not assignable to parameter of type 'Point'.
    ruler.pointToSegmentDistance(q2, p1, p2));
    return Math.min(dist1, dist2);
}
function lineToLineDistance(line1, range1, line2, range2, ruler) {
    if (!isRangeSafe(range1, line1.length) || !isRangeSafe(range2, line2.length)) {
        return NaN;
    }
    let dist = Infinity;
    for (let i = range1[0]; i < range1[1]; ++i) {
        for (let j = range2[0]; j < range2[1]; ++j) {
            if (segmentIntersectSegment(line1[i], line1[i + 1], line2[j], line2[j + 1]))
                return 0;
            dist = Math.min(dist, segmentToSegmentDistance(line1[i], line1[i + 1], line2[j], line2[j + 1], ruler));
        }
    }
    return dist;
}
function pointsToPointsDistance(pointSet1, range1, pointSet2, range2, ruler) {
    if (!isRangeSafe(range1, pointSet1.length) || !isRangeSafe(range2, pointSet2.length)) {
        return NaN;
    }
    let dist = Infinity;
    for (let i = range1[0]; i <= range1[1]; ++i) {
        for (let j = range2[0]; j <= range2[1]; ++j) {
            if ((dist = Math.min(dist, ruler.distance(pointSet1[i], pointSet2[j]))) === 0)
                return dist;
        }
    }
    return dist;
}
function pointToPolygonDistance(point, polygon, ruler) {
    if (pointWithinPolygon(point, polygon, true    /*trueOnBoundary*/))
        return 0;
    let dist = Infinity;
    for (const ring of polygon) {
        const ringLen = ring.length;
        if (ringLen < 2) {
            console.warn('Distance Expression: Invalid polygon!');
            return NaN;
        }
        if (ring[0] !== ring[ringLen - 1]) {
            if ((dist = Math.min(dist, ruler.pointToSegmentDistance(point, ring[ringLen - 1], ring[0]))) === 0)
                return dist;
        }
        if ((dist = Math.min(dist, pointToLineDistance(point, ring, ruler))) === 0)
            return dist;
    }
    return dist;
}
function lineToPolygonDistance(line, range, polygon, ruler) {
    if (!isRangeSafe(range, line.length)) {
        return NaN;
    }
    for (let i = range[0]; i <= range[1]; ++i) {
        if (pointWithinPolygon(line[i], polygon, true    /*trueOnBoundary*/))
            return 0;
    }
    let dist = Infinity;
    for (let i = range[0]; i < range[1]; ++i) {
        for (const ring of polygon) {
            for (let j = 0, len = ring.length, k = len - 1; j < len; k = j++) {
                if (segmentIntersectSegment(line[i], line[i + 1], ring[k], ring[j]))
                    return 0;
                dist = Math.min(dist, segmentToSegmentDistance(line[i], line[i + 1], ring[k], ring[j], ruler));
            }
        }
    }
    return dist;
}
function polygonIntersect(polygon1, polygon2) {
    for (const ring of polygon1) {
        for (let i = 0; i <= ring.length - 1; ++i) {
            if (pointWithinPolygon(ring[i], polygon2, true    /*trueOnBoundary*/))
                return true;
        }
    }
    return false;
}
function polygonToPolygonDistance(polygon1, polygon2, ruler, currentMiniDist = Infinity) {
    const bbox1 = getPolygonBBox(polygon1);
    const bbox2 = getPolygonBBox(polygon2);
    if (currentMiniDist !== Infinity && bboxToBBoxDistance(bbox1, bbox2, ruler) >= currentMiniDist) {
        return currentMiniDist;
    }
    if (boxWithinBox(bbox1, bbox2)) {
        if (polygonIntersect(polygon1, polygon2))
            return 0;
    } else if (polygonIntersect(polygon2, polygon1)) {
        return 0;
    }
    let dist = currentMiniDist;
    for (const ring1 of polygon1) {
        for (let i = 0, len1 = ring1.length, l = len1 - 1; i < len1; l = i++) {
            for (const ring2 of polygon2) {
                for (let j = 0, len2 = ring2.length, k = len2 - 1; j < len2; k = j++) {
                    if (segmentIntersectSegment(ring1[l], ring1[i], ring2[k], ring2[j]))
                        return 0;
                    dist = Math.min(dist, segmentToSegmentDistance(ring1[l], ring1[i], ring2[k], ring2[j], ruler));
                }
            }
        }
    }
    return dist;
}
function updateQueue(distQueue, miniDist, ruler, pointSet1, pointSet2, r1, r2) {
    if (r1 === null || r2 === null)
        return;
    const tempDist = bboxToBBoxDistance(getBBox(pointSet1, r1), getBBox(pointSet2, r2), ruler);
    if (tempDist < miniDist)
        distQueue.push({
            dist: tempDist,
            range1: r1,
            range2: r2
        });
}
function pointSetToPolygonDistance(pointSets, isLine, polygon, ruler, currentMiniDist = Infinity) {
    let miniDist = Math.min(ruler.distance(pointSets[0], polygon[0][0]), currentMiniDist);
    if (miniDist === 0)
        return miniDist;
    const initialDistPair = {
        dist: 0,
        range1: [
            0,
            pointSets.length - 1
        ],
        range2: [
            0,
            0
        ]
    };
    const distQueue = new TinyQueue([initialDistPair], compareMax$1);
    const setThreshold = isLine ? MIN_LINE_POINT_SIZE : MIN_POINT_SIZE;
    const polyBBox = getPolygonBBox(polygon);
    while (distQueue.length) {
        const distPair = distQueue.pop();
        if (distPair.dist >= miniDist)
            continue;
        const range = distPair.range1;
        if (getRangeSize(range) <= setThreshold) {
            if (!isRangeSafe(range, pointSets.length))
                return NaN;
            if (isLine) {
                const tempDist = lineToPolygonDistance(pointSets, range, polygon, ruler);
                if ((miniDist = Math.min(miniDist, tempDist)) === 0)
                    return miniDist;
            } else {
                for (let i = range[0]; i <= range[1]; ++i) {
                    const tempDist = pointToPolygonDistance(pointSets[i], polygon, ruler);
                    if ((miniDist = Math.min(miniDist, tempDist)) === 0)
                        return miniDist;
                }
            }
        } else {
            const newRanges = splitRange(range, isLine);
            if (newRanges[0] !== null) {
                const tempDist = bboxToBBoxDistance(getBBox(pointSets, newRanges[0]), polyBBox, ruler);
                if (tempDist < miniDist)
                    distQueue.push({
                        dist: tempDist,
                        range1: newRanges[0],
                        range2: [
                            0,
                            0
                        ]
                    });
            }
            if (newRanges[1] !== null) {
                const tempDist = bboxToBBoxDistance(getBBox(pointSets, newRanges[1]), polyBBox, ruler);
                if (tempDist < miniDist)
                    distQueue.push({
                        dist: tempDist,
                        range1: newRanges[1],
                        range2: [
                            0,
                            0
                        ]
                    });
            }
        }
    }
    return miniDist;
}
function pointSetsDistance(pointSet1, isLine1, pointSet2, isLine2, ruler, currentMiniDist = Infinity) {
    let miniDist = Math.min(currentMiniDist, ruler.distance(pointSet1[0], pointSet2[0]));
    if (miniDist === 0)
        return miniDist;
    const initialDistPair = {
        dist: 0,
        range1: [
            0,
            pointSet1.length - 1
        ],
        range2: [
            0,
            pointSet2.length - 1
        ]
    };
    const distQueue = new TinyQueue([initialDistPair], compareMax$1);
    const set1Threshold = isLine1 ? MIN_LINE_POINT_SIZE : MIN_POINT_SIZE;
    const set2Threshold = isLine2 ? MIN_LINE_POINT_SIZE : MIN_POINT_SIZE;
    while (distQueue.length) {
        const distPair = distQueue.pop();
        if (distPair.dist >= miniDist)
            continue;
        const rangeA = distPair.range1;
        const rangeB = distPair.range2;
        if (getRangeSize(rangeA) <= set1Threshold && getRangeSize(rangeB) <= set2Threshold) {
            if (!isRangeSafe(rangeA, pointSet1.length) || !isRangeSafe(rangeB, pointSet2.length)) {
                return NaN;
            }
            if (isLine1 && isLine2) {
                miniDist = Math.min(miniDist, lineToLineDistance(pointSet1, rangeA, pointSet2, rangeB, ruler));
            } else if (!isLine1 && !isLine2) {
                miniDist = Math.min(miniDist, pointsToPointsDistance(pointSet1, rangeA, pointSet2, rangeB, ruler));
            } else if (isLine1 && !isLine2) {
                miniDist = Math.min(miniDist, pointsToLineDistance(pointSet2, rangeB, pointSet1, rangeA, ruler));
            } else if (!isLine1 && isLine2) {
                miniDist = Math.min(miniDist, pointsToLineDistance(pointSet1, rangeA, pointSet2, rangeB, ruler));
            }
            if (miniDist === 0)
                return miniDist;
        } else {
            const newRangesA = splitRange(rangeA, isLine1);
            const newRangesB = splitRange(rangeB, isLine2);
            updateQueue(distQueue, miniDist, ruler, pointSet1, pointSet2, newRangesA[0], newRangesB[0]);
            updateQueue(distQueue, miniDist, ruler, pointSet1, pointSet2, newRangesA[0], newRangesB[1]);
            updateQueue(distQueue, miniDist, ruler, pointSet1, pointSet2, newRangesA[1], newRangesB[0]);
            updateQueue(distQueue, miniDist, ruler, pointSet1, pointSet2, newRangesA[1], newRangesB[1]);
        }
    }
    return miniDist;
}
function pointSetToLinesDistance(pointSet, isLine, lines, ruler, currentMiniDist = Infinity) {
    let dist = currentMiniDist;
    const bbox1 = getBBox(pointSet, [
        0,
        pointSet.length - 1
    ]);
    for (const line of lines) {
        if (dist !== Infinity && bboxToBBoxDistance(bbox1, getBBox(line, [
                0,
                line.length - 1
            ]), ruler) >= dist)
            continue;
        dist = Math.min(dist, pointSetsDistance(pointSet, isLine, line, true, ruler, dist));
        if (dist === 0)
            return dist;
    }
    return dist;
}
function pointSetToPolygonsDistance(points, isLine, polygons, ruler, currentMiniDist = Infinity) {
    let dist = currentMiniDist;
    const bbox1 = getBBox(points, [
        0,
        points.length - 1
    ]);
    for (const polygon of polygons) {
        if (dist !== Infinity && bboxToBBoxDistance(bbox1, getPolygonBBox(polygon), ruler) >= dist)
            continue;
        const tempDist = pointSetToPolygonDistance(points, isLine, polygon, ruler, dist);
        if (isNaN(tempDist))
            return tempDist;
        if ((dist = Math.min(dist, tempDist)) === 0)
            return dist;
    }
    return dist;
}
function polygonsToPolygonsDistance(polygons1, polygons2, ruler) {
    let dist = Infinity;
    for (const polygon1 of polygons1) {
        for (const polygon2 of polygons2) {
            const tempDist = polygonToPolygonDistance(polygon1, polygon2, ruler, dist);
            if (isNaN(tempDist))
                return tempDist;
            if ((dist = Math.min(dist, tempDist)) === 0)
                return dist;
        }
    }
    return dist;
}
function pointsToGeometryDistance(originGeometry, canonical, geometry) {
    const lngLatPoints = [];
    for (const points of originGeometry) {
        for (const point of points) {
            lngLatPoints.push(getLngLatPoint(point, canonical));
        }
    }
    const ruler = new CheapRuler(lngLatPoints[0][1], 'meters');
    if (geometry.type === 'Point' || geometry.type === 'MultiPoint' || geometry.type === 'LineString') {
        return pointSetsDistance(lngLatPoints, false, geometry.type === 'Point' ? [geometry.coordinates] : geometry.coordinates, geometry.type === 'LineString', ruler);
    }
    if (geometry.type === 'MultiLineString') {
        return pointSetToLinesDistance(lngLatPoints, false, geometry.coordinates, ruler);
    }
    if (geometry.type === 'Polygon' || geometry.type === 'MultiPolygon') {
        return pointSetToPolygonsDistance(lngLatPoints, false, geometry.type === 'Polygon' ? [geometry.coordinates] : geometry.coordinates, ruler);
    }
    return null;
}
function linesToGeometryDistance(originGeometry, canonical, geometry) {
    const lngLatLines = [];
    for (const line of originGeometry) {
        const lngLatLine = [];
        for (const point of line) {
            lngLatLine.push(getLngLatPoint(point, canonical));
        }
        lngLatLines.push(lngLatLine);
    }
    const ruler = new CheapRuler(lngLatLines[0][0][1], 'meters');
    if (geometry.type === 'Point' || geometry.type === 'MultiPoint' || geometry.type === 'LineString') {
        return pointSetToLinesDistance(geometry.type === 'Point' ? [geometry.coordinates] : geometry.coordinates, geometry.type === 'LineString', lngLatLines, ruler);
    }
    if (geometry.type === 'MultiLineString') {
        let dist = Infinity;
        for (let i = 0; i < geometry.coordinates.length; i++) {
            const tempDist = pointSetToLinesDistance(geometry.coordinates[i], true, lngLatLines, ruler, dist);
            if (isNaN(tempDist))
                return tempDist;
            if ((dist = Math.min(dist, tempDist)) === 0)
                return dist;
        }
        return dist;
    }
    if (geometry.type === 'Polygon' || geometry.type === 'MultiPolygon') {
        let dist = Infinity;
        for (let i = 0; i < lngLatLines.length; i++) {
            const tempDist = pointSetToPolygonsDistance(lngLatLines[i], true, geometry.type === 'Polygon' ? [geometry.coordinates] : geometry.coordinates, ruler, dist);
            if (isNaN(tempDist))
                return tempDist;
            if ((dist = Math.min(dist, tempDist)) === 0)
                return dist;
        }
        return dist;
    }
    return null;
}
function polygonsToGeometryDistance(originGeometry, canonical, geometry) {
    const lngLatPolygons = [];
    for (const polygon of classifyRings$1(originGeometry)) {
        const lngLatPolygon = [];
        for (let i = 0; i < polygon.length; ++i) {
            lngLatPolygon.push(getLngLatPoints(polygon[i], canonical));
        }
        lngLatPolygons.push(lngLatPolygon);
    }
    const ruler = new CheapRuler(lngLatPolygons[0][0][0][1], 'meters');
    if (geometry.type === 'Point' || geometry.type === 'MultiPoint' || geometry.type === 'LineString') {
        return pointSetToPolygonsDistance(geometry.type === 'Point' ? [geometry.coordinates] : geometry.coordinates, geometry.type === 'LineString', lngLatPolygons, ruler);
    }
    if (geometry.type === 'MultiLineString') {
        let dist = Infinity;
        for (let i = 0; i < geometry.coordinates.length; i++) {
            const tempDist = pointSetToPolygonsDistance(geometry.coordinates[i], true, lngLatPolygons, ruler, dist);
            if (isNaN(tempDist))
                return tempDist;
            if ((dist = Math.min(dist, tempDist)) === 0)
                return dist;
        }
        return dist;
    }
    if (geometry.type === 'Polygon' || geometry.type === 'MultiPolygon') {
        return polygonsToPolygonsDistance(geometry.type === 'Polygon' ? [geometry.coordinates] : geometry.coordinates, lngLatPolygons, ruler);
    }
    return null;
}
function isTypeValid(type) {
    return type === 'Point' || type === 'MultiPoint' || type === 'LineString' || type === 'MultiLineString' || type === 'Polygon' || type === 'MultiPolygon';
}
class Distance {
    constructor(geojson, geometries) {
        this.type = NumberType;
        this.geojson = geojson;
        this.geometries = geometries;
    }
    static parse(args, context) {
        if (args.length !== 2) {
            return context.error(`'distance' expression requires either one argument, but found ' ${ args.length - 1 } instead.`);
        }
        if (isValue(args[1])) {
            const geojson = args[1];
            if (geojson.type === 'FeatureCollection') {
                for (let i = 0; i < geojson.features.length; ++i) {
                    if (isTypeValid(geojson.features[i].geometry.type)) {
                        return new Distance(geojson, geojson.features[i].geometry);
                    }
                }
            } else if (geojson.type === 'Feature') {
                if (isTypeValid(geojson.geometry.type)) {
                    return new Distance(geojson, geojson.geometry);
                }
            } else if (isTypeValid(geojson.type)) {
                return new Distance(geojson, geojson);
            }
        }
        return context.error('\'distance\' expression needs to be an array with format [\'Distance\', GeoJSONObj].');
    }
    evaluate(ctx) {
        const geometry = ctx.geometry();
        const canonical = ctx.canonicalID();
        if (geometry != null && canonical != null) {
            if (ctx.geometryType() === 'Point') {
                return pointsToGeometryDistance(geometry, canonical, this.geometries);
            }
            if (ctx.geometryType() === 'LineString') {
                return linesToGeometryDistance(geometry, canonical, this.geometries);
            }
            if (ctx.geometryType() === 'Polygon') {
                return polygonsToGeometryDistance(geometry, canonical, this.geometries);
            }
            console.warn('Distance Expression: currently only evaluates valid Point/LineString/Polygon geometries.');
        } else {
            console.warn('Distance Expression: requirs valid feature and canonical information.');
        }
        return null;
    }
    eachChild() {
    }
    outputDefined() {
        return true;
    }
    serialize() {
        return [
            'distance',
            this.geojson
        ];
    }
}

function coerceValue(type, value) {
    switch (type) {
    case 'string':
        return toString(value);
    case 'number':
        return +value;
    case 'boolean':
        return !!value;
    case 'color':
        return Color.parse(value);
    case 'formatted': {
            return Formatted.fromString(toString(value));
        }
    case 'resolvedImage': {
            return ResolvedImage.build(toString(value));
        }
    }
    return value;
}
function clampToAllowedNumber(value, min, max, step) {
    if (step !== void 0) {
        value = step * Math.round(value / step);
    }
    if (min !== void 0 && value < min) {
        value = min;
    }
    if (max !== void 0 && value > max) {
        value = max;
    }
    return value;
}
class Config {
    constructor(type, key, scope) {
        this.type = type;
        this.key = key;
        this.scope = scope;
    }
    static parse(args, context) {
        let type = context.expectedType;
        if (type === null || type === void 0) {
            type = ValueType;
        }
        if (args.length < 2 || args.length > 3) {
            return context.error(`Invalid number of arguments for 'config' expression.`);
        }
        const configKey = context.parse(args[1], 1);
        if (!(configKey instanceof Literal)) {
            return context.error(`Key name of 'config' expression must be a string literal.`);
        }
        if (args.length >= 3) {
            const configScope = context.parse(args[2], 2);
            if (!(configScope instanceof Literal)) {
                return context.error(`Scope of 'config' expression must be a string literal.`);
            }
            return new Config(type, toString(configKey.value), toString(configScope.value));
        }
        return new Config(type, toString(configKey.value));
    }
    evaluate(ctx) {
        const FQIDSeparator = '\x1F';
        const configKey = [
            this.key,
            this.scope,
            ctx.scope
        ].filter(Boolean).join(FQIDSeparator);
        const config = ctx.getConfig(configKey);
        if (!config)
            return null;
        const {type, value, values, minValue, maxValue, stepValue} = config;
        const defaultValue = config.default.evaluate(ctx);
        let result = defaultValue;
        if (value) {
            const originalScope = ctx.scope;
            ctx.scope = (originalScope || '').split(FQIDSeparator).slice(1).join(FQIDSeparator);
            result = value.evaluate(ctx);
            ctx.scope = originalScope;
        }
        if (type) {
            result = coerceValue(type, result);
        }
        if (result !== void 0 && (minValue !== void 0 || maxValue !== void 0 || stepValue !== void 0)) {
            if (typeof result === 'number') {
                result = clampToAllowedNumber(result, minValue, maxValue, stepValue);
            } else if (Array.isArray(result)) {
                result = result.map(item => typeof item === 'number' ? clampToAllowedNumber(item, minValue, maxValue, stepValue) : item);
            }
        }
        if (value !== void 0 && result !== void 0 && values && !values.includes(result)) {
            result = defaultValue;
            if (type) {
                result = coerceValue(type, result);
            }
        }
        if (type && type !== this.type || result !== void 0 && typeOf(result) !== this.type) {
            result = coerceValue(this.type.kind, result);
        }
        return result;
    }
    eachChild() {
    }
    outputDefined() {
        return false;
    }
    serialize() {
        const res = [
            'config',
            this.key
        ];
        if (this.scope) {
            res.concat(this.key);
        }
        return res;
    }
}

function isFeatureConstant(e) {
    if (e instanceof CompoundExpression) {
        if (e.name === 'get' && e.args.length === 1) {
            return false;
        } else if (e.name === 'feature-state') {
            return false;
        } else if (e.name === 'has' && e.args.length === 1) {
            return false;
        } else if (e.name === 'properties' || e.name === 'geometry-type' || e.name === 'id') {
            return false;
        } else if (/^filter-/.test(e.name)) {
            return false;
        }
    }
    if (e instanceof Within) {
        return false;
    }
    if (e instanceof Distance) {
        return false;
    }
    let result = true;
    e.eachChild(arg => {
        if (result && !isFeatureConstant(arg)) {
            result = false;
        }
    });
    return result;
}
function isStateConstant(e) {
    if (e instanceof CompoundExpression) {
        if (e.name === 'feature-state') {
            return false;
        }
    }
    let result = true;
    e.eachChild(arg => {
        if (result && !isStateConstant(arg)) {
            result = false;
        }
    });
    return result;
}
function getConfigDependencies(e) {
    if (e instanceof Config) {
        const singleConfig = /* @__PURE__ */
        new Set([e.key]);
        return singleConfig;
    }
    let result = /* @__PURE__ */
    new Set();
    e.eachChild(arg => {
        result = /* @__PURE__ */
        new Set([
            ...result,
            ...getConfigDependencies(arg)
        ]);
    });
    return result;
}
function isGlobalPropertyConstant(e, properties) {
    if (e instanceof CompoundExpression && properties.indexOf(e.name) >= 0) {
        return false;
    }
    let result = true;
    e.eachChild(arg => {
        if (result && !isGlobalPropertyConstant(arg, properties)) {
            result = false;
        }
    });
    return result;
}

class Var {
    constructor(name, boundExpression) {
        this.type = boundExpression.type;
        this.name = name;
        this.boundExpression = boundExpression;
    }
    static parse(args, context) {
        if (args.length !== 2 || typeof args[1] !== 'string')
            return context.error(`'var' expression requires exactly one string literal argument.`);
        const name = args[1];
        if (!context.scope.has(name)) {
            return context.error(`Unknown variable "${ name }". Make sure "${ name }" has been bound in an enclosing "let" expression before using it.`, 1);
        }
        return new Var(name, context.scope.get(name));
    }
    evaluate(ctx) {
        return this.boundExpression.evaluate(ctx);
    }
    eachChild() {
    }
    outputDefined() {
        return false;
    }
    serialize() {
        return [
            'var',
            this.name
        ];
    }
}

class ParsingContext {
    constructor(registry, path = [], expectedType, scope = new Scope(), errors = [], _scope, options) {
        this.registry = registry;
        this.path = path;
        this.key = path.map(part => {
            if (typeof part === 'string') {
                return `['${ part }']`;
            }
            return `[${ part }]`;
        }).join('');
        this.scope = scope;
        this.errors = errors;
        this.expectedType = expectedType;
        this._scope = _scope;
        this.options = options;
    }
    /**
   * @param expr the JSON expression to parse
   * @param index the optional argument index if this expression is an argument of a parent expression that's being parsed
   * @param options
   * @param options.omitTypeAnnotations set true to omit inferred type annotations.  Caller beware: with this option set, the parsed expression's type will NOT satisfy `expectedType` if it would normally be wrapped in an inferred annotation.
   * @private
   */
    parse(expr, index, expectedType, bindings, options = {}) {
        if (index || expectedType) {
            return this.concat(index, null, expectedType, bindings)._parse(expr, options);
        }
        return this._parse(expr, options);
    }
    /**
   * @param expr the JSON expression to parse
   * @param index the optional argument index if parent object being is an argument of another expression
   * @param key key of parent object being parsed
   * @param options
   * @param options.omitTypeAnnotations set true to omit inferred type annotations.  Caller beware: with this option set, the parsed expression's type will NOT satisfy `expectedType` if it would normally be wrapped in an inferred annotation.
   * @private
   */
    parseObjectValue(expr, index, key, expectedType, bindings, options = {}) {
        return this.concat(index, key, expectedType, bindings)._parse(expr, options);
    }
    _parse(expr, options) {
        if (expr === null || typeof expr === 'string' || typeof expr === 'boolean' || typeof expr === 'number') {
            expr = [
                'literal',
                expr
            ];
        }
        function annotate(parsed, type, typeAnnotation) {
            if (typeAnnotation === 'assert') {
                return new Assertion(type, [parsed]);
            } else if (typeAnnotation === 'coerce') {
                return new Coercion(type, [parsed]);
            } else {
                return parsed;
            }
        }
        if (Array.isArray(expr)) {
            if (expr.length === 0) {
                return this.error(`Expected an array with at least one element. If you wanted a literal array, use ["literal", []].`);
            }
            const Expr = typeof expr[0] === 'string' ? this.registry[expr[0]] : void 0;
            if (Expr) {
                let parsed = Expr.parse(expr, this);
                if (!parsed)
                    return null;
                if (this.expectedType) {
                    const expected = this.expectedType;
                    const actual = parsed.type;
                    if ((expected.kind === 'string' || expected.kind === 'number' || expected.kind === 'boolean' || expected.kind === 'object' || expected.kind === 'array') && actual.kind === 'value') {
                        parsed = annotate(parsed, expected, options.typeAnnotation || 'assert');
                    } else if ((expected.kind === 'color' || expected.kind === 'formatted' || expected.kind === 'resolvedImage') && (actual.kind === 'value' || actual.kind === 'string')) {
                        parsed = annotate(parsed, expected, options.typeAnnotation || 'coerce');
                    } else if (this.checkSubtype(expected, actual)) {
                        return null;
                    }
                }
                if (!(parsed instanceof Literal) && parsed.type.kind !== 'resolvedImage' && isConstant(parsed)) {
                    const ec = new EvaluationContext(this._scope, this.options);
                    try {
                        parsed = new Literal(parsed.type, parsed.evaluate(ec));
                    } catch (e) {
                        this.error(e.message);
                        return null;
                    }
                }
                return parsed;
            }
            return Coercion.parse([
                'to-array',
                expr
            ], this);
        } else if (typeof expr === 'undefined') {
            return this.error(`'undefined' value invalid. Use null instead.`);
        } else if (typeof expr === 'object') {
            return this.error(`Bare objects invalid. Use ["literal", {...}] instead.`);
        } else {
            return this.error(`Expected an array, but found ${ typeof expr } instead.`);
        }
    }
    /**
   * Returns a copy of this context suitable for parsing the subexpression at
   * index `index`, optionally appending to 'let' binding map.
   *
   * Note that `errors` property, intended for collecting errors while
   * parsing, is copied by reference rather than cloned.
   * @private
   */
    concat(index, key, expectedType, bindings) {
        let path = typeof index === 'number' ? this.path.concat(index) : this.path;
        path = typeof key === 'string' ? path.concat(key) : path;
        const scope = bindings ? this.scope.concat(bindings) : this.scope;
        return new ParsingContext(this.registry, path, expectedType || null, scope, this.errors, this._scope, this.options);
    }
    /**
   * Push a parsing (or type checking) error into the `this.errors`
   * @param error The message
   * @param keys Optionally specify the source of the error at a child
   * of the current expression at `this.key`.
   * @private
   */
    error(error, ...keys) {
        const key = `${ this.key }${ keys.map(k => `[${ k }]`).join('') }`;
        this.errors.push(new ParsingError(key, error));
    }
    /**
   * Returns null if `t` is a subtype of `expected`; otherwise returns an
   * error message and also pushes it to `this.errors`.
   */
    checkSubtype(expected, t) {
        const error = checkSubtype(expected, t);
        if (error)
            this.error(error);
        return error;
    }
}
var ParsingContext$1 = ParsingContext;
function isConstant(expression) {
    if (expression instanceof Var) {
        return isConstant(expression.boundExpression);
    } else if (expression instanceof CompoundExpression && expression.name === 'error') {
        return false;
    } else if (expression instanceof CollatorExpression) {
        return false;
    } else if (expression instanceof Within) {
        return false;
    } else if (expression instanceof Distance) {
        return false;
    } else if (expression instanceof Config) {
        return false;
    }
    const isTypeAnnotation = expression instanceof Coercion || expression instanceof Assertion;
    let childrenConstant = true;
    expression.eachChild(child => {
        if (isTypeAnnotation) {
            childrenConstant = childrenConstant && isConstant(child);
        } else {
            childrenConstant = childrenConstant && child instanceof Literal;
        }
    });
    if (!childrenConstant) {
        return false;
    }
    return isFeatureConstant(expression) && isGlobalPropertyConstant(expression, [
        'zoom',
        'heatmap-density',
        'line-progress',
        'raster-value',
        'sky-radial-progress',
        'accumulated',
        'is-supported-script',
        'pitch',
        'distance-from-center',
        'measure-light',
        'raster-particle-speed'
    ]);
}

function findStopLessThanOrEqualTo(stops, input) {
    const lastIndex = stops.length - 1;
    let lowerIndex = 0;
    let upperIndex = lastIndex;
    let currentIndex = 0;
    let currentValue, nextValue;
    while (lowerIndex <= upperIndex) {
        currentIndex = Math.floor((lowerIndex + upperIndex) / 2);
        currentValue = stops[currentIndex];
        nextValue = stops[currentIndex + 1];
        if (currentValue <= input) {
            if (currentIndex === lastIndex || input < nextValue) {
                return currentIndex;
            }
            lowerIndex = currentIndex + 1;
        } else if (currentValue > input) {
            upperIndex = currentIndex - 1;
        } else {
            throw new RuntimeError('Input is not a number.');
        }
    }
    return 0;
}

class Step {
    constructor(type, input, stops) {
        this.type = type;
        this.input = input;
        this.labels = [];
        this.outputs = [];
        for (const [label, expression] of stops) {
            this.labels.push(label);
            this.outputs.push(expression);
        }
    }
    static parse(args, context) {
        if (args.length - 1 < 4) {
            return context.error(`Expected at least 4 arguments, but found only ${ args.length - 1 }.`);
        }
        if ((args.length - 1) % 2 !== 0) {
            return context.error(`Expected an even number of arguments.`);
        }
        const input = context.parse(args[1], 1, NumberType);
        if (!input)
            return null;
        const stops = [];
        let outputType = null;
        if (context.expectedType && context.expectedType.kind !== 'value') {
            outputType = context.expectedType;
        }
        for (let i = 1; i < args.length; i += 2) {
            const label = i === 1 ? -Infinity : args[i];
            const value = args[i + 1];
            const labelKey = i;
            const valueKey = i + 1;
            if (typeof label !== 'number') {
                return context.error('Input/output pairs for "step" expressions must be defined using literal numeric values (not computed expressions) for the input values.', labelKey);
            }
            if (stops.length && stops[stops.length - 1][0] >= label) {
                return context.error('Input/output pairs for "step" expressions must be arranged with input values in strictly ascending order.', labelKey);
            }
            const parsed = context.parse(value, valueKey, outputType);
            if (!parsed)
                return null;
            outputType = outputType || parsed.type;
            stops.push([
                label,
                parsed
            ]);
        }
        return new Step(outputType, input, stops);
    }
    evaluate(ctx) {
        const labels = this.labels;
        const outputs = this.outputs;
        if (labels.length === 1) {
            return outputs[0].evaluate(ctx);
        }
        const value = this.input.evaluate(ctx);
        if (value <= labels[0]) {
            return outputs[0].evaluate(ctx);
        }
        const stopCount = labels.length;
        if (value >= labels[stopCount - 1]) {
            return outputs[stopCount - 1].evaluate(ctx);
        }
        const index = findStopLessThanOrEqualTo(labels, value);
        return outputs[index].evaluate(ctx);
    }
    eachChild(fn) {
        fn(this.input);
        for (const expression of this.outputs) {
            fn(expression);
        }
    }
    outputDefined() {
        return this.outputs.every(out => out.outputDefined());
    }
    serialize() {
        const serialized = [
            'step',
            this.input.serialize()
        ];
        for (let i = 0; i < this.labels.length; i++) {
            if (i > 0) {
                serialized.push(this.labels[i]);
            }
            serialized.push(this.outputs[i].serialize());
        }
        return serialized;
    }
}

const Xn = 0.95047, Yn = 1, Zn = 1.08883, t0 = 4 / 29, t1 = 6 / 29, t2 = 3 * t1 * t1, t3 = t1 * t1 * t1, deg2rad = Math.PI / 180, rad2deg = 180 / Math.PI;
function xyz2lab(t) {
    return t > t3 ? Math.pow(t, 1 / 3) : t / t2 + t0;
}
function lab2xyz(t) {
    return t > t1 ? t * t * t : t2 * (t - t0);
}
function xyz2rgb(x) {
    return 255 * (x <= 0.0031308 ? 12.92 * x : 1.055 * Math.pow(x, 1 / 2.4) - 0.055);
}
function rgb2xyz(x) {
    x /= 255;
    return x <= 0.04045 ? x / 12.92 : Math.pow((x + 0.055) / 1.055, 2.4);
}
function rgbToLab(rgbColor) {
    const b = rgb2xyz(rgbColor.r), a = rgb2xyz(rgbColor.g), l = rgb2xyz(rgbColor.b), x = xyz2lab((0.4124564 * b + 0.3575761 * a + 0.1804375 * l) / Xn), y = xyz2lab((0.2126729 * b + 0.7151522 * a + 0.072175 * l) / Yn), z = xyz2lab((0.0193339 * b + 0.119192 * a + 0.9503041 * l) / Zn);
    return {
        l: 116 * y - 16,
        a: 500 * (x - y),
        b: 200 * (y - z),
        alpha: rgbColor.a
    };
}
function labToRgb(labColor) {
    let y = (labColor.l + 16) / 116, x = isNaN(labColor.a) ? y : y + labColor.a / 500, z = isNaN(labColor.b) ? y : y - labColor.b / 200;
    y = Yn * lab2xyz(y);
    x = Xn * lab2xyz(x);
    z = Zn * lab2xyz(z);
    return new Color(xyz2rgb(3.2404542 * x - 1.5371385 * y - 0.4985314 * z), // D65 -> sRGB
    xyz2rgb(-0.969266 * x + 1.8760108 * y + 0.041556 * z), xyz2rgb(0.0556434 * x - 0.2040259 * y + 1.0572252 * z), labColor.alpha);
}
function interpolateLab(from, to, t) {
    return {
        l: number(from.l, to.l, t),
        a: number(from.a, to.a, t),
        b: number(from.b, to.b, t),
        alpha: number(from.alpha, to.alpha, t)
    };
}
function rgbToHcl(rgbColor) {
    const {l, a, b} = rgbToLab(rgbColor);
    const h = Math.atan2(b, a) * rad2deg;
    return {
        h: h < 0 ? h + 360 : h,
        c: Math.sqrt(a * a + b * b),
        l,
        alpha: rgbColor.a
    };
}
function hclToRgb(hclColor) {
    const h = hclColor.h * deg2rad, c = hclColor.c, l = hclColor.l;
    return labToRgb({
        l,
        a: Math.cos(h) * c,
        b: Math.sin(h) * c,
        alpha: hclColor.alpha
    });
}
function interpolateHue(a, b, t) {
    const d = b - a;
    return a + t * (d > 180 || d < -180 ? d - 360 * Math.round(d / 360) : d);
}
function interpolateHcl(from, to, t) {
    return {
        h: interpolateHue(from.h, to.h, t),
        c: number(from.c, to.c, t),
        l: number(from.l, to.l, t),
        alpha: number(from.alpha, to.alpha, t)
    };
}
const lab = {
    forward: rgbToLab,
    reverse: labToRgb,
    interpolate: interpolateLab
};
const hcl = {
    forward: rgbToHcl,
    reverse: hclToRgb,
    interpolate: interpolateHcl
};

var colorSpaces = /*#__PURE__*/Object.freeze({
__proto__: null,
hcl: hcl,
lab: lab
});

class Interpolate {
    constructor(type, operator, interpolation, input, dynamicStops, stops) {
        this.type = type;
        this.operator = operator;
        this.interpolation = interpolation;
        this.input = input;
        this.dynamicStops = dynamicStops;
        this.labels = [];
        this.outputs = [];
        for (const [label, expression] of stops) {
            this.labels.push(label);
            this.outputs.push(expression);
        }
    }
    static interpolationFactor(interpolation, input, lower, upper) {
        let t = 0;
        if (interpolation.name === 'exponential') {
            t = exponentialInterpolation(input, interpolation.base, lower, upper);
        } else if (interpolation.name === 'linear') {
            t = exponentialInterpolation(input, 1, lower, upper);
        } else if (interpolation.name === 'cubic-bezier') {
            const c = interpolation.controlPoints;
            const ub = new UnitBezier(c[0], c[1], c[2], c[3]);
            t = ub.solve(exponentialInterpolation(input, 1, lower, upper));
        }
        return t;
    }
    static parse(args, context) {
        let [operator, interpolation, input, ...rest] = args;
        if (!Array.isArray(interpolation) || interpolation.length === 0) {
            return context.error(`Expected an interpolation type expression.`, 1);
        }
        if (interpolation[0] === 'linear') {
            interpolation = { name: 'linear' };
        } else if (interpolation[0] === 'exponential') {
            const base = interpolation[1];
            if (typeof base !== 'number')
                return context.error(`Exponential interpolation requires a numeric base.`, 1, 1);
            interpolation = {
                name: 'exponential',
                base
            };
        } else if (interpolation[0] === 'cubic-bezier') {
            const controlPoints = interpolation.slice(1);
            if (controlPoints.length !== 4 || controlPoints.some(t => typeof t !== 'number' || t < 0 || t > 1)) {
                return context.error('Cubic bezier interpolation requires four numeric arguments with values between 0 and 1.', 1);
            }
            interpolation = {
                name: 'cubic-bezier',
                controlPoints
            };
        } else {
            return context.error(`Unknown interpolation type ${ String(interpolation[0]) }`, 1, 0);
        }
        if (args.length - 1 < 3) {
            return context.error(`Expected at least 3 arguments, but found only ${ args.length - 1 }.`);
        }
        if (args.length - 1 > 3 && (args.length - 1) % 2 !== 0) {
            return context.error(`Expected an even number of arguments.`);
        }
        input = context.parse(input, 2, NumberType);
        if (!input)
            return null;
        const stops = [];
        let outputType = null;
        if (operator === 'interpolate-hcl' || operator === 'interpolate-lab') {
            outputType = ColorType;
        } else if (context.expectedType && context.expectedType.kind !== 'value') {
            outputType = context.expectedType;
        }
        if (args.length - 1 === 3) {
            const dynamicStops = context.parse(rest[0], 3, ValueType);
            if (!dynamicStops)
                return null;
            return new Interpolate(outputType, operator, interpolation, input, dynamicStops, stops);
        }
        for (let i = 0; i < rest.length; i += 2) {
            const label = rest[i];
            const value = rest[i + 1];
            const labelKey = i + 3;
            const valueKey = i + 4;
            if (typeof label !== 'number') {
                return context.error('Input/output pairs for "interpolate" expressions must be defined using literal numeric values (not computed expressions) for the input values.', labelKey);
            }
            if (stops.length && stops[stops.length - 1][0] >= label) {
                return context.error('Input/output pairs for "interpolate" expressions must be arranged with input values in strictly ascending order.', labelKey);
            }
            const parsed = context.parse(value, valueKey, outputType);
            if (!parsed)
                return null;
            outputType = outputType || parsed.type;
            stops.push([
                label,
                parsed
            ]);
        }
        if (outputType.kind !== 'number' && outputType.kind !== 'color' && !(outputType.kind === 'array' && outputType.itemType.kind === 'number' && typeof outputType.N === 'number')) {
            return context.error(`Type ${ toString$1(outputType) } is not interpolatable.`);
        }
        return new Interpolate(outputType, operator, interpolation, input, null, stops);
    }
    evaluate(ctx) {
        let labels = this.labels;
        let outputs = this.outputs;
        if (this.dynamicStops) {
            const dynamicStopsValue = this.dynamicStops.evaluate(ctx);
            if (dynamicStopsValue.length % 2 !== 0) {
                throw new RuntimeError('Expected an even number of arguments.');
            }
            labels = [];
            outputs = [];
            for (let i = 0; i < dynamicStopsValue.length; i += 2) {
                const label = dynamicStopsValue[i];
                const output = new Literal(NumberType, dynamicStopsValue[i + 1]);
                if (typeof label !== 'number') {
                    throw new RuntimeError('Input/output pairs for "interpolate" expressions must be defined using literal numeric values (not computed expressions) for the input values.');
                }
                if (labels.length && labels[labels.length - 1] >= label) {
                    throw new RuntimeError('Input/output pairs for "interpolate" expressions must be arranged with input values in strictly ascending order.');
                }
                labels.push(label);
                outputs.push(output);
            }
            if (labels.length === 0) {
                throw new RuntimeError('Expected at least one input/output pair.');
            }
        }
        if (labels.length === 1) {
            return outputs[0].evaluate(ctx);
        }
        const value = this.input.evaluate(ctx);
        if (value <= labels[0]) {
            return outputs[0].evaluate(ctx);
        }
        const stopCount = labels.length;
        if (value >= labels[stopCount - 1]) {
            return outputs[stopCount - 1].evaluate(ctx);
        }
        const index = findStopLessThanOrEqualTo(labels, value);
        const lower = labels[index];
        const upper = labels[index + 1];
        const t = Interpolate.interpolationFactor(this.interpolation, value, lower, upper);
        const outputLower = outputs[index].evaluate(ctx);
        const outputUpper = outputs[index + 1].evaluate(ctx);
        if (this.operator === 'interpolate') {
            return interpolate$1[this.type.kind.toLowerCase()](outputLower, outputUpper, t);
        } else if (this.operator === 'interpolate-hcl') {
            return hcl.reverse(hcl.interpolate(hcl.forward(outputLower), hcl.forward(outputUpper), t));
        } else {
            return lab.reverse(lab.interpolate(lab.forward(outputLower), lab.forward(outputUpper), t));
        }
    }
    eachChild(fn) {
        fn(this.input);
        for (const expression of this.outputs) {
            fn(expression);
        }
    }
    outputDefined() {
        return this.outputs.every(out => out.outputDefined());
    }
    serialize() {
        let interpolation;
        if (this.interpolation.name === 'linear') {
            interpolation = ['linear'];
        } else if (this.interpolation.name === 'exponential') {
            if (this.interpolation.base === 1) {
                interpolation = ['linear'];
            } else {
                interpolation = [
                    'exponential',
                    this.interpolation.base
                ];
            }
        } else {
            interpolation = ['cubic-bezier'].concat(this.interpolation.controlPoints);
        }
        const serialized = [
            this.operator,
            interpolation,
            this.input.serialize()
        ];
        if (this.dynamicStops) {
            serialized.push(this.dynamicStops.serialize());
        } else {
            for (let i = 0; i < this.labels.length; i++) {
                serialized.push(this.labels[i], this.outputs[i].serialize());
            }
        }
        return serialized;
    }
}
function exponentialInterpolation(input, base, lowerValue, upperValue) {
    const difference = upperValue - lowerValue;
    const progress = input - lowerValue;
    if (difference === 0) {
        return 0;
    } else if (base === 1) {
        return progress / difference;
    } else {
        return (Math.pow(base, progress) - 1) / (Math.pow(base, difference) - 1);
    }
}

class Coalesce {
    constructor(type, args) {
        this.type = type;
        this.args = args;
    }
    static parse(args, context) {
        if (args.length < 2) {
            return context.error('Expectected at least one argument.');
        }
        let outputType = null;
        const expectedType = context.expectedType;
        if (expectedType && expectedType.kind !== 'value') {
            outputType = expectedType;
        }
        const parsedArgs = [];
        for (const arg of args.slice(1)) {
            const parsed = context.parse(arg, 1 + parsedArgs.length, outputType, void 0, { typeAnnotation: 'omit' });
            if (!parsed)
                return null;
            outputType = outputType || parsed.type;
            parsedArgs.push(parsed);
        }
        const needsAnnotation = expectedType && parsedArgs.some(arg => checkSubtype(expectedType, arg.type));
        return needsAnnotation ? new Coalesce(ValueType, parsedArgs) : new Coalesce(outputType, parsedArgs);
    }
    evaluate(ctx) {
        let result = null;
        let argCount = 0;
        let firstImage;
        for (const arg of this.args) {
            argCount++;
            result = arg.evaluate(ctx);
            if (result && result instanceof ResolvedImage && !result.available) {
                if (!firstImage) {
                    firstImage = result;
                }
                result = null;
                if (argCount === this.args.length) {
                    return firstImage;
                }
            }
            if (result !== null)
                break;
        }
        return result;
    }
    eachChild(fn) {
        this.args.forEach(fn);
    }
    outputDefined() {
        return this.args.every(arg => arg.outputDefined());
    }
    serialize() {
        const serialized = ['coalesce'];
        this.eachChild(child => {
            serialized.push(child.serialize());
        });
        return serialized;
    }
}

class Let {
    constructor(bindings, result) {
        this.type = result.type;
        this.bindings = [].concat(bindings);
        this.result = result;
    }
    evaluate(ctx) {
        return this.result.evaluate(ctx);
    }
    eachChild(fn) {
        for (const binding of this.bindings) {
            fn(binding[1]);
        }
        fn(this.result);
    }
    static parse(args, context) {
        if (args.length < 4)
            return context.error(`Expected at least 3 arguments, but found ${ args.length - 1 } instead.`);
        const bindings = [];
        for (let i = 1; i < args.length - 1; i += 2) {
            const name = args[i];
            if (typeof name !== 'string') {
                return context.error(`Expected string, but found ${ typeof name } instead.`, i);
            }
            if (/[^a-zA-Z0-9_]/.test(name)) {
                return context.error(`Variable names must contain only alphanumeric characters or '_'.`, i);
            }
            const value = context.parse(args[i + 1], i + 1);
            if (!value)
                return null;
            bindings.push([
                name,
                value
            ]);
        }
        const result = context.parse(args[args.length - 1], args.length - 1, context.expectedType, bindings);
        if (!result)
            return null;
        return new Let(bindings, result);
    }
    outputDefined() {
        return this.result.outputDefined();
    }
    serialize() {
        const serialized = ['let'];
        for (const [name, expr] of this.bindings) {
            serialized.push(name, expr.serialize());
        }
        serialized.push(this.result.serialize());
        return serialized;
    }
}

class At {
    constructor(type, index, input) {
        this.type = type;
        this.index = index;
        this.input = input;
    }
    static parse(args, context) {
        if (args.length !== 3)
            return context.error(`Expected 2 arguments, but found ${ args.length - 1 } instead.`);
        const index = context.parse(args[1], 1, NumberType);
        const input = context.parse(args[2], 2, array(context.expectedType || ValueType));
        if (!index || !input)
            return null;
        const t = input.type;
        return new At(t.itemType, index, input);
    }
    evaluate(ctx) {
        const index = this.index.evaluate(ctx);
        const array2 = this.input.evaluate(ctx);
        if (index < 0) {
            throw new RuntimeError(`Array index out of bounds: ${ index } < 0.`);
        }
        if (index > array2.length - 1) {
            throw new RuntimeError(`Array index out of bounds: ${ index } > ${ array2.length - 1 }.`);
        }
        if (index === Math.floor(index)) {
            return array2[index];
        }
        const lowerIndex = Math.floor(index);
        const upperIndex = Math.ceil(index);
        const lowerValue = array2[lowerIndex];
        const upperValue = array2[upperIndex];
        if (typeof lowerValue !== 'number' || typeof upperValue !== 'number') {
            throw new RuntimeError(`Cannot interpolate between non-number values at index ${ index }.`);
        }
        const fraction = index - lowerIndex;
        return lowerValue * (1 - fraction) + upperValue * fraction;
    }
    eachChild(fn) {
        fn(this.index);
        fn(this.input);
    }
    outputDefined() {
        return false;
    }
    serialize() {
        return [
            'at',
            this.index.serialize(),
            this.input.serialize()
        ];
    }
}

class In {
    constructor(needle, haystack) {
        this.type = BooleanType;
        this.needle = needle;
        this.haystack = haystack;
    }
    static parse(args, context) {
        if (args.length !== 3) {
            return context.error(`Expected 2 arguments, but found ${ args.length - 1 } instead.`);
        }
        const needle = context.parse(args[1], 1, ValueType);
        const haystack = context.parse(args[2], 2, ValueType);
        if (!needle || !haystack)
            return null;
        if (!isValidType(needle.type, [
                BooleanType,
                StringType,
                NumberType,
                NullType,
                ValueType
            ])) {
            return context.error(`Expected first argument to be of type boolean, string, number or null, but found ${ toString$1(needle.type) } instead`);
        }
        return new In(needle, haystack);
    }
    evaluate(ctx) {
        const needle = this.needle.evaluate(ctx);
        const haystack = this.haystack.evaluate(ctx);
        if (haystack == null)
            return false;
        if (!isValidNativeType(needle, [
                'boolean',
                'string',
                'number',
                'null'
            ])) {
            throw new RuntimeError(`Expected first argument to be of type boolean, string, number or null, but found ${ toString$1(typeOf(needle)) } instead.`);
        }
        if (!isValidNativeType(haystack, [
                'string',
                'array'
            ])) {
            throw new RuntimeError(`Expected second argument to be of type array or string, but found ${ toString$1(typeOf(haystack)) } instead.`);
        }
        return haystack.indexOf(needle) >= 0;
    }
    eachChild(fn) {
        fn(this.needle);
        fn(this.haystack);
    }
    outputDefined() {
        return true;
    }
    serialize() {
        return [
            'in',
            this.needle.serialize(),
            this.haystack.serialize()
        ];
    }
}

class IndexOf {
    constructor(needle, haystack, fromIndex) {
        this.type = NumberType;
        this.needle = needle;
        this.haystack = haystack;
        this.fromIndex = fromIndex;
    }
    static parse(args, context) {
        if (args.length <= 2 || args.length >= 5) {
            return context.error(`Expected 3 or 4 arguments, but found ${ args.length - 1 } instead.`);
        }
        const needle = context.parse(args[1], 1, ValueType);
        const haystack = context.parse(args[2], 2, ValueType);
        if (!needle || !haystack)
            return null;
        if (!isValidType(needle.type, [
                BooleanType,
                StringType,
                NumberType,
                NullType,
                ValueType
            ])) {
            return context.error(`Expected first argument to be of type boolean, string, number or null, but found ${ toString$1(needle.type) } instead`);
        }
        if (args.length === 4) {
            const fromIndex = context.parse(args[3], 3, NumberType);
            if (!fromIndex)
                return null;
            return new IndexOf(needle, haystack, fromIndex);
        } else {
            return new IndexOf(needle, haystack);
        }
    }
    evaluate(ctx) {
        const needle = this.needle.evaluate(ctx);
        const haystack = this.haystack.evaluate(ctx);
        if (!isValidNativeType(needle, [
                'boolean',
                'string',
                'number',
                'null'
            ])) {
            throw new RuntimeError(`Expected first argument to be of type boolean, string, number or null, but found ${ toString$1(typeOf(needle)) } instead.`);
        }
        if (!isValidNativeType(haystack, [
                'string',
                'array'
            ])) {
            throw new RuntimeError(`Expected second argument to be of type array or string, but found ${ toString$1(typeOf(haystack)) } instead.`);
        }
        if (this.fromIndex) {
            const fromIndex = this.fromIndex.evaluate(ctx);
            return haystack.indexOf(needle, fromIndex);
        }
        return haystack.indexOf(needle);
    }
    eachChild(fn) {
        fn(this.needle);
        fn(this.haystack);
        if (this.fromIndex) {
            fn(this.fromIndex);
        }
    }
    outputDefined() {
        return false;
    }
    serialize() {
        if (this.fromIndex != null && this.fromIndex !== void 0) {
            const fromIndex = this.fromIndex.serialize();
            return [
                'index-of',
                this.needle.serialize(),
                this.haystack.serialize(),
                fromIndex
            ];
        }
        return [
            'index-of',
            this.needle.serialize(),
            this.haystack.serialize()
        ];
    }
}

class Match {
    constructor(inputType, outputType, input, cases, outputs, otherwise) {
        this.inputType = inputType;
        this.type = outputType;
        this.input = input;
        this.cases = cases;
        this.outputs = outputs;
        this.otherwise = otherwise;
    }
    static parse(args, context) {
        if (args.length < 5)
            return context.error(`Expected at least 4 arguments, but found only ${ args.length - 1 }.`);
        if (args.length % 2 !== 1)
            return context.error(`Expected an even number of arguments.`);
        let inputType;
        let outputType;
        if (context.expectedType && context.expectedType.kind !== 'value') {
            outputType = context.expectedType;
        }
        const cases = {};
        const outputs = [];
        for (let i = 2; i < args.length - 1; i += 2) {
            let labels = args[i];
            const value = args[i + 1];
            if (!Array.isArray(labels)) {
                labels = [labels];
            }
            const labelContext = context.concat(i);
            if (labels.length === 0) {
                return labelContext.error('Expected at least one branch label.');
            }
            for (const label of labels) {
                if (typeof label !== 'number' && typeof label !== 'string') {
                    return labelContext.error(`Branch labels must be numbers or strings.`);
                } else if (typeof label === 'number' && Math.abs(label) > Number.MAX_SAFE_INTEGER) {
                    return labelContext.error(`Branch labels must be integers no larger than ${ Number.MAX_SAFE_INTEGER }.`);
                } else if (typeof label === 'number' && Math.floor(label) !== label) {
                    return labelContext.error(`Numeric branch labels must be integer values.`);
                } else if (!inputType) {
                    inputType = typeOf(label);
                } else if (labelContext.checkSubtype(inputType, typeOf(label))) {
                    return null;
                }
                if (typeof cases[String(label)] !== 'undefined') {
                    return labelContext.error('Branch labels must be unique.');
                }
                cases[String(label)] = outputs.length;
            }
            const result = context.parse(value, i, outputType);
            if (!result)
                return null;
            outputType = outputType || result.type;
            outputs.push(result);
        }
        const input = context.parse(args[1], 1, ValueType);
        if (!input)
            return null;
        const otherwise = context.parse(args[args.length - 1], args.length - 1, outputType);
        if (!otherwise)
            return null;
        if (input.type.kind !== 'value' && context.concat(1).checkSubtype(inputType, input.type)) {
            return null;
        }
        return new Match(inputType, outputType, input, cases, outputs, otherwise);
    }
    evaluate(ctx) {
        const input = this.input.evaluate(ctx);
        const output = typeOf(input) === this.inputType && this.outputs[this.cases[input]] || this.otherwise;
        return output.evaluate(ctx);
    }
    eachChild(fn) {
        fn(this.input);
        this.outputs.forEach(fn);
        fn(this.otherwise);
    }
    outputDefined() {
        return this.outputs.every(out => out.outputDefined()) && this.otherwise.outputDefined();
    }
    serialize() {
        const serialized = [
            'match',
            this.input.serialize()
        ];
        const sortedLabels = Object.keys(this.cases).sort();
        const groupedByOutput = [];
        const outputLookup = {};
        for (const label of sortedLabels) {
            const outputIndex = outputLookup[this.cases[label]];
            if (outputIndex === void 0) {
                outputLookup[this.cases[label]] = groupedByOutput.length;
                groupedByOutput.push([
                    this.cases[label],
                    [label]
                ]);
            } else {
                groupedByOutput[outputIndex][1].push(label);
            }
        }
        const coerceLabel = label => this.inputType.kind === 'number' ? Number(label) : label;
        for (const [outputIndex, labels] of groupedByOutput) {
            if (labels.length === 1) {
                serialized.push(coerceLabel(labels[0]));
            } else {
                serialized.push(labels.map(coerceLabel));
            }
            serialized.push(this.outputs[outputIndex].serialize());
        }
        serialized.push(this.otherwise.serialize());
        return serialized;
    }
}

class Case {
    constructor(type, branches, otherwise) {
        this.type = type;
        this.branches = branches;
        this.otherwise = otherwise;
    }
    static parse(args, context) {
        if (args.length < 4)
            return context.error(`Expected at least 3 arguments, but found only ${ args.length - 1 }.`);
        if (args.length % 2 !== 0)
            return context.error(`Expected an odd number of arguments.`);
        let outputType;
        if (context.expectedType && context.expectedType.kind !== 'value') {
            outputType = context.expectedType;
        }
        const branches = [];
        for (let i = 1; i < args.length - 1; i += 2) {
            const test = context.parse(args[i], i, BooleanType);
            if (!test)
                return null;
            const result = context.parse(args[i + 1], i + 1, outputType);
            if (!result)
                return null;
            branches.push([
                test,
                result
            ]);
            outputType = outputType || result.type;
        }
        const otherwise = context.parse(args[args.length - 1], args.length - 1, outputType);
        if (!otherwise)
            return null;
        return new Case(outputType, branches, otherwise);
    }
    evaluate(ctx) {
        for (const [test, expression] of this.branches) {
            if (test.evaluate(ctx)) {
                return expression.evaluate(ctx);
            }
        }
        return this.otherwise.evaluate(ctx);
    }
    eachChild(fn) {
        for (const [test, expression] of this.branches) {
            fn(test);
            fn(expression);
        }
        fn(this.otherwise);
    }
    outputDefined() {
        return this.branches.every(([_, out]) => out.outputDefined()) && this.otherwise.outputDefined();
    }
    serialize() {
        const serialized = ['case'];
        this.eachChild(child => {
            serialized.push(child.serialize());
        });
        return serialized;
    }
}

class Slice {
    constructor(type, input, beginIndex, endIndex) {
        this.type = type;
        this.input = input;
        this.beginIndex = beginIndex;
        this.endIndex = endIndex;
    }
    static parse(args, context) {
        if (args.length <= 2 || args.length >= 5) {
            return context.error(`Expected 3 or 4 arguments, but found ${ args.length - 1 } instead.`);
        }
        const input = context.parse(args[1], 1, ValueType);
        const beginIndex = context.parse(args[2], 2, NumberType);
        if (!input || !beginIndex)
            return null;
        if (!isValidType(input.type, [
                array(ValueType),
                StringType,
                ValueType
            ])) {
            return context.error(`Expected first argument to be of type array or string, but found ${ toString$1(input.type) } instead`);
        }
        if (args.length === 4) {
            const endIndex = context.parse(args[3], 3, NumberType);
            if (!endIndex)
                return null;
            return new Slice(input.type, input, beginIndex, endIndex);
        } else {
            return new Slice(input.type, input, beginIndex);
        }
    }
    evaluate(ctx) {
        const input = this.input.evaluate(ctx);
        const beginIndex = this.beginIndex.evaluate(ctx);
        if (!isValidNativeType(input, [
                'string',
                'array'
            ])) {
            throw new RuntimeError(`Expected first argument to be of type array or string, but found ${ toString$1(typeOf(input)) } instead.`);
        }
        if (this.endIndex) {
            const endIndex = this.endIndex.evaluate(ctx);
            return input.slice(beginIndex, endIndex);
        }
        return input.slice(beginIndex);
    }
    eachChild(fn) {
        fn(this.input);
        fn(this.beginIndex);
        if (this.endIndex) {
            fn(this.endIndex);
        }
    }
    outputDefined() {
        return false;
    }
    serialize() {
        if (this.endIndex != null && this.endIndex !== void 0) {
            const endIndex = this.endIndex.serialize();
            return [
                'slice',
                this.input.serialize(),
                this.beginIndex.serialize(),
                endIndex
            ];
        }
        return [
            'slice',
            this.input.serialize(),
            this.beginIndex.serialize()
        ];
    }
}

function isComparableType(op, type) {
    if (op === '==' || op === '!=') {
        return type.kind === 'boolean' || type.kind === 'string' || type.kind === 'number' || type.kind === 'null' || type.kind === 'value';
    } else {
        return type.kind === 'string' || type.kind === 'number' || type.kind === 'value';
    }
}
function eq(ctx, a, b) {
    return a === b;
}
function neq(ctx, a, b) {
    return a !== b;
}
function lt(ctx, a, b) {
    return a < b;
}
function gt(ctx, a, b) {
    return a > b;
}
function lteq(ctx, a, b) {
    return a <= b;
}
function gteq(ctx, a, b) {
    return a >= b;
}
function eqCollate(ctx, a, b, c) {
    return c.compare(a, b) === 0;
}
function neqCollate(ctx, a, b, c) {
    return !eqCollate(ctx, a, b, c);
}
function ltCollate(ctx, a, b, c) {
    return c.compare(a, b) < 0;
}
function gtCollate(ctx, a, b, c) {
    return c.compare(a, b) > 0;
}
function lteqCollate(ctx, a, b, c) {
    return c.compare(a, b) <= 0;
}
function gteqCollate(ctx, a, b, c) {
    return c.compare(a, b) >= 0;
}
function makeComparison(op, compareBasic, compareWithCollator) {
    const isOrderComparison = op !== '==' && op !== '!=';
    return class Comparison {
        constructor(lhs, rhs, collator) {
            this.type = BooleanType;
            this.lhs = lhs;
            this.rhs = rhs;
            this.collator = collator;
            this.hasUntypedArgument = lhs.type.kind === 'value' || rhs.type.kind === 'value';
        }
        static parse(args, context) {
            if (args.length !== 3 && args.length !== 4)
                return context.error(`Expected two or three arguments.`);
            const op2 = args[0];
            let lhs = context.parse(args[1], 1, ValueType);
            if (!lhs)
                return null;
            if (!isComparableType(op2, lhs.type)) {
                return context.concat(1).error(`"${ op2 }" comparisons are not supported for type '${ toString$1(lhs.type) }'.`);
            }
            let rhs = context.parse(args[2], 2, ValueType);
            if (!rhs)
                return null;
            if (!isComparableType(op2, rhs.type)) {
                return context.concat(2).error(`"${ op2 }" comparisons are not supported for type '${ toString$1(rhs.type) }'.`);
            }
            if (lhs.type.kind !== rhs.type.kind && lhs.type.kind !== 'value' && rhs.type.kind !== 'value') {
                return context.error(`Cannot compare types '${ toString$1(lhs.type) }' and '${ toString$1(rhs.type) }'.`);
            }
            if (isOrderComparison) {
                if (lhs.type.kind === 'value' && rhs.type.kind !== 'value') {
                    lhs = new Assertion(rhs.type, [lhs]);
                } else if (lhs.type.kind !== 'value' && rhs.type.kind === 'value') {
                    rhs = new Assertion(lhs.type, [rhs]);
                }
            }
            let collator = null;
            if (args.length === 4) {
                if (lhs.type.kind !== 'string' && rhs.type.kind !== 'string' && lhs.type.kind !== 'value' && rhs.type.kind !== 'value') {
                    return context.error(`Cannot use collator to compare non-string types.`);
                }
                collator = context.parse(args[3], 3, CollatorType);
                if (!collator)
                    return null;
            }
            return new Comparison(lhs, rhs, collator);
        }
        evaluate(ctx) {
            const lhs = this.lhs.evaluate(ctx);
            const rhs = this.rhs.evaluate(ctx);
            if (isOrderComparison && this.hasUntypedArgument) {
                const lt2 = typeOf(lhs);
                const rt = typeOf(rhs);
                if (lt2.kind !== rt.kind || !(lt2.kind === 'string' || lt2.kind === 'number')) {
                    throw new RuntimeError(`Expected arguments for "${ op }" to be (string, string) or (number, number), but found (${ lt2.kind }, ${ rt.kind }) instead.`);
                }
            }
            if (this.collator && !isOrderComparison && this.hasUntypedArgument) {
                const lt2 = typeOf(lhs);
                const rt = typeOf(rhs);
                if (lt2.kind !== 'string' || rt.kind !== 'string') {
                    return compareBasic(ctx, lhs, rhs);
                }
            }
            return this.collator ? compareWithCollator(ctx, lhs, rhs, this.collator.evaluate(ctx)) : compareBasic(ctx, lhs, rhs);
        }
        eachChild(fn) {
            fn(this.lhs);
            fn(this.rhs);
            if (this.collator) {
                fn(this.collator);
            }
        }
        outputDefined() {
            return true;
        }
        serialize() {
            const serialized = [op];
            this.eachChild(child => {
                serialized.push(child.serialize());
            });
            return serialized;
        }
    };
}
const Equals = makeComparison('==', eq, eqCollate);
const NotEquals = makeComparison('!=', neq, neqCollate);
const LessThan = makeComparison('<', lt, ltCollate);
const GreaterThan = makeComparison('>', gt, gtCollate);
const LessThanOrEqual = makeComparison('<=', lteq, lteqCollate);
const GreaterThanOrEqual = makeComparison('>=', gteq, gteqCollate);

class NumberFormat {
    // Default 3
    constructor(number, locale, currency, unit, minFractionDigits, maxFractionDigits) {
        this.type = StringType;
        this.number = number;
        this.locale = locale;
        this.currency = currency;
        this.unit = unit;
        this.minFractionDigits = minFractionDigits;
        this.maxFractionDigits = maxFractionDigits;
    }
    static parse(args, context) {
        if (args.length !== 3)
            return context.error(`Expected two arguments.`);
        const number = context.parse(args[1], 1, NumberType);
        if (!number)
            return null;
        const options = args[2];
        if (typeof options !== 'object' || Array.isArray(options))
            return context.error(`NumberFormat options argument must be an object.`);
        let locale = null;
        if (options['locale']) {
            locale = context.parseObjectValue(options['locale'], 2, 'locale', StringType);
            if (!locale)
                return null;
        }
        let currency = null;
        if (options['currency']) {
            currency = context.parseObjectValue(options['currency'], 2, 'currency', StringType);
            if (!currency)
                return null;
        }
        let unit = null;
        if (options['unit']) {
            unit = context.parseObjectValue(options['unit'], 2, 'unit', StringType);
            if (!unit)
                return null;
        }
        let minFractionDigits = null;
        if (options['min-fraction-digits']) {
            minFractionDigits = context.parseObjectValue(options['min-fraction-digits'], 2, 'min-fraction-digits', NumberType);
            if (!minFractionDigits)
                return null;
        }
        let maxFractionDigits = null;
        if (options['max-fraction-digits']) {
            maxFractionDigits = context.parseObjectValue(options['max-fraction-digits'], 2, 'max-fraction-digits', NumberType);
            if (!maxFractionDigits)
                return null;
        }
        return new NumberFormat(number, locale, currency, unit, minFractionDigits, maxFractionDigits);
    }
    evaluate(ctx) {
        return new Intl.NumberFormat(this.locale ? this.locale.evaluate(ctx) : [], {
            style: this.currency && 'currency' || this.unit && 'unit' || 'decimal',
            currency: this.currency ? this.currency.evaluate(ctx) : void 0,
            unit: this.unit ? this.unit.evaluate(ctx) : void 0,
            minimumFractionDigits: this.minFractionDigits ? this.minFractionDigits.evaluate(ctx) : void 0,
            maximumFractionDigits: this.maxFractionDigits ? this.maxFractionDigits.evaluate(ctx) : void 0
        }).format(this.number.evaluate(ctx));
    }
    eachChild(fn) {
        fn(this.number);
        if (this.locale) {
            fn(this.locale);
        }
        if (this.currency) {
            fn(this.currency);
        }
        if (this.unit) {
            fn(this.unit);
        }
        if (this.minFractionDigits) {
            fn(this.minFractionDigits);
        }
        if (this.maxFractionDigits) {
            fn(this.maxFractionDigits);
        }
    }
    outputDefined() {
        return false;
    }
    serialize() {
        const options = {};
        if (this.locale) {
            options['locale'] = this.locale.serialize();
        }
        if (this.currency) {
            options['currency'] = this.currency.serialize();
        }
        if (this.unit) {
            options['unit'] = this.unit.serialize();
        }
        if (this.minFractionDigits) {
            options['min-fraction-digits'] = this.minFractionDigits.serialize();
        }
        if (this.maxFractionDigits) {
            options['max-fraction-digits'] = this.maxFractionDigits.serialize();
        }
        return [
            'number-format',
            this.number.serialize(),
            options
        ];
    }
}

class Length {
    constructor(input) {
        this.type = NumberType;
        this.input = input;
    }
    static parse(args, context) {
        if (args.length !== 2)
            return context.error(`Expected 1 argument, but found ${ args.length - 1 } instead.`);
        const input = context.parse(args[1], 1);
        if (!input)
            return null;
        if (input.type.kind !== 'array' && input.type.kind !== 'string' && input.type.kind !== 'value')
            return context.error(`Expected argument of type string or array, but found ${ toString$1(input.type) } instead.`);
        return new Length(input);
    }
    evaluate(ctx) {
        const input = this.input.evaluate(ctx);
        if (typeof input === 'string') {
            return input.length;
        } else if (Array.isArray(input)) {
            return input.length;
        } else {
            throw new RuntimeError(`Expected value to be of type string or array, but found ${ toString$1(typeOf(input)) } instead.`);
        }
    }
    eachChild(fn) {
        fn(this.input);
    }
    outputDefined() {
        return false;
    }
    serialize() {
        const serialized = ['length'];
        this.eachChild(child => {
            serialized.push(child.serialize());
        });
        return serialized;
    }
}

function mulberry32(a) {
    return function () {
        a |= 0;
        a = a + 1831565813 | 0;
        let t = Math.imul(a ^ a >>> 15, 1 | a);
        t = t + Math.imul(t ^ t >>> 7, 61 | t) ^ t;
        return ((t ^ t >>> 14) >>> 0) / 4294967296;
    };
}

const expressions = {
    // special forms
    '==': Equals,
    '!=': NotEquals,
    '>': GreaterThan,
    '<': LessThan,
    '>=': GreaterThanOrEqual,
    '<=': LessThanOrEqual,
    'array': Assertion,
    'at': At,
    'boolean': Assertion,
    'case': Case,
    'coalesce': Coalesce,
    'collator': CollatorExpression,
    'format': FormatExpression,
    'image': ImageExpression,
    'in': In,
    'index-of': IndexOf,
    'interpolate': Interpolate,
    'interpolate-hcl': Interpolate,
    'interpolate-lab': Interpolate,
    'length': Length,
    'let': Let,
    'literal': Literal,
    'match': Match,
    'number': Assertion,
    'number-format': NumberFormat,
    'object': Assertion,
    'slice': Slice,
    'step': Step,
    'string': Assertion,
    'to-boolean': Coercion,
    'to-color': Coercion,
    'to-number': Coercion,
    'to-string': Coercion,
    'var': Var,
    'within': Within,
    'distance': Distance,
    'config': Config
};
function rgba(ctx, [r, g, b, a]) {
    r = r.evaluate(ctx);
    g = g.evaluate(ctx);
    b = b.evaluate(ctx);
    const alpha = a ? a.evaluate(ctx) : 1;
    const error = validateRGBA(r, g, b, alpha);
    if (error)
        throw new RuntimeError(error);
    return new Color(r / 255 * alpha, g / 255 * alpha, b / 255 * alpha, alpha);
}
function hsla(ctx, [h, s, l, a]) {
    h = h.evaluate(ctx);
    s = s.evaluate(ctx);
    l = l.evaluate(ctx);
    const alpha = a ? a.evaluate(ctx) : 1;
    const error = validateHSLA(h, s, l, alpha);
    if (error)
        throw new RuntimeError(error);
    const colorFunction = `hsla(${ h }, ${ s }%, ${ l }%, ${ alpha })`;
    const color = Color.parse(colorFunction);
    if (!color)
        throw new RuntimeError(`Failed to parse HSLA color: ${ colorFunction }`);
    return color;
}
function has(key, obj) {
    return key in obj;
}
function get(key, obj) {
    const v = obj[key];
    return typeof v === 'undefined' ? null : v;
}
function binarySearch(v, a, i, j) {
    while (i <= j) {
        const m = i + j >> 1;
        if (a[m] === v)
            return true;
        if (a[m] > v)
            j = m - 1;
        else
            i = m + 1;
    }
    return false;
}
function varargs(type) {
    return { type };
}
function hashString(str) {
    let hash = 0;
    if (str.length === 0) {
        return hash;
    }
    for (let i = 0; i < str.length; i++) {
        const char = str.charCodeAt(i);
        hash = (hash << 5) - hash + char;
        hash = hash & hash;
    }
    return hash;
}
CompoundExpression.register(expressions, {
    'error': [
        ErrorType,
        [StringType],
        (ctx, [v]) => {
            throw new RuntimeError(v.evaluate(ctx));
        }
    ],
    'typeof': [
        StringType,
        [ValueType],
        (ctx, [v]) => toString$1(typeOf(v.evaluate(ctx)))
    ],
    'to-rgba': [
        array(NumberType, 4),
        [ColorType],
        (ctx, [v]) => {
            return v.evaluate(ctx).toRenderColor(null).toArray();
        }
    ],
    'to-hsla': [
        array(NumberType, 4),
        [ColorType],
        (ctx, [v]) => {
            return v.evaluate(ctx).toRenderColor(null).toHslaArray();
        }
    ],
    'rgb': [
        ColorType,
        [
            NumberType,
            NumberType,
            NumberType
        ],
        rgba
    ],
    'rgba': [
        ColorType,
        [
            NumberType,
            NumberType,
            NumberType,
            NumberType
        ],
        rgba
    ],
    'hsl': [
        ColorType,
        [
            NumberType,
            NumberType,
            NumberType
        ],
        hsla
    ],
    'hsla': [
        ColorType,
        [
            NumberType,
            NumberType,
            NumberType,
            NumberType
        ],
        hsla
    ],
    'has': {
        type: BooleanType,
        overloads: [
            [
                [StringType],
                (ctx, [key]) => has(key.evaluate(ctx), ctx.properties())
            ],
            [
                [
                    StringType,
                    ObjectType
                ],
                (ctx, [key, obj]) => has(key.evaluate(ctx), obj.evaluate(ctx))
            ]
        ]
    },
    'get': {
        type: ValueType,
        overloads: [
            [
                [StringType],
                (ctx, [key]) => get(key.evaluate(ctx), ctx.properties())
            ],
            [
                [
                    StringType,
                    ObjectType
                ],
                (ctx, [key, obj]) => get(key.evaluate(ctx), obj.evaluate(ctx))
            ]
        ]
    },
    'feature-state': [
        ValueType,
        [StringType],
        (ctx, [key]) => get(key.evaluate(ctx), ctx.featureState || {})
    ],
    'properties': [
        ObjectType,
        [],
        ctx => ctx.properties()
    ],
    'geometry-type': [
        StringType,
        [],
        ctx => ctx.geometryType()
    ],
    'id': [
        ValueType,
        [],
        ctx => ctx.id()
    ],
    'zoom': [
        NumberType,
        [],
        ctx => ctx.globals.zoom
    ],
    'pitch': [
        NumberType,
        [],
        ctx => ctx.globals.pitch || 0
    ],
    'distance-from-center': [
        NumberType,
        [],
        ctx => ctx.distanceFromCenter()
    ],
    'measure-light': [
        NumberType,
        [StringType],
        (ctx, [s]) => ctx.measureLight(s.evaluate(ctx))
    ],
    'heatmap-density': [
        NumberType,
        [],
        ctx => ctx.globals.heatmapDensity || 0
    ],
    'line-progress': [
        NumberType,
        [],
        ctx => ctx.globals.lineProgress || 0
    ],
    'raster-value': [
        NumberType,
        [],
        ctx => ctx.globals.rasterValue || 0
    ],
    'raster-particle-speed': [
        NumberType,
        [],
        ctx => ctx.globals.rasterParticleSpeed || 0
    ],
    'sky-radial-progress': [
        NumberType,
        [],
        ctx => ctx.globals.skyRadialProgress || 0
    ],
    'accumulated': [
        ValueType,
        [],
        ctx => ctx.globals.accumulated === void 0 ? null : ctx.globals.accumulated
    ],
    '+': [
        NumberType,
        varargs(NumberType),
        (ctx, args) => {
            let result = 0;
            for (const arg of args) {
                result += arg.evaluate(ctx);
            }
            return result;
        }
    ],
    '*': [
        NumberType,
        varargs(NumberType),
        (ctx, args) => {
            let result = 1;
            for (const arg of args) {
                result *= arg.evaluate(ctx);
            }
            return result;
        }
    ],
    '-': {
        type: NumberType,
        overloads: [
            [
                [
                    NumberType,
                    NumberType
                ],
                (ctx, [a, b]) => a.evaluate(ctx) - b.evaluate(ctx)
            ],
            [
                [NumberType],
                (ctx, [a]) => -a.evaluate(ctx)
            ]
        ]
    },
    '/': [
        NumberType,
        [
            NumberType,
            NumberType
        ],
        (ctx, [a, b]) => a.evaluate(ctx) / b.evaluate(ctx)
    ],
    '%': [
        NumberType,
        [
            NumberType,
            NumberType
        ],
        (ctx, [a, b]) => a.evaluate(ctx) % b.evaluate(ctx)
    ],
    'ln2': [
        NumberType,
        [],
        () => Math.LN2
    ],
    'pi': [
        NumberType,
        [],
        () => Math.PI
    ],
    'e': [
        NumberType,
        [],
        () => Math.E
    ],
    '^': [
        NumberType,
        [
            NumberType,
            NumberType
        ],
        (ctx, [b, e]) => Math.pow(b.evaluate(ctx), e.evaluate(ctx))
    ],
    'sqrt': [
        NumberType,
        [NumberType],
        (ctx, [x]) => Math.sqrt(x.evaluate(ctx))
    ],
    'log10': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.log(n.evaluate(ctx)) / Math.LN10
    ],
    'ln': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.log(n.evaluate(ctx))
    ],
    'log2': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.log(n.evaluate(ctx)) / Math.LN2
    ],
    'sin': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.sin(n.evaluate(ctx))
    ],
    'cos': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.cos(n.evaluate(ctx))
    ],
    'tan': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.tan(n.evaluate(ctx))
    ],
    'asin': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.asin(n.evaluate(ctx))
    ],
    'acos': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.acos(n.evaluate(ctx))
    ],
    'atan': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.atan(n.evaluate(ctx))
    ],
    'min': [
        NumberType,
        varargs(NumberType),
        (ctx, args) => Math.min(...args.map(arg => arg.evaluate(ctx)))
    ],
    'max': [
        NumberType,
        varargs(NumberType),
        (ctx, args) => Math.max(...args.map(arg => arg.evaluate(ctx)))
    ],
    'abs': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.abs(n.evaluate(ctx))
    ],
    'round': [
        NumberType,
        [NumberType],
        (ctx, [n]) => {
            const v = n.evaluate(ctx);
            return v < 0 ? -Math.round(-v) : Math.round(v);
        }
    ],
    'floor': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.floor(n.evaluate(ctx))
    ],
    'ceil': [
        NumberType,
        [NumberType],
        (ctx, [n]) => Math.ceil(n.evaluate(ctx))
    ],
    'filter-==': [
        BooleanType,
        [
            StringType,
            ValueType
        ],
        (ctx, [k, v]) => ctx.properties()[k.value] === v.value
    ],
    'filter-id-==': [
        BooleanType,
        [ValueType],
        (ctx, [v]) => ctx.id() === v.value
    ],
    'filter-type-==': [
        BooleanType,
        [StringType],
        (ctx, [v]) => ctx.geometryType() === v.value
    ],
    'filter-<': [
        BooleanType,
        [
            StringType,
            ValueType
        ],
        (ctx, [k, v]) => {
            const a = ctx.properties()[k.value];
            const b = v.value;
            return typeof a === typeof b && a < b;
        }
    ],
    'filter-id-<': [
        BooleanType,
        [ValueType],
        (ctx, [v]) => {
            const a = ctx.id();
            const b = v.value;
            return typeof a === typeof b && a < b;
        }
    ],
    'filter->': [
        BooleanType,
        [
            StringType,
            ValueType
        ],
        (ctx, [k, v]) => {
            const a = ctx.properties()[k.value];
            const b = v.value;
            return typeof a === typeof b && a > b;
        }
    ],
    'filter-id->': [
        BooleanType,
        [ValueType],
        (ctx, [v]) => {
            const a = ctx.id();
            const b = v.value;
            return typeof a === typeof b && a > b;
        }
    ],
    'filter-<=': [
        BooleanType,
        [
            StringType,
            ValueType
        ],
        (ctx, [k, v]) => {
            const a = ctx.properties()[k.value];
            const b = v.value;
            return typeof a === typeof b && a <= b;
        }
    ],
    'filter-id-<=': [
        BooleanType,
        [ValueType],
        (ctx, [v]) => {
            const a = ctx.id();
            const b = v.value;
            return typeof a === typeof b && a <= b;
        }
    ],
    'filter->=': [
        BooleanType,
        [
            StringType,
            ValueType
        ],
        (ctx, [k, v]) => {
            const a = ctx.properties()[k.value];
            const b = v.value;
            return typeof a === typeof b && a >= b;
        }
    ],
    'filter-id->=': [
        BooleanType,
        [ValueType],
        (ctx, [v]) => {
            const a = ctx.id();
            const b = v.value;
            return typeof a === typeof b && a >= b;
        }
    ],
    'filter-has': [
        BooleanType,
        [ValueType],
        (ctx, [k]) => k.value in ctx.properties()
    ],
    'filter-has-id': [
        BooleanType,
        [],
        ctx => ctx.id() !== null && ctx.id() !== void 0
    ],
    'filter-type-in': [
        BooleanType,
        [array(StringType)],
        (ctx, [v]) => v.value.indexOf(ctx.geometryType()) >= 0
    ],
    'filter-id-in': [
        BooleanType,
        [array(ValueType)],
        (ctx, [v]) => v.value.indexOf(ctx.id()) >= 0
    ],
    'filter-in-small': [
        BooleanType,
        [
            StringType,
            array(ValueType)
        ],
        // assumes v is an array literal
        (ctx, [k, v]) => v.value.indexOf(ctx.properties()[k.value]) >= 0
    ],
    'filter-in-large': [
        BooleanType,
        [
            StringType,
            array(ValueType)
        ],
        // assumes v is a array literal with values sorted in ascending order and of a single type
        (ctx, [k, v]) => binarySearch(ctx.properties()[k.value], v.value, 0, v.value.length - 1)
    ],
    'all': {
        type: BooleanType,
        overloads: [
            [
                [
                    BooleanType,
                    BooleanType
                ],
                (ctx, [a, b]) => a.evaluate(ctx) && b.evaluate(ctx)
            ],
            [
                varargs(BooleanType),
                (ctx, args) => {
                    for (const arg of args) {
                        if (!arg.evaluate(ctx))
                            return false;
                    }
                    return true;
                }
            ]
        ]
    },
    'any': {
        type: BooleanType,
        overloads: [
            [
                [
                    BooleanType,
                    BooleanType
                ],
                (ctx, [a, b]) => a.evaluate(ctx) || b.evaluate(ctx)
            ],
            [
                varargs(BooleanType),
                (ctx, args) => {
                    for (const arg of args) {
                        if (arg.evaluate(ctx))
                            return true;
                    }
                    return false;
                }
            ]
        ]
    },
    '!': [
        BooleanType,
        [BooleanType],
        (ctx, [b]) => !b.evaluate(ctx)
    ],
    'is-supported-script': [
        BooleanType,
        [StringType],
        // At parse time this will always return true, so we need to exclude this expression with isGlobalPropertyConstant
        (ctx, [s]) => {
            const isSupportedScript = ctx.globals && ctx.globals.isSupportedScript;
            if (isSupportedScript) {
                return isSupportedScript(s.evaluate(ctx));
            }
            return true;
        }
    ],
    'upcase': [
        StringType,
        [StringType],
        (ctx, [s]) => s.evaluate(ctx).toUpperCase()
    ],
    'downcase': [
        StringType,
        [StringType],
        (ctx, [s]) => s.evaluate(ctx).toLowerCase()
    ],
    'concat': [
        StringType,
        varargs(ValueType),
        (ctx, args) => args.map(arg => toString(arg.evaluate(ctx))).join('')
    ],
    'resolved-locale': [
        StringType,
        [CollatorType],
        (ctx, [collator]) => collator.evaluate(ctx).resolvedLocale()
    ],
    'random': [
        NumberType,
        [
            NumberType,
            NumberType,
            ValueType
        ],
        (ctx, args) => {
            const [min, max, seed] = args.map(arg => arg.evaluate(ctx));
            if (min > max) {
                return min;
            }
            if (min === max) {
                return min;
            }
            let seedVal;
            if (typeof seed === 'string') {
                seedVal = hashString(seed);
            } else if (typeof seed === 'number') {
                seedVal = seed;
            } else {
                throw new RuntimeError(`Invalid seed input: ${ seed }`);
            }
            const random = mulberry32(seedVal)();
            return min + random * (max - min);
        }
    ]
});

function success(value) {
    return {
        result: 'success',
        value
    };
}
function error(value) {
    return {
        result: 'error',
        value
    };
}

function expressionHasParameter(expression, parameter) {
    return !!expression && !!expression.parameters && expression.parameters.indexOf(parameter) > -1;
}
function supportsPropertyExpression(spec) {
    return spec['property-type'] === 'data-driven';
}
function supportsLightExpression(spec) {
    return expressionHasParameter(spec.expression, 'measure-light');
}
function supportsZoomExpression(spec) {
    return expressionHasParameter(spec.expression, 'zoom');
}
function supportsLineProgressExpression(spec) {
    return expressionHasParameter(spec.expression, 'line-progress');
}
function supportsInterpolation(spec) {
    return !!spec.expression && spec.expression.interpolated;
}

function isFunction(value) {
    return typeof value === 'object' && value !== null && !Array.isArray(value);
}
function identityFunction(x) {
    return x;
}
function createFunction(parameters, propertySpec) {
    const isColor = propertySpec.type === 'color';
    const zoomAndFeatureDependent = parameters.stops && typeof parameters.stops[0][0] === 'object';
    const featureDependent = zoomAndFeatureDependent || parameters.property !== void 0;
    const zoomDependent = zoomAndFeatureDependent || !featureDependent;
    const type = parameters.type || (supportsInterpolation(propertySpec) ? 'exponential' : 'interval');
    if (isColor) {
        parameters = extend({}, parameters);
        if (parameters.stops) {
            parameters.stops = parameters.stops.map(stop => {
                return [
                    stop[0],
                    Color.parse(stop[1])
                ];
            });
        }
        if (parameters.default) {
            parameters.default = Color.parse(parameters.default);
        } else {
            parameters.default = Color.parse(propertySpec.default);
        }
    }
    if (parameters.colorSpace && parameters.colorSpace !== 'rgb' && !colorSpaces[parameters.colorSpace]) {
        throw new Error(`Unknown color space: ${ parameters.colorSpace }`);
    }
    let innerFun;
    let hashedStops;
    let categoricalKeyType;
    if (type === 'exponential') {
        innerFun = evaluateExponentialFunction;
    } else if (type === 'interval') {
        innerFun = evaluateIntervalFunction;
    } else if (type === 'categorical') {
        innerFun = evaluateCategoricalFunction;
        hashedStops = /* @__PURE__ */
        Object.create(null);
        for (const stop of parameters.stops) {
            hashedStops[stop[0]] = stop[1];
        }
        categoricalKeyType = typeof parameters.stops[0][0];
    } else if (type === 'identity') {
        innerFun = evaluateIdentityFunction;
    } else {
        throw new Error(`Unknown function type "${ type }"`);
    }
    if (zoomAndFeatureDependent) {
        const featureFunctions = {};
        const zoomStops = [];
        for (let s = 0; s < parameters.stops.length; s++) {
            const stop = parameters.stops[s];
            const zoom = stop[0].zoom;
            if (featureFunctions[zoom] === void 0) {
                featureFunctions[zoom] = {
                    zoom,
                    type: parameters.type,
                    property: parameters.property,
                    default: parameters.default,
                    stops: []
                };
                zoomStops.push(zoom);
            }
            featureFunctions[zoom].stops.push([
                stop[0].value,
                stop[1]
            ]);
        }
        const featureFunctionStops = [];
        for (const z of zoomStops) {
            featureFunctionStops.push([
                featureFunctions[z].zoom,
                createFunction(featureFunctions[z], propertySpec)
            ]);
        }
        const interpolationType = { name: 'linear' };
        return {
            kind: 'composite',
            interpolationType,
            interpolationFactor: Interpolate.interpolationFactor.bind(void 0, interpolationType),
            zoomStops: featureFunctionStops.map(s => s[0]),
            evaluate({zoom}, properties) {
                return evaluateExponentialFunction({
                    stops: featureFunctionStops,
                    base: parameters.base
                }, propertySpec, zoom).evaluate(zoom, properties);
            }
        };
    } else if (zoomDependent) {
        const interpolationType = type === 'exponential' ? {
            name: 'exponential',
            base: parameters.base !== void 0 ? parameters.base : 1
        } : null;
        return {
            kind: 'camera',
            interpolationType,
            interpolationFactor: Interpolate.interpolationFactor.bind(void 0, interpolationType),
            zoomStops: parameters.stops.map(s => s[0]),
            evaluate: ({zoom}) => innerFun(parameters, propertySpec, zoom, hashedStops, categoricalKeyType)
        };
    } else {
        return {
            kind: 'source',
            evaluate(_, feature) {
                const value = feature && feature.properties ? feature.properties[parameters.property] : void 0;
                if (value === void 0) {
                    return coalesce(parameters.default, propertySpec.default);
                }
                return innerFun(parameters, propertySpec, value, hashedStops, categoricalKeyType);
            }
        };
    }
}
function coalesce(a, b, c) {
    if (a !== void 0)
        return a;
    if (b !== void 0)
        return b;
    if (c !== void 0)
        return c;
}
function evaluateCategoricalFunction(parameters, propertySpec, input, hashedStops, keyType) {
    const evaluated = typeof input === keyType ? hashedStops[input] : void 0;
    return coalesce(evaluated, parameters.default, propertySpec.default);
}
function evaluateIntervalFunction(parameters, propertySpec, input) {
    if (getType(input) !== 'number')
        return coalesce(parameters.default, propertySpec.default);
    const n = parameters.stops.length;
    if (n === 1)
        return parameters.stops[0][1];
    if (input <= parameters.stops[0][0])
        return parameters.stops[0][1];
    if (input >= parameters.stops[n - 1][0])
        return parameters.stops[n - 1][1];
    const index = findStopLessThanOrEqualTo(parameters.stops.map(stop => stop[0]), input);
    return parameters.stops[index][1];
}
function evaluateExponentialFunction(parameters, propertySpec, input) {
    const base = parameters.base !== void 0 ? parameters.base : 1;
    if (getType(input) !== 'number')
        return coalesce(parameters.default, propertySpec.default);
    const n = parameters.stops.length;
    if (n === 1)
        return parameters.stops[0][1];
    if (input <= parameters.stops[0][0])
        return parameters.stops[0][1];
    if (input >= parameters.stops[n - 1][0])
        return parameters.stops[n - 1][1];
    const index = findStopLessThanOrEqualTo(parameters.stops.map(stop => stop[0]), input);
    const t = interpolationFactor(input, base, parameters.stops[index][0], parameters.stops[index + 1][0]);
    const outputLower = parameters.stops[index][1];
    const outputUpper = parameters.stops[index + 1][1];
    let interp = interpolate$1[propertySpec.type] || identityFunction;
    if (parameters.colorSpace && parameters.colorSpace !== 'rgb') {
        const colorspace = colorSpaces[parameters.colorSpace];
        interp = (a, b) => colorspace.reverse(colorspace.interpolate(colorspace.forward(a), colorspace.forward(b), t));
    }
    if (typeof outputLower.evaluate === 'function') {
        return {
            evaluate(...args) {
                const evaluatedLower = outputLower.evaluate.apply(void 0, args);
                const evaluatedUpper = outputUpper.evaluate.apply(void 0, args);
                if (evaluatedLower === void 0 || evaluatedUpper === void 0) {
                    return void 0;
                }
                return interp(evaluatedLower, evaluatedUpper, t);
            }
        };
    }
    return interp(outputLower, outputUpper, t);
}
function evaluateIdentityFunction(parameters, propertySpec, input) {
    if (propertySpec.type === 'color') {
        input = Color.parse(input);
    } else if (propertySpec.type === 'formatted') {
        input = Formatted.fromString(input.toString());
    } else if (propertySpec.type === 'resolvedImage') {
        input = ResolvedImage.build(input.toString());
    } else if (getType(input) !== propertySpec.type && (propertySpec.type !== 'enum' || !propertySpec.values[input])) {
        input = void 0;
    }
    return coalesce(input, parameters.default, propertySpec.default);
}
function interpolationFactor(input, base, lowerValue, upperValue) {
    const difference = upperValue - lowerValue;
    const progress = input - lowerValue;
    if (difference === 0) {
        return 0;
    } else if (base === 1) {
        return progress / difference;
    } else {
        return (Math.pow(base, progress) - 1) / (Math.pow(base, difference) - 1);
    }
}

class StyleExpression {
    constructor(expression, propertySpec, scope, options) {
        this.expression = expression;
        this._warningHistory = {};
        this._evaluator = new EvaluationContext(scope, options);
        this._defaultValue = propertySpec ? getDefaultValue(propertySpec) : null;
        this._enumValues = propertySpec && propertySpec.type === 'enum' ? propertySpec.values : null;
        this.configDependencies = getConfigDependencies(expression);
    }
    evaluateWithoutErrorHandling(globals, feature, featureState, canonical, availableImages, formattedSection, featureTileCoord, featureDistanceData) {
        this._evaluator.globals = globals;
        this._evaluator.feature = feature;
        this._evaluator.featureState = featureState;
        this._evaluator.canonical = canonical || null;
        this._evaluator.availableImages = availableImages || null;
        this._evaluator.formattedSection = formattedSection;
        this._evaluator.featureTileCoord = featureTileCoord || null;
        this._evaluator.featureDistanceData = featureDistanceData || null;
        return this.expression.evaluate(this._evaluator);
    }
    evaluate(globals, feature, featureState, canonical, availableImages, formattedSection, featureTileCoord, featureDistanceData) {
        this._evaluator.globals = globals;
        this._evaluator.feature = feature || null;
        this._evaluator.featureState = featureState || null;
        this._evaluator.canonical = canonical || null;
        this._evaluator.availableImages = availableImages || null;
        this._evaluator.formattedSection = formattedSection || null;
        this._evaluator.featureTileCoord = featureTileCoord || null;
        this._evaluator.featureDistanceData = featureDistanceData || null;
        try {
            const val = this.expression.evaluate(this._evaluator);
            if (val === null || val === void 0 || typeof val === 'number' && val !== val) {
                return this._defaultValue;
            }
            if (this._enumValues && !(val in this._enumValues)) {
                throw new RuntimeError(`Expected value to be one of ${ Object.keys(this._enumValues).map(v => JSON.stringify(v)).join(', ') }, but found ${ JSON.stringify(val) } instead.`);
            }
            return val;
        } catch (e) {
            if (!this._warningHistory[e.message]) {
                this._warningHistory[e.message] = true;
                if (typeof console !== 'undefined') {
                    console.warn(`Failed to evaluate expression "${ JSON.stringify(this.expression.serialize()) }". ${ e.message }`);
                }
            }
            return this._defaultValue;
        }
    }
}
function isExpression(expression) {
    return Array.isArray(expression) && expression.length > 0 && typeof expression[0] === 'string' && expression[0] in expressions;
}
function createExpression(expression, propertySpec, scope, options) {
    const parser = new ParsingContext$1(expressions, [], propertySpec ? getExpectedType(propertySpec) : void 0, void 0, void 0, scope, options);
    const parsed = parser.parse(expression, void 0, void 0, void 0, propertySpec && propertySpec.type === 'string' ? { typeAnnotation: 'coerce' } : void 0);
    if (!parsed) {
        return error(parser.errors);
    }
    return success(new StyleExpression(parsed, propertySpec, scope, options));
}
class ZoomConstantExpression {
    constructor(kind, expression, isLightConstant, isLineProgressConstant) {
        this.kind = kind;
        this._styleExpression = expression;
        this.isLightConstant = isLightConstant;
        this.isLineProgressConstant = isLineProgressConstant;
        this.isStateDependent = kind !== 'constant' && !isStateConstant(expression.expression);
        this.configDependencies = getConfigDependencies(expression.expression);
    }
    evaluateWithoutErrorHandling(globals, feature, featureState, canonical, availableImages, formattedSection) {
        return this._styleExpression.evaluateWithoutErrorHandling(globals, feature, featureState, canonical, availableImages, formattedSection);
    }
    evaluate(globals, feature, featureState, canonical, availableImages, formattedSection) {
        return this._styleExpression.evaluate(globals, feature, featureState, canonical, availableImages, formattedSection);
    }
}
class ZoomDependentExpression {
    constructor(kind, expression, zoomStops, interpolationType, isLightConstant, isLineProgressConstant) {
        this.kind = kind;
        this.zoomStops = zoomStops;
        this._styleExpression = expression;
        this.isStateDependent = kind !== 'camera' && !isStateConstant(expression.expression);
        this.isLightConstant = isLightConstant;
        this.isLineProgressConstant = isLineProgressConstant;
        this.configDependencies = getConfigDependencies(expression.expression);
        this.interpolationType = interpolationType;
    }
    evaluateWithoutErrorHandling(globals, feature, featureState, canonical, availableImages, formattedSection) {
        return this._styleExpression.evaluateWithoutErrorHandling(globals, feature, featureState, canonical, availableImages, formattedSection);
    }
    evaluate(globals, feature, featureState, canonical, availableImages, formattedSection) {
        return this._styleExpression.evaluate(globals, feature, featureState, canonical, availableImages, formattedSection);
    }
    interpolationFactor(input, lower, upper) {
        if (this.interpolationType) {
            return Interpolate.interpolationFactor(this.interpolationType, input, lower, upper);
        } else {
            return 0;
        }
    }
}
function createPropertyExpression(expression, propertySpec, scope, options) {
    expression = createExpression(expression, propertySpec, scope, options);
    if (expression.result === 'error') {
        return expression;
    }
    const parsed = expression.value.expression;
    const isFeatureConstant$1 = isFeatureConstant(parsed);
    if (!isFeatureConstant$1 && !supportsPropertyExpression(propertySpec)) {
        return error([new ParsingError('', 'data expressions not supported')]);
    }
    const isZoomConstant = isGlobalPropertyConstant(parsed, [
        'zoom',
        'pitch',
        'distance-from-center'
    ]);
    if (!isZoomConstant && !supportsZoomExpression(propertySpec)) {
        return error([new ParsingError('', 'zoom expressions not supported')]);
    }
    const isLightConstant = isGlobalPropertyConstant(parsed, ['measure-light']);
    if (!isLightConstant && !supportsLightExpression(propertySpec)) {
        return error([new ParsingError('', 'measure-light expression not supported')]);
    }
    const isLineProgressConstant = isGlobalPropertyConstant(parsed, ['line-progress']);
    if (!isLineProgressConstant && !supportsLineProgressExpression(propertySpec)) {
        return error([new ParsingError('', 'line-progress expression not supported')]);
    }
    const canRelaxZoomRestriction = propertySpec.expression && propertySpec.expression.relaxZoomRestriction;
    const zoomCurve = findZoomCurve(parsed);
    if (!zoomCurve && !isZoomConstant && !canRelaxZoomRestriction) {
        return error([new ParsingError('', '"zoom" expression may only be used as input to a top-level "step" or "interpolate" expression, or in the properties of atmosphere.')]);
    } else if (zoomCurve instanceof ParsingError) {
        return error([zoomCurve]);
    } else if (zoomCurve instanceof Interpolate && !supportsInterpolation(propertySpec)) {
        return error([new ParsingError('', '"interpolate" expressions cannot be used with this property')]);
    }
    if (!zoomCurve) {
        return success(isFeatureConstant$1 && isLineProgressConstant ? // @ts-expect-error - TS2339 - Property 'value' does not exist on type 'unknown'.
        new ZoomConstantExpression('constant', expression.value, isLightConstant, isLineProgressConstant) : // @ts-expect-error - TS2339 - Property 'value' does not exist on type 'unknown'.
        new ZoomConstantExpression('source', expression.value, isLightConstant, isLineProgressConstant));
    }
    const interpolationType = zoomCurve instanceof Interpolate ? zoomCurve.interpolation : void 0;
    return success(isFeatureConstant$1 && isLineProgressConstant ? // @ts-expect-error - TS2339 - Property 'value' does not exist on type 'unknown'.
    new ZoomDependentExpression('camera', expression.value, zoomCurve.labels, interpolationType, isLightConstant, isLineProgressConstant) : // @ts-expect-error - TS2339 - Property 'value' does not exist on type 'unknown'.
    new ZoomDependentExpression('composite', expression.value, zoomCurve.labels, interpolationType, isLightConstant, isLineProgressConstant));
}
class StylePropertyFunction {
    constructor(parameters, specification) {
        this._parameters = parameters;
        this._specification = specification;
        extend(this, createFunction(this._parameters, this._specification));
    }
    static deserialize(serialized) {
        return new StylePropertyFunction(serialized._parameters, serialized._specification);
    }
    static serialize(input) {
        return {
            _parameters: input._parameters,
            _specification: input._specification
        };
    }
}
function normalizePropertyExpression(value, specification, scope, options) {
    if (isFunction(value)) {
        return new StylePropertyFunction(value, specification);
    } else if (isExpression(value) || Array.isArray(value) && value.length > 0) {
        const expression = createPropertyExpression(value, specification, scope, options);
        if (expression.result === 'error') {
            throw new Error(expression.value.map(err => `${ err.key }: ${ err.message }`).join(', '));
        }
        return expression.value;
    } else {
        let constant = value;
        if (typeof value === 'string' && specification.type === 'color') {
            constant = Color.parse(value);
        }
        return {
            kind: 'constant',
            configDependencies: /* @__PURE__ */
            new Set(),
            evaluate: () => constant
        };
    }
}
function findZoomCurve(expression) {
    let result = null;
    if (expression instanceof Let) {
        result = findZoomCurve(expression.result);
    } else if (expression instanceof Coalesce) {
        for (const arg of expression.args) {
            result = findZoomCurve(arg);
            if (result) {
                break;
            }
        }
    } else if ((expression instanceof Step || expression instanceof Interpolate) && expression.input instanceof CompoundExpression && expression.input.name === 'zoom') {
        result = expression;
    }
    if (result instanceof ParsingError) {
        return result;
    }
    expression.eachChild(child => {
        const childResult = findZoomCurve(child);
        if (childResult instanceof ParsingError) {
            result = childResult;
        } else if (result && childResult && result !== childResult) {
            result = new ParsingError('', 'Only one zoom-based "step" or "interpolate" subexpression may be used in an expression.');
        }
    });
    return result;
}
function getExpectedType(spec) {
    const types = {
        color: ColorType,
        string: StringType,
        number: NumberType,
        enum: StringType,
        boolean: BooleanType,
        formatted: FormattedType,
        resolvedImage: ResolvedImageType
    };
    if (spec.type === 'array') {
        return array(types[spec.value] || ValueType, spec.length);
    }
    return types[spec.type];
}
function getDefaultValue(spec) {
    if (spec.type === 'color' && (isFunction(spec.default) || Array.isArray(spec.default))) {
        return new Color(0, 0, 0, 0);
    } else if (spec.type === 'color') {
        return Color.parse(spec.default) || null;
    } else if (spec.default === void 0) {
        return null;
    } else {
        return spec.default;
    }
}

var gridIndex;
var hasRequiredGridIndex;

function requireGridIndex () {
	if (hasRequiredGridIndex) return gridIndex;
	hasRequiredGridIndex = 1;
	gridIndex = GridIndex;
	var NUM_PARAMS = 3;
	function GridIndex(extent, n, padding) {
	    var cells = this.cells = [];
	    if (extent instanceof ArrayBuffer) {
	        this.arrayBuffer = extent;
	        var array = new Int32Array(this.arrayBuffer);
	        extent = array[0];
	        n = array[1];
	        padding = array[2];
	        this.d = n + 2 * padding;
	        for (var k = 0; k < this.d * this.d; k++) {
	            var start = array[NUM_PARAMS + k];
	            var end = array[NUM_PARAMS + k + 1];
	            cells.push(start === end ? null : array.subarray(start, end));
	        }
	        var keysOffset = array[NUM_PARAMS + cells.length];
	        var bboxesOffset = array[NUM_PARAMS + cells.length + 1];
	        this.keys = array.subarray(keysOffset, bboxesOffset);
	        this.bboxes = array.subarray(bboxesOffset);
	        this.insert = this._insertReadonly;
	    } else {
	        this.d = n + 2 * padding;
	        for (var i = 0; i < this.d * this.d; i++) {
	            cells.push([]);
	        }
	        this.keys = [];
	        this.bboxes = [];
	    }
	    this.n = n;
	    this.extent = extent;
	    this.padding = padding;
	    this.scale = n / extent;
	    this.uid = 0;
	    var p = padding / n * extent;
	    this.min = -p;
	    this.max = extent + p;
	}
	GridIndex.prototype.insert = function (key, x1, y1, x2, y2) {
	    this._forEachCell(x1, y1, x2, y2, this._insertCell, this.uid++);
	    this.keys.push(key);
	    this.bboxes.push(x1);
	    this.bboxes.push(y1);
	    this.bboxes.push(x2);
	    this.bboxes.push(y2);
	};
	GridIndex.prototype._insertReadonly = function () {
	    throw 'Cannot insert into a GridIndex created from an ArrayBuffer.';
	};
	GridIndex.prototype._insertCell = function (x1, y1, x2, y2, cellIndex, uid) {
	    this.cells[cellIndex].push(uid);
	};
	GridIndex.prototype.query = function (x1, y1, x2, y2, intersectionTest) {
	    var min = this.min;
	    var max = this.max;
	    if (x1 <= min && y1 <= min && max <= x2 && max <= y2 && !intersectionTest) {
	        // We use `Array#slice` because `this.keys` may be a `Int32Array` and
	        // some browsers (Safari and IE) do not support `TypedArray#slice`
	        // https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/TypedArray/slice#Browser_compatibility
	        return Array.prototype.slice.call(this.keys);
	    } else {
	        var result = [];
	        var seenUids = {};
	        this._forEachCell(x1, y1, x2, y2, this._queryCell, result, seenUids, intersectionTest);
	        return result;
	    }
	};
	GridIndex.prototype._queryCell = function (x1, y1, x2, y2, cellIndex, result, seenUids, intersectionTest) {
	    var cell = this.cells[cellIndex];
	    if (cell !== null) {
	        var keys = this.keys;
	        var bboxes = this.bboxes;
	        for (var u = 0; u < cell.length; u++) {
	            var uid = cell[u];
	            if (seenUids[uid] === undefined) {
	                var offset = uid * 4;
	                if (intersectionTest ? intersectionTest(bboxes[offset + 0], bboxes[offset + 1], bboxes[offset + 2], bboxes[offset + 3]) : x1 <= bboxes[offset + 2] && y1 <= bboxes[offset + 3] && x2 >= bboxes[offset + 0] && y2 >= bboxes[offset + 1]) {
	                    seenUids[uid] = true;
	                    result.push(keys[uid]);
	                } else {
	                    seenUids[uid] = false;
	                }
	            }
	        }
	    }
	};
	GridIndex.prototype._forEachCell = function (x1, y1, x2, y2, fn, arg1, arg2, intersectionTest) {
	    var cx1 = this._convertToCellCoord(x1);
	    var cy1 = this._convertToCellCoord(y1);
	    var cx2 = this._convertToCellCoord(x2);
	    var cy2 = this._convertToCellCoord(y2);
	    for (var x = cx1; x <= cx2; x++) {
	        for (var y = cy1; y <= cy2; y++) {
	            var cellIndex = this.d * y + x;
	            if (intersectionTest && !intersectionTest(this._convertFromCellCoord(x), this._convertFromCellCoord(y), this._convertFromCellCoord(x + 1), this._convertFromCellCoord(y + 1)))
	                continue;
	            if (fn.call(this, x1, y1, x2, y2, cellIndex, arg1, arg2, intersectionTest))
	                return;
	        }
	    }
	};
	GridIndex.prototype._convertFromCellCoord = function (x) {
	    return (x - this.padding) / this.scale;
	};
	GridIndex.prototype._convertToCellCoord = function (x) {
	    return Math.max(0, Math.min(this.d - 1, Math.floor(x * this.scale) + this.padding));
	};
	GridIndex.prototype.toArrayBuffer = function () {
	    if (this.arrayBuffer)
	        return this.arrayBuffer;
	    var cells = this.cells;
	    var metadataLength = NUM_PARAMS + this.cells.length + 1 + 1;
	    var totalCellLength = 0;
	    for (var i = 0; i < this.cells.length; i++) {
	        totalCellLength += this.cells[i].length;
	    }
	    var array = new Int32Array(metadataLength + totalCellLength + this.keys.length + this.bboxes.length);
	    array[0] = this.extent;
	    array[1] = this.n;
	    array[2] = this.padding;
	    var offset = metadataLength;
	    for (var k = 0; k < cells.length; k++) {
	        var cell = cells[k];
	        array[NUM_PARAMS + k] = offset;
	        array.set(cell, offset);
	        offset += cell.length;
	    }
	    array[NUM_PARAMS + cells.length] = offset;
	    array.set(this.keys, offset);
	    offset += this.keys.length;
	    array[NUM_PARAMS + cells.length + 1] = offset;
	    array.set(this.bboxes, offset);
	    offset += this.bboxes.length;
	    return array.buffer;
	};
	return gridIndex;
}

var gridIndexExports = requireGridIndex();
var Grid = /*@__PURE__*/getDefaultExportFromCjs(gridIndexExports);

const registry = {};
function register(klass, name, options = {}) {
    Object.defineProperty(klass, '_classRegistryKey', {
        value: name,
        writable: false
    });
    registry[name] = {
        klass,
        omit: options.omit || []
    };
}
register(Object, 'Object');
Grid.serialize = function serialize2(grid, transferables) {
    const buffer = grid.toArrayBuffer();
    if (transferables) {
        transferables.add(buffer);
    }
    return { buffer };
};
Grid.deserialize = function deserialize2(serialized) {
    return new Grid(serialized.buffer);
};
Object.defineProperty(Grid, 'name', { value: 'Grid' });
register(Grid, 'Grid');
if (typeof DOMMatrix !== 'undefined') {
    register(DOMMatrix, 'DOMMatrix');
}
register(Color, 'Color');
register(Error, 'Error');
register(Formatted, 'Formatted');
register(FormattedSection, 'FormattedSection');
register(AJAXError, 'AJAXError');
register(ResolvedImage, 'ResolvedImage');
register(StylePropertyFunction, 'StylePropertyFunction');
register(StyleExpression, 'StyleExpression', { omit: ['_evaluator'] });
register(ImageIdWithOptions, 'ImageIdWithOptions');
register(ZoomDependentExpression, 'ZoomDependentExpression');
register(ZoomConstantExpression, 'ZoomConstantExpression');
register(CompoundExpression, 'CompoundExpression', { omit: ['_evaluate'] });
for (const name in expressions) {
    if (!registry[expressions[name]._classRegistryKey])
        register(expressions[name], `Expression${ name }`);
}
function isArrayBuffer(val) {
    return val && typeof ArrayBuffer !== 'undefined' && (val instanceof ArrayBuffer || val.constructor && val.constructor.name === 'ArrayBuffer');
}
function isImageBitmap(val) {
    return self.ImageBitmap && val instanceof ImageBitmap;
}
function serialize(input, transferables) {
    if (input === null || input === void 0 || typeof input === 'boolean' || typeof input === 'number' || typeof input === 'string' || input instanceof Boolean || input instanceof Number || input instanceof String || input instanceof Date || input instanceof RegExp) {
        return input;
    }
    if (isArrayBuffer(input) || isImageBitmap(input)) {
        if (transferables) {
            transferables.add(input);
        }
        return input;
    }
    if (ArrayBuffer.isView(input)) {
        if (transferables) {
            transferables.add(input.buffer);
        }
        return input;
    }
    if (input instanceof ImageData) {
        if (transferables) {
            transferables.add(input.data.buffer);
        }
        return input;
    }
    if (Array.isArray(input)) {
        const serialized = [];
        for (const item of input) {
            serialized.push(serialize(item, transferables));
        }
        return serialized;
    }
    if (input instanceof Map) {
        const properties = { '$name': 'Map' };
        for (const [key, value] of input.entries()) {
            properties[key] = serialize(value);
        }
        return properties;
    }
    if (input instanceof Set) {
        const properties = { '$name': 'Set' };
        let idx = 0;
        for (const value of input.values()) {
            properties[++idx] = serialize(value);
        }
        return properties;
    }
    if (typeof input === 'object') {
        const klass = input.constructor;
        const name = klass._classRegistryKey;
        if (!name) {
            throw new Error(`Can't serialize object of unregistered class "${ name }".`);
        }
        const properties = klass.serialize ? // (Temporary workaround) allow a class to provide static
        // `serialize()` and `deserialize()` methods to bypass the generic
        // approach.
        // This temporary workaround lets us use the generic serialization
        // approach for objects whose members include instances of dynamic
        // StructArray types. Once we refactor StructArray to be static,
        // we can remove this complexity.
        klass.serialize(input, transferables) : {};
        if (!klass.serialize) {
            for (const key in input) {
                if (!input.hasOwnProperty(key))
                    continue;
                if (registry[name].omit.indexOf(key) >= 0)
                    continue;
                const property = input[key];
                properties[key] = serialize(property, transferables);
            }
            if (input instanceof Error) {
                properties['message'] = input.message;
            }
        }
        if (properties['$name']) {
            throw new Error('$name property is reserved for worker serialization logic.');
        }
        if (name !== 'Object') {
            properties['$name'] = name;
        }
        return properties;
    }
    throw new Error(`can't serialize object of type ${ typeof input }`);
}
function deserialize(input) {
    if (input === null || input === void 0 || typeof input === 'boolean' || typeof input === 'number' || typeof input === 'string' || input instanceof Boolean || input instanceof Number || input instanceof String || input instanceof Date || input instanceof RegExp || isArrayBuffer(input) || isImageBitmap(input) || ArrayBuffer.isView(input) || input instanceof ImageData) {
        return input;
    }
    if (Array.isArray(input)) {
        return input.map(deserialize);
    }
    if (typeof input === 'object') {
        const name = input.$name || 'Object';
        if (name === 'Map') {
            const map = /* @__PURE__ */
            new Map();
            for (const key of Object.keys(input)) {
                if (key === '$name')
                    continue;
                const value = input[key];
                map.set(key, deserialize(value));
            }
            return map;
        }
        if (name === 'Set') {
            const set = /* @__PURE__ */
            new Set();
            for (const key of Object.keys(input)) {
                if (key === '$name')
                    continue;
                const value = input[key];
                set.add(deserialize(value));
            }
            return set;
        }
        const {klass} = registry[name];
        if (!klass) {
            throw new Error(`Can't deserialize unregistered class "${ name }".`);
        }
        if (klass.deserialize) {
            return klass.deserialize(input);
        }
        const result = Object.create(klass.prototype);
        for (const key of Object.keys(input)) {
            if (key === '$name')
                continue;
            const value = input[key];
            result[key] = deserialize(value);
        }
        return result;
    }
    throw new Error(`can't deserialize object of type ${ typeof input }`);
}

const unicodeBlockLookup = {
    // 'Basic Latin': (char) => char >= 0x0000 && char <= 0x007F,
    'Latin-1 Supplement': char => char >= 128 && char <= 255,
    // 'Latin Extended-A': (char) => char >= 0x0100 && char <= 0x017F,
    // 'Latin Extended-B': (char) => char >= 0x0180 && char <= 0x024F,
    // 'IPA Extensions': (char) => char >= 0x0250 && char <= 0x02AF,
    // 'Spacing Modifier Letters': (char) => char >= 0x02B0 && char <= 0x02FF,
    // 'Combining Diacritical Marks': (char) => char >= 0x0300 && char <= 0x036F,
    // 'Greek and Coptic': (char) => char >= 0x0370 && char <= 0x03FF,
    // 'Cyrillic': (char) => char >= 0x0400 && char <= 0x04FF,
    // 'Cyrillic Supplement': (char) => char >= 0x0500 && char <= 0x052F,
    // 'Armenian': (char) => char >= 0x0530 && char <= 0x058F,
    //'Hebrew': (char) => char >= 0x0590 && char <= 0x05FF,
    'Arabic': char => char >= 1536 && char <= 1791,
    //'Syriac': (char) => char >= 0x0700 && char <= 0x074F,
    'Arabic Supplement': char => char >= 1872 && char <= 1919,
    // 'Thaana': (char) => char >= 0x0780 && char <= 0x07BF,
    // 'NKo': (char) => char >= 0x07C0 && char <= 0x07FF,
    // 'Samaritan': (char) => char >= 0x0800 && char <= 0x083F,
    // 'Mandaic': (char) => char >= 0x0840 && char <= 0x085F,
    // 'Syriac Supplement': (char) => char >= 0x0860 && char <= 0x086F,
    'Arabic Extended-A': char => char >= 2208 && char <= 2303,
    // 'Devanagari': (char) => char >= 0x0900 && char <= 0x097F,
    // 'Bengali': (char) => char >= 0x0980 && char <= 0x09FF,
    // 'Gurmukhi': (char) => char >= 0x0A00 && char <= 0x0A7F,
    // 'Gujarati': (char) => char >= 0x0A80 && char <= 0x0AFF,
    // 'Oriya': (char) => char >= 0x0B00 && char <= 0x0B7F,
    // 'Tamil': (char) => char >= 0x0B80 && char <= 0x0BFF,
    // 'Telugu': (char) => char >= 0x0C00 && char <= 0x0C7F,
    // 'Kannada': (char) => char >= 0x0C80 && char <= 0x0CFF,
    // 'Malayalam': (char) => char >= 0x0D00 && char <= 0x0D7F,
    // 'Sinhala': (char) => char >= 0x0D80 && char <= 0x0DFF,
    // 'Thai': (char) => char >= 0x0E00 && char <= 0x0E7F,
    // 'Lao': (char) => char >= 0x0E80 && char <= 0x0EFF,
    // 'Tibetan': (char) => char >= 0x0F00 && char <= 0x0FFF,
    // 'Myanmar': (char) => char >= 0x1000 && char <= 0x109F,
    // 'Georgian': (char) => char >= 0x10A0 && char <= 0x10FF,
    'Hangul Jamo': char => char >= 4352 && char <= 4607,
    // 'Ethiopic': (char) => char >= 0x1200 && char <= 0x137F,
    // 'Ethiopic Supplement': (char) => char >= 0x1380 && char <= 0x139F,
    // 'Cherokee': (char) => char >= 0x13A0 && char <= 0x13FF,
    'Unified Canadian Aboriginal Syllabics': char => char >= 5120 && char <= 5759,
    // 'Ogham': (char) => char >= 0x1680 && char <= 0x169F,
    // 'Runic': (char) => char >= 0x16A0 && char <= 0x16FF,
    // 'Tagalog': (char) => char >= 0x1700 && char <= 0x171F,
    // 'Hanunoo': (char) => char >= 0x1720 && char <= 0x173F,
    // 'Buhid': (char) => char >= 0x1740 && char <= 0x175F,
    // 'Tagbanwa': (char) => char >= 0x1760 && char <= 0x177F,
    'Khmer': char => char >= 6016 && char <= 6143,
    // 'Mongolian': (char) => char >= 0x1800 && char <= 0x18AF,
    'Unified Canadian Aboriginal Syllabics Extended': char => char >= 6320 && char <= 6399,
    // 'Limbu': (char) => char >= 0x1900 && char <= 0x194F,
    // 'Tai Le': (char) => char >= 0x1950 && char <= 0x197F,
    // 'New Tai Lue': (char) => char >= 0x1980 && char <= 0x19DF,
    // 'Khmer Symbols': (char) => char >= 0x19E0 && char <= 0x19FF,
    // 'Buginese': (char) => char >= 0x1A00 && char <= 0x1A1F,
    // 'Tai Tham': (char) => char >= 0x1A20 && char <= 0x1AAF,
    // 'Combining Diacritical Marks Extended': (char) => char >= 0x1AB0 && char <= 0x1AFF,
    // 'Balinese': (char) => char >= 0x1B00 && char <= 0x1B7F,
    // 'Sundanese': (char) => char >= 0x1B80 && char <= 0x1BBF,
    // 'Batak': (char) => char >= 0x1BC0 && char <= 0x1BFF,
    // 'Lepcha': (char) => char >= 0x1C00 && char <= 0x1C4F,
    // 'Ol Chiki': (char) => char >= 0x1C50 && char <= 0x1C7F,
    // 'Cyrillic Extended-C': (char) => char >= 0x1C80 && char <= 0x1C8F,
    // 'Georgian Extended': (char) => char >= 0x1C90 && char <= 0x1CBF,
    // 'Sundanese Supplement': (char) => char >= 0x1CC0 && char <= 0x1CCF,
    // 'Vedic Extensions': (char) => char >= 0x1CD0 && char <= 0x1CFF,
    // 'Phonetic Extensions': (char) => char >= 0x1D00 && char <= 0x1D7F,
    // 'Phonetic Extensions Supplement': (char) => char >= 0x1D80 && char <= 0x1DBF,
    // 'Combining Diacritical Marks Supplement': (char) => char >= 0x1DC0 && char <= 0x1DFF,
    // 'Latin Extended Additional': (char) => char >= 0x1E00 && char <= 0x1EFF,
    // 'Greek Extended': (char) => char >= 0x1F00 && char <= 0x1FFF,
    'General Punctuation': char => char >= 8192 && char <= 8303,
    // 'Superscripts and Subscripts': (char) => char >= 0x2070 && char <= 0x209F,
    // 'Currency Symbols': (char) => char >= 0x20A0 && char <= 0x20CF,
    // 'Combining Diacritical Marks for Symbols': (char) => char >= 0x20D0 && char <= 0x20FF,
    'Letterlike Symbols': char => char >= 8448 && char <= 8527,
    'Number Forms': char => char >= 8528 && char <= 8591,
    // 'Arrows': (char) => char >= 0x2190 && char <= 0x21FF,
    // 'Mathematical Operators': (char) => char >= 0x2200 && char <= 0x22FF,
    'Miscellaneous Technical': char => char >= 8960 && char <= 9215,
    'Control Pictures': char => char >= 9216 && char <= 9279,
    'Optical Character Recognition': char => char >= 9280 && char <= 9311,
    'Enclosed Alphanumerics': char => char >= 9312 && char <= 9471,
    // 'Box Drawing': (char) => char >= 0x2500 && char <= 0x257F,
    // 'Block Elements': (char) => char >= 0x2580 && char <= 0x259F,
    'Geometric Shapes': char => char >= 9632 && char <= 9727,
    'Miscellaneous Symbols': char => char >= 9728 && char <= 9983,
    // 'Dingbats': (char) => char >= 0x2700 && char <= 0x27BF,
    // 'Miscellaneous Mathematical Symbols-A': (char) => char >= 0x27C0 && char <= 0x27EF,
    // 'Supplemental Arrows-A': (char) => char >= 0x27F0 && char <= 0x27FF,
    // 'Braille Patterns': (char) => char >= 0x2800 && char <= 0x28FF,
    // 'Supplemental Arrows-B': (char) => char >= 0x2900 && char <= 0x297F,
    // 'Miscellaneous Mathematical Symbols-B': (char) => char >= 0x2980 && char <= 0x29FF,
    // 'Supplemental Mathematical Operators': (char) => char >= 0x2A00 && char <= 0x2AFF,
    'Miscellaneous Symbols and Arrows': char => char >= 11008 && char <= 11263,
    // 'Glagolitic': (char) => char >= 0x2C00 && char <= 0x2C5F,
    // 'Latin Extended-C': (char) => char >= 0x2C60 && char <= 0x2C7F,
    // 'Coptic': (char) => char >= 0x2C80 && char <= 0x2CFF,
    // 'Georgian Supplement': (char) => char >= 0x2D00 && char <= 0x2D2F,
    // 'Tifinagh': (char) => char >= 0x2D30 && char <= 0x2D7F,
    // 'Ethiopic Extended': (char) => char >= 0x2D80 && char <= 0x2DDF,
    // 'Cyrillic Extended-A': (char) => char >= 0x2DE0 && char <= 0x2DFF,
    // 'Supplemental Punctuation': (char) => char >= 0x2E00 && char <= 0x2E7F,
    'CJK Radicals Supplement': char => char >= 11904 && char <= 12031,
    'Kangxi Radicals': char => char >= 12032 && char <= 12255,
    'Ideographic Description Characters': char => char >= 12272 && char <= 12287,
    'CJK Symbols and Punctuation': char => char >= 12288 && char <= 12351,
    'Hiragana': char => char >= 12352 && char <= 12447,
    'Katakana': char => char >= 12448 && char <= 12543,
    'Bopomofo': char => char >= 12544 && char <= 12591,
    'Hangul Compatibility Jamo': char => char >= 12592 && char <= 12687,
    'Kanbun': char => char >= 12688 && char <= 12703,
    'Bopomofo Extended': char => char >= 12704 && char <= 12735,
    'CJK Strokes': char => char >= 12736 && char <= 12783,
    'Katakana Phonetic Extensions': char => char >= 12784 && char <= 12799,
    'Enclosed CJK Letters and Months': char => char >= 12800 && char <= 13055,
    'CJK Compatibility': char => char >= 13056 && char <= 13311,
    'CJK Unified Ideographs Extension A': char => char >= 13312 && char <= 19903,
    'Yijing Hexagram Symbols': char => char >= 19904 && char <= 19967,
    'CJK Unified Ideographs': char => char >= 19968 && char <= 40959,
    'Yi Syllables': char => char >= 40960 && char <= 42127,
    'Yi Radicals': char => char >= 42128 && char <= 42191,
    // 'Lisu': (char) => char >= 0xA4D0 && char <= 0xA4FF,
    // 'Vai': (char) => char >= 0xA500 && char <= 0xA63F,
    // 'Cyrillic Extended-B': (char) => char >= 0xA640 && char <= 0xA69F,
    // 'Bamum': (char) => char >= 0xA6A0 && char <= 0xA6FF,
    // 'Modifier Tone Letters': (char) => char >= 0xA700 && char <= 0xA71F,
    // 'Latin Extended-D': (char) => char >= 0xA720 && char <= 0xA7FF,
    // 'Syloti Nagri': (char) => char >= 0xA800 && char <= 0xA82F,
    // 'Common Indic Number Forms': (char) => char >= 0xA830 && char <= 0xA83F,
    // 'Phags-pa': (char) => char >= 0xA840 && char <= 0xA87F,
    // 'Saurashtra': (char) => char >= 0xA880 && char <= 0xA8DF,
    // 'Devanagari Extended': (char) => char >= 0xA8E0 && char <= 0xA8FF,
    // 'Kayah Li': (char) => char >= 0xA900 && char <= 0xA92F,
    // 'Rejang': (char) => char >= 0xA930 && char <= 0xA95F,
    'Hangul Jamo Extended-A': char => char >= 43360 && char <= 43391,
    // 'Javanese': (char) => char >= 0xA980 && char <= 0xA9DF,
    // 'Myanmar Extended-B': (char) => char >= 0xA9E0 && char <= 0xA9FF,
    // 'Cham': (char) => char >= 0xAA00 && char <= 0xAA5F,
    // 'Myanmar Extended-A': (char) => char >= 0xAA60 && char <= 0xAA7F,
    // 'Tai Viet': (char) => char >= 0xAA80 && char <= 0xAADF,
    // 'Meetei Mayek Extensions': (char) => char >= 0xAAE0 && char <= 0xAAFF,
    // 'Ethiopic Extended-A': (char) => char >= 0xAB00 && char <= 0xAB2F,
    // 'Latin Extended-E': (char) => char >= 0xAB30 && char <= 0xAB6F,
    // 'Cherokee Supplement': (char) => char >= 0xAB70 && char <= 0xABBF,
    // 'Meetei Mayek': (char) => char >= 0xABC0 && char <= 0xABFF,
    'Hangul Syllables': char => char >= 44032 && char <= 55215,
    'Hangul Jamo Extended-B': char => char >= 55216 && char <= 55295,
    // 'High Surrogates': (char) => char >= 0xD800 && char <= 0xDB7F,
    // 'High Private Use Surrogates': (char) => char >= 0xDB80 && char <= 0xDBFF,
    // 'Low Surrogates': (char) => char >= 0xDC00 && char <= 0xDFFF,
    'Private Use Area': char => char >= 57344 && char <= 63743,
    'CJK Compatibility Ideographs': char => char >= 63744 && char <= 64255,
    // 'Alphabetic Presentation Forms': (char) => char >= 0xFB00 && char <= 0xFB4F,
    'Arabic Presentation Forms-A': char => char >= 64336 && char <= 65023,
    // 'Variation Selectors': (char) => char >= 0xFE00 && char <= 0xFE0F,
    'Vertical Forms': char => char >= 65040 && char <= 65055,
    // 'Combining Half Marks': (char) => char >= 0xFE20 && char <= 0xFE2F,
    'CJK Compatibility Forms': char => char >= 65072 && char <= 65103,
    'Small Form Variants': char => char >= 65104 && char <= 65135,
    'Arabic Presentation Forms-B': char => char >= 65136 && char <= 65279,
    'Halfwidth and Fullwidth Forms': char => char >= 65280 && char <= 65519,
    // 'Specials': (char) => char >= 0xFFF0 && char <= 0xFFFF,
    // 'Linear B Syllabary': (char) => char >= 0x10000 && char <= 0x1007F,
    // 'Linear B Ideograms': (char) => char >= 0x10080 && char <= 0x100FF,
    // 'Aegean Numbers': (char) => char >= 0x10100 && char <= 0x1013F,
    // 'Ancient Greek Numbers': (char) => char >= 0x10140 && char <= 0x1018F,
    // 'Ancient Symbols': (char) => char >= 0x10190 && char <= 0x101CF,
    // 'Phaistos Disc': (char) => char >= 0x101D0 && char <= 0x101FF,
    // 'Lycian': (char) => char >= 0x10280 && char <= 0x1029F,
    // 'Carian': (char) => char >= 0x102A0 && char <= 0x102DF,
    // 'Coptic Epact Numbers': (char) => char >= 0x102E0 && char <= 0x102FF,
    // 'Old Italic': (char) => char >= 0x10300 && char <= 0x1032F,
    // 'Gothic': (char) => char >= 0x10330 && char <= 0x1034F,
    // 'Old Permic': (char) => char >= 0x10350 && char <= 0x1037F,
    // 'Ugaritic': (char) => char >= 0x10380 && char <= 0x1039F,
    // 'Old Persian': (char) => char >= 0x103A0 && char <= 0x103DF,
    // 'Deseret': (char) => char >= 0x10400 && char <= 0x1044F,
    // 'Shavian': (char) => char >= 0x10450 && char <= 0x1047F,
    // 'Osmanya': (char) => char >= 0x10480 && char <= 0x104AF,
    'Osage': char => char >= 66736 && char <= 66815,
    // 'Elbasan': (char) => char >= 0x10500 && char <= 0x1052F,
    // 'Caucasian Albanian': (char) => char >= 0x10530 && char <= 0x1056F,
    // 'Linear A': (char) => char >= 0x10600 && char <= 0x1077F,
    // 'Cypriot Syllabary': (char) => char >= 0x10800 && char <= 0x1083F,
    // 'Imperial Aramaic': (char) => char >= 0x10840 && char <= 0x1085F,
    // 'Palmyrene': (char) => char >= 0x10860 && char <= 0x1087F,
    // 'Nabataean': (char) => char >= 0x10880 && char <= 0x108AF,
    // 'Hatran': (char) => char >= 0x108E0 && char <= 0x108FF,
    // 'Phoenician': (char) => char >= 0x10900 && char <= 0x1091F,
    // 'Lydian': (char) => char >= 0x10920 && char <= 0x1093F,
    // 'Meroitic Hieroglyphs': (char) => char >= 0x10980 && char <= 0x1099F,
    // 'Meroitic Cursive': (char) => char >= 0x109A0 && char <= 0x109FF,
    // 'Kharoshthi': (char) => char >= 0x10A00 && char <= 0x10A5F,
    // 'Old South Arabian': (char) => char >= 0x10A60 && char <= 0x10A7F,
    // 'Old North Arabian': (char) => char >= 0x10A80 && char <= 0x10A9F,
    // 'Manichaean': (char) => char >= 0x10AC0 && char <= 0x10AFF,
    // 'Avestan': (char) => char >= 0x10B00 && char <= 0x10B3F,
    // 'Inscriptional Parthian': (char) => char >= 0x10B40 && char <= 0x10B5F,
    // 'Inscriptional Pahlavi': (char) => char >= 0x10B60 && char <= 0x10B7F,
    // 'Psalter Pahlavi': (char) => char >= 0x10B80 && char <= 0x10BAF,
    // 'Old Turkic': (char) => char >= 0x10C00 && char <= 0x10C4F,
    // 'Old Hungarian': (char) => char >= 0x10C80 && char <= 0x10CFF,
    // 'Hanifi Rohingya': (char) => char >= 0x10D00 && char <= 0x10D3F,
    // 'Rumi Numeral Symbols': (char) => char >= 0x10E60 && char <= 0x10E7F,
    // 'Old Sogdian': (char) => char >= 0x10F00 && char <= 0x10F2F,
    // 'Sogdian': (char) => char >= 0x10F30 && char <= 0x10F6F,
    // 'Elymaic': (char) => char >= 0x10FE0 && char <= 0x10FFF,
    // 'Brahmi': (char) => char >= 0x11000 && char <= 0x1107F,
    // 'Kaithi': (char) => char >= 0x11080 && char <= 0x110CF,
    // 'Sora Sompeng': (char) => char >= 0x110D0 && char <= 0x110FF,
    // 'Chakma': (char) => char >= 0x11100 && char <= 0x1114F,
    // 'Mahajani': (char) => char >= 0x11150 && char <= 0x1117F,
    // 'Sharada': (char) => char >= 0x11180 && char <= 0x111DF,
    // 'Sinhala Archaic Numbers': (char) => char >= 0x111E0 && char <= 0x111FF,
    // 'Khojki': (char) => char >= 0x11200 && char <= 0x1124F,
    // 'Multani': (char) => char >= 0x11280 && char <= 0x112AF,
    // 'Khudawadi': (char) => char >= 0x112B0 && char <= 0x112FF,
    // 'Grantha': (char) => char >= 0x11300 && char <= 0x1137F,
    // 'Newa': (char) => char >= 0x11400 && char <= 0x1147F,
    // 'Tirhuta': (char) => char >= 0x11480 && char <= 0x114DF,
    // 'Siddham': (char) => char >= 0x11580 && char <= 0x115FF,
    // 'Modi': (char) => char >= 0x11600 && char <= 0x1165F,
    // 'Mongolian Supplement': (char) => char >= 0x11660 && char <= 0x1167F,
    // 'Takri': (char) => char >= 0x11680 && char <= 0x116CF,
    // 'Ahom': (char) => char >= 0x11700 && char <= 0x1173F,
    // 'Dogra': (char) => char >= 0x11800 && char <= 0x1184F,
    // 'Warang Citi': (char) => char >= 0x118A0 && char <= 0x118FF,
    // 'Nandinagari': (char) => char >= 0x119A0 && char <= 0x119FF,
    // 'Zanabazar Square': (char) => char >= 0x11A00 && char <= 0x11A4F,
    // 'Soyombo': (char) => char >= 0x11A50 && char <= 0x11AAF,
    // 'Pau Cin Hau': (char) => char >= 0x11AC0 && char <= 0x11AFF,
    // 'Bhaiksuki': (char) => char >= 0x11C00 && char <= 0x11C6F,
    // 'Marchen': (char) => char >= 0x11C70 && char <= 0x11CBF,
    // 'Masaram Gondi': (char) => char >= 0x11D00 && char <= 0x11D5F,
    // 'Gunjala Gondi': (char) => char >= 0x11D60 && char <= 0x11DAF,
    // 'Makasar': (char) => char >= 0x11EE0 && char <= 0x11EFF,
    // 'Tamil Supplement': (char) => char >= 0x11FC0 && char <= 0x11FFF,
    // 'Cuneiform': (char) => char >= 0x12000 && char <= 0x123FF,
    // 'Cuneiform Numbers and Punctuation': (char) => char >= 0x12400 && char <= 0x1247F,
    // 'Early Dynastic Cuneiform': (char) => char >= 0x12480 && char <= 0x1254F,
    // 'Egyptian Hieroglyphs': (char) => char >= 0x13000 && char <= 0x1342F,
    // 'Egyptian Hieroglyph Format Controls': (char) => char >= 0x13430 && char <= 0x1343F,
    // 'Anatolian Hieroglyphs': (char) => char >= 0x14400 && char <= 0x1467F,
    // 'Bamum Supplement': (char) => char >= 0x16800 && char <= 0x16A3F,
    // 'Mro': (char) => char >= 0x16A40 && char <= 0x16A6F,
    // 'Bassa Vah': (char) => char >= 0x16AD0 && char <= 0x16AFF,
    // 'Pahawh Hmong': (char) => char >= 0x16B00 && char <= 0x16B8F,
    // 'Medefaidrin': (char) => char >= 0x16E40 && char <= 0x16E9F,
    // 'Miao': (char) => char >= 0x16F00 && char <= 0x16F9F,
    // 'Ideographic Symbols and Punctuation': (char) => char >= 0x16FE0 && char <= 0x16FFF,
    // 'Tangut': (char) => char >= 0x17000 && char <= 0x187FF,
    // 'Tangut Components': (char) => char >= 0x18800 && char <= 0x18AFF,
    // 'Kana Supplement': (char) => char >= 0x1B000 && char <= 0x1B0FF,
    // 'Kana Extended-A': (char) => char >= 0x1B100 && char <= 0x1B12F,
    // 'Small Kana Extension': (char) => char >= 0x1B130 && char <= 0x1B16F,
    // 'Nushu': (char) => char >= 0x1B170 && char <= 0x1B2FF,
    // 'Duployan': (char) => char >= 0x1BC00 && char <= 0x1BC9F,
    // 'Shorthand Format Controls': (char) => char >= 0x1BCA0 && char <= 0x1BCAF,
    // 'Byzantine Musical Symbols': (char) => char >= 0x1D000 && char <= 0x1D0FF,
    // 'Musical Symbols': (char) => char >= 0x1D100 && char <= 0x1D1FF,
    // 'Ancient Greek Musical Notation': (char) => char >= 0x1D200 && char <= 0x1D24F,
    // 'Mayan Numerals': (char) => char >= 0x1D2E0 && char <= 0x1D2FF,
    // 'Tai Xuan Jing Symbols': (char) => char >= 0x1D300 && char <= 0x1D35F,
    // 'Counting Rod Numerals': (char) => char >= 0x1D360 && char <= 0x1D37F,
    // 'Mathematical Alphanumeric Symbols': (char) => char >= 0x1D400 && char <= 0x1D7FF,
    // 'Sutton SignWriting': (char) => char >= 0x1D800 && char <= 0x1DAAF,
    // 'Glagolitic Supplement': (char) => char >= 0x1E000 && char <= 0x1E02F,
    // 'Nyiakeng Puachue Hmong': (char) => char >= 0x1E100 && char <= 0x1E14F,
    // 'Wancho': (char) => char >= 0x1E2C0 && char <= 0x1E2FF,
    // 'Mende Kikakui': (char) => char >= 0x1E800 && char <= 0x1E8DF,
    // 'Adlam': (char) => char >= 0x1E900 && char <= 0x1E95F,
    // 'Indic Siyaq Numbers': (char) => char >= 0x1EC70 && char <= 0x1ECBF,
    // 'Ottoman Siyaq Numbers': (char) => char >= 0x1ED00 && char <= 0x1ED4F,
    // 'Arabic Mathematical Alphabetic Symbols': (char) => char >= 0x1EE00 && char <= 0x1EEFF,
    // 'Mahjong Tiles': (char) => char >= 0x1F000 && char <= 0x1F02F,
    // 'Domino Tiles': (char) => char >= 0x1F030 && char <= 0x1F09F,
    // 'Playing Cards': (char) => char >= 0x1F0A0 && char <= 0x1F0FF,
    // 'Enclosed Alphanumeric Supplement': (char) => char >= 0x1F100 && char <= 0x1F1FF,
    // 'Enclosed Ideographic Supplement': (char) => char >= 0x1F200 && char <= 0x1F2FF,
    // 'Miscellaneous Symbols and Pictographs': (char) => char >= 0x1F300 && char <= 0x1F5FF,
    // 'Emoticons': (char) => char >= 0x1F600 && char <= 0x1F64F,
    // 'Ornamental Dingbats': (char) => char >= 0x1F650 && char <= 0x1F67F,
    // 'Transport and Map Symbols': (char) => char >= 0x1F680 && char <= 0x1F6FF,
    // 'Alchemical Symbols': (char) => char >= 0x1F700 && char <= 0x1F77F,
    // 'Geometric Shapes Extended': (char) => char >= 0x1F780 && char <= 0x1F7FF,
    // 'Supplemental Arrows-C': (char) => char >= 0x1F800 && char <= 0x1F8FF,
    // 'Supplemental Symbols and Pictographs': (char) => char >= 0x1F900 && char <= 0x1F9FF,
    // 'Chess Symbols': (char) => char >= 0x1FA00 && char <= 0x1FA6F,
    // 'Symbols and Pictographs Extended-A': (char) => char >= 0x1FA70 && char <= 0x1FAFF,
    'CJK Unified Ideographs Extension B': char => char >= 131072 && char <= 173791    // 'CJK Unified Ideographs Extension C': (char) => char >= 0x2A700 && char <= 0x2B73F,
              // 'CJK Unified Ideographs Extension D': (char) => char >= 0x2B740 && char <= 0x2B81F,
              // 'CJK Unified Ideographs Extension E': (char) => char >= 0x2B820 && char <= 0x2CEAF,
              // 'CJK Unified Ideographs Extension F': (char) => char >= 0x2CEB0 && char <= 0x2EBEF,
              // 'CJK Compatibility Ideographs Supplement': (char) => char >= 0x2F800 && char <= 0x2FA1F,
              // 'Tags': (char) => char >= 0xE0000 && char <= 0xE007F,
              // 'Variation Selectors Supplement': (char) => char >= 0xE0100 && char <= 0xE01EF,
              // 'Supplementary Private Use Area-A': (char) => char >= 0xF0000 && char <= 0xFFFFF,
              // 'Supplementary Private Use Area-B': (char) => char >= 0x100000 && char <= 0x10FFFF,
};

function allowsVerticalWritingMode(chars) {
    for (const char of chars) {
        if (charHasUprightVerticalOrientation(char.charCodeAt(0)))
            return true;
    }
    return false;
}
function allowsLetterSpacing(chars) {
    for (const char of chars) {
        if (!charAllowsLetterSpacing(char.charCodeAt(0)))
            return false;
    }
    return true;
}
function charAllowsLetterSpacing(char) {
    if (unicodeBlockLookup['Arabic'](char))
        return false;
    if (unicodeBlockLookup['Arabic Supplement'](char))
        return false;
    if (unicodeBlockLookup['Arabic Extended-A'](char))
        return false;
    if (unicodeBlockLookup['Arabic Presentation Forms-A'](char))
        return false;
    if (unicodeBlockLookup['Arabic Presentation Forms-B'](char))
        return false;
    return true;
}
function charAllowsIdeographicBreaking(char) {
    if (char < 11904)
        return false;
    if (unicodeBlockLookup['Bopomofo Extended'](char))
        return true;
    if (unicodeBlockLookup['Bopomofo'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility Forms'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility Ideographs'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility'](char))
        return true;
    if (unicodeBlockLookup['CJK Radicals Supplement'](char))
        return true;
    if (unicodeBlockLookup['CJK Strokes'](char))
        return true;
    if (unicodeBlockLookup['CJK Symbols and Punctuation'](char))
        return true;
    if (unicodeBlockLookup['CJK Unified Ideographs Extension A'](char))
        return true;
    if (unicodeBlockLookup['CJK Unified Ideographs'](char))
        return true;
    if (unicodeBlockLookup['Enclosed CJK Letters and Months'](char))
        return true;
    if (unicodeBlockLookup['Halfwidth and Fullwidth Forms'](char))
        return true;
    if (unicodeBlockLookup['Hiragana'](char))
        return true;
    if (unicodeBlockLookup['Ideographic Description Characters'](char))
        return true;
    if (unicodeBlockLookup['Kangxi Radicals'](char))
        return true;
    if (unicodeBlockLookup['Katakana Phonetic Extensions'](char))
        return true;
    if (unicodeBlockLookup['Katakana'](char))
        return true;
    if (unicodeBlockLookup['Vertical Forms'](char))
        return true;
    if (unicodeBlockLookup['Yi Radicals'](char))
        return true;
    if (unicodeBlockLookup['Yi Syllables'](char))
        return true;
    return false;
}
function charHasUprightVerticalOrientation(char) {
    if (char === 746 || char === 747) {
        return true;
    }
    if (char < 4352)
        return false;
    if (unicodeBlockLookup['Bopomofo Extended'](char))
        return true;
    if (unicodeBlockLookup['Bopomofo'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility Forms'](char)) {
        if (!(char >= 65097 && char <= 65103)) {
            return true;
        }
    }
    if (unicodeBlockLookup['CJK Compatibility Ideographs'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility'](char))
        return true;
    if (unicodeBlockLookup['CJK Radicals Supplement'](char))
        return true;
    if (unicodeBlockLookup['CJK Strokes'](char))
        return true;
    if (unicodeBlockLookup['CJK Symbols and Punctuation'](char)) {
        if (!(char >= 12296 && char <= 12305) && !(char >= 12308 && char <= 12319) && char !== 12336) {
            return true;
        }
    }
    if (unicodeBlockLookup['CJK Unified Ideographs Extension A'](char))
        return true;
    if (unicodeBlockLookup['CJK Unified Ideographs'](char))
        return true;
    if (unicodeBlockLookup['Enclosed CJK Letters and Months'](char))
        return true;
    if (unicodeBlockLookup['Hangul Compatibility Jamo'](char))
        return true;
    if (unicodeBlockLookup['Hangul Jamo Extended-A'](char))
        return true;
    if (unicodeBlockLookup['Hangul Jamo Extended-B'](char))
        return true;
    if (unicodeBlockLookup['Hangul Jamo'](char))
        return true;
    if (unicodeBlockLookup['Hangul Syllables'](char))
        return true;
    if (unicodeBlockLookup['Hiragana'](char))
        return true;
    if (unicodeBlockLookup['Ideographic Description Characters'](char))
        return true;
    if (unicodeBlockLookup['Kanbun'](char))
        return true;
    if (unicodeBlockLookup['Kangxi Radicals'](char))
        return true;
    if (unicodeBlockLookup['Katakana Phonetic Extensions'](char))
        return true;
    if (unicodeBlockLookup['Katakana'](char)) {
        if (char !== 12540) {
            return true;
        }
    }
    if (unicodeBlockLookup['Halfwidth and Fullwidth Forms'](char)) {
        if (char !== 65288 && char !== 65289 && char !== 65293 && !(char >= 65306 && char <= 65310) && char !== 65339 && char !== 65341 && char !== 65343 && !(char >= 65371 && char <= 65503) && char !== 65507 && !(char >= 65512 && char <= 65519)) {
            return true;
        }
    }
    if (unicodeBlockLookup['Small Form Variants'](char)) {
        if (!(char >= 65112 && char <= 65118) && !(char >= 65123 && char <= 65126)) {
            return true;
        }
    }
    if (unicodeBlockLookup['Unified Canadian Aboriginal Syllabics'](char))
        return true;
    if (unicodeBlockLookup['Unified Canadian Aboriginal Syllabics Extended'](char))
        return true;
    if (unicodeBlockLookup['Vertical Forms'](char))
        return true;
    if (unicodeBlockLookup['Yijing Hexagram Symbols'](char))
        return true;
    if (unicodeBlockLookup['Yi Syllables'](char))
        return true;
    if (unicodeBlockLookup['Yi Radicals'](char))
        return true;
    return false;
}
function charHasNeutralVerticalOrientation(char) {
    if (unicodeBlockLookup['Latin-1 Supplement'](char)) {
        if (char === 167 || char === 169 || char === 174 || char === 177 || char === 188 || char === 189 || char === 190 || char === 215 || char === 247) {
            return true;
        }
    }
    if (unicodeBlockLookup['General Punctuation'](char)) {
        if (char === 8214 || char === 8224 || char === 8225 || char === 8240 || char === 8241 || char === 8251 || char === 8252 || char === 8258 || char === 8263 || char === 8264 || char === 8265 || char === 8273) {
            return true;
        }
    }
    if (unicodeBlockLookup['Letterlike Symbols'](char))
        return true;
    if (unicodeBlockLookup['Number Forms'](char))
        return true;
    if (unicodeBlockLookup['Miscellaneous Technical'](char)) {
        if (char >= 8960 && char <= 8967 || char >= 8972 && char <= 8991 || char >= 8996 && char <= 9000 || char === 9003 || char >= 9085 && char <= 9114 || char >= 9150 && char <= 9165 || char === 9167 || char >= 9169 && char <= 9179 || char >= 9186 && char <= 9215) {
            return true;
        }
    }
    if (unicodeBlockLookup['Control Pictures'](char) && char !== 9251)
        return true;
    if (unicodeBlockLookup['Optical Character Recognition'](char))
        return true;
    if (unicodeBlockLookup['Enclosed Alphanumerics'](char))
        return true;
    if (unicodeBlockLookup['Geometric Shapes'](char))
        return true;
    if (unicodeBlockLookup['Miscellaneous Symbols'](char)) {
        if (!(char >= 9754 && char <= 9759)) {
            return true;
        }
    }
    if (unicodeBlockLookup['Miscellaneous Symbols and Arrows'](char)) {
        if (char >= 11026 && char <= 11055 || char >= 11088 && char <= 11097 || char >= 11192 && char <= 11243) {
            return true;
        }
    }
    if (unicodeBlockLookup['CJK Symbols and Punctuation'](char))
        return true;
    if (unicodeBlockLookup['Katakana'](char))
        return true;
    if (unicodeBlockLookup['Private Use Area'](char))
        return true;
    if (unicodeBlockLookup['CJK Compatibility Forms'](char))
        return true;
    if (unicodeBlockLookup['Small Form Variants'](char))
        return true;
    if (unicodeBlockLookup['Halfwidth and Fullwidth Forms'](char))
        return true;
    if (char === 8734 || char === 8756 || char === 8757 || char >= 9984 && char <= 10087 || char >= 10102 && char <= 10131 || char === 65532 || char === 65533) {
        return true;
    }
    return false;
}
function charHasRotatedVerticalOrientation(char) {
    return !(charHasUprightVerticalOrientation(char) || charHasNeutralVerticalOrientation(char));
}
function charInComplexShapingScript(char) {
    return unicodeBlockLookup['Arabic'](char) || unicodeBlockLookup['Arabic Supplement'](char) || unicodeBlockLookup['Arabic Extended-A'](char) || unicodeBlockLookup['Arabic Presentation Forms-A'](char) || unicodeBlockLookup['Arabic Presentation Forms-B'](char);
}
function charInRTLScript(char) {
    return char >= 1424 && char <= 2303 || unicodeBlockLookup['Arabic Presentation Forms-A'](char) || unicodeBlockLookup['Arabic Presentation Forms-B'](char);
}
function charInSupportedScript(char, canRenderRTL) {
    if (!canRenderRTL && charInRTLScript(char)) {
        return false;
    }
    if (char >= 2304 && char <= 3583 || // Main blocks for Indic scripts and Sinhala
        char >= 3840 && char <= 4255 || // Main blocks for Tibetan and Myanmar
        unicodeBlockLookup['Khmer'](char)) {
        return false;
    }
    return true;
}
function stringContainsRTLText(chars) {
    for (const char of chars) {
        if (charInRTLScript(char.charCodeAt(0))) {
            return true;
        }
    }
    return false;
}
function isStringInSupportedScript(chars, canRenderRTL) {
    for (const char of chars) {
        if (!charInSupportedScript(char.charCodeAt(0), canRenderRTL)) {
            return false;
        }
    }
    return true;
}

const status = {
    unavailable: 'unavailable',
    // Not loaded
    deferred: 'deferred',
    // The plugin URL has been specified, but loading has been deferred
    loading: 'loading',
    // request in-flight
    loaded: 'loaded',
    error: 'error'
};
let _completionCallback = null;
let pluginStatus = status.unavailable;
let pluginURL = null;
const triggerPluginCompletionEvent = function (error) {
    if (error && typeof error === 'string' && error.indexOf('NetworkError') > -1) {
        pluginStatus = status.error;
    }
    if (_completionCallback) {
        _completionCallback(error);
    }
};
function sendPluginStateToWorker() {
    evented.fire(new Event('pluginStateChange', {
        pluginStatus,
        pluginURL
    }));
}
const evented = new Evented();
const getRTLTextPluginStatus = function () {
    return pluginStatus;
};
const registerForPluginStateChange = function (callback) {
    callback({
        pluginStatus,
        pluginURL
    });
    evented.on('pluginStateChange', callback);
    return callback;
};
const setRTLTextPlugin = function (url, callback, deferred = false) {
    if (pluginStatus === status.deferred || pluginStatus === status.loading || pluginStatus === status.loaded) {
        throw new Error('setRTLTextPlugin cannot be called multiple times.');
    }
    pluginURL = exported$1.resolveURL(url);
    pluginStatus = status.deferred;
    _completionCallback = callback;
    sendPluginStateToWorker();
    if (!deferred) {
        downloadRTLTextPlugin();
    }
};
const downloadRTLTextPlugin = function () {
    if (pluginStatus !== status.deferred || !pluginURL) {
        throw new Error('rtl-text-plugin cannot be downloaded unless a pluginURL is specified');
    }
    pluginStatus = status.loading;
    sendPluginStateToWorker();
    if (pluginURL) {
        getArrayBuffer({ url: pluginURL }, error => {
            if (error) {
                triggerPluginCompletionEvent(error);
            } else {
                pluginStatus = status.loaded;
                sendPluginStateToWorker();
            }
        });
    }
};
const plugin = {
    applyArabicShaping: null,
    processBidirectionalText: null,
    processStyledBidirectionalText: null,
    isLoaded() {
        return pluginStatus === status.loaded || // Main Thread: loaded if the completion callback returned successfully
        plugin.applyArabicShaping != null;
    },
    isLoading() {
        return pluginStatus === status.loading;
    },
    setState(state) {
        pluginStatus = state.pluginStatus;
        pluginURL = state.pluginURL;
    },
    isParsed() {
        return plugin.applyArabicShaping != null && plugin.processBidirectionalText != null && plugin.processStyledBidirectionalText != null;
    },
    getPluginURL() {
        return pluginURL;
    }
};
const lazyLoadRTLTextPlugin = function () {
    if (!plugin.isLoading() && !plugin.isLoaded() && getRTLTextPluginStatus() === 'deferred') {
        downloadRTLTextPlugin();
    }
};

class EvaluationParameters {
    // "options" may also be another EvaluationParameters to copy
    constructor(zoom, options) {
        this.zoom = zoom;
        if (options) {
            this.now = options.now;
            this.fadeDuration = options.fadeDuration;
            this.transition = options.transition;
            this.pitch = options.pitch;
            this.brightness = options.brightness;
        } else {
            this.now = 0;
            this.fadeDuration = 0;
            this.transition = {};
            this.pitch = 0;
            this.brightness = 0;
        }
    }
    isSupportedScript(str) {
        return isStringInSupportedScript(str, plugin.isLoaded());
    }
}

class PropertyValue {
    constructor(property, value, scope, options) {
        this.property = property;
        this.value = value;
        this.expression = normalizePropertyExpression(value === void 0 ? property.specification.default : value, property.specification, scope, options);
    }
    isDataDriven() {
        return this.expression.kind === 'source' || this.expression.kind === 'composite';
    }
    possiblyEvaluate(parameters, canonical, availableImages) {
        return this.property.possiblyEvaluate(this, parameters, canonical, availableImages);
    }
}
class TransitionablePropertyValue {
    constructor(property, scope, options) {
        this.property = property;
        this.value = new PropertyValue(property, void 0, scope, options);
    }
    transitioned(parameters, prior) {
        return new TransitioningPropertyValue(this.property, this.value, prior, // eslint-disable-line no-use-before-define
        extend$1({}, parameters.transition, this.transition), parameters.now);
    }
    untransitioned() {
        return new TransitioningPropertyValue(this.property, this.value, null, {}, 0);
    }
}
class Transitionable {
    constructor(properties, scope, options) {
        this._properties = properties;
        this._values = Object.create(properties.defaultTransitionablePropertyValues);
        this._scope = scope;
        this._options = options;
        this.configDependencies = /* @__PURE__ */
        new Set();
    }
    getValue(name) {
        return clone(this._values[name].value.value);
    }
    setValue(name, value) {
        if (!this._values.hasOwnProperty(name)) {
            this._values[name] = new TransitionablePropertyValue(this._values[name].property, this._scope, this._options);
        }
        this._values[name].value = new PropertyValue(this._values[name].property, value === null ? void 0 : clone(value), this._scope, this._options);
        if (this._values[name].value.expression.configDependencies) {
            this.configDependencies = /* @__PURE__ */
            new Set([
                ...this.configDependencies,
                ...this._values[name].value.expression.configDependencies
            ]);
        }
    }
    setTransitionOrValue(properties, options) {
        if (options)
            this._options = options;
        const specProperties = this._properties.properties;
        if (properties) {
            for (const name in properties) {
                const value = properties[name];
                if (endsWith(name, '-transition')) {
                    const propName = name.slice(0, -'-transition'.length);
                    if (specProperties[propName]) {
                        this.setTransition(propName, value);
                    }
                } else if (specProperties.hasOwnProperty(name)) {
                    this.setValue(name, value);
                }
            }
        }
    }
    getTransition(name) {
        return clone(this._values[name].transition);
    }
    setTransition(name, value) {
        if (!this._values.hasOwnProperty(name)) {
            this._values[name] = new TransitionablePropertyValue(this._values[name].property);
        }
        this._values[name].transition = clone(value) || void 0;
    }
    serialize() {
        const result = {};
        for (const property of Object.keys(this._values)) {
            const value = this.getValue(property);
            if (value !== void 0) {
                result[property] = value;
            }
            const transition = this.getTransition(property);
            if (transition !== void 0) {
                result[`${ property }-transition`] = transition;
            }
        }
        return result;
    }
    transitioned(parameters, prior) {
        const result = new Transitioning(this._properties);
        for (const property of Object.keys(this._values)) {
            result._values[property] = this._values[property].transitioned(parameters, prior._values[property]);
        }
        return result;
    }
    untransitioned() {
        const result = new Transitioning(this._properties);
        for (const property of Object.keys(this._values)) {
            result._values[property] = this._values[property].untransitioned();
        }
        return result;
    }
}
class TransitioningPropertyValue {
    constructor(property, value, prior, transition, now) {
        const delay = transition.delay || 0;
        const duration = transition.duration || 0;
        now = now || 0;
        this.property = property;
        this.value = value;
        this.begin = now + delay;
        this.end = this.begin + duration;
        if (property.specification.transition && (transition.delay || transition.duration)) {
            this.prior = prior;
        }
    }
    possiblyEvaluate(parameters, canonical, availableImages) {
        const now = parameters.now || 0;
        const finalValue = this.value.possiblyEvaluate(parameters, canonical, availableImages);
        const prior = this.prior;
        if (!prior) {
            return finalValue;
        } else if (now > this.end) {
            this.prior = null;
            return finalValue;
        } else if (this.value.isDataDriven()) {
            this.prior = null;
            return finalValue;
        } else if (now < this.begin) {
            return prior.possiblyEvaluate(parameters, canonical, availableImages);
        } else {
            const t = (now - this.begin) / (this.end - this.begin);
            return this.property.interpolate(prior.possiblyEvaluate(parameters, canonical, availableImages), finalValue, easeCubicInOut(t));
        }
    }
}
class Transitioning {
    constructor(properties) {
        this._properties = properties;
        this._values = Object.create(properties.defaultTransitioningPropertyValues);
    }
    possiblyEvaluate(parameters, canonical, availableImages) {
        const result = new PossiblyEvaluated(this._properties);
        for (const property of Object.keys(this._values)) {
            result._values[property] = this._values[property].possiblyEvaluate(parameters, canonical, availableImages);
        }
        return result;
    }
    hasTransition() {
        for (const property of Object.keys(this._values)) {
            if (this._values[property].prior) {
                return true;
            }
        }
        return false;
    }
}
class Layout {
    constructor(properties, scope, options) {
        this._properties = properties;
        this._values = Object.create(properties.defaultPropertyValues);
        this._scope = scope;
        this._options = options;
        this.configDependencies = /* @__PURE__ */
        new Set();
    }
    getValue(name) {
        return clone(this._values[name].value);
    }
    setValue(name, value) {
        this._values[name] = new PropertyValue(this._values[name].property, value === null ? void 0 : clone(value), this._scope, this._options);
        if (this._values[name].expression.configDependencies) {
            this.configDependencies = /* @__PURE__ */
            new Set([
                ...this.configDependencies,
                ...this._values[name].expression.configDependencies
            ]);
        }
    }
    serialize() {
        const result = {};
        for (const property of Object.keys(this._values)) {
            const value = this.getValue(property);
            if (value !== void 0) {
                result[property] = value;
            }
        }
        return result;
    }
    possiblyEvaluate(parameters, canonical, availableImages) {
        const result = new PossiblyEvaluated(this._properties);
        for (const property of Object.keys(this._values)) {
            result._values[property] = this._values[property].possiblyEvaluate(parameters, canonical, availableImages);
        }
        return result;
    }
}
class PossiblyEvaluatedPropertyValue {
    constructor(property, value, parameters) {
        this.property = property;
        this.value = value;
        this.parameters = parameters;
    }
    isConstant() {
        return this.value.kind === 'constant';
    }
    constantOr(value) {
        if (this.value.kind === 'constant') {
            return this.value.value;
        } else {
            return value;
        }
    }
    evaluate(feature, featureState, canonical, availableImages) {
        return this.property.evaluate(this.value, this.parameters, feature, featureState, canonical, availableImages);
    }
}
class PossiblyEvaluated {
    constructor(properties) {
        this._properties = properties;
        this._values = Object.create(properties.defaultPossiblyEvaluatedValues);
    }
    get(name) {
        return this._values[name];
    }
}
class DataConstantProperty {
    constructor(specification) {
        this.specification = specification;
    }
    possiblyEvaluate(value, parameters) {
        return value.expression.evaluate(parameters);
    }
    interpolate(a, b, t) {
        const interp = interpolate$1[this.specification.type];
        if (interp) {
            return interp(a, b, t);
        } else {
            return a;
        }
    }
}
class DataDrivenProperty {
    constructor(specification, overrides) {
        this.specification = specification;
        this.overrides = overrides;
    }
    possiblyEvaluate(value, parameters, canonical, availableImages) {
        if (value.expression.kind === 'constant' || value.expression.kind === 'camera') {
            return new PossiblyEvaluatedPropertyValue(this, {
                kind: 'constant',
                value: value.expression.evaluate(parameters, null, {}, canonical, availableImages)
            }, parameters);
        } else {
            return new PossiblyEvaluatedPropertyValue(this, value.expression, parameters);
        }
    }
    interpolate(a, b, t) {
        if (a.value.kind !== 'constant' || b.value.kind !== 'constant') {
            return a;
        }
        if (a.value.value === void 0 || b.value.value === void 0) {
            return new PossiblyEvaluatedPropertyValue(this, {
                kind: 'constant',
                value: void 0
            }, a.parameters);
        }
        const interp = interpolate$1[this.specification.type];
        if (interp) {
            return new PossiblyEvaluatedPropertyValue(this, {
                kind: 'constant',
                value: interp(a.value.value, b.value.value, t)
            }, a.parameters);
        } else {
            return a;
        }
    }
    evaluate(value, parameters, feature, featureState, canonical, availableImages) {
        if (value.kind === 'constant') {
            return value.value;
        } else {
            return value.evaluate(parameters, feature, featureState, canonical, availableImages);
        }
    }
}
class ColorRampProperty {
    constructor(specification) {
        this.specification = specification;
    }
    possiblyEvaluate(value, parameters, canonical, availableImages) {
        return !!value.expression.evaluate(parameters, null, {}, canonical, availableImages);
    }
    interpolate() {
        return false;
    }
}
class DirectionProperty {
    constructor(specification) {
        this.specification = specification;
    }
    possiblyEvaluate(value, parameters) {
        return sphericalDirectionToCartesian(value.expression.evaluate(parameters));
    }
    interpolate(a, b, t) {
        return {
            x: number(a.x, b.x, t),
            y: number(a.y, b.y, t),
            z: number(a.z, b.z, t)
        };
    }
}
class PositionProperty {
    constructor(specification) {
        this.specification = specification;
    }
    possiblyEvaluate(value, parameters) {
        return sphericalPositionToCartesian(value.expression.evaluate(parameters));
    }
    interpolate(a, b, t) {
        return {
            x: number(a.x, b.x, t),
            y: number(a.y, b.y, t),
            z: number(a.z, b.z, t),
            azimuthal: number(a.azimuthal, b.azimuthal, t),
            polar: number(a.polar, b.polar, t)
        };
    }
}
class Properties {
    constructor(properties) {
        this.properties = properties;
        this.defaultPropertyValues = {};
        this.defaultTransitionablePropertyValues = {};
        this.defaultTransitioningPropertyValues = {};
        this.defaultPossiblyEvaluatedValues = {};
        this.overridableProperties = [];
        const defaultParameters = new EvaluationParameters(0, {});
        for (const property in properties) {
            const prop = properties[property];
            if (prop.specification.overridable) {
                this.overridableProperties.push(property);
            }
            const defaultPropertyValue = this.defaultPropertyValues[property] = new PropertyValue(prop, void 0);
            const defaultTransitionablePropertyValue = this.defaultTransitionablePropertyValues[property] = new TransitionablePropertyValue(prop);
            this.defaultTransitioningPropertyValues[property] = defaultTransitionablePropertyValue.untransitioned();
            this.defaultPossiblyEvaluatedValues[property] = defaultPropertyValue.possiblyEvaluate(defaultParameters);
        }
    }
}
register(DataDrivenProperty, 'DataDrivenProperty');
register(DataConstantProperty, 'DataConstantProperty');
register(ColorRampProperty, 'ColorRampProperty');

var spec = JSON.parse('{"$version":8,"$root":{"version":{"required":true,"type":"enum","values":[8]},"fragment":{"type":"boolean"},"name":{"type":"string"},"metadata":{"type":"*"},"center":{"type":"array","value":"number"},"zoom":{"type":"number"},"bearing":{"type":"number","default":0,"period":360},"pitch":{"type":"number","default":0},"light":{"type":"light"},"lights":{"required":false,"type":"array","value":"light-3d"},"terrain":{"type":"terrain","optional":true},"fog":{"type":"fog"},"snow":{"type":"snow","experimental":true},"rain":{"type":"rain","experimental":true},"camera":{"type":"camera"},"color-theme":{"type":"colorTheme"},"indoor":{"type":"indoor","experimental":true},"imports":{"type":"array","value":"import"},"schema":{"type":"schema"},"sources":{"required":true,"type":"sources"},"sprite":{"type":"string"},"glyphs":{"type":"string","default":"mapbox://fonts/mapbox/{fontstack}/{range}.pbf"},"transition":{"type":"transition"},"projection":{"type":"projection"},"layers":{"required":true,"type":"array","value":"layer"},"models":{"type":"models"},"featuresets":{"experimental":true,"type":"featuresets"}},"featuresets":{"experimental":true,"*":{"type":"featureset"}},"featureset":{"experimental":true,"metadata":{"experimental":true,"type":"*"},"selectors":{"experimental":true,"type":"array","value":"selector"}},"selector":{"experimental":true,"layer":{"experimental":true,"type":"string","required":true},"properties":{"experimental":true,"type":"selectorProperty","required":false},"featureNamespace":{"experimental":true,"type":"string","required":false},"_uniqueFeatureID":{"experimental":true,"type":"boolean","private":true,"required":false}},"selectorProperty":{"experimental":true,"*":{"experimental":true,"type":"*"}},"model":{"type":"string","required":true},"import":{"id":{"type":"string","required":true},"url":{"type":"string","required":true},"config":{"type":"config"},"data":{"type":"$root"},"color-theme":{"type":"colorTheme","optional":true}},"config":{"*":{"type":"*"}},"schema":{"*":{"type":"option"}},"option":{"default":{"type":"*","property-type":"data-constant","expression":{},"required":true},"type":{"type":"enum","values":{"string":1,"number":1,"boolean":1,"color":1}},"array":{"type":"boolean"},"minValue":{"type":"number"},"maxValue":{"type":"number"},"stepValue":{"type":"number"},"values":{"type":"array","value":"*"},"metadata":{"type":"*"}},"models":{"*":{"type":"model"}},"light-3d":{"id":{"type":"string","required":true},"properties":{"type":"properties"},"type":{"type":"enum","values":{"ambient":{},"directional":{},"flat":{}}}},"properties":["properties_light_directional","properties_light_ambient","properties_light_flat"],"properties_light_directional":{"direction":{"type":"array","default":[210,30],"minimum":[0,0],"maximum":[360,90],"length":2,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"color":{"type":"color","property-type":"data-constant","default":"#ffffff","expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":0.5,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"cast-shadows":{"type":"boolean","default":false,"property-type":"data-constant"},"shadow-quality":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1,"expression":{"parameters":["zoom"]},"experimental":true},"shadow-intensity":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true}},"properties_light_ambient":{"color":{"type":"color","property-type":"data-constant","default":"#ffffff","expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":0.5,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true}},"properties_light_flat":{"anchor":{"type":"enum","default":"viewport","values":{"map":1,"viewport":1},"property-type":"data-constant","expression":{"parameters":["zoom"]}},"position":{"type":"array","default":[1.15,210,30],"length":3,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"color":{"type":"color","property-type":"data-constant","default":"#ffffff","expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":0.5,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true}},"sources":{"*":{"type":"source"}},"source":["source_vector","source_raster","source_raster_dem","source_raster_array","source_geojson","source_video","source_image","source_model"],"source_vector":{"type":{"required":true,"type":"enum","values":{"vector":1}},"url":{"type":"string"},"tiles":{"type":"array","value":"string"},"bounds":{"type":"array","value":"number","length":4,"default":[-180,-85.051129,180,85.051129]},"scheme":{"type":"enum","values":{"xyz":1,"tms":1},"default":"xyz"},"minzoom":{"type":"number","default":0},"maxzoom":{"type":"number","default":22},"attribution":{"type":"string"},"promoteId":{"type":"promoteId"},"volatile":{"type":"boolean","default":false},"*":{"type":"*"}},"source_raster":{"type":{"required":true,"type":"enum","values":{"raster":1}},"url":{"type":"string"},"tiles":{"type":"array","value":"string"},"bounds":{"type":"array","value":"number","length":4,"default":[-180,-85.051129,180,85.051129]},"minzoom":{"type":"number","default":0},"maxzoom":{"type":"number","default":22},"tileSize":{"type":"number","default":512},"scheme":{"type":"enum","values":{"xyz":1,"tms":1},"default":"xyz"},"attribution":{"type":"string"},"volatile":{"type":"boolean","default":false},"*":{"type":"*"}},"source_raster_dem":{"type":{"required":true,"type":"enum","values":{"raster-dem":1}},"url":{"type":"string"},"tiles":{"type":"array","value":"string"},"bounds":{"type":"array","value":"number","length":4,"default":[-180,-85.051129,180,85.051129]},"minzoom":{"type":"number","default":0},"maxzoom":{"type":"number","default":22},"tileSize":{"type":"number","default":512},"attribution":{"type":"string"},"encoding":{"type":"enum","values":{"terrarium":1,"mapbox":1},"default":"mapbox"},"volatile":{"type":"boolean","default":false},"*":{"type":"*"}},"source_raster_array":{"experimental":true,"type":{"required":true,"type":"enum","values":{"raster-array":1}},"url":{"type":"string"},"tiles":{"type":"array","value":"string"},"bounds":{"type":"array","value":"number","length":4,"default":[-180,-85.051129,180,85.051129]},"minzoom":{"type":"number","default":0},"maxzoom":{"type":"number","default":22},"tileSize":{"type":"number","default":512},"attribution":{"type":"string"},"rasterLayers":{"type":"*"},"volatile":{"type":"boolean","default":false},"*":{"type":"*"}},"source_geojson":{"type":{"required":true,"type":"enum","values":{"geojson":1}},"data":{"type":"*"},"maxzoom":{"type":"number","default":18},"minzoom":{"type":"number","default":0},"attribution":{"type":"string"},"buffer":{"type":"number","default":128,"maximum":512,"minimum":0},"filter":{"type":"*"},"tolerance":{"type":"number","default":0.375},"cluster":{"type":"boolean","default":false},"clusterRadius":{"type":"number","default":50,"minimum":0},"clusterMaxZoom":{"type":"number"},"clusterMinPoints":{"type":"number"},"clusterProperties":{"type":"*"},"lineMetrics":{"type":"boolean","default":false},"generateId":{"type":"boolean","default":false},"promoteId":{"type":"promoteId"},"dynamic":{"type":"boolean","default":false}},"source_video":{"type":{"required":true,"type":"enum","values":{"video":1}},"urls":{"required":true,"type":"array","value":"string"},"coordinates":{"required":true,"type":"array","length":4,"value":{"type":"array","length":2,"value":"number"}}},"source_image":{"type":{"required":true,"type":"enum","values":{"image":1}},"url":{"required":false,"type":"string"},"coordinates":{"required":true,"type":"array","length":4,"value":{"type":"array","length":2,"value":"number"}}},"source_model":{"type":{"required":true,"type":"enum","values":{"model":1,"batched-model":1}},"maxzoom":{"type":"number","default":18},"minzoom":{"type":"number","default":0},"tiles":{"type":"array","value":"string"}},"layer":{"id":{"type":"string","required":true},"type":{"type":"enum","values":{"fill":{},"line":{},"symbol":{},"circle":{},"heatmap":{},"fill-extrusion":{},"raster":{},"raster-particle":{"experimental":true},"hillshade":{},"model":{"experimental":true},"background":{},"sky":{},"slot":{},"clip":{}},"required":true},"metadata":{"type":"*"},"source":{"type":"string"},"source-layer":{"type":"string"},"slot":{"type":"string"},"minzoom":{"type":"number","minimum":0,"maximum":24},"maxzoom":{"type":"number","minimum":0,"maximum":24},"filter":{"type":"filter"},"layout":{"type":"layout"},"paint":{"type":"paint"}},"layout":["layout_clip","layout_fill","layout_line","layout_circle","layout_heatmap","layout_fill-extrusion","layout_symbol","layout_raster","layout_raster-particle","layout_hillshade","layout_background","layout_sky","layout_model"],"layout_background":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_sky":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_model":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"},"model-id":{"type":"string","default":"","property-type":"data-driven","expression":{"parameters":["zoom","feature"]}}},"layout_clip":{"clip-layer-types":{"type":"array","value":"enum","values":{"model":1,"symbol":1},"default":[],"expression":{},"property-type":"data-constant"},"clip-layer-scope":{"type":"array","value":"string","default":[],"expression":{},"property-type":"data-constant"}},"layout_fill":{"fill-sort-key":{"type":"number","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"},"fill-elevation-reference":{"type":"enum","values":{"none":1,"hd-road-base":1,"hd-road-markup":1},"default":"none","experimental":true,"private":true,"expression":{},"property-type":"data-constant"}},"layout_circle":{"circle-sort-key":{"type":"number","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"circle-elevation-reference":{"type":"enum","values":{"none":1,"hd-road-markup":1},"default":"none","experimental":true,"expression":{},"property-type":"data-constant"},"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_heatmap":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_fill-extrusion":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"},"fill-extrusion-edge-radius":{"type":"number","experimental":true,"default":0,"minimum":0,"maximum":1,"expression":{},"property-type":"constant"}},"layout_line":{"line-cap":{"type":"enum","values":{"butt":1,"round":1,"square":1},"default":"butt","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"line-join":{"type":"enum","values":{"bevel":1,"round":1,"miter":1,"none":1},"default":"miter","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"line-miter-limit":{"type":"number","default":2,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"line-round-limit":{"type":"number","default":1.05,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"line-sort-key":{"type":"number","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"line-z-offset":{"type":"number","experimental":true,"default":0,"expression":{"parameters":["zoom","feature","line-progress"]},"property-type":"data-driven"},"line-elevation-reference":{"type":"enum","values":{"none":1,"sea":1,"ground":1,"hd-road-markup":1},"default":"none","experimental":true,"expression":{},"property-type":"data-constant"},"line-cross-slope":{"type":"number","experimental":true,"expression":{},"property-type":"constant"},"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"},"line-width-unit":{"type":"enum","values":{"pixels":1,"meters":1},"default":"pixels","experimental":true,"private":true,"expression":{"parameters":["zoom"]},"property-type":"data-constant"}},"layout_symbol":{"symbol-placement":{"type":"enum","values":{"point":1,"line":1,"line-center":1},"default":"point","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"symbol-spacing":{"type":"number","default":250,"minimum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"symbol-avoid-edges":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"symbol-sort-key":{"type":"number","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"symbol-z-order":{"type":"enum","values":{"auto":1,"viewport-y":1,"source":1},"default":"auto","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"symbol-z-elevate":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"symbol-elevation-reference":{"type":"enum","values":{"sea":1,"ground":1,"hd-road-markup":1},"default":"ground","experimental":true,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-allow-overlap":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-ignore-placement":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-optional":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-rotation-alignment":{"type":"enum","values":{"map":1,"viewport":1,"auto":1},"default":"auto","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-size":{"type":"number","default":1,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-size-scale-range":{"type":"array","value":"number","length":2,"default":[0.8,2],"minimum":0.1,"maximum":10,"experimental":true,"private":true,"expression":{},"property-type":"data-constant"},"icon-text-fit":{"type":"enum","values":{"none":1,"width":1,"height":1,"both":1},"default":"none","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-text-fit-padding":{"type":"array","value":"number","length":4,"default":[0,0,0,0],"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-image":{"type":"resolvedImage","tokens":true,"expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-rotate":{"type":"number","default":0,"period":360,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-padding":{"type":"number","default":2,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"icon-keep-upright":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-offset":{"type":"array","value":"number","length":2,"default":[0,0],"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-anchor":{"type":"enum","values":{"center":1,"left":1,"right":1,"top":1,"bottom":1,"top-left":1,"top-right":1,"bottom-left":1,"bottom-right":1},"default":"center","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"icon-pitch-alignment":{"type":"enum","values":{"map":1,"viewport":1,"auto":1},"default":"auto","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-pitch-alignment":{"type":"enum","values":{"map":1,"viewport":1,"auto":1},"default":"auto","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-rotation-alignment":{"type":"enum","values":{"map":1,"viewport":1,"auto":1},"default":"auto","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-field":{"type":"formatted","default":"","tokens":true,"expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-font":{"type":"array","value":"string","default":["Open Sans Regular","Arial Unicode MS Regular"],"expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-size":{"type":"number","default":16,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-size-scale-range":{"type":"array","value":"number","length":2,"default":[0.8,2],"minimum":0.1,"maximum":10,"experimental":true,"private":true,"expression":{},"property-type":"data-constant"},"text-max-width":{"type":"number","default":10,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-line-height":{"type":"number","default":1.2,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-letter-spacing":{"type":"number","default":0,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-justify":{"type":"enum","values":{"auto":1,"left":1,"center":1,"right":1},"default":"center","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-radial-offset":{"type":"number","default":0,"property-type":"data-driven","expression":{"interpolated":true,"parameters":["zoom","feature"]}},"text-variable-anchor":{"type":"array","value":"enum","values":{"center":1,"left":1,"right":1,"top":1,"bottom":1,"top-left":1,"top-right":1,"bottom-left":1,"bottom-right":1},"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-anchor":{"type":"enum","values":{"center":1,"left":1,"right":1,"top":1,"bottom":1,"top-left":1,"top-right":1,"bottom-left":1,"bottom-right":1},"default":"center","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-max-angle":{"type":"number","default":45,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"text-writing-mode":{"type":"array","value":"enum","values":{"horizontal":1,"vertical":1},"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-rotate":{"type":"number","default":0,"period":360,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-padding":{"type":"number","default":2,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"text-keep-upright":{"type":"boolean","default":true,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-transform":{"type":"enum","values":{"none":1,"uppercase":1,"lowercase":1},"default":"none","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-offset":{"type":"array","value":"number","length":2,"default":[0,0],"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"},"text-allow-overlap":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-ignore-placement":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"text-optional":{"type":"boolean","default":false,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_raster":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_raster-particle":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"layout_hillshade":{"visibility":{"type":"enum","values":{"visible":1,"none":1},"default":"visible","expression":{},"property-type":"constant"}},"filter":{"type":"array","value":"*"},"filter_symbol":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature","pitch","distance-from-center"]}},"filter_fill":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_hillshade":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_raster":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_raster-particle":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_clip":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_model":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_line":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_circle":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_fill-extrusion":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_heatmap":{"type":"boolean","default":false,"property-type":"data-driven","expression":{"parameters":["zoom","feature"]}},"filter_operator":{"type":"enum","values":{"==":1,"!=":1,">":1,">=":1,"<":1,"<=":1,"in":1,"!in":1,"all":1,"any":1,"none":1,"has":1,"!has":1}},"geometry_type":{"type":"enum","values":{"Point":1,"LineString":1,"Polygon":1}},"function":{"expression":{"type":"expression"},"stops":{"type":"array","value":"function_stop"},"base":{"type":"number","default":1,"minimum":0},"property":{"type":"string","default":"$zoom"},"type":{"type":"enum","values":{"identity":1,"exponential":1,"interval":1,"categorical":1},"default":"exponential"},"colorSpace":{"type":"enum","values":{"rgb":1,"lab":1,"hcl":1},"default":"rgb"},"default":{"type":"*","required":false}},"function_stop":{"type":"array","minimum":0,"maximum":24,"value":["number","color"],"length":2},"expression":{"type":"array","value":"*","minimum":1},"fog":{"range":{"type":"array","default":[0.5,10],"minimum":-20,"maximum":20,"length":2,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true}},"color":{"type":"color","property-type":"data-constant","default":"#ffffff","expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"high-color":{"type":"color","property-type":"data-constant","default":"#245cdf","expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"space-color":{"type":"color","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],4,"#010b19",7,"#367ab9"],"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"horizon-blend":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],4,0.2,7,0.1],"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"star-intensity":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],5,0.35,6,0],"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"vertical-range":{"type":"array","default":[0,0],"minimum":0,"length":2,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true}}},"snow":{"density":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],11,0,13,0.85],"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"color":{"type":"color","property-type":"data-constant","default":"#ffffff","experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"opacity":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"vignette":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],11,0,13,0.3],"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"vignette-color":{"type":"color","property-type":"data-constant","default":"#ffffff","experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"center-thinning":{"type":"number","property-type":"data-constant","default":0.4,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"direction":{"type":"array","default":[0,50],"minimum":0,"maximum":360,"length":2,"value":"number","property-type":"data-constant","transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true}},"flake-size":{"type":"number","property-type":"data-constant","default":0.71,"minimum":0,"maximum":5,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true}},"rain":{"density":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],11,0,13,0.5],"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"color":{"type":"color","property-type":"data-constant","default":["interpolate",["linear"],["measure-light","brightness"],0,"#03113d",0.3,"#a8adbc"],"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"opacity":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["measure-light","brightness"],0,0.88,1,0.7],"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"vignette":{"type":"number","property-type":"data-constant","default":["interpolate",["linear"],["zoom"],11,0,13,1],"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"vignette-color":{"type":"color","property-type":"data-constant","default":["interpolate",["linear"],["measure-light","brightness"],0,"#001736",0.3,"#464646"],"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"center-thinning":{"type":"number","property-type":"data-constant","default":0.57,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true},"direction":{"type":"array","default":[0,80],"minimum":0,"maximum":360,"length":2,"value":"number","property-type":"data-constant","transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true}},"droplet-size":{"type":"array","default":[2.6,18.2],"minimum":0,"maximum":50,"length":2,"value":"number","property-type":"data-constant","transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true}},"distortion-strength":{"type":"number","property-type":"data-constant","default":0.7,"minimum":0,"maximum":1,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"],"relaxZoomRestriction":true},"transition":true}},"camera":{"camera-projection":{"type":"enum","values":{"perspective":1,"orthographic":1},"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"default":"perspective","property-type":"data-constant"}},"colorTheme":{"data":{"type":"string","property-type":"data-constant","expression":{}}},"indoor":{"floorplanFeaturesetId":{"type":"string","experimental":true,"property-type":"data-constant","expression":{}},"buildingFeaturesetId":{"type":"string","experimental":true,"property-type":"data-constant","expression":{}}},"light":{"anchor":{"type":"enum","default":"viewport","values":{"map":1,"viewport":1},"property-type":"data-constant","expression":{"parameters":["zoom"]}},"position":{"type":"array","default":[1.15,210,30],"length":3,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"color":{"type":"color","property-type":"data-constant","default":"#ffffff","expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"intensity":{"type":"number","property-type":"data-constant","default":0.5,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true}},"projection":{"name":{"type":"enum","values":{"albers":1,"equalEarth":1,"equirectangular":1,"lambertConformalConic":1,"mercator":1,"naturalEarth":1,"winkelTripel":1,"globe":1},"default":"mercator","required":true},"center":{"type":"array","length":2,"value":"number","property-type":"data-constant","minimum":[-180,-90],"maximum":[180,90]},"parallels":{"type":"array","length":2,"value":"number","property-type":"data-constant","minimum":[-90,-90],"maximum":[90,90]}},"terrain":{"source":{"type":"string","required":true},"exaggeration":{"type":"number","property-type":"data-constant","default":1,"minimum":0,"maximum":1000,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true}},"paint":["paint_fill","paint_line","paint_circle","paint_heatmap","paint_fill-extrusion","paint_symbol","paint_raster","paint_raster-particle","paint_hillshade","paint_background","paint_sky","paint_model"],"paint_fill":{"fill-antialias":{"type":"boolean","default":true,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"fill-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"fill-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"fill-outline-color":{"type":"color","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"fill-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"fill-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"fill-pattern":{"type":"resolvedImage","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"fill-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"fill-z-offset":{"type":"number","default":0,"minimum":0,"transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"}},"paint_fill-extrusion":{"fill-extrusion-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"fill-extrusion-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"fill-extrusion-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"fill-extrusion-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"fill-extrusion-pattern":{"type":"resolvedImage","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"fill-extrusion-height":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state"]},"property-type":"data-driven"},"fill-extrusion-base":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state"]},"property-type":"data-driven"},"fill-extrusion-height-alignment":{"type":"enum","experimental":true,"values":{"terrain":1,"flat":1},"default":"flat","property-type":"data-constant"},"fill-extrusion-base-alignment":{"type":"enum","experimental":true,"values":{"terrain":1,"flat":1},"default":"terrain","property-type":"data-constant"},"fill-extrusion-vertical-gradient":{"type":"boolean","default":true,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"fill-extrusion-ambient-occlusion-intensity":{"property-type":"data-constant","type":"number","default":0,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"fill-extrusion-ambient-occlusion-radius":{"property-type":"data-constant","type":"number","default":3,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"fill-extrusion-ambient-occlusion-wall-radius":{"property-type":"data-constant","type":"number","experimental":true,"default":3,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"fill-extrusion-ambient-occlusion-ground-radius":{"property-type":"data-constant","type":"number","experimental":true,"default":3,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true},"fill-extrusion-ambient-occlusion-ground-attenuation":{"property-type":"data-constant","type":"number","experimental":true,"default":0.69,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"fill-extrusion-flood-light-color":{"property-type":"data-constant","type":"color","experimental":true,"default":"#ffffff","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]}},"fill-extrusion-flood-light-intensity":{"property-type":"data-constant","type":"number","experimental":true,"default":0,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]}},"fill-extrusion-flood-light-wall-radius":{"property-type":"data-driven","type":"number","experimental":true,"default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["feature","feature-state"]}},"fill-extrusion-flood-light-ground-radius":{"property-type":"data-driven","type":"number","experimental":true,"default":0,"transition":true,"expression":{"interpolated":true,"parameters":["feature","feature-state"]}},"fill-extrusion-flood-light-ground-attenuation":{"property-type":"data-constant","type":"number","experimental":true,"default":0.69,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"fill-extrusion-vertical-scale":{"property-type":"data-constant","type":"number","experimental":true,"default":1,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"fill-extrusion-rounded-roof":{"property-type":"data-constant","type":"boolean","default":true,"experimental":true,"expression":{"parameters":["zoom"]}},"fill-extrusion-cutoff-fade-range":{"type":"number","default":0,"minimum":0,"maximum":1,"expression":{},"property-type":"data-constant"},"fill-extrusion-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light","feature-state"]},"property-type":"data-driven"},"fill-extrusion-line-width":{"type":"number","default":0,"minimum":0,"transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"fill-extrusion-cast-shadows":{"type":"boolean","default":true,"property-type":"data-constant"}},"paint_line":{"line-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"line-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"line-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"line-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"line-width":{"type":"number","default":1,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light","line-progress"]},"property-type":"data-driven"},"line-gap-width":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"line-offset":{"type":"number","default":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"line-blur":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"line-dasharray":{"type":"array","value":"number","minimum":0,"expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"line-pattern":{"type":"resolvedImage","expression":{"parameters":["zoom","feature"]},"property-type":"data-driven"},"line-gradient":{"type":"color","expression":{"interpolated":true,"parameters":["line-progress"]},"property-type":"color-ramp"},"line-trim-offset":{"type":"array","value":"number","length":2,"default":[0,0],"minimum":[0,0],"maximum":[1,1],"property-type":"constant"},"line-trim-fade-range":{"type":"array","value":"number","experimental":true,"length":2,"default":[0,0],"minimum":[0,0],"maximum":[1,1],"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"line-trim-color":{"type":"color","experimental":true,"default":"transparent","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"line-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"line-border-width":{"type":"number","private":true,"default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state"]},"property-type":"data-driven"},"line-border-color":{"type":"color","private":true,"default":"rgba(0, 0, 0, 0)","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state"]},"property-type":"data-driven"},"line-occlusion-opacity":{"type":"number","default":0,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"transition":true,"property-type":"data-constant"}},"paint_circle":{"circle-radius":{"type":"number","default":5,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-blur":{"type":"number","default":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"circle-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"circle-pitch-scale":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"circle-pitch-alignment":{"type":"enum","values":{"map":1,"viewport":1},"default":"viewport","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"circle-stroke-width":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-stroke-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-stroke-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"circle-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"}},"paint_heatmap":{"heatmap-radius":{"type":"number","default":30,"minimum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"heatmap-weight":{"type":"number","default":1,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"heatmap-intensity":{"type":"number","default":1,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"heatmap-color":{"type":"color","default":["interpolate",["linear"],["heatmap-density"],0,"rgba(0, 0, 255, 0)",0.1,"royalblue",0.3,"cyan",0.5,"lime",0.7,"yellow",1,"red"],"expression":{"interpolated":true,"parameters":["heatmap-density"]},"property-type":"color-ramp"},"heatmap-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"}},"paint_symbol":{"icon-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-occlusion-opacity":{"type":"number","minimum":0,"maximum":1,"default":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-emissive-strength":{"type":"number","default":1,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light","feature-state"]},"property-type":"data-driven"},"text-emissive-strength":{"type":"number","default":1,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light","feature-state"]},"property-type":"data-driven"},"icon-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-halo-color":{"type":"color","default":"rgba(0, 0, 0, 0)","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-halo-width":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-halo-blur":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"icon-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"icon-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-image-cross-fade":{"type":"number","property-type":"data-driven","default":0,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"transition":true},"text-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-occlusion-opacity":{"type":"number","minimum":0,"maximum":1,"default":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-color":{"type":"color","default":"#000000","transition":true,"overridable":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-halo-color":{"type":"color","default":"rgba(0, 0, 0, 0)","transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-halo-width":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-halo-blur":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","feature","feature-state","measure-light"]},"property-type":"data-driven"},"text-translate":{"type":"array","value":"number","length":2,"default":[0,0],"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"text-translate-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"icon-color-saturation":{"type":"number","default":0,"minimum":-1,"maximum":1,"expression":{},"property-type":"data-constant"},"icon-color-contrast":{"type":"number","default":0,"minimum":-1,"maximum":1,"expression":{},"property-type":"data-constant"},"icon-color-brightness-min":{"type":"number","default":0,"minimum":0,"maximum":1,"expression":{},"property-type":"data-constant"},"icon-color-brightness-max":{"type":"number","default":1,"minimum":0,"maximum":1,"expression":{},"property-type":"data-constant"},"symbol-z-offset":{"type":"number","default":0,"minimum":0,"transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom","feature"]},"property-type":"data-driven"}},"paint_raster":{"raster-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-color":{"type":"color","expression":{"interpolated":true,"parameters":["raster-value"]},"property-type":"color-ramp"},"raster-color-mix":{"type":"array","default":[0.2126,0.7152,0.0722,0],"length":4,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"raster-color-range":{"type":"array","length":2,"value":"number","property-type":"data-constant","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]}},"raster-hue-rotate":{"type":"number","default":0,"period":360,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-brightness-min":{"type":"number","default":0,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-brightness-max":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-saturation":{"type":"number","default":0,"minimum":-1,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-contrast":{"type":"number","default":0,"minimum":-1,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-resampling":{"type":"enum","values":{"linear":1,"nearest":1},"default":"linear","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"raster-fade-duration":{"type":"number","default":300,"minimum":0,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"raster-array-band":{"type":"string","required":false,"experimental":true,"property-type":"data-constant"},"raster-elevation":{"type":"number","default":0,"minimum":0,"transition":true,"experimental":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"}},"paint_raster-particle":{"raster-particle-array-band":{"type":"string","required":false,"property-type":"data-constant"},"raster-particle-count":{"type":"number","default":512,"minimum":1,"property-type":"data-constant"},"raster-particle-color":{"type":"color","expression":{"interpolated":true,"parameters":["raster-particle-speed"]},"property-type":"color-ramp"},"raster-particle-max-speed":{"type":"number","default":1,"minimum":1,"property-type":"data-constant"},"raster-particle-speed-factor":{"type":"number","default":0.2,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-particle-fade-opacity-factor":{"type":"number","default":0.98,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"raster-particle-reset-rate-factor":{"type":"number","default":0.8,"minimum":0,"maximum":1,"property-type":"data-constant"},"raster-particle-elevation":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"}},"paint_hillshade":{"hillshade-illumination-direction":{"type":"number","default":335,"minimum":0,"maximum":359,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"hillshade-illumination-anchor":{"type":"enum","values":{"map":1,"viewport":1},"default":"viewport","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"hillshade-exaggeration":{"type":"number","default":0.5,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"hillshade-shadow-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"hillshade-highlight-color":{"type":"color","default":"#FFFFFF","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"hillshade-accent-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"},"hillshade-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"}},"paint_background":{"background-pitch-alignment":{"type":"enum","values":{"map":1,"viewport":1},"default":"map","expression":{"parameters":[]},"experimental":true,"property-type":"data-constant"},"background-color":{"type":"color","default":"#000000","transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"background-pattern":{"type":"resolvedImage","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"background-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"},"background-emissive-strength":{"type":"number","default":0,"minimum":0,"transition":true,"expression":{"interpolated":true,"parameters":["zoom","measure-light"]},"property-type":"data-constant"}},"paint_sky":{"sky-type":{"type":"enum","values":{"gradient":1,"atmosphere":1},"default":"atmosphere","expression":{"parameters":["zoom"]},"property-type":"data-constant"},"sky-atmosphere-sun":{"type":"array","value":"number","length":2,"minimum":[0,0],"maximum":[360,180],"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"sky-atmosphere-sun-intensity":{"type":"number","default":10,"minimum":0,"maximum":100,"property-type":"data-constant"},"sky-gradient-center":{"type":"array","value":"number","default":[0,0],"length":2,"minimum":[0,0],"maximum":[360,180],"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"sky-gradient-radius":{"type":"number","default":90,"minimum":0,"maximum":180,"expression":{"parameters":["zoom"]},"property-type":"data-constant"},"sky-gradient":{"type":"color","default":["interpolate",["linear"],["sky-radial-progress"],0.8,"#87ceeb",1,"white"],"expression":{"interpolated":true,"parameters":["sky-radial-progress"]},"property-type":"color-ramp"},"sky-atmosphere-halo-color":{"type":"color","default":"white","property-type":"data-constant"},"sky-atmosphere-color":{"type":"color","default":"white","property-type":"data-constant"},"sky-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant"}},"paint_model":{"model-opacity":{"type":"number","default":1,"minimum":0,"maximum":1,"transition":true,"expression":{"interpolated":true,"parameters":["feature","feature-state","zoom"]},"property-type":"data-driven"},"model-rotation":{"type":"array","value":"number","length":3,"default":[0,0,0],"period":360,"property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state","zoom"]},"transition":true},"model-scale":{"type":"array","value":"number","length":3,"default":[1,1,1],"property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state","zoom"]},"transition":true},"model-translation":{"type":"array","value":"number","length":3,"default":[0,0,0],"property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state","zoom"]},"transition":true},"model-color":{"type":"color","default":"#ffffff","property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state","measure-light","zoom"]},"transition":true},"model-color-mix-intensity":{"type":"number","property-type":"data-driven","default":0,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["feature","feature-state","measure-light"]},"transition":true},"model-type":{"type":"enum","values":{"common-3d":1,"location-indicator":1},"default":"common-3d","property-type":"data-constant"},"model-cast-shadows":{"type":"boolean","default":true,"property-type":"data-constant"},"model-receive-shadows":{"type":"boolean","default":true,"property-type":"data-constant"},"model-ambient-occlusion-intensity":{"type":"number","default":1,"minimum":0,"maximum":1,"expression":{"interpolated":true,"parameters":["zoom"]},"property-type":"data-constant","transition":true},"model-emissive-strength":{"type":"number","property-type":"data-driven","default":0,"minimum":0,"maximum":5,"expression":{"interpolated":true,"parameters":["feature","feature-state","measure-light"]},"transition":true},"model-roughness":{"type":"number","default":1,"minimum":0,"maximum":1,"property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state"]},"transition":true},"model-height-based-emissive-strength-multiplier":{"type":"array","default":[1,1,1,1,0],"length":5,"value":"number","property-type":"data-driven","expression":{"interpolated":true,"parameters":["feature","feature-state","measure-light"]},"transition":true},"model-cutoff-fade-range":{"type":"number","default":0,"minimum":0,"maximum":1,"expression":{},"property-type":"data-constant"},"model-front-cutoff":{"type":"array","private":true,"value":"number","property-type":"data-constant","expression":{"interpolated":true,"parameters":["zoom"]},"length":3,"default":[0,0,1],"minimum":[0,0,0],"maximum":[1,1,1]}},"transition":{"duration":{"type":"number","default":300,"minimum":0},"delay":{"type":"number","default":0,"minimum":0}},"property-type":{"data-driven":{"type":"property-type"},"color-ramp":{"type":"property-type"},"data-constant":{"type":"property-type"},"constant":{"type":"property-type"}},"promoteId":{"*":{"type":"*"}}}');

function unbundle(value) {
    if (value instanceof Number || value instanceof String || value instanceof Boolean) {
        return value.valueOf();
    } else {
        return value;
    }
}
function deepUnbundle(value) {
    if (Array.isArray(value)) {
        return value.map(deepUnbundle);
    } else if (value instanceof Object && !(value instanceof Number || value instanceof String || value instanceof Boolean)) {
        const unbundledValue = {};
        for (const key in value) {
            unbundledValue[key] = deepUnbundle(value[key]);
        }
        return unbundledValue;
    }
    return unbundle(value);
}

function isExpressionFilter(filter) {
    if (filter === true || filter === false) {
        return true;
    }
    if (!Array.isArray(filter) || filter.length === 0) {
        return false;
    }
    switch (filter[0]) {
    case 'has':
        return filter.length >= 2 && filter[1] !== '$id' && filter[1] !== '$type';
    case 'in':
        return filter.length >= 3 && (typeof filter[1] !== 'string' || Array.isArray(filter[2]));
    case '!in':
    case '!has':
    case 'none':
        return false;
    case '==':
    case '!=':
    case '>':
    case '>=':
    case '<':
    case '<=':
        return filter.length !== 3 || (Array.isArray(filter[1]) || Array.isArray(filter[2]));
    case 'any':
    case 'all':
        for (const f of filter.slice(1)) {
            if (!isExpressionFilter(f) && typeof f !== 'boolean') {
                return false;
            }
        }
        return true;
    default:
        return true;
    }
}
function createFilter(filter, scope = '', options = null, layerType = 'fill') {
    if (filter === null || filter === void 0) {
        return {
            filter: () => true,
            needGeometry: false,
            needFeature: false
        };
    }
    if (!isExpressionFilter(filter)) {
        filter = convertFilter(filter);
    }
    const filterExp = filter;
    let staticFilter = true;
    try {
        staticFilter = extractStaticFilter(filterExp);
    } catch (e) {
        console.warn(`Failed to extract static filter. Filter will continue working, but at higher memory usage and slower framerate.
This is most likely a bug, please report this via https://github.com/mapbox/mapbox-gl-js/issues/new?assignees=&labels=&template=Bug_report.md
and paste the contents of this message in the report.
Thank you!
Filter Expression:
${ JSON.stringify(filterExp, null, 2) }
        `);
    }
    let filterFunc = null;
    let filterSpec = null;
    if (layerType !== 'background' && layerType !== 'sky' && layerType !== 'slot') {
        filterSpec = spec[`filter_${ layerType }`];
        const compiledStaticFilter = createExpression(staticFilter, filterSpec, scope, options);
        if (compiledStaticFilter.result === 'error') {
            throw new Error(compiledStaticFilter.value.map(err => `${ err.key }: ${ err.message }`).join(', '));
        } else {
            filterFunc = (globalProperties, feature, canonical) => compiledStaticFilter.value.evaluate(globalProperties, feature, {}, canonical);
        }
    }
    let dynamicFilterFunc = null;
    let needFeature = null;
    if (staticFilter !== filterExp) {
        const compiledDynamicFilter = createExpression(filterExp, filterSpec, scope, options);
        if (compiledDynamicFilter.result === 'error') {
            throw new Error(compiledDynamicFilter.value.map(err => `${ err.key }: ${ err.message }`).join(', '));
        } else {
            dynamicFilterFunc = (globalProperties, feature, canonical, featureTileCoord, featureDistanceData) => compiledDynamicFilter.value.evaluate(globalProperties, feature, {}, canonical, void 0, void 0, featureTileCoord, featureDistanceData);
            needFeature = !isFeatureConstant(compiledDynamicFilter.value.expression);
        }
    }
    filterFunc = filterFunc;
    const needGeometry = geometryNeeded(staticFilter);
    return {
        filter: filterFunc,
        dynamicFilter: dynamicFilterFunc ? dynamicFilterFunc : void 0,
        needGeometry,
        needFeature: !!needFeature
    };
}
function extractStaticFilter(filter) {
    if (!isDynamicFilter(filter)) {
        return filter;
    }
    let result = deepUnbundle(filter);
    unionDynamicBranches(result);
    result = collapseDynamicBooleanExpressions(result);
    return result;
}
function collapseDynamicBooleanExpressions(expression) {
    if (!Array.isArray(expression)) {
        return expression;
    }
    const collapsed = collapsedExpression(expression);
    if (collapsed === true) {
        return collapsed;
    } else {
        return collapsed.map(subExpression => collapseDynamicBooleanExpressions(subExpression));
    }
}
function unionDynamicBranches(filter) {
    let isBranchingDynamically = false;
    const branches = [];
    if (filter[0] === 'case') {
        for (let i = 1; i < filter.length - 1; i += 2) {
            isBranchingDynamically = isBranchingDynamically || isDynamicFilter(filter[i]);
            branches.push(filter[i + 1]);
        }
        branches.push(filter[filter.length - 1]);
    } else if (filter[0] === 'match') {
        isBranchingDynamically = isBranchingDynamically || isDynamicFilter(filter[1]);
        for (let i = 2; i < filter.length - 1; i += 2) {
            branches.push(filter[i + 1]);
        }
        branches.push(filter[filter.length - 1]);
    } else if (filter[0] === 'step') {
        isBranchingDynamically = isBranchingDynamically || isDynamicFilter(filter[1]);
        for (let i = 1; i < filter.length - 1; i += 2) {
            branches.push(filter[i + 1]);
        }
    }
    if (isBranchingDynamically) {
        filter.length = 0;
        filter.push('any', ...branches);
    }
    for (let i = 1; i < filter.length; i++) {
        unionDynamicBranches(filter[i]);
    }
}
function isDynamicFilter(filter) {
    if (!Array.isArray(filter)) {
        return false;
    }
    if (isRootExpressionDynamic(filter[0])) {
        return true;
    }
    for (let i = 1; i < filter.length; i++) {
        const child = filter[i];
        if (isDynamicFilter(child)) {
            return true;
        }
    }
    return false;
}
function isRootExpressionDynamic(expression) {
    return expression === 'pitch' || expression === 'distance-from-center';
}
const dynamicConditionExpressions = /* @__PURE__ */
new Set([
    'in',
    '==',
    '!=',
    '>',
    '>=',
    '<',
    '<=',
    'to-boolean'
]);
function collapsedExpression(expression) {
    if (dynamicConditionExpressions.has(expression[0])) {
        for (let i = 1; i < expression.length; i++) {
            const param = expression[i];
            if (isDynamicFilter(param)) {
                return true;
            }
        }
    }
    return expression;
}
function compare(a, b) {
    return a < b ? -1 : a > b ? 1 : 0;
}
function geometryNeeded(filter) {
    if (!Array.isArray(filter))
        return false;
    if (filter[0] === 'within' || filter[0] === 'distance')
        return true;
    for (let index = 1; index < filter.length; index++) {
        if (geometryNeeded(filter[index]))
            return true;
    }
    return false;
}
function convertFilter(filter) {
    if (!filter)
        return true;
    const op = filter[0];
    if (filter.length <= 1)
        return op !== 'any';
    const converted = op === '==' ? convertComparisonOp(filter[1], filter[2], '==') : op === '!=' ? convertNegation(convertComparisonOp(filter[1], filter[2], '==')) : op === '<' || op === '>' || op === '<=' || op === '>=' ? convertComparisonOp(filter[1], filter[2], op) : op === 'any' ? convertDisjunctionOp(filter.slice(1)) : // @ts-expect-error - TS2769 - No overload matches this call.
    op === 'all' ? ['all'].concat(filter.slice(1).map(convertFilter)) : // @ts-expect-error - TS2769 - No overload matches this call.
    op === 'none' ? ['all'].concat(filter.slice(1).map(convertFilter).map(convertNegation)) : op === 'in' ? convertInOp(filter[1], filter.slice(2)) : op === '!in' ? convertNegation(convertInOp(filter[1], filter.slice(2))) : op === 'has' ? convertHasOp(filter[1]) : op === '!has' ? convertNegation(convertHasOp(filter[1])) : true;
    return converted;
}
function convertComparisonOp(property, value, op) {
    switch (property) {
    case '$type':
        return [
            `filter-type-${ op }`,
            value
        ];
    case '$id':
        return [
            `filter-id-${ op }`,
            value
        ];
    default:
        return [
            `filter-${ op }`,
            property,
            value
        ];
    }
}
function convertDisjunctionOp(filters) {
    return ['any'].concat(filters.map(convertFilter));
}
function convertInOp(property, values) {
    if (values.length === 0) {
        return false;
    }
    switch (property) {
    case '$type':
        return [
            `filter-type-in`,
            [
                'literal',
                values
            ]
        ];
    case '$id':
        return [
            `filter-id-in`,
            [
                'literal',
                values
            ]
        ];
    default:
        if (values.length > 200 && !values.some(v => typeof v !== typeof values[0])) {
            return [
                'filter-in-large',
                property,
                [
                    'literal',
                    values.sort(compare)
                ]
            ];
        } else {
            return [
                'filter-in-small',
                property,
                [
                    'literal',
                    values
                ]
            ];
        }
    }
}
function convertHasOp(property) {
    switch (property) {
    case '$type':
        return true;
    case '$id':
        return [`filter-has-id`];
    default:
        return [
            `filter-has`,
            property
        ];
    }
}
function convertNegation(filter) {
    return [
        '!',
        filter
    ];
}

const FQIDSeparator = '\x1F';
function isFQID(id) {
    return id.indexOf(FQIDSeparator) >= 0;
}
function makeFQID(id, scope) {
    if (!scope)
        return id;
    return `${ id }${ FQIDSeparator }${ scope }`;
}
function getNameFromFQID(fqid) {
    const sep = fqid.indexOf(FQIDSeparator);
    return sep >= 0 ? fqid.slice(0, sep) : fqid;
}
function getScopeFromFQID(fqid) {
    const sep = fqid.indexOf(FQIDSeparator);
    return sep >= 0 ? fqid.slice(sep + 1) : '';
}

const TRANSITION_SUFFIX = '-transition';
const drapedLayers = /* @__PURE__ */
new Set([
    'fill',
    'line',
    'background',
    'hillshade',
    'raster'
]);
class StyleLayer extends Evented {
    constructor(layer, properties, scope, lut, options) {
        super();
        this.id = layer.id;
        this.fqid = makeFQID(this.id, scope);
        this.type = layer.type;
        this.scope = scope;
        this.lut = lut;
        this.options = options;
        this._featureFilter = {
            filter: () => true,
            needGeometry: false,
            needFeature: false
        };
        this._filterCompiled = false;
        this.configDependencies = /* @__PURE__ */
        new Set();
        if (layer.type === 'custom')
            return;
        layer = layer;
        this.metadata = layer.metadata;
        this.minzoom = layer.minzoom;
        this.maxzoom = layer.maxzoom;
        if (layer.type && layer.type !== 'background' && layer.type !== 'sky' && layer.type !== 'slot') {
            this.source = layer.source;
            this.sourceLayer = layer['source-layer'];
            this.filter = layer.filter;
            const filterSpec = spec[`filter_${ layer.type }`];
            const compiledStaticFilter = createExpression(this.filter, filterSpec);
            if (compiledStaticFilter.result !== 'error') {
                this.configDependencies = /* @__PURE__ */
                new Set([
                    ...this.configDependencies,
                    ...compiledStaticFilter.value.configDependencies
                ]);
            }
        }
        if (layer.slot)
            this.slot = layer.slot;
        if (properties.layout) {
            this._unevaluatedLayout = new Layout(properties.layout, this.scope, options);
            this.configDependencies = /* @__PURE__ */
            new Set([
                ...this.configDependencies,
                ...this._unevaluatedLayout.configDependencies
            ]);
        }
        if (properties.paint) {
            this._transitionablePaint = new Transitionable(properties.paint, this.scope, options);
            for (const property in layer.paint) {
                this.setPaintProperty(property, layer.paint[property]);
            }
            for (const property in layer.layout) {
                this.setLayoutProperty(property, layer.layout[property]);
            }
            this.configDependencies = /* @__PURE__ */
            new Set([
                ...this.configDependencies,
                ...this._transitionablePaint.configDependencies
            ]);
            this._transitioningPaint = this._transitionablePaint.untransitioned();
            this.paint = new PossiblyEvaluated(properties.paint);
        }
    }
    // No-op in the StyleLayer class, must be implemented by each concrete StyleLayer
    onAdd(_map) {
    }
    // No-op in the StyleLayer class, must be implemented by each concrete StyleLayer
    onRemove(_map) {
    }
    isDraped(_sourceCache) {
        return !this.is3D() && drapedLayers.has(this.type);
    }
    getLayoutProperty(name) {
        if (name === 'visibility') {
            return this.visibility;
        }
        return this._unevaluatedLayout.getValue(name);
    }
    setLayoutProperty(name, value) {
        if (this.type === 'custom' && name === 'visibility') {
            this.visibility = value;
            return;
        }
        const layout = this._unevaluatedLayout;
        const specProps = layout._properties.properties;
        if (!specProps[name])
            return;
        layout.setValue(name, value);
        this.configDependencies = /* @__PURE__ */
        new Set([
            ...this.configDependencies,
            ...layout.configDependencies
        ]);
        if (name === 'visibility') {
            this.possiblyEvaluateVisibility();
        }
    }
    possiblyEvaluateVisibility() {
        if (!this._unevaluatedLayout._values.visibility) {
            return;
        }
        this.visibility = this._unevaluatedLayout._values.visibility.possiblyEvaluate({ zoom: 0 });
    }
    getPaintProperty(name) {
        if (endsWith(name, TRANSITION_SUFFIX)) {
            return this._transitionablePaint.getTransition(name.slice(0, -TRANSITION_SUFFIX.length));
        } else {
            return this._transitionablePaint.getValue(name);
        }
    }
    setPaintProperty(name, value) {
        const paint = this._transitionablePaint;
        const specProps = paint._properties.properties;
        if (endsWith(name, TRANSITION_SUFFIX)) {
            const propName = name.slice(0, -TRANSITION_SUFFIX.length);
            if (specProps[propName]) {
                paint.setTransition(propName, value || void 0);
            }
            return false;
        }
        if (!specProps[name])
            return false;
        const transitionable = paint._values[name];
        const wasDataDriven = transitionable.value.isDataDriven();
        const oldValue = transitionable.value;
        paint.setValue(name, value);
        this.configDependencies = /* @__PURE__ */
        new Set([
            ...this.configDependencies,
            ...paint.configDependencies
        ]);
        this._handleSpecialPaintPropertyUpdate(name);
        const newValue = paint._values[name].value;
        const isDataDriven = newValue.isDataDriven();
        const isPattern = endsWith(name, 'pattern') || name === 'line-dasharray';
        return isDataDriven || wasDataDriven || isPattern || this._handleOverridablePaintPropertyUpdate(name, oldValue, newValue);
    }
    _handleSpecialPaintPropertyUpdate(_) {
    }
    getProgramIds() {
        return null;
    }
    // eslint-disable-next-line no-unused-vars
    getDefaultProgramParams(name, zoom, lut) {
        return null;
    }
    // eslint-disable-next-line no-unused-vars
    _handleOverridablePaintPropertyUpdate(name, oldValue, newValue) {
        return false;
    }
    isHidden(zoom) {
        if (this.minzoom && zoom < this.minzoom)
            return true;
        if (this.maxzoom && zoom >= this.maxzoom)
            return true;
        return this.visibility === 'none';
    }
    updateTransitions(parameters) {
        this._transitioningPaint = this._transitionablePaint.transitioned(parameters, this._transitioningPaint);
    }
    hasTransition() {
        return this._transitioningPaint.hasTransition();
    }
    recalculate(parameters, availableImages) {
        if (this._unevaluatedLayout) {
            this.layout = this._unevaluatedLayout.possiblyEvaluate(parameters, void 0, availableImages);
        }
        this.paint = this._transitioningPaint.possiblyEvaluate(parameters, void 0, availableImages);
    }
    serialize() {
        const output = {
            'id': this.id,
            'type': this.type,
            'slot': this.slot,
            'source': this.source,
            'source-layer': this.sourceLayer,
            'metadata': this.metadata,
            'minzoom': this.minzoom,
            'maxzoom': this.maxzoom,
            'filter': this.filter,
            'layout': this._unevaluatedLayout && this._unevaluatedLayout.serialize(),
            'paint': this._transitionablePaint && this._transitionablePaint.serialize()
        };
        return filterObject(output, (value, key) => {
            return value !== void 0 && !(key === 'layout' && !Object.keys(value).length) && !(key === 'paint' && !Object.keys(value).length);
        });
    }
    is3D() {
        return false;
    }
    isSky() {
        return false;
    }
    isTileClipped() {
        return false;
    }
    hasOffscreenPass() {
        return false;
    }
    hasShadowPass() {
        return false;
    }
    canCastShadows() {
        return false;
    }
    hasLightBeamPass() {
        return false;
    }
    cutoffRange() {
        return 0;
    }
    tileCoverLift() {
        return 0;
    }
    resize() {
    }
    isStateDependent() {
        for (const property in this.paint._values) {
            const value = this.paint.get(property);
            if (!(value instanceof PossiblyEvaluatedPropertyValue) || !supportsPropertyExpression(value.property.specification)) {
                continue;
            }
            if ((value.value.kind === 'source' || value.value.kind === 'composite') && value.value.isStateDependent) {
                return true;
            }
        }
        return false;
    }
    compileFilter(options) {
        if (!this._filterCompiled) {
            this._featureFilter = createFilter(this.filter, this.scope, options);
            this._filterCompiled = true;
        }
    }
    invalidateCompiledFilter() {
        this._filterCompiled = false;
    }
    dynamicFilter() {
        return this._featureFilter.dynamicFilter;
    }
    dynamicFilterNeedsFeature() {
        return this._featureFilter.needFeature;
    }
    getLayerRenderingStats() {
        return this._stats;
    }
    resetLayerRenderingStats(painter) {
        if (this._stats) {
            if (painter.renderPass === 'shadow') {
                this._stats.numRenderedVerticesInShadowPass = 0;
            } else {
                this._stats.numRenderedVerticesInTransparentPass = 0;
            }
        }
    }
    // @ts-expect-error - TS2355 - A function whose declared type is neither 'undefined', 'void', nor 'any' must return a value.
    queryRadius(_bucket) {
    }
    queryIntersectsFeature(_queryGeometry, _feature, _featureState, _geometry, _zoom, _transform, _pixelPosMatrix, _elevationHelper, _layoutVertexArrayOffset) {
    }
}

const viewTypes = {
    'Int8': Int8Array,
    'Uint8': Uint8Array,
    'Int16': Int16Array,
    'Uint16': Uint16Array,
    'Int32': Int32Array,
    'Uint32': Uint32Array,
    'Float32': Float32Array
};
class Struct {
    /**
   * @param {StructArray} structArray The StructArray the struct is stored in
   * @param {number} index The index of the struct in the StructArray.
   * @private
   */
    constructor(structArray, index) {
        this._structArray = structArray;
        this._pos1 = index * this.size;
        this._pos2 = this._pos1 / 2;
        this._pos4 = this._pos1 / 4;
        this._pos8 = this._pos1 / 8;
    }
}
const DEFAULT_CAPACITY = 128;
const RESIZE_MULTIPLIER = 5;
class StructArray {
    constructor() {
        this.isTransferred = false;
        this.capacity = -1;
        this.resize(0);
    }
    /**
   * Serialize a StructArray instance.  Serializes both the raw data and the
   * metadata needed to reconstruct the StructArray base class during
   * deserialization.
   * @private
   */
    static serialize(array, transferables) {
        array._trim();
        if (transferables) {
            array.isTransferred = true;
            transferables.add(array.arrayBuffer);
        }
        return {
            length: array.length,
            arrayBuffer: array.arrayBuffer
        };
    }
    static deserialize(input) {
        const structArray = Object.create(this.prototype);
        structArray.arrayBuffer = input.arrayBuffer;
        structArray.length = input.length;
        structArray.capacity = input.arrayBuffer.byteLength / structArray.bytesPerElement;
        structArray._refreshViews();
        return structArray;
    }
    /**
   * Resize the array to discard unused capacity.
   */
    _trim() {
        if (this.length !== this.capacity) {
            this.capacity = this.length;
            this.arrayBuffer = this.arrayBuffer.slice(0, this.length * this.bytesPerElement);
            this._refreshViews();
        }
    }
    /**
   * Resets the the length of the array to 0 without de-allocating capacity.
   */
    clear() {
        this.length = 0;
    }
    /**
   * Resize the array.
   * If `n` is greater than the current length then additional elements with undefined values are added.
   * If `n` is less than the current length then the array will be reduced to the first `n` elements.
   * @param {number} n The new size of the array.
   */
    resize(n) {
        this.reserve(n);
        this.length = n;
    }
    /**
   * Indicate a planned increase in size, so that any necessary allocation may
   * be done once, ahead of time.
   * @param {number} n The expected size of the array.
   */
    reserve(n) {
        if (n > this.capacity) {
            this.capacity = Math.max(n, Math.floor(this.capacity * RESIZE_MULTIPLIER), DEFAULT_CAPACITY);
            this.arrayBuffer = new ArrayBuffer(this.capacity * this.bytesPerElement);
            const oldUint8Array = this.uint8;
            this._refreshViews();
            if (oldUint8Array)
                this.uint8.set(oldUint8Array);
        }
    }
    /**
   * Create TypedArray views for the current ArrayBuffer.
   */
    _refreshViews() {
        throw new Error('StructArray#_refreshViews() must be implemented by each concrete StructArray layout');
    }
    emplace(..._) {
        throw new Error('StructArray#emplace() must be implemented by each concrete StructArray layout');
    }
    emplaceBack(..._) {
        throw new Error('StructArray#emplaceBack() must be implemented by each concrete StructArray layout');
    }
    destroy() {
        this.int8 = this.uint8 = this.int16 = this.uint16 = this.int32 = this.uint32 = this.float32 = null;
        this.arrayBuffer = null;
    }
}
function createLayout(members, alignment = 1) {
    let offset = 0;
    let maxSize = 0;
    const layoutMembers = members.map(member => {
        const typeSize = sizeOf(member.type);
        const memberOffset = offset = align$1(offset, Math.max(alignment, typeSize));
        const components = member.components || 1;
        maxSize = Math.max(maxSize, typeSize);
        offset += typeSize * components;
        return {
            name: member.name,
            type: member.type,
            components,
            offset: memberOffset
        };
    });
    const size = align$1(offset, Math.max(maxSize, alignment));
    return {
        members: layoutMembers,
        size,
        alignment
    };
}
function sizeOf(type) {
    return viewTypes[type].BYTES_PER_ELEMENT;
}
function align$1(offset, size) {
    return Math.ceil(offset / size) * size;
}

class StructArrayLayout2i4 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1);
    }
    emplace(i, v0, v1) {
        const o2 = i * 2;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        return i;
    }
}
StructArrayLayout2i4.prototype.bytesPerElement = 4;
register(StructArrayLayout2i4, 'StructArrayLayout2i4');
class StructArrayLayout3i6 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2);
    }
    emplace(i, v0, v1, v2) {
        const o2 = i * 3;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        return i;
    }
}
StructArrayLayout3i6.prototype.bytesPerElement = 6;
register(StructArrayLayout3i6, 'StructArrayLayout3i6');
class StructArrayLayout4i8 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3);
    }
    emplace(i, v0, v1, v2, v3) {
        const o2 = i * 4;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 3] = v3;
        return i;
    }
}
StructArrayLayout4i8.prototype.bytesPerElement = 8;
register(StructArrayLayout4i8, 'StructArrayLayout4i8');
class StructArrayLayout5i10 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4);
    }
    emplace(i, v0, v1, v2, v3, v4) {
        const o2 = i * 5;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 3] = v3;
        this.int16[o2 + 4] = v4;
        return i;
    }
}
StructArrayLayout5i10.prototype.bytesPerElement = 10;
register(StructArrayLayout5i10, 'StructArrayLayout5i10');
class StructArrayLayout2i4ub1f12 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6) {
        const o2 = i * 6;
        const o1 = i * 12;
        const o4 = i * 3;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.uint8[o1 + 4] = v2;
        this.uint8[o1 + 5] = v3;
        this.uint8[o1 + 6] = v4;
        this.uint8[o1 + 7] = v5;
        this.float32[o4 + 2] = v6;
        return i;
    }
}
StructArrayLayout2i4ub1f12.prototype.bytesPerElement = 12;
register(StructArrayLayout2i4ub1f12, 'StructArrayLayout2i4ub1f12');
class StructArrayLayout3f12 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2);
    }
    emplace(i, v0, v1, v2) {
        const o4 = i * 3;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        return i;
    }
}
StructArrayLayout3f12.prototype.bytesPerElement = 12;
register(StructArrayLayout3f12, 'StructArrayLayout3f12');
class StructArrayLayout4ui1f12 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4);
    }
    emplace(i, v0, v1, v2, v3, v4) {
        const o2 = i * 6;
        const o4 = i * 3;
        this.uint16[o2 + 0] = v0;
        this.uint16[o2 + 1] = v1;
        this.uint16[o2 + 2] = v2;
        this.uint16[o2 + 3] = v3;
        this.float32[o4 + 2] = v4;
        return i;
    }
}
StructArrayLayout4ui1f12.prototype.bytesPerElement = 12;
register(StructArrayLayout4ui1f12, 'StructArrayLayout4ui1f12');
class StructArrayLayout4ui8 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3);
    }
    emplace(i, v0, v1, v2, v3) {
        const o2 = i * 4;
        this.uint16[o2 + 0] = v0;
        this.uint16[o2 + 1] = v1;
        this.uint16[o2 + 2] = v2;
        this.uint16[o2 + 3] = v3;
        return i;
    }
}
StructArrayLayout4ui8.prototype.bytesPerElement = 8;
register(StructArrayLayout4ui8, 'StructArrayLayout4ui8');
class StructArrayLayout6i12 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5);
    }
    emplace(i, v0, v1, v2, v3, v4, v5) {
        const o2 = i * 6;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 3] = v3;
        this.int16[o2 + 4] = v4;
        this.int16[o2 + 5] = v5;
        return i;
    }
}
StructArrayLayout6i12.prototype.bytesPerElement = 12;
register(StructArrayLayout6i12, 'StructArrayLayout6i12');
class StructArrayLayout4i4ui4i24 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11) {
        const o2 = i * 12;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 3] = v3;
        this.uint16[o2 + 4] = v4;
        this.uint16[o2 + 5] = v5;
        this.uint16[o2 + 6] = v6;
        this.uint16[o2 + 7] = v7;
        this.int16[o2 + 8] = v8;
        this.int16[o2 + 9] = v9;
        this.int16[o2 + 10] = v10;
        this.int16[o2 + 11] = v11;
        return i;
    }
}
StructArrayLayout4i4ui4i24.prototype.bytesPerElement = 24;
register(StructArrayLayout4i4ui4i24, 'StructArrayLayout4i4ui4i24');
class StructArrayLayout3i3f20 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5);
    }
    emplace(i, v0, v1, v2, v3, v4, v5) {
        const o2 = i * 10;
        const o4 = i * 5;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.float32[o4 + 2] = v3;
        this.float32[o4 + 3] = v4;
        this.float32[o4 + 4] = v5;
        return i;
    }
}
StructArrayLayout3i3f20.prototype.bytesPerElement = 20;
register(StructArrayLayout3i3f20, 'StructArrayLayout3i3f20');
class StructArrayLayout4f16 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3);
    }
    emplace(i, v0, v1, v2, v3) {
        const o4 = i * 4;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        return i;
    }
}
StructArrayLayout4f16.prototype.bytesPerElement = 16;
register(StructArrayLayout4f16, 'StructArrayLayout4f16');
class StructArrayLayout1ul4 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint32 = new Uint32Array(this.arrayBuffer);
    }
    emplaceBack(v0) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0);
    }
    emplace(i, v0) {
        const o4 = i * 1;
        this.uint32[o4 + 0] = v0;
        return i;
    }
}
StructArrayLayout1ul4.prototype.bytesPerElement = 4;
register(StructArrayLayout1ul4, 'StructArrayLayout1ul4');
class StructArrayLayout2ui4 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1);
    }
    emplace(i, v0, v1) {
        const o2 = i * 2;
        this.uint16[o2 + 0] = v0;
        this.uint16[o2 + 1] = v1;
        return i;
    }
}
StructArrayLayout2ui4.prototype.bytesPerElement = 4;
register(StructArrayLayout2ui4, 'StructArrayLayout2ui4');
class StructArrayLayout5i4f1i1ul2ui40 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
        this.uint32 = new Uint32Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12) {
        const o2 = i * 20;
        const o4 = i * 10;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 3] = v3;
        this.int16[o2 + 4] = v4;
        this.float32[o4 + 3] = v5;
        this.float32[o4 + 4] = v6;
        this.float32[o4 + 5] = v7;
        this.float32[o4 + 6] = v8;
        this.int16[o2 + 14] = v9;
        this.uint32[o4 + 8] = v10;
        this.uint16[o2 + 18] = v11;
        this.uint16[o2 + 19] = v12;
        return i;
    }
}
StructArrayLayout5i4f1i1ul2ui40.prototype.bytesPerElement = 40;
register(StructArrayLayout5i4f1i1ul2ui40, 'StructArrayLayout5i4f1i1ul2ui40');
class StructArrayLayout3i2i2i16 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6) {
        const o2 = i * 8;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.int16[o2 + 4] = v3;
        this.int16[o2 + 5] = v4;
        this.int16[o2 + 6] = v5;
        this.int16[o2 + 7] = v6;
        return i;
    }
}
StructArrayLayout3i2i2i16.prototype.bytesPerElement = 16;
register(StructArrayLayout3i2i2i16, 'StructArrayLayout3i2i2i16');
class StructArrayLayout2f1f2i16 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4);
    }
    emplace(i, v0, v1, v2, v3, v4) {
        const o4 = i * 4;
        const o2 = i * 8;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.int16[o2 + 6] = v3;
        this.int16[o2 + 7] = v4;
        return i;
    }
}
StructArrayLayout2f1f2i16.prototype.bytesPerElement = 16;
register(StructArrayLayout2f1f2i16, 'StructArrayLayout2f1f2i16');
class StructArrayLayout2ub4f20 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5);
    }
    emplace(i, v0, v1, v2, v3, v4, v5) {
        const o1 = i * 20;
        const o4 = i * 5;
        this.uint8[o1 + 0] = v0;
        this.uint8[o1 + 1] = v1;
        this.float32[o4 + 1] = v2;
        this.float32[o4 + 2] = v3;
        this.float32[o4 + 3] = v4;
        this.float32[o4 + 4] = v5;
        return i;
    }
}
StructArrayLayout2ub4f20.prototype.bytesPerElement = 20;
register(StructArrayLayout2ub4f20, 'StructArrayLayout2ub4f20');
class StructArrayLayout3ui6 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2);
    }
    emplace(i, v0, v1, v2) {
        const o2 = i * 3;
        this.uint16[o2 + 0] = v0;
        this.uint16[o2 + 1] = v1;
        this.uint16[o2 + 2] = v2;
        return i;
    }
}
StructArrayLayout3ui6.prototype.bytesPerElement = 6;
register(StructArrayLayout3ui6, 'StructArrayLayout3ui6');
class StructArrayLayout3i2f2ui3ul3ui2f3ub1ul1i1ub60 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
        this.uint32 = new Uint32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20) {
        const o2 = i * 30;
        const o4 = i * 15;
        const o1 = i * 60;
        this.int16[o2 + 0] = v0;
        this.int16[o2 + 1] = v1;
        this.int16[o2 + 2] = v2;
        this.float32[o4 + 2] = v3;
        this.float32[o4 + 3] = v4;
        this.uint16[o2 + 8] = v5;
        this.uint16[o2 + 9] = v6;
        this.uint32[o4 + 5] = v7;
        this.uint32[o4 + 6] = v8;
        this.uint32[o4 + 7] = v9;
        this.uint16[o2 + 16] = v10;
        this.uint16[o2 + 17] = v11;
        this.uint16[o2 + 18] = v12;
        this.float32[o4 + 10] = v13;
        this.float32[o4 + 11] = v14;
        this.uint8[o1 + 48] = v15;
        this.uint8[o1 + 49] = v16;
        this.uint8[o1 + 50] = v17;
        this.uint32[o4 + 13] = v18;
        this.int16[o2 + 28] = v19;
        this.uint8[o1 + 58] = v20;
        return i;
    }
}
StructArrayLayout3i2f2ui3ul3ui2f3ub1ul1i1ub60.prototype.bytesPerElement = 60;
register(StructArrayLayout3i2f2ui3ul3ui2f3ub1ul1i1ub60, 'StructArrayLayout3i2f2ui3ul3ui2f3ub1ul1i1ub60');
class StructArrayLayout2f9i15ui1ul4f1ub80 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
        this.uint32 = new Uint32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20, v21, v22, v23, v24, v25, v26, v27, v28, v29, v30, v31) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20, v21, v22, v23, v24, v25, v26, v27, v28, v29, v30, v31);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15, v16, v17, v18, v19, v20, v21, v22, v23, v24, v25, v26, v27, v28, v29, v30, v31) {
        const o4 = i * 20;
        const o2 = i * 40;
        const o1 = i * 80;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.int16[o2 + 4] = v2;
        this.int16[o2 + 5] = v3;
        this.int16[o2 + 6] = v4;
        this.int16[o2 + 7] = v5;
        this.int16[o2 + 8] = v6;
        this.int16[o2 + 9] = v7;
        this.int16[o2 + 10] = v8;
        this.int16[o2 + 11] = v9;
        this.int16[o2 + 12] = v10;
        this.uint16[o2 + 13] = v11;
        this.uint16[o2 + 14] = v12;
        this.uint16[o2 + 15] = v13;
        this.uint16[o2 + 16] = v14;
        this.uint16[o2 + 17] = v15;
        this.uint16[o2 + 18] = v16;
        this.uint16[o2 + 19] = v17;
        this.uint16[o2 + 20] = v18;
        this.uint16[o2 + 21] = v19;
        this.uint16[o2 + 22] = v20;
        this.uint16[o2 + 23] = v21;
        this.uint16[o2 + 24] = v22;
        this.uint16[o2 + 25] = v23;
        this.uint16[o2 + 26] = v24;
        this.uint16[o2 + 27] = v25;
        this.uint32[o4 + 14] = v26;
        this.float32[o4 + 15] = v27;
        this.float32[o4 + 16] = v28;
        this.float32[o4 + 17] = v29;
        this.float32[o4 + 18] = v30;
        this.uint8[o1 + 76] = v31;
        return i;
    }
}
StructArrayLayout2f9i15ui1ul4f1ub80.prototype.bytesPerElement = 80;
register(StructArrayLayout2f9i15ui1ul4f1ub80, 'StructArrayLayout2f9i15ui1ul4f1ub80');
class StructArrayLayout1f4 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0);
    }
    emplace(i, v0) {
        const o4 = i * 1;
        this.float32[o4 + 0] = v0;
        return i;
    }
}
StructArrayLayout1f4.prototype.bytesPerElement = 4;
register(StructArrayLayout1f4, 'StructArrayLayout1f4');
class StructArrayLayout5f20 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4);
    }
    emplace(i, v0, v1, v2, v3, v4) {
        const o4 = i * 5;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        this.float32[o4 + 4] = v4;
        return i;
    }
}
StructArrayLayout5f20.prototype.bytesPerElement = 20;
register(StructArrayLayout5f20, 'StructArrayLayout5f20');
class StructArrayLayout7f28 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6) {
        const o4 = i * 7;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        this.float32[o4 + 4] = v4;
        this.float32[o4 + 5] = v5;
        this.float32[o4 + 6] = v6;
        return i;
    }
}
StructArrayLayout7f28.prototype.bytesPerElement = 28;
register(StructArrayLayout7f28, 'StructArrayLayout7f28');
class StructArrayLayout11f44 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10) {
        const o4 = i * 11;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        this.float32[o4 + 4] = v4;
        this.float32[o4 + 5] = v5;
        this.float32[o4 + 6] = v6;
        this.float32[o4 + 7] = v7;
        this.float32[o4 + 8] = v8;
        this.float32[o4 + 9] = v9;
        this.float32[o4 + 10] = v10;
        return i;
    }
}
StructArrayLayout11f44.prototype.bytesPerElement = 44;
register(StructArrayLayout11f44, 'StructArrayLayout11f44');
class StructArrayLayout9f36 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8) {
        const o4 = i * 9;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        this.float32[o4 + 4] = v4;
        this.float32[o4 + 5] = v5;
        this.float32[o4 + 6] = v6;
        this.float32[o4 + 7] = v7;
        this.float32[o4 + 8] = v8;
        return i;
    }
}
StructArrayLayout9f36.prototype.bytesPerElement = 36;
register(StructArrayLayout9f36, 'StructArrayLayout9f36');
class StructArrayLayout2f8 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1);
    }
    emplace(i, v0, v1) {
        const o4 = i * 2;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        return i;
    }
}
StructArrayLayout2f8.prototype.bytesPerElement = 8;
register(StructArrayLayout2f8, 'StructArrayLayout2f8');
class StructArrayLayout1ul3ui12 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint32 = new Uint32Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3);
    }
    emplace(i, v0, v1, v2, v3) {
        const o4 = i * 3;
        const o2 = i * 6;
        this.uint32[o4 + 0] = v0;
        this.uint16[o2 + 2] = v1;
        this.uint16[o2 + 3] = v2;
        this.uint16[o2 + 4] = v3;
        return i;
    }
}
StructArrayLayout1ul3ui12.prototype.bytesPerElement = 12;
register(StructArrayLayout1ul3ui12, 'StructArrayLayout1ul3ui12');
class StructArrayLayout1ui2 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
    }
    emplaceBack(v0) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0);
    }
    emplace(i, v0) {
        const o2 = i * 1;
        this.uint16[o2 + 0] = v0;
        return i;
    }
}
StructArrayLayout1ui2.prototype.bytesPerElement = 2;
register(StructArrayLayout1ui2, 'StructArrayLayout1ui2');
class StructArrayLayout16f64 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6, v7, v8, v9, v10, v11, v12, v13, v14, v15) {
        const o4 = i * 16;
        this.float32[o4 + 0] = v0;
        this.float32[o4 + 1] = v1;
        this.float32[o4 + 2] = v2;
        this.float32[o4 + 3] = v3;
        this.float32[o4 + 4] = v4;
        this.float32[o4 + 5] = v5;
        this.float32[o4 + 6] = v6;
        this.float32[o4 + 7] = v7;
        this.float32[o4 + 8] = v8;
        this.float32[o4 + 9] = v9;
        this.float32[o4 + 10] = v10;
        this.float32[o4 + 11] = v11;
        this.float32[o4 + 12] = v12;
        this.float32[o4 + 13] = v13;
        this.float32[o4 + 14] = v14;
        this.float32[o4 + 15] = v15;
        return i;
    }
}
StructArrayLayout16f64.prototype.bytesPerElement = 64;
register(StructArrayLayout16f64, 'StructArrayLayout16f64');
class StructArrayLayout4ui3f20 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.uint16 = new Uint16Array(this.arrayBuffer);
        this.float32 = new Float32Array(this.arrayBuffer);
    }
    emplaceBack(v0, v1, v2, v3, v4, v5, v6) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0, v1, v2, v3, v4, v5, v6);
    }
    emplace(i, v0, v1, v2, v3, v4, v5, v6) {
        const o2 = i * 10;
        const o4 = i * 5;
        this.uint16[o2 + 0] = v0;
        this.uint16[o2 + 1] = v1;
        this.uint16[o2 + 2] = v2;
        this.uint16[o2 + 3] = v3;
        this.float32[o4 + 2] = v4;
        this.float32[o4 + 3] = v5;
        this.float32[o4 + 4] = v6;
        return i;
    }
}
StructArrayLayout4ui3f20.prototype.bytesPerElement = 20;
register(StructArrayLayout4ui3f20, 'StructArrayLayout4ui3f20');
class StructArrayLayout1i2 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
        this.int16 = new Int16Array(this.arrayBuffer);
    }
    emplaceBack(v0) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0);
    }
    emplace(i, v0) {
        const o2 = i * 1;
        this.int16[o2 + 0] = v0;
        return i;
    }
}
StructArrayLayout1i2.prototype.bytesPerElement = 2;
register(StructArrayLayout1i2, 'StructArrayLayout1i2');
class StructArrayLayout1ub1 extends StructArray {
    _refreshViews() {
        this.uint8 = new Uint8Array(this.arrayBuffer);
    }
    emplaceBack(v0) {
        const i = this.length;
        this.resize(i + 1);
        return this.emplace(i, v0);
    }
    emplace(i, v0) {
        const o1 = i * 1;
        this.uint8[o1 + 0] = v0;
        return i;
    }
}
StructArrayLayout1ub1.prototype.bytesPerElement = 1;
register(StructArrayLayout1ub1, 'StructArrayLayout1ub1');
class CollisionBoxStruct extends Struct {
    get projectedAnchorX() {
        return this._structArray.int16[this._pos2 + 0];
    }
    get projectedAnchorY() {
        return this._structArray.int16[this._pos2 + 1];
    }
    get projectedAnchorZ() {
        return this._structArray.int16[this._pos2 + 2];
    }
    get tileAnchorX() {
        return this._structArray.int16[this._pos2 + 3];
    }
    get tileAnchorY() {
        return this._structArray.int16[this._pos2 + 4];
    }
    get x1() {
        return this._structArray.float32[this._pos4 + 3];
    }
    get y1() {
        return this._structArray.float32[this._pos4 + 4];
    }
    get x2() {
        return this._structArray.float32[this._pos4 + 5];
    }
    get y2() {
        return this._structArray.float32[this._pos4 + 6];
    }
    get padding() {
        return this._structArray.int16[this._pos2 + 14];
    }
    get featureIndex() {
        return this._structArray.uint32[this._pos4 + 8];
    }
    get sourceLayerIndex() {
        return this._structArray.uint16[this._pos2 + 18];
    }
    get bucketIndex() {
        return this._structArray.uint16[this._pos2 + 19];
    }
}
CollisionBoxStruct.prototype.size = 40;
class CollisionBoxArray extends StructArrayLayout5i4f1i1ul2ui40 {
    /**
   * Return the CollisionBoxStruct at the given location in the array.
   * @param {number} index The index of the element.
   * @private
   */
    get(index) {
        return new CollisionBoxStruct(this, index);
    }
}
register(CollisionBoxArray, 'CollisionBoxArray');
class PlacedSymbolStruct extends Struct {
    get projectedAnchorX() {
        return this._structArray.int16[this._pos2 + 0];
    }
    get projectedAnchorY() {
        return this._structArray.int16[this._pos2 + 1];
    }
    get projectedAnchorZ() {
        return this._structArray.int16[this._pos2 + 2];
    }
    get tileAnchorX() {
        return this._structArray.float32[this._pos4 + 2];
    }
    get tileAnchorY() {
        return this._structArray.float32[this._pos4 + 3];
    }
    get glyphStartIndex() {
        return this._structArray.uint16[this._pos2 + 8];
    }
    get numGlyphs() {
        return this._structArray.uint16[this._pos2 + 9];
    }
    get vertexStartIndex() {
        return this._structArray.uint32[this._pos4 + 5];
    }
    get lineStartIndex() {
        return this._structArray.uint32[this._pos4 + 6];
    }
    get lineLength() {
        return this._structArray.uint32[this._pos4 + 7];
    }
    get segment() {
        return this._structArray.uint16[this._pos2 + 16];
    }
    get lowerSize() {
        return this._structArray.uint16[this._pos2 + 17];
    }
    get upperSize() {
        return this._structArray.uint16[this._pos2 + 18];
    }
    get lineOffsetX() {
        return this._structArray.float32[this._pos4 + 10];
    }
    get lineOffsetY() {
        return this._structArray.float32[this._pos4 + 11];
    }
    get writingMode() {
        return this._structArray.uint8[this._pos1 + 48];
    }
    get placedOrientation() {
        return this._structArray.uint8[this._pos1 + 49];
    }
    set placedOrientation(x) {
        this._structArray.uint8[this._pos1 + 49] = x;
    }
    get hidden() {
        return this._structArray.uint8[this._pos1 + 50];
    }
    set hidden(x) {
        this._structArray.uint8[this._pos1 + 50] = x;
    }
    get crossTileID() {
        return this._structArray.uint32[this._pos4 + 13];
    }
    set crossTileID(x) {
        this._structArray.uint32[this._pos4 + 13] = x;
    }
    get associatedIconIndex() {
        return this._structArray.int16[this._pos2 + 28];
    }
    get flipState() {
        return this._structArray.uint8[this._pos1 + 58];
    }
    set flipState(x) {
        this._structArray.uint8[this._pos1 + 58] = x;
    }
}
PlacedSymbolStruct.prototype.size = 60;
class PlacedSymbolArray extends StructArrayLayout3i2f2ui3ul3ui2f3ub1ul1i1ub60 {
    /**
   * Return the PlacedSymbolStruct at the given location in the array.
   * @param {number} index The index of the element.
   * @private
   */
    get(index) {
        return new PlacedSymbolStruct(this, index);
    }
}
register(PlacedSymbolArray, 'PlacedSymbolArray');
class SymbolInstanceStruct extends Struct {
    get tileAnchorX() {
        return this._structArray.float32[this._pos4 + 0];
    }
    get tileAnchorY() {
        return this._structArray.float32[this._pos4 + 1];
    }
    get projectedAnchorX() {
        return this._structArray.int16[this._pos2 + 4];
    }
    get projectedAnchorY() {
        return this._structArray.int16[this._pos2 + 5];
    }
    get projectedAnchorZ() {
        return this._structArray.int16[this._pos2 + 6];
    }
    get rightJustifiedTextSymbolIndex() {
        return this._structArray.int16[this._pos2 + 7];
    }
    get centerJustifiedTextSymbolIndex() {
        return this._structArray.int16[this._pos2 + 8];
    }
    get leftJustifiedTextSymbolIndex() {
        return this._structArray.int16[this._pos2 + 9];
    }
    get verticalPlacedTextSymbolIndex() {
        return this._structArray.int16[this._pos2 + 10];
    }
    get placedIconSymbolIndex() {
        return this._structArray.int16[this._pos2 + 11];
    }
    get verticalPlacedIconSymbolIndex() {
        return this._structArray.int16[this._pos2 + 12];
    }
    get key() {
        return this._structArray.uint16[this._pos2 + 13];
    }
    get textBoxStartIndex() {
        return this._structArray.uint16[this._pos2 + 14];
    }
    get textBoxEndIndex() {
        return this._structArray.uint16[this._pos2 + 15];
    }
    get verticalTextBoxStartIndex() {
        return this._structArray.uint16[this._pos2 + 16];
    }
    get verticalTextBoxEndIndex() {
        return this._structArray.uint16[this._pos2 + 17];
    }
    get iconBoxStartIndex() {
        return this._structArray.uint16[this._pos2 + 18];
    }
    get iconBoxEndIndex() {
        return this._structArray.uint16[this._pos2 + 19];
    }
    get verticalIconBoxStartIndex() {
        return this._structArray.uint16[this._pos2 + 20];
    }
    get verticalIconBoxEndIndex() {
        return this._structArray.uint16[this._pos2 + 21];
    }
    get featureIndex() {
        return this._structArray.uint16[this._pos2 + 22];
    }
    get numHorizontalGlyphVertices() {
        return this._structArray.uint16[this._pos2 + 23];
    }
    get numVerticalGlyphVertices() {
        return this._structArray.uint16[this._pos2 + 24];
    }
    get numIconVertices() {
        return this._structArray.uint16[this._pos2 + 25];
    }
    get numVerticalIconVertices() {
        return this._structArray.uint16[this._pos2 + 26];
    }
    get useRuntimeCollisionCircles() {
        return this._structArray.uint16[this._pos2 + 27];
    }
    get crossTileID() {
        return this._structArray.uint32[this._pos4 + 14];
    }
    set crossTileID(x) {
        this._structArray.uint32[this._pos4 + 14] = x;
    }
    get textOffset0() {
        return this._structArray.float32[this._pos4 + 15];
    }
    get textOffset1() {
        return this._structArray.float32[this._pos4 + 16];
    }
    get collisionCircleDiameter() {
        return this._structArray.float32[this._pos4 + 17];
    }
    get zOffset() {
        return this._structArray.float32[this._pos4 + 18];
    }
    set zOffset(x) {
        this._structArray.float32[this._pos4 + 18] = x;
    }
    get hasIconTextFit() {
        return this._structArray.uint8[this._pos1 + 76];
    }
}
SymbolInstanceStruct.prototype.size = 80;
class SymbolInstanceArray extends StructArrayLayout2f9i15ui1ul4f1ub80 {
    /**
   * Return the SymbolInstanceStruct at the given location in the array.
   * @param {number} index The index of the element.
   * @private
   */
    get(index) {
        return new SymbolInstanceStruct(this, index);
    }
}
register(SymbolInstanceArray, 'SymbolInstanceArray');
class GlyphOffsetArray extends StructArrayLayout1f4 {
    getoffsetX(index) {
        return this.float32[index * 1 + 0];
    }
}
register(GlyphOffsetArray, 'GlyphOffsetArray');
class SymbolLineVertexArray extends StructArrayLayout2i4 {
    getx(index) {
        return this.int16[index * 2 + 0];
    }
    gety(index) {
        return this.int16[index * 2 + 1];
    }
}
register(SymbolLineVertexArray, 'SymbolLineVertexArray');
class FeatureIndexStruct extends Struct {
    get featureIndex() {
        return this._structArray.uint32[this._pos4 + 0];
    }
    get sourceLayerIndex() {
        return this._structArray.uint16[this._pos2 + 2];
    }
    get bucketIndex() {
        return this._structArray.uint16[this._pos2 + 3];
    }
    get layoutVertexArrayOffset() {
        return this._structArray.uint16[this._pos2 + 4];
    }
}
FeatureIndexStruct.prototype.size = 12;
class FeatureIndexArray extends StructArrayLayout1ul3ui12 {
    /**
   * Return the FeatureIndexStruct at the given location in the array.
   * @param {number} index The index of the element.
   * @private
   */
    get(index) {
        return new FeatureIndexStruct(this, index);
    }
}
register(FeatureIndexArray, 'FeatureIndexArray');
class FillExtrusionCentroidArray extends StructArrayLayout2ui4 {
    geta_centroid_pos0(index) {
        return this.uint16[index * 2 + 0];
    }
    geta_centroid_pos1(index) {
        return this.uint16[index * 2 + 1];
    }
}
register(FillExtrusionCentroidArray, 'FillExtrusionCentroidArray');
class FillExtrusionWallStruct extends Struct {
    get a_join_normal_inside0() {
        return this._structArray.int16[this._pos2 + 0];
    }
    get a_join_normal_inside1() {
        return this._structArray.int16[this._pos2 + 1];
    }
    get a_join_normal_inside2() {
        return this._structArray.int16[this._pos2 + 2];
    }
}
FillExtrusionWallStruct.prototype.size = 6;
class FillExtrusionWallArray extends StructArrayLayout3i6 {
    /**
   * Return the FillExtrusionWallStruct at the given location in the array.
   * @param {number} index The index of the element.
   * @private
   */
    get(index) {
        return new FillExtrusionWallStruct(this, index);
    }
}
register(FillExtrusionWallArray, 'FillExtrusionWallArray');

const circleAttributes = createLayout([{
        name: 'a_pos',
        components: 2,
        type: 'Int16'
    }], 4);
const circleGlobeAttributesExt = createLayout([
    {
        name: 'a_pos_3',
        components: 3,
        type: 'Int16'
    },
    {
        name: 'a_pos_normal_3',
        components: 3,
        type: 'Int16'
    }
]);

class SegmentVector {
    constructor(segments = []) {
        this.segments = segments;
    }
    _prepareSegment(numVertices, vertexArrayLength, indexArrayLength, sortKey) {
        let segment = this.segments[this.segments.length - 1];
        if (numVertices > SegmentVector.MAX_VERTEX_ARRAY_LENGTH)
            warnOnce(`Max vertices per segment is ${ SegmentVector.MAX_VERTEX_ARRAY_LENGTH }: bucket requested ${ numVertices }`);
        if (!segment || segment.vertexLength + numVertices > SegmentVector.MAX_VERTEX_ARRAY_LENGTH || segment.sortKey !== sortKey) {
            segment = {
                vertexOffset: vertexArrayLength,
                primitiveOffset: indexArrayLength,
                vertexLength: 0,
                primitiveLength: 0
            };
            if (sortKey !== void 0)
                segment.sortKey = sortKey;
            this.segments.push(segment);
        }
        return segment;
    }
    prepareSegment(numVertices, layoutVertexArray, indexArray, sortKey) {
        return this._prepareSegment(numVertices, layoutVertexArray.length, indexArray.length, sortKey);
    }
    get() {
        return this.segments;
    }
    destroy() {
        for (const segment of this.segments) {
            for (const k in segment.vaos) {
                segment.vaos[k].destroy();
            }
        }
    }
    static simpleSegment(vertexOffset, primitiveOffset, vertexLength, primitiveLength) {
        return new SegmentVector([{
                vertexOffset,
                primitiveOffset,
                vertexLength,
                primitiveLength,
                vaos: {},
                sortKey: 0
            }]);
    }
}
SegmentVector.MAX_VERTEX_ARRAY_LENGTH = Math.pow(2, 16) - 1;
register(SegmentVector, 'SegmentVector');

function packUint8ToFloat(a, b) {
    a = clamp(Math.floor(a), 0, 255);
    b = clamp(Math.floor(b), 0, 255);
    return 256 * a + b;
}

const patternAttributes = createLayout([
    // [tl.x, tl.y, br.x, br.y]
    {
        name: 'a_pattern',
        components: 4,
        type: 'Uint16'
    },
    {
        name: 'a_pixel_ratio',
        components: 1,
        type: 'Float32'
    }
]);

const dashAttributes = createLayout([{
        name: 'a_dash',
        components: 4,
        type: 'Uint16'
    }    // [x, y, width, unused]
]);

class FeaturePositionMap {
    constructor() {
        this.ids = [];
        this.uniqueIds = [];
        this.positions = [];
        this.indexed = false;
    }
    add(id, index, start, end) {
        this.ids.push(getNumericId(id));
        this.positions.push(index, start, end);
    }
    eachPosition(id, fn) {
        const intId = getNumericId(id);
        let i = 0;
        let j = this.ids.length - 1;
        while (i < j) {
            const m = i + j >> 1;
            if (this.ids[m] >= intId) {
                j = m;
            } else {
                i = m + 1;
            }
        }
        while (this.ids[i] === intId) {
            const index = this.positions[3 * i];
            const start = this.positions[3 * i + 1];
            const end = this.positions[3 * i + 2];
            fn(index, start, end);
            i++;
        }
    }
    static serialize(map, transferables) {
        const ids = new Float64Array(map.ids);
        const positions = new Uint32Array(map.positions);
        sort$1(ids, positions, 0, ids.length - 1);
        if (transferables) {
            transferables.add(ids.buffer);
            transferables.add(positions.buffer);
        }
        return {
            ids,
            positions
        };
    }
    static deserialize(obj) {
        const map = new FeaturePositionMap();
        map.ids = obj.ids;
        map.positions = obj.positions;
        let prev;
        for (const id of map.ids) {
            if (id !== prev)
                map.uniqueIds.push(id);
            prev = id;
        }
        map.indexed = true;
        return map;
    }
}
function getNumericId(value) {
    const numValue = +value;
    if (!isNaN(numValue) && Number.MIN_SAFE_INTEGER <= numValue && numValue <= Number.MAX_SAFE_INTEGER) {
        return numValue;
    }
    return murmur3(String(value));
}
function sort$1(ids, positions, left, right) {
    while (left < right) {
        const pivot = ids[left + right >> 1];
        let i = left - 1;
        let j = right + 1;
        while (true) {
            do
                i++;
            while (ids[i] < pivot);
            do
                j--;
            while (ids[j] > pivot);
            if (i >= j)
                break;
            swap$1(ids, i, j);
            swap$1(positions, 3 * i, 3 * j);
            swap$1(positions, 3 * i + 1, 3 * j + 1);
            swap$1(positions, 3 * i + 2, 3 * j + 2);
        }
        if (j - left < right - j) {
            sort$1(ids, positions, left, j);
            left = j + 1;
        } else {
            sort$1(ids, positions, j + 1, right);
            right = j;
        }
    }
}
function swap$1(arr, i, j) {
    const tmp = arr[i];
    arr[i] = arr[j];
    arr[j] = tmp;
}
register(FeaturePositionMap, 'FeaturePositionMap');

class Uniform {
    constructor(context) {
        this.gl = context.gl;
        this.initialized = false;
    }
    fetchUniformLocation(program, name) {
        if (!this.location && !this.initialized) {
            this.location = this.gl.getUniformLocation(program, name);
            this.initialized = true;
        }
        return !!this.location;
    }
    set(_program, _name, _v) {
        throw new Error('Uniform#set() must be implemented by each concrete Uniform');
    }
}
class Uniform1i extends Uniform {
    constructor(context) {
        super(context);
        this.current = 0;
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (this.current !== v) {
            this.current = v;
            this.gl.uniform1i(this.location, v);
        }
    }
}
class Uniform1f extends Uniform {
    constructor(context) {
        super(context);
        this.current = 0;
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (this.current !== v) {
            this.current = v;
            this.gl.uniform1f(this.location, v);
        }
    }
}
class Uniform2f extends Uniform {
    constructor(context) {
        super(context);
        this.current = [
            0,
            0
        ];
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (v[0] !== this.current[0] || v[1] !== this.current[1]) {
            this.current = v;
            this.gl.uniform2f(this.location, v[0], v[1]);
        }
    }
}
class Uniform3f extends Uniform {
    constructor(context) {
        super(context);
        this.current = [
            0,
            0,
            0
        ];
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (v[0] !== this.current[0] || v[1] !== this.current[1] || v[2] !== this.current[2]) {
            this.current = v;
            this.gl.uniform3f(this.location, v[0], v[1], v[2]);
        }
    }
}
class Uniform4f extends Uniform {
    constructor(context) {
        super(context);
        this.current = [
            0,
            0,
            0,
            0
        ];
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (v[0] !== this.current[0] || v[1] !== this.current[1] || v[2] !== this.current[2] || v[3] !== this.current[3]) {
            this.current = v;
            this.gl.uniform4f(this.location, v[0], v[1], v[2], v[3]);
        }
    }
}
class UniformColor extends Uniform {
    constructor(context) {
        super(context);
        this.current = Color.transparent.toRenderColor(null);
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (v.r !== this.current.r || v.g !== this.current.g || v.b !== this.current.b || v.a !== this.current.a) {
            this.current = v;
            this.gl.uniform4f(this.location, v.r, v.g, v.b, v.a);
        }
    }
}
const emptyMat4 = new Float32Array(16);
class UniformMatrix4f extends Uniform {
    constructor(context) {
        super(context);
        this.current = emptyMat4;
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        if (v[12] !== this.current[12] || v[0] !== this.current[0]) {
            this.current = v;
            this.gl.uniformMatrix4fv(this.location, false, v);
            return;
        }
        for (let i = 1; i < 16; i++) {
            if (v[i] !== this.current[i]) {
                this.current = v;
                this.gl.uniformMatrix4fv(this.location, false, v);
                break;
            }
        }
    }
}
const emptyMat3 = new Float32Array(9);
class UniformMatrix3f extends Uniform {
    constructor(context) {
        super(context);
        this.current = emptyMat3;
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        for (let i = 0; i < 9; i++) {
            if (v[i] !== this.current[i]) {
                this.current = v;
                this.gl.uniformMatrix3fv(this.location, false, v);
                break;
            }
        }
    }
}
const emptyMat2 = new Float32Array(4);
class UniformMatrix2f extends Uniform {
    constructor(context) {
        super(context);
        this.current = emptyMat2;
    }
    set(program, name, v) {
        if (!this.fetchUniformLocation(program, name))
            return;
        for (let i = 0; i < 4; i++) {
            if (v[i] !== this.current[i]) {
                this.current = v;
                this.gl.uniformMatrix2fv(this.location, false, v);
                break;
            }
        }
    }
}

function packColor(color) {
    return [
        packUint8ToFloat(255 * color.r, 255 * color.g),
        packUint8ToFloat(255 * color.b, 255 * color.a)
    ];
}
class ConstantBinder {
    constructor(value, names, type, context) {
        this.value = value;
        this.uniformNames = names.map(name => `u_${ name }`);
        this.type = type;
        this.context = context;
    }
    setUniform(program, uniform, globals, currentValue, uniformName) {
        const value = currentValue.constantOr(this.value);
        if (value instanceof Color) {
            uniform.set(program, uniformName, value.toRenderColor(this.ignoreLut ? null : this.context.lut));
        } else {
            uniform.set(program, uniformName, value);
        }
    }
    getBinding(context, _) {
        return this.type === 'color' ? new UniformColor(context) : new Uniform1f(context);
    }
}
class PatternConstantBinder {
    constructor(value, names) {
        this.uniformNames = names.map(name => `u_${ name }`);
        this.pattern = null;
        this.pixelRatio = 1;
    }
    setConstantPatternPositions(posTo) {
        this.pixelRatio = posTo.pixelRatio || 1;
        this.pattern = posTo.tl.concat(posTo.br);
    }
    setUniform(program, uniform, globals, currentValue, uniformName) {
        const pos = uniformName === 'u_pattern' || uniformName === 'u_dash' ? this.pattern : uniformName === 'u_pixel_ratio' ? this.pixelRatio : null;
        if (pos)
            uniform.set(program, uniformName, pos);
    }
    getBinding(context, name) {
        return name === 'u_pattern' || name === 'u_dash' ? new Uniform4f(context) : new Uniform1f(context);
    }
}
class SourceExpressionBinder {
    constructor(expression, names, type, PaintVertexArray) {
        this.expression = expression;
        this.type = type;
        this.maxValue = 0;
        this.paintVertexAttributes = names.map(name => ({
            name: `a_${ name }`,
            type: 'Float32',
            components: type === 'color' ? 2 : 1,
            offset: 0
        }));
        this.paintVertexArray = new PaintVertexArray();
    }
    populatePaintArray(newLength, feature, imagePositions, availableImages, canonical, brightness, formattedSection) {
        const start = this.paintVertexArray.length;
        const value = this.expression.kind === 'composite' || this.expression.kind === 'source' ? this.expression.evaluate(new EvaluationParameters(0, { brightness }), feature, {}, canonical, availableImages, formattedSection) : this.expression.kind === 'constant' && this.expression.value;
        if (this.lutExpression && (this.lutExpression.kind === 'composite' || this.lutExpression.kind === 'source'))
            this.ignoreLut = this.lutExpression.evaluate(new EvaluationParameters(0, { brightness }), feature, {}, canonical, availableImages, formattedSection) === 'none';
        this.paintVertexArray.resize(newLength);
        this._setPaintValue(start, newLength, value, this.context);
    }
    updatePaintArray(start, end, feature, featureState, availableImages, spritePositions, brightness) {
        const value = this.expression.kind === 'composite' || this.expression.kind === 'source' ? this.expression.evaluate({
            zoom: 0,
            brightness
        }, feature, featureState, void 0, availableImages) : this.expression.kind === 'constant' && this.expression.value;
        if (this.lutExpression && (this.lutExpression.kind === 'composite' || this.lutExpression.kind === 'source'))
            this.ignoreLut = this.lutExpression.evaluate({
                zoom: 0,
                brightness
            }, feature, featureState, void 0, availableImages) === 'none';
        this._setPaintValue(start, end, value, this.context);
    }
    _setPaintValue(start, end, value, context) {
        if (this.type === 'color') {
            const color = packColor(value.toRenderColor(this.ignoreLut ? null : context.lut));
            for (let i = start; i < end; i++) {
                this.paintVertexArray.emplace(i, color[0], color[1]);
            }
        } else {
            for (let i = start; i < end; i++) {
                this.paintVertexArray.emplace(i, value);
            }
            this.maxValue = Math.max(this.maxValue, Math.abs(value));
        }
    }
    upload(context) {
        if (this.paintVertexArray && this.paintVertexArray.arrayBuffer) {
            if (this.paintVertexBuffer && this.paintVertexBuffer.buffer) {
                this.paintVertexBuffer.updateData(this.paintVertexArray);
            } else {
                const dynamicDraw = this.lutExpression && this.lutExpression.kind !== 'constant' && (this.lutExpression.isStateDependent || !this.lutExpression.isLightConstant) || this.expression.kind !== 'constant' && (this.expression.isStateDependent || !this.expression.isLightConstant);
                this.paintVertexBuffer = context.createVertexBuffer(this.paintVertexArray, this.paintVertexAttributes, dynamicDraw);
            }
        }
    }
    destroy() {
        if (this.paintVertexBuffer) {
            this.paintVertexBuffer.destroy();
        }
    }
}
class CompositeExpressionBinder {
    constructor(expression, names, type, useIntegerZoom, context, PaintVertexArray) {
        this.expression = expression;
        this.uniformNames = names.map(name => `u_${ name }_t`);
        this.type = type;
        this.useIntegerZoom = useIntegerZoom;
        this.context = context;
        this.maxValue = 0;
        this.paintVertexAttributes = names.map(name => ({
            name: `a_${ name }`,
            type: 'Float32',
            components: type === 'color' ? 4 : 2,
            offset: 0
        }));
        this.paintVertexArray = new PaintVertexArray();
    }
    populatePaintArray(newLength, feature, imagePositions, availableImages, canonical, brightness, formattedSection) {
        const min = this.expression.evaluate(new EvaluationParameters(this.context.zoom, { brightness }), feature, {}, canonical, availableImages, formattedSection);
        const max = this.expression.evaluate(new EvaluationParameters(this.context.zoom + 1, { brightness }), feature, {}, canonical, availableImages, formattedSection);
        if (this.lutExpression && (this.lutExpression.kind === 'composite' || this.lutExpression.kind === 'source'))
            this.ignoreLut = this.lutExpression.evaluate(new EvaluationParameters(this.context.zoom, { brightness }), feature, {}, canonical, availableImages, formattedSection) === 'none';
        const start = this.paintVertexArray.length;
        this.paintVertexArray.resize(newLength);
        this._setPaintValue(start, newLength, min, max, this.context);
    }
    updatePaintArray(start, end, feature, featureState, availableImages, spritePositions, brightness) {
        const min = this.expression.evaluate({
            zoom: this.context.zoom,
            brightness
        }, feature, featureState, void 0, availableImages);
        const max = this.expression.evaluate({
            zoom: this.context.zoom + 1,
            brightness
        }, feature, featureState, void 0, availableImages);
        if (this.lutExpression && (this.lutExpression.kind === 'composite' || this.lutExpression.kind === 'source'))
            this.ignoreLut = this.lutExpression.evaluate({
                zoom: this.context.zoom,
                brightness
            }, feature, featureState, void 0, availableImages) === 'none';
        this._setPaintValue(start, end, min, max, this.context);
    }
    _setPaintValue(start, end, min, max, context) {
        if (this.type === 'color') {
            const minColor = packColor(min.toRenderColor(this.ignoreLut ? null : context.lut));
            const maxColor = packColor(min.toRenderColor(this.ignoreLut ? null : context.lut));
            for (let i = start; i < end; i++) {
                this.paintVertexArray.emplace(i, minColor[0], minColor[1], maxColor[0], maxColor[1]);
            }
        } else {
            for (let i = start; i < end; i++) {
                this.paintVertexArray.emplace(i, min, max);
            }
            this.maxValue = Math.max(this.maxValue, Math.abs(min), Math.abs(max));
        }
    }
    upload(context) {
        if (this.paintVertexArray && this.paintVertexArray.arrayBuffer) {
            if (this.paintVertexBuffer && this.paintVertexBuffer.buffer) {
                this.paintVertexBuffer.updateData(this.paintVertexArray);
            } else {
                this.paintVertexBuffer = context.createVertexBuffer(this.paintVertexArray, this.paintVertexAttributes, this.expression.isStateDependent || !this.expression.isLightConstant);
            }
        }
    }
    destroy() {
        if (this.paintVertexBuffer) {
            this.paintVertexBuffer.destroy();
        }
    }
    setUniform(program, uniform, globals, _, uniformName) {
        const currentZoom = this.useIntegerZoom ? Math.floor(globals.zoom) : globals.zoom;
        const factor = clamp(this.expression.interpolationFactor(currentZoom, this.context.zoom, this.context.zoom + 1), 0, 1);
        uniform.set(program, uniformName, factor);
    }
    getBinding(context, _) {
        return new Uniform1f(context);
    }
}
class PatternCompositeBinder {
    constructor(expression, names, type, PaintVertexArray, layerId) {
        this.expression = expression;
        this.layerId = layerId;
        this.paintVertexAttributes = (type === 'array' ? dashAttributes : patternAttributes).members;
        for (let i = 0; i < names.length; ++i) {
        }
        this.paintVertexArray = new PaintVertexArray();
    }
    populatePaintArray(length, feature, imagePositions, _availableImages) {
        const start = this.paintVertexArray.length;
        this.paintVertexArray.resize(length);
        this._setPaintValues(start, length, feature.patterns && feature.patterns[this.layerId], imagePositions);
    }
    updatePaintArray(start, end, feature, featureState, availableImages, imagePositions, _) {
        this._setPaintValues(start, end, feature.patterns && feature.patterns[this.layerId], imagePositions);
    }
    _setPaintValues(start, end, patterns, positions) {
        if (!positions || !patterns)
            return;
        const pos = positions[patterns];
        if (!pos)
            return;
        const {tl, br, pixelRatio} = pos;
        for (let i = start; i < end; i++) {
            this.paintVertexArray.emplace(i, tl[0], tl[1], br[0], br[1], pixelRatio);
        }
    }
    upload(context) {
        if (this.paintVertexArray && this.paintVertexArray.arrayBuffer) {
            this.paintVertexBuffer = context.createVertexBuffer(this.paintVertexArray, this.paintVertexAttributes, this.expression.isStateDependent || !this.expression.isLightConstant);
        }
    }
    destroy() {
        if (this.paintVertexBuffer)
            this.paintVertexBuffer.destroy();
    }
}
class ProgramConfiguration {
    constructor(layer, context, filterProperties = () => true) {
        this.binders = {};
        this._buffers = [];
        this.context = context;
        const keys = [];
        for (const property in layer.paint._values) {
            const value = layer.paint.get(property);
            const valueUseTheme = layer.paint.get(`${ property }-use-theme`);
            if (property.endsWith('-use-theme'))
                continue;
            if (!filterProperties(property))
                continue;
            if (!(value instanceof PossiblyEvaluatedPropertyValue) || !supportsPropertyExpression(value.property.specification)) {
                continue;
            }
            const names = paintAttributeNames(property, layer.type);
            const expression = value.value;
            const type = value.property.specification.type;
            const useIntegerZoom = !!value.property.useIntegerZoom;
            const isPattern = property === 'line-dasharray' || property.endsWith('pattern');
            const sourceException = property === 'line-dasharray' && layer.layout.get('line-cap').value.kind !== 'constant' || valueUseTheme && valueUseTheme.value.kind !== 'constant';
            if (expression.kind === 'constant' && !sourceException) {
                this.binders[property] = isPattern ? new PatternConstantBinder(expression.value, names) : new ConstantBinder(expression.value, names, type, context);
                keys.push(`/u_${ property }`);
            } else if (expression.kind === 'source' || sourceException || isPattern) {
                const StructArrayLayout = layoutType(property, type, 'source');
                this.binders[property] = isPattern ? // @ts-expect-error - TS2345 - Argument of type 'PossiblyEvaluatedValue<any>' is not assignable to parameter of type 'CompositeExpression'.
                new PatternCompositeBinder(expression, names, type, StructArrayLayout, layer.id) : new SourceExpressionBinder(expression, names, type, StructArrayLayout);
                keys.push(`/a_${ property }`);
            } else {
                const StructArrayLayout = layoutType(property, type, 'composite');
                this.binders[property] = new CompositeExpressionBinder(expression, names, type, useIntegerZoom, context, StructArrayLayout);
                keys.push(`/z_${ property }`);
            }
            if (valueUseTheme) {
                this.binders[property].ignoreLut = valueUseTheme.constantOr('default') === 'none';
                const expressionLUT = valueUseTheme.value;
                this.binders[property].lutExpression = expressionLUT;
                this.binders[property].checkUseTheme = true;
            }
        }
        this.cacheKey = keys.sort().join('');
    }
    getMaxValue(property) {
        const binder = this.binders[property];
        return binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder ? binder.maxValue : 0;
    }
    populatePaintArrays(newLength, feature, imagePositions, availableImages, canonical, brightness, formattedSection) {
        for (const property in this.binders) {
            const binder = this.binders[property];
            binder.context = this.context;
            if (binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder)
                binder.populatePaintArray(newLength, feature, imagePositions, availableImages, canonical, brightness, formattedSection);
            else if (binder.lutExpression) {
                if (binder instanceof ConstantBinder && binder.lutExpression && (binder.lutExpression.kind === 'composite' || binder.lutExpression.kind === 'source'))
                    binder.ignoreLut = binder.lutExpression.evaluate(new EvaluationParameters(0, { brightness }), feature, {}, canonical, availableImages, formattedSection) === 'none';
            }
        }
    }
    setConstantPatternPositions(posTo) {
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof PatternConstantBinder)
                binder.setConstantPatternPositions(posTo);
        }
    }
    updatePaintArrays(featureStates, featureMap, featureMapWithoutIds, vtLayer, layer, availableImages, imagePositions, isBrightnessChanged, brightness) {
        let dirty = false;
        const keys = Object.keys(featureStates);
        const featureStateUpdate = keys.length !== 0 && !isBrightnessChanged;
        const ids = featureStateUpdate ? keys : featureMap.uniqueIds;
        this.context.lut = layer.lut;
        for (const property in this.binders) {
            const binder = this.binders[property];
            binder.context = this.context;
            const isExpressionNotConst = binder.expression && binder.expression.kind && binder.expression.kind !== 'constant';
            if ((binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder) && isExpressionNotConst && (binder.expression.isStateDependent === true || binder.expression.isLightConstant === false)) {
                const value = layer.paint.get(property);
                binder.expression = value.value;
                for (const id of ids) {
                    const state = featureStates[id.toString()];
                    featureMap.eachPosition(id, (index, start, end) => {
                        const feature = vtLayer.feature(index);
                        binder.updatePaintArray(start, end, feature, state, availableImages, imagePositions, brightness);
                    });
                }
                if (!featureStateUpdate) {
                    for (const id of featureMapWithoutIds.uniqueIds) {
                        const state = featureStates[id.toString()];
                        featureMapWithoutIds.eachPosition(id, (index, start, end) => {
                            const feature = vtLayer.feature(index);
                            binder.updatePaintArray(start, end, feature, state, availableImages, imagePositions, brightness);
                        });
                    }
                }
                dirty = true;
            }
        }
        return dirty;
    }
    defines() {
        const result = [];
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof ConstantBinder || binder instanceof PatternConstantBinder) {
                result.push(...binder.uniformNames.map(name => `#define HAS_UNIFORM_${ name }`));
            }
        }
        return result;
    }
    getBinderAttributes() {
        const result = [];
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder) {
                for (let i = 0; i < binder.paintVertexAttributes.length; i++) {
                    result.push(binder.paintVertexAttributes[i].name);
                }
            }
        }
        return result;
    }
    getBinderUniforms() {
        const uniforms = [];
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof ConstantBinder || binder instanceof PatternConstantBinder || binder instanceof CompositeExpressionBinder) {
                for (const uniformName of binder.uniformNames) {
                    uniforms.push(uniformName);
                }
            }
        }
        return uniforms;
    }
    getPaintVertexBuffers() {
        return this._buffers;
    }
    getUniforms(context) {
        const uniforms = [];
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof ConstantBinder || binder instanceof PatternConstantBinder || binder instanceof CompositeExpressionBinder) {
                for (const name of binder.uniformNames) {
                    uniforms.push({
                        name,
                        property,
                        binding: binder.getBinding(context, name)
                    });
                }
            }
        }
        return uniforms;
    }
    setUniforms(program, context, binderUniforms, properties, globals) {
        for (const {name, property, binding} of binderUniforms) {
            if (this.binders[property].checkUseTheme && this.binders[property] instanceof ConstantBinder) {
                const pvalue = properties.get(`${ property }-use-theme`);
                if (pvalue.isConstant())
                    this.binders[property].ignoreLut = pvalue.constantOr('default') === 'none';
            }
            this.binders[property].setUniform(program, binding, globals, properties.get(property), name);
        }
    }
    updatePaintBuffers() {
        this._buffers = [];
        for (const property in this.binders) {
            const binder = this.binders[property];
            if ((binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder) && binder.paintVertexBuffer) {
                this._buffers.push(binder.paintVertexBuffer);
            }
        }
    }
    upload(context) {
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder)
                binder.upload(context);
        }
        this.updatePaintBuffers();
    }
    destroy() {
        for (const property in this.binders) {
            const binder = this.binders[property];
            if (binder instanceof SourceExpressionBinder || binder instanceof CompositeExpressionBinder || binder instanceof PatternCompositeBinder)
                binder.destroy();
        }
    }
}
class ProgramConfigurationSet {
    constructor(layers, context, filterProperties = () => true) {
        this.programConfigurations = {};
        for (const layer of layers) {
            this.programConfigurations[layer.id] = new ProgramConfiguration(layer, context, filterProperties);
        }
        this.needsUpload = false;
        this._featureMap = new FeaturePositionMap();
        this._featureMapWithoutIds = new FeaturePositionMap();
        this._bufferOffset = 0;
        this._idlessCounter = 0;
    }
    populatePaintArrays(length, feature, index, imagePositions, availableImages, canonical, brightness, formattedSection) {
        for (const key in this.programConfigurations) {
            this.programConfigurations[key].populatePaintArrays(length, feature, imagePositions, availableImages, canonical, brightness, formattedSection);
        }
        if (feature.id !== void 0) {
            this._featureMap.add(feature.id, index, this._bufferOffset, length);
        } else {
            this._featureMapWithoutIds.add(this._idlessCounter, index, this._bufferOffset, length);
            this._idlessCounter += 1;
        }
        this._bufferOffset = length;
        this.needsUpload = true;
    }
    updatePaintArrays(featureStates, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness) {
        for (const layer of layers) {
            this.needsUpload = this.programConfigurations[layer.id].updatePaintArrays(featureStates, this._featureMap, this._featureMapWithoutIds, vtLayer, layer, availableImages, imagePositions, isBrightnessChanged, brightness || 0) || this.needsUpload;
        }
    }
    get(layerId) {
        return this.programConfigurations[layerId];
    }
    upload(context) {
        if (!this.needsUpload)
            return;
        for (const layerId in this.programConfigurations) {
            this.programConfigurations[layerId].upload(context);
        }
        this.needsUpload = false;
    }
    destroy() {
        for (const layerId in this.programConfigurations) {
            this.programConfigurations[layerId].destroy();
        }
    }
}
const attributeNameExceptions = {
    'text-opacity': ['opacity'],
    'icon-opacity': ['opacity'],
    'text-occlusion-opacity': ['occlusion_opacity'],
    'icon-occlusion-opacity': ['occlusion_opacity'],
    'text-color': ['fill_color'],
    'icon-color': ['fill_color'],
    'text-emissive-strength': ['emissive_strength'],
    'icon-emissive-strength': ['emissive_strength'],
    'text-halo-color': ['halo_color'],
    'icon-halo-color': ['halo_color'],
    'text-halo-blur': ['halo_blur'],
    'icon-halo-blur': ['halo_blur'],
    'text-halo-width': ['halo_width'],
    'icon-halo-width': ['halo_width'],
    'symbol-z-offset': ['z_offset'],
    'line-gap-width': ['gapwidth'],
    'line-pattern': [
        'pattern',
        'pixel_ratio'
    ],
    'fill-pattern': [
        'pattern',
        'pixel_ratio'
    ],
    'fill-extrusion-pattern': [
        'pattern',
        'pixel_ratio'
    ],
    'line-dasharray': ['dash']
};
function paintAttributeNames(property, type) {
    return attributeNameExceptions[property] || [property.replace(`${ type }-`, '').replace(/-/g, '_')];
}
const propertyExceptions = {
    'line-pattern': {
        'source': StructArrayLayout4ui1f12,
        'composite': StructArrayLayout4ui1f12
    },
    'fill-pattern': {
        'source': StructArrayLayout4ui1f12,
        'composite': StructArrayLayout4ui1f12
    },
    'fill-extrusion-pattern': {
        'source': StructArrayLayout4ui1f12,
        'composite': StructArrayLayout4ui1f12
    },
    'line-dasharray': {
        // temporary layout
        'source': StructArrayLayout4ui8,
        'composite': StructArrayLayout4ui8
    }
};
const defaultLayouts = {
    'color': {
        'source': StructArrayLayout2f8,
        'composite': StructArrayLayout4f16
    },
    'number': {
        'source': StructArrayLayout1f4,
        'composite': StructArrayLayout2f8
    }
};
function layoutType(property, type, binderType) {
    const layoutException = propertyExceptions[property];
    return layoutException && layoutException[binderType] || defaultLayouts[type][binderType];
}
register(ConstantBinder, 'ConstantBinder');
register(PatternConstantBinder, 'PatternConstantBinder');
register(SourceExpressionBinder, 'SourceExpressionBinder');
register(PatternCompositeBinder, 'PatternCompositeBinder');
register(CompositeExpressionBinder, 'CompositeExpressionBinder');
register(ProgramConfiguration, 'ProgramConfiguration', { omit: ['_buffers'] });
register(ProgramConfigurationSet, 'ProgramConfigurationSet');

const GLOBE_RADIUS = EXTENT / Math.PI / 2;
const GLOBE_ZOOM_THRESHOLD_MIN = 5;
const GLOBE_ZOOM_THRESHOLD_MAX = 6;
const GLOBE_SCALE_MATCH_LATITUDE = 45;
const GLOBE_NORMALIZATION_BIT_RANGE = 15;
const GLOBE_NORMALIZATION_MASK = (1 << GLOBE_NORMALIZATION_BIT_RANGE - 1) - 1;
const GLOBE_VERTEX_GRID_SIZE = 64;
const GLOBE_LATITUDINAL_GRID_LOD_TABLE = [
    GLOBE_VERTEX_GRID_SIZE,
    GLOBE_VERTEX_GRID_SIZE / 2,
    GLOBE_VERTEX_GRID_SIZE / 4
];
const TILE_SIZE = 512;
const GLOBE_MIN = -GLOBE_RADIUS;
const GLOBE_MAX = GLOBE_RADIUS;

function csLatLngToECEF(cosLat, sinLat, lng, radius = GLOBE_RADIUS) {
    lng = degToRad(lng);
    const sx = cosLat * Math.sin(lng) * radius;
    const sy = -sinLat * radius;
    const sz = cosLat * Math.cos(lng) * radius;
    return [
        sx,
        sy,
        sz
    ];
}
function ecefToLatLng([x, y, z]) {
    const radius = Math.hypot(x, y, z);
    const lng = Math.atan2(x, z);
    const lat = Math.PI * 0.5 - Math.acos(-y / radius);
    return new LngLat(radToDeg(lng), radToDeg(lat));
}
function latLngToECEF(lat, lng, radius) {
    return csLatLngToECEF(Math.cos(degToRad(lat)), Math.sin(degToRad(lat)), lng, radius);
}
const earthRadius = 6371008.8;
const earthCircumference = 2 * Math.PI * earthRadius;
class LngLat {
    constructor(lng, lat) {
        if (isNaN(lng) || isNaN(lat)) {
            throw new Error(`Invalid LngLat object: (${ lng }, ${ lat })`);
        }
        this.lng = +lng;
        this.lat = +lat;
        if (this.lat > 90 || this.lat < -90) {
            throw new Error('Invalid LngLat latitude value: must be between -90 and 90');
        }
    }
    /**
   * Returns a new `LngLat` object whose longitude is wrapped to the range (-180, 180).
   *
   * @returns {LngLat} The wrapped `LngLat` object.
   * @example
   * const ll = new mapboxgl.LngLat(286.0251, 40.7736);
   * const wrapped = ll.wrap();
   * console.log(wrapped.lng); // = -73.9749
   */
    wrap() {
        return new LngLat(wrap$1(this.lng, -180, 180), this.lat);
    }
    /**
   * Returns the coordinates represented as an array of two numbers.
   *
   * @returns {Array<number>} The coordinates represeted as an array of longitude and latitude.
   * @example
   * const ll = new mapboxgl.LngLat(-73.9749, 40.7736);
   * ll.toArray(); // = [-73.9749, 40.7736]
   */
    toArray() {
        return [
            this.lng,
            this.lat
        ];
    }
    /**
   * Returns the coordinates represent as a string.
   *
   * @returns {string} The coordinates represented as a string of the format `'LngLat(lng, lat)'`.
   * @example
   * const ll = new mapboxgl.LngLat(-73.9749, 40.7736);
   * ll.toString(); // = "LngLat(-73.9749, 40.7736)"
   */
    toString() {
        return `LngLat(${ this.lng }, ${ this.lat })`;
    }
    /**
   * Returns the approximate distance between a pair of coordinates in meters.
   * Uses the Haversine Formula (from R.W. Sinnott, "Virtues of the Haversine", Sky and Telescope, vol. 68, no. 2, 1984, p. 159).
   *
   * @param {LngLat} lngLat Coordinates to compute the distance to.
   * @returns {number} Distance in meters between the two coordinates.
   * @example
   * const newYork = new mapboxgl.LngLat(-74.0060, 40.7128);
   * const losAngeles = new mapboxgl.LngLat(-118.2437, 34.0522);
   * newYork.distanceTo(losAngeles); // = 3935751.690893987, "true distance" using a non-spherical approximation is ~3966km
   */
    distanceTo(lngLat) {
        const rad = Math.PI / 180;
        const lat1 = this.lat * rad;
        const lat2 = lngLat.lat * rad;
        const a = Math.sin(lat1) * Math.sin(lat2) + Math.cos(lat1) * Math.cos(lat2) * Math.cos((lngLat.lng - this.lng) * rad);
        const maxMeters = earthRadius * Math.acos(Math.min(a, 1));
        return maxMeters;
    }
    /**
   * Returns a `LngLatBounds` from the coordinates extended by a given `radius`. The returned `LngLatBounds` completely contains the `radius`.
   *
   * @param {number} [radius=0] Distance in meters from the coordinates to extend the bounds.
   * @returns {LngLatBounds} A new `LngLatBounds` object representing the coordinates extended by the `radius`.
   * @example
   * const ll = new mapboxgl.LngLat(-73.9749, 40.7736);
   * ll.toBounds(100).toArray(); // = [[-73.97501862141328, 40.77351016847229], [-73.97478137858673, 40.77368983152771]]
   */
    toBounds(radius = 0) {
        const earthCircumferenceInMetersAtEquator = 40075017;
        const latAccuracy = 360 * radius / earthCircumferenceInMetersAtEquator, lngAccuracy = latAccuracy / Math.cos(Math.PI / 180 * this.lat);
        return new LngLatBounds({
            lng: this.lng - lngAccuracy,
            lat: this.lat - latAccuracy
        }, {
            lng: this.lng + lngAccuracy,
            lat: this.lat + latAccuracy
        });
    }
    toEcef(altitude) {
        const altInEcef = altitude * GLOBE_RADIUS / earthRadius;
        const radius = GLOBE_RADIUS + altInEcef;
        return latLngToECEF(this.lat, this.lng, radius);
    }
    /**
   * Converts an array of two numbers or an object with `lng` and `lat` or `lon` and `lat` properties
   * to a `LngLat` object.
   *
   * If a `LngLat` object is passed in, the function returns it unchanged.
   *
   * @param {LngLatLike} input An array of two numbers or object to convert, or a `LngLat` object to return.
   * @returns {LngLat} A new `LngLat` object, if a conversion occurred, or the original `LngLat` object.
   * @example
   * const arr = [-73.9749, 40.7736];
   * const ll = mapboxgl.LngLat.convert(arr);
   * console.log(ll);   // = LngLat {lng: -73.9749, lat: 40.7736}
   */
    static convert(input) {
        if (input instanceof LngLat) {
            return input;
        }
        if (Array.isArray(input) && (input.length === 2 || input.length === 3)) {
            return new LngLat(Number(input[0]), Number(input[1]));
        }
        if (!Array.isArray(input) && typeof input === 'object' && input !== null) {
            return new LngLat(Number('lng' in input ? input.lng : input.lon), Number(input.lat));
        }
        throw new Error('`LngLatLike` argument must be specified as a LngLat instance, an object {lng: <lng>, lat: <lat>}, an object {lon: <lng>, lat: <lat>}, or an array of [<lng>, <lat>]');
    }
}
class LngLatBounds {
    constructor(sw, ne) {
        if (!sw) ; else if (ne) {
            this.setSouthWest(sw).setNorthEast(ne);
        } else if (sw.length === 4) {
            const bounds = sw;
            this.setSouthWest([
                bounds[0],
                bounds[1]
            ]).setNorthEast([
                bounds[2],
                bounds[3]
            ]);
        } else {
            const bounds = sw;
            this.setSouthWest(bounds[0]).setNorthEast(bounds[1]);
        }
    }
    /**
   * Set the northeast corner of the bounding box.
   *
   * @param {LngLatLike} ne A {@link LngLatLike} object describing the northeast corner of the bounding box.
   * @returns {LngLatBounds} Returns itself to allow for method chaining.
   * @example
   * const sw = new mapboxgl.LngLat(-73.9876, 40.7661);
   * const ne = new mapboxgl.LngLat(-73.9397, 40.8002);
   * const llb = new mapboxgl.LngLatBounds(sw, ne);
   * llb.setNorthEast([-73.9397, 42.8002]);
   */
    setNorthEast(ne) {
        this._ne = ne instanceof LngLat ? new LngLat(ne.lng, ne.lat) : LngLat.convert(ne);
        return this;
    }
    /**
   * Set the southwest corner of the bounding box.
   *
   * @param {LngLatLike} sw A {@link LngLatLike} object describing the southwest corner of the bounding box.
   * @returns {LngLatBounds} Returns itself to allow for method chaining.
   * @example
   * const sw = new mapboxgl.LngLat(-73.9876, 40.7661);
   * const ne = new mapboxgl.LngLat(-73.9397, 40.8002);
   * const llb = new mapboxgl.LngLatBounds(sw, ne);
   * llb.setSouthWest([-73.9876, 40.2661]);
   */
    setSouthWest(sw) {
        this._sw = sw instanceof LngLat ? new LngLat(sw.lng, sw.lat) : LngLat.convert(sw);
        return this;
    }
    /**
   * Extend the bounds to include a given LngLatLike or LngLatBoundsLike.
   *
   * @param {LngLatLike|LngLatBoundsLike} obj Object to extend to.
   * @returns {LngLatBounds} Returns itself to allow for method chaining.
   * @example
   * const sw = new mapboxgl.LngLat(-73.9876, 40.7661);
   * const ne = new mapboxgl.LngLat(-73.9397, 40.8002);
   * const llb = new mapboxgl.LngLatBounds(sw, ne);
   * llb.extend([-72.9876, 42.2661]);
   */
    extend(obj) {
        const sw = this._sw, ne = this._ne;
        let sw2, ne2;
        if (obj instanceof LngLat) {
            sw2 = obj;
            ne2 = obj;
        } else if (obj instanceof LngLatBounds) {
            sw2 = obj._sw;
            ne2 = obj._ne;
            if (!sw2 || !ne2)
                return this;
        } else if (Array.isArray(obj)) {
            if (obj.length === 4 || obj.every(Array.isArray)) {
                const lngLatBoundsObj = obj;
                return this.extend(LngLatBounds.convert(lngLatBoundsObj));
            } else {
                const lngLatObj = obj;
                return this.extend(LngLat.convert(lngLatObj));
            }
        } else if (typeof obj === 'object' && obj !== null && obj.hasOwnProperty('lat') && (obj.hasOwnProperty('lon') || obj.hasOwnProperty('lng'))) {
            return this.extend(LngLat.convert(obj));
        } else {
            return this;
        }
        if (!sw && !ne) {
            this._sw = new LngLat(sw2.lng, sw2.lat);
            this._ne = new LngLat(ne2.lng, ne2.lat);
        } else {
            sw.lng = Math.min(sw2.lng, sw.lng);
            sw.lat = Math.min(sw2.lat, sw.lat);
            ne.lng = Math.max(ne2.lng, ne.lng);
            ne.lat = Math.max(ne2.lat, ne.lat);
        }
        return this;
    }
    /**
   * Returns the geographical coordinate equidistant from the bounding box's corners.
   *
   * @returns {LngLat} The bounding box's center.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getCenter(); // = LngLat {lng: -73.96365, lat: 40.78315}
   */
    getCenter() {
        return new LngLat((this._sw.lng + this._ne.lng) / 2, (this._sw.lat + this._ne.lat) / 2);
    }
    /**
   * Returns the southwest corner of the bounding box.
   *
   * @returns {LngLat} The southwest corner of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getSouthWest(); // LngLat {lng: -73.9876, lat: 40.7661}
   */
    getSouthWest() {
        return this._sw;
    }
    /**
   * Returns the northeast corner of the bounding box.
   *
   * @returns {LngLat} The northeast corner of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getNorthEast(); // LngLat {lng: -73.9397, lat: 40.8002}
   */
    getNorthEast() {
        return this._ne;
    }
    /**
   * Returns the northwest corner of the bounding box.
   *
   * @returns {LngLat} The northwest corner of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getNorthWest(); // LngLat {lng: -73.9876, lat: 40.8002}
   */
    getNorthWest() {
        return new LngLat(this.getWest(), this.getNorth());
    }
    /**
   * Returns the southeast corner of the bounding box.
   *
   * @returns {LngLat} The southeast corner of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getSouthEast(); // LngLat {lng: -73.9397, lat: 40.7661}
   */
    getSouthEast() {
        return new LngLat(this.getEast(), this.getSouth());
    }
    /**
   * Returns the west edge of the bounding box.
   *
   * @returns {number} The west edge of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getWest(); // -73.9876
   */
    getWest() {
        return this._sw.lng;
    }
    /**
   * Returns the south edge of the bounding box.
   *
   * @returns {number} The south edge of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getSouth(); // 40.7661
   */
    getSouth() {
        return this._sw.lat;
    }
    /**
   * Returns the east edge of the bounding box.
   *
   * @returns {number} The east edge of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getEast(); // -73.9397
   */
    getEast() {
        return this._ne.lng;
    }
    /**
   * Returns the north edge of the bounding box.
   *
   * @returns {number} The north edge of the bounding box.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.getNorth(); // 40.8002
   */
    getNorth() {
        return this._ne.lat;
    }
    /**
   * Returns the bounding box represented as an array.
   *
   * @returns {Array<Array<number>>} The bounding box represented as an array, consisting of the
   * southwest and northeast coordinates of the bounding represented as arrays of numbers.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.toArray(); // = [[-73.9876, 40.7661], [-73.9397, 40.8002]]
   */
    toArray() {
        return [
            this._sw.toArray(),
            this._ne.toArray()
        ];
    }
    /**
   * Return the bounding box represented as a string.
   *
   * @returns {string} The bounding box represents as a string of the format
   * `'LngLatBounds(LngLat(lng, lat), LngLat(lng, lat))'`.
   * @example
   * const llb = new mapboxgl.LngLatBounds([-73.9876, 40.7661], [-73.9397, 40.8002]);
   * llb.toString(); // = "LngLatBounds(LngLat(-73.9876, 40.7661), LngLat(-73.9397, 40.8002))"
   */
    toString() {
        return `LngLatBounds(${ this._sw.toString() }, ${ this._ne.toString() })`;
    }
    /**
   * Check if the bounding box is an empty/`null`-type box.
   *
   * @returns {boolean} True if bounds have been defined, otherwise false.
   * @example
   * const llb = new mapboxgl.LngLatBounds();
   * llb.isEmpty(); // true
   * llb.setNorthEast([-73.9876, 40.7661]);
   * llb.setSouthWest([-73.9397, 40.8002]);
   * llb.isEmpty(); // false
   */
    isEmpty() {
        return !(this._sw && this._ne);
    }
    /**
  * Check if the point is within the bounding box.
  *
  * @param {LngLatLike} lnglat Geographic point to check against.
  * @returns {boolean} True if the point is within the bounding box.
  * @example
  * const llb = new mapboxgl.LngLatBounds(
  *   new mapboxgl.LngLat(-73.9876, 40.7661),
  *   new mapboxgl.LngLat(-73.9397, 40.8002)
  * );
  *
  * const ll = new mapboxgl.LngLat(-73.9567, 40.7789);
  *
  * console.log(llb.contains(ll)); // = true
  */
    contains(lnglat) {
        const {lng, lat} = LngLat.convert(lnglat);
        const containsLatitude = this._sw.lat <= lat && lat <= this._ne.lat;
        let containsLongitude = this._sw.lng <= lng && lng <= this._ne.lng;
        if (this._sw.lng > this._ne.lng) {
            containsLongitude = this._sw.lng >= lng && lng >= this._ne.lng;
        }
        return containsLatitude && containsLongitude;
    }
    /**
   * Converts an array to a `LngLatBounds` object.
   *
   * If a `LngLatBounds` object is passed in, the function returns it unchanged.
   *
   * Internally, the function calls `LngLat#convert` to convert arrays to `LngLat` values.
   *
   * @param {LngLatBoundsLike} input An array of two coordinates to convert, or a `LngLatBounds` object to return.
   * @returns {LngLatBounds} A new `LngLatBounds` object, if a conversion occurred, or the original `LngLatBounds` object.
   * @example
   * const arr = [[-73.9876, 40.7661], [-73.9397, 40.8002]];
   * const llb = mapboxgl.LngLatBounds.convert(arr);
   * console.log(llb);   // = LngLatBounds {_sw: LngLat {lng: -73.9876, lat: 40.7661}, _ne: LngLat {lng: -73.9397, lat: 40.8002}}
   */
    static convert(input) {
        if (!input)
            return;
        if (input instanceof LngLatBounds)
            return input;
        return new LngLatBounds(input);
    }
}

const DEFAULT_MIN_ZOOM = 0;
const DEFAULT_MAX_ZOOM = 25.5;
function circumferenceAtLatitude(latitude) {
    return earthCircumference * Math.cos(latitude * Math.PI / 180);
}
function mercatorXfromLng(lng) {
    return (180 + lng) / 360;
}
function mercatorYfromLat(lat) {
    return (180 - 180 / Math.PI * Math.log(Math.tan(Math.PI / 4 + lat * Math.PI / 360))) / 360;
}
function mercatorZfromAltitude(altitude, lat) {
    return altitude / circumferenceAtLatitude(lat);
}
function lngFromMercatorX(x) {
    return x * 360 - 180;
}
function latFromMercatorY(y) {
    const y2 = 180 - y * 360;
    return 360 / Math.PI * Math.atan(Math.exp(y2 * Math.PI / 180)) - 90;
}
function altitudeFromMercatorZ(z, y) {
    return z * circumferenceAtLatitude(latFromMercatorY(y));
}
const MAX_MERCATOR_LATITUDE = 85.051129;
function getLatitudeScale(lat) {
    return Math.cos(degToRad(clamp(lat, -MAX_MERCATOR_LATITUDE, MAX_MERCATOR_LATITUDE)));
}
function getMetersPerPixelAtLatitude(lat, zoom) {
    const constrainedZoom = clamp(zoom, DEFAULT_MIN_ZOOM, DEFAULT_MAX_ZOOM);
    const constrainedScale = Math.pow(2, constrainedZoom);
    return getLatitudeScale(lat) * earthCircumference / (constrainedScale * 512);
}
function mercatorScale(lat) {
    return 1 / Math.cos(lat * Math.PI / 180);
}
function tileToMeter(canonical, tileYCoordinate = 0) {
    const circumferenceAtEquator = 40075017;
    const mercatorY = (canonical.y + tileYCoordinate / EXTENT) / (1 << canonical.z);
    const exp = Math.exp(Math.PI * (1 - 2 * mercatorY));
    return circumferenceAtEquator * 2 * exp / (exp * exp + 1) / EXTENT / (1 << canonical.z);
}
class MercatorCoordinate {
    constructor(x, y, z = 0) {
        this.x = +x;
        this.y = +y;
        this.z = +z;
    }
    /**
   * Project a `LngLat` to a `MercatorCoordinate`.
   *
   * @param {LngLatLike} lngLatLike The location to project.
   * @param {number} altitude The altitude in meters of the position.
   * @returns {MercatorCoordinate} The projected mercator coordinate.
   * @example
   * const coord = mapboxgl.MercatorCoordinate.fromLngLat({lng: 0, lat: 0}, 0);
   * console.log(coord); // MercatorCoordinate(0.5, 0.5, 0)
   */
    static fromLngLat(lngLatLike, altitude = 0) {
        const lngLat = LngLat.convert(lngLatLike);
        return new MercatorCoordinate(mercatorXfromLng(lngLat.lng), mercatorYfromLat(lngLat.lat), mercatorZfromAltitude(altitude, lngLat.lat));
    }
    /**
   * Returns the `LngLat` for the coordinate.
   *
   * @returns {LngLat} The `LngLat` object.
   * @example
   * const coord = new mapboxgl.MercatorCoordinate(0.5, 0.5, 0);
   * const lngLat = coord.toLngLat(); // LngLat(0, 0)
   */
    toLngLat() {
        return new LngLat(lngFromMercatorX(this.x), latFromMercatorY(this.y));
    }
    /**
   * Returns the altitude in meters of the coordinate.
   *
   * @returns {number} The altitude in meters.
   * @example
   * const coord = new mapboxgl.MercatorCoordinate(0, 0, 0.02);
   * coord.toAltitude(); // 6914.281956295339
   */
    toAltitude() {
        return altitudeFromMercatorZ(this.z, this.y);
    }
    /**
   * Returns the distance of 1 meter in `MercatorCoordinate` units at this latitude.
   *
   * For coordinates in real world units using meters, this naturally provides the scale
   * to transform into `MercatorCoordinate`s.
   *
   * @returns {number} Distance of 1 meter in `MercatorCoordinate` units.
   * @example
   * // Calculate a new MercatorCoordinate that is 150 meters west of the other coord.
   * const coord = new mapboxgl.MercatorCoordinate(0.5, 0.25, 0);
   * const offsetInMeters = 150;
   * const offsetInMercatorCoordinateUnits = offsetInMeters * coord.meterInMercatorCoordinateUnits();
   * const westCoord = new mapboxgl.MercatorCoordinate(coord.x - offsetInMercatorCoordinateUnits, coord.y, coord.z);
   */
    meterInMercatorCoordinateUnits() {
        return 1 / earthCircumference * mercatorScale(latFromMercatorY(this.y));
    }
}

function pointToLineDist(px, py, ax, ay, bx, by) {
    const dx = ax - bx;
    const dy = ay - by;
    return Math.abs((ay - py) * dx - (ax - px) * dy) / Math.hypot(dx, dy);
}
function addResampled(resampled, mx0, my0, mx2, my2, start, end, reproject, tolerance) {
    const mx1 = (mx0 + mx2) / 2;
    const my1 = (my0 + my2) / 2;
    const mid = new Point(mx1, my1);
    reproject(mid);
    const err = pointToLineDist(mid.x, mid.y, start.x, start.y, end.x, end.y);
    if (err >= tolerance) {
        addResampled(resampled, mx0, my0, mx1, my1, start, mid, reproject, tolerance);
        addResampled(resampled, mx1, my1, mx2, my2, mid, end, reproject, tolerance);
    } else {
        resampled.push(end);
    }
}
function resample$1(line, reproject, tolerance) {
    let prev = line[0];
    let mx0 = prev.x;
    let my0 = prev.y;
    reproject(prev);
    const resampled = [prev];
    for (let i = 1; i < line.length; i++) {
        const point = line[i];
        const {x, y} = point;
        reproject(point);
        addResampled(resampled, mx0, my0, x, y, prev, point, reproject, tolerance);
        mx0 = x;
        my0 = y;
        prev = point;
    }
    return resampled;
}
function addResampledPred(resampled, a, b, pred) {
    const split = pred(a, b);
    if (split) {
        const mid = a.add(b)._mult(0.5);
        addResampledPred(resampled, a, mid, pred);
        addResampledPred(resampled, mid, b, pred);
    } else {
        resampled.push(b);
    }
}
function resamplePred(line, predicate) {
    let prev = line[0];
    const resampled = [prev];
    for (let i = 1; i < line.length; i++) {
        const point = line[i];
        addResampledPred(resampled, prev, point, predicate);
        prev = point;
    }
    return resampled;
}

const BITS = 15;
const MAX = Math.pow(2, BITS - 1) - 1;
const MIN = -MAX - 1;
function preparePoint(point, scale) {
    const x = Math.round(point.x * scale);
    const y = Math.round(point.y * scale);
    point.x = clamp(x, MIN, MAX);
    point.y = clamp(y, MIN, MAX);
    if (x < point.x || x > point.x + 1 || y < point.y || y > point.y + 1) {
        warnOnce('Geometry exceeds allowed extent, reduce your vector tile buffer size');
    }
    return point;
}
function loadGeometry(feature, canonical, tileTransform) {
    const geometry = feature.loadGeometry();
    const extent = feature.extent;
    const extentScale = EXTENT / extent;
    if (canonical && tileTransform && tileTransform.projection.isReprojectedInTileSpace) {
        const z2 = 1 << canonical.z;
        const {scale, x, y, projection} = tileTransform;
        const reproject = p => {
            const lng = lngFromMercatorX((canonical.x + p.x / extent) / z2);
            const lat = latFromMercatorY((canonical.y + p.y / extent) / z2);
            const p2 = projection.project(lng, lat);
            p.x = (p2.x * scale - x) * extent;
            p.y = (p2.y * scale - y) * extent;
        };
        for (let i = 0; i < geometry.length; i++) {
            if (feature.type !== 1) {
                geometry[i] = resample$1(geometry[i], reproject, 1);
            } else {
                const line = [];
                for (const p of geometry[i]) {
                    if (p.x < 0 || p.x >= extent || p.y < 0 || p.y >= extent)
                        continue;
                    reproject(p);
                    line.push(p);
                }
                geometry[i] = line;
            }
        }
    }
    for (const line of geometry) {
        for (const p of line) {
            preparePoint(p, extentScale);
        }
    }
    return geometry;
}

function toEvaluationFeature(feature, needGeometry) {
    return {
        type: feature.type,
        id: feature.id,
        properties: feature.properties,
        geometry: needGeometry ? loadGeometry(feature) : []
    };
}

function addCircleVertex(layoutVertexArray, x, y, extrudeX, extrudeY) {
    layoutVertexArray.emplaceBack(x * 2 + (extrudeX + 1) / 2, y * 2 + (extrudeY + 1) / 2);
}
function addGlobeExtVertex$1(vertexArray, pos, normal) {
    const encode = 1 << 14;
    vertexArray.emplaceBack(pos.x, pos.y, pos.z, normal[0] * encode, normal[1] * encode, normal[2] * encode);
}
class CircleBucket {
    constructor(options) {
        this.zoom = options.zoom;
        this.overscaling = options.overscaling;
        this.layers = options.layers;
        this.layerIds = this.layers.map(layer => layer.fqid);
        this.index = options.index;
        this.hasPattern = false;
        this.projection = options.projection;
        this.layoutVertexArray = new StructArrayLayout2i4();
        this.indexArray = new StructArrayLayout3ui6();
        this.segments = new SegmentVector();
        this.programConfigurations = new ProgramConfigurationSet(options.layers, {
            zoom: options.zoom,
            lut: options.lut
        });
        this.stateDependentLayerIds = this.layers.filter(l => l.isStateDependent()).map(l => l.id);
    }
    updateFootprints(_id, _footprints) {
    }
    populate(features, options, canonical, tileTransform) {
        const styleLayer = this.layers[0];
        const bucketFeatures = [];
        let circleSortKey = null;
        if (styleLayer.type === 'circle') {
            circleSortKey = styleLayer.layout.get('circle-sort-key');
        }
        for (const {feature, id, index, sourceLayerIndex} of features) {
            const needGeometry = this.layers[0]._featureFilter.needGeometry;
            const evaluationFeature = toEvaluationFeature(feature, needGeometry);
            if (!this.layers[0]._featureFilter.filter(new EvaluationParameters(this.zoom), evaluationFeature, canonical))
                continue;
            const sortKey = circleSortKey ? circleSortKey.evaluate(evaluationFeature, {}, canonical) : void 0;
            const bucketFeature = {
                id,
                properties: feature.properties,
                type: feature.type,
                sourceLayerIndex,
                index,
                geometry: needGeometry ? evaluationFeature.geometry : loadGeometry(feature, canonical, tileTransform),
                patterns: {},
                sortKey
            };
            bucketFeatures.push(bucketFeature);
        }
        if (circleSortKey) {
            bucketFeatures.sort((a, b) => {
                return a.sortKey - b.sortKey;
            });
        }
        let globeProjection = null;
        if (tileTransform.projection.name === 'globe') {
            this.globeExtVertexArray = new StructArrayLayout6i12();
            globeProjection = tileTransform.projection;
        }
        for (const bucketFeature of bucketFeatures) {
            const {geometry, index, sourceLayerIndex} = bucketFeature;
            const feature = features[index].feature;
            this.addFeature(bucketFeature, geometry, index, options.availableImages, canonical, globeProjection, options.brightness);
            options.featureIndex.insert(feature, geometry, index, sourceLayerIndex, this.index);
        }
    }
    update(states, vtLayer, availableImages, imagePositions, layers, isBrightnessChanged, brightness) {
        this.programConfigurations.updatePaintArrays(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness);
    }
    isEmpty() {
        return this.layoutVertexArray.length === 0;
    }
    uploadPending() {
        return !this.uploaded || this.programConfigurations.needsUpload;
    }
    upload(context) {
        if (!this.uploaded) {
            this.layoutVertexBuffer = context.createVertexBuffer(this.layoutVertexArray, circleAttributes.members);
            this.indexBuffer = context.createIndexBuffer(this.indexArray);
            if (this.globeExtVertexArray) {
                this.globeExtVertexBuffer = context.createVertexBuffer(this.globeExtVertexArray, circleGlobeAttributesExt.members);
            }
        }
        this.programConfigurations.upload(context);
        this.uploaded = true;
    }
    destroy() {
        if (!this.layoutVertexBuffer)
            return;
        this.layoutVertexBuffer.destroy();
        this.indexBuffer.destroy();
        this.programConfigurations.destroy();
        this.segments.destroy();
        if (this.globeExtVertexBuffer) {
            this.globeExtVertexBuffer.destroy();
        }
    }
    addFeature(feature, geometry, index, availableImages, canonical, projection, brightness) {
        for (const ring of geometry) {
            for (const point of ring) {
                const x = point.x;
                const y = point.y;
                if (x < 0 || x >= EXTENT || y < 0 || y >= EXTENT)
                    continue;
                if (projection) {
                    const projectedPoint = projection.projectTilePoint(x, y, canonical);
                    const normal = projection.upVector(canonical, x, y);
                    const array = this.globeExtVertexArray;
                    addGlobeExtVertex$1(array, projectedPoint, normal);
                    addGlobeExtVertex$1(array, projectedPoint, normal);
                    addGlobeExtVertex$1(array, projectedPoint, normal);
                    addGlobeExtVertex$1(array, projectedPoint, normal);
                }
                const segment = this.segments.prepareSegment(4, this.layoutVertexArray, this.indexArray, feature.sortKey);
                const index2 = segment.vertexLength;
                addCircleVertex(this.layoutVertexArray, x, y, -1, -1);
                addCircleVertex(this.layoutVertexArray, x, y, 1, -1);
                addCircleVertex(this.layoutVertexArray, x, y, 1, 1);
                addCircleVertex(this.layoutVertexArray, x, y, -1, 1);
                this.indexArray.emplaceBack(index2, index2 + 1, index2 + 2);
                this.indexArray.emplaceBack(index2, index2 + 2, index2 + 3);
                segment.vertexLength += 4;
                segment.primitiveLength += 2;
            }
        }
        this.programConfigurations.populatePaintArrays(this.layoutVertexArray.length, feature, index, {}, availableImages, canonical, brightness);
    }
}
register(CircleBucket, 'CircleBucket', { omit: ['layers'] });

function polygonIntersectsPolygon(polygonA, polygonB) {
    for (let i = 0; i < polygonA.length; i++) {
        if (polygonContainsPoint(polygonB, polygonA[i]))
            return true;
    }
    for (let i = 0; i < polygonB.length; i++) {
        if (polygonContainsPoint(polygonA, polygonB[i]))
            return true;
    }
    if (lineIntersectsLine(polygonA, polygonB))
        return true;
    return false;
}
function polygonIntersectsBufferedPoint(polygon, point, radius) {
    if (polygonContainsPoint(polygon, point))
        return true;
    if (pointIntersectsBufferedLine(point, polygon, radius))
        return true;
    return false;
}
function polygonIntersectsMultiPolygon(polygon, multiPolygon) {
    if (polygon.length === 1) {
        return multiPolygonContainsPoint(multiPolygon, polygon[0]);
    }
    for (let m = 0; m < multiPolygon.length; m++) {
        const ring = multiPolygon[m];
        for (let n = 0; n < ring.length; n++) {
            if (polygonContainsPoint(polygon, ring[n]))
                return true;
        }
    }
    for (let i = 0; i < polygon.length; i++) {
        if (multiPolygonContainsPoint(multiPolygon, polygon[i]))
            return true;
    }
    for (let k = 0; k < multiPolygon.length; k++) {
        if (lineIntersectsLine(polygon, multiPolygon[k]))
            return true;
    }
    return false;
}
function polygonIntersectsBufferedMultiLine(polygon, multiLine, radius) {
    for (let i = 0; i < multiLine.length; i++) {
        const line = multiLine[i];
        if (polygon.length >= 3) {
            for (let k = 0; k < line.length; k++) {
                if (polygonContainsPoint(polygon, line[k]))
                    return true;
            }
        }
        if (lineIntersectsBufferedLine(polygon, line, radius))
            return true;
    }
    return false;
}
function lineIntersectsBufferedLine(lineA, lineB, radius) {
    if (lineA.length > 1) {
        if (lineIntersectsLine(lineA, lineB))
            return true;
        for (let j = 0; j < lineB.length; j++) {
            if (pointIntersectsBufferedLine(lineB[j], lineA, radius))
                return true;
        }
    }
    for (let k = 0; k < lineA.length; k++) {
        if (pointIntersectsBufferedLine(lineA[k], lineB, radius))
            return true;
    }
    return false;
}
function lineIntersectsLine(lineA, lineB) {
    if (lineA.length === 0 || lineB.length === 0)
        return false;
    for (let i = 0; i < lineA.length - 1; i++) {
        const a0 = lineA[i];
        const a1 = lineA[i + 1];
        for (let j = 0; j < lineB.length - 1; j++) {
            const b0 = lineB[j];
            const b1 = lineB[j + 1];
            if (lineSegmentIntersectsLineSegment(a0, a1, b0, b1))
                return true;
        }
    }
    return false;
}
function lineSegmentIntersectsLineSegment(a0, a1, b0, b1) {
    return isCounterClockwise(a0, b0, b1) !== isCounterClockwise(a1, b0, b1) && isCounterClockwise(a0, a1, b0) !== isCounterClockwise(a0, a1, b1);
}
function pointIntersectsBufferedLine(p, line, radius) {
    const radiusSquared = radius * radius;
    if (line.length === 1)
        return p.distSqr(line[0]) < radiusSquared;
    for (let i = 1; i < line.length; i++) {
        const v = line[i - 1], w = line[i];
        if (distToSegmentSquared(p, v, w) < radiusSquared)
            return true;
    }
    return false;
}
function distToSegmentSquared(p, v, w) {
    const l2 = v.distSqr(w);
    if (l2 === 0)
        return p.distSqr(v);
    const t = ((p.x - v.x) * (w.x - v.x) + (p.y - v.y) * (w.y - v.y)) / l2;
    if (t < 0)
        return p.distSqr(v);
    if (t > 1)
        return p.distSqr(w);
    return p.distSqr(w.sub(v)._mult(t)._add(v));
}
function multiPolygonContainsPoint(rings, p) {
    let c = false, ring, p1, p2;
    for (let k = 0; k < rings.length; k++) {
        ring = rings[k];
        for (let i = 0, j = ring.length - 1; i < ring.length; j = i++) {
            p1 = ring[i];
            p2 = ring[j];
            if (p1.y > p.y !== p2.y > p.y && p.x < (p2.x - p1.x) * (p.y - p1.y) / (p2.y - p1.y) + p1.x) {
                c = !c;
            }
        }
    }
    return c;
}
function polygonContainsPoint(ring, p) {
    let c = false;
    for (let i = 0, j = ring.length - 1; i < ring.length; j = i++) {
        const p1 = ring[i];
        const p2 = ring[j];
        if (p1.y > p.y !== p2.y > p.y && p.x < (p2.x - p1.x) * (p.y - p1.y) / (p2.y - p1.y) + p1.x) {
            c = !c;
        }
    }
    return c;
}
function polygonIntersectsBox(ring, boxX1, boxY1, boxX2, boxY2) {
    for (const p of ring) {
        if (boxX1 <= p.x && boxY1 <= p.y && boxX2 >= p.x && boxY2 >= p.y)
            return true;
    }
    const corners = [
        new Point(boxX1, boxY1),
        new Point(boxX1, boxY2),
        new Point(boxX2, boxY2),
        new Point(boxX2, boxY1)
    ];
    if (ring.length > 2) {
        for (const corner of corners) {
            if (polygonContainsPoint(ring, corner))
                return true;
        }
    }
    for (let i = 0; i < ring.length - 1; i++) {
        const p1 = ring[i];
        const p2 = ring[i + 1];
        if (edgeIntersectsBox(p1, p2, corners))
            return true;
    }
    return false;
}
function edgeIntersectsBox(e1, e2, corners) {
    const tl = corners[0];
    const br = corners[2];
    if (e1.x < tl.x && e2.x < tl.x || e1.x > br.x && e2.x > br.x || e1.y < tl.y && e2.y < tl.y || e1.y > br.y && e2.y > br.y)
        return false;
    const dir = isCounterClockwise(e1, e2, corners[0]);
    return dir !== isCounterClockwise(e1, e2, corners[1]) || dir !== isCounterClockwise(e1, e2, corners[2]) || dir !== isCounterClockwise(e1, e2, corners[3]);
}
function triangleLeftSideOfEdge(a, b, p0, p1, p2, padding) {
    let nx = b.y - a.y;
    let ny = a.x - b.x;
    padding = padding || 0;
    if (padding) {
        const nLenSq = nx * nx + ny * ny;
        if (nLenSq === 0) {
            return true;
        }
        const len = Math.sqrt(nLenSq);
        nx /= len;
        ny /= len;
    }
    if ((p0.x - a.x) * nx + (p0.y - a.y) * ny - padding < 0) {
        return false;
    } else if ((p1.x - a.x) * nx + (p1.y - a.y) * ny - padding < 0) {
        return false;
    } else if ((p2.x - a.x) * nx + (p2.y - a.y) * ny - padding < 0) {
        return false;
    }
    return true;
}
function triangleIntersectsTriangle(a0, b0, c0, a1, b1, c1, padding) {
    if (triangleLeftSideOfEdge(a0, b0, a1, b1, c1, padding)) {
        return false;
    } else if (triangleLeftSideOfEdge(b0, c0, a1, b1, c1, padding)) {
        return false;
    } else if (triangleLeftSideOfEdge(c0, a0, a1, b1, c1, padding)) {
        return false;
    } else if (triangleLeftSideOfEdge(a1, b1, a0, b0, c0, padding)) {
        return false;
    } else if (triangleLeftSideOfEdge(b1, c1, a0, b0, c0, padding)) {
        return false;
    } else if (triangleLeftSideOfEdge(c1, a1, a0, b0, c0, padding)) {
        return false;
    }
    return true;
}

function getMaximumPaintValue(property, layer, bucket) {
    const value = layer.paint.get(property).value;
    if (value.kind === 'constant') {
        return value.value;
    } else {
        return bucket.programConfigurations.get(layer.id).getMaxValue(property);
    }
}
function translateDistance(translate2) {
    return Math.sqrt(translate2[0] * translate2[0] + translate2[1] * translate2[1]);
}
function translate(queryGeometry, translate2, translateAnchor, bearing, pixelsToTileUnits) {
    if (!translate2[0] && !translate2[1]) {
        return queryGeometry;
    }
    const pt = Point.convert(translate2)._mult(pixelsToTileUnits);
    if (translateAnchor === 'viewport') {
        pt._rotate(-bearing);
    }
    const translated = [];
    for (let i = 0; i < queryGeometry.length; i++) {
        const point = queryGeometry[i];
        translated.push(point.sub(pt));
    }
    return translated;
}
function tilespaceTranslate(translate2, translateAnchor, bearing, pixelsToTileUnits) {
    const pt = Point.convert(translate2)._mult(pixelsToTileUnits);
    if (translateAnchor === 'viewport') {
        pt._rotate(-bearing);
    }
    return pt;
}

let layout$e;
const getLayoutProperties$c = () => layout$e || (layout$e = new Properties({
    'circle-sort-key': new DataDrivenProperty(spec['layout_circle']['circle-sort-key']),
    'circle-elevation-reference': new DataConstantProperty(spec['layout_circle']['circle-elevation-reference']),
    'visibility': new DataConstantProperty(spec['layout_circle']['visibility'])
}));
let paint$d;
const getPaintProperties$d = () => paint$d || (paint$d = new Properties({
    'circle-radius': new DataDrivenProperty(spec['paint_circle']['circle-radius']),
    'circle-color': new DataDrivenProperty(spec['paint_circle']['circle-color']),
    'circle-blur': new DataDrivenProperty(spec['paint_circle']['circle-blur']),
    'circle-opacity': new DataDrivenProperty(spec['paint_circle']['circle-opacity']),
    'circle-translate': new DataConstantProperty(spec['paint_circle']['circle-translate']),
    'circle-translate-anchor': new DataConstantProperty(spec['paint_circle']['circle-translate-anchor']),
    'circle-pitch-scale': new DataConstantProperty(spec['paint_circle']['circle-pitch-scale']),
    'circle-pitch-alignment': new DataConstantProperty(spec['paint_circle']['circle-pitch-alignment']),
    'circle-stroke-width': new DataDrivenProperty(spec['paint_circle']['circle-stroke-width']),
    'circle-stroke-color': new DataDrivenProperty(spec['paint_circle']['circle-stroke-color']),
    'circle-stroke-opacity': new DataDrivenProperty(spec['paint_circle']['circle-stroke-opacity']),
    'circle-emissive-strength': new DataConstantProperty(spec['paint_circle']['circle-emissive-strength']),
    'circle-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    }),
    'circle-stroke-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    })
}));

var whootsJs = {exports: {}};

var hasRequiredWhootsJs;

function requireWhootsJs () {
	if (hasRequiredWhootsJs) return whootsJs.exports;
	hasRequiredWhootsJs = 1;
	(function (module, exports) {
		(function (global, factory) {
		    factory(exports) ;
		}(this, function (exports) {
		    /**
		 * getURL
		 *
		 * @param    {String}  baseUrl  Base url of the WMS server
		 * @param    {String}  layer    Layer name
		 * @param    {Number}  x        Tile coordinate x
		 * @param    {Number}  y        Tile coordinate y
		 * @param    {Number}  z        Tile zoom
		 * @param    {Object}  [options]
		 * @param    {String}  [options.format='image/png']
		 * @param    {String}  [options.service='WMS']
		 * @param    {String}  [options.version='1.1.1']
		 * @param    {String}  [options.request='GetMap']
		 * @param    {String}  [options.srs='EPSG:3857']
		 * @param    {Number}  [options.width='256']
		 * @param    {Number}  [options.height='256']
		 * @returns  {String}  url
		 * @example
		 * var baseUrl = 'http://geodata.state.nj.us/imagerywms/Natural2015';
		 * var layer = 'Natural2015';
		 * var url = whoots.getURL(baseUrl, layer, 154308, 197167, 19);
		 */
		    function getURL(baseUrl, layer, x, y, z, options) {
		        options = options || {};
		        var url = baseUrl + '?' + [
		            'bbox=' + getTileBBox(x, y, z),
		            'format=' + (options.format || 'image/png'),
		            'service=' + (options.service || 'WMS'),
		            'version=' + (options.version || '1.1.1'),
		            'request=' + (options.request || 'GetMap'),
		            'srs=' + (options.srs || 'EPSG:3857'),
		            'width=' + (options.width || 256),
		            'height=' + (options.height || 256),
		            'layers=' + layer
		        ].join('&');
		        return url;
		    }
		    /**
		 * getTileBBox
		 *
		 * @param    {Number}  x  Tile coordinate x
		 * @param    {Number}  y  Tile coordinate y
		 * @param    {Number}  z  Tile zoom
		 * @returns  {String}  String of the bounding box
		 */
		    function getTileBBox(x, y, z) {
		        // for Google/OSM tile scheme we need to alter the y
		        y = Math.pow(2, z) - y - 1;
		        var min = getMercCoords(x * 256, y * 256, z), max = getMercCoords((x + 1) * 256, (y + 1) * 256, z);
		        return min[0] + ',' + min[1] + ',' + max[0] + ',' + max[1];
		    }
		    /**
		 * getMercCoords
		 *
		 * @param    {Number}  x  Pixel coordinate x
		 * @param    {Number}  y  Pixel coordinate y
		 * @param    {Number}  z  Tile zoom
		 * @returns  {Array}   [x, y]
		 */
		    function getMercCoords(x, y, z) {
		        var resolution = 2 * Math.PI * 6378137 / 256 / Math.pow(2, z), merc_x = x * resolution - 2 * Math.PI * 6378137 / 2, merc_y = y * resolution - 2 * Math.PI * 6378137 / 2;
		        return [
		            merc_x,
		            merc_y
		        ];
		    }
		    exports.getURL = getURL;
		    exports.getTileBBox = getTileBBox;
		    exports.getMercCoords = getMercCoords;
		    Object.defineProperty(exports, '__esModule', { value: true });
		})); 
	} (whootsJs, whootsJs.exports));
	return whootsJs.exports;
}

var whootsJsExports = requireWhootsJs();

class CanonicalTileID {
    constructor(z, x, y) {
        this.z = z;
        this.x = x;
        this.y = y;
        this.key = calculateKey(0, z, z, x, y);
    }
    equals(id) {
        return this.z === id.z && this.x === id.x && this.y === id.y;
    }
    // given a list of urls, choose a url template and return a tile URL
    url(urls, scheme) {
        const bbox = whootsJsExports.getTileBBox(this.x, this.y, this.z);
        const quadkey = getQuadkey(this.z, this.x, this.y);
        return urls[(this.x + this.y) % urls.length].replace('{prefix}', (this.x % 16).toString(16) + (this.y % 16).toString(16)).replace(/{z}/g, String(this.z)).replace(/{x}/g, String(this.x)).replace(/{y}/g, String(scheme === 'tms' ? Math.pow(2, this.z) - this.y - 1 : this.y)).replace('{quadkey}', quadkey).replace('{bbox-epsg-3857}', bbox);
    }
    toString() {
        return `${ this.z }/${ this.x }/${ this.y }`;
    }
}
class UnwrappedTileID {
    constructor(wrap, canonical) {
        this.wrap = wrap;
        this.canonical = canonical;
        this.key = calculateKey(wrap, canonical.z, canonical.z, canonical.x, canonical.y);
    }
}
class OverscaledTileID {
    constructor(overscaledZ, wrap, z, x, y) {
        this.overscaledZ = overscaledZ;
        this.wrap = wrap;
        this.canonical = new CanonicalTileID(z, +x, +y);
        this.key = wrap === 0 && overscaledZ === z ? this.canonical.key : calculateKey(wrap, overscaledZ, z, x, y);
    }
    equals(id) {
        return this.overscaledZ === id.overscaledZ && this.wrap === id.wrap && this.canonical.equals(id.canonical);
    }
    scaledTo(targetZ) {
        const zDifference = this.canonical.z - targetZ;
        if (targetZ > this.canonical.z) {
            return new OverscaledTileID(targetZ, this.wrap, this.canonical.z, this.canonical.x, this.canonical.y);
        } else {
            return new OverscaledTileID(targetZ, this.wrap, targetZ, this.canonical.x >> zDifference, this.canonical.y >> zDifference);
        }
    }
    /*
   * calculateScaledKey is an optimization:
   * when withWrap == true, implements the same as this.scaledTo(z).key,
   * when withWrap == false, implements the same as this.scaledTo(z).wrapped().key.
   */
    calculateScaledKey(targetZ, withWrap = true) {
        if (this.overscaledZ === targetZ && withWrap)
            return this.key;
        if (targetZ > this.canonical.z) {
            return calculateKey(this.wrap * +withWrap, targetZ, this.canonical.z, this.canonical.x, this.canonical.y);
        } else {
            const zDifference = this.canonical.z - targetZ;
            return calculateKey(this.wrap * +withWrap, targetZ, targetZ, this.canonical.x >> zDifference, this.canonical.y >> zDifference);
        }
    }
    isChildOf(parent) {
        if (parent.wrap !== this.wrap) {
            return false;
        }
        const zDifference = this.canonical.z - parent.canonical.z;
        return parent.overscaledZ === 0 || parent.overscaledZ < this.overscaledZ && parent.canonical.z < this.canonical.z && parent.canonical.x === this.canonical.x >> zDifference && parent.canonical.y === this.canonical.y >> zDifference;
    }
    children(sourceMaxZoom) {
        if (this.overscaledZ >= sourceMaxZoom) {
            return [new OverscaledTileID(this.overscaledZ + 1, this.wrap, this.canonical.z, this.canonical.x, this.canonical.y)];
        }
        const z = this.canonical.z + 1;
        const x = this.canonical.x * 2;
        const y = this.canonical.y * 2;
        return [
            new OverscaledTileID(z, this.wrap, z, x, y),
            new OverscaledTileID(z, this.wrap, z, x + 1, y),
            new OverscaledTileID(z, this.wrap, z, x, y + 1),
            new OverscaledTileID(z, this.wrap, z, x + 1, y + 1)
        ];
    }
    isLessThan(rhs) {
        if (this.wrap < rhs.wrap)
            return true;
        if (this.wrap > rhs.wrap)
            return false;
        if (this.overscaledZ < rhs.overscaledZ)
            return true;
        if (this.overscaledZ > rhs.overscaledZ)
            return false;
        if (this.canonical.x < rhs.canonical.x)
            return true;
        if (this.canonical.x > rhs.canonical.x)
            return false;
        if (this.canonical.y < rhs.canonical.y)
            return true;
        return false;
    }
    wrapped() {
        return new OverscaledTileID(this.overscaledZ, 0, this.canonical.z, this.canonical.x, this.canonical.y);
    }
    unwrapTo(wrap) {
        return new OverscaledTileID(this.overscaledZ, wrap, this.canonical.z, this.canonical.x, this.canonical.y);
    }
    overscaleFactor() {
        return Math.pow(2, this.overscaledZ - this.canonical.z);
    }
    toUnwrapped() {
        return new UnwrappedTileID(this.wrap, this.canonical);
    }
    toString() {
        return `${ this.overscaledZ }/${ this.canonical.x }/${ this.canonical.y }`;
    }
}
function calculateKey(wrap, overscaledZ, z, x, y) {
    const dim = 1 << Math.min(z, 22);
    let xy = dim * (y % dim) + x % dim;
    if (wrap && z < 22) {
        const bitsAvailable = 2 * (22 - z);
        xy += dim * dim * ((wrap < 0 ? -2 * wrap - 1 : 2 * wrap) % (1 << bitsAvailable));
    }
    const key = (xy * 32 + z) * 16 + (overscaledZ - z);
    return key;
}
function getQuadkey(z, x, y) {
    let quadkey = '', mask;
    for (let i = z; i > 0; i--) {
        mask = 1 << i - 1;
        quadkey += (x & mask ? 1 : 0) + (y & mask ? 2 : 0);
    }
    return quadkey;
}
const neighborCoord = [
    coord => {
        let x = coord.canonical.x - 1;
        let w = coord.wrap;
        if (x < 0) {
            x = (1 << coord.canonical.z) - 1;
            w--;
        }
        return new OverscaledTileID(coord.overscaledZ, w, coord.canonical.z, x, coord.canonical.y);
    },
    coord => {
        let x = coord.canonical.x + 1;
        let w = coord.wrap;
        if (x === 1 << coord.canonical.z) {
            x = 0;
            w++;
        }
        return new OverscaledTileID(coord.overscaledZ, w, coord.canonical.z, x, coord.canonical.y);
    },
    coord => new OverscaledTileID(coord.overscaledZ, coord.wrap, coord.canonical.z, coord.canonical.x, (coord.canonical.y === 0 ? 1 << coord.canonical.z : coord.canonical.y) - 1),
    coord => new OverscaledTileID(coord.overscaledZ, coord.wrap, coord.canonical.z, coord.canonical.x, coord.canonical.y === (1 << coord.canonical.z) - 1 ? 0 : coord.canonical.y + 1)
];
register(CanonicalTileID, 'CanonicalTileID');
register(OverscaledTileID, 'OverscaledTileID', {
    omit: [
        'projMatrix',
        'expandedProjMatrix'
    ]
});

const layout$d = createLayout([
    {
        type: 'Float32',
        name: 'a_globe_pos',
        components: 3
    },
    {
        type: 'Float32',
        name: 'a_uv',
        components: 2
    }
]);
const {members: members$5, size: size$5, alignment: alignment$5} = layout$d;

const posAttributesGlobeExt = createLayout([{
        name: 'a_pos_3',
        components: 3,
        type: 'Int16'
    }]);
var posAttributes = createLayout([{
        name: 'a_pos',
        type: 'Int16',
        components: 2
    }]);

class Ray2D {
    constructor(pos_, dir_) {
        this.pos = pos_;
        this.dir = dir_;
    }
    intersectsPlane(pt, normal, out) {
        const D = cjsExports.vec2.dot(normal, this.dir);
        if (Math.abs(D) < 0.000001) {
            return false;
        }
        const t = ((pt[0] - this.pos[0]) * normal[0] + (pt[1] - this.pos[1]) * normal[1]) / D;
        out[0] = this.pos[0] + this.dir[0] * t;
        out[1] = this.pos[1] + this.dir[1] * t;
        return true;
    }
}
class Ray {
    constructor(pos_, dir_) {
        this.pos = pos_;
        this.dir = dir_;
    }
    intersectsPlane(pt, normal, out) {
        const D = cjsExports.vec3.dot(normal, this.dir);
        if (Math.abs(D) < 0.000001) {
            return false;
        }
        const t = ((pt[0] - this.pos[0]) * normal[0] + (pt[1] - this.pos[1]) * normal[1] + (pt[2] - this.pos[2]) * normal[2]) / D;
        out[0] = this.pos[0] + this.dir[0] * t;
        out[1] = this.pos[1] + this.dir[1] * t;
        out[2] = this.pos[2] + this.dir[2] * t;
        return true;
    }
    closestPointOnSphere(center, r, out) {
        if (cjsExports.vec3.equals(this.pos, center) || r === 0) {
            out[0] = out[1] = out[2] = 0;
            return false;
        }
        const [dx, dy, dz] = this.dir;
        const px = this.pos[0] - center[0];
        const py = this.pos[1] - center[1];
        const pz = this.pos[2] - center[2];
        const a = dx * dx + dy * dy + dz * dz;
        const b = 2 * (px * dx + py * dy + pz * dz);
        const c = px * px + py * py + pz * pz - r * r;
        const d = b * b - 4 * a * c;
        if (d < 0) {
            const t = Math.max(-b / 2, 0);
            const gx = px + dx * t;
            const gy = py + dy * t;
            const gz = pz + dz * t;
            const glen = Math.hypot(gx, gy, gz);
            out[0] = gx * r / glen;
            out[1] = gy * r / glen;
            out[2] = gz * r / glen;
            return false;
        } else {
            const t = (-b - Math.sqrt(d)) / (2 * a);
            if (t < 0) {
                const plen = Math.hypot(px, py, pz);
                out[0] = px * r / plen;
                out[1] = py * r / plen;
                out[2] = pz * r / plen;
                return false;
            } else {
                out[0] = px + dx * t;
                out[1] = py + dy * t;
                out[2] = pz + dz * t;
                return true;
            }
        }
    }
}
class FrustumCorners {
    constructor(TL_, TR_, BR_, BL_, horizon_) {
        this.TL = TL_;
        this.TR = TR_;
        this.BR = BR_;
        this.BL = BL_;
        this.horizon = horizon_;
    }
    static fromInvProjectionMatrix(invProj, horizonFromTop, viewportHeight) {
        const TLClip = [
            -1,
            1,
            1
        ];
        const TRClip = [
            1,
            1,
            1
        ];
        const BRClip = [
            1,
            -1,
            1
        ];
        const BLClip = [
            -1,
            -1,
            1
        ];
        const TL = cjsExports.vec3.transformMat4(TLClip, TLClip, invProj);
        const TR = cjsExports.vec3.transformMat4(TRClip, TRClip, invProj);
        const BR = cjsExports.vec3.transformMat4(BRClip, BRClip, invProj);
        const BL = cjsExports.vec3.transformMat4(BLClip, BLClip, invProj);
        return new FrustumCorners(TL, TR, BR, BL, horizonFromTop / viewportHeight);
    }
}
function projectPoints(points, origin, axis) {
    let min = Infinity;
    let max = -Infinity;
    const vec = [];
    for (const point of points) {
        cjsExports.vec3.sub(vec, point, origin);
        const projection = cjsExports.vec3.dot(vec, axis);
        min = Math.min(min, projection);
        max = Math.max(max, projection);
    }
    return [
        min,
        max
    ];
}
function intersectsFrustum(frustum, aabbPoints) {
    let fullyInside = true;
    for (let p = 0; p < frustum.planes.length; p++) {
        const plane = frustum.planes[p];
        let pointsInside = 0;
        for (let i = 0; i < aabbPoints.length; i++) {
            pointsInside += cjsExports.vec3.dot(plane, aabbPoints[i]) + plane[3] >= 0;
        }
        if (pointsInside === 0)
            return 0;
        if (pointsInside !== aabbPoints.length)
            fullyInside = false;
    }
    return fullyInside ? 2 : 1;
}
function intersectsFrustumPrecise(frustum, aabbPoints) {
    for (const proj of frustum.projections) {
        const projectedAabb = projectPoints(aabbPoints, frustum.points[0], proj.axis);
        if (proj.projection[1] < projectedAabb[0] || proj.projection[0] > projectedAabb[1]) {
            return 0;
        }
    }
    return 1;
}
const NEAR_TL = 0;
const NEAR_TR = 1;
const NEAR_BR = 2;
const NEAR_BL = 3;
const FAR_TL = 4;
const FAR_TR = 5;
const FAR_BR = 6;
const FAR_BL = 7;
function pointsInsideOfPlane(points, plane) {
    let pointsInside = 0;
    const p = [
        0,
        0,
        0,
        0
    ];
    for (let i = 0; i < points.length; i++) {
        p[0] = points[i][0];
        p[1] = points[i][1];
        p[2] = points[i][2];
        p[3] = 1;
        if (cjsExports.vec4.dot(p, plane) >= 0) {
            pointsInside++;
        }
    }
    return pointsInside;
}
class Frustum {
    constructor(points_, planes_) {
        this.points = points_ || new Array(8).fill([
            0,
            0,
            0
        ]);
        this.planes = planes_ || new Array(6).fill([
            0,
            0,
            0,
            0
        ]);
        this.bounds = Aabb.fromPoints(this.points);
        this.projections = [];
        this.frustumEdges = [
            cjsExports.vec3.sub([], this.points[NEAR_BR], this.points[NEAR_BL]),
            cjsExports.vec3.sub([], this.points[NEAR_TL], this.points[NEAR_BL]),
            cjsExports.vec3.sub([], this.points[FAR_TL], this.points[NEAR_TL]),
            cjsExports.vec3.sub([], this.points[FAR_TR], this.points[NEAR_TR]),
            cjsExports.vec3.sub([], this.points[FAR_BR], this.points[NEAR_BR]),
            cjsExports.vec3.sub([], this.points[FAR_BL], this.points[NEAR_BL])
        ];
        for (const edge of this.frustumEdges) {
            const axis0 = [
                0,
                -edge[2],
                edge[1]
            ];
            const axis1 = [
                edge[2],
                0,
                -edge[0]
            ];
            this.projections.push({
                axis: axis0,
                projection: projectPoints(this.points, this.points[0], axis0)
            });
            this.projections.push({
                axis: axis1,
                projection: projectPoints(this.points, this.points[0], axis1)
            });
        }
    }
    static fromInvProjectionMatrix(invProj, worldSize, zoom, zInMeters) {
        const clipSpaceCorners = [
            [
                -1,
                1,
                -1,
                1
            ],
            [
                1,
                1,
                -1,
                1
            ],
            [
                1,
                -1,
                -1,
                1
            ],
            [
                -1,
                -1,
                -1,
                1
            ],
            [
                -1,
                1,
                1,
                1
            ],
            [
                1,
                1,
                1,
                1
            ],
            [
                1,
                -1,
                1,
                1
            ],
            [
                -1,
                -1,
                1,
                1
            ]
        ];
        const scale = Math.pow(2, zoom);
        const frustumCoords = clipSpaceCorners.map(v => {
            const s = cjsExports.vec4.transformMat4([], v, invProj);
            const k = 1 / s[3] / worldSize * scale;
            return cjsExports.vec4.mul(s, s, [
                k,
                k,
                zInMeters ? 1 / s[3] : k,
                k
            ]);
        });
        const frustumPlanePointIndices = [
            [
                NEAR_TL,
                NEAR_TR,
                NEAR_BR
            ],
            // near
            [
                FAR_BR,
                FAR_TR,
                FAR_TL
            ],
            // far
            [
                NEAR_TL,
                NEAR_BL,
                FAR_BL
            ],
            // left
            [
                NEAR_BR,
                NEAR_TR,
                FAR_TR
            ],
            // right
            [
                NEAR_BL,
                NEAR_BR,
                FAR_BR
            ],
            // bottom
            [
                NEAR_TL,
                FAR_TL,
                FAR_TR
            ]    // top
        ];
        const frustumPlanes = frustumPlanePointIndices.map(p => {
            const a = cjsExports.vec3.sub([], frustumCoords[p[0]], frustumCoords[p[1]]);
            const b = cjsExports.vec3.sub([], frustumCoords[p[2]], frustumCoords[p[1]]);
            const n = cjsExports.vec3.normalize([], cjsExports.vec3.cross([], a, b));
            const d = -cjsExports.vec3.dot(n, frustumCoords[p[1]]);
            return n.concat(d);
        });
        const frustumPoints = [];
        for (let i = 0; i < frustumCoords.length; i++) {
            frustumPoints.push([
                frustumCoords[i][0],
                frustumCoords[i][1],
                frustumCoords[i][2]
            ]);
        }
        return new Frustum(frustumPoints, frustumPlanes);
    }
    // Performs precise intersection test between the frustum and the provided convex hull.
    // The hull consits of vertices, faces (defined as planes) and a list of edges.
    // Intersection test is performed using separating axis theoreom.
    intersectsPrecise(vertices, faces, edges) {
        for (let i = 0; i < faces.length; i++) {
            if (!pointsInsideOfPlane(vertices, faces[i])) {
                return 0;
            }
        }
        for (let i = 0; i < this.planes.length; i++) {
            if (!pointsInsideOfPlane(vertices, this.planes[i])) {
                return 0;
            }
        }
        for (const edge of edges) {
            for (const frustumEdge of this.frustumEdges) {
                const axis = cjsExports.vec3.cross([], edge, frustumEdge);
                const len = cjsExports.vec3.length(axis);
                if (len === 0) {
                    continue;
                }
                cjsExports.vec3.scale(axis, axis, 1 / len);
                const projA = projectPoints(this.points, this.points[0], axis);
                const projB = projectPoints(vertices, this.points[0], axis);
                if (projA[0] > projB[1] || projB[0] > projA[1]) {
                    return 0;
                }
            }
        }
        return 1;
    }
    containsPoint(point) {
        for (const plane of this.planes) {
            const normal = [
                plane[0],
                plane[1],
                plane[2]
            ];
            const distance = plane[3];
            if (cjsExports.vec3.dot(normal, point) + distance < 0) {
                return false;
            }
        }
        return true;
    }
}
class Aabb {
    static fromPoints(points) {
        const min = [
            Infinity,
            Infinity,
            Infinity
        ];
        const max = [
            -Infinity,
            -Infinity,
            -Infinity
        ];
        for (const p of points) {
            cjsExports.vec3.min(min, min, p);
            cjsExports.vec3.max(max, max, p);
        }
        return new Aabb(min, max);
    }
    static fromTileIdAndHeight(id, minHeight, maxHeight) {
        const tiles = 1 << id.canonical.z;
        const x = id.canonical.x;
        const y = id.canonical.y;
        return new Aabb([
            x / tiles,
            y / tiles,
            minHeight
        ], [
            (x + 1) / tiles,
            (y + 1) / tiles,
            maxHeight
        ]);
    }
    static applyTransform(aabb, transform) {
        const corners = aabb.getCorners();
        for (let i = 0; i < corners.length; ++i) {
            cjsExports.vec3.transformMat4(corners[i], corners[i], transform);
        }
        return Aabb.fromPoints(corners);
    }
    // A fast version of applyTransform. Note that it breaks down for non-uniform
    // scale and complex projection matrices.
    static applyTransformFast(aabb, transform) {
        const min = [
            transform[12],
            transform[13],
            transform[14]
        ];
        const max = [...min];
        for (let i = 0; i < 3; i++) {
            for (let j = 0; j < 3; j++) {
                const value = transform[j * 4 + i];
                const a = value * aabb.min[j];
                const b = value * aabb.max[j];
                min[i] += Math.min(a, b);
                max[i] += Math.max(a, b);
            }
        }
        return new Aabb(min, max);
    }
    static projectAabbCorners(aabb, transform) {
        const corners = aabb.getCorners();
        for (let i = 0; i < corners.length; ++i) {
            cjsExports.vec3.transformMat4(corners[i], corners[i], transform);
        }
        return corners;
    }
    constructor(min_, max_) {
        this.min = min_;
        this.max = max_;
        this.center = cjsExports.vec3.scale([], cjsExports.vec3.add([], this.min, this.max), 0.5);
    }
    quadrant(index) {
        const split = [
            index % 2 === 0,
            index < 2
        ];
        const qMin = cjsExports.vec3.clone(this.min);
        const qMax = cjsExports.vec3.clone(this.max);
        for (let axis = 0; axis < split.length; axis++) {
            qMin[axis] = split[axis] ? this.min[axis] : this.center[axis];
            qMax[axis] = split[axis] ? this.center[axis] : this.max[axis];
        }
        qMax[2] = this.max[2];
        return new Aabb(qMin, qMax);
    }
    distanceX(point) {
        const pointOnAabb = Math.max(Math.min(this.max[0], point[0]), this.min[0]);
        return pointOnAabb - point[0];
    }
    distanceY(point) {
        const pointOnAabb = Math.max(Math.min(this.max[1], point[1]), this.min[1]);
        return pointOnAabb - point[1];
    }
    distanceZ(point) {
        const pointOnAabb = Math.max(Math.min(this.max[2], point[2]), this.min[2]);
        return pointOnAabb - point[2];
    }
    getCorners() {
        const mn = this.min;
        const mx = this.max;
        return [
            [
                mn[0],
                mn[1],
                mn[2]
            ],
            [
                mx[0],
                mn[1],
                mn[2]
            ],
            [
                mx[0],
                mx[1],
                mn[2]
            ],
            [
                mn[0],
                mx[1],
                mn[2]
            ],
            [
                mn[0],
                mn[1],
                mx[2]
            ],
            [
                mx[0],
                mn[1],
                mx[2]
            ],
            [
                mx[0],
                mx[1],
                mx[2]
            ],
            [
                mn[0],
                mx[1],
                mx[2]
            ]
        ];
    }
    // Performs conservative intersection test using separating axis theorem.
    // Some accuracy is traded for better performance. False positive rate is < 1%.
    // Flat intersection test checks only x and y dimensions of the aabb.
    // Returns 0 if there's no intersection, 1 if shapes are intersecting and
    // 2 if the aabb if fully inside the frustum.
    intersects(frustum) {
        if (!this.intersectsAabb(frustum.bounds)) {
            return 0;
        }
        return intersectsFrustum(frustum, this.getCorners());
    }
    intersectsFlat(frustum) {
        if (!this.intersectsAabb(frustum.bounds)) {
            return 0;
        }
        const aabbPoints = [
            [
                this.min[0],
                this.min[1],
                0
            ],
            [
                this.max[0],
                this.min[1],
                0
            ],
            [
                this.max[0],
                this.max[1],
                0
            ],
            [
                this.min[0],
                this.max[1],
                0
            ]
        ];
        return intersectsFrustum(frustum, aabbPoints);
    }
    // Performs precise intersection test using separating axis theorem.
    // It is possible run only edge cases that were not covered in intersects().
    // Flat intersection test checks only x and y dimensions of the aabb.
    intersectsPrecise(frustum, edgeCasesOnly) {
        if (!edgeCasesOnly) {
            const intersects = this.intersects(frustum);
            if (!intersects) {
                return 0;
            }
        }
        return intersectsFrustumPrecise(frustum, this.getCorners());
    }
    intersectsPreciseFlat(frustum, edgeCasesOnly) {
        if (!edgeCasesOnly) {
            const intersects = this.intersectsFlat(frustum);
            if (!intersects) {
                return 0;
            }
        }
        const aabbPoints = [
            [
                this.min[0],
                this.min[1],
                0
            ],
            [
                this.max[0],
                this.min[1],
                0
            ],
            [
                this.max[0],
                this.max[1],
                0
            ],
            [
                this.min[0],
                this.max[1],
                0
            ]
        ];
        return intersectsFrustumPrecise(frustum, aabbPoints);
    }
    intersectsAabb(aabb) {
        for (let axis = 0; axis < 3; ++axis) {
            if (this.min[axis] > aabb.max[axis] || aabb.min[axis] > this.max[axis]) {
                return false;
            }
        }
        return true;
    }
    intersectsAabbXY(aabb) {
        if (this.min[0] > aabb.max[0] || aabb.min[0] > this.max[0]) {
            return false;
        }
        if (this.min[1] > aabb.max[1] || aabb.min[1] > this.max[1]) {
            return false;
        }
        return true;
    }
    encapsulate(aabb) {
        for (let i = 0; i < 3; i++) {
            this.min[i] = Math.min(this.min[i], aabb.min[i]);
            this.max[i] = Math.max(this.max[i], aabb.max[i]);
        }
    }
    encapsulatePoint(point) {
        for (let i = 0; i < 3; i++) {
            this.min[i] = Math.min(this.min[i], point[i]);
            this.max[i] = Math.max(this.max[i], point[i]);
        }
    }
    closestPoint(point) {
        return [
            Math.max(Math.min(this.max[0], point[0]), this.min[0]),
            Math.max(Math.min(this.max[1], point[1]), this.min[1]),
            Math.max(Math.min(this.max[2], point[2]), this.min[2])
        ];
    }
}
register(Aabb, 'Aabb');

function globeMetersToEcef(d) {
    return d * GLOBE_RADIUS / earthRadius;
}
const GLOBE_LOW_ZOOM_TILE_AABBS = [
    // z == 0
    new Aabb([
        GLOBE_MIN,
        GLOBE_MIN,
        GLOBE_MIN
    ], [
        GLOBE_MAX,
        GLOBE_MAX,
        GLOBE_MAX
    ]),
    // z == 1
    new Aabb([
        GLOBE_MIN,
        GLOBE_MIN,
        GLOBE_MIN
    ], [
        0,
        0,
        GLOBE_MAX
    ]),
    // x=0, y=0
    new Aabb([
        0,
        GLOBE_MIN,
        GLOBE_MIN
    ], [
        GLOBE_MAX,
        0,
        GLOBE_MAX
    ]),
    // x=1, y=0
    new Aabb([
        GLOBE_MIN,
        0,
        GLOBE_MIN
    ], [
        0,
        GLOBE_MAX,
        GLOBE_MAX
    ]),
    // x=0, y=1
    new Aabb([
        0,
        0,
        GLOBE_MIN
    ], [
        GLOBE_MAX,
        GLOBE_MAX,
        GLOBE_MAX
    ])    // x=1, y=1
];
function globePointCoordinate(tr, x, y, clampToHorizon = true) {
    const point0 = cjsExports.vec3.scale([], tr._camera.position, tr.worldSize);
    const point1 = [
        x,
        y,
        1,
        1
    ];
    cjsExports.vec4.transformMat4(point1, point1, tr.pixelMatrixInverse);
    cjsExports.vec4.scale(point1, point1, 1 / point1[3]);
    const p0p1 = cjsExports.vec3.sub([], point1, point0);
    const dir = cjsExports.vec3.normalize([], p0p1);
    const m = tr.globeMatrix;
    const globeCenter = [
        m[12],
        m[13],
        m[14]
    ];
    const p0toCenter = cjsExports.vec3.sub([], globeCenter, point0);
    const p0toCenterDist = cjsExports.vec3.length(p0toCenter);
    const centerDir = cjsExports.vec3.normalize([], p0toCenter);
    const radius = tr.worldSize / (2 * Math.PI);
    const cosAngle = cjsExports.vec3.dot(centerDir, dir);
    const origoTangentAngle = Math.asin(radius / p0toCenterDist);
    const origoDirAngle = Math.acos(cosAngle);
    if (origoTangentAngle < origoDirAngle) {
        if (!clampToHorizon)
            return null;
        const clampedP1 = [];
        const origoToP1 = [];
        cjsExports.vec3.scale(clampedP1, dir, p0toCenterDist / cosAngle);
        cjsExports.vec3.normalize(origoToP1, cjsExports.vec3.sub(origoToP1, clampedP1, p0toCenter));
        cjsExports.vec3.normalize(dir, cjsExports.vec3.add(dir, p0toCenter, cjsExports.vec3.scale(dir, origoToP1, Math.tan(origoTangentAngle) * p0toCenterDist)));
    }
    const pointOnGlobe = [];
    const ray = new Ray(point0, dir);
    ray.closestPointOnSphere(globeCenter, radius, pointOnGlobe);
    const xa = cjsExports.vec3.normalize([], getColumn(m, 0));
    const ya = cjsExports.vec3.normalize([], getColumn(m, 1));
    const za = cjsExports.vec3.normalize([], getColumn(m, 2));
    const xp = cjsExports.vec3.dot(xa, pointOnGlobe);
    const yp = cjsExports.vec3.dot(ya, pointOnGlobe);
    const zp = cjsExports.vec3.dot(za, pointOnGlobe);
    const lat = radToDeg(Math.asin(-yp / radius));
    let lng = radToDeg(Math.atan2(xp, zp));
    lng = tr.center.lng + shortestAngle(tr.center.lng, lng);
    const mx = mercatorXfromLng(lng);
    const my = clamp(mercatorYfromLat(lat), 0, 1);
    return new MercatorCoordinate(mx, my);
}
class Arc {
    constructor(p0, p1, center) {
        this.a = cjsExports.vec3.sub([], p0, center);
        this.b = cjsExports.vec3.sub([], p1, center);
        this.center = center;
        const an = cjsExports.vec3.normalize([], this.a);
        const bn = cjsExports.vec3.normalize([], this.b);
        this.angle = Math.acos(cjsExports.vec3.dot(an, bn));
    }
}
function slerp(a, b, angle, t) {
    const sina = Math.sin(angle);
    return a * (Math.sin((1 - t) * angle) / sina) + b * (Math.sin(t * angle) / sina);
}
function localExtremum(arc, dim) {
    if (arc.angle === 0) {
        return null;
    }
    let t;
    if (arc.a[dim] === 0) {
        t = 1 / arc.angle * 0.5 * Math.PI;
    } else {
        t = 1 / arc.angle * Math.atan(arc.b[dim] / arc.a[dim] / Math.sin(arc.angle) - 1 / Math.tan(arc.angle));
    }
    if (t < 0 || t > 1) {
        return null;
    }
    return slerp(arc.a[dim], arc.b[dim], arc.angle, clamp(t, 0, 1)) + arc.center[dim];
}
function globeTileBounds(id) {
    if (id.z <= 1) {
        return GLOBE_LOW_ZOOM_TILE_AABBS[id.z + id.y * 2 + id.x];
    }
    const bounds = tileCornersToBounds(id);
    const corners = boundsToECEF(bounds);
    return Aabb.fromPoints(corners);
}
function interpolateVec3(from, to, phase) {
    cjsExports.vec3.scale(from, from, 1 - phase);
    return cjsExports.vec3.scaleAndAdd(from, from, to, phase);
}
function transitionTileAABBinECEF(id, tr) {
    const phase = globeToMercatorTransition(tr.zoom);
    if (phase === 0) {
        return globeTileBounds(id);
    }
    const bounds = tileCornersToBounds(id);
    const corners = boundsToECEF(bounds);
    const w = mercatorXfromLng(bounds.getWest()) * tr.worldSize;
    const e = mercatorXfromLng(bounds.getEast()) * tr.worldSize;
    const n = mercatorYfromLat(bounds.getNorth()) * tr.worldSize;
    const s = mercatorYfromLat(bounds.getSouth()) * tr.worldSize;
    const nw = [
        w,
        n,
        0
    ];
    const ne = [
        e,
        n,
        0
    ];
    const sw = [
        w,
        s,
        0
    ];
    const se = [
        e,
        s,
        0
    ];
    const worldToECEFMatrix = cjsExports.mat4.invert([], tr.globeMatrix);
    cjsExports.vec3.transformMat4(nw, nw, worldToECEFMatrix);
    cjsExports.vec3.transformMat4(ne, ne, worldToECEFMatrix);
    cjsExports.vec3.transformMat4(sw, sw, worldToECEFMatrix);
    cjsExports.vec3.transformMat4(se, se, worldToECEFMatrix);
    corners[0] = interpolateVec3(corners[0], sw, phase);
    corners[1] = interpolateVec3(corners[1], se, phase);
    corners[2] = interpolateVec3(corners[2], ne, phase);
    corners[3] = interpolateVec3(corners[3], nw, phase);
    return Aabb.fromPoints(corners);
}
function transformPoints(corners, globeMatrix, scale) {
    for (const corner of corners) {
        cjsExports.vec3.transformMat4(corner, corner, globeMatrix);
        cjsExports.vec3.scale(corner, corner, scale);
    }
}
function aabbForTileOnGlobe(tr, numTiles, tileId, extendToPoles) {
    const scale = numTiles / tr.worldSize;
    const m = tr.globeMatrix;
    if (tileId.z <= 1) {
        const corners2 = globeTileBounds(tileId).getCorners();
        transformPoints(corners2, m, scale);
        return Aabb.fromPoints(corners2);
    }
    const bounds = tileCornersToBounds(tileId, extendToPoles);
    const corners = boundsToECEF(bounds, GLOBE_RADIUS + globeMetersToEcef(tr._tileCoverLift));
    transformPoints(corners, m, scale);
    const mx = Number.MAX_VALUE;
    const cornerMax = [
        -mx,
        -mx,
        -mx
    ];
    const cornerMin = [
        mx,
        mx,
        mx
    ];
    if (bounds.contains(tr.center)) {
        for (const corner of corners) {
            cjsExports.vec3.min(cornerMin, cornerMin, corner);
            cjsExports.vec3.max(cornerMax, cornerMax, corner);
        }
        cornerMax[2] = 0;
        const point = tr.point;
        const center = [
            point.x * scale,
            point.y * scale,
            0
        ];
        cjsExports.vec3.min(cornerMin, cornerMin, center);
        cjsExports.vec3.max(cornerMax, cornerMax, center);
        return new Aabb(cornerMin, cornerMax);
    }
    if (tr._tileCoverLift > 0) {
        for (const corner of corners) {
            cjsExports.vec3.min(cornerMin, cornerMin, corner);
            cjsExports.vec3.max(cornerMax, cornerMax, corner);
        }
        return new Aabb(cornerMin, cornerMax);
    }
    const arcCenter = [
        m[12] * scale,
        m[13] * scale,
        m[14] * scale
    ];
    const tileCenter = bounds.getCenter();
    const centerLat = clamp(tr.center.lat, -MAX_MERCATOR_LATITUDE, MAX_MERCATOR_LATITUDE);
    const tileCenterLat = clamp(tileCenter.lat, -MAX_MERCATOR_LATITUDE, MAX_MERCATOR_LATITUDE);
    const camX = mercatorXfromLng(tr.center.lng);
    const camY = mercatorYfromLat(centerLat);
    let dx = camX - mercatorXfromLng(tileCenter.lng);
    const dy = camY - mercatorYfromLat(tileCenterLat);
    if (dx > 0.5) {
        dx -= 1;
    } else if (dx < -0.5) {
        dx += 1;
    }
    let closestArcIdx = 0;
    if (Math.abs(dx) > Math.abs(dy)) {
        closestArcIdx = dx >= 0 ? 1 : 3;
    } else {
        closestArcIdx = dy >= 0 ? 0 : 2;
        const yAxis = [
            m[4] * scale,
            m[5] * scale,
            m[6] * scale
        ];
        const shift = -Math.sin(degToRad(dy >= 0 ? bounds.getSouth() : bounds.getNorth())) * GLOBE_RADIUS;
        cjsExports.vec3.scaleAndAdd(arcCenter, arcCenter, yAxis, shift);
    }
    const arcStart = corners[closestArcIdx];
    const arcEnd = corners[(closestArcIdx + 1) % 4];
    const closestArc = new Arc(arcStart, arcEnd, arcCenter);
    const arcExtremum = [
        localExtremum(closestArc, 0) || arcStart[0],
        localExtremum(closestArc, 1) || arcStart[1],
        localExtremum(closestArc, 2) || arcStart[2]
    ];
    const phase = globeToMercatorTransition(tr.zoom);
    if (phase > 0) {
        const mercatorCorners = mercatorTileCornersInCameraSpace(tileId, numTiles, tr._pixelsPerMercatorPixel, camX, camY);
        for (let i = 0; i < corners.length; i++) {
            interpolateVec3(corners[i], mercatorCorners[i], phase);
        }
        const mercatorMidpoint = cjsExports.vec3.add([], mercatorCorners[closestArcIdx], mercatorCorners[(closestArcIdx + 1) % 4]);
        cjsExports.vec3.scale(mercatorMidpoint, mercatorMidpoint, 0.5);
        interpolateVec3(arcExtremum, mercatorMidpoint, phase);
    }
    for (const corner of corners) {
        cjsExports.vec3.min(cornerMin, cornerMin, corner);
        cjsExports.vec3.max(cornerMax, cornerMax, corner);
    }
    cornerMin[2] = Math.min(arcStart[2], arcEnd[2]);
    cjsExports.vec3.min(cornerMin, cornerMin, arcExtremum);
    cjsExports.vec3.max(cornerMax, cornerMax, arcExtremum);
    return new Aabb(cornerMin, cornerMax);
}
function tileCornersToBounds({x, y, z}, extendToPoles = false) {
    const s = 1 / (1 << z);
    const sw = new LngLat(lngFromMercatorX(x * s), y === (1 << z) - 1 && extendToPoles ? -90 : latFromMercatorY((y + 1) * s));
    const ne = new LngLat(lngFromMercatorX((x + 1) * s), y === 0 && extendToPoles ? 90 : latFromMercatorY(y * s));
    return new LngLatBounds(sw, ne);
}
function mercatorTileCornersInCameraSpace({x, y, z}, numTiles, mercatorScale, camX, camY) {
    const tileScale = 1 / (1 << z);
    let w = x * tileScale;
    let e = w + tileScale;
    let n = y * tileScale;
    let s = n + tileScale;
    let wrap = 0;
    const tileCenterXFromCamera = (w + e) / 2 - camX;
    if (tileCenterXFromCamera > 0.5) {
        wrap = -1;
    } else if (tileCenterXFromCamera < -0.5) {
        wrap = 1;
    }
    camX *= numTiles;
    camY *= numTiles;
    w = ((w + wrap) * numTiles - camX) * mercatorScale + camX;
    e = ((e + wrap) * numTiles - camX) * mercatorScale + camX;
    n = (n * numTiles - camY) * mercatorScale + camY;
    s = (s * numTiles - camY) * mercatorScale + camY;
    return [
        [
            w,
            s,
            0
        ],
        [
            e,
            s,
            0
        ],
        [
            e,
            n,
            0
        ],
        [
            w,
            n,
            0
        ]
    ];
}
function boundsToECEF(bounds, radius = GLOBE_RADIUS) {
    const ny = degToRad(bounds.getNorth());
    const sy = degToRad(bounds.getSouth());
    const cosN = Math.cos(ny);
    const cosS = Math.cos(sy);
    const sinN = Math.sin(ny);
    const sinS = Math.sin(sy);
    const w = bounds.getWest();
    const e = bounds.getEast();
    return [
        csLatLngToECEF(cosS, sinS, w, radius),
        csLatLngToECEF(cosS, sinS, e, radius),
        csLatLngToECEF(cosN, sinN, e, radius),
        csLatLngToECEF(cosN, sinN, w, radius)
    ];
}
function tileCoordToECEF(x, y, id, radius) {
    const tileCount = 1 << id.z;
    const mercatorX = (x / EXTENT + id.x) / tileCount;
    const mercatorY = (y / EXTENT + id.y) / tileCount;
    const lat = latFromMercatorY(mercatorY);
    const lng = lngFromMercatorX(mercatorX);
    const pos = latLngToECEF(lat, lng, radius);
    return pos;
}
function globeECEFOrigin(tileMatrix, id) {
    const origin = [
        0,
        0,
        0
    ];
    const bounds = globeTileBounds(id.canonical);
    const normalizationMatrix = globeNormalizeECEF(bounds);
    cjsExports.vec3.transformMat4(origin, origin, normalizationMatrix);
    cjsExports.vec3.transformMat4(origin, origin, tileMatrix);
    return origin;
}
function globeECEFNormalizationScale({min, max}) {
    return GLOBE_NORMALIZATION_MASK / Math.max(max[0] - min[0], max[1] - min[1], max[2] - min[2]);
}
const tempMatrix = new Float64Array(16);
function globeNormalizeECEF(bounds) {
    const scale = globeECEFNormalizationScale(bounds);
    const m = cjsExports.mat4.fromScaling(tempMatrix, [
        scale,
        scale,
        scale
    ]);
    return cjsExports.mat4.translate(m, m, cjsExports.vec3.negate([], bounds.min));
}
function globeDenormalizeECEF(bounds) {
    const m = cjsExports.mat4.fromTranslation(tempMatrix, bounds.min);
    const scale = 1 / globeECEFNormalizationScale(bounds);
    return cjsExports.mat4.scale(m, m, [
        scale,
        scale,
        scale
    ]);
}
function globeECEFUnitsToPixelScale(worldSize) {
    const localRadius = EXTENT / (2 * Math.PI);
    const wsRadius = worldSize / (2 * Math.PI);
    return wsRadius / localRadius;
}
function globePixelsToTileUnits(zoom, id) {
    const ecefPerPixel = EXTENT / (TILE_SIZE * Math.pow(2, zoom));
    const normCoeff = globeECEFNormalizationScale(globeTileBounds(id));
    return ecefPerPixel * normCoeff;
}
function calculateGlobePosMatrix(x, y, worldSize, lng, lat) {
    const scale = globeECEFUnitsToPixelScale(worldSize);
    const offset = [
        x,
        y,
        -worldSize / (2 * Math.PI)
    ];
    const m = cjsExports.mat4.identity(new Float64Array(16));
    cjsExports.mat4.translate(m, m, offset);
    cjsExports.mat4.scale(m, m, [
        scale,
        scale,
        scale
    ]);
    cjsExports.mat4.rotateX(m, m, degToRad(-lat));
    cjsExports.mat4.rotateY(m, m, degToRad(-lng));
    return m;
}
function calculateGlobeMatrix(tr) {
    const {x, y} = tr.point;
    const {lng, lat} = tr._center;
    return calculateGlobePosMatrix(x, y, tr.worldSize, lng, lat);
}
function calculateGlobeLabelMatrix(tr, id) {
    const {x, y} = tr.point;
    const m = calculateGlobePosMatrix(x, y, tr.worldSize / tr._pixelsPerMercatorPixel, 0, 0);
    return cjsExports.mat4.multiply(m, m, globeDenormalizeECEF(globeTileBounds(id)));
}
function calculateGlobeMercatorMatrix(tr) {
    const zScale = tr.pixelsPerMeter;
    const ws = zScale / mercatorZfromAltitude(1, tr.center.lat);
    const posMatrix = cjsExports.mat4.identity(new Float64Array(16));
    cjsExports.mat4.translate(posMatrix, posMatrix, [
        tr.point.x,
        tr.point.y,
        0
    ]);
    cjsExports.mat4.scale(posMatrix, posMatrix, [
        ws,
        ws,
        zScale
    ]);
    return Float32Array.from(posMatrix);
}
function globeToMercatorTransition(zoom) {
    return smoothstep(GLOBE_ZOOM_THRESHOLD_MIN, GLOBE_ZOOM_THRESHOLD_MAX, zoom);
}
function globePoleMatrixForTile(z, x, tr) {
    const poleMatrix = cjsExports.mat4.identity(new Float64Array(16));
    const numTiles = 1 << z;
    const xOffsetAngle = (x / numTiles - 0.5) * Math.PI * 2;
    cjsExports.mat4.rotateY(poleMatrix, tr.globeMatrix, xOffsetAngle);
    return Float32Array.from(poleMatrix);
}
function globeUseCustomAntiAliasing(painter, context, transform) {
    const transitionT = globeToMercatorTransition(transform.zoom);
    const useContextAA = painter.style.map._antialias;
    const disabled = painter.terrain && painter.terrain.exaggeration() > 0;
    return transitionT === 0 && !useContextAA && !disabled;
}
function getGridMatrix(id, bounds, latitudinalLod, worldSize) {
    const n = bounds.getNorth();
    const s = bounds.getSouth();
    const w = bounds.getWest();
    const e = bounds.getEast();
    const tiles = 1 << id.z;
    const tileWidth = e - w;
    const tileHeight = n - s;
    const tileToLng = tileWidth / GLOBE_VERTEX_GRID_SIZE;
    const tileToLat = -tileHeight / GLOBE_LATITUDINAL_GRID_LOD_TABLE[latitudinalLod];
    const matrix = [
        0,
        tileToLng,
        0,
        tileToLat,
        0,
        0,
        n,
        w,
        0
    ];
    if (id.z > 0) {
        const pixelPadding = 0.5;
        const padding = pixelPadding * 360 / worldSize;
        const xScale = padding / tileWidth + 1;
        const yScale = padding / tileHeight + 1;
        const padMatrix = [
            xScale,
            0,
            0,
            0,
            yScale,
            0,
            -0.5 * padding / tileToLng,
            0.5 * padding / tileToLat,
            1
        ];
        cjsExports.mat3.multiply(matrix, matrix, padMatrix);
    }
    matrix[2] = tiles;
    matrix[5] = id.x;
    matrix[8] = id.y;
    return matrix;
}
function getLatitudinalLod(lat) {
    const UPPER_LATITUDE = MAX_MERCATOR_LATITUDE - 5;
    lat = clamp(lat, -UPPER_LATITUDE, UPPER_LATITUDE) / UPPER_LATITUDE * 90;
    const t = Math.pow(Math.abs(Math.sin(degToRad(lat))), 3);
    const lod = Math.round(t * (GLOBE_LATITUDINAL_GRID_LOD_TABLE.length - 1));
    return lod;
}
function globeCenterToScreenPoint(tr) {
    const pos = [
        0,
        0,
        0
    ];
    const matrix = cjsExports.mat4.identity(new Float64Array(16));
    cjsExports.mat4.multiply(matrix, tr.pixelMatrix, tr.globeMatrix);
    cjsExports.vec3.transformMat4(pos, pos, matrix);
    return new Point(pos[0], pos[1]);
}
function cameraPositionInECEF(tr) {
    const centerToPivot = latLngToECEF(tr._center.lat, tr._center.lng);
    const south = cjsExports.vec3.fromValues(0, 1, 0);
    let axis = cjsExports.vec3.cross([], south, centerToPivot);
    const rotation = cjsExports.mat4.fromRotation([], -tr.angle, centerToPivot);
    axis = cjsExports.vec3.transformMat4(axis, axis, rotation);
    cjsExports.mat4.fromRotation(rotation, -tr._pitch, axis);
    const pivotToCamera = cjsExports.vec3.normalize([], centerToPivot);
    cjsExports.vec3.scale(pivotToCamera, pivotToCamera, globeMetersToEcef(tr.cameraToCenterDistance / tr.pixelsPerMeter));
    cjsExports.vec3.transformMat4(pivotToCamera, pivotToCamera, rotation);
    return cjsExports.vec3.add([], centerToPivot, pivotToCamera);
}
function globeTiltAtLngLat(tr, lngLat) {
    const centerToPoint = latLngToECEF(lngLat.lat, lngLat.lng);
    const centerToCamera = cameraPositionInECEF(tr);
    const pointToCamera = cjsExports.vec3.subtract([], centerToCamera, centerToPoint);
    return cjsExports.vec3.angle(pointToCamera, centerToPoint);
}
function isLngLatBehindGlobe(tr, lngLat) {
    return globeTiltAtLngLat(tr, lngLat) > Math.PI / 2 * 1.01;
}
function polesInViewport(tr) {
    const ecefToScreenMatrix = cjsExports.mat4.identity(new Float64Array(16));
    cjsExports.mat4.multiply(ecefToScreenMatrix, tr.pixelMatrix, tr.globeMatrix);
    const north = [
        0,
        GLOBE_MIN,
        0
    ];
    const south = [
        0,
        GLOBE_MAX,
        0
    ];
    cjsExports.vec3.transformMat4(north, north, ecefToScreenMatrix);
    cjsExports.vec3.transformMat4(south, south, ecefToScreenMatrix);
    const northInViewport = north[0] > 0 && north[0] <= tr.width && north[1] > 0 && north[1] <= tr.height && !isLngLatBehindGlobe(tr, new LngLat(tr.center.lat, 90));
    const southInViewport = south[0] > 0 && south[0] <= tr.width && south[1] > 0 && south[1] <= tr.height && !isLngLatBehindGlobe(tr, new LngLat(tr.center.lat, -90));
    return [
        northInViewport,
        southInViewport
    ];
}
const POLE_RAD = degToRad(85);
const POLE_COS = Math.cos(POLE_RAD);
const POLE_SIN = Math.sin(POLE_RAD);
const EMBED_SKIRTS = true;
class GlobeSharedBuffers {
    constructor(context) {
        this._createGrid(context);
        this._createPoles(context);
    }
    destroy() {
        this._poleIndexBuffer.destroy();
        this._gridBuffer.destroy();
        this._gridIndexBuffer.destroy();
        this._poleNorthVertexBuffer.destroy();
        this._poleSouthVertexBuffer.destroy();
        for (const segments of this._poleSegments)
            segments.destroy();
        for (const segments of this._gridSegments) {
            segments.withSkirts.destroy();
            segments.withoutSkirts.destroy();
        }
    }
    // Generate terrain grid vertices and indices for all LOD's
    //
    // Grid vertices memory layout:
    //
    //          First line Skirt
    //          ┌───────────────┐
    //          │┌─────────────┐│
    // Left     ││┼┼┼┼┼┼┼┼┼┼┼┼┼││ Right
    // Border   ││┼┼┼┼┼┼┼┼┼┼┼┼┼││ Border
    // Skirt    │├─────────────┤│ Skirt
    //          ││  Main Grid  ││
    //          │├─────────────┤│
    //          ││┼┼┼┼┼┼┼┼┼┼┼┼┼││
    //          ││┼┼┼┼┼┼┼┼┼┼┼┼┼││
    //          │└─────────────┘│
    //          ├───────────────┤
    //          ├───────────────┤
    //          └───────────────┘
    //      Bottom Skirt = Number of LOD's
    //
    _fillGridMeshWithLods(longitudinalCellsCount, latitudinalLods) {
        const vertices = new StructArrayLayout2i4();
        const indices = new StructArrayLayout3ui6();
        const segments = [];
        const xVertices = longitudinalCellsCount + 1 + 2 * (1 );
        const yVerticesHighLodNoStrip = latitudinalLods[0] + 1;
        const yVerticesHighLodWithStrip = latitudinalLods[0] + 1 + (1 + latitudinalLods.length );
        const prepareVertex = (x, y, isSkirt) => {
            let adjustedX = ((() => {
                if (x === xVertices - 1) {
                    return x - 2;
                } else if (x === 0) {
                    return x;
                } else {
                    return x - 1;
                }
            })());
            const skirtOffset = 24575;
            adjustedX += isSkirt ? skirtOffset : 0;
            return [
                adjustedX,
                y
            ];
        };
        {
            for (let x = 0; x < xVertices; ++x) {
                vertices.emplaceBack(...prepareVertex(x, 0, true));
            }
        }
        for (let y = 0; y < yVerticesHighLodNoStrip; ++y) {
            for (let x = 0; x < xVertices; ++x) {
                const isSideBorder = x === 0 || x === xVertices - 1;
                vertices.emplaceBack(...prepareVertex(x, y, isSideBorder && EMBED_SKIRTS));
            }
        }
        {
            for (let lodIdx = 0; lodIdx < latitudinalLods.length; ++lodIdx) {
                const lastYRowForLod = latitudinalLods[lodIdx];
                for (let x = 0; x < xVertices; ++x) {
                    vertices.emplaceBack(...prepareVertex(x, lastYRowForLod, true));
                }
            }
        }
        for (let lodIdx = 0; lodIdx < latitudinalLods.length; ++lodIdx) {
            const indexOffset = indices.length;
            const yVerticesLod = latitudinalLods[lodIdx] + 1 + 2 * (1 );
            const skirtsOnlyIndices = new StructArrayLayout3ui6();
            for (let y = 0; y < yVerticesLod - 1; y++) {
                const isLastLine = y === yVerticesLod - 2;
                const offsetToNextRow = isLastLine && EMBED_SKIRTS ? xVertices * (yVerticesHighLodWithStrip - latitudinalLods.length + lodIdx - y) : xVertices;
                for (let x = 0; x < xVertices - 1; x++) {
                    const idx = y * xVertices + x;
                    const isSkirt = (y === 0 || isLastLine || x === 0 || x === xVertices - 2);
                    if (isSkirt) {
                        skirtsOnlyIndices.emplaceBack(idx + 1, idx, idx + offsetToNextRow);
                        skirtsOnlyIndices.emplaceBack(idx + offsetToNextRow, idx + offsetToNextRow + 1, idx + 1);
                    } else {
                        indices.emplaceBack(idx + 1, idx, idx + offsetToNextRow);
                        indices.emplaceBack(idx + offsetToNextRow, idx + offsetToNextRow + 1, idx + 1);
                    }
                }
            }
            const withoutSkirts = SegmentVector.simpleSegment(0, indexOffset, vertices.length, indices.length - indexOffset);
            for (let i = 0; i < skirtsOnlyIndices.uint16.length; i += 3) {
                indices.emplaceBack(skirtsOnlyIndices.uint16[i], skirtsOnlyIndices.uint16[i + 1], skirtsOnlyIndices.uint16[i + 2]);
            }
            const withSkirts = SegmentVector.simpleSegment(0, indexOffset, vertices.length, indices.length - indexOffset);
            segments.push({
                withoutSkirts,
                withSkirts
            });
        }
        return {
            vertices,
            indices,
            segments
        };
    }
    _createGrid(context) {
        const gridWithLods = this._fillGridMeshWithLods(GLOBE_VERTEX_GRID_SIZE, GLOBE_LATITUDINAL_GRID_LOD_TABLE);
        this._gridSegments = gridWithLods.segments;
        this._gridBuffer = context.createVertexBuffer(gridWithLods.vertices, posAttributes.members);
        this._gridIndexBuffer = context.createIndexBuffer(gridWithLods.indices, true);
    }
    _createPoles(context) {
        const poleIndices = new StructArrayLayout3ui6();
        for (let i = 0; i <= GLOBE_VERTEX_GRID_SIZE; i++) {
            poleIndices.emplaceBack(0, i + 1, i + 2);
        }
        this._poleIndexBuffer = context.createIndexBuffer(poleIndices, true);
        const northVertices = new StructArrayLayout5f20();
        const southVertices = new StructArrayLayout5f20();
        const texturedNorthVertices = new StructArrayLayout5f20();
        const texturedSouthVertices = new StructArrayLayout5f20();
        const polePrimitives = GLOBE_VERTEX_GRID_SIZE;
        const poleVertices = GLOBE_VERTEX_GRID_SIZE + 2;
        this._poleSegments = [];
        for (let zoom = 0, offset = 0; zoom < GLOBE_ZOOM_THRESHOLD_MIN; zoom++) {
            const tiles = 1 << zoom;
            const endAngle = 360 / tiles;
            northVertices.emplaceBack(0, -GLOBE_RADIUS, 0, 0.5, 0);
            southVertices.emplaceBack(0, -GLOBE_RADIUS, 0, 0.5, 1);
            texturedNorthVertices.emplaceBack(0, -GLOBE_RADIUS, 0, 0.5, 0.5);
            texturedSouthVertices.emplaceBack(0, -GLOBE_RADIUS, 0, 0.5, 0.5);
            for (let i = 0; i <= GLOBE_VERTEX_GRID_SIZE; i++) {
                let uvX = i / GLOBE_VERTEX_GRID_SIZE;
                let uvY = 0;
                const angle = number(0, endAngle, uvX);
                const [gx, gy, gz] = csLatLngToECEF(POLE_COS, POLE_SIN, angle, GLOBE_RADIUS);
                northVertices.emplaceBack(gx, gy, gz, uvX, uvY);
                southVertices.emplaceBack(gx, gy, gz, uvX, 1 - uvY);
                const rad = degToRad(angle);
                uvX = 0.5 + 0.5 * Math.sin(rad);
                uvY = 0.5 + 0.5 * Math.cos(rad);
                texturedNorthVertices.emplaceBack(gx, gy, gz, uvX, uvY);
                texturedSouthVertices.emplaceBack(gx, gy, gz, uvX, 1 - uvY);
            }
            this._poleSegments.push(SegmentVector.simpleSegment(offset, 0, poleVertices, polePrimitives));
            offset += poleVertices;
        }
        this._poleNorthVertexBuffer = context.createVertexBuffer(northVertices, members$5, false);
        this._poleSouthVertexBuffer = context.createVertexBuffer(southVertices, members$5, false);
        this._texturedPoleNorthVertexBuffer = context.createVertexBuffer(texturedNorthVertices, members$5, false);
        this._texturedPoleSouthVertexBuffer = context.createVertexBuffer(texturedSouthVertices, members$5, false);
    }
    getGridBuffers(latitudinalLod, withSkirts) {
        return [
            this._gridBuffer,
            this._gridIndexBuffer,
            withSkirts ? this._gridSegments[latitudinalLod].withSkirts : this._gridSegments[latitudinalLod].withoutSkirts
        ];
    }
    getPoleBuffers(z, textured) {
        return [
            textured ? this._texturedPoleNorthVertexBuffer : this._poleNorthVertexBuffer,
            textured ? this._texturedPoleSouthVertexBuffer : this._poleSouthVertexBuffer,
            this._poleIndexBuffer,
            this._poleSegments[z]
        ];
    }
}

const circleUniforms = context => ({
    'u_camera_to_center_distance': new Uniform1f(context),
    'u_extrude_scale': new UniformMatrix2f(context),
    'u_device_pixel_ratio': new Uniform1f(context),
    'u_matrix': new UniformMatrix4f(context),
    'u_inv_rot_matrix': new UniformMatrix4f(context),
    'u_merc_center': new Uniform2f(context),
    'u_tile_id': new Uniform3f(context),
    'u_zoom_transition': new Uniform1f(context),
    'u_up_dir': new Uniform3f(context),
    'u_emissive_strength': new Uniform1f(context)
});
const identityMatrix = cjsExports.mat4.create();
const circleUniformValues = (painter, coord, tile, invMatrix, mercatorCenter, layer) => {
    const transform = painter.transform;
    const isGlobe = transform.projection.name === 'globe';
    let extrudeScale;
    if (layer.paint.get('circle-pitch-alignment') === 'map') {
        if (isGlobe) {
            const s = globePixelsToTileUnits(transform.zoom, coord.canonical) * transform._pixelsPerMercatorPixel;
            extrudeScale = Float32Array.from([
                s,
                0,
                0,
                s
            ]);
        } else {
            extrudeScale = transform.calculatePixelsToTileUnitsMatrix(tile);
        }
    } else {
        extrudeScale = new Float32Array([
            transform.pixelsToGLUnits[0],
            0,
            0,
            transform.pixelsToGLUnits[1]
        ]);
    }
    const values = {
        'u_camera_to_center_distance': painter.transform.getCameraToCenterDistance(transform.projection),
        'u_matrix': painter.translatePosMatrix(coord.projMatrix, tile, layer.paint.get('circle-translate'), layer.paint.get('circle-translate-anchor')),
        'u_device_pixel_ratio': exported$1.devicePixelRatio,
        'u_extrude_scale': extrudeScale,
        'u_inv_rot_matrix': identityMatrix,
        'u_merc_center': [
            0,
            0
        ],
        'u_tile_id': [
            0,
            0,
            0
        ],
        'u_zoom_transition': 0,
        'u_up_dir': [
            0,
            0,
            0
        ],
        'u_emissive_strength': layer.paint.get('circle-emissive-strength')
    };
    if (isGlobe) {
        values['u_inv_rot_matrix'] = invMatrix;
        values['u_merc_center'] = mercatorCenter;
        values['u_tile_id'] = [
            coord.canonical.x,
            coord.canonical.y,
            1 << coord.canonical.z
        ];
        values['u_zoom_transition'] = globeToMercatorTransition(transform.zoom);
        const x = mercatorCenter[0] * EXTENT;
        const y = mercatorCenter[1] * EXTENT;
        values['u_up_dir'] = transform.projection.upVector(new CanonicalTileID(0, 0, 0), x, y);
    }
    return values;
};
const circleDefinesValues = layer => {
    const values = [];
    if (layer.paint.get('circle-pitch-alignment') === 'map')
        values.push('PITCH_WITH_MAP');
    if (layer.paint.get('circle-pitch-scale') === 'map')
        values.push('SCALE_WITH_MAP');
    return values;
};

class CircleStyleLayer extends StyleLayer {
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$c(),
            paint: getPaintProperties$d()
        };
        super(layer, properties, scope, lut, options);
    }
    createBucket(parameters) {
        return new CircleBucket(parameters);
    }
    queryRadius(bucket) {
        const circleBucket = bucket;
        return getMaximumPaintValue('circle-radius', this, circleBucket) + getMaximumPaintValue('circle-stroke-width', this, circleBucket) + translateDistance(this.paint.get('circle-translate'));
    }
    queryIntersectsFeature(queryGeometry, feature, featureState, geometry, zoom, transform, pixelPosMatrix, elevationHelper) {
        const translation = tilespaceTranslate(this.paint.get('circle-translate'), this.paint.get('circle-translate-anchor'), transform.angle, queryGeometry.pixelToTileUnitsFactor);
        const size = this.paint.get('circle-radius').evaluate(feature, featureState) + this.paint.get('circle-stroke-width').evaluate(feature, featureState);
        return queryIntersectsCircle(queryGeometry, geometry, transform, pixelPosMatrix, elevationHelper, this.paint.get('circle-pitch-alignment') === 'map', this.paint.get('circle-pitch-scale') === 'map', translation, size);
    }
    getProgramIds() {
        return ['circle'];
    }
    getDefaultProgramParams(_, zoom, lut) {
        const definesValues = circleDefinesValues(this);
        return {
            config: new ProgramConfiguration(this, {
                zoom,
                lut
            }),
            defines: definesValues,
            overrideFog: false
        };
    }
}
function queryIntersectsCircle(queryGeometry, geometry, transform, pixelPosMatrix, elevationHelper, alignWithMap, scaleWithMap, translation, size) {
    if (alignWithMap && queryGeometry.queryGeometry.isAboveHorizon)
        return false;
    if (alignWithMap)
        size *= queryGeometry.pixelToTileUnitsFactor;
    const tileId = queryGeometry.tileID.canonical;
    const elevationScale = transform.projection.upVectorScale(tileId, transform.center.lat, transform.worldSize).metersToTile;
    for (const ring of geometry) {
        for (const point of ring) {
            const translatedPoint = point.add(translation);
            const z = elevationHelper && transform.elevation ? transform.elevation.exaggeration() * elevationHelper.getElevationAt(translatedPoint.x, translatedPoint.y, true) : 0;
            const reproj = transform.projection.projectTilePoint(translatedPoint.x, translatedPoint.y, tileId);
            if (z > 0) {
                const dir = transform.projection.upVector(tileId, translatedPoint.x, translatedPoint.y);
                reproj.x += dir[0] * elevationScale * z;
                reproj.y += dir[1] * elevationScale * z;
                reproj.z += dir[2] * elevationScale * z;
            }
            const transformedPoint = alignWithMap ? translatedPoint : projectPoint(reproj.x, reproj.y, reproj.z, pixelPosMatrix);
            const transformedPolygon = alignWithMap ? queryGeometry.tilespaceRays.map(r => intersectAtHeight(r, z)) : queryGeometry.queryGeometry.screenGeometry;
            const projectedCenter = cjsExports.vec4.transformMat4([], [
                reproj.x,
                reproj.y,
                reproj.z,
                1
            ], pixelPosMatrix);
            if (!scaleWithMap && alignWithMap) {
                size *= projectedCenter[3] / transform.cameraToCenterDistance;
            } else if (scaleWithMap && !alignWithMap) {
                size *= transform.cameraToCenterDistance / projectedCenter[3];
            }
            if (alignWithMap) {
                const lat = latFromMercatorY((point.y / EXTENT + tileId.y) / (1 << tileId.z));
                const scale = transform.projection.pixelsPerMeter(lat, 1) / mercatorZfromAltitude(1, lat);
                size /= scale;
            }
            if (polygonIntersectsBufferedPoint(transformedPolygon, transformedPoint, size))
                return true;
        }
    }
    return false;
}
function projectPoint(x, y, z, pixelPosMatrix) {
    const point = cjsExports.vec4.transformMat4([], [
        x,
        y,
        z,
        1
    ], pixelPosMatrix);
    return new Point(point[0] / point[3], point[1] / point[3]);
}
const origin = cjsExports.vec3.fromValues(0, 0, 0);
const up = cjsExports.vec3.fromValues(0, 0, 1);
function intersectAtHeight(r, z) {
    const intersectionPt = cjsExports.vec3.create();
    origin[2] = z;
    r.intersectsPlane(origin, up, intersectionPt);
    return new Point(intersectionPt[0], intersectionPt[1]);
}

class HeatmapBucket extends CircleBucket {
}
register(HeatmapBucket, 'HeatmapBucket', { omit: ['layers'] });

let layout$c;
const getLayoutProperties$b = () => layout$c || (layout$c = new Properties({ 'visibility': new DataConstantProperty(spec['layout_heatmap']['visibility']) }));
let paint$c;
const getPaintProperties$c = () => paint$c || (paint$c = new Properties({
    'heatmap-radius': new DataDrivenProperty(spec['paint_heatmap']['heatmap-radius']),
    'heatmap-weight': new DataDrivenProperty(spec['paint_heatmap']['heatmap-weight']),
    'heatmap-intensity': new DataConstantProperty(spec['paint_heatmap']['heatmap-intensity']),
    'heatmap-color': new ColorRampProperty(spec['paint_heatmap']['heatmap-color']),
    'heatmap-opacity': new DataConstantProperty(spec['paint_heatmap']['heatmap-opacity']),
    'heatmap-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    })
}));

function createImage(image, {width, height}, channels, data) {
    if (!data) {
        data = new Uint8Array(width * height * channels);
    } else if (data instanceof Uint8ClampedArray) {
        data = new Uint8Array(data.buffer);
    } else if (data.length !== width * height * channels) {
        throw new RangeError('mismatched image size');
    }
    image.width = width;
    image.height = height;
    image.data = data;
    return image;
}
function resizeImage(image, newImage, channels) {
    const {width, height} = newImage;
    if (width === image.width && height === image.height) {
        return;
    }
    copyImage(image, newImage, {
        x: 0,
        y: 0
    }, {
        x: 0,
        y: 0
    }, {
        width: Math.min(image.width, width),
        height: Math.min(image.height, height)
    }, channels, null);
    image.width = width;
    image.height = height;
    image.data = newImage.data;
}
function copyImage(srcImg, dstImg, srcPt, dstPt, size, channels, lut, overrideRGBWithWhite) {
    if (size.width === 0 || size.height === 0) {
        return dstImg;
    }
    if (size.width > srcImg.width || size.height > srcImg.height || srcPt.x > srcImg.width - size.width || srcPt.y > srcImg.height - size.height) {
        throw new RangeError('out of range source coordinates for image copy');
    }
    if (size.width > dstImg.width || size.height > dstImg.height || dstPt.x > dstImg.width - size.width || dstPt.y > dstImg.height - size.height) {
        throw new RangeError('out of range destination coordinates for image copy');
    }
    const srcData = srcImg.data;
    const dstData = dstImg.data;
    const overrideRGB = channels === 4 && overrideRGBWithWhite;
    for (let y = 0; y < size.height; y++) {
        const srcOffset = ((srcPt.y + y) * srcImg.width + srcPt.x) * channels;
        const dstOffset = ((dstPt.y + y) * dstImg.width + dstPt.x) * channels;
        if (overrideRGB) {
            for (let i = 0; i < size.width; i++) {
                const srcByteOffset = srcOffset + i * channels + 3;
                const dstPixelOffset = dstOffset + i * channels;
                dstData[dstPixelOffset + 0] = 255;
                dstData[dstPixelOffset + 1] = 255;
                dstData[dstPixelOffset + 2] = 255;
                dstData[dstPixelOffset + 3] = srcData[srcByteOffset];
            }
        } else if (lut) {
            for (let i = 0; i < size.width; i++) {
                const srcByteOffset = srcOffset + i * channels;
                const dstPixelOffset = dstOffset + i * channels;
                const alpha = srcData[srcByteOffset + 3];
                const color = new Color(srcData[srcByteOffset + 0] / 255 * alpha, srcData[srcByteOffset + 1] / 255 * alpha, srcData[srcByteOffset + 2] / 255 * alpha, alpha);
                const shifted = color.toRenderColor(lut).toArray();
                dstData[dstPixelOffset + 0] = shifted[0];
                dstData[dstPixelOffset + 1] = shifted[1];
                dstData[dstPixelOffset + 2] = shifted[2];
                dstData[dstPixelOffset + 3] = shifted[3];
            }
        } else {
            for (let i = 0; i < size.width * channels; i++) {
                const srcByte = srcOffset + i;
                dstData[dstOffset + i] = srcData[srcByte];
            }
        }
    }
    return dstImg;
}
class AlphaImage {
    constructor(size, data) {
        createImage(this, size, 1, data);
    }
    resize(size) {
        resizeImage(this, new AlphaImage(size), 1);
    }
    clone() {
        return new AlphaImage({
            width: this.width,
            height: this.height
        }, new Uint8Array(this.data));
    }
    static copy(srcImg, dstImg, srcPt, dstPt, size) {
        copyImage(srcImg, dstImg, srcPt, dstPt, size, 1, null);
    }
}
class RGBAImage {
    constructor(size, data) {
        createImage(this, size, 4, data);
    }
    resize(size) {
        resizeImage(this, new RGBAImage(size), 4);
    }
    replace(data, copy) {
        if (copy) {
            this.data.set(data);
        } else if (data instanceof Uint8ClampedArray) {
            this.data = new Uint8Array(data.buffer);
        } else {
            this.data = data;
        }
    }
    clone() {
        return new RGBAImage({
            width: this.width,
            height: this.height
        }, new Uint8Array(this.data));
    }
    static copy(srcImg, dstImg, srcPt, dstPt, size, lut, overrideRGBWithWhite) {
        copyImage(srcImg, dstImg, srcPt, dstPt, size, 4, lut, overrideRGBWithWhite);
    }
}
class Float32Image {
    constructor(size, data) {
        this.width = size.width;
        this.height = size.height;
        if (data instanceof Uint8Array) {
            this.data = new Float32Array(data.buffer);
        } else {
            this.data = data;
        }
    }
}
register(AlphaImage, 'AlphaImage');
register(RGBAImage, 'RGBAImage');

function renderColorRamp(params) {
    const evaluationGlobals = {};
    const width = params.resolution || 256;
    const height = params.clips ? params.clips.length : 1;
    const image = params.image || new RGBAImage({
        width,
        height
    });
    const renderPixel = (stride, index, progress) => {
        evaluationGlobals[params.evaluationKey] = progress;
        const pxColor = params.expression.evaluate(evaluationGlobals);
        if (!pxColor)
            return;
        image.data[stride + index + 0] = Math.floor(pxColor.r * 255 / pxColor.a);
        image.data[stride + index + 1] = Math.floor(pxColor.g * 255 / pxColor.a);
        image.data[stride + index + 2] = Math.floor(pxColor.b * 255 / pxColor.a);
        image.data[stride + index + 3] = Math.floor(pxColor.a * 255);
    };
    if (!params.clips) {
        for (let i = 0, j = 0; i < width; i++, j += 4) {
            const progress = i / (width - 1);
            renderPixel(0, j, progress);
        }
    } else {
        for (let clip = 0, stride = 0; clip < height; ++clip, stride += width * 4) {
            for (let i = 0, j = 0; i < width; i++, j += 4) {
                const progress = i / (width - 1);
                const {start, end} = params.clips[clip];
                const evaluationProgress = start * (1 - progress) + end * progress;
                renderPixel(stride, j, evaluationProgress);
            }
        }
    }
    return image;
}

class HeatmapStyleLayer extends StyleLayer {
    createBucket(parameters) {
        return new HeatmapBucket(parameters);
    }
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$b(),
            paint: getPaintProperties$c()
        };
        super(layer, properties, scope, lut, options);
        this._updateColorRamp();
    }
    _handleSpecialPaintPropertyUpdate(name) {
        if (name === 'heatmap-color') {
            this._updateColorRamp();
        }
    }
    _updateColorRamp() {
        const expression = this._transitionablePaint._values['heatmap-color'].value.expression;
        this.colorRamp = renderColorRamp({
            expression,
            evaluationKey: 'heatmapDensity',
            image: this.colorRamp
        });
        this.colorRampTexture = null;
    }
    resize() {
        if (this.heatmapFbo) {
            this.heatmapFbo.destroy();
            this.heatmapFbo = null;
        }
    }
    queryRadius(bucket) {
        return getMaximumPaintValue('heatmap-radius', this, bucket);
    }
    queryIntersectsFeature(queryGeometry, feature, featureState, geometry, zoom, transform, pixelPosMatrix, elevationHelper) {
        const size = this.paint.get('heatmap-radius').evaluate(feature, featureState);
        return queryIntersectsCircle(queryGeometry, geometry, transform, pixelPosMatrix, elevationHelper, true, true, new Point(0, 0), size);
    }
    hasOffscreenPass() {
        return this.paint.get('heatmap-opacity') !== 0 && this.visibility !== 'none';
    }
    getProgramIds() {
        return [
            'heatmap',
            'heatmapTexture'
        ];
    }
    getDefaultProgramParams(name, zoom, lut) {
        if (name === 'heatmap') {
            return {
                config: new ProgramConfiguration(this, {
                    zoom,
                    lut
                }),
                overrideFog: false
            };
        }
        return {};
    }
}

let layout$b;
const getLayoutProperties$a = () => layout$b || (layout$b = new Properties({ 'visibility': new DataConstantProperty(spec['layout_hillshade']['visibility']) }));
let paint$b;
const getPaintProperties$b = () => paint$b || (paint$b = new Properties({
    'hillshade-illumination-direction': new DataConstantProperty(spec['paint_hillshade']['hillshade-illumination-direction']),
    'hillshade-illumination-anchor': new DataConstantProperty(spec['paint_hillshade']['hillshade-illumination-anchor']),
    'hillshade-exaggeration': new DataConstantProperty(spec['paint_hillshade']['hillshade-exaggeration']),
    'hillshade-shadow-color': new DataConstantProperty(spec['paint_hillshade']['hillshade-shadow-color']),
    'hillshade-highlight-color': new DataConstantProperty(spec['paint_hillshade']['hillshade-highlight-color']),
    'hillshade-accent-color': new DataConstantProperty(spec['paint_hillshade']['hillshade-accent-color']),
    'hillshade-emissive-strength': new DataConstantProperty(spec['paint_hillshade']['hillshade-emissive-strength']),
    'hillshade-shadow-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    }),
    'hillshade-highlight-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    }),
    'hillshade-accent-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    })
}));

class HillshadeStyleLayer extends StyleLayer {
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$a(),
            paint: getPaintProperties$b()
        };
        super(layer, properties, scope, lut, options);
    }
    shouldRedrape() {
        return this.hasOffscreenPass() && this.paint.get('hillshade-illumination-anchor') === 'viewport';
    }
    hasOffscreenPass() {
        return this.paint.get('hillshade-exaggeration') !== 0 && this.visibility !== 'none';
    }
    getProgramIds() {
        return [
            'hillshade',
            'hillshadePrepare'
        ];
    }
    // eslint-disable-next-line no-unused-vars
    getDefaultProgramParams(name, zoom, lut) {
        return { overrideFog: false };
    }
}

const layout$a = createLayout([{
        name: 'a_pos',
        components: 2,
        type: 'Int16'
    }], 4);
const {members: members$4, size: size$4, alignment: alignment$4} = layout$a;

function earcut(data, holeIndices, dim = 2) {
    const hasHoles = holeIndices && holeIndices.length;
    const outerLen = hasHoles ? holeIndices[0] * dim : data.length;
    let outerNode = linkedList(data, 0, outerLen, dim, true);
    const triangles = [];
    if (!outerNode || outerNode.next === outerNode.prev)
        return triangles;
    let minX, minY, invSize;
    if (hasHoles)
        outerNode = eliminateHoles(data, holeIndices, outerNode, dim);
    // if the shape is not too simple, we'll use z-order curve hash later; calculate polygon bbox
    if (data.length > 80 * dim) {
        minX = Infinity;
        minY = Infinity;
        let maxX = -Infinity;
        let maxY = -Infinity;
        for (let i = dim; i < outerLen; i += dim) {
            const x = data[i];
            const y = data[i + 1];
            if (x < minX)
                minX = x;
            if (y < minY)
                minY = y;
            if (x > maxX)
                maxX = x;
            if (y > maxY)
                maxY = y;
        }
        // minX, minY and invSize are later used to transform coords into integers for z-order calculation
        invSize = Math.max(maxX - minX, maxY - minY);
        invSize = invSize !== 0 ? 32767 / invSize : 0;
    }
    earcutLinked(outerNode, triangles, dim, minX, minY, invSize, 0);
    return triangles;
}
// create a circular doubly linked list from polygon points in the specified winding order
function linkedList(data, start, end, dim, clockwise) {
    let last;
    if (clockwise === signedArea(data, start, end, dim) > 0) {
        for (let i = start; i < end; i += dim)
            last = insertNode(i / dim | 0, data[i], data[i + 1], last);
    } else {
        for (let i = end - dim; i >= start; i -= dim)
            last = insertNode(i / dim | 0, data[i], data[i + 1], last);
    }
    if (last && equals(last, last.next)) {
        removeNode(last);
        last = last.next;
    }
    return last;
}
// eliminate colinear or duplicate points
function filterPoints(start, end) {
    if (!start)
        return start;
    if (!end)
        end = start;
    let p = start, again;
    do {
        again = false;
        if (!p.steiner && (equals(p, p.next) || area(p.prev, p, p.next) === 0)) {
            removeNode(p);
            p = end = p.prev;
            if (p === p.next)
                break;
            again = true;
        } else {
            p = p.next;
        }
    } while (again || p !== end);
    return end;
}
// main ear slicing loop which triangulates a polygon (given as a linked list)
function earcutLinked(ear, triangles, dim, minX, minY, invSize, pass) {
    if (!ear)
        return;
    // interlink polygon nodes in z-order
    if (!pass && invSize)
        indexCurve(ear, minX, minY, invSize);
    let stop = ear;
    // iterate through ears, slicing them one by one
    while (ear.prev !== ear.next) {
        const prev = ear.prev;
        const next = ear.next;
        if (invSize ? isEarHashed(ear, minX, minY, invSize) : isEar(ear)) {
            triangles.push(prev.i, ear.i, next.i);
            // cut off the triangle
            removeNode(ear);
            // skipping the next vertex leads to less sliver triangles
            ear = next.next;
            stop = next.next;
            continue;
        }
        ear = next;
        // if we looped through the whole remaining polygon and can't find any more ears
        if (ear === stop) {
            // try filtering points and slicing again
            if (!pass) {
                earcutLinked(filterPoints(ear), triangles, dim, minX, minY, invSize, 1);    // if this didn't work, try curing all small self-intersections locally
            } else if (pass === 1) {
                ear = cureLocalIntersections(filterPoints(ear), triangles);
                earcutLinked(ear, triangles, dim, minX, minY, invSize, 2);    // as a last resort, try splitting the remaining polygon into two
            } else if (pass === 2) {
                splitEarcut(ear, triangles, dim, minX, minY, invSize);
            }
            break;
        }
    }
}
// check whether a polygon node forms a valid ear with adjacent nodes
function isEar(ear) {
    const a = ear.prev, b = ear, c = ear.next;
    if (area(a, b, c) >= 0)
        return false;
    // reflex, can't be an ear
    // now make sure we don't have other points inside the potential ear
    const ax = a.x, bx = b.x, cx = c.x, ay = a.y, by = b.y, cy = c.y;
    // triangle bbox; min & max are calculated like this for speed
    const x0 = ax < bx ? ax < cx ? ax : cx : bx < cx ? bx : cx, y0 = ay < by ? ay < cy ? ay : cy : by < cy ? by : cy, x1 = ax > bx ? ax > cx ? ax : cx : bx > cx ? bx : cx, y1 = ay > by ? ay > cy ? ay : cy : by > cy ? by : cy;
    let p = c.next;
    while (p !== a) {
        if (p.x >= x0 && p.x <= x1 && p.y >= y0 && p.y <= y1 && pointInTriangle(ax, ay, bx, by, cx, cy, p.x, p.y) && area(p.prev, p, p.next) >= 0)
            return false;
        p = p.next;
    }
    return true;
}
function isEarHashed(ear, minX, minY, invSize) {
    const a = ear.prev, b = ear, c = ear.next;
    if (area(a, b, c) >= 0)
        return false;
    // reflex, can't be an ear
    const ax = a.x, bx = b.x, cx = c.x, ay = a.y, by = b.y, cy = c.y;
    // triangle bbox; min & max are calculated like this for speed
    const x0 = ax < bx ? ax < cx ? ax : cx : bx < cx ? bx : cx, y0 = ay < by ? ay < cy ? ay : cy : by < cy ? by : cy, x1 = ax > bx ? ax > cx ? ax : cx : bx > cx ? bx : cx, y1 = ay > by ? ay > cy ? ay : cy : by > cy ? by : cy;
    // z-order range for the current triangle bbox;
    const minZ = zOrder(x0, y0, minX, minY, invSize), maxZ = zOrder(x1, y1, minX, minY, invSize);
    let p = ear.prevZ, n = ear.nextZ;
    // look for points inside the triangle in both directions
    while (p && p.z >= minZ && n && n.z <= maxZ) {
        if (p.x >= x0 && p.x <= x1 && p.y >= y0 && p.y <= y1 && p !== a && p !== c && pointInTriangle(ax, ay, bx, by, cx, cy, p.x, p.y) && area(p.prev, p, p.next) >= 0)
            return false;
        p = p.prevZ;
        if (n.x >= x0 && n.x <= x1 && n.y >= y0 && n.y <= y1 && n !== a && n !== c && pointInTriangle(ax, ay, bx, by, cx, cy, n.x, n.y) && area(n.prev, n, n.next) >= 0)
            return false;
        n = n.nextZ;
    }
    // look for remaining points in decreasing z-order
    while (p && p.z >= minZ) {
        if (p.x >= x0 && p.x <= x1 && p.y >= y0 && p.y <= y1 && p !== a && p !== c && pointInTriangle(ax, ay, bx, by, cx, cy, p.x, p.y) && area(p.prev, p, p.next) >= 0)
            return false;
        p = p.prevZ;
    }
    // look for remaining points in increasing z-order
    while (n && n.z <= maxZ) {
        if (n.x >= x0 && n.x <= x1 && n.y >= y0 && n.y <= y1 && n !== a && n !== c && pointInTriangle(ax, ay, bx, by, cx, cy, n.x, n.y) && area(n.prev, n, n.next) >= 0)
            return false;
        n = n.nextZ;
    }
    return true;
}
// go through all polygon nodes and cure small local self-intersections
function cureLocalIntersections(start, triangles) {
    let p = start;
    do {
        const a = p.prev, b = p.next.next;
        if (!equals(a, b) && intersects(a, p, p.next, b) && locallyInside(a, b) && locallyInside(b, a)) {
            triangles.push(a.i, p.i, b.i);
            // remove two nodes involved
            removeNode(p);
            removeNode(p.next);
            p = start = b;
        }
        p = p.next;
    } while (p !== start);
    return filterPoints(p);
}
// try splitting polygon into two and triangulate them independently
function splitEarcut(start, triangles, dim, minX, minY, invSize) {
    // look for a valid diagonal that divides the polygon into two
    let a = start;
    do {
        let b = a.next.next;
        while (b !== a.prev) {
            if (a.i !== b.i && isValidDiagonal(a, b)) {
                // split the polygon in two by the diagonal
                let c = splitPolygon(a, b);
                // filter colinear points around the cuts
                a = filterPoints(a, a.next);
                c = filterPoints(c, c.next);
                // run earcut on each half
                earcutLinked(a, triangles, dim, minX, minY, invSize, 0);
                earcutLinked(c, triangles, dim, minX, minY, invSize, 0);
                return;
            }
            b = b.next;
        }
        a = a.next;
    } while (a !== start);
}
// link every hole into the outer loop, producing a single-ring polygon without holes
function eliminateHoles(data, holeIndices, outerNode, dim) {
    const queue = [];
    for (let i = 0, len = holeIndices.length; i < len; i++) {
        const start = holeIndices[i] * dim;
        const end = i < len - 1 ? holeIndices[i + 1] * dim : data.length;
        const list = linkedList(data, start, end, dim, false);
        if (list === list.next)
            list.steiner = true;
        queue.push(getLeftmost(list));
    }
    queue.sort(compareX);
    // process holes from left to right
    for (let i = 0; i < queue.length; i++) {
        outerNode = eliminateHole(queue[i], outerNode);
    }
    return outerNode;
}
function compareX(a, b) {
    return a.x - b.x;
}
// find a bridge between vertices that connects hole with an outer ring and and link it
function eliminateHole(hole, outerNode) {
    const bridge = findHoleBridge(hole, outerNode);
    if (!bridge) {
        return outerNode;
    }
    const bridgeReverse = splitPolygon(bridge, hole);
    // filter collinear points around the cuts
    filterPoints(bridgeReverse, bridgeReverse.next);
    return filterPoints(bridge, bridge.next);
}
// David Eberly's algorithm for finding a bridge between hole and outer polygon
function findHoleBridge(hole, outerNode) {
    let p = outerNode;
    const hx = hole.x;
    const hy = hole.y;
    let qx = -Infinity;
    let m;
    // find a segment intersected by a ray from the hole's leftmost point to the left;
    // segment's endpoint with lesser x will be potential connection point
    do {
        if (hy <= p.y && hy >= p.next.y && p.next.y !== p.y) {
            const x = p.x + (hy - p.y) * (p.next.x - p.x) / (p.next.y - p.y);
            if (x <= hx && x > qx) {
                qx = x;
                m = p.x < p.next.x ? p : p.next;
                if (x === hx)
                    return m;    // hole touches outer segment; pick leftmost endpoint
            }
        }
        p = p.next;
    } while (p !== outerNode);
    if (!m)
        return null;
    // look for points inside the triangle of hole point, segment intersection and endpoint;
    // if there are no points found, we have a valid connection;
    // otherwise choose the point of the minimum angle with the ray as connection point
    const stop = m;
    const mx = m.x;
    const my = m.y;
    let tanMin = Infinity;
    p = m;
    do {
        if (hx >= p.x && p.x >= mx && hx !== p.x && pointInTriangle(hy < my ? hx : qx, hy, mx, my, hy < my ? qx : hx, hy, p.x, p.y)) {
            const tan = Math.abs(hy - p.y) / (hx - p.x);
            // tangential
            if (locallyInside(p, hole) && (tan < tanMin || tan === tanMin && (p.x > m.x || p.x === m.x && sectorContainsSector(m, p)))) {
                m = p;
                tanMin = tan;
            }
        }
        p = p.next;
    } while (p !== stop);
    return m;
}
// whether sector in vertex m contains sector in vertex p in the same coordinates
function sectorContainsSector(m, p) {
    return area(m.prev, m, p.prev) < 0 && area(p.next, m, m.next) < 0;
}
// interlink polygon nodes in z-order
function indexCurve(start, minX, minY, invSize) {
    let p = start;
    do {
        if (p.z === 0)
            p.z = zOrder(p.x, p.y, minX, minY, invSize);
        p.prevZ = p.prev;
        p.nextZ = p.next;
        p = p.next;
    } while (p !== start);
    p.prevZ.nextZ = null;
    p.prevZ = null;
    sortLinked(p);
}
// Simon Tatham's linked list merge sort algorithm
// http://www.chiark.greenend.org.uk/~sgtatham/algorithms/listsort.html
function sortLinked(list) {
    let numMerges;
    let inSize = 1;
    do {
        let p = list;
        let e;
        list = null;
        let tail = null;
        numMerges = 0;
        while (p) {
            numMerges++;
            let q = p;
            let pSize = 0;
            for (let i = 0; i < inSize; i++) {
                pSize++;
                q = q.nextZ;
                if (!q)
                    break;
            }
            let qSize = inSize;
            while (pSize > 0 || qSize > 0 && q) {
                if (pSize !== 0 && (qSize === 0 || !q || p.z <= q.z)) {
                    e = p;
                    p = p.nextZ;
                    pSize--;
                } else {
                    e = q;
                    q = q.nextZ;
                    qSize--;
                }
                if (tail)
                    tail.nextZ = e;
                else
                    list = e;
                e.prevZ = tail;
                tail = e;
            }
            p = q;
        }
        tail.nextZ = null;
        inSize *= 2;
    } while (numMerges > 1);
    return list;
}
// z-order of a point given coords and inverse of the longer side of data bbox
function zOrder(x, y, minX, minY, invSize) {
    // coords are transformed into non-negative 15-bit integer range
    x = (x - minX) * invSize | 0;
    y = (y - minY) * invSize | 0;
    x = (x | x << 8) & 16711935;
    x = (x | x << 4) & 252645135;
    x = (x | x << 2) & 858993459;
    x = (x | x << 1) & 1431655765;
    y = (y | y << 8) & 16711935;
    y = (y | y << 4) & 252645135;
    y = (y | y << 2) & 858993459;
    y = (y | y << 1) & 1431655765;
    return x | y << 1;
}
// find the leftmost node of a polygon ring
function getLeftmost(start) {
    let p = start, leftmost = start;
    do {
        if (p.x < leftmost.x || p.x === leftmost.x && p.y < leftmost.y)
            leftmost = p;
        p = p.next;
    } while (p !== start);
    return leftmost;
}
// check if a point lies within a convex triangle
function pointInTriangle(ax, ay, bx, by, cx, cy, px, py) {
    return (cx - px) * (ay - py) >= (ax - px) * (cy - py) && (ax - px) * (by - py) >= (bx - px) * (ay - py) && (bx - px) * (cy - py) >= (cx - px) * (by - py);
}
// check if a diagonal between two polygon nodes is valid (lies in polygon interior)
function isValidDiagonal(a, b) {
    return a.next.i !== b.i && a.prev.i !== b.i && !intersectsPolygon(a, b) && (locallyInside(a, b) && locallyInside(b, a) && middleInside(a, b) && (area(a.prev, a, b.prev) || area(a, b.prev, b)) || // does not create opposite-facing sectors
    equals(a, b) && area(a.prev, a, a.next) > 0 && area(b.prev, b, b.next) > 0);    // special zero-length case
}
// signed area of a triangle
function area(p, q, r) {
    return (q.y - p.y) * (r.x - q.x) - (q.x - p.x) * (r.y - q.y);
}
// check if two points are equal
function equals(p1, p2) {
    return p1.x === p2.x && p1.y === p2.y;
}
// check if two segments intersect
function intersects(p1, q1, p2, q2) {
    const o1 = sign(area(p1, q1, p2));
    const o2 = sign(area(p1, q1, q2));
    const o3 = sign(area(p2, q2, p1));
    const o4 = sign(area(p2, q2, q1));
    if (o1 !== o2 && o3 !== o4)
        return true;
    // general case
    if (o1 === 0 && onSegment(p1, p2, q1))
        return true;
    // p1, q1 and p2 are collinear and p2 lies on p1q1
    if (o2 === 0 && onSegment(p1, q2, q1))
        return true;
    // p1, q1 and q2 are collinear and q2 lies on p1q1
    if (o3 === 0 && onSegment(p2, p1, q2))
        return true;
    // p2, q2 and p1 are collinear and p1 lies on p2q2
    if (o4 === 0 && onSegment(p2, q1, q2))
        return true;
    // p2, q2 and q1 are collinear and q1 lies on p2q2
    return false;
}
// for collinear points p, q, r, check if point q lies on segment pr
function onSegment(p, q, r) {
    return q.x <= Math.max(p.x, r.x) && q.x >= Math.min(p.x, r.x) && q.y <= Math.max(p.y, r.y) && q.y >= Math.min(p.y, r.y);
}
function sign(num) {
    return num > 0 ? 1 : num < 0 ? -1 : 0;
}
// check if a polygon diagonal intersects any polygon segments
function intersectsPolygon(a, b) {
    let p = a;
    do {
        if (p.i !== a.i && p.next.i !== a.i && p.i !== b.i && p.next.i !== b.i && intersects(p, p.next, a, b))
            return true;
        p = p.next;
    } while (p !== a);
    return false;
}
// check if a polygon diagonal is locally inside the polygon
function locallyInside(a, b) {
    return area(a.prev, a, a.next) < 0 ? area(a, b, a.next) >= 0 && area(a, a.prev, b) >= 0 : area(a, b, a.prev) < 0 || area(a, a.next, b) < 0;
}
// check if the middle point of a polygon diagonal is inside the polygon
function middleInside(a, b) {
    let p = a;
    let inside = false;
    const px = (a.x + b.x) / 2;
    const py = (a.y + b.y) / 2;
    do {
        if (p.y > py !== p.next.y > py && p.next.y !== p.y && px < (p.next.x - p.x) * (py - p.y) / (p.next.y - p.y) + p.x)
            inside = !inside;
        p = p.next;
    } while (p !== a);
    return inside;
}
// link two polygon vertices with a bridge; if the vertices belong to the same ring, it splits polygon into two;
// if one belongs to the outer ring and another to a hole, it merges it into a single ring
function splitPolygon(a, b) {
    const a2 = createNode(a.i, a.x, a.y), b2 = createNode(b.i, b.x, b.y), an = a.next, bp = b.prev;
    a.next = b;
    b.prev = a;
    a2.next = an;
    an.prev = a2;
    b2.next = a2;
    a2.prev = b2;
    bp.next = b2;
    b2.prev = bp;
    return b2;
}
// create a node and optionally link it with previous one (in a circular doubly linked list)
function insertNode(i, x, y, last) {
    const p = createNode(i, x, y);
    if (!last) {
        p.prev = p;
        p.next = p;
    } else {
        p.next = last.next;
        p.prev = last;
        last.next.prev = p;
        last.next = p;
    }
    return p;
}
function removeNode(p) {
    p.next.prev = p.prev;
    p.prev.next = p.next;
    if (p.prevZ)
        p.prevZ.nextZ = p.nextZ;
    if (p.nextZ)
        p.nextZ.prevZ = p.prevZ;
}
function createNode(i, x, y) {
    return {
        i,
        // vertex index in coordinates array
        x,
        y,
        // vertex coordinates
        prev: null,
        // previous and next vertex nodes in a polygon ring
        next: null,
        z: 0,
        // z-order curve value
        prevZ: null,
        // previous and next nodes in z-order
        nextZ: null,
        steiner: false    // indicates whether this is a steiner point
    };
}
function signedArea(data, start, end, dim) {
    let sum = 0;
    for (let i = start, j = end - dim; i < end; i += dim) {
        sum += (data[j] - data[i]) * (data[i + 1] + data[j + 1]);
        j = i;
    }
    return sum;
}

function classifyRings(rings, maxRings) {
    const len = rings.length;
    if (len <= 1)
        return [rings];
    const polygons = [];
    let polygon, ccw;
    for (let i = 0; i < len; i++) {
        const area = calculateSignedArea$1(rings[i]);
        if (area === 0)
            continue;
        rings[i].area = Math.abs(area);
        if (ccw === void 0)
            ccw = area < 0;
        if (ccw === area < 0) {
            if (polygon)
                polygons.push(polygon);
            polygon = [rings[i]];
        } else {
            polygon.push(rings[i]);
        }
    }
    if (polygon)
        polygons.push(polygon);
    if (maxRings > 1) {
        for (let j = 0; j < polygons.length; j++) {
            if (polygons[j].length <= maxRings)
                continue;
            quickselect(polygons[j], maxRings, 1, polygons[j].length - 1, compareAreas);
            polygons[j] = polygons[j].slice(0, maxRings);
        }
    }
    return polygons;
}
function compareAreas(a, b) {
    return b.area - a.area;
}

function addPattern(pattern, patterns, pixelRatio = 1) {
    if (!pattern) {
        return null;
    }
    const patternId = typeof pattern === 'string' ? pattern : pattern.getPrimary().id;
    if (!patterns[patternId]) {
        patterns[patternId] = [];
    }
    const patternPrimary = ResolvedImage.from(patternId).getPrimary().scaleSelf(pixelRatio);
    patterns[patternId].push(patternPrimary);
    return patternPrimary.serialize();
}
function hasPattern(type, layers, pixelRatio, options) {
    const patterns = options.patternDependencies;
    let hasPattern2 = false;
    for (const layer of layers) {
        const patternProperty = layer.paint.get(`${ type }-pattern`);
        if (!patternProperty.isConstant()) {
            hasPattern2 = true;
        }
        const constantPattern = patternProperty.constantOr(null);
        if (addPattern(constantPattern, patterns, pixelRatio)) {
            hasPattern2 = true;
        }
    }
    return hasPattern2;
}
function addPatternDependencies(type, layers, patternFeature, zoom, pixelRatio, options) {
    const patterns = options.patternDependencies;
    for (const layer of layers) {
        const patternProperty = layer.paint.get(`${ type }-pattern`);
        const patternPropertyValue = patternProperty.value;
        if (patternPropertyValue.kind !== 'constant') {
            let pattern = patternPropertyValue.evaluate({ zoom }, patternFeature, {}, options.availableImages);
            pattern = pattern && pattern.name ? pattern.name : pattern;
            const patternSerialized = addPattern(pattern, patterns, pixelRatio);
            if (patternSerialized) {
                patternFeature.patterns[layer.id] = patternSerialized;
            }
        }
    }
    return patternFeature;
}

const EARCUT_MAX_RINGS$1 = 500;
class FillBucket {
    constructor(options) {
        this.zoom = options.zoom;
        this.pixelRatio = options.pixelRatio;
        this.overscaling = options.overscaling;
        this.layers = options.layers;
        this.layerIds = this.layers.map(layer => layer.fqid);
        this.index = options.index;
        this.hasPattern = false;
        this.patternFeatures = [];
        this.layoutVertexArray = new StructArrayLayout2i4();
        this.indexArray = new StructArrayLayout3ui6();
        this.indexArray2 = new StructArrayLayout2ui4();
        this.programConfigurations = new ProgramConfigurationSet(options.layers, {
            zoom: options.zoom,
            lut: options.lut
        });
        this.segments = new SegmentVector();
        this.segments2 = new SegmentVector();
        this.stateDependentLayerIds = this.layers.filter(l => l.isStateDependent()).map(l => l.id);
        this.projection = options.projection;
    }
    updateFootprints(_id, _footprints) {
    }
    populate(features, options, canonical, tileTransform) {
        this.hasPattern = hasPattern('fill', this.layers, this.pixelRatio, options);
        const fillSortKey = this.layers[0].layout.get('fill-sort-key');
        const bucketFeatures = [];
        for (const {feature, id, index, sourceLayerIndex} of features) {
            const needGeometry = this.layers[0]._featureFilter.needGeometry;
            const evaluationFeature = toEvaluationFeature(feature, needGeometry);
            if (!this.layers[0]._featureFilter.filter(new EvaluationParameters(this.zoom), evaluationFeature, canonical))
                continue;
            const sortKey = fillSortKey ? fillSortKey.evaluate(evaluationFeature, {}, canonical, options.availableImages) : void 0;
            const bucketFeature = {
                id,
                properties: feature.properties,
                type: feature.type,
                sourceLayerIndex,
                index,
                geometry: needGeometry ? evaluationFeature.geometry : loadGeometry(feature, canonical, tileTransform),
                patterns: {},
                sortKey
            };
            bucketFeatures.push(bucketFeature);
        }
        if (fillSortKey) {
            bucketFeatures.sort((a, b) => {
                return a.sortKey - b.sortKey;
            });
        }
        for (const bucketFeature of bucketFeatures) {
            const {geometry, index, sourceLayerIndex} = bucketFeature;
            if (this.hasPattern) {
                const patternFeature = addPatternDependencies('fill', this.layers, bucketFeature, this.zoom, this.pixelRatio, options);
                this.patternFeatures.push(patternFeature);
            } else {
                this.addFeature(bucketFeature, geometry, index, canonical, {}, options.availableImages, options.brightness, options.elevationFeatures);
            }
            const feature = features[index].feature;
            options.featureIndex.insert(feature, geometry, index, sourceLayerIndex, this.index);
        }
    }
    update(states, vtLayer, availableImages, imagePositions, layers, isBrightnessChanged, brightness) {
        this.programConfigurations.updatePaintArrays(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness);
    }
    addFeatures(options, canonical, imagePositions, availableImages, _, brightness) {
        for (const feature of this.patternFeatures) {
            this.addFeature(feature, feature.geometry, feature.index, canonical, imagePositions, availableImages, brightness, options.elevationFeatures);
        }
    }
    isEmpty() {
        return this.layoutVertexArray.length === 0;
    }
    uploadPending() {
        return !this.uploaded || this.programConfigurations.needsUpload;
    }
    upload(context) {
        if (!this.uploaded) {
            this.layoutVertexBuffer = context.createVertexBuffer(this.layoutVertexArray, members$4);
            this.indexBuffer = context.createIndexBuffer(this.indexArray);
            this.indexBuffer2 = context.createIndexBuffer(this.indexArray2);
        }
        this.programConfigurations.upload(context);
        this.uploaded = true;
    }
    destroy() {
        if (!this.layoutVertexBuffer)
            return;
        this.layoutVertexBuffer.destroy();
        this.indexBuffer.destroy();
        this.indexBuffer2.destroy();
        this.programConfigurations.destroy();
        this.segments.destroy();
        this.segments2.destroy();
    }
    addFeature(feature, geometry, index, canonical, imagePositions, availableImages = [], brightness, elevationFeatures) {
        for (const polygon of classifyRings(geometry, EARCUT_MAX_RINGS$1)) {
            let numVertices = 0;
            for (const ring of polygon) {
                numVertices += ring.length;
            }
            const triangleSegment = this.segments.prepareSegment(numVertices, this.layoutVertexArray, this.indexArray);
            const triangleIndex = triangleSegment.vertexLength;
            const flattened = [];
            const holeIndices = [];
            for (const ring of polygon) {
                if (ring.length === 0) {
                    continue;
                }
                if (ring !== polygon[0]) {
                    holeIndices.push(flattened.length / 2);
                }
                const lineSegment = this.segments2.prepareSegment(ring.length, this.layoutVertexArray, this.indexArray2);
                const lineIndex = lineSegment.vertexLength;
                this.layoutVertexArray.emplaceBack(ring[0].x, ring[0].y);
                this.indexArray2.emplaceBack(lineIndex + ring.length - 1, lineIndex);
                flattened.push(ring[0].x);
                flattened.push(ring[0].y);
                for (let i = 1; i < ring.length; i++) {
                    this.layoutVertexArray.emplaceBack(ring[i].x, ring[i].y);
                    this.indexArray2.emplaceBack(lineIndex + i - 1, lineIndex + i);
                    flattened.push(ring[i].x);
                    flattened.push(ring[i].y);
                }
                lineSegment.vertexLength += ring.length;
                lineSegment.primitiveLength += ring.length;
            }
            const indices = earcut(flattened, holeIndices);
            for (let i = 0; i < indices.length; i += 3) {
                this.indexArray.emplaceBack(triangleIndex + indices[i], triangleIndex + indices[i + 1], triangleIndex + indices[i + 2]);
            }
            triangleSegment.vertexLength += numVertices;
            triangleSegment.primitiveLength += indices.length / 3;
        }
        this.programConfigurations.populatePaintArrays(this.layoutVertexArray.length, feature, index, imagePositions, availableImages, canonical, brightness);
    }
}
register(FillBucket, 'FillBucket', {
    omit: [
        'layers',
        'patternFeatures'
    ]
});

let layout$9;
const getLayoutProperties$9 = () => layout$9 || (layout$9 = new Properties({
    'fill-sort-key': new DataDrivenProperty(spec['layout_fill']['fill-sort-key']),
    'visibility': new DataConstantProperty(spec['layout_fill']['visibility']),
    'fill-elevation-reference': new DataConstantProperty(spec['layout_fill']['fill-elevation-reference'])
}));
let paint$a;
const getPaintProperties$a = () => paint$a || (paint$a = new Properties({
    'fill-antialias': new DataConstantProperty(spec['paint_fill']['fill-antialias']),
    'fill-opacity': new DataDrivenProperty(spec['paint_fill']['fill-opacity']),
    'fill-color': new DataDrivenProperty(spec['paint_fill']['fill-color']),
    'fill-outline-color': new DataDrivenProperty(spec['paint_fill']['fill-outline-color']),
    'fill-translate': new DataConstantProperty(spec['paint_fill']['fill-translate']),
    'fill-translate-anchor': new DataConstantProperty(spec['paint_fill']['fill-translate-anchor']),
    'fill-pattern': new DataDrivenProperty(spec['paint_fill']['fill-pattern']),
    'fill-emissive-strength': new DataConstantProperty(spec['paint_fill']['fill-emissive-strength']),
    'fill-z-offset': new DataDrivenProperty(spec['paint_fill']['fill-z-offset']),
    'fill-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    }),
    'fill-outline-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    })
}));

class FillStyleLayer extends StyleLayer {
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$9(),
            paint: getPaintProperties$a()
        };
        super(layer, properties, scope, lut, options);
    }
    getProgramIds() {
        const pattern = this.paint.get('fill-pattern');
        const image = pattern && pattern.constantOr(1);
        const ids = [image ? 'fillPattern' : 'fill'];
        if (this.paint.get('fill-antialias')) {
            ids.push(image && !this.getPaintProperty('fill-outline-color') ? 'fillOutlinePattern' : 'fillOutline');
        }
        return ids;
    }
    getDefaultProgramParams(name, zoom, lut) {
        return {
            config: new ProgramConfiguration(this, {
                zoom,
                lut
            }),
            overrideFog: false
        };
    }
    recalculate(parameters, availableImages) {
        super.recalculate(parameters, availableImages);
        const outlineColor = this.paint._values['fill-outline-color'];
        if (outlineColor.value.kind === 'constant' && outlineColor.value.value === void 0) {
            this.paint._values['fill-outline-color'] = this.paint._values['fill-color'];
        }
    }
    createBucket(parameters) {
        return new FillBucket(parameters);
    }
    queryRadius() {
        return translateDistance(this.paint.get('fill-translate'));
    }
    queryIntersectsFeature(queryGeometry, feature, featureState, geometry, zoom, transform) {
        if (queryGeometry.queryGeometry.isAboveHorizon)
            return false;
        const translatedPolygon = translate(queryGeometry.tilespaceGeometry, this.paint.get('fill-translate'), this.paint.get('fill-translate-anchor'), transform.angle, queryGeometry.pixelToTileUnitsFactor);
        return polygonIntersectsMultiPolygon(translatedPolygon, geometry);
    }
    isTileClipped() {
        return true;
    }
    is3D() {
        return this.paint.get('fill-z-offset').constantOr(1) !== 0;
    }
}

class TriangleGridIndex {
    constructor(vertices, indices, cellCount, maxCellSize) {
        this.triangleCount = indices.length / 3;
        this.min = new Point(0, 0);
        this.max = new Point(0, 0);
        this.xScale = 0;
        this.yScale = 0;
        this.cellsX = 0;
        this.cellsY = 0;
        this.cells = [];
        this.payload = [];
        if (this.triangleCount === 0 || vertices.length === 0) {
            return;
        }
        const [min, max] = [
            vertices[0].clone(),
            vertices[0].clone()
        ];
        for (let i = 1; i < vertices.length; ++i) {
            const v = vertices[i];
            min.x = Math.min(min.x, v.x);
            min.y = Math.min(min.y, v.y);
            max.x = Math.max(max.x, v.x);
            max.y = Math.max(max.y, v.y);
        }
        if (maxCellSize) {
            const optimalCellCount = Math.ceil(Math.max(max.x - min.x, max.y - min.y) / maxCellSize);
            cellCount = Math.max(cellCount, optimalCellCount);
        }
        if (cellCount === 0) {
            return;
        }
        this.min = min;
        this.max = max;
        const size = this.max.sub(this.min);
        size.x = Math.max(size.x, 1);
        size.y = Math.max(size.y, 1);
        const maxExt = Math.max(size.x, size.y);
        const cellSize = maxExt / cellCount;
        this.cellsX = Math.max(1, Math.ceil(size.x / cellSize));
        this.cellsY = Math.max(1, Math.ceil(size.y / cellSize));
        this.xScale = 1 / cellSize;
        this.yScale = 1 / cellSize;
        const associatedTriangles = [];
        for (let t = 0; t < this.triangleCount; t++) {
            const v0 = vertices[indices[t * 3 + 0]].sub(this.min);
            const v1 = vertices[indices[t * 3 + 1]].sub(this.min);
            const v2 = vertices[indices[t * 3 + 2]].sub(this.min);
            const minx = toCellIdx(Math.floor(Math.min(v0.x, v1.x, v2.x)), this.xScale, this.cellsX);
            const maxx = toCellIdx(Math.floor(Math.max(v0.x, v1.x, v2.x)), this.xScale, this.cellsX);
            const miny = toCellIdx(Math.floor(Math.min(v0.y, v1.y, v2.y)), this.yScale, this.cellsY);
            const maxy = toCellIdx(Math.floor(Math.max(v0.y, v1.y, v2.y)), this.yScale, this.cellsY);
            const c00 = new Point(0, 0);
            const c10 = new Point(0, 0);
            const c01 = new Point(0, 0);
            const c11 = new Point(0, 0);
            for (let y = miny; y <= maxy; ++y) {
                c00.y = c10.y = y * cellSize;
                c01.y = c11.y = (y + 1) * cellSize;
                for (let x = minx; x <= maxx; ++x) {
                    c00.x = c01.x = x * cellSize;
                    c10.x = c11.x = (x + 1) * cellSize;
                    if (!triangleIntersectsTriangle(v0, v1, v2, c00, c10, c11) && !triangleIntersectsTriangle(v0, v1, v2, c00, c11, c01)) {
                        continue;
                    }
                    associatedTriangles.push({
                        cellIdx: y * this.cellsX + x,
                        triIdx: t
                    });
                }
            }
        }
        if (associatedTriangles.length === 0) {
            return;
        }
        associatedTriangles.sort((a, b) => a.cellIdx - b.cellIdx || a.triIdx - b.triIdx);
        let idx = 0;
        while (idx < associatedTriangles.length) {
            const cellIdx = associatedTriangles[idx].cellIdx;
            const cell = {
                start: this.payload.length,
                len: 0
            };
            while (idx < associatedTriangles.length && associatedTriangles[idx].cellIdx === cellIdx) {
                ++cell.len;
                this.payload.push(associatedTriangles[idx++].triIdx);
            }
            this.cells[cellIdx] = cell;
        }
    }
    _lazyInitLookup() {
        if (!this.lookup) {
            this.lookup = new Uint8Array(Math.ceil(this.triangleCount / 8));
        }
        this.lookup.fill(0);
    }
    queryPoint(p, out) {
        if (this.triangleCount === 0 || this.cells.length === 0) {
            return;
        }
        if (p.x > this.max.x || this.min.x > p.x || p.y > this.max.y || this.min.y > p.y) {
            return;
        }
        const x = toCellIdx(p.x - this.min.x, this.xScale, this.cellsX);
        const y = toCellIdx(p.y - this.min.y, this.yScale, this.cellsY);
        const cell = this.cells[y * this.cellsX + x];
        if (!cell) {
            return;
        }
        this._lazyInitLookup();
        for (let i = 0; i < cell.len; i++) {
            const triIdx = this.payload[cell.start + i];
            const byte = Math.floor(triIdx / 8);
            const bit = 1 << triIdx % 8;
            if (this.lookup[byte] & bit) {
                continue;
            }
            this.lookup[byte] |= bit;
            out.push(triIdx);
            if (out.length === this.triangleCount) {
                return;
            }
        }
    }
    query(bbMin, bbMax, out) {
        if (this.triangleCount === 0 || this.cells.length === 0) {
            return;
        }
        if (bbMin.x > this.max.x || this.min.x > bbMax.x) {
            return;
        } else if (bbMin.y > this.max.y || this.min.y > bbMax.y) {
            return;
        }
        this._lazyInitLookup();
        const mnx = toCellIdx(bbMin.x - this.min.x, this.xScale, this.cellsX);
        const mxx = toCellIdx(bbMax.x - this.min.x, this.xScale, this.cellsX);
        const mny = toCellIdx(bbMin.y - this.min.y, this.yScale, this.cellsY);
        const mxy = toCellIdx(bbMax.y - this.min.y, this.yScale, this.cellsY);
        for (let y = mny; y <= mxy; y++) {
            for (let x = mnx; x <= mxx; x++) {
                const cell = this.cells[y * this.cellsX + x];
                if (!cell) {
                    continue;
                }
                for (let i = 0; i < cell.len; i++) {
                    const triIdx = this.payload[cell.start + i];
                    const byte = Math.floor(triIdx / 8);
                    const bit = 1 << triIdx % 8;
                    if (this.lookup[byte] & bit) {
                        continue;
                    }
                    this.lookup[byte] |= bit;
                    out.push(triIdx);
                    if (out.length === this.triangleCount) {
                        return;
                    }
                }
            }
        }
    }
}
function toCellIdx(p, scale, cells) {
    return Math.max(0, Math.min(cells - 1, Math.floor(p * scale)));
}
register(TriangleGridIndex, 'TriangleGridIndex');

class ClipBucket {
    constructor(options) {
        this.zoom = options.zoom;
        this.layers = options.layers;
        this.layerIds = this.layers.map(layer => layer.fqid);
        this.index = options.index;
        this.hasPattern = false;
        this.stateDependentLayerIds = this.layers.filter(l => l.isStateDependent()).map(l => l.id);
        this.footprints = [];
    }
    updateFootprints(id, footprints) {
        for (const footprint of this.footprints) {
            footprints.push({
                footprint,
                id
            });
        }
    }
    populate(features, options, canonical, tileTransform) {
        const bucketFeatures = [];
        for (const {feature, id, index, sourceLayerIndex} of features) {
            const needGeometry = this.layers[0]._featureFilter.needGeometry;
            const evaluationFeature = toEvaluationFeature(feature, needGeometry);
            if (!this.layers[0]._featureFilter.filter(new EvaluationParameters(this.zoom), evaluationFeature, canonical))
                continue;
            const bucketFeature = {
                id,
                properties: feature.properties,
                type: feature.type,
                sourceLayerIndex,
                index,
                geometry: needGeometry ? evaluationFeature.geometry : loadGeometry(feature, canonical, tileTransform),
                patterns: {}
            };
            bucketFeatures.push(bucketFeature);
        }
        for (const bucketFeature of bucketFeatures) {
            const {geometry, index, sourceLayerIndex} = bucketFeature;
            this.addFeature(bucketFeature, geometry, index, canonical, {}, options.availableImages, options.brightness);
            const feature = features[index].feature;
            options.featureIndex.insert(feature, geometry, index, sourceLayerIndex, this.index);
        }
    }
    isEmpty() {
        return this.footprints.length === 0;
    }
    uploadPending() {
        return false;
    }
    upload(_context) {
    }
    update(_states, _vtLayer, _availableImages, _imagePositions, layers, isBrightnessChanged, brightness) {
    }
    destroy() {
    }
    addFeature(feature, geometry, index, canonical, imagePositions, _availableImages = [], _brightness) {
        for (const polygon of classifyRings(geometry, 2)) {
            const points = [];
            const flattened = [];
            const holeIndices = [];
            const min = new Point(Infinity, Infinity);
            const max = new Point(-Infinity, -Infinity);
            for (const ring of polygon) {
                if (ring.length === 0) {
                    continue;
                }
                if (ring !== polygon[0]) {
                    holeIndices.push(flattened.length / 2);
                }
                for (let i = 0; i < ring.length; i++) {
                    flattened.push(ring[i].x);
                    flattened.push(ring[i].y);
                    points.push(ring[i]);
                    min.x = Math.min(min.x, ring[i].x);
                    min.y = Math.min(min.y, ring[i].y);
                    max.x = Math.max(max.x, ring[i].x);
                    max.y = Math.max(max.y, ring[i].y);
                }
            }
            const indices = earcut(flattened, holeIndices);
            const grid = new TriangleGridIndex(points, indices, 8, 256);
            this.footprints.push({
                vertices: points,
                indices,
                grid,
                min,
                max
            });
        }
    }
}
register(ClipBucket, 'ClipBucket', { omit: ['layers'] });

let layout$8;
const getLayoutProperties$8 = () => layout$8 || (layout$8 = new Properties({
    'clip-layer-types': new DataConstantProperty(spec['layout_clip']['clip-layer-types']),
    'clip-layer-scope': new DataConstantProperty(spec['layout_clip']['clip-layer-scope'])
}));
let paint$9;
const getPaintProperties$9 = () => paint$9 || (paint$9 = new Properties({}));

class ClipStyleLayer extends StyleLayer {
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$8(),
            paint: getPaintProperties$9()
        };
        super(layer, properties, scope, lut, options);
    }
    recalculate(parameters, availableImages) {
        super.recalculate(parameters, availableImages);
    }
    createBucket(parameters) {
        return new ClipBucket(parameters);
    }
    isTileClipped() {
        return true;
    }
    is3D() {
        return true;
    }
}

const fillExtrusionAttributes = createLayout([{
        name: 'a_pos_normal_ed',
        components: 4,
        type: 'Int16'
    }]);
const fillExtrusionGroundAttributes = createLayout([
    {
        name: 'a_pos_end',
        components: 4,
        type: 'Int16'
    },
    {
        name: 'a_angular_offset_factor',
        components: 1,
        type: 'Int16'
    }
]);
const centroidAttributes = createLayout([{
        name: 'a_centroid_pos',
        components: 2,
        type: 'Uint16'
    }]);
const wallAttributes = createLayout([{
        name: 'a_join_normal_inside',
        components: 3,
        type: 'Int16'
    }]);
const hiddenByLandmarkAttributes = createLayout([{
        name: 'a_hidden_by_landmark',
        components: 1,
        type: 'Uint8'
    }]);
const fillExtrusionAttributesExt = createLayout([
    {
        name: 'a_pos_3',
        components: 3,
        type: 'Int16'
    },
    {
        name: 'a_pos_normal_3',
        components: 3,
        type: 'Int16'
    }
]);
const {members: members$3, size: size$3, alignment: alignment$3} = fillExtrusionAttributes;

var vectorTile = {};

var vectortilefeature;
var hasRequiredVectortilefeature;

function requireVectortilefeature () {
	if (hasRequiredVectortilefeature) return vectortilefeature;
	hasRequiredVectortilefeature = 1;
	var Point = requirePointGeometry();
	vectortilefeature = VectorTileFeature;
	function VectorTileFeature(pbf, end, extent, keys, values) {
	    // Public
	    this.properties = {};
	    this.extent = extent;
	    this.type = 0;
	    // Private
	    this._pbf = pbf;
	    this._geometry = -1;
	    this._keys = keys;
	    this._values = values;
	    pbf.readFields(readFeature, this, end);
	}
	function readFeature(tag, feature, pbf) {
	    if (tag == 1)
	        feature.id = pbf.readVarint();
	    else if (tag == 2)
	        readTag(pbf, feature);
	    else if (tag == 3)
	        feature.type = pbf.readVarint();
	    else if (tag == 4)
	        feature._geometry = pbf.pos;
	}
	function readTag(pbf, feature) {
	    var end = pbf.readVarint() + pbf.pos;
	    while (pbf.pos < end) {
	        var key = feature._keys[pbf.readVarint()], value = feature._values[pbf.readVarint()];
	        feature.properties[key] = value;
	    }
	}
	VectorTileFeature.types = [
	    'Unknown',
	    'Point',
	    'LineString',
	    'Polygon'
	];
	VectorTileFeature.prototype.loadGeometry = function () {
	    var pbf = this._pbf;
	    pbf.pos = this._geometry;
	    var end = pbf.readVarint() + pbf.pos, cmd = 1, length = 0, x = 0, y = 0, lines = [], line;
	    while (pbf.pos < end) {
	        if (length <= 0) {
	            var cmdLen = pbf.readVarint();
	            cmd = cmdLen & 7;
	            length = cmdLen >> 3;
	        }
	        length--;
	        if (cmd === 1 || cmd === 2) {
	            x += pbf.readSVarint();
	            y += pbf.readSVarint();
	            if (cmd === 1) {
	                // moveTo
	                if (line)
	                    lines.push(line);
	                line = [];
	            }
	            line.push(new Point(x, y));
	        } else if (cmd === 7) {
	            // Workaround for https://github.com/mapbox/mapnik-vector-tile/issues/90
	            if (line) {
	                line.push(line[0].clone());    // closePolygon
	            }
	        } else {
	            throw new Error('unknown command ' + cmd);
	        }
	    }
	    if (line)
	        lines.push(line);
	    return lines;
	};
	VectorTileFeature.prototype.bbox = function () {
	    var pbf = this._pbf;
	    pbf.pos = this._geometry;
	    var end = pbf.readVarint() + pbf.pos, cmd = 1, length = 0, x = 0, y = 0, x1 = Infinity, x2 = -Infinity, y1 = Infinity, y2 = -Infinity;
	    while (pbf.pos < end) {
	        if (length <= 0) {
	            var cmdLen = pbf.readVarint();
	            cmd = cmdLen & 7;
	            length = cmdLen >> 3;
	        }
	        length--;
	        if (cmd === 1 || cmd === 2) {
	            x += pbf.readSVarint();
	            y += pbf.readSVarint();
	            if (x < x1)
	                x1 = x;
	            if (x > x2)
	                x2 = x;
	            if (y < y1)
	                y1 = y;
	            if (y > y2)
	                y2 = y;
	        } else if (cmd !== 7) {
	            throw new Error('unknown command ' + cmd);
	        }
	    }
	    return [
	        x1,
	        y1,
	        x2,
	        y2
	    ];
	};
	VectorTileFeature.prototype.toGeoJSON = function (x, y, z) {
	    var size = this.extent * Math.pow(2, z), x0 = this.extent * x, y0 = this.extent * y, coords = this.loadGeometry(), type = VectorTileFeature.types[this.type], i, j;
	    function project(line) {
	        for (var j = 0; j < line.length; j++) {
	            var p = line[j], y2 = 180 - (p.y + y0) * 360 / size;
	            line[j] = [
	                (p.x + x0) * 360 / size - 180,
	                360 / Math.PI * Math.atan(Math.exp(y2 * Math.PI / 180)) - 90
	            ];
	        }
	    }
	    switch (this.type) {
	    case 1:
	        var points = [];
	        for (i = 0; i < coords.length; i++) {
	            points[i] = coords[i][0];
	        }
	        coords = points;
	        project(coords);
	        break;
	    case 2:
	        for (i = 0; i < coords.length; i++) {
	            project(coords[i]);
	        }
	        break;
	    case 3:
	        coords = classifyRings(coords);
	        for (i = 0; i < coords.length; i++) {
	            for (j = 0; j < coords[i].length; j++) {
	                project(coords[i][j]);
	            }
	        }
	        break;
	    }
	    if (coords.length === 1) {
	        coords = coords[0];
	    } else {
	        type = 'Multi' + type;
	    }
	    var result = {
	        type: 'Feature',
	        geometry: {
	            type: type,
	            coordinates: coords
	        },
	        properties: this.properties
	    };
	    if ('id' in this) {
	        result.id = this.id;
	    }
	    return result;
	};
	// classifies an array of rings into polygons with outer rings and holes
	function classifyRings(rings) {
	    var len = rings.length;
	    if (len <= 1)
	        return [rings];
	    var polygons = [], polygon, ccw;
	    for (var i = 0; i < len; i++) {
	        var area = signedArea(rings[i]);
	        if (area === 0)
	            continue;
	        if (ccw === undefined)
	            ccw = area < 0;
	        if (ccw === area < 0) {
	            if (polygon)
	                polygons.push(polygon);
	            polygon = [rings[i]];
	        } else {
	            polygon.push(rings[i]);
	        }
	    }
	    if (polygon)
	        polygons.push(polygon);
	    return polygons;
	}
	function signedArea(ring) {
	    var sum = 0;
	    for (var i = 0, len = ring.length, j = len - 1, p1, p2; i < len; j = i++) {
	        p1 = ring[i];
	        p2 = ring[j];
	        sum += (p2.x - p1.x) * (p1.y + p2.y);
	    }
	    return sum;
	}
	return vectortilefeature;
}

var vectortilelayer;
var hasRequiredVectortilelayer;

function requireVectortilelayer () {
	if (hasRequiredVectortilelayer) return vectortilelayer;
	hasRequiredVectortilelayer = 1;
	var VectorTileFeature = requireVectortilefeature();
	vectortilelayer = VectorTileLayer;
	function VectorTileLayer(pbf, end) {
	    // Public
	    this.version = 1;
	    this.name = null;
	    this.extent = 4096;
	    this.length = 0;
	    // Private
	    this._pbf = pbf;
	    this._keys = [];
	    this._values = [];
	    this._features = [];
	    pbf.readFields(readLayer, this, end);
	    this.length = this._features.length;
	}
	function readLayer(tag, layer, pbf) {
	    if (tag === 15)
	        layer.version = pbf.readVarint();
	    else if (tag === 1)
	        layer.name = pbf.readString();
	    else if (tag === 5)
	        layer.extent = pbf.readVarint();
	    else if (tag === 2)
	        layer._features.push(pbf.pos);
	    else if (tag === 3)
	        layer._keys.push(pbf.readString());
	    else if (tag === 4)
	        layer._values.push(readValueMessage(pbf));
	}
	function readValueMessage(pbf) {
	    var value = null, end = pbf.readVarint() + pbf.pos;
	    while (pbf.pos < end) {
	        var tag = pbf.readVarint() >> 3;
	        value = tag === 1 ? pbf.readString() : tag === 2 ? pbf.readFloat() : tag === 3 ? pbf.readDouble() : tag === 4 ? pbf.readVarint64() : tag === 5 ? pbf.readVarint() : tag === 6 ? pbf.readSVarint() : tag === 7 ? pbf.readBoolean() : null;
	    }
	    return value;
	}
	// return feature `i` from this layer as a `VectorTileFeature`
	VectorTileLayer.prototype.feature = function (i) {
	    if (i < 0 || i >= this._features.length)
	        throw new Error('feature index out of bounds');
	    this._pbf.pos = this._features[i];
	    var end = this._pbf.readVarint() + this._pbf.pos;
	    return new VectorTileFeature(this._pbf, end, this.extent, this._keys, this._values);
	};
	return vectortilelayer;
}

var vectortile;
var hasRequiredVectortile;

function requireVectortile () {
	if (hasRequiredVectortile) return vectortile;
	hasRequiredVectortile = 1;
	var VectorTileLayer = requireVectortilelayer();
	vectortile = VectorTile;
	function VectorTile(pbf, end) {
	    this.layers = pbf.readFields(readTile, {}, end);
	}
	function readTile(tag, layers, pbf) {
	    if (tag === 3) {
	        var layer = new VectorTileLayer(pbf, pbf.readVarint() + pbf.pos);
	        if (layer.length)
	            layers[layer.name] = layer;
	    }
	}
	return vectortile;
}

var hasRequiredVectorTile;

function requireVectorTile () {
	if (hasRequiredVectorTile) return vectorTile;
	hasRequiredVectorTile = 1;
	vectorTile.VectorTile = requireVectortile();
	vectorTile.VectorTileFeature = requireVectortilefeature();
	vectorTile.VectorTileLayer = requireVectortilelayer();
	return vectorTile;
}

var vectorTileExports = requireVectorTile();

class Point3D extends Point {
    constructor(x, y, z) {
        super(x, y);
        this.z = z;
    }
}
class Point4D extends Point3D {
    // used for line progress and interpolated on clipping
    constructor(x, y, z, w) {
        super(x, y, z);
        this.w = w;
    }
}
function clipPolygon(polygons, clipAxis1, clipAxis2, axis) {
    const intersectX = (ring, ax, ay, bx, by, x) => {
        ring.push(new Point(x, ay + (by - ay) * ((x - ax) / (bx - ax))));
    };
    const intersectY = (ring, ax, ay, bx, by, y) => {
        ring.push(new Point(ax + (bx - ax) * ((y - ay) / (by - ay)), y));
    };
    const polygonsClipped = [];
    const intersect = axis === 0 ? intersectX : intersectY;
    for (const polygon of polygons) {
        const polygonClipped = [];
        for (const ring of polygon) {
            if (ring.length <= 2) {
                continue;
            }
            const clipped = [];
            for (let i = 0; i < ring.length - 1; i++) {
                const ax = ring[i].x;
                const ay = ring[i].y;
                const bx = ring[i + 1].x;
                const by = ring[i + 1].y;
                const a2 = axis === 0 ? ax : ay;
                const b = axis === 0 ? bx : by;
                if (a2 < clipAxis1) {
                    if (b > clipAxis1) {
                        intersect(clipped, ax, ay, bx, by, clipAxis1);
                    }
                } else if (a2 > clipAxis2) {
                    if (b < clipAxis2) {
                        intersect(clipped, ax, ay, bx, by, clipAxis2);
                    }
                } else {
                    clipped.push(ring[i]);
                }
                if (b < clipAxis1 && a2 >= clipAxis1) {
                    intersect(clipped, ax, ay, bx, by, clipAxis1);
                }
                if (b > clipAxis2 && a2 <= clipAxis2) {
                    intersect(clipped, ax, ay, bx, by, clipAxis2);
                }
            }
            let last = ring[ring.length - 1];
            const a = axis === 0 ? last.x : last.y;
            if (a >= clipAxis1 && a <= clipAxis2) {
                clipped.push(last);
            }
            if (clipped.length) {
                last = clipped[clipped.length - 1];
                if (clipped[0].x !== last.x || clipped[0].y !== last.y) {
                    clipped.push(clipped[0]);
                }
                polygonClipped.push(clipped);
            }
        }
        if (polygonClipped.length) {
            polygonsClipped.push(polygonClipped);
        }
    }
    return polygonsClipped;
}
function subdividePolygons(polygons, bounds, gridSizeX, gridSizeY, padding = 0, splitFn) {
    const outPolygons = [];
    if (!polygons.length || !gridSizeX || !gridSizeY) {
        return outPolygons;
    }
    const addResult = (clipped, bounds2) => {
        for (const polygon of clipped) {
            outPolygons.push({
                polygon,
                bounds: bounds2
            });
        }
    };
    const hSplits = Math.ceil(Math.log2(gridSizeX));
    const vSplits = Math.ceil(Math.log2(gridSizeY));
    const initialSplits = hSplits - vSplits;
    const splits = [];
    for (let i = 0; i < Math.abs(initialSplits); i++) {
        splits.push(initialSplits > 0 ? 0 : 1);
    }
    for (let i = 0; i < Math.min(hSplits, vSplits); i++) {
        splits.push(0);
        splits.push(1);
    }
    let split = polygons;
    split = clipPolygon(split, bounds[0].y - padding, bounds[1].y + padding, 1);
    split = clipPolygon(split, bounds[0].x - padding, bounds[1].x + padding, 0);
    if (!split.length) {
        return outPolygons;
    }
    const stack = [];
    if (splits.length) {
        stack.push({
            polygons: split,
            bounds,
            depth: 0
        });
    } else {
        addResult(split, bounds);
    }
    while (stack.length) {
        const frame = stack.pop();
        const depth = frame.depth;
        const axis = splits[depth];
        const bboxMin = frame.bounds[0];
        const bboxMax = frame.bounds[1];
        const splitMin = axis === 0 ? bboxMin.x : bboxMin.y;
        const splitMax = axis === 0 ? bboxMax.x : bboxMax.y;
        const splitMid = splitFn ? splitFn(axis, splitMin, splitMax) : 0.5 * (splitMin + splitMax);
        const lclip = clipPolygon(frame.polygons, splitMin - padding, splitMid + padding, axis);
        const rclip = clipPolygon(frame.polygons, splitMid - padding, splitMax + padding, axis);
        if (lclip.length) {
            const bbMaxX = axis === 0 ? splitMid : bboxMax.x;
            const bbMaxY = axis === 1 ? splitMid : bboxMax.y;
            const bbMax = new Point(bbMaxX, bbMaxY);
            const lclipBounds = [
                bboxMin,
                bbMax
            ];
            if (splits.length > depth + 1) {
                stack.push({
                    polygons: lclip,
                    bounds: lclipBounds,
                    depth: depth + 1
                });
            } else {
                addResult(lclip, lclipBounds);
            }
        }
        if (rclip.length) {
            const bbMinX = axis === 0 ? splitMid : bboxMin.x;
            const bbMinY = axis === 1 ? splitMid : bboxMin.y;
            const bbMin = new Point(bbMinX, bbMinY);
            const rclipBounds = [
                bbMin,
                bboxMax
            ];
            if (splits.length > depth + 1) {
                stack.push({
                    polygons: rclip,
                    bounds: rclipBounds,
                    depth: depth + 1
                });
            } else {
                addResult(rclip, rclipBounds);
            }
        }
    }
    return outPolygons;
}
function clipFirst(a, b, axis, clip) {
    const axis1 = axis === 'x' ? 'y' : 'x';
    const ratio = (clip - a[axis]) / (b[axis] - a[axis]);
    a[axis1] = a[axis1] + (b[axis1] - a[axis1]) * ratio;
    a[axis] = clip;
    if (a.hasOwnProperty('z')) {
        a['z'] = number(a['z'], b['z'], ratio);
    }
    if (a.hasOwnProperty('w')) {
        a['w'] = number(a['w'], b['w'], ratio);
    }
}
function clipLine$1(p0, p1, boundsMin, boundsMax) {
    const clipAxis1 = boundsMin;
    const clipAxis2 = boundsMax;
    for (const axis of [
            'x',
            'y'
        ]) {
        let a = p0;
        let b = p1;
        if (a[axis] >= b[axis]) {
            a = p1;
            b = p0;
        }
        if (a[axis] < clipAxis1 && b[axis] > clipAxis1) {
            clipFirst(a, b, axis, clipAxis1);
        }
        if (a[axis] < clipAxis2 && b[axis] > clipAxis2) {
            clipFirst(b, a, axis, clipAxis2);
        }
    }
}

const ReplacementOrderLandmark = Number.MAX_SAFE_INTEGER;
function scopeSkipsClipping(scope, scopes) {
    return scopes.length !== 0 && scopes.find(el => {
        return el === scope;
    }) === void 0;
}
function skipClipping(region, layerIndex, mask, scope) {
    return region.order < layerIndex || region.order === ReplacementOrderLandmark || !(region.clipMask & mask) || scopeSkipsClipping(scope, region.clipScope);
}
class ReplacementSource {
    constructor() {
        this._updateTime = 0;
        this._sourceIds = [];
        this._activeRegions = [];
        this._prevRegions = [];
        this._globalClipBounds = {
            min: new Point(Infinity, Infinity),
            max: new Point(-Infinity, -Infinity)
        };
    }
    clear() {
        if (this._activeRegions.length > 0) {
            ++this._updateTime;
        }
        this._activeRegions = [];
        this._prevRegions = [];
    }
    get updateTime() {
        return this._updateTime;
    }
    getReplacementRegionsForTile(id, checkAgainstGlobalClipBounds = false) {
        const tileBounds = transformAabbToMerc(new Point(0, 0), new Point(EXTENT, EXTENT), id);
        const result = [];
        if (checkAgainstGlobalClipBounds) {
            if (!regionsOverlap(tileBounds, this._globalClipBounds))
                return result;
        }
        for (const region of this._activeRegions) {
            if (region.hiddenByOverlap) {
                continue;
            }
            if (!regionsOverlap(tileBounds, region)) {
                continue;
            }
            const bounds = transformAabbToTile(region.min, region.max, id);
            result.push({
                min: bounds.min,
                max: bounds.max,
                sourceId: this._sourceIds[region.priority],
                footprint: region.footprint,
                footprintTileId: region.tileId,
                order: region.order,
                clipMask: region.clipMask,
                clipScope: region.clipScope
            });
        }
        return result;
    }
    setSources(sources) {
        this._setSources(sources.map(source => {
            return {
                getSourceId: () => {
                    return source.cache.id;
                },
                getFootprints: () => {
                    const footprints = [];
                    for (const id of source.cache.getVisibleCoordinates()) {
                        const tile = source.cache.getTile(id);
                        const bucket = tile.buckets[source.layer];
                        if (bucket) {
                            bucket.updateFootprints(id.toUnwrapped(), footprints);
                        }
                    }
                    return footprints;
                },
                getOrder: () => {
                    return source.order;
                },
                getClipMask: () => {
                    return source.clipMask;
                },
                getClipScope: () => {
                    return source.clipScope;
                }
            };
        }));
    }
    _addSource(source) {
        const footprints = source.getFootprints();
        if (footprints.length === 0) {
            return;
        }
        const order = source.getOrder();
        const clipMask = source.getClipMask();
        const clipScope = source.getClipScope();
        for (const fp of footprints) {
            if (!fp.footprint) {
                continue;
            }
            const bounds = transformAabbToMerc(fp.footprint.min, fp.footprint.max, fp.id);
            this._activeRegions.push({
                min: bounds.min,
                max: bounds.max,
                hiddenByOverlap: false,
                priority: this._sourceIds.length,
                tileId: fp.id,
                footprint: fp.footprint,
                order,
                clipMask,
                clipScope
            });
        }
        this._sourceIds.push(source.getSourceId());
    }
    _computeReplacement() {
        this._activeRegions.sort((a, b) => {
            return a.priority - b.priority || comparePoint(a.min, b.min) || comparePoint(a.max, b.max) || a.order - b.order || a.clipMask - b.clipMask || compareClipScopes(a.clipScope, b.clipScope);
        });
        let regionsChanged = this._activeRegions.length !== this._prevRegions.length;
        if (!regionsChanged) {
            let idx = 0;
            while (!regionsChanged && idx !== this._activeRegions.length) {
                const curr = this._activeRegions[idx];
                const prev = this._prevRegions[idx];
                regionsChanged = curr.priority !== prev.priority || !boundsEquals(curr, prev) || curr.order !== prev.order || (curr.clipMask !== prev.clipMask || !deepEqual(curr.clipScope, prev.clipScope));
                ++idx;
            }
        }
        if (regionsChanged) {
            ++this._updateTime;
            for (const region of this._activeRegions) {
                if (region.order !== ReplacementOrderLandmark) {
                    this._globalClipBounds.min.x = Math.min(this._globalClipBounds.min.x, region.min.x);
                    this._globalClipBounds.min.y = Math.min(this._globalClipBounds.min.y, region.min.y);
                    this._globalClipBounds.max.x = Math.max(this._globalClipBounds.max.x, region.max.x);
                    this._globalClipBounds.max.y = Math.max(this._globalClipBounds.max.y, region.max.y);
                }
            }
            const firstRegionOfNextPriority = idx => {
                const regs = this._activeRegions;
                if (idx >= regs.length) {
                    return idx;
                }
                const priority = regs[idx].priority;
                while (idx < regs.length && regs[idx].priority === priority) {
                    ++idx;
                }
                return idx;
            };
            if (this._sourceIds.length > 1) {
                let rangeBegin = 0;
                let rangeEnd = firstRegionOfNextPriority(rangeBegin);
                while (rangeBegin !== rangeEnd) {
                    let idx = rangeBegin;
                    const prevRangeEnd = rangeBegin;
                    while (idx !== rangeEnd) {
                        const active = this._activeRegions[idx];
                        active.hiddenByOverlap = false;
                        for (let prevIdx = 0; prevIdx < prevRangeEnd; prevIdx++) {
                            const prev = this._activeRegions[prevIdx];
                            if (prev.hiddenByOverlap) {
                                continue;
                            }
                            if (active.order !== ReplacementOrderLandmark) {
                                continue;
                            }
                            if (regionsOverlap(active, prev)) {
                                active.hiddenByOverlap = footprintsIntersect(active.footprint, active.tileId, prev.footprint, prev.tileId);
                                if (active.hiddenByOverlap) {
                                    break;
                                }
                            }
                        }
                        ++idx;
                    }
                    rangeBegin = rangeEnd;
                    rangeEnd = firstRegionOfNextPriority(rangeBegin);
                }
            }
        }
    }
    _setSources(sources) {
        [this._prevRegions, this._activeRegions] = [
            this._activeRegions,
            []
        ];
        this._sourceIds = [];
        for (let i = sources.length - 1; i >= 0; i--) {
            this._addSource(sources[i]);
        }
        this._computeReplacement();
    }
}
function comparePoint(a, b) {
    return a.x - b.x || a.y - b.y;
}
function compareClipScopes(a, b) {
    const concat = (t, n) => {
        return t + n;
    };
    return a.length - b.length || a.reduce(concat, '').localeCompare(b.reduce(concat, ''));
}
function boundsEquals(a, b) {
    return comparePoint(a.min, b.min) === 0 && comparePoint(a.max, b.max) === 0;
}
function regionsOverlap(a, b) {
    if (a.min.x > b.max.x || a.max.x < b.min.x)
        return false;
    else if (a.min.y > b.max.y || a.max.y < b.min.y)
        return false;
    return true;
}
function regionsEquals(a, b) {
    if (a.length !== b.length) {
        return false;
    }
    for (let i = 0; i < a.length; i++) {
        if (a[i].sourceId !== b[i].sourceId || !boundsEquals(a[i], b[i]) || a[i].order !== b[i].order || a[i].clipMask !== b[i].clipMask || !deepEqual(a[i].clipScope, b[i].clipScope)) {
            return false;
        }
    }
    return true;
}
function transformAabbToMerc(min, max, id) {
    const invExtent = 1 / EXTENT;
    const invTiles = 1 / (1 << id.canonical.z);
    const minx = (min.x * invExtent + id.canonical.x) * invTiles + id.wrap;
    const maxx = (max.x * invExtent + id.canonical.x) * invTiles + id.wrap;
    const miny = (min.y * invExtent + id.canonical.y) * invTiles;
    const maxy = (max.y * invExtent + id.canonical.y) * invTiles;
    return {
        min: new Point(minx, miny),
        max: new Point(maxx, maxy)
    };
}
function transformAabbToTile(min, max, id) {
    const tiles = 1 << id.canonical.z;
    const minx = ((min.x - id.wrap) * tiles - id.canonical.x) * EXTENT;
    const maxx = ((max.x - id.wrap) * tiles - id.canonical.x) * EXTENT;
    const miny = (min.y * tiles - id.canonical.y) * EXTENT;
    const maxy = (max.y * tiles - id.canonical.y) * EXTENT;
    return {
        min: new Point(minx, miny),
        max: new Point(maxx, maxy)
    };
}
function footprintTrianglesIntersect(footprint, vertices, indices, indexOffset, indexCount, baseVertex, padding) {
    const fpIndices = footprint.indices;
    const fpVertices = footprint.vertices;
    const candidateTriangles = [];
    for (let i = indexOffset; i < indexOffset + indexCount; i += 3) {
        const a = vertices[indices[i + 0] + baseVertex];
        const b = vertices[indices[i + 1] + baseVertex];
        const c = vertices[indices[i + 2] + baseVertex];
        const mnx = Math.min(a.x, b.x, c.x);
        const mxx = Math.max(a.x, b.x, c.x);
        const mny = Math.min(a.y, b.y, c.y);
        const mxy = Math.max(a.y, b.y, c.y);
        candidateTriangles.length = 0;
        footprint.grid.query(new Point(mnx, mny), new Point(mxx, mxy), candidateTriangles);
        for (let j = 0; j < candidateTriangles.length; j++) {
            const triIdx = candidateTriangles[j];
            const v0 = fpVertices[fpIndices[triIdx * 3 + 0]];
            const v1 = fpVertices[fpIndices[triIdx * 3 + 1]];
            const v2 = fpVertices[fpIndices[triIdx * 3 + 2]];
            if (triangleIntersectsTriangle(v0, v1, v2, a, b, c, padding)) {
                return true;
            }
        }
    }
    return false;
}
function footprintsIntersect(a, aTile, b, bTile) {
    if (!a || !b) {
        return false;
    }
    let queryVertices = a.vertices;
    if (!aTile.canonical.equals(bTile.canonical) || aTile.wrap !== bTile.wrap) {
        if (b.vertices.length < a.vertices.length) {
            return footprintsIntersect(b, bTile, a, aTile);
        }
        const srcId = aTile.canonical;
        const dstId = bTile.canonical;
        const zDiff = Math.pow(2, dstId.z - srcId.z);
        queryVertices = a.vertices.map(v => {
            const x = (v.x + srcId.x * EXTENT) * zDiff - dstId.x * EXTENT;
            const y = (v.y + srcId.y * EXTENT) * zDiff - dstId.y * EXTENT;
            return new Point(x, y);
        });
    }
    return footprintTrianglesIntersect(b, queryVertices, a.indices, 0, a.indices.length, 0, 0);
}
function transformPointToTile(x, y, src, dst) {
    const zDiff = Math.pow(2, dst.z - src.z);
    const xf = (x + src.x * EXTENT) * zDiff - dst.x * EXTENT;
    const yf = (y + src.y * EXTENT) * zDiff - dst.y * EXTENT;
    return new Point(xf, yf);
}
function pointInFootprint(p, footprint) {
    const candidateTriangles = [];
    footprint.grid.queryPoint(p, candidateTriangles);
    const fpIndices = footprint.indices;
    const fpVertices = footprint.vertices;
    for (let j = 0; j < candidateTriangles.length; j++) {
        const triIdx = candidateTriangles[j];
        const triangle = [
            fpVertices[fpIndices[triIdx * 3 + 0]],
            fpVertices[fpIndices[triIdx * 3 + 1]],
            fpVertices[fpIndices[triIdx * 3 + 2]]
        ];
        if (polygonContainsPoint(triangle, p)) {
            return true;
        }
    }
    return false;
}

function isClockWise(vertices) {
    let signedArea = 0;
    const n = vertices.length;
    for (let i = 0; i < n; i++) {
        const x1 = vertices[i].x;
        const y1 = vertices[i].y;
        const x2 = vertices[(i + 1) % n].x;
        const y2 = vertices[(i + 1) % n].y;
        signedArea += (x2 - x1) * (y2 + y1);
    }
    return signedArea >= 0;
}
function createLineWallGeometry(vertices) {
    const isPolygon = vertices[0].x === vertices[vertices.length - 1].x && vertices[0].y === vertices[vertices.length - 1].y;
    const isCW = isClockWise(vertices);
    if (!isCW) {
        vertices = vertices.reverse();
    }
    const wallGeometry = {
        geometry: [],
        joinNormals: [],
        indices: []
    };
    const innerWall = [];
    const outerWall = [];
    const joinNormals = [];
    let len = vertices.length;
    while (len >= 2 && vertices[len - 1].equals(vertices[len - 2])) {
        len--;
    }
    if (len < (isPolygon ? 3 : 2))
        return wallGeometry;
    let first = 0;
    while (first < len - 1 && vertices[first].equals(vertices[first + 1])) {
        first++;
    }
    let currentVertex;
    let prevVertex = void 0;
    let nextVertex = void 0;
    let prevNormal = void 0;
    let nextNormal = void 0;
    if (isPolygon) {
        currentVertex = vertices[len - 2];
        nextNormal = vertices[first].sub(currentVertex)._unit()._perp();
    }
    for (let i = first; i < len; i++) {
        nextVertex = i === len - 1 ? isPolygon ? vertices[first + 1] : void 0 : // if it's a polygon, treat the last vertex like the first
        vertices[i + 1];
        if (nextVertex && vertices[i].equals(nextVertex))
            continue;
        if (nextNormal)
            prevNormal = nextNormal;
        if (currentVertex)
            prevVertex = currentVertex;
        currentVertex = vertices[i];
        nextNormal = nextVertex ? nextVertex.sub(currentVertex)._unit()._perp() : prevNormal;
        prevNormal = prevNormal || nextNormal;
        let joinNormal = prevNormal.add(nextNormal);
        if (joinNormal.x !== 0 || joinNormal.y !== 0) {
            joinNormal._unit();
        }
        const cosHalfAngle = joinNormal.x * nextNormal.x + joinNormal.y * nextNormal.y;
        const miterLength = cosHalfAngle !== 0 ? 1 / cosHalfAngle : Infinity;
        const lineTurnsLeft = prevNormal.x * nextNormal.y - prevNormal.y * nextNormal.x > 0;
        let currentJoin = 'miter';
        const miterLimit = 2;
        if (currentJoin === 'miter' && miterLength > miterLimit) {
            currentJoin = 'bevel';
        }
        if (currentJoin === 'bevel') {
            if (miterLength > 100)
                currentJoin = 'flipbevel';
            if (miterLength < miterLimit)
                currentJoin = 'miter';
        }
        const addWallJoin = (vert, normal, outerOffset, innerOffset) => {
            const innerPoint = new Point(vert.x, vert.y);
            const outerPoint = new Point(vert.x, vert.y);
            innerPoint.x += normal.x * innerOffset;
            innerPoint.y += normal.y * innerOffset;
            outerPoint.x -= normal.x * Math.max(outerOffset, 1);
            outerPoint.y -= normal.y * Math.max(outerOffset, 1);
            joinNormals.push(normal);
            innerWall.push(innerPoint);
            outerWall.push(outerPoint);
        };
        if (currentJoin === 'miter') {
            joinNormal._mult(miterLength);
            addWallJoin(currentVertex, joinNormal, 0, 0);
        } else if (currentJoin === 'flipbevel') {
            joinNormal = nextNormal.mult(-1);
            addWallJoin(currentVertex, joinNormal, 0, 0);
            addWallJoin(currentVertex, joinNormal.mult(-1), 0, 0);
        } else {
            const offset = -Math.sqrt(miterLength * miterLength - 1);
            const offsetA = lineTurnsLeft ? offset : 0;
            const offsetB = lineTurnsLeft ? 0 : offset;
            if (prevVertex) {
                addWallJoin(currentVertex, prevNormal, offsetA, offsetB);
            }
            if (nextVertex) {
                addWallJoin(currentVertex, nextNormal, offsetA, offsetB);
            }
        }
    }
    wallGeometry.geometry = [
        ...innerWall,
        ...outerWall.reverse(),
        innerWall[0]
    ];
    wallGeometry.joinNormals = [
        ...joinNormals,
        ...joinNormals.reverse(),
        joinNormals[joinNormals.length - 1]
    ];
    const numPoints = wallGeometry.geometry.length - 1;
    for (let i = 0; i < numPoints / 2; i++) {
        if (i + 1 < numPoints / 2) {
            let indexA = i;
            let indexB = i + 1;
            let indexC = numPoints - 1 - i;
            let indexD = numPoints - 2 - i;
            indexA = indexA === 0 ? numPoints - 1 : indexA - 1;
            indexB = indexB === 0 ? numPoints - 1 : indexB - 1;
            indexC = indexC === 0 ? numPoints - 1 : indexC - 1;
            indexD = indexD === 0 ? numPoints - 1 : indexD - 1;
            wallGeometry.indices.push(indexC);
            wallGeometry.indices.push(indexB);
            wallGeometry.indices.push(indexA);
            wallGeometry.indices.push(indexC);
            wallGeometry.indices.push(indexD);
            wallGeometry.indices.push(indexB);
        }
    }
    return wallGeometry;
}
const tileCorners = [
    new Point(0, 0),
    new Point(EXTENT, 0),
    new Point(EXTENT, EXTENT),
    new Point(0, EXTENT)
];
function dropBufferConnectionLines(polygon, isPolygon) {
    const lineSegments = [];
    let lineSegment = [];
    if (!isPolygon || polygon.length < 2) {
        return [polygon];
    } else if (polygon.length === 2) {
        if (edgeIntersectsBox(polygon[0], polygon[1], tileCorners)) {
            return [polygon];
        }
        return [];
    } else {
        for (let i = 0; i < polygon.length + 2; i++) {
            const p0 = i === 0 ? polygon[polygon.length - 1] : polygon[(i - 1) % polygon.length];
            const p1 = polygon[i % polygon.length];
            const p2 = polygon[(i + 1) % polygon.length];
            const intersectsPrev = edgeIntersectsBox(p0, p1, tileCorners);
            const intersectsNext = edgeIntersectsBox(p1, p2, tileCorners);
            const addPoint = intersectsPrev || intersectsNext;
            if (addPoint) {
                lineSegment.push(p1);
            }
            if (!addPoint || !intersectsNext) {
                if (lineSegment.length > 0) {
                    if (lineSegment.length > 1) {
                        lineSegments.push(lineSegment);
                    }
                    lineSegment = [];
                }
            }
        }
    }
    if (lineSegment.length > 1) {
        lineSegments.push(lineSegment);
    }
    return lineSegments;
}

const vectorTileFeatureTypes$2 = vectorTileExports.VectorTileFeature.types;
const EARCUT_MAX_RINGS = 500;
const fillExtrusionDefaultDataDrivenProperties = [
    'fill-extrusion-base',
    'fill-extrusion-height',
    'fill-extrusion-color',
    'fill-extrusion-pattern',
    'fill-extrusion-flood-light-wall-radius',
    'fill-extrusion-line-width',
    'fill-extrusion-emissive-strength'
];
const fillExtrusionGroundDataDrivenProperties = ['fill-extrusion-flood-light-ground-radius'];
const FACTOR = Math.pow(2, 13);
const TANGENT_CUTOFF = 4;
const NORM = Math.pow(2, 15) - 1;
const QUAD_VERTS = 4;
const QUAD_TRIS = 2;
const TILE_REGIONS = 4;
const HIDDEN_CENTROID = new Point(0, 1);
const HIDDEN_BY_REPLACEMENT = 2147483648;
const ELEVATION_SCALE = 7;
const ELEVATION_OFFSET = 450;
function addVertex$1(vertexArray, x, y, nxRatio, nySign, normalUp, top, e) {
    vertexArray.emplaceBack(// a_pos_normal_ed:
    // Encode top and side/up normal using the least significant bits
    (x << 1) + top, (y << 1) + normalUp, // dxdy is signed, encode quadrant info using the least significant bit
    (Math.floor(nxRatio * FACTOR) << 1) + nySign, // edgedistance (used for wrapping patterns around extrusion sides)
    Math.round(e));
}
function addWallVertex(vertexArray, joinNormal, inside) {
    vertexArray.emplaceBack(// a_join_normal_inside:
    joinNormal.x * EXTENT, joinNormal.y * EXTENT, inside ? 1 : 0);
}
function addGroundVertex(vertexArray, p, q, start, bottom, angle) {
    vertexArray.emplaceBack(p.x, p.y, (q.x << 1) + start, (q.y << 1) + bottom, angle);
}
function addGlobeExtVertex(vertexArray, pos, normal) {
    const encode = 1 << 14;
    vertexArray.emplaceBack(pos.x, pos.y, pos.z, normal[0] * encode, normal[1] * encode, normal[2] * encode);
}
class FootprintSegment {
    constructor() {
        this.vertexOffset = 0;
        this.vertexCount = 0;
        this.indexOffset = 0;
        this.indexCount = 0;
    }
}
class PartData {
    constructor() {
        this.centroidXY = new Point(0, 0);
        this.vertexArrayOffset = 0;
        this.vertexCount = 0;
        this.groundVertexArrayOffset = 0;
        this.groundVertexCount = 0;
        this.flags = 0;
        this.footprintSegIdx = -1;
        this.footprintSegLen = 0;
        this.polygonSegIdx = -1;
        this.polygonSegLen = 0;
        this.min = new Point(Number.MAX_VALUE, Number.MAX_VALUE);
        this.max = new Point(-Number.MAX_VALUE, -Number.MAX_VALUE);
        this.height = 0;
    }
    span() {
        return new Point(this.max.x - this.min.x, this.max.y - this.min.y);
    }
}
class BorderCentroidData {
    constructor() {
        this.acc = new Point(0, 0);
        this.accCount = 0;
        this.centroidDataIndex = 0;
    }
    startRing(data, p) {
        if (data.min.x === Number.MAX_VALUE) {
            data.min.x = data.max.x = p.x;
            data.min.y = data.max.y = p.y;
        }
    }
    appendEdge(data, p, prev) {
        this.accCount++;
        this.acc._add(p);
        let checkBorders = !!this.borders;
        if (p.x < data.min.x) {
            data.min.x = p.x;
            checkBorders = true;
        } else if (p.x > data.max.x) {
            data.max.x = p.x;
            checkBorders = true;
        }
        if (p.y < data.min.y) {
            data.min.y = p.y;
            checkBorders = true;
        } else if (p.y > data.max.y) {
            data.max.y = p.y;
            checkBorders = true;
        }
        if (((p.x === 0 || p.x === EXTENT) && p.x === prev.x) !== ((p.y === 0 || p.y === EXTENT) && p.y === prev.y)) {
            this.processBorderOverlap(p, prev);
        }
        if (checkBorders) {
            this.checkBorderIntersection(p, prev);
        }
    }
    checkBorderIntersection(p, prev) {
        if (prev.x < 0 !== p.x < 0) {
            this.addBorderIntersection(0, number(prev.y, p.y, (0 - prev.x) / (p.x - prev.x)));
        }
        if (prev.x > EXTENT !== p.x > EXTENT) {
            this.addBorderIntersection(1, number(prev.y, p.y, (EXTENT - prev.x) / (p.x - prev.x)));
        }
        if (prev.y < 0 !== p.y < 0) {
            this.addBorderIntersection(2, number(prev.x, p.x, (0 - prev.y) / (p.y - prev.y)));
        }
        if (prev.y > EXTENT !== p.y > EXTENT) {
            this.addBorderIntersection(3, number(prev.x, p.x, (EXTENT - prev.y) / (p.y - prev.y)));
        }
    }
    addBorderIntersection(index, i) {
        if (!this.borders) {
            this.borders = [
                [
                    Number.MAX_VALUE,
                    -Number.MAX_VALUE
                ],
                [
                    Number.MAX_VALUE,
                    -Number.MAX_VALUE
                ],
                [
                    Number.MAX_VALUE,
                    -Number.MAX_VALUE
                ],
                [
                    Number.MAX_VALUE,
                    -Number.MAX_VALUE
                ]
            ];
        }
        const b = this.borders[index];
        if (i < b[0])
            b[0] = i;
        if (i > b[1])
            b[1] = i;
    }
    processBorderOverlap(p, prev) {
        if (p.x === prev.x) {
            if (p.y === prev.y)
                return;
            const index = p.x === 0 ? 0 : 1;
            this.addBorderIntersection(index, prev.y);
            this.addBorderIntersection(index, p.y);
        } else {
            const index = p.y === 0 ? 2 : 3;
            this.addBorderIntersection(index, prev.x);
            this.addBorderIntersection(index, p.x);
        }
    }
    centroid() {
        if (this.accCount === 0) {
            return new Point(0, 0);
        }
        return new Point(Math.floor(Math.max(0, this.acc.x) / this.accCount), Math.floor(Math.max(0, this.acc.y) / this.accCount));
    }
    intersectsCount() {
        if (!this.borders) {
            return 0;
        }
        return this.borders.reduce((acc, p) => acc + +(p[0] !== Number.MAX_VALUE), 0);
    }
}
function concavity(a, b) {
    return a.x * b.y - a.y * b.x < 0 ? -1 : 1;
}
function tanAngleClamped(angle) {
    return Math.min(TANGENT_CUTOFF, Math.max(-TANGENT_CUTOFF, Math.tan(angle))) / TANGENT_CUTOFF * NORM;
}
function getAngularOffsetFactor(na, nb) {
    const nm = na.add(nb)._unit();
    const cosHalfAngle = clamp(na.x * nm.x + na.y * nm.y, -1, 1);
    const factor = tanAngleClamped(Math.acos(cosHalfAngle)) * concavity(na, nb);
    return factor;
}
const borderCheck = [
    a => {
        return a.x < 0;
    },
    // left
    a => {
        return a.x > EXTENT;
    },
    // right
    a => {
        return a.y < 0;
    },
    // top
    a => {
        return a.y > EXTENT;
    }    // bottom
];
function getTileRegions(pa, pb, na, maxRadius) {
    const regions = [4];
    if (maxRadius === 0)
        return regions;
    na._mult(maxRadius);
    const c = pa.sub(na);
    const d = pb.sub(na);
    const points = [
        pa,
        pb,
        c,
        d
    ];
    for (let i = 0; i < TILE_REGIONS; i++) {
        for (const point of points) {
            if (borderCheck[i](point)) {
                regions.push(i);
                break;
            }
        }
    }
    return regions;
}
class GroundEffect {
    constructor(options) {
        this.vertexArray = new StructArrayLayout5i10();
        this.indexArray = new StructArrayLayout3ui6();
        const filtered = property => {
            return fillExtrusionGroundDataDrivenProperties.includes(property);
        };
        this.programConfigurations = new ProgramConfigurationSet(options.layers, {
            zoom: options.zoom,
            lut: options.lut
        }, filtered);
        this._segments = new SegmentVector();
        this.hiddenByLandmarkVertexArray = new StructArrayLayout1ub1();
        this._segmentToGroundQuads = {};
        this._segmentToGroundQuads[0] = [];
        this._segmentToRegionTriCounts = {};
        this._segmentToRegionTriCounts[0] = [
            0,
            0,
            0,
            0,
            0
        ];
        this.regionSegments = {};
        this.regionSegments[4] = new SegmentVector();
    }
    getDefaultSegment() {
        return this.regionSegments[4];
    }
    hasData() {
        return this.vertexArray.length !== 0;
    }
    addData(polyline, bounds, maxRadius, roundedEdges = false) {
        const n = polyline.length;
        if (n > 2) {
            let sid = Math.max(0, this._segments.get().length - 1);
            const numNewVerts = n * 4;
            const numExistingVerts = this.vertexArray.length;
            const numExistingTris = this._segmentToGroundQuads[sid].length * QUAD_TRIS;
            const segment = this._segments._prepareSegment(numNewVerts, numExistingVerts, numExistingTris);
            const newSegmentAdded = sid !== this._segments.get().length - 1;
            if (newSegmentAdded) {
                sid++;
                this._segmentToGroundQuads[sid] = [];
                this._segmentToRegionTriCounts[sid] = [
                    0,
                    0,
                    0,
                    0,
                    0
                ];
            }
            let prevFactor;
            {
                const pa = polyline[n - 1];
                const pb = polyline[0];
                const pc = polyline[1];
                const na = pb.sub(pa)._perp()._unit();
                const nb = pc.sub(pb)._perp()._unit();
                prevFactor = getAngularOffsetFactor(na, nb);
            }
            for (let i = 0; i < n; i++) {
                const j = i === n - 1 ? 0 : i + 1;
                const k = j === n - 1 ? 0 : j + 1;
                const pa = polyline[i];
                const pb = polyline[j];
                const pc = polyline[k];
                const na = pb.sub(pa)._perp()._unit();
                const nb = pc.sub(pb)._perp()._unit();
                const factor = getAngularOffsetFactor(na, nb);
                const a0 = prevFactor;
                const a1 = factor;
                if (isEdgeOutsideBounds(pa, pb, bounds) || roundedEdges && pointOutsideBounds$1(pa, bounds) && pointOutsideBounds$1(pb, bounds)) {
                    prevFactor = factor;
                    continue;
                }
                const idx = segment.vertexLength;
                addGroundVertex(this.vertexArray, pa, pb, 1, 1, a0);
                addGroundVertex(this.vertexArray, pa, pb, 1, 0, a0);
                addGroundVertex(this.vertexArray, pa, pb, 0, 1, a1);
                addGroundVertex(this.vertexArray, pa, pb, 0, 0, a1);
                segment.vertexLength += QUAD_VERTS;
                const regions = getTileRegions(pa, pb, na, maxRadius);
                for (const rid of regions) {
                    this._segmentToGroundQuads[sid].push({
                        id: idx,
                        region: rid
                    });
                    this._segmentToRegionTriCounts[sid][rid] += QUAD_TRIS;
                    segment.primitiveLength += QUAD_TRIS;
                }
                prevFactor = factor;
            }
        }
    }
    prepareBorderSegments() {
        if (!this.hasData())
            return;
        const segments = this._segments.get();
        const numSegments = segments.length;
        for (let i = 0; i < numSegments; i++) {
            const groundQuads = this._segmentToGroundQuads[i];
            groundQuads.sort((a, b) => {
                return a.region - b.region;
            });
        }
        for (let i = 0; i < numSegments; i++) {
            const groundQuads = this._segmentToGroundQuads[i];
            const segment = segments[i];
            const regionTriCounts = this._segmentToRegionTriCounts[i];
            regionTriCounts.reduce((acc, a) => {
                return acc + a;
            }, 0);
            let regionTriCountOffset = 0;
            for (let k = 0; k <= TILE_REGIONS; k++) {
                const triCount = regionTriCounts[k];
                if (triCount !== 0) {
                    let segmentVector = this.regionSegments[k];
                    if (!segmentVector) {
                        segmentVector = this.regionSegments[k] = new SegmentVector();
                    }
                    const nSegment = {
                        vertexOffset: segment.vertexOffset,
                        primitiveOffset: segment.primitiveOffset + regionTriCountOffset,
                        vertexLength: segment.vertexLength,
                        primitiveLength: triCount
                    };
                    segmentVector.get().push(nSegment);
                }
                regionTriCountOffset += triCount;
            }
            for (let j = 0; j < groundQuads.length; j++) {
                const groundQuad = groundQuads[j];
                const idx = groundQuad.id;
                this.indexArray.emplaceBack(idx, idx + 1, idx + 3);
                this.indexArray.emplaceBack(idx, idx + 3, idx + 2);
            }
        }
        this._segmentToGroundQuads = null;
        this._segmentToRegionTriCounts = null;
        this._segments.destroy();
        this._segments = null;
    }
    addPaintPropertiesData(feature, index, imagePositions, availableImages, canonical, brightness) {
        if (!this.hasData())
            return;
        this.programConfigurations.populatePaintArrays(this.vertexArray.length, feature, index, imagePositions, availableImages, canonical, brightness);
    }
    upload(context) {
        if (!this.hasData())
            return;
        this.vertexBuffer = context.createVertexBuffer(this.vertexArray, fillExtrusionGroundAttributes.members);
        this.indexBuffer = context.createIndexBuffer(this.indexArray);
    }
    uploadPaintProperties(context) {
        if (!this.hasData())
            return;
        this.programConfigurations.upload(context);
    }
    update(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness) {
        if (!this.hasData())
            return;
        this.programConfigurations.updatePaintArrays(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness);
    }
    updateHiddenByLandmark(data) {
        if (!this.hasData())
            return;
        const offset = data.groundVertexArrayOffset;
        const vertexArrayBounds = data.groundVertexCount + data.groundVertexArrayOffset;
        if (data.groundVertexCount === 0)
            return;
        const hide = data.flags & HIDDEN_BY_REPLACEMENT ? 1 : 0;
        for (let i = offset; i < vertexArrayBounds; ++i) {
            this.hiddenByLandmarkVertexArray.emplace(i, hide);
        }
        this._needsHiddenByLandmarkUpdate = true;
    }
    uploadHiddenByLandmark(context) {
        if (!this.hasData() || !this._needsHiddenByLandmarkUpdate) {
            return;
        }
        if (!this.hiddenByLandmarkVertexBuffer && this.hiddenByLandmarkVertexArray.length > 0) {
            this.hiddenByLandmarkVertexBuffer = context.createVertexBuffer(this.hiddenByLandmarkVertexArray, hiddenByLandmarkAttributes.members, true);
        } else if (this.hiddenByLandmarkVertexBuffer) {
            this.hiddenByLandmarkVertexBuffer.updateData(this.hiddenByLandmarkVertexArray);
        }
        this._needsHiddenByLandmarkUpdate = false;
    }
    destroy() {
        if (!this.vertexBuffer)
            return;
        this.vertexBuffer.destroy();
        this.indexBuffer.destroy();
        if (this.hiddenByLandmarkVertexBuffer) {
            this.hiddenByLandmarkVertexBuffer.destroy();
        }
        if (this._segments)
            this._segments.destroy();
        this.programConfigurations.destroy();
        for (let i = 0; i <= TILE_REGIONS; i++) {
            const segments = this.regionSegments[i];
            if (segments) {
                segments.destroy();
            }
        }
    }
}
class FillExtrusionBucket {
    constructor(options) {
        this.zoom = options.zoom;
        this.canonical = options.canonical;
        this.overscaling = options.overscaling;
        this.layers = options.layers;
        this.pixelRatio = options.pixelRatio;
        this.layerIds = this.layers.map(layer => layer.fqid);
        this.index = options.index;
        this.hasPattern = false;
        this.edgeRadius = 0;
        this.projection = options.projection;
        this.activeReplacements = [];
        this.replacementUpdateTime = 0;
        this.centroidData = [];
        this.footprintIndices = new StructArrayLayout3ui6();
        this.footprintVertices = new StructArrayLayout2i4();
        this.footprintSegments = [];
        this.layoutVertexArray = new StructArrayLayout4i8();
        this.centroidVertexArray = new FillExtrusionCentroidArray();
        this.wallVertexArray = new FillExtrusionWallArray();
        this.indexArray = new StructArrayLayout3ui6();
        const filtered = property => {
            return fillExtrusionDefaultDataDrivenProperties.includes(property);
        };
        this.programConfigurations = new ProgramConfigurationSet(options.layers, {
            zoom: options.zoom,
            lut: options.lut
        }, filtered);
        this.segments = new SegmentVector();
        this.stateDependentLayerIds = this.layers.filter(l => l.isStateDependent()).map(l => l.id);
        this.groundEffect = new GroundEffect(options);
        this.maxHeight = 0;
        this.partLookup = {};
        this.triangleSubSegments = [];
        this.polygonSegments = [];
    }
    updateFootprints(_id, _footprints) {
    }
    populate(features, options, canonical, tileTransform) {
        this.features = [];
        this.hasPattern = hasPattern('fill-extrusion', this.layers, this.pixelRatio, options);
        this.featuresOnBorder = [];
        this.borderFeatureIndices = [
            [],
            [],
            [],
            []
        ];
        this.borderDoneWithNeighborZ = [
            -1,
            -1,
            -1,
            -1
        ];
        this.selfDEMTileTimestamp = Number.MAX_VALUE;
        this.borderDEMTileTimestamp = [
            Number.MAX_VALUE,
            Number.MAX_VALUE,
            Number.MAX_VALUE,
            Number.MAX_VALUE
        ];
        this.tileToMeter = tileToMeter(canonical);
        this.edgeRadius = this.layers[0].layout.get('fill-extrusion-edge-radius') / this.tileToMeter;
        this.wallMode = this.layers[0].paint.get('fill-extrusion-line-width').constantOr(1) !== 0;
        for (const {feature, id, index, sourceLayerIndex} of features) {
            const needGeometry = this.layers[0]._featureFilter.needGeometry;
            const evaluationFeature = toEvaluationFeature(feature, needGeometry);
            if (!this.layers[0]._featureFilter.filter(new EvaluationParameters(this.zoom), evaluationFeature, canonical))
                continue;
            const bucketFeature = {
                id,
                sourceLayerIndex,
                index,
                geometry: needGeometry ? evaluationFeature.geometry : loadGeometry(feature, canonical, tileTransform),
                properties: feature.properties,
                type: feature.type,
                patterns: {}
            };
            const vertexArrayOffset = this.layoutVertexArray.length;
            const featureIsPolygon = vectorTileFeatureTypes$2[bucketFeature.type] === 'Polygon';
            if (this.hasPattern) {
                this.features.push(addPatternDependencies('fill-extrusion', this.layers, bucketFeature, this.zoom, this.pixelRatio, options));
            } else {
                if (this.wallMode) {
                    for (const polygon of bucketFeature.geometry) {
                        for (const line of dropBufferConnectionLines(polygon, featureIsPolygon)) {
                            this.addFeature(bucketFeature, [line], index, canonical, {}, options.availableImages, tileTransform, options.brightness);
                        }
                    }
                } else {
                    this.addFeature(bucketFeature, bucketFeature.geometry, index, canonical, {}, options.availableImages, tileTransform, options.brightness);
                }
            }
            options.featureIndex.insert(feature, bucketFeature.geometry, index, sourceLayerIndex, this.index, vertexArrayOffset);
        }
        this.sortBorders();
        if (this.projection.name === 'mercator') {
            this.splitToSubtiles();
        }
        this.groundEffect.prepareBorderSegments();
        this.polygonSegments.length = 0;
    }
    addFeatures(options, canonical, imagePositions, availableImages, tileTransform, brightness) {
        for (const feature of this.features) {
            const featureIsPolygon = vectorTileFeatureTypes$2[feature.type] === 'Polygon';
            const {geometry} = feature;
            if (this.wallMode) {
                for (const polygon of geometry) {
                    for (const line of dropBufferConnectionLines(polygon, featureIsPolygon)) {
                        this.addFeature(feature, [line], feature.index, canonical, imagePositions, availableImages, tileTransform, brightness);
                    }
                }
            } else {
                this.addFeature(feature, geometry, feature.index, canonical, imagePositions, availableImages, tileTransform, brightness);
            }
        }
        this.sortBorders();
        if (this.projection.name === 'mercator') {
            this.splitToSubtiles();
        }
    }
    update(states, vtLayer, availableImages, imagePositions, layers, isBrightnessChanged, brightness) {
        this.programConfigurations.updatePaintArrays(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness);
        this.groundEffect.update(states, vtLayer, layers, availableImages, imagePositions, isBrightnessChanged, brightness);
    }
    isEmpty() {
        return this.layoutVertexArray.length === 0;
    }
    uploadPending() {
        return !this.uploaded || this.programConfigurations.needsUpload || this.groundEffect.programConfigurations.needsUpload;
    }
    upload(context) {
        if (!this.uploaded) {
            this.layoutVertexBuffer = context.createVertexBuffer(this.layoutVertexArray, members$3);
            this.indexBuffer = context.createIndexBuffer(this.indexArray);
            this.wallVertexBuffer = context.createVertexBuffer(this.wallVertexArray, wallAttributes.members);
            if (this.layoutVertexExtArray) {
                this.layoutVertexExtBuffer = context.createVertexBuffer(this.layoutVertexExtArray, fillExtrusionAttributesExt.members, true);
            }
            this.groundEffect.upload(context);
        }
        this.groundEffect.uploadPaintProperties(context);
        this.programConfigurations.upload(context);
        this.uploaded = true;
    }
    uploadCentroid(context) {
        this.groundEffect.uploadHiddenByLandmark(context);
        if (!this.needsCentroidUpdate) {
            return;
        }
        if (!this.centroidVertexBuffer && this.centroidVertexArray.length > 0) {
            this.centroidVertexBuffer = context.createVertexBuffer(this.centroidVertexArray, centroidAttributes.members, true);
        } else if (this.centroidVertexBuffer) {
            this.centroidVertexBuffer.updateData(this.centroidVertexArray);
        }
        this.needsCentroidUpdate = false;
    }
    destroy() {
        if (!this.layoutVertexBuffer)
            return;
        this.layoutVertexBuffer.destroy();
        if (this.centroidVertexBuffer) {
            this.centroidVertexBuffer.destroy();
        }
        if (this.layoutVertexExtBuffer) {
            this.layoutVertexExtBuffer.destroy();
        }
        this.groundEffect.destroy();
        this.indexBuffer.destroy();
        this.programConfigurations.destroy();
        this.segments.destroy();
    }
    addFeature(feature, geometry, index, canonical, imagePositions, availableImages, tileTransform, brightness) {
        const floodLightRadius = this.layers[0].paint.get('fill-extrusion-flood-light-ground-radius').evaluate(feature, {});
        const maxRadius = floodLightRadius / this.tileToMeter;
        const tileBounds = [
            new Point(0, 0),
            new Point(EXTENT, EXTENT)
        ];
        const projection = tileTransform.projection;
        const isGlobe = projection.name === 'globe';
        const isPolygon = this.wallMode || vectorTileFeatureTypes$2[feature.type] === 'Polygon';
        const borderCentroidData = new BorderCentroidData();
        borderCentroidData.centroidDataIndex = this.centroidData.length;
        const centroid = new PartData();
        const base = this.layers[0].paint.get('fill-extrusion-base').evaluate(feature, {}, canonical);
        const onGround = base <= 0;
        const height = this.layers[0].paint.get('fill-extrusion-height').evaluate(feature, {}, canonical);
        centroid.height = height;
        centroid.vertexArrayOffset = this.layoutVertexArray.length;
        centroid.groundVertexArrayOffset = this.groundEffect.vertexArray.length;
        if (isGlobe && !this.layoutVertexExtArray) {
            this.layoutVertexExtArray = new StructArrayLayout6i12();
        }
        let wallGeometry;
        if (this.wallMode) {
            if (isGlobe) {
                warnOnce('Non zero fill-extrusion-line-width is not yet supported on globe.');
                return;
            }
            if (geometry.length !== 1) {
                return;
            }
            wallGeometry = createLineWallGeometry(geometry[0]);
            geometry = [wallGeometry.geometry];
        }
        const isPointOnInnerWall = (index2, polygon) => {
            return index2 < (polygon.length - 1) / 2 || index2 === polygon.length - 1;
        };
        const polygons = this.wallMode ? [geometry] : classifyRings(geometry, EARCUT_MAX_RINGS);
        for (let i = polygons.length - 1; i >= 0; i--) {
            const polygon = polygons[i];
            if (polygon.length === 0 || isEntirelyOutside(polygon[0])) {
                polygons.splice(i, 1);
            }
        }
        let clippedPolygons;
        if (isGlobe) {
            clippedPolygons = resampleFillExtrusionPolygonsForGlobe(polygons, tileBounds, canonical);
        } else {
            clippedPolygons = [];
            for (const polygon of polygons) {
                clippedPolygons.push({
                    polygon,
                    bounds: tileBounds
                });
            }
        }
        const edgeRadius = isPolygon ? this.edgeRadius : 0;
        const optimiseGround = edgeRadius > 0 && this.zoom < 17;
        const isDuplicate = (coords, a) => {
            if (coords.length === 0)
                return false;
            const b = coords[coords.length - 1];
            return a.x === b.x && a.y === b.y;
        };
        for (const {polygon, bounds} of clippedPolygons) {
            let topIndex = 0;
            let numVertices = 0;
            for (const ring of polygon) {
                if (isPolygon && !ring[0].equals(ring[ring.length - 1]))
                    ring.push(ring[0]);
                numVertices += isPolygon ? ring.length - 1 : ring.length;
            }
            const segment = this.segments.prepareSegment((isPolygon ? 5 : 4) * numVertices, this.layoutVertexArray, this.indexArray);
            if (centroid.footprintSegIdx < 0) {
                centroid.footprintSegIdx = this.footprintSegments.length;
            }
            if (centroid.polygonSegIdx < 0) {
                centroid.polygonSegIdx = this.polygonSegments.length;
            }
            const polygonSeg = {
                triangleArrayOffset: this.indexArray.length,
                triangleCount: 0,
                triangleSegIdx: this.segments.segments.length - 1
            };
            const fpSegment = new FootprintSegment();
            fpSegment.vertexOffset = this.footprintVertices.length;
            fpSegment.indexOffset = this.footprintIndices.length * 3;
            fpSegment.ringIndices = [];
            if (isPolygon) {
                const flattened = [];
                const holeIndices = [];
                topIndex = segment.vertexLength;
                for (let r = 0; r < polygon.length; r++) {
                    const ring = polygon[r];
                    if (ring.length && r !== 0) {
                        holeIndices.push(flattened.length / 2);
                    }
                    const groundPolyline = [];
                    let na, nb;
                    {
                        const p0 = ring[0];
                        const p1 = ring[1];
                        na = p1.sub(p0)._perp()._unit();
                    }
                    fpSegment.ringIndices.push(ring.length - 1);
                    for (let i = 1; i < ring.length; i++) {
                        const p1 = ring[i];
                        const p2 = ring[i === ring.length - 1 ? 1 : i + 1];
                        const q = p1.clone();
                        if (edgeRadius) {
                            nb = p2.sub(p1)._perp()._unit();
                            const nm = na.add(nb)._unit();
                            const cosHalfAngle = na.x * nm.x + na.y * nm.y;
                            const offset = edgeRadius * Math.min(4, 1 / cosHalfAngle);
                            q.x += offset * nm.x;
                            q.y += offset * nm.y;
                            q.x = Math.round(q.x);
                            q.y = Math.round(q.y);
                            na = nb;
                        }
                        if (onGround && (edgeRadius === 0 || optimiseGround) && !isDuplicate(groundPolyline, q)) {
                            groundPolyline.push(q);
                        }
                        addVertex$1(this.layoutVertexArray, q.x, q.y, 0, 0, 1, 1, 0);
                        if (this.wallMode) {
                            const isInside = isPointOnInnerWall(i, ring);
                            const joinNormal = wallGeometry.joinNormals[i];
                            addWallVertex(this.wallVertexArray, joinNormal, !isInside);
                        }
                        segment.vertexLength++;
                        this.footprintVertices.emplaceBack(p1.x, p1.y);
                        flattened.push(p1.x, p1.y);
                        if (isGlobe) {
                            const array = this.layoutVertexExtArray;
                            const projectedP = projection.projectTilePoint(q.x, q.y, canonical);
                            const n = projection.upVector(canonical, q.x, q.y);
                            addGlobeExtVertex(array, projectedP, n);
                        }
                    }
                    if (onGround && (edgeRadius === 0 || optimiseGround)) {
                        if (groundPolyline.length !== 0 && isDuplicate(groundPolyline, groundPolyline[0])) {
                            groundPolyline.pop();
                        }
                        this.groundEffect.addData(groundPolyline, bounds, maxRadius);
                    }
                }
                const indices = this.wallMode ? wallGeometry.indices : earcut(flattened, holeIndices);
                for (let j = 0; j < indices.length; j += 3) {
                    this.footprintIndices.emplaceBack(fpSegment.vertexOffset + indices[j + 0], fpSegment.vertexOffset + indices[j + 1], fpSegment.vertexOffset + indices[j + 2]);
                    this.indexArray.emplaceBack(topIndex + indices[j], topIndex + indices[j + 2], topIndex + indices[j + 1]);
                    segment.primitiveLength++;
                }
                fpSegment.indexCount += indices.length;
                fpSegment.vertexCount += this.footprintVertices.length - fpSegment.vertexOffset;
            }
            for (let r = 0; r < polygon.length; r++) {
                const ring = polygon[r];
                borderCentroidData.startRing(centroid, ring[0]);
                let isPrevCornerConcave = ring.length > 4 && isAOConcaveAngle(ring[ring.length - 2], ring[0], ring[1]);
                let offsetPrev = edgeRadius ? getRoundedEdgeOffset(ring[ring.length - 2], ring[0], ring[1], edgeRadius) : 0;
                const groundPolyline = [];
                let kFirst;
                let na, nb;
                {
                    const p0 = ring[0];
                    const p1 = ring[1];
                    na = p1.sub(p0)._perp()._unit();
                }
                let cap = true;
                for (let i = 1, edgeDistance = 0; i < ring.length; i++) {
                    let p0 = ring[i - 1];
                    let p1 = ring[i];
                    const p2 = ring[i === ring.length - 1 ? 1 : i + 1];
                    borderCentroidData.appendEdge(centroid, p1, p0);
                    if (isEdgeOutsideBounds(p1, p0, bounds)) {
                        if (edgeRadius) {
                            na = p2.sub(p1)._perp()._unit();
                            cap = !cap;
                        }
                        continue;
                    }
                    const d = p1.sub(p0)._perp();
                    const nxRatio = d.x / (Math.abs(d.x) + Math.abs(d.y));
                    const nySign = d.y > 0 ? 1 : 0;
                    const dist = p0.dist(p1);
                    if (edgeDistance + dist > 32768)
                        edgeDistance = 0;
                    if (edgeRadius) {
                        nb = p2.sub(p1)._perp()._unit();
                        const cosHalfAngle = getCosHalfAngle(na, nb);
                        let offsetNext = _getRoundedEdgeOffset(p0, p1, p2, cosHalfAngle, edgeRadius);
                        if (isNaN(offsetNext))
                            offsetNext = 0;
                        const nEdge = p1.sub(p0)._unit();
                        p0 = p0.add(nEdge.mult(offsetPrev))._round();
                        p1 = p1.add(nEdge.mult(-offsetNext))._round();
                        offsetPrev = offsetNext;
                        na = nb;
                        if (onGround && this.zoom >= 17) {
                            if (!isDuplicate(groundPolyline, p0))
                                groundPolyline.push(p0);
                            if (!isDuplicate(groundPolyline, p1))
                                groundPolyline.push(p1);
                        }
                    }
                    const k = segment.vertexLength;
                    const isConcaveCorner = ring.length > 4 && isAOConcaveAngle(p0, p1, p2);
                    let encodedEdgeDistance = encodeAOToEdgeDistance(edgeDistance, isPrevCornerConcave, cap);
                    addVertex$1(this.layoutVertexArray, p0.x, p0.y, nxRatio, nySign, 0, 0, encodedEdgeDistance);
                    addVertex$1(this.layoutVertexArray, p0.x, p0.y, nxRatio, nySign, 0, 1, encodedEdgeDistance);
                    if (this.wallMode) {
                        const isInside = isPointOnInnerWall(i - 1, ring);
                        const joinNormal = wallGeometry.joinNormals[i - 1];
                        addWallVertex(this.wallVertexArray, joinNormal, isInside);
                        addWallVertex(this.wallVertexArray, joinNormal, isInside);
                    }
                    edgeDistance += dist;
                    encodedEdgeDistance = encodeAOToEdgeDistance(edgeDistance, isConcaveCorner, !cap);
                    isPrevCornerConcave = isConcaveCorner;
                    addVertex$1(this.layoutVertexArray, p1.x, p1.y, nxRatio, nySign, 0, 0, encodedEdgeDistance);
                    addVertex$1(this.layoutVertexArray, p1.x, p1.y, nxRatio, nySign, 0, 1, encodedEdgeDistance);
                    if (this.wallMode) {
                        const isInside = isPointOnInnerWall(i, ring);
                        const joinNormal = wallGeometry.joinNormals[i];
                        addWallVertex(this.wallVertexArray, joinNormal, isInside);
                        addWallVertex(this.wallVertexArray, joinNormal, isInside);
                    }
                    segment.vertexLength += 4;
                    this.indexArray.emplaceBack(k + 0, k + 1, k + 2);
                    this.indexArray.emplaceBack(k + 1, k + 3, k + 2);
                    segment.primitiveLength += 2;
                    if (edgeRadius) {
                        const t0 = topIndex + (i === 1 ? ring.length - 2 : i - 2);
                        const t1 = i === 1 ? topIndex : t0 + 1;
                        this.indexArray.emplaceBack(k + 1, t0, k + 3);
                        this.indexArray.emplaceBack(t0, t1, k + 3);
                        segment.primitiveLength += 2;
                        if (kFirst === void 0) {
                            kFirst = k;
                        }
                        if (!isEdgeOutsideBounds(p2, ring[i], bounds)) {
                            const l = i === ring.length - 1 ? kFirst : segment.vertexLength;
                            this.indexArray.emplaceBack(k + 2, k + 3, l);
                            this.indexArray.emplaceBack(k + 3, l + 1, l);
                            this.indexArray.emplaceBack(k + 3, t1, l + 1);
                            segment.primitiveLength += 3;
                        }
                        cap = !cap;
                    }
                    if (isGlobe) {
                        const array = this.layoutVertexExtArray;
                        const projectedP0 = projection.projectTilePoint(p0.x, p0.y, canonical);
                        const projectedP1 = projection.projectTilePoint(p1.x, p1.y, canonical);
                        const n0 = projection.upVector(canonical, p0.x, p0.y);
                        const n1 = projection.upVector(canonical, p1.x, p1.y);
                        addGlobeExtVertex(array, projectedP0, n0);
                        addGlobeExtVertex(array, projectedP0, n0);
                        addGlobeExtVertex(array, projectedP1, n1);
                        addGlobeExtVertex(array, projectedP1, n1);
                    }
                }
                if (isPolygon)
                    topIndex += ring.length - 1;
                if (onGround && edgeRadius && this.zoom >= 17) {
                    if (groundPolyline.length !== 0 && isDuplicate(groundPolyline, groundPolyline[0])) {
                        groundPolyline.pop();
                    }
                    this.groundEffect.addData(groundPolyline, bounds, maxRadius, edgeRadius > 0);
                }
            }
            this.footprintSegments.push(fpSegment);
            polygonSeg.triangleCount = this.indexArray.length - polygonSeg.triangleArrayOffset;
            this.polygonSegments.push(polygonSeg);
            ++centroid.footprintSegLen;
            ++centroid.polygonSegLen;
        }
        centroid.vertexCount = this.layoutVertexArray.length - centroid.vertexArrayOffset;
        centroid.groundVertexCount = this.groundEffect.vertexArray.length - centroid.groundVertexArrayOffset;
        if (centroid.vertexCount === 0) {
            return;
        }
        centroid.centroidXY = borderCentroidData.borders ? HIDDEN_CENTROID : this.encodeCentroid(borderCentroidData, centroid);
        this.centroidData.push(centroid);
        if (borderCentroidData.borders) {
            this.featuresOnBorder.push(borderCentroidData);
            const borderIndex = this.featuresOnBorder.length - 1;
            for (let i = 0; i < borderCentroidData.borders.length; i++) {
                if (borderCentroidData.borders[i][0] !== Number.MAX_VALUE) {
                    this.borderFeatureIndices[i].push(borderIndex);
                }
            }
        }
        this.programConfigurations.populatePaintArrays(this.layoutVertexArray.length, feature, index, imagePositions, availableImages, canonical, brightness);
        this.groundEffect.addPaintPropertiesData(feature, index, imagePositions, availableImages, canonical, brightness);
        this.maxHeight = Math.max(this.maxHeight, height);
    }
    sortBorders() {
        for (let i = 0; i < this.borderFeatureIndices.length; i++) {
            const borders = this.borderFeatureIndices[i];
            borders.sort((a, b) => this.featuresOnBorder[a].borders[i][0] - this.featuresOnBorder[b].borders[i][0]);
        }
    }
    splitToSubtiles() {
        const segmentedFeatures = [];
        for (let centroidIdx = 0; centroidIdx < this.centroidData.length; centroidIdx++) {
            const part = this.centroidData[centroidIdx];
            const right = +(part.min.x + part.max.x > EXTENT);
            const bottom = +(part.min.y + part.max.y > EXTENT);
            const subtile = bottom * 2 + (right ^ bottom);
            for (let i = 0; i < part.polygonSegLen; i++) {
                const polySegIdx = part.polygonSegIdx + i;
                segmentedFeatures.push({
                    centroidIdx,
                    subtile,
                    polygonSegmentIdx: polySegIdx,
                    triangleSegmentIdx: this.polygonSegments[polySegIdx].triangleSegIdx
                });
            }
        }
        const sortedTriangles = new StructArrayLayout3ui6();
        segmentedFeatures.sort((a, b) => a.triangleSegmentIdx === b.triangleSegmentIdx ? a.subtile - b.subtile : a.triangleSegmentIdx - b.triangleSegmentIdx);
        let segmentIdx = 0;
        let segmentBeginIndex = 0;
        let segmentEndIndex = 0;
        for (const segmentedFeature of segmentedFeatures) {
            if (segmentedFeature.triangleSegmentIdx !== segmentIdx) {
                break;
            }
            segmentEndIndex++;
        }
        const segmentedFeaturesEndIndex = segmentedFeatures.length;
        while (segmentBeginIndex !== segmentedFeatures.length) {
            segmentIdx = segmentedFeatures[segmentBeginIndex].triangleSegmentIdx;
            let subTileIdx = 0;
            let featuresBeginIndex = segmentBeginIndex;
            let featuresEndIndex = segmentBeginIndex;
            for (let seg = featuresBeginIndex; seg < segmentEndIndex; seg++) {
                if (segmentedFeatures[seg].subtile !== subTileIdx) {
                    break;
                }
                featuresEndIndex++;
            }
            while (featuresBeginIndex !== segmentEndIndex) {
                const featuresBegin = segmentedFeatures[featuresBeginIndex];
                subTileIdx = featuresBegin.subtile;
                const subtileMin = this.centroidData[featuresBegin.centroidIdx].min.clone();
                const subtileMax = this.centroidData[featuresBegin.centroidIdx].max.clone();
                const segment = {
                    vertexOffset: this.segments.segments[segmentIdx].vertexOffset,
                    primitiveOffset: sortedTriangles.length,
                    vertexLength: this.segments.segments[segmentIdx].vertexLength,
                    primitiveLength: 0,
                    sortKey: void 0,
                    vaos: {}
                };
                for (let featureIdx = featuresBeginIndex; featureIdx < featuresEndIndex; featureIdx++) {
                    const feature = segmentedFeatures[featureIdx];
                    const data = this.polygonSegments[feature.polygonSegmentIdx];
                    const centroidMin = this.centroidData[feature.centroidIdx].min;
                    const centroidMax = this.centroidData[feature.centroidIdx].max;
                    const iArray = this.indexArray.uint16;
                    for (let i = data.triangleArrayOffset; i < data.triangleArrayOffset + data.triangleCount; i++) {
                        sortedTriangles.emplaceBack(iArray[i * 3], iArray[i * 3 + 1], iArray[i * 3 + 2]);
                    }
                    segment.primitiveLength += data.triangleCount;
                    subtileMin.x = Math.min(subtileMin.x, centroidMin.x);
                    subtileMin.y = Math.min(subtileMin.y, centroidMin.y);
                    subtileMax.x = Math.max(subtileMax.x, centroidMax.x);
                    subtileMax.y = Math.max(subtileMax.y, centroidMax.y);
                }
                if (segment.primitiveLength > 0) {
                    this.triangleSubSegments.push({
                        segment,
                        min: subtileMin,
                        max: subtileMax
                    });
                }
                featuresBeginIndex = featuresEndIndex;
                for (let seg = featuresBeginIndex; seg < segmentEndIndex; seg++) {
                    if (segmentedFeatures[seg].subtile !== segmentedFeatures[featuresBeginIndex].subtile) {
                        break;
                    }
                    featuresEndIndex++;
                }
            }
            segmentBeginIndex = segmentEndIndex;
            for (let seg = segmentBeginIndex; seg < segmentedFeaturesEndIndex; seg++) {
                if (segmentedFeatures[seg].triangleSegmentIdx !== segmentedFeatures[segmentBeginIndex].triangleSegmentIdx) {
                    break;
                }
                segmentEndIndex++;
            }
        }
        sortedTriangles._trim();
        this.indexArray = sortedTriangles;
    }
    getVisibleSegments(renderId, elevation, frustum) {
        const outSegments = new SegmentVector();
        if (this.wallMode) {
            for (const subSegment of this.triangleSubSegments) {
                outSegments.segments.push(subSegment.segment);
            }
            return outSegments;
        }
        let minZ = 0;
        let maxZ = 0;
        const tiles = 1 << renderId.canonical.z;
        if (elevation) {
            const minmax = elevation.getMinMaxForTile(renderId);
            if (minmax) {
                minZ = minmax.min;
                maxZ = minmax.max;
            }
        }
        maxZ += this.maxHeight;
        const id = renderId.toUnwrapped();
        let activeSegment;
        const tileMin = [
            id.canonical.x / tiles + id.wrap,
            id.canonical.y / tiles
        ];
        const tileMax = [
            (id.canonical.x + 1) / tiles + id.wrap,
            (id.canonical.y + 1) / tiles
        ];
        const mix = (a, b, c) => {
            return [
                a[0] * (1 - c[0]) + b[0] * c[0],
                a[1] * (1 - c[1]) + b[1] * c[1]
            ];
        };
        const fracMin = [];
        const fracMax = [];
        for (const subSegment of this.triangleSubSegments) {
            fracMin[0] = subSegment.min.x / EXTENT;
            fracMin[1] = subSegment.min.y / EXTENT;
            fracMax[0] = subSegment.max.x / EXTENT;
            fracMax[1] = subSegment.max.y / EXTENT;
            const aabbMin = mix(tileMin, tileMax, fracMin);
            const aabbMax = mix(tileMin, tileMax, fracMax);
            const aabb = new Aabb([
                aabbMin[0],
                aabbMin[1],
                minZ
            ], [
                aabbMax[0],
                aabbMax[1],
                maxZ
            ]);
            if (aabb.intersectsPrecise(frustum) === 0) {
                if (activeSegment) {
                    outSegments.segments.push(activeSegment);
                    activeSegment = void 0;
                }
                continue;
            }
            const renderSegment = subSegment.segment;
            if (activeSegment && activeSegment.vertexOffset !== renderSegment.vertexOffset) {
                outSegments.segments.push(activeSegment);
                activeSegment = void 0;
            }
            if (!activeSegment) {
                activeSegment = {
                    vertexOffset: renderSegment.vertexOffset,
                    primitiveLength: renderSegment.primitiveLength,
                    vertexLength: renderSegment.vertexLength,
                    primitiveOffset: renderSegment.primitiveOffset,
                    sortKey: void 0,
                    vaos: {}
                };
            } else {
                activeSegment.vertexLength += renderSegment.vertexLength;
                activeSegment.primitiveLength += renderSegment.primitiveLength;
            }
        }
        if (activeSegment) {
            outSegments.segments.push(activeSegment);
        }
        return outSegments;
    }
    // Encoded centroid x and y:
    //     x     y
    // ---------------------------------------------
    //     0     0    Default, no flat roof.
    //     0     1    Hide, used to hide parts of buildings on border while expecting the other side to get loaded
    //    >0     0    Elevation encoded to uint16 word
    //    >0    >0    Encoded centroid position and x & y span
    encodeCentroid(borderCentroidData, data) {
        const c = borderCentroidData.centroid();
        const span = data.span();
        const spanX = Math.min(7, Math.round(span.x * this.tileToMeter / 10));
        const spanY = Math.min(7, Math.round(span.y * this.tileToMeter / 10));
        return new Point(clamp(c.x, 1, EXTENT - 1) << 3 | spanX, clamp(c.y, 1, EXTENT - 1) << 3 | spanY);
    }
    // Border centroid data is unreliable for elevating parts split on tile borders.
    // It is used only for synchronous lowering of splits as the centroid (not the size information in split parts) is consistent.
    encodeBorderCentroid(borderCentroidData) {
        if (!borderCentroidData.borders)
            return new Point(0, 0);
        const b = borderCentroidData.borders;
        const notOnBorder = Number.MAX_VALUE;
        const span = 6;
        if (b[0][0] !== notOnBorder || b[1][0] !== notOnBorder) {
            const x = (b[0][0] !== notOnBorder ? 0 : 8191 << 3) | span;
            const index = b[0][0] !== notOnBorder ? 0 : 1;
            return new Point(x, ((b[index][0] + b[index][1]) / 2 | 0) << 3 | span);
        } else {
            const y = (b[2][0] !== notOnBorder ? 0 : 8191 << 3) | span;
            const index = b[2][0] !== notOnBorder ? 2 : 3;
            return new Point(((b[index][0] + b[index][1]) / 2 | 0) << 3 | span, y);
        }
    }
    showCentroid(borderCentroidData) {
        const c = this.centroidData[borderCentroidData.centroidDataIndex];
        c.flags &= HIDDEN_BY_REPLACEMENT;
        c.centroidXY.x = 0;
        c.centroidXY.y = 0;
        this.writeCentroidToBuffer(c);
    }
    writeCentroidToBuffer(data) {
        this.groundEffect.updateHiddenByLandmark(data);
        const offset = data.vertexArrayOffset;
        const vertexArrayBounds = data.vertexCount + data.vertexArrayOffset;
        const c = data.flags & HIDDEN_BY_REPLACEMENT ? HIDDEN_CENTROID : data.centroidXY;
        const firstX = this.centroidVertexArray.geta_centroid_pos0(offset);
        const firstY = this.centroidVertexArray.geta_centroid_pos1(offset);
        if (firstY === c.y && firstX === c.x) {
            return;
        }
        for (let i = offset; i < vertexArrayBounds; ++i) {
            this.centroidVertexArray.emplace(i, c.x, c.y);
        }
        this.needsCentroidUpdate = true;
    }
    createCentroidsBuffer() {
        this.centroidVertexArray.resize(this.layoutVertexArray.length);
        this.groundEffect.hiddenByLandmarkVertexArray.resize(this.groundEffect.vertexArray.length);
        for (const centroid of this.centroidData) {
            this.writeCentroidToBuffer(centroid);
        }
    }
    updateReplacement(coord, source, layerIndex) {
        if (source.updateTime === this.replacementUpdateTime) {
            return;
        }
        this.replacementUpdateTime = source.updateTime;
        const newReplacements = source.getReplacementRegionsForTile(coord.toUnwrapped());
        if (regionsEquals(this.activeReplacements, newReplacements)) {
            return;
        }
        this.activeReplacements = newReplacements;
        if (this.centroidVertexArray.length === 0) {
            this.createCentroidsBuffer();
        } else {
            for (const centroid of this.centroidData) {
                centroid.flags &= ~HIDDEN_BY_REPLACEMENT;
            }
        }
        const transformedVertices = [];
        for (const region of this.activeReplacements) {
            if (region.order < layerIndex)
                continue;
            const padding = Math.max(1, Math.pow(2, region.footprintTileId.canonical.z - coord.canonical.z));
            for (const centroid of this.centroidData) {
                if (centroid.flags & HIDDEN_BY_REPLACEMENT) {
                    continue;
                }
                if (region.min.x > centroid.max.x || centroid.min.x > region.max.x) {
                    continue;
                } else if (region.min.y > centroid.max.y || centroid.min.y > region.max.y) {
                    continue;
                }
                for (let i = 0; i < centroid.footprintSegLen; i++) {
                    const seg = this.footprintSegments[centroid.footprintSegIdx + i];
                    transformedVertices.length = 0;
                    transformFootprintVertices(this.footprintVertices, seg.vertexOffset, seg.vertexCount, region.footprintTileId.canonical, coord.canonical, transformedVertices);
                    if (footprintTrianglesIntersect(region.footprint, transformedVertices, this.footprintIndices.uint16, seg.indexOffset, seg.indexCount, -seg.vertexOffset, -padding)) {
                        centroid.flags |= HIDDEN_BY_REPLACEMENT;
                        break;
                    }
                }
            }
        }
        for (const centroid of this.centroidData) {
            this.writeCentroidToBuffer(centroid);
        }
        this.borderDoneWithNeighborZ = [
            -1,
            -1,
            -1,
            -1
        ];
    }
    footprintContainsPoint(x, y, centroid) {
        let c = false;
        for (let s = 0; s < centroid.footprintSegLen; s++) {
            const seg = this.footprintSegments[centroid.footprintSegIdx + s];
            let startRing = 0;
            for (const endRing of seg.ringIndices) {
                for (let i = startRing, j = endRing + startRing - 1; i < endRing + startRing; j = i++) {
                    const x1 = this.footprintVertices.int16[(i + seg.vertexOffset) * 2 + 0];
                    const y1 = this.footprintVertices.int16[(i + seg.vertexOffset) * 2 + 1];
                    const x2 = this.footprintVertices.int16[(j + seg.vertexOffset) * 2 + 0];
                    const y2 = this.footprintVertices.int16[(j + seg.vertexOffset) * 2 + 1];
                    if (y1 > y !== y2 > y && x < (x2 - x1) * (y - y1) / (y2 - y1) + x1) {
                        c = !c;
                    }
                }
                startRing = endRing;
            }
        }
        return c;
    }
    getHeightAtTileCoord(x, y) {
        let height = Number.NEGATIVE_INFINITY;
        let hidden = true;
        const lookupKey = (x + EXTENT) * 4 * EXTENT + (y + EXTENT);
        if (this.partLookup.hasOwnProperty(lookupKey)) {
            const centroid = this.partLookup[lookupKey];
            return centroid ? {
                height: centroid.height,
                hidden: !!(centroid.flags & HIDDEN_BY_REPLACEMENT)
            } : void 0;
        }
        for (const centroid of this.centroidData) {
            if (x > centroid.max.x || centroid.min.x > x || y > centroid.max.y || centroid.min.y > y) {
                continue;
            }
            if (this.footprintContainsPoint(x, y, centroid)) {
                if (centroid && centroid.height > height) {
                    height = centroid.height;
                    this.partLookup[lookupKey] = centroid;
                    hidden = !!(centroid.flags & HIDDEN_BY_REPLACEMENT);
                }
            }
        }
        if (height === Number.NEGATIVE_INFINITY) {
            this.partLookup[lookupKey] = void 0;
            return;
        }
        return {
            height,
            hidden
        };
    }
}
function getCosHalfAngle(na, nb) {
    const nm = na.add(nb)._unit();
    const cosHalfAngle = na.x * nm.x + na.y * nm.y;
    return cosHalfAngle;
}
function getRoundedEdgeOffset(p0, p1, p2, edgeRadius) {
    const na = p1.sub(p0)._perp()._unit();
    const nb = p2.sub(p1)._perp()._unit();
    const cosHalfAngle = getCosHalfAngle(na, nb);
    return _getRoundedEdgeOffset(p0, p1, p2, cosHalfAngle, edgeRadius);
}
function _getRoundedEdgeOffset(p0, p1, p2, cosHalfAngle, edgeRadius) {
    const sinHalfAngle = Math.sqrt(1 - cosHalfAngle * cosHalfAngle);
    return Math.min(p0.dist(p1) / 3, p1.dist(p2) / 3, edgeRadius * sinHalfAngle / cosHalfAngle);
}
register(FillExtrusionBucket, 'FillExtrusionBucket', {
    omit: [
        'layers',
        'features'
    ]
});
register(PartData, 'PartData');
register(FootprintSegment, 'FootprintSegment');
register(BorderCentroidData, 'BorderCentroidData');
register(GroundEffect, 'GroundEffect');
function isEdgeOutsideBounds(p1, p2, bounds) {
    return p1.x < bounds[0].x && p2.x < bounds[0].x || p1.x > bounds[1].x && p2.x > bounds[1].x || p1.y < bounds[0].y && p2.y < bounds[0].y || p1.y > bounds[1].y && p2.y > bounds[1].y;
}
function pointOutsideBounds$1(p, bounds) {
    return p.x < bounds[0].x || p.x > bounds[1].x || p.y < bounds[0].y || p.y > bounds[1].y;
}
function isEntirelyOutside(ring) {
    return ring.every(p => p.x <= 0) || ring.every(p => p.x >= EXTENT) || ring.every(p => p.y <= 0) || ring.every(p => p.y >= EXTENT);
}
function isAOConcaveAngle(p2, p1, p3) {
    if (p2.x < 0 || p2.x >= EXTENT || p1.x < 0 || p1.x >= EXTENT || p3.x < 0 || p3.x >= EXTENT) {
        return false;
    }
    const a = p3.sub(p1);
    const an = a.perp();
    const b = p2.sub(p1);
    const ab = a.x * b.x + a.y * b.y;
    const cosAB = ab / Math.sqrt((a.x * a.x + a.y * a.y) * (b.x * b.x + b.y * b.y));
    const dotProductWithNormal = an.x * b.x + an.y * b.y;
    return cosAB > -0.866 && dotProductWithNormal < 0;
}
function encodeAOToEdgeDistance(edgeDistance, isConcaveCorner, edgeStart) {
    const encodedEdgeDistance = isConcaveCorner ? edgeDistance | 2 : edgeDistance & ~2;
    return edgeStart ? encodedEdgeDistance | 1 : encodedEdgeDistance & ~1;
}
function fillExtrusionHeightLift() {
    const angle = Math.PI / 32;
    const tanAngle = Math.tan(angle);
    const r = earthRadius;
    return r * Math.sqrt(1 + 2 * tanAngle * tanAngle) - r;
}
function resampleFillExtrusionPolygonsForGlobe(polygons, tileBounds, tileID) {
    const cellCount = 360 / 32;
    const tiles = 1 << tileID.z;
    const leftLng = lngFromMercatorX(tileID.x / tiles);
    const rightLng = lngFromMercatorX((tileID.x + 1) / tiles);
    const topLat = latFromMercatorY(tileID.y / tiles);
    const bottomLat = latFromMercatorY((tileID.y + 1) / tiles);
    const cellCountOnXAxis = Math.ceil((rightLng - leftLng) / cellCount);
    const cellCountOnYAxis = Math.ceil((topLat - bottomLat) / cellCount);
    const splitFn = (axis, min, max) => {
        if (axis === 0) {
            return 0.5 * (min + max);
        } else {
            const maxLat = latFromMercatorY((tileID.y + min / EXTENT) / tiles);
            const minLat = latFromMercatorY((tileID.y + max / EXTENT) / tiles);
            const midLat = 0.5 * (minLat + maxLat);
            return (mercatorYfromLat(midLat) * tiles - tileID.y) * EXTENT;
        }
    };
    return subdividePolygons(polygons, tileBounds, cellCountOnXAxis, cellCountOnYAxis, 1, splitFn);
}
function transformFootprintVertices(vertices, offset, count, footprintId, centroidId, out) {
    const zDiff = Math.pow(2, footprintId.z - centroidId.z);
    for (let i = 0; i < count; i++) {
        let x = vertices.int16[(i + offset) * 2 + 0];
        let y = vertices.int16[(i + offset) * 2 + 1];
        x = (x + centroidId.x * EXTENT) * zDiff - footprintId.x * EXTENT;
        y = (y + centroidId.y * EXTENT) * zDiff - footprintId.y * EXTENT;
        out.push(new Point(x, y));
    }
}

let layout$7;
const getLayoutProperties$7 = () => layout$7 || (layout$7 = new Properties({
    'visibility': new DataConstantProperty(spec['layout_fill-extrusion']['visibility']),
    'fill-extrusion-edge-radius': new DataConstantProperty(spec['layout_fill-extrusion']['fill-extrusion-edge-radius'])
}));
let paint$8;
const getPaintProperties$8 = () => paint$8 || (paint$8 = new Properties({
    'fill-extrusion-opacity': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-opacity']),
    'fill-extrusion-color': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-color']),
    'fill-extrusion-translate': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-translate']),
    'fill-extrusion-translate-anchor': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-translate-anchor']),
    'fill-extrusion-pattern': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-pattern']),
    'fill-extrusion-height': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-height']),
    'fill-extrusion-base': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-base']),
    'fill-extrusion-height-alignment': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-height-alignment']),
    'fill-extrusion-base-alignment': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-base-alignment']),
    'fill-extrusion-vertical-gradient': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-vertical-gradient']),
    'fill-extrusion-ambient-occlusion-intensity': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-ambient-occlusion-intensity']),
    'fill-extrusion-ambient-occlusion-radius': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-ambient-occlusion-radius']),
    'fill-extrusion-ambient-occlusion-wall-radius': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-ambient-occlusion-wall-radius']),
    'fill-extrusion-ambient-occlusion-ground-radius': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-ambient-occlusion-ground-radius']),
    'fill-extrusion-ambient-occlusion-ground-attenuation': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-ambient-occlusion-ground-attenuation']),
    'fill-extrusion-flood-light-color': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-flood-light-color']),
    'fill-extrusion-flood-light-intensity': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-flood-light-intensity']),
    'fill-extrusion-flood-light-wall-radius': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-flood-light-wall-radius']),
    'fill-extrusion-flood-light-ground-radius': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-flood-light-ground-radius']),
    'fill-extrusion-flood-light-ground-attenuation': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-flood-light-ground-attenuation']),
    'fill-extrusion-vertical-scale': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-vertical-scale']),
    'fill-extrusion-rounded-roof': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-rounded-roof']),
    'fill-extrusion-cutoff-fade-range': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-cutoff-fade-range']),
    'fill-extrusion-emissive-strength': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-emissive-strength']),
    'fill-extrusion-line-width': new DataDrivenProperty(spec['paint_fill-extrusion']['fill-extrusion-line-width']),
    'fill-extrusion-cast-shadows': new DataConstantProperty(spec['paint_fill-extrusion']['fill-extrusion-cast-shadows']),
    'fill-extrusion-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    }),
    'fill-extrusion-flood-light-color-use-theme': new DataDrivenProperty({
        'type': 'string',
        'default': 'default',
        'property-type': 'data-driven'
    })
}));

class FillExtrusionStyleLayer extends StyleLayer {
    constructor(layer, scope, lut, options) {
        const properties = {
            layout: getLayoutProperties$7(),
            paint: getPaintProperties$8()
        };
        super(layer, properties, scope, lut, options);
        this._stats = {
            numRenderedVerticesInShadowPass: 0,
            numRenderedVerticesInTransparentPass: 0
        };
    }
    createBucket(parameters) {
        return new FillExtrusionBucket(parameters);
    }
    queryRadius() {
        return translateDistance(this.paint.get('fill-extrusion-translate'));
    }
    is3D() {
        return true;
    }
    hasShadowPass() {
        return this.paint.get('fill-extrusion-cast-shadows');
    }
    cutoffRange() {
        return this.paint.get('fill-extrusion-cutoff-fade-range');
    }
    canCastShadows() {
        return true;
    }
    getProgramIds() {
        const patternProperty = this.paint.get('fill-extrusion-pattern');
        const image = patternProperty.constantOr(1);
        return [image ? 'fillExtrusionPattern' : 'fillExtrusion'];
    }
    queryIntersectsFeature(queryGeometry, feature, featureState, geometry, zoom, transform, pixelPosMatrix, elevationHelper, layoutVertexArrayOffset) {
        const translation = tilespaceTranslate(this.paint.get('fill-extrusion-translate'), this.paint.get('fill-extrusion-translate-anchor'), transform.angle, queryGeometry.pixelToTileUnitsFactor);
        const height = this.paint.get('fill-extrusion-height').evaluate(feature, featureState);
        const base = this.paint.get('fill-extrusion-base').evaluate(feature, featureState);
        const centroid = [
            0,
            0
        ];
        const terrainVisible = elevationHelper && transform.elevation;
        const exaggeration = transform.elevation ? transform.elevation.exaggeration() : 1;
        const bucket = queryGeometry.tile.getBucket(this);
        if (terrainVisible && bucket instanceof FillExtrusionBucket) {
            const centroidVertexArray = bucket.centroidVertexArray;
            const centroidOffset = layoutVertexArrayOffset + 1;
            if (centroidOffset < centroidVertexArray.length) {
                centroid[0] = centroidVertexArray.geta_centroid_pos0(centroidOffset);
                centroid[1] = centroidVertexArray.geta_centroid_pos1(centroidOffset);
            }
        }
        const isHidden = centroid[0] === 0 && centroid[1] === 1;
        if (isHidden)
            return false;
        if (transform.projection.name === 'globe') {
            const bounds = [
                new Point(0, 0),
                new Point(EXTENT, EXTENT)
            ];
            const resampledGeometry = resampleFillExtrusionPolygonsForGlobe([geometry], bounds, queryGeometry.tileID.canonical);
            geometry = resampledGeometry.map(clipped => clipped.polygon).flat();
        }
        const demSampler = terrainVisible ? elevationHelper : null;
        const [projectedBase, projectedTop] = projectExtrusion(transform, geometry, base, height, translation, pixelPosMatrix, demSampler, centroid, exaggeration, transform.center.lat, queryGeometry.tileID.canonical);
        const screenQuery = queryGeometry.queryGeometry;
        const projectedQueryGeometry = screenQuery.isPointQuery() ? screenQuery.screenBounds : screenQuery.screenGeometry;
        return checkIntersection(projectedBase, projectedTop, projectedQueryGeometry);
    }
}
function dot(a, b) {
    return a.x * b.x + a.y * b.y;
}
function getIntersectionDistance(projectedQueryGeometry, projectedFace) {
    if (projectedQueryGeometry.length === 1) {
        let i = 0;
        const a = projectedFace[i++];
        let b;
        while (!b || a.equals(b)) {
            b = projectedFace[i++];
            if (!b)
                return Infinity;
        }
        for (; i < projectedFace.length; i++) {
            const c = projectedFace[i];
            const p = projectedQueryGeometry[0];
            const ab = b.sub(a);
            const ac = c.sub(a);
            const ap = p.sub(a);
            const dotABAB = dot(ab, ab);
            const dotABAC = dot(ab, ac);
            const dotACAC = dot(ac, ac);
            const dotAPAB = dot(ap, ab);
            const dotAPAC = dot(ap, ac);
            const denom = dotABAB * dotACAC - dotABAC * dotABAC;
            const v = (dotACAC * dotAPAB - dotABAC * dotAPAC) / denom;
            const w = (dotABAB * dotAPAC - dotABAC * dotAPAB) / denom;
            const u = 1 - v - w;
            const distance = a.z * u + b.z * v + c.z * w;
            if (isFinite(distance))
                return distance;
        }
        return Infinity;
    } else {
        let closestDistance = Infinity;
        for (const p of projectedFace) {
            closestDistance = Math.min(closestDistance, p.z);
        }
        return closestDistance;
    }
}
function checkIntersection(projectedBase, projectedTop, projectedQueryGeometry) {
    let closestDistance = Infinity;
    if (polygonIntersectsMultiPolygon(projectedQueryGeometry, projectedTop)) {
        closestDistance = getIntersectionDistance(projectedQueryGeometry, projectedTop[0]);
    }
    for (let r = 0; r < projectedTop.length; r++) {
        const ringTop = projectedTop[r];
        const ringBase = projectedBase[r];
        for (let p = 0; p < ringTop.length - 1; p++) {
            const topA = ringTop[p];
            const topB = ringTop[p + 1];
            const baseA = ringBase[p];
            const baseB = ringBase[p + 1];
            const face = [
                topA,
                topB,
                baseB,
                baseA,
                topA
            ];
            if (polygonIntersectsPolygon(projectedQueryGeometry, face)) {
                closestDistance = Math.min(closestDistance, getIntersectionDistance(projectedQueryGeometry, face));
            }
        }
    }
    return closestDistance === Infinity ? false : closestDistance;
}
function projectExtrusion(tr, geometry, zBase, zTop, translation, m, demSampler, centroid, exaggeration, lat, tileID) {
    if (tr.projection.name === 'globe') {
        return projectExtrusionGlobe(tr, geometry, zBase, zTop, translation, m, demSampler, centroid, exaggeration, lat, tileID);
    } else {
        if (demSampler) {
            return projectExtrusion3D(geometry, zBase, zTop, translation, m, demSampler, centroid, exaggeration, lat);
        } else {
            return projectExtrusion2D(geometry, zBase, zTop, translation, m);
        }
    }
}
function projectExtrusionGlobe(tr, geometry, zBase, zTop, translation, m, demSampler, centroid, exaggeration, lat, tileID) {
    const projectedBase = [];
    const projectedTop = [];
    const elevationScale = tr.projection.upVectorScale(tileID, tr.center.lat, tr.worldSize).metersToTile;
    const basePoint = [
        0,
        0,
        0,
        1
    ];
    const topPoint = [
        0,
        0,
        0,
        1
    ];
    const setPoint = (point, x, y, z) => {
        point[0] = x;
        point[1] = y;
        point[2] = z;
        point[3] = 1;
    };
    const lift = fillExtrusionHeightLift();
    if (zBase > 0) {
        zBase += lift;
    }
    zTop += lift;
    for (const r of geometry) {
        const ringBase = [];
        const ringTop = [];
        for (const p of r) {
            const x = p.x + translation.x;
            const y = p.y + translation.y;
            const reproj = tr.projection.projectTilePoint(x, y, tileID);
            const dir = tr.projection.upVector(tileID, p.x, p.y);
            let zBasePoint = zBase;
            let zTopPoint = zTop;
            if (demSampler) {
                const offset = getTerrainHeightOffset(x, y, zBase, zTop, demSampler, centroid, exaggeration, lat);
                zBasePoint += offset.base;
                zTopPoint += offset.top;
            }
            if (zBase !== 0) {
                setPoint(basePoint, reproj.x + dir[0] * elevationScale * zBasePoint, reproj.y + dir[1] * elevationScale * zBasePoint, reproj.z + dir[2] * elevationScale * zBasePoint);
            } else {
                setPoint(basePoint, reproj.x, reproj.y, reproj.z);
            }
            setPoint(topPoint, reproj.x + dir[0] * elevationScale * zTopPoint, reproj.y + dir[1] * elevationScale * zTopPoint, reproj.z + dir[2] * elevationScale * zTopPoint);
            cjsExports.vec3.transformMat4(basePoint, basePoint, m);
            cjsExports.vec3.transformMat4(topPoint, topPoint, m);
            ringBase.push(new Point3D(basePoint[0], basePoint[1], basePoint[2]));
            ringTop.push(new Point3D(topPoint[0], topPoint[1], topPoint[2]));
        }
        projectedBase.push(ringBase);
        projectedTop.push(ringTop);
    }
    return [
        projectedBase,
        projectedTop
    ];
}
function projectExtrusion2D(geometry, zBase, zTop, translation, m) {
    const projectedBase = [];
    const projectedTop = [];
    const baseXZ = m[8] * zBase;
    const baseYZ = m[9] * zBase;
    const baseZZ = m[10] * zBase;
    const baseWZ = m[11] * zBase;
    const topXZ = m[8] * zTop;
    const topYZ = m[9] * zTop;
    const topZZ = m[10] * zTop;
    const topWZ = m[11] * zTop;
    for (const r of geometry) {
        const ringBase = [];
        const ringTop = [];
        for (const p of r) {
            const x = p.x + translation.x;
            const y = p.y + translation.y;
            const sX = m[0] * x + m[4] * y + m[12];
            const sY = m[1] * x + m[5] * y + m[13];
            const sZ = m[2] * x + m[6] * y + m[14];
            const sW = m[3] * x + m[7] * y + m[15];
            const baseX = sX + baseXZ;
            const baseY = sY + baseYZ;
            const baseZ = sZ + baseZZ;
            const baseW = Math.max(sW + baseWZ, 0.00001);
            const topX = sX + topXZ;
            const topY = sY + topYZ;
            const topZ = sZ + topZZ;
            const topW = Math.max(sW + topWZ, 0.00001);
            ringBase.push(new Point3D(baseX / baseW, baseY / baseW, baseZ / baseW));
            ringTop.push(new Point3D(topX / topW, topY / topW, topZ / topW));
        }
        projectedBase.push(ringBase);
        projectedTop.push(ringTop);
    }
    return [
        projectedBase,
        projectedTop
    ];
}
function projectExtrusion3D(geometry, zBase, zTop, translation, m, demSampler, centroid, exaggeration, lat) {
    const projectedBase = [];
    const projectedTop = [];
    const v = [
        0,
        0,
        0,
        1
    ];
    for (const r of geometry) {
        const ringBase = [];
        const ringTop = [];
        for (const p of r) {
            const x = p.x + translation.x;
            const y = p.y + translation.y;
            const heightOffset = getTerrainHeightOffset(x, y, zBase, zTop, demSampler, centroid, exaggeration, lat);
            v[0] = x;
            v[1] = y;
            v[2] = heightOffset.base;
            v[3] = 1;
            cjsExports.vec4.transformMat4(v, v, m);
            v[3] = Math.max(v[3], 0.00001);
            const base = new Point3D(v[0] / v[3], v[1] / v[3], v[2] / v[3]);
            v[0] = x;
            v[1] = y;
            v[2] = heightOffset.top;
            v[3] = 1;
            cjsExports.vec4.transformMat4(v, v, m);
            v[3] = Math.max(v[3], 0.00001);
            const top = new Point3D(v[0] / v[3], v[1] / v[3], v[2] / v[3]);
            ringBase.push(base);
            ringTop.push(top);
        }
        projectedBase.push(ringBase);
        projectedTop.push(ringTop);
    }
    return [
        projectedBase,
        projectedTop
    ];
}
function getTerrainHeightOffset(x, y, zBase, zTop, demSampler, centroid, exaggeration, lat) {
    const ele = exaggeration * demSampler.getElevationAt(x, y, true, true);
    const flatRoof = centroid[0] !== 0;
    const centroidElevation = flatRoof ? centroid[1] === 0 ? exaggeration * elevationFromUint16(centroid[0]) : exaggeration * flatElevation(demSampler, centroid, lat) : ele;
    return {
        base: ele + (zBase === 0 ? -1 : zBase),
        // Use -1 instead of -5 in shader to prevent picking underground
        top: flatRoof ? Math.max(centroidElevation + zTop, ele + zBase + 2) : ele + zTop
    };
}
function elevationFromUint16(n) {
    return n / ELEVATION_SCALE - ELEVATION_OFFSET;
}
function flatElevation(demSampler, centroid, lat) {
    const posX = Math.floor(centroid[0] / 8);
    const posY = Math.floor(centroid[1] / 8);
    const spanX = 10 * (centroid[0] - posX * 8);
    const spanY = 10 * (centroid[1] - posY * 8);
    const z = demSampler.getElevationAt(posX, posY, true, true);
    const meterToDEM = demSampler.getMeterToDEM(lat);
    const wX = Math.floor(0.5 * (spanX * meterToDEM - 1));
    const wY = Math.floor(0.5 * (spanY * meterToDEM - 1));
    const posPx = demSampler.tileCoordToPixel(posX, posY);
    const offsetX = 2 * wX + 1;
    const offsetY = 2 * wY + 1;
    const corners = fourSample(demSampler, posPx.x - wX, posPx.y - wY, offsetX, offsetY);
    const diffX = Math.abs(corners[0] - corners[1]);
    const diffY = Math.abs(corners[2] - corners[3]);
    const diffZ = Math.abs(corners[0] - corners[2]);
    const diffW = Math.abs(corners[1] - corners[3]);
    const diffSumX = diffX + diffY;
    const diffSumY = diffZ + diffW;
    const slopeX = Math.min(0.25, meterToDEM * 0.5 * diffSumX / offsetX);
    const slopeY = Math.min(0.25, meterToDEM * 0.5 * diffSumY / offsetY);
    return z + Math.max(slopeX * spanX, slopeY * spanY);
}
function fourSample(demSampler, posX, posY, offsetX, offsetY) {
    return [
        demSampler.getElevationAtPixel(posX, posY, true),
        demSampler.getElevationAtPixel(posX + offsetY, posY, true),
        demSampler.getElevationAtPixel(posX, posY + offsetY, true),
        demSampler.getElevationAtPixel(posX + offsetX, posY + offsetY, true)
    ];
}

const lineLayoutAttributes = createLayout([
    {
        name: 'a_pos_normal',
        components: 2,
        type: 'Int16'
    },
    {
        name: 'a_data',
        components: 4,
        type: 'Uint8'
    },
    {
        name: 'a_linesofar',
        components: 1,
        type: 'Float32'
    }
], 4);
const lineZOffsetAttributes = createLayout([{
        name: 'a_z_offset_width',
        components: 3,
        type: 'Float32'
    }], 4);
const {members: members$2, size: size$2, alignment: alignment$2} = lineLayoutAttributes;

const lineLayoutAttributesExt = createLayout([{
        name: 'a_packed',
        components: 3,
        type: 'Float32'
    }]);
const {members: members$1, size: size$1, alignment: alignment$1} = lineLayoutAttributesExt;

const lineLayoutAttributesPattern = createLayout([{
        name: 'a_pattern_data',
        components: 3,
        type: 'Float32'
    }]);
const {members, size, alignment} = lineLayoutAttributesPattern;

class LineAtlas {
    constructor(width, height) {
        this.width = width;
        this.height = height;
        this.nextRow = 0;
        this.image = new AlphaImage({
            width,
            height
        });
        this.positions = {};
        this.uploaded = false;
    }
    /**
   * Get a dash line pattern.
   *
   * @param {Array<number>} dasharray
   * @param {string} lineCap the type of line caps to be added to dashes
   * @returns {Object} position of dash texture in { y, height, width }
   * @private
   */
    getDash(dasharray, lineCap) {
        const key = this.getKey(dasharray, lineCap);
        return this.positions[key];
    }
    trim() {
        const width = this.width;
        const height = this.height = nextPowerOfTwo(this.nextRow);
        this.image.resize({
            width,
            height
        });
    }
    getKey(dasharray, lineCap) {
        return dasharray.join(',') + lineCap;
    }
    getDashRanges(dasharray, lineAtlasWidth, stretch) {
        const oddDashArray = dasharray.length % 2 === 1;
        const ranges = [];
        let left = oddDashArray ? -dasharray[dasharray.length - 1] * stretch : 0;
        let right = dasharray[0] * stretch;
        let isDash = true;
        ranges.push({
            left,
            right,
            isDash,
            zeroLength: dasharray[0] === 0
        });
        let currentDashLength = dasharray[0];
        for (let i = 1; i < dasharray.length; i++) {
            isDash = !isDash;
            const dashLength = dasharray[i];
            left = currentDashLength * stretch;
            currentDashLength += dashLength;
            right = currentDashLength * stretch;
            ranges.push({
                left,
                right,
                isDash,
                zeroLength: dashLength === 0
            });
        }
        return ranges;
    }
    addRoundDash(ranges, stretch, n) {
        const halfStretch = stretch / 2;
        for (let y = -n; y <= n; y++) {
            const row = this.nextRow + n + y;
            const index = this.width * row;
            let currIndex = 0;
            let range = ranges[currIndex];
            for (let x = 0; x < this.width; x++) {
                if (x / range.right > 1) {
                    range = ranges[++currIndex];
                }
                const distLeft = Math.abs(x - range.left);
                const distRight = Math.abs(x - range.right);
                const minDist = Math.min(distLeft, distRight);
                let signedDistance;
                const distMiddle = y / n * (halfStretch + 1);
                if (range.isDash) {
                    const distEdge = halfStretch - Math.abs(distMiddle);
                    signedDistance = Math.sqrt(minDist * minDist + distEdge * distEdge);
                } else {
                    signedDistance = halfStretch - Math.sqrt(minDist * minDist + distMiddle * distMiddle);
                }
                this.image.data[index + x] = Math.max(0, Math.min(255, signedDistance + 128));
            }
        }
    }
    addRegularDash(ranges, capLength) {
        for (let i = ranges.length - 1; i >= 0; --i) {
            const part = ranges[i];
            const next = ranges[i + 1];
            if (part.zeroLength) {
                ranges.splice(i, 1);
            } else if (next && next.isDash === part.isDash) {
                next.left = part.left;
                ranges.splice(i, 1);
            }
        }
        const first = ranges[0];
        const last = ranges[ranges.length - 1];
        if (first.isDash === last.isDash) {
            first.left = last.left - this.width;
            last.right = first.right + this.width;
        }
        const index = this.width * this.nextRow;
        let currIndex = 0;
        let range = ranges[currIndex];
        for (let x = 0; x < this.width; x++) {
            if (x / range.right > 1) {
                range = ranges[++currIndex];
            }
            const distLeft = Math.abs(x - range.left);
            const distRight = Math.abs(x - range.right);
            const minDist = Math.min(distLeft, distRight);
            const signedDistance = (range.isDash ? minDist : -minDist) + capLength;
            this.image.data[index + x] = Math.max(0, Math.min(255, signedDistance + 128));
        }
    }
    addDash(dasharray, lineCap) {
        const key = this.getKey(dasharray, lineCap);
        if (this.positions[key])
            return this.positions[key];
        const round = lineCap === 'round';
        const n = round ? 7 : 0;
        const height = 2 * n + 1;
        if (this.nextRow + height > this.height) {
            warnOnce('LineAtlas out of space');
            return null;
        }
        if (dasharray.length === 0) {
            dasharray.push(1);
        }
        let length = 0;
        for (let i = 0; i < dasharray.length; i++) {
            if (dasharray[i] < 0) {
                warnOnce('Negative value is found in line dasharray, replacing values with 0');
                dasharray[i] = 0;
            }
            length += dasharray[i];
        }
        if (length !== 0) {
            const stretch = this.width / length;
            const ranges = this.getDashRanges(dasharray, this.width, stretch);
            if (round) {
                this.addRoundDash(ranges, stretch, n);
            } else {
                const capLength = lineCap === 'square' ? 0.5 * stretch : 0;
                this.addRegularDash(ranges, capLength);
            }
        }
        const y = this.nextRow + n;
        this.nextRow += height;
        const pos = {
            tl: [
                y,
                n
            ],
            br: [
                length,
                0
            ]
        };
        this.positions[key] = pos;
        return pos;
    }
}
register(LineAtlas, 'LineAtlas');

const vectorTileFeatureTypes$1 = vectorTileExports.VectorTileFeature.types;
const EXTRUDE_SCALE = 63;
const COS_HALF_SHARP_CORNER = Math.cos(75 / 2 * (Math.PI / 180));
const COS_STRAIGHT_CORNER = Math.cos(5 * (Math.PI / 180));
const DEG_PER_TRIANGLE = 20;
class LineBucket {
    constructor(options) {
        this.evaluationGlobals = {
            'zoom': 0,
            'lineProgress': void 0
        };
        this.zoom = options.zoom;
        this.evaluationGlobals.zoom = this.zoom;
        this.overscaling = options.overscaling;
        this.pixelRatio = options.pixelRatio;
        this.layers = options.layers;
        this.layerIds = this.layers.map(layer => layer.fqid);
        this.index = options.index;
        this.projection = options.projection;
        this.hasPattern = false;
        this.hasZOffset = false;
        this.hasCrossSlope = false;
        this.patternFeatures = [];
        this.lineClipsArray = [];