package Lire::Max;

use strict;

use base qw/ Lire::SimpleStat /;

use Carp;

use Lire::Utils qw/ sql_quote_name /;

=pod

=head1 NAME

Lire::Max - Lire class that implements the max operator

=head1 SYNOPSIS

 use Lire::Max

=head1 DESCRIPTION

Class that implements the max operator. This operator will find the
minimum value appearing in a DLF field among a group of DLF records.

=head1 METHODS

=head2 new( %params )

Creates a new Lire::Max object.

=cut

sub new {
    my $self = bless {}, shift;

    $self->init( @_, 'op' => 'max' );

    return $self;
}

# Implements Lire::Aggregate::sql_aggr_expr
sub sql_aggr_expr {
    my $self = $_[0];

    return 'max(' . sql_quote_name( $self->{'field'} ) . ')';
}

# Implements Lire::ReportOperator::init_group_data
sub init_group_data {
    my $scalar = undef;
    return \$scalar;
}

# Implements Lire::ReportOperator::merge_group_data
sub merge_group_data {
    my ( $self, $value, $data ) = @_;

    return if lc $value->{'value'} eq 'nan';

    $$data = $value->{'value'}
      unless ( defined $$data );

    # To merge two max, we keep the highest
    $$data = $value->{'value'}
      if $value->{'value'} > $$data;

    return;
}

# Implements Lire::ReportOperator::end_group_data
sub end_group_data {
    my ( $self, $data ) = @_;

    $$data = "NaN" unless defined $$data;

    return;
}


# keep perl happy
1;

__END__

=head1 SEE ALSO

Lire::ReportSpec(3pm), Lire::ReportOperator(3pm), Lire::Aggregator(3pm), Lire::Aggregate(3pm).

=head1 AUTHOR

  Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Max.pm,v 1.10 2008/03/09 19:27:31 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001, 2002 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html. 

=cut
