RSpec.describe Msf::Exploit::Remote::Pkcs12::StoredPkcs12 do
  let(:private) {
    FactoryBot.build(
      :metasploit_credential_pkcs12,
      id: rand(1..1000)
    )
  }
  let(:pkcs12) {
    FactoryBot.build(
      :metasploit_credential_core,
      private: private
    )
  }

  subject(:stored_pkcs12) { described_class.new(pkcs12) }

  describe '#id' do
    it 'returns the expected ID value' do
      expect(stored_pkcs12.id).to eq(pkcs12.id)
    end
  end

  describe '#openssl_pkcs12' do
    it 'returns an OpenSSL::PKCS12 object' do
      expect(stored_pkcs12.openssl_pkcs12).to be_a(OpenSSL::PKCS12)
    end

    it 'returns the expected Pkcs12' do
      raw_pkcs12 = Base64.strict_decode64(pkcs12.private.data)
      expect(stored_pkcs12.openssl_pkcs12.to_der).to eq(raw_pkcs12)
    end
  end

  describe '#private_cred' do
    it 'returns the expected private credential value' do
      expect(stored_pkcs12.private_cred).to eq(private)
    end
  end

  describe '#username' do
    it 'returns the expected username value' do
      expect(stored_pkcs12.username).to eq(pkcs12.public.username)
    end
  end

  describe '#realm' do
    it 'returns the expected realm value' do
      expect(stored_pkcs12.realm).to eq(pkcs12.realm.value)
    end
  end

  context 'with metadata' do
    let(:adcs_ca) { 'test_CA' }
    let(:adcs_template) { 'test_template' }
    let(:status) { 'inactive' }
    let(:metadata) {
      {
        adcs_ca: adcs_ca,
        adcs_template: adcs_template,
        status: status
      }
    }
    let(:private) {
      FactoryBot.build(
        :metasploit_credential_pkcs12,
        metadata: metadata
      )
    }

    describe '#adcs_ca' do
      it 'returns the expected ADCS CA value' do
        expect(stored_pkcs12.adcs_ca).to eq(adcs_ca)
      end
    end

    describe '#adcs_template' do
      it 'returns the expected ADCS template value' do
        expect(stored_pkcs12.adcs_template).to eq(adcs_template)
      end
    end

    describe '#status' do
      it 'returns the expected status value' do
        expect(stored_pkcs12.status).to eq(status)
      end
    end
  end

  describe '#expired?' do
    context 'when the certificate is valid within the not_before/not_after' do
      it 'returns false' do
        expect(stored_pkcs12.expired?).to be false
      end

      context 'with a password-protected Pkcs12' do
        let(:passwd) { 'test_password' }
        let(:private) {
          FactoryBot.build(
            :metasploit_credential_pkcs12_with_pkcs12_password,
            pkcs12_password: passwd
          )
        }

        it 'returns false' do
          expect(stored_pkcs12.expired?).to be false
        end
      end
    end

    context 'when the certificate not_after date is in the past' do
      let(:private) {
        FactoryBot.build(
          :metasploit_credential_pkcs12,
          not_before: Time.now - 2.hours,
          not_after: Time.now - 1.hour
        )
      }
      it 'returns true' do
        expect(stored_pkcs12.expired?).to be true
      end
    end

    context 'when the certificate not_before date is in the future' do
      let(:private) {
        FactoryBot.build(
          :metasploit_credential_pkcs12,
          not_before: Time.now + 1.hour,
          not_after: Time.now + 2.hours
        )
      }
      it 'returns true' do
        expect(stored_pkcs12.expired?).to be true
      end
    end
  end
end

