﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/lakeformation/LakeFormation_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace LakeFormation {
namespace Model {

/**
 * <p>A structure used as a protocol between query engines and Lake Formation or
 * Glue. Contains both a Lake Formation generated authorization identifier and
 * information from the request's authorization context.</p><p><h3>See Also:</h3>
 * <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/lakeformation-2017-03-31/QuerySessionContext">AWS
 * API Reference</a></p>
 */
class QuerySessionContext {
 public:
  AWS_LAKEFORMATION_API QuerySessionContext() = default;
  AWS_LAKEFORMATION_API QuerySessionContext(Aws::Utils::Json::JsonView jsonValue);
  AWS_LAKEFORMATION_API QuerySessionContext& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_LAKEFORMATION_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>A unique identifier generated by the query engine for the query.</p>
   */
  inline const Aws::String& GetQueryId() const { return m_queryId; }
  inline bool QueryIdHasBeenSet() const { return m_queryIdHasBeenSet; }
  template <typename QueryIdT = Aws::String>
  void SetQueryId(QueryIdT&& value) {
    m_queryIdHasBeenSet = true;
    m_queryId = std::forward<QueryIdT>(value);
  }
  template <typename QueryIdT = Aws::String>
  QuerySessionContext& WithQueryId(QueryIdT&& value) {
    SetQueryId(std::forward<QueryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A timestamp provided by the query engine for when the query started.</p>
   */
  inline const Aws::Utils::DateTime& GetQueryStartTime() const { return m_queryStartTime; }
  inline bool QueryStartTimeHasBeenSet() const { return m_queryStartTimeHasBeenSet; }
  template <typename QueryStartTimeT = Aws::Utils::DateTime>
  void SetQueryStartTime(QueryStartTimeT&& value) {
    m_queryStartTimeHasBeenSet = true;
    m_queryStartTime = std::forward<QueryStartTimeT>(value);
  }
  template <typename QueryStartTimeT = Aws::Utils::DateTime>
  QuerySessionContext& WithQueryStartTime(QueryStartTimeT&& value) {
    SetQueryStartTime(std::forward<QueryStartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An identifier string for the consumer cluster.</p>
   */
  inline const Aws::String& GetClusterId() const { return m_clusterId; }
  inline bool ClusterIdHasBeenSet() const { return m_clusterIdHasBeenSet; }
  template <typename ClusterIdT = Aws::String>
  void SetClusterId(ClusterIdT&& value) {
    m_clusterIdHasBeenSet = true;
    m_clusterId = std::forward<ClusterIdT>(value);
  }
  template <typename ClusterIdT = Aws::String>
  QuerySessionContext& WithClusterId(ClusterIdT&& value) {
    SetClusterId(std::forward<ClusterIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A cryptographically generated query identifier generated by Glue or Lake
   * Formation.</p>
   */
  inline const Aws::String& GetQueryAuthorizationId() const { return m_queryAuthorizationId; }
  inline bool QueryAuthorizationIdHasBeenSet() const { return m_queryAuthorizationIdHasBeenSet; }
  template <typename QueryAuthorizationIdT = Aws::String>
  void SetQueryAuthorizationId(QueryAuthorizationIdT&& value) {
    m_queryAuthorizationIdHasBeenSet = true;
    m_queryAuthorizationId = std::forward<QueryAuthorizationIdT>(value);
  }
  template <typename QueryAuthorizationIdT = Aws::String>
  QuerySessionContext& WithQueryAuthorizationId(QueryAuthorizationIdT&& value) {
    SetQueryAuthorizationId(std::forward<QueryAuthorizationIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An opaque string-string map passed by the query engine.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetAdditionalContext() const { return m_additionalContext; }
  inline bool AdditionalContextHasBeenSet() const { return m_additionalContextHasBeenSet; }
  template <typename AdditionalContextT = Aws::Map<Aws::String, Aws::String>>
  void SetAdditionalContext(AdditionalContextT&& value) {
    m_additionalContextHasBeenSet = true;
    m_additionalContext = std::forward<AdditionalContextT>(value);
  }
  template <typename AdditionalContextT = Aws::Map<Aws::String, Aws::String>>
  QuerySessionContext& WithAdditionalContext(AdditionalContextT&& value) {
    SetAdditionalContext(std::forward<AdditionalContextT>(value));
    return *this;
  }
  template <typename AdditionalContextKeyT = Aws::String, typename AdditionalContextValueT = Aws::String>
  QuerySessionContext& AddAdditionalContext(AdditionalContextKeyT&& key, AdditionalContextValueT&& value) {
    m_additionalContextHasBeenSet = true;
    m_additionalContext.emplace(std::forward<AdditionalContextKeyT>(key), std::forward<AdditionalContextValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_queryId;

  Aws::Utils::DateTime m_queryStartTime{};

  Aws::String m_clusterId;

  Aws::String m_queryAuthorizationId;

  Aws::Map<Aws::String, Aws::String> m_additionalContext;
  bool m_queryIdHasBeenSet = false;
  bool m_queryStartTimeHasBeenSet = false;
  bool m_clusterIdHasBeenSet = false;
  bool m_queryAuthorizationIdHasBeenSet = false;
  bool m_additionalContextHasBeenSet = false;
};

}  // namespace Model
}  // namespace LakeFormation
}  // namespace Aws
