# (c) Copyright 2010-2011. CodeWeavers, Inc.

import os.path
import re

import cxconfig
import cxobservable
import cxutils
import distversion


#####
#
# The current product and cellar identifiers
#
#####

_PRODUCT_ID = None
def get_product_id():
    """Gets and checks the effective product id.

    An invalid product id is a fatal error and will raise an exception.
    """
    # pylint: disable=W0603
    global _PRODUCT_ID
    if _PRODUCT_ID is None:
        filename = os.path.join(cxutils.CX_ROOT, ".productid")
        if os.path.exists(filename):
            try:
                with open(filename, 'r', encoding='utf8') as thisfile:
                    _PRODUCT_ID = thisfile.readline().rstrip()
            except OSError as ose:
                raise RuntimeError("unable to open, read or close '%s'" % filename) from ose
        else:
            _PRODUCT_ID = distversion.BUILTIN_PRODUCT_ID
        if len(_PRODUCT_ID) < 4:
            raise RuntimeError("product id '%s' is too short" % _PRODUCT_ID)

        if not re.search("^[a-zA-Z0-9_][a-zA-Z0-9_][a-zA-Z0-9_][a-zA-Z0-9_][a-zA-Z0-9_]*$", _PRODUCT_ID):
            raise RuntimeError("product id '%s' contains bad characters" % _PRODUCT_ID)
    return _PRODUCT_ID


_CELLAR_ID = None
def get_cellar_id():
    """Gets and checks the cellar identifier.

    The cellar is where the CrossOver bottles and associated data (user
    configuration, Windows installer cache, CrossTies, etc) are stored.
    The cellar identifier sets the location of the cellar and is also part of
    the unique identifier used to tag the associated resources such as the
    bottle menus and associations.

    An invalid cellar identifier is a fatal error and will raise an exception.
    """
    # pylint: disable=W0603
    global _CELLAR_ID
    if _CELLAR_ID is None:
        _CELLAR_ID = get_product_id()
        if distversion.IS_PREVIEW and _CELLAR_ID == "cxpreview":
            _CELLAR_ID = "cxoffice"
        if distversion.IS_MACOSX:
            _CELLAR_ID = distversion.PRODUCT_NAME
            if distversion.IS_PREVIEW:
                _CELLAR_ID = "CrossOver"
    return _CELLAR_ID


#####
#
# CrossOver's important directories
#
#####

def get_home_dir():
    try:
        home = os.environ["HOME"]
        stat = os.stat(home)
        euid = os.geteuid()
        if euid == 0 and stat.st_uid != euid:
            raise RuntimeError("$HOME (%s) does not belong to root" % home)
    except KeyError:
        raise RuntimeError("$HOME is not set!") #pylint: disable=W0707
    except OSError as ose:
        import errno
        if ose.errno == errno.ENOENT:
            raise RuntimeError("$HOME (%s) does not exist!" % home) #pylint: disable=W0707
        raise

    return home


def get_user_dir():
    home = get_home_dir()
    if distversion.IS_MACOSX:
        return os.path.join(home, "Library/Application Support/", get_cellar_id())

    return os.path.join(home, "." + get_cellar_id())


def get_icons_dir():
    if distversion.IS_MACOSX:
        return os.path.join(get_home_dir(), "Library/Caches/", "com.codeweavers.%s" % get_cellar_id(), "icons")

    return os.path.join(get_user_dir(), "icons")


def get_installer_dir():
    if distversion.IS_MACOSX:
        return os.path.join(get_home_dir(), "Library/Caches/", "com.codeweavers.%s" % get_cellar_id(), "installers")

    return os.path.join(get_user_dir(), "installers")


def get_managed_dir():
    if distversion.IS_MACOSX:
        return os.path.join("/Library/Application Support", get_cellar_id())

    return os.path.join(cxutils.CX_ROOT, "support")


def is_root_install():
    """Returns True if this install is owned by root.
       This is intended to distinguish between system-wide and private installs.
    """
    stat = os.stat(cxutils.CX_ROOT)
    return stat.st_uid == 0

def is_32bit_install():
    if not os.path.exists(os.path.join(cxutils.CX_ROOT, 'lib', 'wine', 'i386-unix')):
        return False
    try:
        return os.readlink(os.path.join(cxutils.CX_ROOT, 'bin', 'wineserver')) == 'wineserver32'
    except OSError:
        # not a symlink
        return True

def is_64bit_install():
    if os.path.exists(os.path.join(cxutils.CX_ROOT, 'lib', 'wine', 'aarch64-unix')):
        return True
    if not os.path.exists(os.path.join(cxutils.CX_ROOT, 'lib', 'wine', 'x86_64-unix')):
        return False
    try:
        return os.readlink(os.path.join(cxutils.CX_ROOT, 'bin', 'wineserver')) == 'wineserver64'
    except OSError:
        # not a symlink
        return True

def is_crostini():
    """Returns True if we're on Crostini (Linux on Chrome OS). We check
    the presence of the file below which says no password is needed to
    become root on Crostini."""
    return os.path.isfile("/etc/sudoers.d/10-cros-nopasswd")


#####
#
# CrossOver's configuration
#
#####

_CONFIG = None

def get_config():
    """Returns the CrossOver configuration, merged from the relevant sources
    as needed.

    To avoid races when changing the configuration file it is necessary to
    follow these rules:
    - The data obtained prior to locking the writable configuration file with
      must not be trusted. This is because the configuration file was not yet
      locked and thus may have changed between the two calls.
    - So the proper way to test the value of a setting and then modify it
      race-free is:
          config = cxproduct.get_config()

          # This returns the file the changes should be saved into
          wconfig = config.get_save_config()

          # This locks the file and re-reads it if needed, ensuring wconfig
          # *and* config see the very latest data.
          wconfig.lock_file()

          if config['section']['field'] ...:
              wconfig['section']['field'] = ...

          # Saves the changes. If nothing was changed above then the file is
          # left untouched. Then the file is unlocked.
          wconfig.save_and_unlock_file()

          All the while config can still be used in a read-only fashion and is
          guaranteed to be in sync with the latest modifications made in
          wconfig.
    """
    # pylint: disable=W0603
    global _CONFIG
    if _CONFIG is None:
        _CONFIG = cxconfig.Stack()

        global_config_file = os.path.join(cxutils.CX_ROOT, 'etc', get_product_id() + ".conf")
        _CONFIG.addconfig(global_config_file)

        try:
            user_config_file = os.path.join(get_user_dir(), get_cellar_id() + ".conf")
            _CONFIG.addconfig(user_config_file)
        except KeyError:
            # $HOME is not set, ignore it.
            pass
        except RuntimeError:
            # $HOME is wrong, ignore it.
            pass
    return _CONFIG


def get_config_value(section, name, default):
    """Obtains a config value from the CrossOver configuration file."""
    config = get_config()
    return config[section].get(name, default)


def set_config_value(section, field, value):
    """Saves the specified setting in the CrossOver configuration file.

    This function must not be used if value is based on  another configuration
    setting as this would have races.
    """
    config = get_config()
    wconfig = config.get_save_config()
    wconfig.lock_file()
    wconfig[section][field] = value
    wconfig.save_and_unlock_file()


#####
#
# CrossOver's environment settings
#
#####

_ENVIRON = None

def refresh_environment():
    # pylint: disable=W0603
    global _ENVIRON
    _ENVIRON = None

def _refresh_environment(config, *_args):
    """Refreshes the block holding CrossOver's environment settings."""

    # And a few CrossOver-specific variables
    if 'CX_ROOT' not in os.environ:
        os.environ['CX_ROOT'] = cxutils.CX_ROOT

    if 'CX_BOTTLE_PATH' not in os.environ:
        if distversion.IS_MACOSX:
            os.environ['CX_BOTTLE_PATH'] = os.path.join(get_user_dir(), "Bottles")
        else:
            os.environ['CX_BOTTLE_PATH'] = get_user_dir()

    if 'CX_MANAGED_BOTTLE_PATH' not in os.environ:
        if distversion.IS_MACOSX:
            os.environ['CX_MANAGED_BOTTLE_PATH'] = os.path.join(get_managed_dir(), "Bottles")
        else:
            os.environ['CX_MANAGED_BOTTLE_PATH'] = get_managed_dir()

    if config is None:
        config = get_config()
        # Note that adding ourselves as an observer means we will be called
        # from multiple threads. No locking is needed however.
        config.add_observer(cxobservable.ANY_EVENT, _refresh_environment)

    # Process each configuration file individually and in the right order in
    # case the more specific ones build on the more general ones.
    configs = config.configs()
    for conffile in reversed(configs):
        section = conffile["EnvironmentVariables"]
        # Then process the environment variables in the order in which they
        # appear in the file in case they use environment variables defined
        # before.
        for key in section.sortedkeys():
            name = section.fieldname(key)
            os.environ[name] = cxutils.expand_unix_string(os.environ, section[key])

    # Allow setting the bottle path through the config as well.
    bottle_path = config['CrossOver'].get('BottlePath')
    if bottle_path:
        os.environ['CX_BOTTLE_PATH'] = bottle_path

    # pylint: disable=W0603
    global _ENVIRON
    _ENVIRON = os.environ.copy()


def get_bottle_path():
    if _ENVIRON is None:
        _refresh_environment(None)
    return _ENVIRON['CX_BOTTLE_PATH']


def set_bottle_path(path):
    if 'CX_BOTTLE_PATH' in os.environ:
        del os.environ['CX_BOTTLE_PATH']

    set_config_value('CrossOver', 'BottlePath', path)
    refresh_environment()


def get_managed_bottle_path():
    if _ENVIRON is None:
        _refresh_environment(None)
    return _ENVIRON['CX_MANAGED_BOTTLE_PATH']


#####
#
# Inter-product configuration (currently only used on Unix)
#
#####

_CROSS_PRODUCT_CONFIG = None

def get_cross_product_config():
    """Returns a configuration object containing the registered CrossOver
    product."""
    # pylint: disable=W0603
    global _CROSS_PRODUCT_CONFIG
    if _CROSS_PRODUCT_CONFIG is None:
        _CROSS_PRODUCT_CONFIG = cxconfig.Stack()
        _CROSS_PRODUCT_CONFIG.addconfig('/etc/crossover.conf')
        if 'HOME' in os.environ and (os.geteuid() != 0 or \
                cxutils.CX_ROOT.startswith(os.environ['HOME'])):
            _CROSS_PRODUCT_CONFIG.addconfig(os.path.join(os.environ['HOME'],
                                                         '.crossover.conf'))
    return _CROSS_PRODUCT_CONFIG


#####
#
# Product discovery
#
#####

def this_product():
    """Returns a dictionary describing the current product.

    The following values will be present:
    * name - Human-readable product name
    * productid - The product ID used to test <cxversion> tags.
    * productversion - The CrossOver version used to test <cxversion> tags.
    * publicversion - The public version.
    * root - The value of CX_ROOT."""
    result = {}
    result['root'] = cxutils.CX_ROOT
    result['name'] = distversion.PRODUCT_NAME
    result['productid'] = distversion.BUILTIN_PRODUCT_ID
    result['productversion'] = distversion.CX_VERSION
    result['publicversion'] = distversion.PUBLIC_VERSION
    return result
