﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class PutRolePermissionsBoundaryRequest : public IAMRequest {
 public:
  AWS_IAM_API PutRolePermissionsBoundaryRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutRolePermissionsBoundary"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>The name (friendly name, not ARN) of the IAM role for which you want to set
   * the permissions boundary.</p>
   */
  inline const Aws::String& GetRoleName() const { return m_roleName; }
  inline bool RoleNameHasBeenSet() const { return m_roleNameHasBeenSet; }
  template <typename RoleNameT = Aws::String>
  void SetRoleName(RoleNameT&& value) {
    m_roleNameHasBeenSet = true;
    m_roleName = std::forward<RoleNameT>(value);
  }
  template <typename RoleNameT = Aws::String>
  PutRolePermissionsBoundaryRequest& WithRoleName(RoleNameT&& value) {
    SetRoleName(std::forward<RoleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the managed policy that is used to set the permissions boundary
   * for the role.</p> <p>A permissions boundary policy defines the maximum
   * permissions that identity-based policies can grant to an entity, but does not
   * grant permissions. Permissions boundaries do not define the maximum permissions
   * that a resource-based policy can grant to an entity. To learn more, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_boundaries.html">Permissions
   * boundaries for IAM entities</a> in the <i>IAM User Guide</i>.</p> <p>For more
   * information about policy types, see <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policy-types">Policy
   * types </a> in the <i>IAM User Guide</i>.</p>
   */
  inline const Aws::String& GetPermissionsBoundary() const { return m_permissionsBoundary; }
  inline bool PermissionsBoundaryHasBeenSet() const { return m_permissionsBoundaryHasBeenSet; }
  template <typename PermissionsBoundaryT = Aws::String>
  void SetPermissionsBoundary(PermissionsBoundaryT&& value) {
    m_permissionsBoundaryHasBeenSet = true;
    m_permissionsBoundary = std::forward<PermissionsBoundaryT>(value);
  }
  template <typename PermissionsBoundaryT = Aws::String>
  PutRolePermissionsBoundaryRequest& WithPermissionsBoundary(PermissionsBoundaryT&& value) {
    SetPermissionsBoundary(std::forward<PermissionsBoundaryT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_roleName;
  bool m_roleNameHasBeenSet = false;

  Aws::String m_permissionsBoundary;
  bool m_permissionsBoundaryHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
