﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/iam/IAMRequest.h>
#include <aws/iam/IAM_EXPORTS.h>

#include <utility>

namespace Aws {
namespace IAM {
namespace Model {

/**
 */
class GetMFADeviceRequest : public IAMRequest {
 public:
  AWS_IAM_API GetMFADeviceRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "GetMFADevice"; }

  AWS_IAM_API Aws::String SerializePayload() const override;

 protected:
  AWS_IAM_API void DumpBodyToUrl(Aws::Http::URI& uri) const override;

 public:
  ///@{
  /**
   * <p>Serial number that uniquely identifies the MFA device. For this API, we only
   * accept FIDO security key <a
   * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/reference-arns.html">ARNs</a>.</p>
   */
  inline const Aws::String& GetSerialNumber() const { return m_serialNumber; }
  inline bool SerialNumberHasBeenSet() const { return m_serialNumberHasBeenSet; }
  template <typename SerialNumberT = Aws::String>
  void SetSerialNumber(SerialNumberT&& value) {
    m_serialNumberHasBeenSet = true;
    m_serialNumber = std::forward<SerialNumberT>(value);
  }
  template <typename SerialNumberT = Aws::String>
  GetMFADeviceRequest& WithSerialNumber(SerialNumberT&& value) {
    SetSerialNumber(std::forward<SerialNumberT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The friendly name identifying the user.</p>
   */
  inline const Aws::String& GetUserName() const { return m_userName; }
  inline bool UserNameHasBeenSet() const { return m_userNameHasBeenSet; }
  template <typename UserNameT = Aws::String>
  void SetUserName(UserNameT&& value) {
    m_userNameHasBeenSet = true;
    m_userName = std::forward<UserNameT>(value);
  }
  template <typename UserNameT = Aws::String>
  GetMFADeviceRequest& WithUserName(UserNameT&& value) {
    SetUserName(std::forward<UserNameT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_serialNumber;
  bool m_serialNumberHasBeenSet = false;

  Aws::String m_userName;
  bool m_userNameHasBeenSet = false;
};

}  // namespace Model
}  // namespace IAM
}  // namespace Aws
