﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/config/ConfigServiceRequest.h>
#include <aws/config/ConfigService_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace ConfigService {
namespace Model {

/**
 */
class PutResourceConfigRequest : public ConfigServiceRequest {
 public:
  AWS_CONFIGSERVICE_API PutResourceConfigRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "PutResourceConfig"; }

  AWS_CONFIGSERVICE_API Aws::String SerializePayload() const override;

  AWS_CONFIGSERVICE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The type of the resource. The custom resource type must be registered with
   * CloudFormation. </p>  <p>You cannot use the organization names “amzn”,
   * “amazon”, “alexa”, “custom” with custom resource types. It is the first part of
   * the ResourceType up to the first ::.</p>
   */
  inline const Aws::String& GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  template <typename ResourceTypeT = Aws::String>
  void SetResourceType(ResourceTypeT&& value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = std::forward<ResourceTypeT>(value);
  }
  template <typename ResourceTypeT = Aws::String>
  PutResourceConfigRequest& WithResourceType(ResourceTypeT&& value) {
    SetResourceType(std::forward<ResourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Version of the schema registered for the ResourceType in CloudFormation.</p>
   */
  inline const Aws::String& GetSchemaVersionId() const { return m_schemaVersionId; }
  inline bool SchemaVersionIdHasBeenSet() const { return m_schemaVersionIdHasBeenSet; }
  template <typename SchemaVersionIdT = Aws::String>
  void SetSchemaVersionId(SchemaVersionIdT&& value) {
    m_schemaVersionIdHasBeenSet = true;
    m_schemaVersionId = std::forward<SchemaVersionIdT>(value);
  }
  template <typename SchemaVersionIdT = Aws::String>
  PutResourceConfigRequest& WithSchemaVersionId(SchemaVersionIdT&& value) {
    SetSchemaVersionId(std::forward<SchemaVersionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Unique identifier of the resource.</p>
   */
  inline const Aws::String& GetResourceId() const { return m_resourceId; }
  inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
  template <typename ResourceIdT = Aws::String>
  void SetResourceId(ResourceIdT&& value) {
    m_resourceIdHasBeenSet = true;
    m_resourceId = std::forward<ResourceIdT>(value);
  }
  template <typename ResourceIdT = Aws::String>
  PutResourceConfigRequest& WithResourceId(ResourceIdT&& value) {
    SetResourceId(std::forward<ResourceIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Name of the resource.</p>
   */
  inline const Aws::String& GetResourceName() const { return m_resourceName; }
  inline bool ResourceNameHasBeenSet() const { return m_resourceNameHasBeenSet; }
  template <typename ResourceNameT = Aws::String>
  void SetResourceName(ResourceNameT&& value) {
    m_resourceNameHasBeenSet = true;
    m_resourceName = std::forward<ResourceNameT>(value);
  }
  template <typename ResourceNameT = Aws::String>
  PutResourceConfigRequest& WithResourceName(ResourceNameT&& value) {
    SetResourceName(std::forward<ResourceNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The configuration object of the resource in valid JSON format. It must match
   * the schema registered with CloudFormation.</p>  <p>The configuration JSON
   * must not exceed 64 KB.</p>
   */
  inline const Aws::String& GetConfiguration() const { return m_configuration; }
  inline bool ConfigurationHasBeenSet() const { return m_configurationHasBeenSet; }
  template <typename ConfigurationT = Aws::String>
  void SetConfiguration(ConfigurationT&& value) {
    m_configurationHasBeenSet = true;
    m_configuration = std::forward<ConfigurationT>(value);
  }
  template <typename ConfigurationT = Aws::String>
  PutResourceConfigRequest& WithConfiguration(ConfigurationT&& value) {
    SetConfiguration(std::forward<ConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Tags associated with the resource.</p>  <p>This field is not to be
   * confused with the Amazon Web Services-wide tag feature for Amazon Web Services
   * resources. Tags for <code>PutResourceConfig</code> are tags that you supply for
   * the configuration items of your custom resources.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  PutResourceConfigRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  PutResourceConfigRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceType;

  Aws::String m_schemaVersionId;

  Aws::String m_resourceId;

  Aws::String m_resourceName;

  Aws::String m_configuration;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_resourceTypeHasBeenSet = false;
  bool m_schemaVersionIdHasBeenSet = false;
  bool m_resourceIdHasBeenSet = false;
  bool m_resourceNameHasBeenSet = false;
  bool m_configurationHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace ConfigService
}  // namespace Aws
