﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ram/RAMRequest.h>
#include <aws/ram/RAM_EXPORTS.h>
#include <aws/ram/model/ResourceOwner.h>

#include <utility>

namespace Aws {
namespace RAM {
namespace Model {

/**
 */
class ListPrincipalsRequest : public RAMRequest {
 public:
  AWS_RAM_API ListPrincipalsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "ListPrincipals"; }

  AWS_RAM_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Specifies that you want to list information for only resource shares that
   * match the following:</p> <ul> <li> <p> <b> <code>SELF</code> </b> – principals
   * that your account is sharing resources with</p> </li> <li> <p> <b>
   * <code>OTHER-ACCOUNTS</code> </b> – principals that are sharing resources with
   * your account</p> </li> </ul>
   */
  inline ResourceOwner GetResourceOwner() const { return m_resourceOwner; }
  inline bool ResourceOwnerHasBeenSet() const { return m_resourceOwnerHasBeenSet; }
  inline void SetResourceOwner(ResourceOwner value) {
    m_resourceOwnerHasBeenSet = true;
    m_resourceOwner = value;
  }
  inline ListPrincipalsRequest& WithResourceOwner(ResourceOwner value) {
    SetResourceOwner(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that you want to list principal information for the resource share
   * with the specified <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a>.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  ListPrincipalsRequest& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that you want to list information for only the listed
   * principals.</p> <p>You can include the following values:</p> <ul> <li> <p>An
   * Amazon Web Services account ID, for example: <code>123456789012</code> </p>
   * </li> <li> <p>An <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Name (ARN)</a> of an organization in Organizations, for example:
   * <code>organizations::123456789012:organization/o-exampleorgid</code> </p> </li>
   * <li> <p>An ARN of an organizational unit (OU) in Organizations, for example:
   * <code>organizations::123456789012:ou/o-exampleorgid/ou-examplerootid-exampleouid123</code>
   * </p> </li> <li> <p>An ARN of an IAM role, for example:
   * <code>iam::123456789012:role/rolename</code> </p> </li> <li> <p>An ARN of an IAM
   * user, for example: <code>iam::123456789012user/username</code> </p> </li> <li>
   * <p>A service principal name, for example: <code>service-id.amazonaws.com</code>
   * </p> </li> </ul>  <p>Not all resource types can be shared with IAM roles
   * and users. For more information, see <a
   * href="https://docs.aws.amazon.com/ram/latest/userguide/permissions.html#permissions-rbp-supported-resource-types">Sharing
   * with IAM roles and users</a> in the <i>Resource Access Manager User
   * Guide</i>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetPrincipals() const { return m_principals; }
  inline bool PrincipalsHasBeenSet() const { return m_principalsHasBeenSet; }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  void SetPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals = std::forward<PrincipalsT>(value);
  }
  template <typename PrincipalsT = Aws::Vector<Aws::String>>
  ListPrincipalsRequest& WithPrincipals(PrincipalsT&& value) {
    SetPrincipals(std::forward<PrincipalsT>(value));
    return *this;
  }
  template <typename PrincipalsT = Aws::String>
  ListPrincipalsRequest& AddPrincipals(PrincipalsT&& value) {
    m_principalsHasBeenSet = true;
    m_principals.emplace_back(std::forward<PrincipalsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that you want to list information for only principals associated
   * with resource shares that include the specified resource type.</p> <p>For a list
   * of valid values, query the <a>ListResourceTypes</a> operation.</p>
   */
  inline const Aws::String& GetResourceType() const { return m_resourceType; }
  inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
  template <typename ResourceTypeT = Aws::String>
  void SetResourceType(ResourceTypeT&& value) {
    m_resourceTypeHasBeenSet = true;
    m_resourceType = std::forward<ResourceTypeT>(value);
  }
  template <typename ResourceTypeT = Aws::String>
  ListPrincipalsRequest& WithResourceType(ResourceTypeT&& value) {
    SetResourceType(std::forward<ResourceTypeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that you want to list information for only principals associated
   * with the resource shares specified by a list the <a
   * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
   * Resource Names (ARNs)</a>.</p>
   */
  inline const Aws::Vector<Aws::String>& GetResourceShareArns() const { return m_resourceShareArns; }
  inline bool ResourceShareArnsHasBeenSet() const { return m_resourceShareArnsHasBeenSet; }
  template <typename ResourceShareArnsT = Aws::Vector<Aws::String>>
  void SetResourceShareArns(ResourceShareArnsT&& value) {
    m_resourceShareArnsHasBeenSet = true;
    m_resourceShareArns = std::forward<ResourceShareArnsT>(value);
  }
  template <typename ResourceShareArnsT = Aws::Vector<Aws::String>>
  ListPrincipalsRequest& WithResourceShareArns(ResourceShareArnsT&& value) {
    SetResourceShareArns(std::forward<ResourceShareArnsT>(value));
    return *this;
  }
  template <typename ResourceShareArnsT = Aws::String>
  ListPrincipalsRequest& AddResourceShareArns(ResourceShareArnsT&& value) {
    m_resourceShareArnsHasBeenSet = true;
    m_resourceShareArns.emplace_back(std::forward<ResourceShareArnsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies that you want to receive the next page of results. Valid only if
   * you received a <code>NextToken</code> response in the previous request. If you
   * did, it indicates that more output is available. Set this parameter to the value
   * provided by the previous call's <code>NextToken</code> response to request the
   * next page of results.</p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  ListPrincipalsRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the total number of results that you want included on each page of
   * the response. If you do not include this parameter, it defaults to a value that
   * is specific to the operation. If additional items exist beyond the number you
   * specify, the <code>NextToken</code> response element is returned with a value
   * (not null). Include the specified value as the <code>NextToken</code> request
   * parameter in the next call to the operation to get the next part of the results.
   * Note that the service might return fewer results than the maximum even when
   * there are more results available. You should check <code>NextToken</code> after
   * every operation to ensure that you receive all of the results.</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline ListPrincipalsRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}
 private:
  ResourceOwner m_resourceOwner{ResourceOwner::NOT_SET};

  Aws::String m_resourceArn;

  Aws::Vector<Aws::String> m_principals;

  Aws::String m_resourceType;

  Aws::Vector<Aws::String> m_resourceShareArns;

  Aws::String m_nextToken;

  int m_maxResults{0};
  bool m_resourceOwnerHasBeenSet = false;
  bool m_resourceArnHasBeenSet = false;
  bool m_principalsHasBeenSet = false;
  bool m_resourceTypeHasBeenSet = false;
  bool m_resourceShareArnsHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
};

}  // namespace Model
}  // namespace RAM
}  // namespace Aws
