﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/personalize/PersonalizeRequest.h>
#include <aws/personalize/Personalize_EXPORTS.h>
#include <aws/personalize/model/BatchSegmentJobInput.h>
#include <aws/personalize/model/BatchSegmentJobOutput.h>
#include <aws/personalize/model/Tag.h>

#include <utility>

namespace Aws {
namespace Personalize {
namespace Model {

/**
 */
class CreateBatchSegmentJobRequest : public PersonalizeRequest {
 public:
  AWS_PERSONALIZE_API CreateBatchSegmentJobRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateBatchSegmentJob"; }

  AWS_PERSONALIZE_API Aws::String SerializePayload() const override;

  AWS_PERSONALIZE_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The name of the batch segment job to create.</p>
   */
  inline const Aws::String& GetJobName() const { return m_jobName; }
  inline bool JobNameHasBeenSet() const { return m_jobNameHasBeenSet; }
  template <typename JobNameT = Aws::String>
  void SetJobName(JobNameT&& value) {
    m_jobNameHasBeenSet = true;
    m_jobName = std::forward<JobNameT>(value);
  }
  template <typename JobNameT = Aws::String>
  CreateBatchSegmentJobRequest& WithJobName(JobNameT&& value) {
    SetJobName(std::forward<JobNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the solution version you want the batch
   * segment job to use to generate batch segments.</p>
   */
  inline const Aws::String& GetSolutionVersionArn() const { return m_solutionVersionArn; }
  inline bool SolutionVersionArnHasBeenSet() const { return m_solutionVersionArnHasBeenSet; }
  template <typename SolutionVersionArnT = Aws::String>
  void SetSolutionVersionArn(SolutionVersionArnT&& value) {
    m_solutionVersionArnHasBeenSet = true;
    m_solutionVersionArn = std::forward<SolutionVersionArnT>(value);
  }
  template <typename SolutionVersionArnT = Aws::String>
  CreateBatchSegmentJobRequest& WithSolutionVersionArn(SolutionVersionArnT&& value) {
    SetSolutionVersionArn(std::forward<SolutionVersionArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the filter to apply to the batch segment job. For more information
   * on using filters, see <a
   * href="https://docs.aws.amazon.com/personalize/latest/dg/filter-batch.html">Filtering
   * batch recommendations</a>.</p>
   */
  inline const Aws::String& GetFilterArn() const { return m_filterArn; }
  inline bool FilterArnHasBeenSet() const { return m_filterArnHasBeenSet; }
  template <typename FilterArnT = Aws::String>
  void SetFilterArn(FilterArnT&& value) {
    m_filterArnHasBeenSet = true;
    m_filterArn = std::forward<FilterArnT>(value);
  }
  template <typename FilterArnT = Aws::String>
  CreateBatchSegmentJobRequest& WithFilterArn(FilterArnT&& value) {
    SetFilterArn(std::forward<FilterArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The number of predicted users generated by the batch segment job for each
   * line of input data. The maximum number of users per segment is 5 million.</p>
   */
  inline int GetNumResults() const { return m_numResults; }
  inline bool NumResultsHasBeenSet() const { return m_numResultsHasBeenSet; }
  inline void SetNumResults(int value) {
    m_numResultsHasBeenSet = true;
    m_numResults = value;
  }
  inline CreateBatchSegmentJobRequest& WithNumResults(int value) {
    SetNumResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 path for the input data used to generate the batch segment
   * job.</p>
   */
  inline const BatchSegmentJobInput& GetJobInput() const { return m_jobInput; }
  inline bool JobInputHasBeenSet() const { return m_jobInputHasBeenSet; }
  template <typename JobInputT = BatchSegmentJobInput>
  void SetJobInput(JobInputT&& value) {
    m_jobInputHasBeenSet = true;
    m_jobInput = std::forward<JobInputT>(value);
  }
  template <typename JobInputT = BatchSegmentJobInput>
  CreateBatchSegmentJobRequest& WithJobInput(JobInputT&& value) {
    SetJobInput(std::forward<JobInputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 path for the bucket where the job's output will be stored.</p>
   */
  inline const BatchSegmentJobOutput& GetJobOutput() const { return m_jobOutput; }
  inline bool JobOutputHasBeenSet() const { return m_jobOutputHasBeenSet; }
  template <typename JobOutputT = BatchSegmentJobOutput>
  void SetJobOutput(JobOutputT&& value) {
    m_jobOutputHasBeenSet = true;
    m_jobOutput = std::forward<JobOutputT>(value);
  }
  template <typename JobOutputT = BatchSegmentJobOutput>
  CreateBatchSegmentJobRequest& WithJobOutput(JobOutputT&& value) {
    SetJobOutput(std::forward<JobOutputT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the Amazon Identity and Access Management role that has
   * permissions to read and write to your input and output Amazon S3 buckets
   * respectively.</p>
   */
  inline const Aws::String& GetRoleArn() const { return m_roleArn; }
  inline bool RoleArnHasBeenSet() const { return m_roleArnHasBeenSet; }
  template <typename RoleArnT = Aws::String>
  void SetRoleArn(RoleArnT&& value) {
    m_roleArnHasBeenSet = true;
    m_roleArn = std::forward<RoleArnT>(value);
  }
  template <typename RoleArnT = Aws::String>
  CreateBatchSegmentJobRequest& WithRoleArn(RoleArnT&& value) {
    SetRoleArn(std::forward<RoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <a
   * href="https://docs.aws.amazon.com/personalize/latest/dg/tagging-resources.html">tags</a>
   * to apply to the batch segment job.</p>
   */
  inline const Aws::Vector<Tag>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Vector<Tag>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Vector<Tag>>
  CreateBatchSegmentJobRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsT = Tag>
  CreateBatchSegmentJobRequest& AddTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace_back(std::forward<TagsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_jobName;

  Aws::String m_solutionVersionArn;

  Aws::String m_filterArn;

  int m_numResults{0};

  BatchSegmentJobInput m_jobInput;

  BatchSegmentJobOutput m_jobOutput;

  Aws::String m_roleArn;

  Aws::Vector<Tag> m_tags;
  bool m_jobNameHasBeenSet = false;
  bool m_solutionVersionArnHasBeenSet = false;
  bool m_filterArnHasBeenSet = false;
  bool m_numResultsHasBeenSet = false;
  bool m_jobInputHasBeenSet = false;
  bool m_jobOutputHasBeenSet = false;
  bool m_roleArnHasBeenSet = false;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace Personalize
}  // namespace Aws
