"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerForecastRoutes = registerForecastRoutes;
var _lodash = require("lodash");
var _constants = require("../utils/constants");
var _helpers = require("../utils/helpers");
var _forecastHelpers = require("./utils/forecastHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerForecastRoutes(apiRouter, forecastService) {
  // create forecaster
  apiRouter.post('/forecasters', forecastService.putForecaster);
  apiRouter.post('/forecasters/{dataSourceId}', forecastService.putForecaster);

  // put forecaster
  apiRouter.put('/forecasters/{forecasterId}', forecastService.putForecaster);
  apiRouter.put('/forecasters/{forecasterId}/{dataSourceId}', forecastService.putForecaster);

  // FIXME: routes not used in the UI, therefore no data source id
  apiRouter.post('/forecasters/_search', forecastService.searchForecaster);

  /**
   * Search forecast results routes
   * 
   * We use 'by-source' and 'by-index' path segments to avoid route conflicts between
   * paths with different parameter types. Without these segments, routes like:
   *   /forecasters/results/_search/{resultIndex}
   *   /forecasters/results/_search/{dataSourceId}
   * would conflict because OpenSearch can't distinguish between parameter types in the same position.
   * 
   * Current route structure:
   * 1. Search by source (no params)     : /by-source/_search
   * 2. Search by source with ID         : /by-source/{dataSourceId}/_search
   * 3. Search by index pattern          : /by-index/{resultIndex}/_search
   * 4. Search by both index and source  : /by-index/{resultIndex}/by-source/{dataSourceId}/_search
   */

  // Search with no parameters
  apiRouter.post('/forecasters/results/by-source/_search', forecastService.searchResults);

  // Search by data source ID
  apiRouter.post('/forecasters/results/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // Search by result index pattern
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/_search', forecastService.searchResults);

  // Search by both result index and data source ID
  apiRouter.post('/forecasters/results/by-index/{resultIndex}/by-source/{dataSourceId}/_search', forecastService.searchResults);

  // list forecasters
  apiRouter.get('/forecasters/_list', forecastService.getForecasters);
  apiRouter.get('/forecasters/_list/{dataSourceId}', forecastService.getForecasters);

  // run once forecaster
  apiRouter.post('/forecasters/{forecasterId}/_run_once', forecastService.runOnceForecaster);
  apiRouter.post('/forecasters/{forecasterId}/_run_once/{dataSourceId}', forecastService.runOnceForecaster);

  // get forecaster forecast results
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}', forecastService.getForecastResults);
  apiRouter.get('/forecasters/{id}/results/{isRunOnce}/{resultIndex}/{dataSourceId}', forecastService.getForecastResults);

  // delete forecaster
  apiRouter.delete('/forecasters/{forecasterId}', forecastService.deleteForecaster);
  apiRouter.delete('/forecasters/{forecasterId}/{dataSourceId}', forecastService.deleteForecaster);

  // start forecaster
  apiRouter.post('/forecasters/{forecasterId}/start', forecastService.startForecaster);
  apiRouter.post('/forecasters/{forecasterId}/start/{dataSourceId}', forecastService.startForecaster);

  // stop forecaster
  apiRouter.post('/forecasters/{forecasterId}/stop', forecastService.stopForecaster);
  apiRouter.post('/forecasters/{forecasterId}/stop/{dataSourceId}', forecastService.stopForecaster);
  apiRouter.get('/forecasters/{forecasterId}/_profile', forecastService.getForecasterProfile);

  // get forecaster
  apiRouter.get('/forecasters/{forecasterId}', forecastService.getForecaster);
  apiRouter.get('/forecasters/{forecasterId}/{dataSourceId}', forecastService.getForecaster);

  // match forecaster
  apiRouter.get('/forecasters/{forecasterName}/_match', forecastService.matchForecaster);
  apiRouter.get('/forecasters/{forecasterName}/_match/{dataSourceId}', forecastService.matchForecaster);

  // get forecaster count
  apiRouter.get('/forecasters/_count', forecastService.getForecasterCount);
  apiRouter.get('/forecasters/_count/{dataSourceId}', forecastService.getForecasterCount);

  // post get top forecast results
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}', forecastService.getTopForecastResults);
  apiRouter.post('/forecasters/{forecasterId}/_topForecasts/{isRunOnce}/{dataSourceId}', forecastService.getTopForecastResults);

  // validate forecaster
  apiRouter.post('/forecasters/_validate/{validationType}', forecastService.validateForecaster);
  apiRouter.post('/forecasters/_validate/{validationType}/{dataSourceId}', forecastService.validateForecaster);

  // suggest forecaster
  apiRouter.post('/forecasters/_suggest/{suggestType}', forecastService.suggestForecaster);
  apiRouter.post('/forecasters/_suggest/{suggestType}/{dataSourceId}', forecastService.suggestForecaster);
}
class ForecastService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "deleteForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        // dataSourceId will be "" and fall back to use the existing client for local cluster
        // On the other hand, in MDS world, the open search legacy client (this.client) will
        // be undefined and it will pickup the data source client 
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.deleteForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - deleteForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "runOnceForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId = '',
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.runOnceForecaster', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            // return taskId
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - runOnceForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "putForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;

        //@ts-ignore
        const ifSeqNo = request.body.seqNo;
        //@ts-ignore
        const ifPrimaryTerm = request.body.primaryTerm;
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        let params = {
          forecasterId: forecasterId,
          ifSeqNo: ifSeqNo,
          ifPrimaryTerm: ifPrimaryTerm,
          body: requestBody
        };
        let response;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        if ((0, _lodash.isNumber)(ifSeqNo) && (0, _lodash.isNumber)(ifPrimaryTerm)) {
          response = await callWithRequest('forecast.updateForecaster', params);
        } else {
          response = await callWithRequest('forecast.createForecaster', {
            body: params.body
          });
        }
        const resp = {
          ...response.forecaster,
          id: response._id,
          primaryTerm: response._primary_term,
          seqNo: response._seq_no
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: (0, _forecastHelpers.convertForecastKeysToCamelCase)(resp)
          }
        });
      } catch (err) {
        console.log('Forecast - PutForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "validateForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          validationType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.validateForecaster', {
          body: requestBody,
          validationType: validationType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - validateForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "suggestForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          suggestType
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const requestBody = JSON.stringify((0, _forecastHelpers.convertForecastKeysToSnakeCase)(request.body));
        const response = await callWithRequest('forecast.suggestForecaster', {
          body: requestBody,
          suggestType: suggestType
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - suggestForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const forecasterResponse = await callWithRequest('forecast.getForecaster', {
          forecasterId
        });

        // Populating static forecaster fields
        const staticFields = {
          id: forecasterResponse._id,
          primaryTerm: forecasterResponse._primary_term,
          seqNo: forecasterResponse._seq_no,
          // the backend returns a response with forecaster field.
          ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse.forecaster)
        };

        // Get real-time and run-once task info to populate the
        // task and job-related fields
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        // let realtimeTasksResponse = {} as any;
        // let runOnceTasksResponse = {} as any;
        // try {
        //   const callWithRequest = getClientBasedOnDataSource(
        //     context,
        //     this.dataSourceEnabled,
        //     request,
        //     dataSourceId,
        //     this.client
        //   );

        //   realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, true),
        //   });

        //   runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
        //     body: getLatestTaskForForecasterQuery(forecasterId, false),
        //   });
        // } catch (err) {
        //   if (!isIndexNotFoundError(err)) {
        //     throw err;
        //   }
        // }

        // const realtimeTask = get(
        //   get(realtimeTasksResponse, 'hits.hits', []).map((taskResponse: any) => {
        //     return {
        //       id: get(taskResponse, '_id'),
        //       ...get(taskResponse, '_source'),
        //     };
        //   }),
        //   0
        // );
        // const runOnceTask = get(
        //   get(runOnceTasksResponse, 'hits.hits', []).map(
        //     (taskResponse: any) => {
        //       return {
        //         id: get(taskResponse, '_id'),
        //         ...get(taskResponse, '_source'),
        //       };
        //     }
        //   ),
        //   0
        // );

        const taskAndJobFields = (0, _forecastHelpers.convertTaskAndJobFieldsToCamelCase)(forecasterResponse.realtime_task, forecasterResponse.run_once_task, forecasterResponse.forecaster_job);

        // Combine the static and task-and-job-related fields into
        // a final response
        const finalResponse = {
          ...staticFields,
          ...taskAndJobFields
        };
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: finalResponse
          }
        });
      } catch (err) {
        // if the forecaster is not found (e.g., deleted while on the detail page), return an empty response
        // this is to avoid the error message from the frontend where the forecaster is not found
        // the error is triggered by useEffect of useFetchForecasterInfo in ForecasterDetail
        if (err.statusCode === 404) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {}
            }
          });
        }
        console.log('Forecast - Unable to get forecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "startForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        var _request$body, _request$body2;
        const {
          forecasterId
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        //@ts-ignore
        const startTime = (_request$body = request.body) === null || _request$body === void 0 ? void 0 : _request$body.startTime;
        //@ts-ignore
        const endTime = (_request$body2 = request.body) === null || _request$body2 === void 0 ? void 0 : _request$body2.endTime;
        let requestParams = {
          forecasterId: forecasterId
        };
        let requestPath = 'forecast.startForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, requestParams);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - startForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "stopForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        // Extract required parameters with specific type assertion.
        // 'forecasterId' is expected to always be present in the route path.
        let {
          forecasterId
        } = request.params;
        // Extract optional parameters separately.
        // 'dataSourceId' might be missing from the route path (hence '?').
        // Provide a default value ('') if it's not present using destructuring default assignment.
        const {
          dataSourceId = ''
        } = request.params;
        const requestPath = 'forecast.stopForecaster';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - stopForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterProfile", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterId
        } = request.params;
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.forecasterProfile', {
          forecasterId
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Forecast - forecasterProfile', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "searchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = JSON.stringify(request.body);
        const response = await this.client.asScoped(request).callAsCurrentUser('forecast.searchForecaster', {
          body: requestBody
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);
        const forecasters = (0, _lodash.get)(response, 'hits.hits', []).map(forecaster => ({
          ...(0, _forecastHelpers.convertForecastKeysToCamelCase)(forecaster._source),
          id: forecaster._id,
          seqNo: forecaster._seq_no,
          primaryTerm: forecaster._primary_term
        }));
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters,
              forecasters
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasters", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.searchForecaster', {
          body: {}
        });
        const totalForecasters = (0, _lodash.get)(response, 'hits.total.value', 0);

        //Get all forecasters from search forecaster API
        const allForecasters = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, forecasterResponse) => ({
          ...acc,
          [forecasterResponse._id]: {
            id: forecasterResponse._id,
            primaryTerm: forecasterResponse._primary_term,
            seqNo: forecasterResponse._seq_no,
            ...(0, _forecastHelpers.convertStaticFieldsToCamelCase)(forecasterResponse._source)
          }
        }), {});

        // Fetch the latest realtime and runOnce tasks for all forecasters
        // using terms aggregations
        // We wrap these calls in a try/catch, and suppress any index_not_found_exceptions
        // which can occur if no forecaster jobs have been ran on a new cluster.
        let realtimeTasksResponse = {};
        let runOnceTasksResponse = {};
        try {
          realtimeTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(true)
          });
          runOnceTasksResponse = await callWithRequest('forecast.searchTasks', {
            body: (0, _forecastHelpers.getLatestForecasterTasksQuery)(false)
          });
        } catch (err) {
          if (!(0, _forecastHelpers.isIndexNotFoundError)(err)) {
            throw err;
          }
        }
        const realtimeTasks = (0, _lodash.get)(realtimeTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              realtimeTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});
        const runOnceTasks = (0, _lodash.get)(runOnceTasksResponse, 'aggregations.forecasters.buckets', []).reduce((acc, bucket) => {
          return {
            ...acc,
            [bucket.key]: {
              runOnceTask: (0, _lodash.get)(bucket, 'latest_tasks.hits.hits.0', undefined)
            }
          };
        }, {});

        // Get real-time and runOnce task info by looping through each forecaster & retrieving
        //    - curState by getting real-time task state
        //    - enabledTime by getting real-time task's execution_start time
        //    - taskId by getting historical task's _id
        const forecastersArray = Object.values(allForecasters);
        forecastersArray.forEach(forecaster => {
          const realtimeTask = (0, _lodash.get)(realtimeTasks[forecaster.id], 'realtimeTask._source');
          const runOnceTask = (0, _lodash.get)(runOnceTasks[forecaster.id], 'runOnceTask._source');
          forecaster.curState = (0, _forecastHelpers.combineTaskState)(realtimeTask, runOnceTask);
          forecaster.realTimeLastUpdateTime = (0, _lodash.get)(realtimeTask, 'last_update_time');
          forecaster.runOnceLastUpdateTime = (0, _lodash.get)(runOnceTask, 'last_update_time');
          forecaster.stateError = (0, _lodash.get)(realtimeTask, 'error') || (0, _lodash.get)(runOnceTask, 'error');
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalForecasters: totalForecasters,
              forecasterList: forecastersArray
            }
          }
        });
      } catch (err) {
        console.log('Forecaster - Unable to search forecasters', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasterList: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecastResults", async (context, request, opensearchDashboardsResponse) => {
      let {
        id,
        isRunOnce,
        resultIndex
      } = request.params;
      const {
        dataSourceId = ''
      } = request.params;
      if (!resultIndex) {
        // Not strictly querying custom ⇒ default to ''
        resultIndex = '';
      } else if (!resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
        // If resultIndex is given but not valid, revert to default
        resultIndex = '';
      }
      isRunOnce = JSON.parse(isRunOnce);

      // Search by task id if runOnce, or by forecaster id if realtime
      const searchTerm = isRunOnce ? {
        task_id: id
      } : {
        forecaster_id: id
      };
      try {
        const {
          size = 20,
          sortDirection = _constants.SORT_DIRECTION.DESC,
          sortField = _constants.FORECASTER_DOC_FIELDS.DATA_START_TIME,
          startTime = 0,
          endTime = 0,
          fieldName = '',
          entityList = '',
          dawnEpoch = 0,
          maxEntities = 0
        } = request.query;

        //Allowed sorting columns
        const sortQueryMap = {
          [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_START_TIME]: sortDirection
          },
          [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: {
            [_constants.FORECASTER_DOC_FIELDS.DATA_END_TIME]: sortDirection
          }
        };
        let sort = {};
        const sortQuery = sortQueryMap[sortField];
        if (sortQuery) {
          sort = sortQuery;
        }

        //Preparing search request
        const requestBody = {
          sort,
          size,
          query: {
            bool: {
              filter: [{
                term: searchTerm
              }]
            }
          }
        };

        // If querying RT results: remove any results that include a task_id, as this indicates
        // a runOnce result from a runOnce task.
        if (!isRunOnce) {
          requestBody.query.bool = {
            ...requestBody.query.bool,
            ...{
              must_not: {
                exists: {
                  field: 'task_id'
                }
              }
            }
          };
        }
        try {
          // Get current number of filters to determine the index for adding new date range filter
          // This includes the initial term filter and any entity filters that were added
          let filterSize = requestBody.query.bool.filter.length;
          if (fieldName) {
            (startTime || endTime) && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.format`, 'epoch_millis');
            startTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.gte`, startTime);
            endTime && (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${fieldName}.lte`, endTime);
          }
          filterSize = requestBody.query.bool.filter.length;

          // Add dawnEpoch filter if it exists
          if (dawnEpoch > 0) {
            (0, _lodash.set)(requestBody.query.bool.filter, `${filterSize}.range.${_constants.FORECASTER_DOC_FIELDS.EXECUTION_END_TIME}.gte`, dawnEpoch);
          }
        } catch (error) {
          console.log('wrong date range filter', error);
        }

        // ─────────────────────────────────────────────────────────────
        // If maxEntities > 0, find top N entity_ids.
        // ─────────────────────────────────────────────────────────────
        let restrictedEntityIds = [];
        if (maxEntities > 0) {
          const entityListAsObj = entityList.length === 0 ? {} : JSON.parse(entityList);
          const entityFilters = (0, _lodash.isEmpty)(entityListAsObj) ? {} : (0, _forecastHelpers.buildEntityListQuery)(entityListAsObj);

          // Only clone and modify requestBody if entityFilters exists and is not empty/null
          let queryForAggregation;
          if (entityFilters && typeof entityFilters === 'object' && Object.keys(entityFilters).length > 0) {
            // Create a deep clone of the request body
            const clonedRequestBody = JSON.parse(JSON.stringify(requestBody));

            // Add entity filters to the filter array of the cloned request body
            if (!clonedRequestBody.query) {
              clonedRequestBody.query = {
                bool: {
                  filter: []
                }
              };
            } else if (!clonedRequestBody.query.bool) {
              clonedRequestBody.query.bool = {
                filter: []
              };
            } else if (!clonedRequestBody.query.bool.filter) {
              clonedRequestBody.query.bool.filter = [];
            }

            // Add the entity filter object to the filter array
            clonedRequestBody.query.bool.filter.push(entityFilters);
            queryForAggregation = clonedRequestBody.query;
          } else {
            // Use the original requestBody if no entity filters to add
            queryForAggregation = requestBody.query;
          }

          // Example aggregatorRequestBody:
          // {
          //   "size": 0,
          //   "query": {
          //     "bool": {
          //       "filter": [
          //         {"term": {"task_id": "BsLQbZUBxkwQb14j93bF"}},
          //         {"range": {"execution_end_time": {"gte": "0"}}},
          //         {
          //           "bool": {
          //             "should": [
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_3"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               },
          //               {
          //                 "bool": {
          //                   "must": [
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "service"}}, {"term": {"entity.value": "app_6"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     },
          //                     {
          //                       "nested": {
          //                         "path": "entity",
          //                         "query": {"bool": {"must": [{"term": {"entity.name": "host"}}, {"term": {"entity.value": "server_1"}}]}},
          //                         "ignore_unmapped": false,
          //                         "score_mode": "avg"
          //                       }
          //                     }
          //                   ]
          //                 }
          //               }
          //             ],
          //             "minimum_should_match": 1
          //           }
          //         }
          //       ]
          //     }
          //   },
          //   "aggs": {
          //     "top_entities": {
          //       "terms": {
          //         "field": "entity_id",
          //         "size": 5,
          //         "order": {"_count": "desc"}
          //       }
          //     }
          //   }
          // }

          // Now use the appropriate query in aggregatorRequestBody
          const aggregatorRequestBody = {
            size: 0,
            query: queryForAggregation,
            aggs: {
              top_entities: {
                terms: {
                  field: 'entity_id',
                  size: maxEntities,
                  order: {
                    _count: 'desc'
                  }
                }
              }
            }
          };

          // We'll call the same or custom search method:
          const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
          const aggResponse = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: aggregatorRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            resultIndex: resultIndex,
            body: aggregatorRequestBody
          });

          // Extract top entity_ids
          const topEntityBuckets = (0, _lodash.get)(aggResponse, 'aggregations.top_entities.buckets', []);
          restrictedEntityIds = topEntityBuckets.map(b => b.key);

          // If no entities matched, return empty
          if (!restrictedEntityIds.length) {
            return opensearchDashboardsResponse.ok({
              body: {
                ok: true,
                response: {
                  totalResults: 0,
                  results: []
                }
              }
            });
          }
        }

        // ─────────────────────────────────────────────────────────────
        // Add a terms filter to restrict final hits if we have top entities
        // ─────────────────────────────────────────────────────────────
        if (restrictedEntityIds.length > 0) {
          requestBody.query.bool.filter.push({
            terms: {
              entity_id: restrictedEntityIds
            }
          });
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Add pagination with search_after
        let allResults = [];
        let lastSort = null;
        let hasMoreResults = true;
        let totalHits = 0;

        // Create a copy of your existing requestBody to use in the pagination loop
        const paginatedRequestBody = {
          ...requestBody,
          "track_total_hits": true // Add this to ensure accurate total count for large result sets
        };

        // Add sort if not already present
        if (!paginatedRequestBody.sort) {
          paginatedRequestBody.sort = [{
            [sortField]: sortDirection.toLowerCase()
          }, {
            "_id": "asc"
          } // Secondary sort for tiebreaker
          ];
        }

        // Execute paginated search
        while (hasMoreResults) {
          // Add search_after for subsequent pages
          if (lastSort) {
            paginatedRequestBody.search_after = lastSort;
          }

          // Your existing API call, but with our paginated request body
          const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
            body: paginatedRequestBody
          }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
            ...requestParams,
            body: paginatedRequestBody
          });
          const hits = (0, _lodash.get)(response, 'hits.hits', []);

          // Track total hits from first page
          if (!lastSort) {
            totalHits = (0, _lodash.get)(response, 'hits.total.value', 0);
          }
          if (hits.length === 0 || hits.length < size) {
            hasMoreResults = false;
          }
          if (hits.length > 0) {
            // Save sort values from last hit for next iteration
            lastSort = hits[hits.length - 1].sort;

            // Collect results
            allResults.push(...hits);
          }
        }
        const groupedResults = new Map();
        allResults.forEach(result => {
          const source = result._source;
          const key = `${source.forecaster_id}|${source.entity_id || 'default'}|${source.data_end_time}`;
          if (!groupedResults.has(key)) {
            groupedResults.set(key, {
              featureData: null,
              forecasts: []
            });
          }
          if (source.feature_data) {
            groupedResults.get(key).featureData = result;
          } else {
            groupedResults.get(key).forecasts.push(result);
          }
        });
        const forecastResult = [];

        // Process each group
        groupedResults.forEach(({
          featureData,
          forecasts
        }) => {
          if (!featureData) return; // Skip if no feature data found

          // Check if any forecast has horizon_index
          const hasHorizonIndex = forecasts.some(forecast => forecast._source.horizon_index != null);
          if (hasHorizonIndex) {
            // Sort forecasts by horizon_index and combine into arrays
            const sortedForecasts = (0, _lodash.orderBy)(forecasts, ['_source.horizon_index'], ['asc']);
            const forecastValues = [];
            const forecastLowerBounds = [];
            const forecastUpperBounds = [];
            const forecastStartTimes = [];
            const forecastEndTimes = [];
            sortedForecasts.forEach(forecast => {
              const source = forecast._source;
              forecastValues.push(source.forecast_value != null && source.forecast_value !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_value)) : 0);
              forecastLowerBounds.push(source.forecast_lower_bound != null && source.forecast_lower_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_lower_bound)) : 0);
              forecastUpperBounds.push(source.forecast_upper_bound != null && source.forecast_upper_bound !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(source.forecast_upper_bound)) : 0);
              forecastStartTimes.push(source.forecast_data_start_time);
              forecastEndTimes.push(source.forecast_data_end_time);
            });
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: forecastValues,
              forecastLowerBound: forecastLowerBounds,
              forecastUpperBound: forecastUpperBounds,
              forecastStartTime: forecastStartTimes,
              forecastEndTime: forecastEndTimes,
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          } else {
            // Direct push for single forecasts without horizon_index
            forecastResult.push({
              startTime: featureData._source.data_start_time,
              endTime: featureData._source.data_end_time,
              plotTime: featureData._source.data_end_time,
              forecastValue: [],
              forecastLowerBound: [],
              forecastUpperBound: [],
              forecastStartTime: [],
              forecastEndTime: [],
              ...(featureData._source.entity != null ? {
                entity: featureData._source.entity,
                entityId: featureData._source.entity_id
              } : {}),
              features: this.getFeatureData(featureData)
            });
          }
        });

        // Sort final results by plotTime
        const sortedForecastResult = (0, _lodash.orderBy)(forecastResult, ['plotTime'], ['asc']);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalResults: totalHits,
              results: sortedForecastResult
            }
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to get results', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getTopForecastResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        let {
          forecasterId,
          isRunOnce
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        isRunOnce = JSON.parse(isRunOnce);
        const requestPath = 'forecast.topForecastResults';
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);

        // Define proper types for OpenSearch query structures

        // Extract the query parameters from the request body with defaults
        const {
          split_by = '',
          filter_by = '',
          build_in_query = '',
          forecast_from = 0,
          threshold = 0,
          relation_to_threshold = '',
          filter_query = {},
          subaggregations = []
        } = request.body || {};

        // Build query object with appropriate parameters
        const queryBody = {};

        // Add split_by if present
        if (split_by) {
          queryBody.split_by = split_by;
        }

        // Add filter_by and related parameters
        if (filter_by) {
          queryBody.filter_by = filter_by;
          if (filter_by === 'BUILD_IN_QUERY' && build_in_query) {
            queryBody.build_in_query = build_in_query;

            // Add threshold parameters if build_in_query is DISTANCE_TO_THRESHOLD_VALUE
            if (build_in_query === 'DISTANCE_TO_THRESHOLD_VALUE') {
              queryBody.threshold = threshold;
              queryBody.relation_to_threshold = relation_to_threshold;
            }
          } else if (filter_by === 'CUSTOM_QUERY') {
            // Add custom query parameters - check if the objects are not empty
            if (Object.keys(filter_query).length > 0) {
              queryBody.filter_query = filter_query;
            }
            if (subaggregations.length > 0) {
              queryBody.subaggregations = subaggregations;
            }
          }
        }

        // Add forecast_from timestamp if present
        if (forecast_from) {
          queryBody.forecast_from = forecast_from;
        }

        // Add run_once to body if isRunOnce is true
        const requestBody = {
          ...queryBody,
          ...(isRunOnce && {
            run_once: true
          })
        };
        const response = await callWithRequest(requestPath, {
          forecasterId: forecasterId,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getTopForecastResults', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "matchForecaster", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          forecasterName
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.matchForecaster', {
          forecasterName
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - matchForecaster', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getForecasterCount", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          dataSourceId = ''
        } = request.params;
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('forecast.forecasterCount');
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - getForecasterCount', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getFeatureData", rawResult => {
      const featureResult = {};
      rawResult._source.feature_data.forEach(featureData => {
        featureResult[featureData.feature_id] = {
          startTime: rawResult._source.data_start_time,
          endTime: rawResult._source.data_end_time,
          plotTime: rawResult._source.data_end_time,
          data: featureData.data != null && featureData.data !== 'NaN' ? (0, _helpers.toFixedNumberForForecast)(Number.parseFloat(featureData.data)) : 0,
          name: featureData.feature_name
        };
      });
      return featureResult;
    });
    _defineProperty(this, "searchResults", async (context, request, opensearchDashboardsResponse) => {
      try {
        var {
          resultIndex
        } = request.params;
        const {
          dataSourceId = ''
        } = request.params;
        if (!resultIndex || !resultIndex.startsWith(_constants.CUSTOM_FORECAST_RESULT_INDEX_PREFIX)) {
          // Set resultIndex as '' means no custom result index specified, will only search forecast result from default index.
          resultIndex = '';
        }
        let requestParams = {
          resultIndex: resultIndex
        };
        const requestBody = JSON.stringify(request.body);
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = !resultIndex ? await callWithRequest('forecast.searchResults', {
          body: requestBody
        }) : await callWithRequest('forecast.searchResultsFromCustomResultIndex', {
          ...requestParams,
          body: requestBody
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: response
          }
        });
      } catch (err) {
        console.log('Forecast - Unable to search forecast result', err);
        if ((0, _forecastHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                totalForecasters: 0,
                forecasters: []
              }
            }
          });
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _forecastHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = ForecastService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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