/*
    SPDX-FileCopyrightText: 2022 Volker Krause <vkrause@kde.org>
    SPDX-License-Identifier: LGPL-2.0-or-later
*/

import QtQuick
import QtQuick.Layouts
import QtQuick.Controls as QQC2
import org.kde.kirigami as Kirigami
import org.kde.kirigamiaddons.formcard as FormCard
import org.kde.kitinerary
import org.kde.itinerary

FormCard.FormCardPage {
    id: root

    title: i18n("Program Membership")

    property string passId
    property var programMembership

    data: [
        Connections {
            target: PassManager
            function onPassChanged(passId) {
                if (passId == root.passId)
                    root.programMembership = PassManager.pass(passId);
            }
        },
        BarcodeScanModeButton {
            id: scanModeController
            page: root
            visible: barcodeContainer.visible
        },
        Kirigami.PromptDialog {
            id: deleteWarningDialog

            title: i18n("Delete Pass")
            subtitle: i18n("Do you really want to delete this pass?")

            standardButtons: QQC2.Dialog.Cancel

            customFooterActions: [
                Kirigami.Action {
                    text: i18n("Delete")
                    icon.name: "edit-delete"
                    onTriggered: {
                        PassManager.remove(passId)
                        root.QQC2.ApplicationWindow.window.pageStack.pop();
                    }
                }
            ]
        }
    ]

    ColumnLayout {
        spacing: 0

        CardPageTitle {
            emojiIcon: "🎫"
            text: programMembership.programName
            Layout.topMargin: Kirigami.Units.smallSpacing
        }

        FormCard.FormCard {
            visible: programMembership.token.length > 0
            BarcodeContainer {
                id: barcodeContainer
                Layout.alignment: Qt.AlignCenter
                Layout.fillWidth: true
                Layout.topMargin: Kirigami.Units.largeSpacing
                Layout.bottomMargin: Kirigami.Units.largeSpacing

                barcodeType: programMembership.tokenType
                barcodeContent: programMembership.tokenData
                onDoubleClicked: scanModeController.toggle()
            }
        }

        FormCard.FormHeader {
            title: i18n("Membership")
        }
        FormCard.FormCard {
            FormCard.FormTextDelegate {
                id: memberNameLabel
                text: i18n("Member")
                description: programMembership.member.name
                visible: programMembership.member.name !== ""
            }
            FormCard.FormDelegateSeparator {
                visible: memberNameLabel.visible
            }
            FormCard.FormTextDelegate {
                id: membershipNumberLabel
                text: i18n("Number")
                description: programMembership.membershipNumber
                visible: programMembership.membershipNumber !== ""

                trailing: QQC2.ToolButton {
                    display: QQC2.AbstractButton.IconOnly
                    text: i18nc("@info:tooltip", "Copy to Clipboard")
                    icon.name: "edit-copy"
                    onClicked: {
                        Clipboard.saveText(programMembership.membershipNumber);
                        root.QQC2.ApplicationWindow.window.showPassiveNotification(i18n("Program membership number copied to clipboard"));
                    }

                    QQC2.ToolTip.text: text
                    QQC2.ToolTip.delay: Kirigami.Units.toolTipDelay
                    QQC2.ToolTip.visible: hovered
                }
            }
            FormCard.FormDelegateSeparator {
                visible: membershipNumberLabel.visible
            }
            FormCard.FormTextDelegate {
                text: i18n("Valid from")
                description: Localizer.formatDateOrDateTimeLocal(programMembership, "validFrom")
                visible: description !== ""
            }
            FormCard.FormTextDelegate {
                text: i18n("Valid until")
                description: Localizer.formatDateOrDateTimeLocal(programMembership, "validUntil")
                visible: description !== ""
            }
        }

        DocumentsCard {
            documentIds: PassManager.documentIds(programMembership)
            onAddDocument: (file) => {
                ApplicationController.addDocumentToPass(root.passId, file);
                documentIds = Qt.binding(function() { return PassManager.documentIds(programMembership) });
            }
            onRemoveDocument: (docId) => { ApplicationController.removeDocumentFromPass(root.passId, docId); }
        }

        FormCard.FormHeader {
            title: i18n("Actions")
        }

        FormCard.FormCard {
            FormCard.FormButtonDelegate {
                id: passButton
                readonly property string passId: PkPassManager.passId(root.programMembership)
                icon.name: passButton.passId !== "" ? "image://org.kde.pkpass/" + passButton.passId + "/icon" : ""
                text: i18n("Show Pass")
                visible: PkPassManager.hasPass(passButton.passId)
                onClicked: root.QQC2.ApplicationWindow.window.pageStack.push(pkpassComponent, {"passId": passButton.passId });

                Component {
                    id: pkpassComponent
                    PkPassPage {
                        pass: PkPassManager.pass(passButton.passId)
                    }
                }
            }
            FormCard.FormButtonDelegate {
                icon.name: "document-edit"
                text: i18n("Edit")
                onClicked: {
                    const programMembershipEditor = Qt.createComponent("org.kde.itinerary", "ProgramMembershipEditor");
                    root.QQC2.ApplicationWindow.window.pageStack.push(programMembershipEditor, {
                        passId: root.passId,
                        programMembership: root.programMembership,
                        pageStack: pageStack,
                    });
                }
            }
            FormCard.FormButtonDelegate {
                icon.name: "edit-delete"
                text: i18n("Delete")
                onClicked: deleteWarningDialog.open()
            }
        }
    }

    // spacer for the floating buttons
    Item {
        visible: scanModeController.visible
        implicitHeight: root.width < Kirigami.Units.gridUnit * 30 + scanModeController.width * 2 ? scanModeController.height : 0
    }
}
