#include <syslog.h>
#include <glib-2.0/glib.h>
#include <syslog.h>
#include "json.hpp"
#include "eventlog_stub.h"
#include "common.h"
#include "cachedata.h"

using namespace std;
using json = nlohmann::json;

static GMainLoop *g_mainloop = nullptr;

static REPORTMODE getReportMode(string &eventinfo)
{
    REPORTMODE reportMode = BROADCAST;

    if (json::accept(eventinfo.c_str())) {
        json eventData = json::parse(eventinfo.c_str());
        // 旧接口（上报）
        if (!eventData.contains(MODE_FIELD)) {
            reportMode = REPORT;
        } else { // 新接口
            if (eventData.contains(POLICY_FIELD)) {
                json policyData = eventData[POLICY_FIELD];
                if (policyData.contains(REPORT_MODE_FIELD)) {
                    json modeFiled = policyData[REPORT_MODE_FIELD];
                    if (modeFiled.is_number())
                        reportMode = static_cast<REPORTMODE>(modeFiled.get<int>());
                }
                // 删除权限配置内容（外界不需要关注）
                eventData.erase(POLICY_FIELD);
            }
            eventData.erase(MODE_FIELD);
            eventinfo = eventData.dump();
        }
    }

    return reportMode;
}

// 接收 deepin-event-log 的日志事件
static gboolean OnHandleReportEventLog(
        EventLog1 *object,
        GDBusMethodInvocation *invocation,
        const gchar *arg_packagename,
        const gchar *arg_eventinfo)
{
    string eventinfo(arg_eventinfo);
    string package(arg_packagename);
    syslog(LOG_DEBUG, "received event log from package:%s", package.c_str());

    // 包名最长 128
    if (package.length() > MAX_PACKAGE_LEN)
        package = package.substr(0, MAX_PACKAGE_LEN);

    REPORTMODE reportMode = getReportMode(eventinfo);

    // 上报或广播
    if (reportMode == REPORT) {
        CacheData::get_mutable_instance().AddData(package, eventinfo);
    } else if (reportMode == BROADCAST) {
        event_log1_emit_event_log_sent(object, eventinfo.c_str());
    } else if (reportMode == REPORT_AND_BROADCAST) {
        CacheData::get_mutable_instance().AddData(package, eventinfo);
        event_log1_emit_event_log_sent(object, eventinfo.c_str());
    } else {
        event_log1_emit_event_log_sent(object, eventinfo.c_str());
    }

    event_log1_complete_report_event_log(object, invocation);

    return TRUE;
}

// 总线已建立
static void BusAcquiredCallback(GDBusConnection *connection,
                                const gchar *name,
                                gpointer user_data)
{
    syslog(LOG_DEBUG, "bus acquired");
}

// 已获取 bus name
static void NameAcquiredCallback(GDBusConnection *connection,
                                 const gchar *name,
                                 gpointer user_data)
{
    syslog(LOG_DEBUG, "name acquired");
    GError *error = nullptr;
    EventLog1 *eventlogif = event_log1_skeleton_new();
    // 信号注册：当信号 handle_report_event_log（ReportEventLog） 发射后，回调函数 OnHandleReportEventLog 就会被调用
    g_signal_connect(eventlogif, "handle_report_event_log", G_CALLBACK(OnHandleReportEventLog), nullptr);

    // 将创建的 skeleton 对象输出到总线上
    if(!g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(eventlogif), connection, OBJECT_PATH, &error))
    {
        syslog(LOG_ERR, "export connection error:%s", error->message);
        g_error_free(error);
    }

    syslog(LOG_DEBUG, "service start");
}

// 无法建立与总线的连接，或者无法获取 bus name
static void NameLostCallback(GDBusConnection *connection,
                             const gchar *name,
                             gpointer user_data)
{
    syslog(LOG_WARNING, "name lost");

    // 退出主循环
    g_main_loop_quit(g_mainloop);
}

int main()
{
    // 启用日志记录（路径：/var/log/messages、 /var/log/syslog、/var/log/debug）
    openlog(SYSLOG_NAME, LOG_ODELAY | LOG_PID, LOG_USER);

    // 创建 GMainLoop 对象（主事件循环）
    g_mainloop = g_main_loop_new(nullptr, FALSE);

    // 初始化缓存
    CacheData::get_mutable_instance().Init();

    // 连接 D-Bus，获取总线上的名字（在总线上注册 Server）
    guint nameid = g_bus_own_name(G_BUS_TYPE_SYSTEM, SERVICE_NAME, G_BUS_NAME_OWNER_FLAGS_NONE,
                                  BusAcquiredCallback, NameAcquiredCallback, NameLostCallback, nullptr, nullptr);

    // 进入主循环（将不停地检查事件源中是否有新事件并进行分发，直到处理来自某个事件源的事件时，触发了 g_main_loop_quit() 调用退出主循环为止）
    g_main_loop_run(g_mainloop);

    g_bus_unown_name(nameid);

    // 减少 g_mainloop 引用计数（如果引用计数为 0，释放 g_mainloop）
    g_main_loop_unref(g_mainloop);

    // 关闭已打开的 system log 的连接
    closelog();

    return 0;
}
